/*
 * glChess - A 3D chess interface
 *
 * Copyright (C) 2001  Robert Ancell <bob27@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include "global.h"
#include "draw.h"

extern float cog_height[6];

/*
 * Flag the view to be redrawn 
 */
void post_redisplay(Game * game)
{
  game->redisplay = TRUE;
}

/*
 * Reshape the GL view 
 */
void reshape(Game * game, int w, int h)
{
  glViewport(0, 0, (GLsizei) w, (GLsizei) h);
  reset_perspective(game);
  glLoadIdentity();
}

/*
 * Recalculate the projection matrix 
 */
void redo_perspective(Game * game, GLint x, GLint y)
{
  if (game->camera->mode == MODE_ORTHO)
  {
    GLfloat width;

    /*
     * Get the smallest edge and get the scale required to make it length 
     * 100 
     */
    if (x < y)
    {				/*
				 * If the width is less than the height 
				 */
      width = 100.0 * (GLfloat) y / (GLfloat) x;
      glOrtho(-50, 50, -0.5 * width, 0.5 * width, 0.0, 200.0);
    } else
    {				/*
				 * If the height is less than the width 
				 */

      width = 100.0 * (GLfloat) x / (GLfloat) y;
      glOrtho(-0.5 * width, 0.5 * width, -50.0, 50.0, 0.0, 200.0);
    }
  } else
    gluPerspective(60.0, (GLfloat) x / (GLfloat) y, 0.5, 10000.0);
}

/*
 * Reset then recalculate the projection matrix 
 */
void reset_perspective(Game * game)
{
  GLint viewport[4];

  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();

  glGetIntegerv(GL_VIEWPORT, viewport);
  redo_perspective(game, viewport[2], viewport[3]);

  glMatrixMode(GL_MODELVIEW);
}

/*
 * Draws a games board 
 */
void draw_board(Game * game)
{
  int x, y, colour;
  Player *player;
  GLfloat white_texture_colour[4] = { 1.0f, 1.0f, 1.0f, 0.5f };

  glPushAttrib(GL_LIGHTING | GL_TEXTURE_BIT);

  glNormal3f(0.0f, 1.0f, 0.0f);

  if (game->is_texture)
  {
    glEnable(GL_TEXTURE_2D);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
  } else
    glDisable(GL_TEXTURE_2D);

  for (x = 0; x < 8; x++)
    for (y = 0; y < 8; y++)
    {
      /*
       * Get board piece colour 
       */
      colour = (x % 2) != (y % 2);
      if (colour)
	player = game->white_player;
      else
	player = game->black_player;

      if (game->is_texture)
      {
	if (game->is_light)
	  glMaterialfv(GL_FRONT, GL_DIFFUSE, white_texture_colour);
	else
	  glColor4f(1.0f, 1.0f, 1.0f, 0.5f);

	glBindTexture(GL_TEXTURE_2D, player->board_texture);
      } else
      {
	if (game->is_light)
	{
	  glMaterialf(GL_FRONT, GL_SHININESS, 0.8);
	  glMaterialfv(GL_FRONT, GL_DIFFUSE, player->board_colour);
	} else
	{
	  if (colour)		/*
				 * white 
				 */
	    glColor4f(0.78, 0.76, 0.40, 0.5);
	  else			/*
				   * black 
				 */
	    glColor4f(0.47, 0.64, 0.43, 0.5);
	}
      }

      /*
       * draw segment 
       */
      glBegin(GL_QUADS);

      glTexCoord2f(0.0f, 0.0f);
      glVertex3f(x * 10.0f - 40.0f, 0.0f, 40.0f - y * 10.0f);
      glTexCoord2f(1.0f, 0.0f);
      glVertex3f(x * 10.0f - 30.0f, 0.0f, 40.0f - y * 10.0f);
      glTexCoord2f(1.0f, 1.0f);
      glVertex3f(x * 10.0f - 30.0f, 0.0f, 30.0f - y * 10.0f);
      glTexCoord2f(0.0f, 1.0f);
      glVertex3f(x * 10.0f - 40.0f, 0.0f, 30.0f - y * 10.0f);

      glEnd();
    }

  glPopAttrib();
}

/*
 * Draws the co-ordinates around the edge of the board 
 */
void draw_coords(Game * game)
{
  int rows, cols;
  GLfloat *number_colour = game->number_colour;
  GLfloat *letter_colour = game->letter_colour;

  glPushAttrib(GL_LIGHTING | GL_TEXTURE_BIT);

  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  /*
   * Draw the numbers 
   */
  glColor4fv(number_colour);
  glMaterialfv(GL_FRONT, GL_DIFFUSE, number_colour);
  for (cols = 0; cols < 8; cols++)
  {
    glBindTexture(GL_TEXTURE_2D, game->number_textures[cols]);

    glBegin(GL_QUADS);
    glTexCoord2f(0.0f, 0.0f);
    glVertex3f(-47.5f, 0.0f, 37.5f - cols * 10.0f);
    glTexCoord2f(1.0f, 0.0f);
    glVertex3f(-42.5f, 0.0f, 37.5f - cols * 10.0f);
    glTexCoord2f(1.0f, 1.0f);
    glVertex3f(-42.5f, 0.0f, 32.5f - cols * 10.0f);
    glTexCoord2f(0.0f, 1.0f);
    glVertex3f(-47.5f, 0.0f, 32.5f - cols * 10.0f);
    glEnd();

    glBegin(GL_QUADS);
    glTexCoord2f(0.0f, 0.0f);
    glVertex3f(47.5f, 0.0f, 32.5f - cols * 10.0f);
    glTexCoord2f(1.0f, 0.0f);
    glVertex3f(42.5f, 0.0f, 32.5f - cols * 10.0f);
    glTexCoord2f(1.0f, 1.0f);
    glVertex3f(42.5f, 0.0f, 37.5f - cols * 10.0f);
    glTexCoord2f(0.0f, 1.0f);
    glVertex3f(47.5f, 0.0f, 37.5f - cols * 10.0f);
    glEnd();
  }

  /*
   * Draw the letters 
   */
  glColor4fv(letter_colour);
  glMaterialfv(GL_FRONT, GL_DIFFUSE, letter_colour);
  for (rows = 0; rows < 8; rows++)
  {
    glBindTexture(GL_TEXTURE_2D, game->letter_textures[rows]);

    glBegin(GL_QUADS);
    glTexCoord2f(0.0f, 0.0f);
    glVertex3f(rows * 10.0f - 37.5f, 0.0f, 47.5f);
    glTexCoord2f(1.0f, 0.0f);
    glVertex3f(rows * 10.0f - 32.5f, 0.0f, 47.5f);
    glTexCoord2f(1.0f, 1.0f);
    glVertex3f(rows * 10.0f - 32.5f, 0.0f, 42.5f);
    glTexCoord2f(0.0f, 1.0f);
    glVertex3f(rows * 10.0f - 37.5f, 0.0f, 42.5f);
    glEnd();

    glBegin(GL_QUADS);
    glTexCoord2f(0.0f, 0.0f);
    glVertex3f(rows * 10.0f - 32.5f, 0.0f, -47.5f);
    glTexCoord2f(1.0f, 0.0f);
    glVertex3f(rows * 10.0f - 37.5f, 0.0f, -47.5f);
    glTexCoord2f(1.0f, 1.0f);
    glVertex3f(rows * 10.0f - 37.5f, 0.0f, -42.5f);
    glTexCoord2f(0.0f, 1.0f);
    glVertex3f(rows * 10.0f - 32.5f, 0.0f, -42.5f);
    glEnd();
  }

  glPopAttrib();
}

/*
 * Draws a games pieces 
 */
void draw_pieces(Game * game)
{
  int x, y, piece, colour, wkr;
  Player *player;
  GLfloat white_texture_colour[4] = { 1.0, 1.0f, 1.0f, 1.0f };

  glPushAttrib(GL_LIGHTING | GL_TEXTURE_BIT);

  if (game->is_texture)
  {
    glEnable(GL_TEXTURE_2D);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
  } else
    glDisable(GL_TEXTURE_2D);

  for (x = 0; x < 8; x++)
    for (y = 0; y < 8; y++)
    {
      wkr = 0;			/*
				 * white knight rotate flag 
				 */
      piece = game->board[x][y];

      if (piece != 0)
      {
	if (piece < 10)
	{			/*
				 * white 
				 */
	  player = game->white_player;
	  colour = 1;
	} else
	{			/*
				 * black 
				 */

	  player = game->black_player;
	  colour = 0;
	}

	if (game->is_light)
	{
	  glMaterialf(GL_FRONT, GL_SHININESS, 0.5);
	  glMaterialfv(GL_FRONT, GL_DIFFUSE, player->piece_colour);
	  if (colour && piece == KNIGHT)	/*
						 * white knight 
						 */
	    wkr = 1;		/*
				 * white knight 
				 */
	  else if (!colour)	/*
				 * black 
				 */
	    piece -= 10;
	} else
	{
	  glColor4fv(player->piece_colour);
	  if (colour)
	  {			/*
				 * white 
				 */
	    if (piece == KNIGHT)	/*
					 * white knight 
					 */
	      wkr = 1;
	  } else		/*
				   * black 
				 */
	    piece -= 10;
	}

	if (game->is_texture)
	{
	  glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
	  glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE,
		       white_texture_colour);
	  glBindTexture(GL_TEXTURE_2D, player->piece_texture);
	}

	glPushMatrix();
	glTranslatef(x * 10 - 35, 0.0, 35 - y * 10);
	if (wkr == 1)		/*
				 * is white knight 
				 */
	  glRotatef(180.0f, 0.0f, 1.0f, 0.0f);
	/*
	 * Don't draw the selected piece -- later 
	 */
	if (!((x == game->selected[0]) && (y == game->selected[1])))
	  glCallList(piece);
	glPopMatrix();
      }
    }

  glPopAttrib();
}

/*
 * Draw the selected piece (is transparent and glows) 
 */
void draw_selected_piece(Game * game)
{
  Player *player = game->current_player;
  int wkr = 0;
  int p = game->board[game->selected[0]][game->selected[1]];
  GLfloat white_texture_colour[4] = { 1.0f, 1.0f, 1.0f, 0.8f };

  glPushAttrib(GL_ENABLE_BIT | GL_LIGHTING);

  if (game->is_texture)
  {
    glColor4f(1.0f, 1.0f, 1.0f, 0.8f);
    glMaterialfv(GL_FRONT, GL_EMISSION, white_texture_colour);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, white_texture_colour);
    glEnable(GL_TEXTURE_2D);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
    glBindTexture(GL_TEXTURE_2D, player->selected_piece_texture);
  } else
  {
    glColor4fv(player->selected_colour);
    glMaterialfv(GL_FRONT, GL_EMISSION, player->selected_colour);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, player->selected_colour);
  }

  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  if (p < 10)
  {
    if (p == 4)
      wkr = 1;
  } else
    p -= 10;

  glPushMatrix();
  glTranslatef((float) game->selected[0] * 10.0f - 35.0f,
	       0.0f, 35.0f - (float) game->selected[1] * 10.0f);
  if (wkr == 1)			/*
				 * is white knight 
				 */
    glRotatef(180.0f, 0.0f, 1.0f, 0.0f);
  glCallList(p);
  glPopMatrix();

  glPopAttrib();
}

/*
 * Places the lights where they should be 
 */
void place_lights(Game * game)
{
  GLfloat light0_pos[4] = { 0.0, 50.0, 0.0, 1.0 };

  if (game->selected[0] != -1)
  {
    int p = game->board[game->selected[0]][game->selected[1]];
    GLfloat light1_pos[4];
    /*
     * FIXME: brown pieces have practically no blue in them 
     */
    GLfloat light1_diffuse[] = { 0.0f, 0.0f, 0.8f };
    GLfloat light1_attenuation[] = { 0.01f, 0.001f, 0.0001f };

    if (p > 10)
      p -= 10;

    light1_pos[0] = game->selected[0] * 10.0f - 35.0f;
    light1_pos[1] = cog_height[p - 1];
    light1_pos[2] = 35.0f - game->selected[1] * 10.0f;
    light1_pos[3] = 1.0f;

    glEnable(GL_LIGHT1);

    glLightfv(GL_LIGHT1, GL_DIFFUSE, light1_diffuse);
    glLightfv(GL_LIGHT1, GL_POSITION, light1_pos);
    glLightf(GL_LIGHT1, GL_CONSTANT_ATTENUATION, light1_attenuation[0]);
    glLightf(GL_LIGHT1, GL_LINEAR_ATTENUATION, light1_attenuation[1]);
    glLightf(GL_LIGHT1, GL_QUADRATIC_ATTENUATION, light1_attenuation[2]);
    glLightfv(GL_LIGHT0, GL_POSITION, light0_pos);
  } else
  {
    glLightfv(GL_LIGHT0, GL_POSITION, light0_pos);
    glDisable(GL_LIGHT1);
  }

}

/*
 * Draw the scene for a game 
 */
void display(Game * game)
{
  /*
   * Clear the buffers 
   */
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  if (game->is_reflect && game->camera->mode != MODE_ORTHO)
    glClear(GL_STENCIL_BUFFER_BIT);

  glMatrixMode(GL_MODELVIEW);

  /*
   * Reset the transformation matrix and translate camera 
   */
  glLoadIdentity();

  /*
   * Translate camera based on view mode 
   */
  switch (game->camera->mode)
  {
  case MODE_FREE:
    glRotatef(-game->camera->free_rot[2], 0.0, 0.0, 1.0);
    glRotatef(-game->camera->free_rot[1], 1.0, 0.0, 0.0);
    glRotatef(-game->camera->free_rot[0], 0.0, 1.0, 0.0);

    glTranslatef(-game->camera->free_pos[0],
		 -game->camera->free_pos[1], -game->camera->free_pos[2]);
    break;
  case MODE_ROTATING:
    glRotatef(game->camera->rotating_pitch, 1.0, 0.0, 0.0);
    glTranslatef(0.0, -game->camera->rotating_height,
		 -game->camera->rotating_rad);
    glRotatef(-game->camera->rotating_rot, 0.0, 1.0, 0.0);
    break;
  case MODE_ORTHO:
    glRotatef(90.0, 1.0, 0.0, 0.0);
    glTranslatef(0.0, -50.0, 0.0);
    break;
  case MODE_TRACKBALL:
    glTranslatef(0.0f, 0.0f, -game->camera->track_rad);
    glRotatef(game->camera->track_rot[1], 1.0f, 0.0f, 0.0f);
    glRotatef(game->camera->track_rot[0], 0.0f, 1.0f, 0.0f);
    break;
  default:
    break;
  }

  /*
   * Place lights 
   */
  if (game->is_light)
    place_lights(game);

  glEnable(GL_DEPTH_TEST);

  /*
   * Draw the pieces 
   */
  draw_pieces(game);

  /*
   * Draw the co-ordinates [1-8] [a-h] 
   */
  if (game->is_coord)
    draw_coords(game);

  /*
   * Make a stencil of the floor if reflections are done 
   */
  if (game->is_reflect && game->camera->mode != MODE_ORTHO)
  {
    /*
     * Save the old scene attributes 
     */
    glPushAttrib(GL_ENABLE_BIT);	/*
					 * Save states 
					 */
    glPushAttrib(GL_COLOR_BUFFER_BIT | GL_FOG_BIT);	/*
							 * Disable drawing 
							 */

    /*
     * Disable stuff 
     */
    glDisable(GL_LIGHTING);
    glDisable(GL_FOG);
    /*
     * Don't draw to the screen or the depth buffer at this moment 
     */
    glColorMask(GL_FALSE, GL_FALSE, GL_FALSE, GL_FALSE);
    glDepthMask(GL_FALSE);

    /*
     * Write to the stencil buffer 
     */
    glEnable(GL_STENCIL_TEST);
    glStencilOp(GL_KEEP, GL_KEEP, GL_REPLACE);
    glStencilFunc(GL_ALWAYS, 1, 0xffffffff);
  }

  /*
   * Draw the board (and the co-ordinates) 
   */
  draw_board(game);

  if (game->is_reflect && game->camera->mode != MODE_ORTHO)
  {				/*
				 * draw the
				 * reflected
				 * pieces 
				 */
    /*
     * Re-enable drawing (GL_COLOR_BUFFER_BIT) 
     */
    glPopAttrib();
    /*
     * Re-enable writing to the depth buffer 
     */
    glDepthMask(GL_TRUE);

    /*
     * Draw only if stencil is set to 1 
     */
    glStencilFunc(GL_EQUAL, 1, 0xffffffff);
    glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);

    /*
     * Reflect in floor 
     */
    glPushMatrix();
    glScalef(1.0f, -1.0f, 1.0f);
    if (game->is_light)
    {
      place_lights(game);
      glEnable(GL_LIGHTING);
    }
    glCullFace(GL_FRONT);
    draw_pieces(game);
    if (game->selected[0] != -1)
      draw_selected_piece(game);
    glPopMatrix();

    /*
     * Prepare to draw board 
     */
    glCullFace(GL_BACK);
    glEnable(GL_BLEND);
    glDisable(GL_DEPTH_TEST);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

    /*
     * Draw the board 
     */
    /*
     * FIXME: this doesn't make any sense at all!!!! 
     */
    glPushMatrix();
    glScalef(1.0, -1.0, 1.0);
    draw_board(game);
    glPopMatrix();

    /*
     * Restore the scene attributes 
     */
    glPopAttrib();
  }

  if (game->selected[0] != -1)
    draw_selected_piece(game);
}
