/*
 * glChess - A 3D chess interface
 *
 * Copyright (C) 2001  Robert Ancell <bob27@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "global.h"
#include "config.h"
#include "player.h"

#define MAX_WORD_LENGTH 50

/*
 * Get one string from a file 
 */
char *get_file_word(FILE * file)
{
  char word[MAX_WORD_LENGTH + 1];
  int i = 0, input_char, is_word;

  do
  {
    /*
     * Clear whitespace 
     */
    do
      input_char = fgetc(file);
    while (isspace(input_char));
    is_word = 1;

    /*
     * Remove comments 
     */
    if (input_char == '#')
    {
      while ((input_char != EOF) && (input_char != '\n'))
	input_char = fgetc(file);
      is_word = 0;
    }
  }
  while (!is_word);		/*
				 * Repeat until whitespace has been cleared
				 * and no comment is present 
				 */

  if (input_char == EOF)
    return NULL;

  /*
   * Get strings (eg "Eat me") 
   */
  if (input_char == '\"')
  {
    do
    {
      input_char = fgetc(file);
      word[i] = (char) input_char;
      i++;
    }
    while ((input_char != '\"') && (input_char != EOF)
	   && (i < MAX_WORD_LENGTH));
    word[i - 1] = '\0';
  } else
  {				/*
				 * Get a word surrounded by whitespace 
				 */

    while (!isspace(input_char) && (input_char != EOF)
	   && (i < MAX_WORD_LENGTH))
    {
      word[i] = (char) input_char;
      input_char = fgetc(file);
      i++;
    }
    word[i] = '\0';
  }

  return strdup(word);
}

/*
 * Load a rc (config) file 
 */
int open_rc(Game * game, const char *file_name)
{
  FILE *rc;
  char *option, *value;

  rc = fopen(file_name, "r");
  if (rc == NULL)
  {
    fprintf(stderr, "%s not found trying /etc/glchessrc\n", file_name);
    rc = fopen("/etc/glchessrc", "r");
    if (rc == NULL)
    {
      fprintf(stderr, "/etc/glchessrc not found, trying ./glchessrc\n");
      rc = fopen("./glchessrc", "r");
    }
  }

  if (rc == NULL)
  {
    fprintf(stderr, "Error opening rc config file...\n");
    return -1;
  }

  while (1)
  {
    /*
     * Get the next available option 
     */
    option = get_file_word(rc);

    /*
     * If no more options 
     */
    if (option == NULL)
      return 1;

    if ((value = get_file_word(rc)) == NULL)
    {
      free(option);
      fprintf(stderr,
	      "Final option found without a corresponding value in %s\n",
	      file_name);
    }

    if (set_variable(game, option, value, rc) == -1)
      fprintf(stderr, "Unknown option in %s.\n", file_name);
  }
}

/*
 * Saves a rc (config) file 
 */
int save_rc(Game * game, const char *fname)
{
  FILE *rc = fopen(fname, "w");
  int i;

  if (rc == NULL)
  {
    fprintf(stderr, "Error opening rc config file (%s)...\n", fname);
    return -1;
  }

  fprintf(rc,
	  "#This is the glChess configuration file, automatically generated by glChess\n");
  fprintf(rc, "is_reflect              %d\n", game->is_reflect);
  fprintf(rc, "is_light                %d\n", game->is_light);
  fprintf(rc, "is_smooth               %d\n", game->is_smooth);
  fprintf(rc, "is_texture              %d\n", game->is_texture);
  fprintf(rc, "is_fog                  %d\n", game->is_fog);
  fprintf(rc, "camera_mode             %d\n", game->camera->mode);
  fprintf(rc, "\n");
  fprintf(rc, "texture_dir             %s\n", game->texture_dir);
  fprintf(rc, "number_colour          ");
  for (i = 0; i < 4; i++)
    fprintf(rc, " %f", game->number_colour[i]);
  fprintf(rc, "\n");
  fprintf(rc, "letter_colour          ");
  for (i = 0; i < 4; i++)
    fprintf(rc, " %f", game->letter_colour[i]);
  fprintf(rc, "\n");
  fprintf(rc, "\n");
  fprintf(rc, "white_name              \"%s\"\n",
	  game->white_player->name);
  fprintf(rc, "white_piece_colour     ");
  for (i = 0; i < 4; i++)
    fprintf(rc, " %f", game->white_player->piece_colour[i]);
  fprintf(rc, "\n");
  fprintf(rc, "white_selected_colour  ");
  for (i = 0; i < 4; i++)
    fprintf(rc, " %f", game->white_player->selected_colour[i]);
  fprintf(rc, "\n");
  fprintf(rc, "white_board_colour     ");
  for (i = 0; i < 4; i++)
    fprintf(rc, " %f", game->white_player->board_colour[i]);
  fprintf(rc, "\n");
  fprintf(rc, "white_texture_dir       %s\n",
	  game->white_player->texture_dir);
  fprintf(rc, "white_piece_texture     %s\n",
	  game->white_player->piece_texture_file);
  fprintf(rc, "white_selected_texture  %s\n",
	  game->white_player->selected_piece_texture_file);
  fprintf(rc, "white_board_texture     %s\n",
	  game->white_player->board_texture_file);
  fprintf(rc, "\n");
  fprintf(rc, "black_name              \"%s\"\n",
	  game->black_player->name);
  fprintf(rc, "black_piece_colour     ");
  for (i = 0; i < 4; i++)
    fprintf(rc, " %f", game->black_player->piece_colour[i]);
  fprintf(rc, "\n");
  fprintf(rc, "black_selected_colour  ");
  for (i = 0; i < 4; i++)
    fprintf(rc, " %f", game->black_player->selected_colour[i]);
  fprintf(rc, "\n");
  fprintf(rc, "black_board_colour     ");
  for (i = 0; i < 4; i++)
    fprintf(rc, " %f", game->black_player->board_colour[i]);
  fprintf(rc, "\n");
  fprintf(rc, "black_texture_dir       %s\n",
	  game->black_player->texture_dir);
  fprintf(rc, "black_piece_texture     %s\n",
	  game->black_player->piece_texture_file);
  fprintf(rc, "black_selected_texture  %s\n",
	  game->black_player->selected_piece_texture_file);
  fprintf(rc, "black_board_texture     %s\n",
	  game->black_player->board_texture_file);
  fprintf(rc, "\n");
  fprintf(rc, "cecp_mode               ");
  if (game->cecp.mode == CECP_MODE_HARD)
    fprintf(rc, "HARD\n");
  else
    fprintf(rc, "EASY\n");
  fprintf(rc, "cecp_search_depth       %d\n", game->cecp.search_depth);
  fprintf(rc, "cecp_ai_player          ");
  if (game->cecp.ai_player == WHITE PLAYER)
    fprintf(rc, "WHITE\n");
  else
    fprintf(rc, "BLACK\n");
  fprintf(rc, "\n");
  fprintf(rc, "beep_signal             %d\n", game->cecp.beep_signal);

  return TRUE;
}

/*
 * Gets four floats from the input file 
 */
GLfloat *get_colour(FILE * file, char **var_value)
{
  GLfloat *colour = malloc(sizeof(GLfloat) * 4);
  int i;

  colour[0] = (GLfloat) atof(*var_value);
  for (i = 1; i < 4; i++)
  {
    free(*var_value);
    *var_value = get_file_word(file);
    colour[i] = (GLfloat) atof(*var_value);
  }

  return colour;
}

/*
 * Set the variable named by a string 
 */
int set_variable(Game * game, char *var_name, char *var_value, FILE * file)
{
  const char vars[][30] =
      { "is_light", "is_smooth", "is_reflect", "is_texture", "is_fog",
    "camera_mode",
    "white_name",
    "white_piece_colour", "white_selected_colour",
    "white_board_colour",
    "white_texture_dir", "white_piece_texture",
    "white_selected_texture", "white_board_texture",
    "black_name",
    "black_piece_colour", "black_selected_colour",
    "black_board_colour",
    "black_texture_dir", "black_piece_texture",
    "black_selected_texture", "black_board_texture",
    "texture_dir", "number_colour", "letter_colour",
    "cecp_mode", "cecp_search_depth", "cecp_ai_player",
    "beep_signal"
  };
  int i, n_vars = 29;

  for (i = 0; i < n_vars; i++)
    if (!strcmp(var_name, vars[i]))
    {
      switch (i)
      {
      case 0:			/*
				 * is_light 
				 */
	game->is_light = atoi(var_value);
	break;
      case 1:			/*
				 * is_smooth 
				 */
	game->is_smooth = atoi(var_value);
	break;
      case 2:			/*
				 * is_reflect 
				 */
	game->is_reflect = atoi(var_value);
	break;
      case 3:			/*
				 * is_texture 
				 */
	game->is_texture = atoi(var_value);
	break;
      case 4:			/*
				 * is_fog 
				 */
	game->is_fog = atoi(var_value);
      case 5:			/*
				 * camera_mode 
				 */
	game->camera->mode = atoi(var_value);
	break;
      case 6:			/*
				 * white_name 
				 */
	set_player_name(game->white_player, var_value);
	break;
      case 7:			/*
				 * white_piece_colour 
				 */
	free(game->white_player->piece_colour);
	game->white_player->piece_colour = get_colour(file, &var_value);
	break;
      case 8:			/*
				 * white_selected_colour 
				 */
	free(game->white_player->selected_colour);
	game->white_player->selected_colour = get_colour(file, &var_value);
	break;
      case 9:			/*
				 * white_board_colour 
				 */
	free(game->white_player->board_colour);
	game->white_player->board_colour = get_colour(file, &var_value);
	break;
      case 10:			/*
				 * white_texture_dir 
				 */
	free(game->white_player->texture_dir);
	game->white_player->texture_dir = strdup(var_value);
	break;
      case 11:			/*
				 * white_piece_texture 
				 */
	free(game->white_player->piece_texture_file);
	game->white_player->piece_texture_file = strdup(var_value);
	break;
      case 12:			/*
				 * white_selected_texture 
				 */
	free(game->white_player->selected_piece_texture_file);
	game->white_player->selected_piece_texture_file =
	    strdup(var_value);
	break;
      case 13:			/*
				 * white_board_texture 
				 */
	free(game->white_player->board_texture_file);
	game->white_player->board_texture_file = strdup(var_value);
	break;
      case 14:			/*
				 * black_name 
				 */
	set_player_name(game->black_player, var_value);
	break;
      case 15:			/*
				 * black_piece_colour 
				 */
	free(game->black_player->piece_colour);
	game->black_player->piece_colour = get_colour(file, &var_value);
	break;
      case 16:			/*
				 * black_selected_colour 
				 */
	free(game->black_player->selected_colour);
	game->black_player->selected_colour = get_colour(file, &var_value);
	break;
      case 17:			/*
				 * black_board_colour 
				 */
	free(game->black_player->board_colour);
	game->black_player->board_colour = get_colour(file, &var_value);
	break;
      case 18:			/*
				 * black_texture_dir 
				 */
	free(game->black_player->texture_dir);
	game->black_player->texture_dir = strdup(var_value);
	break;
      case 19:			/*
				 * black_piece_texture 
				 */
	free(game->black_player->piece_texture_file);
	game->black_player->piece_texture_file = strdup(var_value);
	break;
      case 20:			/*
				 * black_selected_texture 
				 */
	free(game->black_player->selected_piece_texture_file);
	game->black_player->selected_piece_texture_file =
	    strdup(var_value);
	break;
      case 21:			/*
				 * black_board_texture 
				 */
	free(game->black_player->board_texture_file);
	game->black_player->board_texture_file = strdup(var_value);
	break;
      case 22:			/*
				 * texture_dir 
				 */
	free(game->texture_dir);
	game->texture_dir = strdup(var_value);
	break;
      case 23:			/*
				 * number_colour 
				 */
	free(game->number_colour);
	game->number_colour = get_colour(file, &var_value);
	break;
      case 24:			/*
				 * letter_colour 
				 */
	free(game->letter_colour);
	game->letter_colour = get_colour(file, &var_value);
	break;
      case 25:			/*
				 * cecp_mode 
				 */
	if (!strcmp(var_value, "EASY"))
	  game->cecp.mode = CECP_MODE_EASY;
	else if (!strcmp(var_value, "HARD"))
	  game->cecp.mode = CECP_MODE_HARD;
	else
	{
	  fprintf(stderr, "Warning: Unknown CECP mode: %s\n", var_value);
	  game->cecp.mode = CECP_MODE_UNKNOWN;
	}
	break;
      case 26:			/*
				 * cecp_search_depth 
				 */
	game->cecp.search_depth = atoi(var_value);
	break;
      case 27:			/*
				 * cecp_ai_player 
				 */
	if (!strcmp(var_value, "WHITE"))
	  game->cecp.ai_player = WHITE PLAYER;
	else if (!strcmp(var_value, "BLACK"))
	  game->cecp.ai_player = BLACK PLAYER;
	else
	{
	  fprintf(stderr, "Warning: Unknown CECP player: %s\n", var_value);
	  game->cecp.ai_player = BLACK PLAYER;
	}
	break;
      case 28:			/*
				 * beep_signal 
				 */
	game->cecp.beep_signal = atoi(var_value);
	break;
      }
    }

  free(var_name);
  free(var_value);

  return 1;
}
