/***************************************************************************

	Exidy 440 system

    driver by Aaron Giles


	Currently implemented:
		* Crossbow
		* Cheyenne
		* Combat
		* Crackshot
		* Chiller
		* Top Secret
		* Clay Pigeon
		* Hit 'n Miss
		* Who Dunit
		* Showdown


****************************************************************************

	Memory map

****************************************************************************

	========================================================================
	CPU #1
	========================================================================
	0000-1FFF   R/W   xxxxxxxx    Image RAM (64 images x 128 bytes, each image is 16x16x4bpp)
	            R/W   xxxx----       (left pixel)
	            R/W   ----xxxx       (right pixel)
	2000-209F   R/W   xxxxxxxx    Sprite RAM (40 entries x 4 bytes)
	            R/W   xxxxxxxx       (0: Y position, inverted)
	            R/W   -------x       (1: X position, upper bit, inverted)
	            R/W   xxxxxxxx       (2: X position, lower 8 bits, inverted)
	            R/W   --xxxxxx       (3: image number, inverted)
	20A0-29FF   R/W   xxxxxxxx    Stack RAM
	2A00-2AFF   R/W   xxxxxxxx    Scanline I/O region
	            R/W   xxxx----       (left pixel)
	            R/W   ----xxxx       (right pixel)
	2B00        R     xxxxxxxx    Current vertical beam position
	2B01        R     xxxxxxxx    Latched horizontal collision/beam detect location
	2B01        W     --------    Clear VBLANK FIRQ
	2B02        R/W   xxxxxxxx    Scanline I/O selector
	2B03        R     xxxxxxxx    Input port
	            R     x-------       (VBLANK FIRQ latched, active low)
	            R     -x------       (collision/beam detect FIRQ latched, active low)
	            R     --xxxx--       (4 position DIP switch, active low)
					R     ------x-       (trigger, active low)
					R     -------x       (mirror of VBLANK FIRQ latch, Whodunit only)
	2B03        W     xxxxxxxx    Control port
	            W     xxxx----       (ROM bank select, 0-15)
	            W     ----x---       (collision/beam detect FIRQ enable)
	            W     -----x--       (select FIRQ source: 0=collision, 1=beam detect)
	            W     ------x-       (select palette bank for I/O)
	            W     -------x       (select palette bank for display)
	2C00-2DFF   R/W   xxxxxxxx    Palette RAM (2 banks x 512 bytes)
	            R/W   x-------       (0: collision trigger bit)
	            R/W   -xxxxx--       (0: red component)
	            R/W   ------xx       (0: green component, upper 2 bits)
	            R/W   xxx-----       (1: green component, lower 3 bits)
	            R/W   ---xxxxx       (1: blue component)
	2E00-2E1F   R/W   xxxxxxxx    Sound command
	2E20-2E3F   R     ----xxxx    Coin counter, clearing IRQ
	            R     ----x---       (acknowledgement of sound command read, active high)
	            R     -----x--       (unknown use, active low - gives "coin mech jammed" message)
	            R     ------x-       (coin 2, active low)
	            R     -------x       (coin 1, active low)
	2E40-2E5F   W     -------x    Clear coin counter
	2E60-2E7F   R     xxxxxxxx    8 position DIP switch, active low
	2E80-2E9F   R     --xxxxxx    Player control bits
	            R     -------x       (start button, active low)
	2EA0-2EBF   R     ----xxxx    Coin counter, non-clearing
	            R     ----x---       (acknowledgement of sound command read, active high)
	            R     -----x--       (unknown use, active low - gives "coin mech jammed" message)
	            R     ------x-       (coin 2, active low)
	            R     -------x       (coin 1, active low)
	2EC0-2EDF   R     xxxxxxxx    Copy protection check (Clay Pigeon)
	2EC0-2EDF	R/W	xxxxxxxx		Copy protection and I/O space (Top Secret)
	3000-3FFF   R/W   xxxxxxxx    RAM
	4000-7FFF   R     xxxxxxxx    Banked ROM
	6000-7FFF   R/W   xxxxxxxx    EEROM (when bank 15 is selected only)
	8000-FFFF   R     xxxxxxxx    Program ROM
	========================================================================
	Interrupts:
		NMI not connected
		IRQ generated by coins, detect on audio PCB
		FIRQ generated by VBLANK, collision detect, and beam detect
	========================================================================


	========================================================================
	CPU #2
	========================================================================
	0000-7FFF   R     xxxxxxxx    Banked ROM (never access because DMA switches constantly)
	8000-8016   R/W   xxxxxxxx    MC6844 DMA controller I/O
	8400-8407   R/W   xxxxxxxx    Volume control for each channel: left1, right1, left2, right2...
	8800        R     xxxxxxxx    Sound command
	9400-9403   W     xxxxxxxx    Bank select for each channel
	            W     ----x---       (select bank 3, active high)
	            W     -----x--       (select bank 2, active high)
	            W     ------x-       (select bank 1, active high)
	            W     -------x       (select bank 0, active high)
	9800        W     --------    Clear sound IRQ
	A000-BFFF   R/W   xxxxxxxx    RAM
	E000-FFFF   R     xxxxxxxx    Program ROM
	========================================================================
	Interrupts:
		NMI not connected
		IRQ generated by VSYNC
		FIRQ generated by write to sound command register
	========================================================================

***************************************************************************

	Differences between machines (assuming Crossbow is the base)

	Cheyenne (reports Hardware Error 5 if any of these are not true)
		* a read from $2B03 must return 0 in bit 0

		conclusion: LS244 at 18J on main board needs to generate negative logic

	Combat (reports Hardware Error 1 if any of these are not true)
		* a read from $2E80 must return 0's in bits 1,2,3,4,5
		* a read from $2B03 must return 0 in bit 0

		conclusion: LS244 at 18J on main board needs to generate negative logic
						LS367 at 1A on audio board needs to generate negative logic

	Crackshot (reports Hardware Error W if any of these are not true)
		* a read from $2E20 must return 0 in bit 2
		* a read from $2E80 must return 0's in bits 1,2,3,4,5
		* a read from $2B03 must return 0 in bit 0

		conclusion: LS244 at 18J on main board needs to generate negative logic
						LS367 at 1A on audio board needs to generate negative logic
						LS128 at 2A on audio board needs negative logic into pin 5

	Chiller (reports Hardware Error R if any of these are not true)
		* a read from $2E20 must return 0 in bit 2
		* a read from $2E80 must return 0's in bits 1,2,3,4,5
		* a read from $2B03 must return 0 in bit 0

		conclusion: LS244 at 18J on main board needs to generate negative logic
						LS367 at 1A on audio board needs to generate negative logic
						LS128 at 2A on audio board needs negative logic into pin 5

	Top Secret (reports Hardware Error H if any of these are not true)
		* a read from $2EC0 must return $00
		* a read from $2E20 must return 0 in bit 2
		* a read from $2E80 must return 0's in bits 2,3,4,5
		* a read from $2B03 must return 0 in bit 0

		conclusion: LS244 at 18J on main board needs to generate negative logic
								and must connect the inputs for bits 0 and 1
						LS367 at 1A on audio board needs to generate negative logic
						LS128 at 2A on audio board needs negative logic into pin 5
						the I/O space at $2EC0 must be hooked up to something

	Clay Pigeon (reports Hardware Error N if any of these are not true)
		* a read from $2EC3 must return $76
		* a read from $2E20 must return 0 in bit 2
		* a read from $2E80 must return 0's in bits 1,2,3,4,5
		* a read from $2B03 must return 0 in bit 0

		conclusion: LS244 at 18J on main board needs to generate negative logic
						LS367 at 1A on audio board needs to generate negative logic
						LS128 at 2A on audio board needs negative logic into pin 5
						something must be hooked up to pin 7 of the LS42 at 3A on
								the audio board to put $76 on the data lines going
								back to the logic board when triggered

	Hit'N Miss (reports Hardware Error H if any of these are not true)
		* a read from $2E20 must return 0 in bit 2
		* a read from $2E80 must return 0's in bits 2,3,4,5
		* a read from $2B03 must return the same value for bit 0 as for bit 1

		conclusion: LS244 at 18J on main board needs to generate negative logic
								and must connect the inputs for bits 0 and 1
						LS367 at 1A on audio board needs to generate negative logic
						LS128 at 2A on audio board needs negative logic into pin 5

	Who Dunit (reports Hardware Error H if any of these are not true)
		* a read from $2E20 must return 0 in bit 2
		* a read from $2E80 must return 0's in bits 2,3,4,5
		* a read from $2B03 must return the same value for bit 0 as for bit 7

		conclusion: LS244 at 18J on main board needs to generate negative logic
								and must connect the inputs for bits 0 and 7
						LS367 at 1A on audio board needs to generate negative logic
						LS128 at 2A on audio board needs negative logic into pin 5

	Showdown
		* a read from $2E20 must return 0 in bit 2
		* a read from $2E80 must return 0's in bits 1,2,3,4,5
		* a read from $2B03 must return 0 in bit 0
		* the PLD at 1E on the logic board must recognize the sequence $0055,$00ed
			and return the following data from reads in the first 24 bytes:
				0x15,0x40,0xc1,0x8d,0x4c,0x84,0x0e,0xce,
				0x52,0xd0,0x99,0x48,0x80,0x09,0xc9,0x45,
				0xc4,0x8e,0x5a,0x92,0x18,0xd8,0x51,0xc0
		* the PLD at 1E on the logic board must also recognize the sequence $0055,$1243
			and return the following data from reads in the first 24 bytes:
				0x11,0x51,0xc0,0x89,0x4d,0x85,0x0c,0xcc,
				0x46,0xd2,0x98,0x59,0x91,0x08,0xc8,0x41,
				0xc5,0x8c,0x4e,0x86,0x1a,0xda,0x50,0xd1

		conclusion: something must emulate this PLD; also, not sure where the
								bytes to compare against came from, so it's possible
								that the real machine isn't expecting the same values
						LS244 at 18J on main board needs to generate negative logic
						LS367 at 1A on audio board needs to generate negative logic
						LS128 at 2A on audio board needs negative logic into pin 5

***************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"


/* globals */
UINT8 exidy440_bank;
UINT8 exidy440_topsecret;


/* local variables */
static UINT8 port_0_xor;
static UINT8 port_2_xor;
static UINT8 port_3_xor;
static UINT8 mirror_vblank_bit;
static UINT8 mirror_trigger_bit;
static UINT8 copy_protection_read;
static UINT8 coin_state;
static UINT8 last_coins;
static UINT8 showdown_bank_triggered;


/* sound driver data & functions */
extern UINT8 exidy440_sound_command;
extern UINT8 exidy440_sound_command_ack;
extern UINT8 *exidy440_m6844_data;
extern UINT8 *exidy440_sound_banks;
extern UINT8 *exidy440_sound_volume;

int exidy440_sh_start(const struct MachineSound *msound);
void exidy440_sh_stop(void);
void exidy440_sh_update(void);
int exidy440_sound_interrupt(void);

READ_HANDLER( exidy440_m6844_r );
WRITE_HANDLER( exidy440_m6844_w );
READ_HANDLER( exidy440_sound_command_r );
WRITE_HANDLER( exidy440_sound_volume_w );
WRITE_HANDLER( exidy440_sound_interrupt_clear_w );


/* video driver data & functions */
extern UINT8 *spriteram;
extern UINT8 *exidy440_imageram;
extern UINT8 *exidy440_scanline;
extern UINT8 exidy440_firq_vblank;
extern UINT8 exidy440_firq_beam;
extern UINT8 topsecex_yscroll;

int exidy440_vh_start(void);
void exidy440_vh_stop(void);
void exidy440_vh_screenrefresh(struct osd_bitmap *bitmap,int full_refresh);
void topsecex_vh_screenrefresh(struct osd_bitmap *bitmap,int full_refresh);
int exidy440_vblank_interrupt(void);

READ_HANDLER( exidy440_videoram_r );
WRITE_HANDLER( exidy440_videoram_w );
READ_HANDLER( exidy440_paletteram_r );
WRITE_HANDLER( exidy440_paletteram_w );
WRITE_HANDLER( exidy440_control_w );
READ_HANDLER( exidy440_vertical_pos_r );
READ_HANDLER( exidy440_horizontal_pos_r );
WRITE_HANDLER( exidy440_interrupt_clear_w );



/*************************************
 *
 *	EEROM save/load
 *
 *************************************/

static void nvram_handler(void *file,int read_or_write)
{
	if (read_or_write)
		/* the EEROM lives in the uppermost 8k of the top bank */
		osd_fwrite(file, &memory_region(REGION_CPU1)[0x10000 + 15 * 0x4000 + 0x2000], 0x2000);
	else
	{
		if (file)
			osd_fread(file, &memory_region(REGION_CPU1)[0x10000 + 15 * 0x4000 + 0x2000], 0x2000);
		else
			memset(&memory_region(REGION_CPU1)[0x10000 + 15 * 0x4000 + 0x2000], 0, 0x2000);
	}
}


/*************************************
 *
 *	Interrupt handling
 *
 *************************************/

static void handle_coins(void)
{
	int coins;

	/* if we got a coin, set the IRQ on the main CPU */
	coins = input_port_3_r(0) & 3;
	if (((coins ^ last_coins) & 0x01) && (coins & 0x01) == 0)
	{
		coin_state &= ~0x01;
		cpu_set_irq_line(0, 0, ASSERT_LINE);
	}
	if (((coins ^ last_coins) & 0x02) && (coins & 0x02) == 0)
	{
		coin_state &= ~0x02;
		cpu_set_irq_line(0, 0, ASSERT_LINE);
	}
	last_coins = coins;
}


static int main_interrupt(void)
{
	/* generate coin interrupts */
	handle_coins();
	return exidy440_vblank_interrupt();
}


static void init_machine(void)
{
	exidy440_bank = 0;
	cpu_setbank(1, &memory_region(REGION_CPU1)[0x10000]);

	last_coins = input_port_3_r(0) & 3;
	coin_state = 3;
}



/*************************************
 *
 *	Primary input port read
 *
 *************************************/

static READ_HANDLER( input_r )
{
	int result = input_port_0_r(offset);

	/* the FIRQ cause is reflected in the upper 2 bits */
	if (exidy440_firq_vblank) result ^= 0x80;
	if (exidy440_firq_beam) result ^= 0x40;

	/* Whodunit needs the VBLANK bit mirrored to bit 0 */
	if (mirror_vblank_bit && exidy440_firq_vblank) result ^= 0x01;

	/* Hit'N Miss needs the trigger bit mirrored to bit 0 */
	if (mirror_trigger_bit) result = (result & 0xfe) | ((result >> 1) & 1);

	/* return with the appropriate XOR */
	return result ^ port_0_xor;
}



/*************************************
 *
 *	Banked RAM (actually EEROM)
 *
 *************************************/

static WRITE_HANDLER( bankram_w )
{
	/* EEROM lives in the upper 8k of bank 15 */
	if (exidy440_bank == 15 && offset >= 0x2000)
	{
		memory_region(REGION_CPU1)[0x10000 + 15 * 0x4000 + offset] = data;
		logerror("W EEROM[%04X] = %02X\n", offset - 0x2000, data);
	}

	/* everything else is ROM and we ignore it */
}



/*************************************
 *
 *	Audio board I/O reads
 *
 *************************************/

static READ_HANDLER( io1_r )
{
	int result = 0xff;

	switch (offset & 0xe0)
	{
		case 0x00:										/* sound command */
			result = exidy440_sound_command;
			break;

		case 0x20:										/* coin bits I/O1 */
			result = (input_port_3_r(offset) & 0xfc) | coin_state;
			result ^= port_3_xor;

			/* sound command acknowledgements come on bit 3 here */
			if (!exidy440_sound_command_ack)
				result ^= 0x08;

			/* I/O1 accesses clear the CIRQ flip/flop */
			cpu_set_irq_line(0, 0, CLEAR_LINE);
			break;

		case 0x40:										/* clear coin counters I/O2 */
			result = 0xff;
			break;

		case 0x60:										/* dip switches (8) */
			result = input_port_1_r(offset);
			break;

		case 0x80:										/* player control bits */
			result = input_port_2_r(offset) ^ port_2_xor;
			break;

		case 0xa0:										/* coin bits I/O3 */
			result = (input_port_3_r(offset) & 0xfc) | coin_state;
			result ^= port_3_xor;

			/* sound command acknowledgements come on bit 3 here */
			if (exidy440_sound_command_ack)
				result ^= 0x08;
			break;

		case 0xc0:
			/* for Clay Pigeon and Top Secret */
			if (offset < 0xc4)
				return copy_protection_read;

			/* for Top Secret only */
			if (offset == 0xc5)
				return (input_port_5_r(offset) & 1) ? 0x01 : 0x02;
			else if (offset == 0xc6)
				return input_port_4_r(offset);
			else if (offset == 0xc7)
				return input_port_6_r(offset);
			else
				return 0;
	}

	return result;
}



/*************************************
 *
 *	Audio board I/O writes
 *
 *************************************/

static void delayed_sound_command_w(int param)
{
	exidy440_sound_command = param;
	exidy440_sound_command_ack = 0;

	/* cause an FIRQ on the sound CPU */
	cpu_set_irq_line(1, 1, ASSERT_LINE);
}


static WRITE_HANDLER( io1_w )
{
	logerror("W I/O1[%02X]=%02X\n", offset, data);

	/* switch off the upper 4 bits of the offset */
	switch (offset & 0xe0)
	{
		case 0x00:										/* sound command */
			timer_set(TIME_NOW, data, delayed_sound_command_w);
			break;

		case 0x20:										/* coin bits I/O1 */

			/* accesses here clear the CIRQ flip/flop */
			cpu_set_irq_line(0, 0, CLEAR_LINE);
			break;

		case 0x40:										/* clear coin counters I/O2 */
			coin_state = 3;
			break;

		case 0x60:										/* dip switches (8) */
			break;

		case 0x80:										/* player control bits */
			break;

		case 0xa0:										/* coin bits I/O3 */
			break;

		case 0xc0:
			/* for Top Secret only */
			if (offset == 0xc1)
				topsecex_yscroll = data;
			break;
	}
}



/*************************************
 *
 *	Game-specific handlers
 *
 *************************************/

READ_HANDLER( showdown_pld_trigger_r )
{
	/* bank 0 is where the PLD lives - a read here will set the trigger */
	if (exidy440_bank == 0)
		showdown_bank_triggered = 1;

	/* just return the value from the current bank */
	return memory_region(REGION_CPU1)[0x10000 + exidy440_bank * 0x4000 + 0x0055 + offset];
}


READ_HANDLER( showdown_pld_select1_r )
{
	/* bank 0 is where the PLD lives - a read here after a trigger will set bank "1" */
	if (exidy440_bank == 0 && showdown_bank_triggered)
	{
		static const UINT8 bankdata[0x18] =
		{
			0x15,0x40,0xc1,0x8d,0x4c,0x84,0x0e,0xce,
			0x52,0xd0,0x99,0x48,0x80,0x09,0xc9,0x45,
			0xc4,0x8e,0x5a,0x92,0x18,0xd8,0x51,0xc0
		};

		/* clear the trigger and copy the expected 24 bytes to the RAM area */
		showdown_bank_triggered = 0;
		memcpy(&memory_region(REGION_CPU1)[0x10000], bankdata, 0x18);
	}

	/* just return the value from the current bank */
	return memory_region(REGION_CPU1)[0x10000 + exidy440_bank * 0x4000 + 0x00ed + offset];
}


READ_HANDLER( showdown_pld_select2_r )
{
	/* bank 0 is where the PLD lives - a read here after a trigger will set bank "2" */
	if (exidy440_bank == 0 && showdown_bank_triggered)
	{
		static const UINT8 bankdata[0x18] =
		{
			0x11,0x51,0xc0,0x89,0x4d,0x85,0x0c,0xcc,
			0x46,0xd2,0x98,0x59,0x91,0x08,0xc8,0x41,
			0xc5,0x8c,0x4e,0x86,0x1a,0xda,0x50,0xd1
		};

		/* clear the trigger and copy the expected 24 bytes to the RAM area */
		showdown_bank_triggered = 0;
		memcpy(&memory_region(REGION_CPU1)[0x10000], bankdata, 0x18);
	}

	/* just return the value from the current bank */
	return memory_region(REGION_CPU1)[0x10000 + exidy440_bank * 0x4000 + 0x1243 + offset];
}



/*************************************
 *
 *	Main CPU memory handlers
 *
 *************************************/

static MEMORY_READ_START( readmem_cpu1 )
	{ 0x0000, 0x1fff, MRA_RAM },
	{ 0x2000, 0x209f, MRA_RAM },
	{ 0x20a0, 0x29ff, MRA_RAM },
	{ 0x2a00, 0x2aff, exidy440_videoram_r },
	{ 0x2b00, 0x2b00, exidy440_vertical_pos_r },
	{ 0x2b01, 0x2b01, exidy440_horizontal_pos_r },
	{ 0x2b02, 0x2b02, MRA_RAM },
	{ 0x2b03, 0x2b03, input_r },
	{ 0x2c00, 0x2dff, exidy440_paletteram_r },
	{ 0x2e00, 0x2eff, io1_r },
	{ 0x3000, 0x3fff, MRA_RAM },
	{ 0x4000, 0x7fff, MRA_BANK1 },
	{ 0x8000, 0xffff, MRA_ROM },
MEMORY_END


static MEMORY_WRITE_START( writemem_cpu1 )
	{ 0x0000, 0x1fff, MWA_RAM, &exidy440_imageram },
	{ 0x2000, 0x209f, MWA_RAM, &spriteram },
	{ 0x20a0, 0x29ff, MWA_RAM },
	{ 0x2a00, 0x2aff, exidy440_videoram_w },
	{ 0x2b01, 0x2b01, exidy440_interrupt_clear_w },
	{ 0x2b02, 0x2b02, MWA_RAM, &exidy440_scanline },
	{ 0x2b03, 0x2b03, exidy440_control_w },
	{ 0x2c00, 0x2dff, exidy440_paletteram_w },
	{ 0x2e00, 0x2eff, io1_w },
	{ 0x3000, 0x3fff, MWA_RAM },
	{ 0x4000, 0x7fff, bankram_w },
	{ 0x8000, 0xffff, MWA_ROM },
MEMORY_END



/*************************************
 *
 *	Sound CPU memory handlers
 *
 *************************************/

static MEMORY_READ_START( readmem_cpu2 )
	{ 0x8000, 0x8016, exidy440_m6844_r },
	{ 0x8400, 0x8407, MRA_RAM },
	{ 0x8800, 0x8800, exidy440_sound_command_r },
	{ 0x9800, 0x9800, MRA_NOP },
	{ 0xa000, 0xbfff, MRA_RAM },
	{ 0xe000, 0xffff, MRA_ROM },
MEMORY_END


static MEMORY_WRITE_START( writemem_cpu2 )
	{ 0x8000, 0x8016, exidy440_m6844_w, &exidy440_m6844_data },
	{ 0x8400, 0x8407, exidy440_sound_volume_w, &exidy440_sound_volume },
	{ 0x9400, 0x9403, MWA_RAM, &exidy440_sound_banks },
	{ 0x9800, 0x9800, exidy440_sound_interrupt_clear_w },
	{ 0xa000, 0xbfff, MWA_RAM },
	{ 0xe000, 0xffff, MWA_ROM },
MEMORY_END



/*************************************
 *
 *	Port definitions
 *
 *************************************/

#define COINAGE \
	PORT_DIPNAME( 0x0f, 0x0f, DEF_STR( Coinage ) ) \
	PORT_DIPSETTING(    0x03, DEF_STR( 4C_1C ) ) \
	PORT_DIPSETTING(    0x07, DEF_STR( 3C_1C ) ) \
	PORT_DIPSETTING(    0x02, DEF_STR( 4C_2C ) ) \
	PORT_DIPSETTING(    0x0b, DEF_STR( 2C_1C ) ) \
	PORT_DIPSETTING(    0x06, DEF_STR( 3C_2C ) ) \
	PORT_DIPSETTING(    0x01, DEF_STR( 4C_3C ) ) \
	PORT_DIPSETTING(    0x00, DEF_STR( 4C_4C ) ) \
	PORT_DIPSETTING(    0x05, DEF_STR( 3C_3C ) ) \
	PORT_DIPSETTING(    0x0a, DEF_STR( 2C_2C ) ) \
	PORT_DIPSETTING(    0x0f, DEF_STR( 1C_1C ) ) \
	PORT_DIPSETTING(    0x04, DEF_STR( 3C_4C ) ) \
	PORT_DIPSETTING(    0x09, DEF_STR( 2C_3C ) ) \
	PORT_DIPSETTING(    0x08, DEF_STR( 2C_4C ) ) \
	PORT_DIPSETTING(    0x0e, DEF_STR( 1C_2C ) ) \
	PORT_DIPSETTING(    0x0d, DEF_STR( 1C_3C ) ) \
	PORT_DIPSETTING(    0x0c, DEF_STR( 1C_4C ) )


INPUT_PORTS_START( crossbow )
	PORT_START				/* player inputs and logic board dips */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x08, "2" )
	PORT_DIPSETTING(    0x0c, "3" )
	PORT_DIPSETTING(    0x04, "4" )
	PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Easy" )
	PORT_DIPSETTING(    0x30, "Normal" )
	PORT_DIPSETTING(    0x20, "Hard" )
	PORT_DIPSETTING(    0x10, "Hardest" )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START 				/* audio board dips */
	COINAGE
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START				/* start button */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0xfe, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* coin counters */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0xfc, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* fake analog X */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_X, 50, 10, 0, 255 )

	PORT_START				/* fake analog Y */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_Y, 70, 10, 0, 255 )
INPUT_PORTS_END


INPUT_PORTS_START( cheyenne )
	PORT_START				/* player inputs and logic board dips */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x08, "2" )
	PORT_DIPSETTING(    0x0c, "3" )
	PORT_DIPSETTING(    0x04, "4" )
	PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Easy" )
	PORT_DIPSETTING(    0x30, "Normal" )
	PORT_DIPSETTING(    0x20, "Hard" )
	PORT_DIPSETTING(    0x10, "Hardest" )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START 				/* audio board dips */
	COINAGE
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START				/* start button */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0xfe, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* coin counters */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0xfc, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* fake analog X */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_X, 50, 10, 0, 255 )

	PORT_START				/* fake analog Y */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_Y, 70, 10, 0, 255 )
INPUT_PORTS_END


INPUT_PORTS_START( combat )
	PORT_START				/* player inputs and logic board dips */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x08, "2" )
	PORT_DIPSETTING(    0x0c, "3" )
	PORT_DIPSETTING(    0x04, "4" )
	PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Easy" )
	PORT_DIPSETTING(    0x30, "Normal" )
	PORT_DIPSETTING(    0x20, "Hard" )
	PORT_DIPSETTING(    0x10, "Hardest" )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START 				/* audio board dips */
	COINAGE
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START				/* start button */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0xfe, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* coin counters */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0xfc, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* fake analog X */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_X, 50, 10, 0, 255 )

	PORT_START				/* fake analog Y */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_Y, 70, 10, 0, 255 )
INPUT_PORTS_END


INPUT_PORTS_START( cracksht )
	PORT_START				/* player inputs and logic board dips */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_DIPNAME( 0x0c, 0x0c, "Seconds" )
	PORT_DIPSETTING(    0x08, "20" )
	PORT_DIPSETTING(    0x0c, "30" )
	PORT_DIPSETTING(    0x04, "40" )
	PORT_DIPSETTING(    0x00, "50" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Easy" )
	PORT_DIPSETTING(    0x30, "Normal" )
	PORT_DIPSETTING(    0x20, "Hard" )
	PORT_DIPSETTING(    0x10, "Hardest" )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START 				/* audio board dips */
	COINAGE
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START				/* start button */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0xfe, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* coin counters */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0xfc, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* fake analog X */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_X, 50, 10, 0, 255 )

	PORT_START				/* fake analog Y */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_Y, 70, 10, 0, 255 )
INPUT_PORTS_END


INPUT_PORTS_START( claypign )
	PORT_START				/* player inputs and logic board dips */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x0c, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Easy" )
	PORT_DIPSETTING(    0x30, "Normal" )
	PORT_DIPSETTING(    0x20, "Hard" )
	PORT_DIPSETTING(    0x10, "Hardest" )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START 				/* audio board dips */
	COINAGE
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START				/* start button */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0xfe, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* coin counters */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0xfc, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* fake analog X */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_X, 50, 10, 0, 255 )

	PORT_START				/* fake analog Y */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_Y, 70, 10, 0, 255 )
INPUT_PORTS_END


INPUT_PORTS_START( chiller )
	PORT_START				/* player inputs and logic board dips */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_DIPNAME( 0x0c, 0x0c, "Seconds" )
	PORT_DIPSETTING(    0x08, "30" )
	PORT_DIPSETTING(    0x0c, "45" )
	PORT_DIPSETTING(    0x04, "60" )
	PORT_DIPSETTING(    0x00, "70" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Easy" )
	PORT_DIPSETTING(    0x30, "Normal" )
	PORT_DIPSETTING(    0x20, "Hard" )
	PORT_DIPSETTING(    0x10, "Hardest" )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START 				/* audio board dips */
	COINAGE
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* start button */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0xfe, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* coin counters */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0xfc, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* fake analog X */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_X, 50, 10, 0, 255 )

	PORT_START				/* fake analog Y */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_Y, 70, 10, 0, 255 )
INPUT_PORTS_END


INPUT_PORTS_START( topsecex )
	PORT_START				/* player inputs and logic board dips */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x08, "3" )
	PORT_DIPSETTING(    0x0c, "4" )
	PORT_DIPSETTING(    0x04, "5" )
	PORT_DIPSETTING(    0x00, "6" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Easy" )
	PORT_DIPSETTING(    0x30, "Normal" )
	PORT_DIPSETTING(    0x20, "Hard" )
	PORT_DIPSETTING(    0x10, "Hardest" )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START 				/* audio board dips */
	COINAGE
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* start button */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0xfe, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* coin counters */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0xfc, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* fake analog X */
	PORT_ANALOG( 0xff, 0x00, IPT_TRACKBALL_X | IPF_REVERSE, 50, 10, -127, 127 )

	PORT_START				/* fake analog Y */
	PORT_BIT(  0xff, IP_ACTIVE_LOW, IPT_BUTTON1 )

	PORT_START				/* start button */
	PORT_BITX( 0x01, IP_ACTIVE_LOW, IPT_BUTTON3, "Fireball", IP_KEY_DEFAULT, IP_JOY_DEFAULT )
	PORT_BITX( 0x02, IP_ACTIVE_LOW, IPT_BUTTON4, "Laser", IP_KEY_DEFAULT, IP_JOY_DEFAULT )
	PORT_BITX( 0x04, IP_ACTIVE_LOW, IPT_BUTTON5, "Missile", IP_KEY_DEFAULT, IP_JOY_DEFAULT )
	PORT_BITX( 0x08, IP_ACTIVE_LOW, IPT_BUTTON6, "Oil", IP_KEY_DEFAULT, IP_JOY_DEFAULT )
	PORT_BITX( 0x10, IP_ACTIVE_LOW, IPT_BUTTON7, "Turbo", IP_KEY_DEFAULT, IP_JOY_DEFAULT )
	PORT_BITX( 0x20, IP_ACTIVE_LOW, IPT_BUTTON8, "Shield", IP_KEY_DEFAULT, IP_JOY_DEFAULT )
	PORT_BIT(  0x40, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BITX( 0x80, IP_ACTIVE_LOW, IPT_START1, "Top Secret", IP_KEY_DEFAULT, IP_JOY_DEFAULT )
INPUT_PORTS_END


INPUT_PORTS_START( hitnmiss )
	PORT_START				/* player inputs and logic board dips */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_DIPNAME( 0x0c, 0x0c, "Seconds" )
	PORT_DIPSETTING(    0x08, "20" )
	PORT_DIPSETTING(    0x0c, "30" )
	PORT_DIPSETTING(    0x04, "40" )
	PORT_DIPSETTING(    0x00, "50" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Easy" )
	PORT_DIPSETTING(    0x30, "Normal" )
	PORT_DIPSETTING(    0x20, "Hard" )
	PORT_DIPSETTING(    0x10, "Hardest" )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START 				/* audio board dips */
	COINAGE
	PORT_BIT( 0x70, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START				/* start button */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0xfe, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* coin counters */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0xfc, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* fake analog X */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_X, 50, 10, 0, 255 )

	PORT_START				/* fake analog Y */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_Y, 70, 10, 0, 255 )
INPUT_PORTS_END


INPUT_PORTS_START( whodunit )
	PORT_START				/* player inputs and logic board dips */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_DIPNAME( 0x0c, 0x0c, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x08, "2" )
	PORT_DIPSETTING(    0x0c, "3" )
	PORT_DIPSETTING(    0x04, "4" )
	PORT_DIPSETTING(    0x00, "5" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Easy" )
	PORT_DIPSETTING(    0x30, "Normal" )
	PORT_DIPSETTING(    0x20, "Hard" )
	PORT_DIPSETTING(    0x10, "Hardest" )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START 				/* audio board dips */
	COINAGE
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* start button */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0xfe, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* coin counters */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0xfc, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* fake analog X */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_X, 50, 10, 0, 255 )

	PORT_START				/* fake analog Y */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_Y, 70, 10, 0, 255 )
INPUT_PORTS_END


INPUT_PORTS_START( showdown )
	PORT_START				/* player inputs and logic board dips */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_DIPNAME( 0x0c, 0x0c, "Hands" )
	PORT_DIPSETTING(    0x08, "1" )
	PORT_DIPSETTING(    0x0c, "2" )
	PORT_DIPSETTING(    0x04, "3" )
	PORT_DIPSETTING(    0x00, "4" )
	PORT_DIPNAME( 0x30, 0x30, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x00, "Easy" )
	PORT_DIPSETTING(    0x30, "Normal" )
	PORT_DIPSETTING(    0x20, "Hard" )
	PORT_DIPSETTING(    0x10, "Hardest" )
	PORT_BIT( 0xc0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START 				/* audio board dips */
	COINAGE
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* start button */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0xfe, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* coin counters */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0xfc, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START				/* fake analog X */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_X, 50, 10, 0, 255 )

	PORT_START				/* fake analog Y */
	PORT_ANALOG( 0xff, 0x80, IPT_AD_STICK_Y, 70, 10, 0, 255 )
INPUT_PORTS_END



/*************************************
 *
 *	Sound definitions
 *
 *************************************/

static struct CustomSound_interface custom_interface =
{
	exidy440_sh_start,
	exidy440_sh_stop,
	exidy440_sh_update
};



/*************************************
 *
 *	Machine driver
 *
 *************************************/

static const struct MachineDriver machine_driver_exidy440 =
{
	/* basic machine hardware */
	{
		{
			CPU_M6809,
			12979200/8,                     /* 12MHz/8 */
			readmem_cpu1,writemem_cpu1,0,0,
			main_interrupt,1
		},
		{
			CPU_M6809 | CPU_AUDIO_CPU,
			12979200/4/4,                   /* 12MHz/4 into XTAL, which is 4x clock */
			readmem_cpu2,writemem_cpu2,0,0,
			exidy440_sound_interrupt,1
		}
	},
	60, DEFAULT_REAL_60HZ_VBLANK_DURATION,	/* frames per second, vblank duration */
	1,
	init_machine,

	/* video hardware */
	320, 240, { 0, 319, 0, 239 },
	0,
	257, 512,
	0,

	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE,
	0,
	exidy440_vh_start,
	exidy440_vh_stop,
	exidy440_vh_screenrefresh,

	/* sound hardware */
	SOUND_SUPPORTS_STEREO,0,0,0,
	{
		{
			SOUND_CUSTOM,
			&custom_interface
		}
	},

	nvram_handler
};



/*************************************
 *
 *	Driver initialization
 *
 *************************************/

#define SET_PARAMS(top,p0,p2,p3,mv,mt,cpr) \
	exidy440_topsecret 		= top;\
	port_0_xor 				= p0;\
	port_2_xor 				= p2;\
	port_3_xor				= p3;\
	mirror_vblank_bit 		= mv;\
	mirror_trigger_bit 		= mt;\
	copy_protection_read 	= cpr

static void init_crossbow(void) { SET_PARAMS(0, 0x00, 0x00, 0x00, 0, 0, 0x00); }
static void init_cheyenne(void) { SET_PARAMS(0, 0xff, 0x00, 0x00, 0, 0, 0x00); }
static void init_combat  (void)   { SET_PARAMS(0, 0xff, 0xff, 0x00, 0, 0, 0x00); }
static void init_cracksht(void) { SET_PARAMS(0, 0xff, 0xff, 0x04, 0, 0, 0x00); }
static void init_claypign(void) { SET_PARAMS(0, 0xff, 0xff, 0x04, 0, 0, 0x76); }
static void init_chiller (void)  { SET_PARAMS(0, 0xff, 0xff, 0x04, 0, 0, 0x00); }
static void init_topsecex(void) { SET_PARAMS(1, 0xff, 0xff, 0x04, 0, 0, 0x00); }
static void init_hitnmiss(void) { SET_PARAMS(0, 0xff, 0xff, 0x04, 0, 1, 0x00); }
static void init_whodunit(void) { SET_PARAMS(0, 0xff, 0xff, 0x04, 1, 0, 0x00); }
static void init_showdown(void)
{
	SET_PARAMS(0, 0xff, 0xff, 0x04, 0, 0, 0x00);

	/* set up the fake PLD */
	showdown_bank_triggered = 0;
	install_mem_read_handler(0, 0x4055, 0x4055, showdown_pld_trigger_r);
	install_mem_read_handler(0, 0x40ed, 0x40ed, showdown_pld_select1_r);
	install_mem_read_handler(0, 0x5243, 0x5243, showdown_pld_select2_r);

	/* ensure that we're triggered to bank "1" to start */
	exidy440_bank = 0;
	showdown_pld_trigger_r(0);
	showdown_pld_select1_r(0);
}



/*************************************
 *
 *	ROM definitions
 *
 *************************************/

ROM_START( crossbow )
	ROM_REGION( 0x50000, REGION_CPU1, 0 )     /* 64k for code for the first CPU, plus lots of banked ROMs */
	ROM_LOAD( "xbl-2.1a",   0x08000, 0x2000, 0xbd53ac46 )
	ROM_LOAD( "xbl-2.3a",   0x0a000, 0x2000, 0x703e1376 )
	ROM_LOAD( "xbl-2.4a",   0x0c000, 0x2000, 0x52c5daa1 )
	ROM_LOAD( "xbl-2.6a",   0x0e000, 0x2000, 0xf42a68f7 )
	ROM_LOAD( "xbl-1.1e",   0x10000, 0x2000, 0x2834258e )
	ROM_LOAD( "xbl-1.3e",   0x12000, 0x2000, 0x587b186c )
	ROM_LOAD( "xbl-1.4e",   0x14000, 0x2000, 0x23fbfa8e )
	ROM_LOAD( "xbl-1.6e",   0x16000, 0x2000, 0xa3ebcc92 )
	ROM_LOAD( "xbl-1.7e",   0x18000, 0x2000, 0x945b3a68 )
	ROM_LOAD( "xbl-1.8e",   0x1a000, 0x2000, 0x0d1c5d24 )
	ROM_LOAD( "xbl-1.10e",  0x1c000, 0x2000, 0xca30788b )
	ROM_LOAD( "xbl-1.11e",  0x1e000, 0x2000, 0x6661c5ee )
	ROM_LOAD( "xbl-1.1d",   0x20000, 0x2000, 0xa1416191 )
	ROM_LOAD( "xbl-1.3d",   0x22000, 0x2000, 0x7322b5e1 )
	ROM_LOAD( "xbl-1.4d",   0x24000, 0x2000, 0x425d51ef )
	ROM_LOAD( "xbl-1.6d",   0x26000, 0x2000, 0xc923c9f5 )
	ROM_LOAD( "xbl-1.7d",   0x28000, 0x2000, 0x46cdf117 )
	ROM_LOAD( "xbl-1.8d",   0x2a000, 0x2000, 0x62bad9b6 )
	ROM_LOAD( "xbl-1.10d",  0x2c000, 0x2000, 0xd4aaa382 )
	ROM_LOAD( "xbl-1.11d",  0x2e000, 0x2000, 0xefc77790 )
	ROM_LOAD( "xbl-1.1c",   0x30000, 0x2000, 0xdbbd35cb )
	ROM_LOAD( "xbl-1.3c",   0x32000, 0x2000, 0xf011f98d )
	ROM_LOAD( "xbl-1.4c",   0x34000, 0x2000, 0x1257b947 )
	ROM_LOAD( "xbl-1.6c",   0x36000, 0x2000, 0x48da9081 )
	ROM_LOAD( "xbl-1.7c",   0x38000, 0x2000, 0x8d4d4855 )
	ROM_LOAD( "xbl-1.8c",   0x3a000, 0x2000, 0x4c52b85a )
	ROM_LOAD( "xbl-1.10c",  0x3c000, 0x2000, 0x5986130b )
	ROM_LOAD( "xbl-1.11c",  0x3e000, 0x2000, 0x163a6ae0 )
	ROM_LOAD( "xbl-1.1b",   0x40000, 0x2000, 0x36ea0269 )
	ROM_LOAD( "xbl-1.3b",   0x42000, 0x2000, 0x4a03c2c9 )
	ROM_LOAD( "xbl-1.4b",   0x44000, 0x2000, 0x7e21c624 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "xba-11.1h",  0x0e000, 0x2000, 0x1b61d0c1 )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "xba-1.2k",   0x00000, 0x2000, 0xb6e57685 )
	ROM_LOAD( "xba-1.2l",   0x02000, 0x2000, 0x2c24cb35 )
	ROM_LOAD( "xba-1.2m",   0x04000, 0x2000, 0xf3a4f2be )
	ROM_LOAD( "xba-1.2n",   0x06000, 0x2000, 0x15cf362d )
	ROM_LOAD( "xba-1.2p",   0x08000, 0x2000, 0x56f53af9 )
	ROM_LOAD( "xba-1.2r",   0x0a000, 0x2000, 0x3d8277b0 )
	ROM_LOAD( "xba-1.2s",   0x0c000, 0x2000, 0x14dd8993 )
	ROM_LOAD( "xba-1.2t",   0x0e000, 0x2000, 0xdfa783e4 )
	ROM_LOAD( "xba-1.1k",   0x10000, 0x2000, 0x4f01f9e6 )
	ROM_LOAD( "xba-1.1l",   0x12000, 0x2000, 0xfb119acf )
	ROM_LOAD( "xba-1.1m",   0x14000, 0x2000, 0x18d097ac )
	ROM_LOAD( "xba-1.1n",   0x16000, 0x2000, 0x2e855698 )
	ROM_LOAD( "xba-1.1p",   0x18000, 0x2000, 0x788bfac6 )
	ROM_LOAD( "xba-1.1r",   0x1a000, 0x2000, 0xb8ec43b3 )
	ROM_LOAD( "xba-1.1s",   0x1c000, 0x2000, 0xc9ead134 )
	ROM_LOAD( "xba-1.1t",   0x1e000, 0x2000, 0x5f41c282 )
ROM_END


ROM_START( cheyenne )
	ROM_REGION( 0x50000, REGION_CPU1, 0 )     /* 64k for code for the first CPU, plus lots of banked ROMs */
	ROM_LOAD( "cyl-1.1a",   0x08000, 0x2000, 0x504c3fa6 )
	ROM_LOAD( "cyl-1.3a",   0x0a000, 0x2000, 0x09b7903b )
	ROM_LOAD( "cyl-1.4a",   0x0c000, 0x2000, 0xb708646b )
	ROM_LOAD( "cyl-1.6a",   0x0e000, 0x2000, 0x5d1e708d )
	ROM_LOAD( "cyl-1.1e",   0x10000, 0x2000, 0x8778e317 )
	ROM_LOAD( "cyl-1.3e",   0x12000, 0x2000, 0xc8a9ca1b )
	ROM_LOAD( "cyl-1.4e",   0x14000, 0x2000, 0x86c4125a )
	ROM_LOAD( "cyl-1.6e",   0x16000, 0x2000, 0x51f4f060 )
	ROM_LOAD( "cyl-1.7e",   0x18000, 0x2000, 0x4924d0c1 )
	ROM_LOAD( "cyl-1.8e",   0x1a000, 0x2000, 0x5c7c4dd7 )
	ROM_LOAD( "cyl-1.10e",  0x1c000, 0x2000, 0x57232888 )
	ROM_LOAD( "cyl-1.11e",  0x1e000, 0x2000, 0x2a767252 )
	ROM_LOAD( "cyl-1.1d",   0x20000, 0x2000, 0xcd590e99 )
	ROM_LOAD( "cyl-1.3d",   0x22000, 0x2000, 0x1fddccdb )
	ROM_LOAD( "cyl-1.4d",   0x24000, 0x2000, 0x6c5ee6d7 )
	ROM_LOAD( "cyl-1.6d",   0x26000, 0x2000, 0x0e7c16c2 )
	ROM_LOAD( "cyl-1.7d",   0x28000, 0x2000, 0xabe11728 )
	ROM_LOAD( "cyl-1.8d",   0x2a000, 0x2000, 0x95bb9a72 )
	ROM_LOAD( "cyl-1.10d",  0x2c000, 0x2000, 0x5bc251be )
	ROM_LOAD( "cyl-1.7c",   0x38000, 0x2000, 0xe9f6ce96 )
	ROM_LOAD( "cyl-1.8c",   0x3a000, 0x2000, 0xcb3f8e9e )
	ROM_LOAD( "cyl-1.10c",  0x3c000, 0x2000, 0x49f90633 )
	ROM_LOAD( "cyl-1.11c",  0x3e000, 0x2000, 0x70b69cf1 )
	ROM_LOAD( "cyl-1.1b",   0x40000, 0x2000, 0xc372e018 )
	ROM_LOAD( "cyl-1.3b",   0x42000, 0x2000, 0x6a583feb )
	ROM_LOAD( "cyl-1.4b",   0x44000, 0x2000, 0x670e127d )
	ROM_LOAD( "cyl-1.6b",   0x46000, 0x2000, 0xed245268 )
	ROM_LOAD( "cyl-1.7b",   0x48000, 0x2000, 0xdcc56d6c )
	ROM_LOAD( "cyl-1.8b",   0x4a000, 0x2000, 0xc0653d3e )
	ROM_LOAD( "cyl-1.10b",  0x4c000, 0x2000, 0x7fc67d19 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "cya-1.1h",   0x0e000, 0x2000, 0x5aed3d8c )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "cya-1.2k",   0x00000, 0x2000, 0xdc2b716d )
	ROM_LOAD( "cya-1.2l",   0x02000, 0x2000, 0x091ad047 )
	ROM_LOAD( "cya-1.2m",   0x04000, 0x2000, 0x59085362 )
	ROM_LOAD( "cya-1.2n",   0x06000, 0x2000, 0x9c2e23c7 )
	ROM_LOAD( "cya-1.2p",   0x08000, 0x2000, 0xeff18766 )
	ROM_LOAD( "cya-1.2r",   0x0a000, 0x2000, 0x8e730c98 )
	ROM_LOAD( "cya-1.2s",   0x0c000, 0x2000, 0x46515454 )
	ROM_LOAD( "cya-1.2t",   0x0e000, 0x2000, 0x5868fa84 )
	ROM_LOAD( "cya-1.1k",   0x10000, 0x2000, 0x45a306a6 )
	ROM_LOAD( "cya-1.1l",   0x12000, 0x2000, 0x3c7e2127 )
	ROM_LOAD( "cya-1.1m",   0x14000, 0x2000, 0x39ddc9f7 )
	ROM_LOAD( "cya-1.1n",   0x16000, 0x2000, 0x5fcee4fd )
	ROM_LOAD( "cya-1.1p",   0x18000, 0x2000, 0x81a4a876 )
	ROM_LOAD( "cya-1.1r",   0x1a000, 0x2000, 0xdfd84e73 )
ROM_END


ROM_START( combat )
	ROM_REGION( 0x50000, REGION_CPU1, 0 )     /* 64k for code for the first CPU, plus lots of banked ROMs */
	ROM_LOAD( "1a",   0x08000, 0x2000, 0x159a573b )
	ROM_LOAD( "3a",   0x0a000, 0x2000, 0x59ae51a7 )
	ROM_LOAD( "4a",   0x0c000, 0x2000, 0x95a1f3d0 )
	ROM_LOAD( "6a",   0x0e000, 0x2000, 0xaf3fef5f )
	ROM_LOAD( "6d",   0x26000, 0x2000, 0x43d3eb61 )
	ROM_LOAD( "7d",   0x28000, 0x2000, 0xef31659c )
	ROM_LOAD( "8d",   0x2a000, 0x2000, 0xfb29c5cd )
	ROM_LOAD( "10d",  0x2c000, 0x2000, 0x2ca0eaa4 )
	ROM_LOAD( "11d",  0x2e000, 0x2000, 0xcc9f2001 )
	ROM_LOAD( "1c",   0x30000, 0x2000, 0xb7b9c5ad )
	ROM_LOAD( "3c",   0x32000, 0x2000, 0xb700e6ec )
	ROM_LOAD( "4c",   0x34000, 0x2000, 0x89fc2b2d )
	ROM_LOAD( "6c",   0x36000, 0x2000, 0x6a8d0dcf )
	ROM_LOAD( "7c",   0x38000, 0x2000, 0x9df7172d )
	ROM_LOAD( "8c",   0x3a000, 0x2000, 0x63b2e4f3 )
	ROM_LOAD( "10c",  0x3c000, 0x2000, 0x3b430adc )
	ROM_LOAD( "11c",  0x3e000, 0x2000, 0x04301032 )
	ROM_LOAD( "1b",   0x40000, 0x2000, 0x70e25cae )
	ROM_LOAD( "3b",   0x42000, 0x2000, 0xd09d167e )
	ROM_LOAD( "4b",   0x44000, 0x2000, 0xf46aba0d )
	ROM_LOAD( "6b",   0x46000, 0x2000, 0x8eb46f40 )
	ROM_LOAD( "7b",   0x48000, 0x2000, 0x3be9b1bd )
	ROM_LOAD( "8b",   0x4a000, 0x2000, 0xae977f4c )
	ROM_LOAD( "10b",  0x4c000, 0x2000, 0x502da003 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "1h",  0x0e000, 0x2000, 0x8f3dd350 )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "2k", 		 0x00000, 0x2000, 0x1c9df8b5 )
	ROM_LOAD( "2l", 		 0x02000, 0x2000, 0x6b733306 )
	ROM_LOAD( "2m", 		 0x04000, 0x2000, 0xdc074733 )
	ROM_LOAD( "2n", 		 0x06000, 0x2000, 0x7985867f )
	ROM_LOAD( "2p",  		 0x08000, 0x2000, 0x88684dcf )
	ROM_LOAD( "2r", 		 0x0a000, 0x2000, 0x5857321e )
	ROM_LOAD( "2s", 		 0x0c000, 0x2000, 0x371e5235 )
	ROM_LOAD( "2t",  		 0x0e000, 0x2000, 0x7ae65f05 )
	ROM_LOAD( "1k",   	 0x10000, 0x2000, 0xf748ea87 )
	ROM_LOAD( "xba-1.2s", 0x16000, 0x2000, 0x14dd8993 )	/* from Crossbow */
	ROM_LOAD( "xba-1.1n", 0x18000, 0x2000, 0x2e855698 )	/* from Crossbow */
	ROM_LOAD( "xba-1.1p", 0x1a000, 0x2000, 0x788bfac6 )	/* from Crossbow */
	ROM_LOAD( "xba-1.2l", 0x1c000, 0x2000, 0x2c24cb35 )	/* from Crossbow */
	ROM_LOAD( "xba-1.1t", 0x1e000, 0x2000, 0x5f41c282 )	/* from Crossbow */
ROM_END


ROM_START( cracksht )
	ROM_REGION( 0x50000, REGION_CPU1, 0 )     /* 64k for code for the first CPU, plus lots of banked ROMs */
	ROM_LOAD( "csl2.1a",   0x08000, 0x2000, 0x16fd0171 )
	ROM_LOAD( "csl2.3a",   0x0a000, 0x2000, 0x906f3209 )
	ROM_LOAD( "csl2.4a",   0x0c000, 0x2000, 0x9996d2bf )
	ROM_LOAD( "csl2.6a",   0x0e000, 0x2000, 0xc8d6e945 )
	ROM_LOAD( "csl2.11d",  0x2e000, 0x2000, 0xb1173dd3 )
	ROM_LOAD( "csl2.1c",   0x30000, 0x2000, 0xe44975a7 )
	ROM_LOAD( "csl2.3c",   0x32000, 0x2000, 0xa3ab11e9 )
	ROM_LOAD( "csl2.4c",   0x34000, 0x2000, 0x89266302 )
	ROM_LOAD( "csl2.6c",   0x36000, 0x2000, 0xbb0f8d32 )
	ROM_LOAD( "csl2.7c",   0x38000, 0x2000, 0xe203ed0b )
	ROM_LOAD( "csl2.8c",   0x3a000, 0x2000, 0x3e028a62 )
	ROM_LOAD( "csl2.10c",  0x3c000, 0x2000, 0xc5494f9f )
	ROM_LOAD( "csl2.11c",  0x3e000, 0x2000, 0x0159bdcb )
	ROM_LOAD( "csl2.1b",   0x40000, 0x2000, 0x8adf33fc )
	ROM_LOAD( "csl2.3b",   0x42000, 0x2000, 0x7561be69 )
	ROM_LOAD( "csl2.4b",   0x44000, 0x2000, 0x848e3aff )
	ROM_LOAD( "csl2.6b",   0x46000, 0x2000, 0xd0fd87df )
	ROM_LOAD( "csl2.7b",   0x48000, 0x2000, 0x7e0a6a31 )
	ROM_LOAD( "csl2.8b",   0x4a000, 0x2000, 0xaf1c8cb8 )
	ROM_LOAD( "csl2.10b",  0x4c000, 0x2000, 0x8a0d6ad0 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "csa3.1h",   0x0e000, 0x2000, 0x5ba8b4ac )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "csa3.2k",   0x00000, 0x2000, 0x067a4f71 )
	ROM_LOAD( "csa3.2l",   0x02000, 0x2000, 0x5716c59e )
	ROM_LOAD( "csa3.2m",   0x04000, 0x2000, 0xb3ff659b )
	ROM_LOAD( "csa3.2n",   0x06000, 0x2000, 0xa8968342 )
	ROM_LOAD( "csa3.2p",   0x08000, 0x2000, 0x5db225b8 )
	ROM_LOAD( "csa3.2r",   0x0a000, 0x2000, 0xfda2669d )
	ROM_LOAD( "csa3.2s",   0x0c000, 0x2000, 0xe8d2413f )
	ROM_LOAD( "csa3.2t",   0x0e000, 0x2000, 0x841a1855 )
	ROM_LOAD( "csa3.1k",   0x10000, 0x2000, 0x27dda69b )
	ROM_LOAD( "csa3.1l",   0x12000, 0x2000, 0x86eea479 )
	ROM_LOAD( "csa3.1m",   0x14000, 0x2000, 0x2c24cb35 )
	ROM_LOAD( "csa3.1n",   0x16000, 0x2000, 0xf3a4f2be )
	ROM_LOAD( "csa3.1p",   0x18000, 0x2000, 0x14dd8993 )
	ROM_LOAD( "csa3.1r",   0x1a000, 0x2000, 0xdfa783e4 )
	ROM_LOAD( "csa3.1s",   0x1c000, 0x2000, 0x18d097ac )
	ROM_LOAD( "csa3.1t",   0x1e000, 0x2000, 0x5f41c282 )
ROM_END


ROM_START( claypign )
	ROM_REGION( 0x50000, REGION_CPU1, 0 )     /* 64k for code for the first CPU, plus lots of banked ROMs */
	ROM_LOAD( "claypige.1a",   0x08000, 0x2000, 0x446d7004 )
	ROM_LOAD( "claypige.3a",   0x0a000, 0x2000, 0xdf39701b )
	ROM_LOAD( "claypige.4a",   0x0c000, 0x2000, 0xf205afb8 )
	ROM_LOAD( "claypige.6a",   0x0e000, 0x2000, 0x97c36c6c )
	ROM_LOAD( "claypige.10c",  0x3c000, 0x2000, 0x3d2957cd )
	ROM_LOAD( "claypige.11c",  0x3e000, 0x2000, 0xe162a3af )
	ROM_LOAD( "claypige.1b",   0x40000, 0x2000, 0x90f1e534 )
	ROM_LOAD( "claypige.3b",   0x42000, 0x2000, 0x150c5993 )
	ROM_LOAD( "claypige.4b",   0x44000, 0x2000, 0xdabb99fb )
	ROM_LOAD( "claypige.6b",   0x46000, 0x2000, 0xc3b86d26 )
	ROM_LOAD( "claypige.7b",   0x48000, 0x2000, 0x6140b026 )
	ROM_LOAD( "claypige.8b",   0x4a000, 0x2000, 0xd0f9d170 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "claypige.h1",   0x0e000, 0x2000, 0x9eedc68d )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "claypige.k2",   0x00000, 0x2000, 0x0dd93c6c )
	ROM_LOAD( "claypige.l2",   0x02000, 0x2000, 0xe1d67c42 )
	ROM_LOAD( "claypige.m2",   0x04000, 0x2000, 0xb56d8bd5 )
	ROM_LOAD( "claypige.n2",   0x06000, 0x2000, 0x9e381cb5 )
	ROM_LOAD( "xba-1.2l",   	0x08000, 0x2000, 0x2c24cb35 )		/* from Crossbow */
	ROM_LOAD( "xba-1.2k",		0x0a000, 0x2000, 0xb6e57685 )		/* from Crossbow */
	ROM_LOAD( "xba-1.1m",  		0x0c000, 0x2000, 0x18d097ac )		/* from Crossbow */
	ROM_LOAD( "xba-1.1t",   	0x0e000, 0x2000, 0x5f41c282 )		/* from Crossbow */
	ROM_LOAD( "claypige.k1",   0x10000, 0x2000, 0x07f12d18 )
	ROM_LOAD( "claypige.l1",   0x12000, 0x2000, 0xf448eb4f )
	ROM_LOAD( "claypige.m1",   0x14000, 0x2000, 0x36865f5b )
ROM_END


ROM_START( chiller )
	ROM_REGION( 0x50000, REGION_CPU1, 0 )     /* 64k for code for the first CPU, plus lots of banked ROMs */
	ROM_LOAD( "chl3.1a",   0x08000, 0x2000, 0x996ad02e )
	ROM_LOAD( "chl3.3a",   0x0a000, 0x2000, 0x17e6f904 )
	ROM_LOAD( "chl3.4a",   0x0c000, 0x2000, 0xf30d6e32 )
	ROM_LOAD( "chl3.6a",   0x0e000, 0x2000, 0xf64fa8fe )
	ROM_LOAD( "chl3.1d",   0x20000, 0x2000, 0xce4aa4b0 )
	ROM_LOAD( "chl3.3d",   0x22000, 0x2000, 0xa234952e )
	ROM_LOAD( "chl3.4d",   0x24000, 0x2000, 0x645dbae9 )
	ROM_LOAD( "chl3.6d",   0x26000, 0x2000, 0x440a5cd7 )
	ROM_LOAD( "chl3.7d",   0x28000, 0x2000, 0x062a541f )
	ROM_LOAD( "chl3.8d",   0x2a000, 0x2000, 0x31ff8f48 )
	ROM_LOAD( "chl3.10d",  0x2c000, 0x2000, 0x5bceb965 )
	ROM_LOAD( "chl3.11d",  0x2e000, 0x2000, 0xe16b5db3 )
	ROM_LOAD( "chl3.1c",   0x30000, 0x2000, 0xebfd29e8 )
	ROM_LOAD( "chl3.3c",   0x32000, 0x2000, 0xa04261e5 )
	ROM_LOAD( "chl3.4c",   0x34000, 0x2000, 0x6fcbb15b )
	ROM_LOAD( "chl3.6c",   0x36000, 0x2000, 0xbd0e0689 )
	ROM_LOAD( "chl3.7c",   0x38000, 0x2000, 0x2715571e )
	ROM_LOAD( "chl3.8c",   0x3a000, 0x2000, 0x364d9450 )
	ROM_LOAD( "chl3.10c",  0x3c000, 0x2000, 0x13180106 )
	ROM_LOAD( "chl3.11c",  0x3e000, 0x2000, 0x4a7ffe6f )
	ROM_LOAD( "chl3.1b",   0x40000, 0x2000, 0x20c19bb6 )
	ROM_LOAD( "chl3.3b",   0x42000, 0x2000, 0xe1f07ace )
	ROM_LOAD( "chl3.4b",   0x44000, 0x2000, 0x140d95db )
	ROM_LOAD( "chl3.6b",   0x46000, 0x2000, 0xfaaf7cc8 )
	ROM_LOAD( "chl3.7b",   0x48000, 0x2000, 0x5512b7e6 )
	ROM_LOAD( "chl3.8b",   0x4a000, 0x2000, 0x6172b12f )
	ROM_LOAD( "chl3.10b",  0x4c000, 0x2000, 0x5d15342a )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "cha3.1h",   0x0f000, 0x1000, 0xb195cbba )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "cha3.2k",   0x00000, 0x2000, 0x814a1c6e )
	ROM_LOAD( "cha3.2l",   0x02000, 0x2000, 0xb326007f )
	ROM_LOAD( "cha3.2m",   0x04000, 0x2000, 0x11075e8c )
	ROM_LOAD( "cha3.2n",   0x06000, 0x2000, 0x8c3f6184 )
	ROM_LOAD( "cha3.2p",   0x08000, 0x2000, 0x3a8b4d0f )
	ROM_LOAD( "cha3.2r",   0x0a000, 0x2000, 0xfc6c4e00 )
	ROM_LOAD( "cha3.2s",   0x0c000, 0x2000, 0x2440d5f3 )
	ROM_LOAD( "cha3.2t",   0x0e000, 0x2000, 0x9b2ce556 )
	ROM_LOAD( "cha3.1k",   0x10000, 0x2000, 0x27f86fab )
	ROM_LOAD( "cha3.1l",   0x12000, 0x2000, 0x581dfde7 )
	ROM_LOAD( "cha3.1m",   0x14000, 0x2000, 0x36d47696 )
	ROM_LOAD( "cha3.1n",   0x16000, 0x2000, 0xce47bffe )
	ROM_LOAD( "cha3.1p",   0x18000, 0x2000, 0x788bfac6 )
	ROM_LOAD( "cha3.1r",   0x1a000, 0x2000, 0xb8ec43b3 )
	ROM_LOAD( "cha3.1s",   0x1c000, 0x2000, 0x5f41c282 )
	ROM_LOAD( "cha3.1t",   0x1e000, 0x2000, 0x3a3a48af )
ROM_END


ROM_START( topsecex )
	ROM_REGION( 0x50000, REGION_CPU1, 0 )     /* 64k for code for the first CPU, plus lots of banked ROMs */
	ROM_LOAD( "tsl1.a1",   0x08000, 0x2000, 0x30ff2142 )
	ROM_LOAD( "tsl1.a3",   0x0a000, 0x2000, 0x9295e5b7 )
	ROM_LOAD( "tsl1.a4",   0x0c000, 0x2000, 0x402abca4 )
	ROM_LOAD( "tsl1.a6",   0x0e000, 0x2000, 0x66eac7d8 )
	ROM_LOAD( "tsl1.e3",   0x12000, 0x2000, 0xf5b291fc )
	ROM_LOAD( "tsl1.e4",   0x14000, 0x2000, 0xb6c659ae )
	ROM_LOAD( "tsl1.e6",   0x16000, 0x2000, 0xcf457523 )
	ROM_LOAD( "tsl1.e7",   0x18000, 0x2000, 0x5a29304c )
	ROM_LOAD( "tsl1.e8",   0x1a000, 0x2000, 0x0750893b )
	ROM_LOAD( "tsl1.e10",  0x1c000, 0x2000, 0xfb87a723 )
	ROM_LOAD( "tsl1.e11",  0x1e000, 0x2000, 0xecf78fac )
	ROM_LOAD( "tsl1.d1",   0x20000, 0x2000, 0x3a316cbe )
	ROM_LOAD( "tsl1.d3",   0x22000, 0x2000, 0x58408a5f )
	ROM_LOAD( "tsl1.d4",   0x24000, 0x2000, 0xc3c85c13 )
	ROM_LOAD( "tsl1.d6",   0x26000, 0x2000, 0xf26a2864 )
	ROM_LOAD( "tsl1.d7",   0x28000, 0x2000, 0x53195dc6 )
	ROM_LOAD( "tsl1.d8",   0x2a000, 0x2000, 0x4fcfb3c8 )
	ROM_LOAD( "tsl1.d10",  0x2c000, 0x2000, 0x6e20af8d )
	ROM_LOAD( "tsl1.d11",  0x2e000, 0x2000, 0x58c670e7 )
	ROM_LOAD( "tsl1.c1",   0x30000, 0x2000, 0x630521f8 )
	ROM_LOAD( "tsl1.c3",   0x32000, 0x2000, 0xd0d7d908 )
	ROM_LOAD( "tsl1.c4",   0x34000, 0x2000, 0xdc2193c4 )
	ROM_LOAD( "tsl1.c6",   0x36000, 0x2000, 0xde417d5f )
	ROM_LOAD( "tsl1.c7",   0x38000, 0x2000, 0xd75708c3 )
	ROM_LOAD( "tsl1.c8",   0x3a000, 0x2000, 0x69f639fd )
	ROM_LOAD( "tsl1.c10",  0x3c000, 0x2000, 0x5977e312 )
	ROM_LOAD( "tsl1.c11",  0x3e000, 0x2000, 0x07a6a534 )
	ROM_LOAD( "tsl1.b1",   0x40000, 0x2000, 0x771ec128 )
	ROM_LOAD( "tsl1.b3",   0x42000, 0x2000, 0xe57af802 )
	ROM_LOAD( "tsl1.b4",   0x44000, 0x2000, 0x7d63fe09 )
	ROM_LOAD( "tsl1.b6",   0x46000, 0x2000, 0xe6c85d95 )
	ROM_LOAD( "tsl1.b7",   0x48000, 0x2000, 0xbabb7e24 )
	ROM_LOAD( "tsl1.b8",   0x4a000, 0x2000, 0xcc770802 )
	ROM_LOAD( "tsl1.b10",  0x4c000, 0x2000, 0x079d0a1d )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "tsa1.1h",   0x0e000, 0x2000, 0x35a1dd40 )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "tsa1.2k",   0x00000, 0x2000, 0xc0b7c8f9 )
	ROM_LOAD( "tsa1.2l",   0x02000, 0x2000, 0xd46f2f23 )
	ROM_LOAD( "tsa1.2m",   0x04000, 0x2000, 0x04722ee4 )
	ROM_LOAD( "tsa1.2n",   0x06000, 0x2000, 0x266348a2 )
	ROM_LOAD( "tsa1.2p",   0x08000, 0x2000, 0x33491a21 )
	ROM_LOAD( "tsa1.2r",   0x0a000, 0x2000, 0x669fb97a )
	ROM_LOAD( "tsa1.2s",   0x0c000, 0x2000, 0xa9167bc2 )
	ROM_LOAD( "tsa1.2t",   0x0e000, 0x2000, 0x26bcd7ff )
	ROM_LOAD( "tsa1.1k",   0x10000, 0x2000, 0x60e9035d )
	ROM_LOAD( "tsa1.1l",   0x12000, 0x2000, 0x7de3bfa7 )
	ROM_LOAD( "tsa1.1m",   0x14000, 0x2000, 0x77a86aef )
	ROM_LOAD( "tsa1.1n",   0x16000, 0x2000, 0x4ffc26a7 )
	ROM_LOAD( "tsa1.1p",   0x18000, 0x2000, 0xd0c699aa )
	ROM_LOAD( "tsa1.1r",   0x1a000, 0x2000, 0x753f0a5f )
	ROM_LOAD( "tsa1.1s",   0x1c000, 0x2000, 0x745f9340 )
	ROM_LOAD( "tsa1.1t",   0x1e000, 0x2000, 0x0e74b7a6 )
ROM_END


ROM_START( hitnmiss )
	ROM_REGION( 0x50000, REGION_CPU1, 0 )     /* 64k for code for the first CPU, plus lots of banked ROMs */
	ROM_LOAD( "hml3.a1",   0x08000, 0x2000, 0xd79ae18e )
	ROM_LOAD( "hml3.a3",   0x0a000, 0x2000, 0x61baf38b )
	ROM_LOAD( "hml3.a4",   0x0c000, 0x2000, 0x60ca260b )
	ROM_LOAD( "hml3.a6",   0x0e000, 0x2000, 0x073305d8 )
	ROM_LOAD( "hml3.d6",   0x26000, 0x2000, 0x79578952 )
	ROM_LOAD( "hml3.d7",   0x28000, 0x2000, 0x8043b78e )
	ROM_LOAD( "hml3.d8",   0x2a000, 0x2000, 0xa6494e2e )
	ROM_LOAD( "hml3.d10",  0x2c000, 0x2000, 0x0810cc84 )
	ROM_LOAD( "hml3.d11",  0x2e000, 0x2000, 0x9f5c3799 )
	ROM_LOAD( "hml3.c1",   0x30000, 0x2000, 0x6606d5a8 )
	ROM_LOAD( "hml3.c3",   0x32000, 0x2000, 0xf6b12e48 )
	ROM_LOAD( "hml3.c4",   0x34000, 0x2000, 0xe5031d44 )
	ROM_LOAD( "hml3.c6",   0x36000, 0x2000, 0x1b0f2f28 )
	ROM_LOAD( "hml3.c7",   0x38000, 0x2000, 0x44920233 )
	ROM_LOAD( "hml3.c8",   0x3a000, 0x2000, 0x7db26fad )
	ROM_LOAD( "hml3.c10",  0x3c000, 0x2000, 0xb8f99481 )
	ROM_LOAD( "hml3.c11",  0x3e000, 0x2000, 0xc2a0d170 )
	ROM_LOAD( "hml3.b1",   0x40000, 0x2000, 0x945cb27c )
	ROM_LOAD( "hml3.b3",   0x42000, 0x2000, 0x3f022689 )
	ROM_LOAD( "hml3.b4",   0x44000, 0x2000, 0xd63fd250 )
	ROM_LOAD( "hml3.b6",   0x46000, 0x2000, 0xafc89eed )
	ROM_LOAD( "hml3.b7",   0x48000, 0x2000, 0xf3a12a58 )
	ROM_LOAD( "hml3.b8",   0x4a000, 0x2000, 0xe0a5a6aa )
	ROM_LOAD( "hml3.b10",  0x4c000, 0x2000, 0xde65dfdc )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "hma3.1h",  0x0e000, 0x2000, 0xf718da36 )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "hm2.2k",   0x00000, 0x2000, 0xd3583b62 )
	ROM_LOAD( "hm2.2l",   0x02000, 0x2000, 0xc059d51e )
	ROM_LOAD( "hma.2m",   0x04000, 0x2000, 0x09bb8495 )
	ROM_LOAD( "hma.2n",   0x06000, 0x2000, 0xe3d290df )
	ROM_LOAD( "hma.2p",   0x08000, 0x2000, 0xf93d1ac0 )
	ROM_LOAD( "hma.2r",   0x0a000, 0x2000, 0x0f3a090e )
	ROM_LOAD( "hma.2s",   0x0c000, 0x2000, 0xc5d35f84 )
	ROM_LOAD( "hma.2t",   0x0e000, 0x2000, 0x9aa71457 )
	ROM_LOAD( "xba.1n",   0x16000, 0x2000, 0x2e855698 )
	ROM_LOAD( "hma.1p",   0x18000, 0x2000, 0x021d89dd )
	ROM_LOAD( "hma.1r",   0x1a000, 0x2000, 0xe8bb33bc )
	ROM_LOAD( "hma.1s",   0x1c000, 0x2000, 0x65f1aa6e )
	ROM_LOAD( "hma.1t",   0x1e000, 0x2000, 0xeb35dfcc )
ROM_END


ROM_START( hitnmis2 )
	ROM_REGION( 0x50000, REGION_CPU1, 0 )     /* 64k for code for the first CPU, plus lots of banked ROMs */
	ROM_LOAD( "hml2.a1",   0x08000, 0x2000, 0x322f7e83 )
	ROM_LOAD( "hml2.a3",   0x0a000, 0x2000, 0x0e12a721 )
	ROM_LOAD( "hml2.a4",   0x0c000, 0x2000, 0x6cec8ad2 )
	ROM_LOAD( "hml2.a6",   0x0e000, 0x2000, 0x008803ec )
	ROM_LOAD( "hml2.d4",   0x24000, 0x2000, 0x62790789 )
	ROM_LOAD( "hml2.d6",   0x26000, 0x2000, 0x02d2d07e )
	ROM_LOAD( "hml2.d7",   0x28000, 0x2000, 0x0f795f7a )
	ROM_LOAD( "hml2.d8",   0x2a000, 0x2000, 0xfe40c51d )
	ROM_LOAD( "hml2.d10",  0x2c000, 0x2000, 0x9362c90c )
	ROM_LOAD( "hml2.d11",  0x2e000, 0x2000, 0x0f77322f )
	ROM_LOAD( "hml2.c1",   0x30000, 0x2000, 0x8e5803ff )
	ROM_LOAD( "hml2.c3",   0x32000, 0x2000, 0xa7474441 )
	ROM_LOAD( "hml2.c4",   0x34000, 0x2000, 0xc74b9610 )
	ROM_LOAD( "hml2.c6",   0x36000, 0x2000, 0xaca63300 )
	ROM_LOAD( "hml2.c7",   0x38000, 0x2000, 0x1990305e )
	ROM_LOAD( "hml2.c8",   0x3a000, 0x2000, 0xbf08cf05 )
	ROM_LOAD( "hml2.c10",  0x3c000, 0x2000, 0x971323ca )
	ROM_LOAD( "hml2.c11",  0x3e000, 0x2000, 0xdd172feb )
	ROM_LOAD( "hml2.b1",   0x40000, 0x2000, 0xaf1fce57 )
	ROM_LOAD( "hml2.b3",   0x42000, 0x2000, 0x0d16ef47 )
	ROM_LOAD( "hml2.b4",   0x44000, 0x2000, 0xd5a8ff68 )
	ROM_LOAD( "hml2.b6",   0x46000, 0x2000, 0x13f439b1 )
	ROM_LOAD( "hml2.b7",   0x48000, 0x2000, 0x9088c16d )
	ROM_LOAD( "hml2.b8",   0x4a000, 0x2000, 0x9c2db94a )
	ROM_LOAD( "hml2.b10",  0x4c000, 0x2000, 0xf01bd7d4 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "hma2.1h",  0x0e000, 0x2000, 0x9be48f45 )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "hm2.2k",   0x00000, 0x2000, 0xd3583b62 )
	ROM_LOAD( "hm2.2l",   0x02000, 0x2000, 0xc059d51e )
	ROM_LOAD( "hma.2m",   0x04000, 0x2000, 0x09bb8495 )
	ROM_LOAD( "hma.2n",   0x06000, 0x2000, 0xe3d290df )
	ROM_LOAD( "hma.2p",   0x08000, 0x2000, 0xf93d1ac0 )
	ROM_LOAD( "hma.2r",   0x0a000, 0x2000, 0x0f3a090e )
	ROM_LOAD( "hma.2s",   0x0c000, 0x2000, 0xc5d35f84 )
	ROM_LOAD( "hma.2t",   0x0e000, 0x2000, 0x9aa71457 )
	ROM_LOAD( "xba.1n",   0x16000, 0x2000, 0x2e855698 )
	ROM_LOAD( "hma.1p",   0x18000, 0x2000, 0x021d89dd )
	ROM_LOAD( "hma.1r",   0x1a000, 0x2000, 0xe8bb33bc )
	ROM_LOAD( "hma.1s",   0x1c000, 0x2000, 0x65f1aa6e )
	ROM_LOAD( "hma.1t",   0x1e000, 0x2000, 0xeb35dfcc )
ROM_END


ROM_START( whodunit )
	ROM_REGION( 0x50000, REGION_CPU1, 0 )     /* 64k for code for the first CPU, plus lots of banked ROMs */
	ROM_LOAD( "wdl8.1a",   0x08000, 0x2000, 0x50658904 )
	ROM_LOAD( "wdl8.3a",   0x0a000, 0x2000, 0x5d1530f8 )
	ROM_LOAD( "wdl8.4a",   0x0c000, 0x2000, 0x0323d6b8 )
	ROM_LOAD( "wdl8.6a",   0x0e000, 0x2000, 0x771b3fb1 )
	ROM_LOAD( "wdl8.4e",   0x14000, 0x2000, 0x33e44369 )
	ROM_LOAD( "wdl8.6e",   0x16000, 0x2000, 0x64b1d850 )
	ROM_LOAD( "wdl8.7e",   0x18000, 0x2000, 0xaa54cf90 )
	ROM_LOAD( "wdl8.8e",   0x1a000, 0x2000, 0xcbd61200 )
	ROM_LOAD( "wdl8.10e",  0x1c000, 0x2000, 0xf24adde5 )
	ROM_LOAD( "wdl8.11e",  0x1e000, 0x2000, 0xad6fe69e )
	ROM_LOAD( "wdl6.1d",   0x20000, 0x2000, 0x3572fb71 )
	ROM_LOAD( "wdl6.3d",   0x22000, 0x2000, 0x158074f4 )
	ROM_LOAD( "wdl8.4d",   0x24000, 0x2000, 0x601d8bd0 )
	ROM_LOAD( "wdl8.6d",   0x26000, 0x2000, 0xb72e8f63 )
	ROM_LOAD( "wdl6.7d",   0x28000, 0x2000, 0xe3f55a4b )
	ROM_LOAD( "wdl6.8d",   0x2a000, 0x2000, 0x932689c8 )
	ROM_LOAD( "wdl6.10d",  0x2c000, 0x2000, 0x0c4348f2 )
	ROM_LOAD( "wdl8.11d",  0x2e000, 0x2000, 0x92391ffe )
	ROM_LOAD( "wdl8.1c",   0x30000, 0x2000, 0x21c62c90 )
	ROM_LOAD( "wdl8.3c",   0x32000, 0x2000, 0x5a8123be )
	ROM_LOAD( "wdl6.4c",   0x34000, 0x2000, 0x7b58dfac )
	ROM_LOAD( "wdl6.6c",   0x36000, 0x2000, 0x9be0b9a9 )
	ROM_LOAD( "wdl6.7c",   0x38000, 0x2000, 0x3cb3faae )
	ROM_LOAD( "wdl6.8c",   0x3a000, 0x2000, 0x51e95d91 )
	ROM_LOAD( "wdl6.10c",  0x3c000, 0x2000, 0xeab12084 )
	ROM_LOAD( "wdl6.11c",  0x3e000, 0x2000, 0xfe2c532f )
	ROM_LOAD( "wdl6.1b",   0x40000, 0x2000, 0xcdaa5ca0 )
	ROM_LOAD( "wdl6.3b",   0x42000, 0x2000, 0x9bbc8161 )
	ROM_LOAD( "wdl8.4b",   0x44000, 0x2000, 0xeb7dc583 )
	ROM_LOAD( "wdl8.6b",   0x46000, 0x2000, 0x91c049a5 )
	ROM_LOAD( "wdl6.7b",   0x48000, 0x2000, 0xfe0a2d00 )
	ROM_LOAD( "wdl8.8b",   0x4a000, 0x2000, 0x33792758 )
	ROM_LOAD( "wdl6.10b",  0x4c000, 0x2000, 0x2f48cfdb )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "wda8.h1",  0x0e000, 0x2000, 0x0090e5a7 )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "wda6.k2",   0x00000, 0x2000, 0xd4951375 )
	ROM_LOAD( "wda6.l2",   0x02000, 0x2000, 0xbe8dcf07 )
	ROM_LOAD( "wda6.m2",   0x04000, 0x2000, 0xfb389e2d )
	ROM_LOAD( "wda6.n2",   0x06000, 0x2000, 0x3849bc78 )
	ROM_LOAD( "wda6.p2",   0x08000, 0x2000, 0xd0dcea80 )
	ROM_LOAD( "wda6.r2",   0x0a000, 0x2000, 0x748b0930 )
	ROM_LOAD( "wda6.s2",   0x0c000, 0x2000, 0x23d5c5a9 )
	ROM_LOAD( "wda6.t2",   0x0e000, 0x2000, 0xa807536d )
	ROM_LOAD( "wda8.l1",   0x12000, 0x2000, 0x27b856bd )
	ROM_LOAD( "wda8.m1",   0x14000, 0x2000, 0x8e15c601 )
	ROM_LOAD( "xba1.1n",   0x16000, 0x2000, 0x2e855698 )
	ROM_LOAD( "wda6.p1",   0x18000, 0x2000, 0x3ffaaa22 )
	ROM_LOAD( "wda6.r1",   0x1a000, 0x2000, 0x0579a3b8 )
	ROM_LOAD( "wda6.s1",   0x1c000, 0x2000, 0xf55c3c6e )
	ROM_LOAD( "wda6.t1",   0x1e000, 0x2000, 0x38363b52 )
ROM_END


ROM_START( showdown )
	ROM_REGION( 0x50000, REGION_CPU1, 0 )     /* 64k for code for the first CPU, plus lots of banked ROMs */
	ROM_LOAD( "showda1.bin",   0x08000, 0x2000, 0xe4031507 )
	ROM_LOAD( "showd3a.bin",   0x0a000, 0x2000, 0xe7de171e )
	ROM_LOAD( "showd4a.bin",   0x0c000, 0x2000, 0x5c8683c9 )
	ROM_LOAD( "showd6a.bin",   0x0e000, 0x2000, 0x4a408379 )
	ROM_LOAD( "showd11e.bin",  0x1e000, 0x2000, 0x1c6b34e5 )
	ROM_LOAD( "showd1d.bin",   0x20000, 0x2000, 0xdb4c8cf6 )
	ROM_LOAD( "showd3d.bin",   0x22000, 0x2000, 0x24242867 )
	ROM_LOAD( "showd4d.bin",   0x24000, 0x2000, 0x36f247e9 )
	ROM_LOAD( "showd6d.bin",   0x26000, 0x2000, 0xc9b14d8d )
	ROM_LOAD( "showd7d.bin",   0x28000, 0x2000, 0xfd054cd2 )
	ROM_LOAD( "showd8d.bin",   0x2a000, 0x2000, 0x8bf32822 )
	ROM_LOAD( "showd10d.bin",  0x2c000, 0x2000, 0xa2051da2 )
	ROM_LOAD( "showd11d.bin",  0x2e000, 0x2000, 0x0748f345 )
	ROM_LOAD( "showd1c.bin",   0x30000, 0x2000, 0xc016cf73 )
	ROM_LOAD( "showd3c.bin",   0x32000, 0x2000, 0x652503ee )
	ROM_LOAD( "showd4c.bin",   0x34000, 0x2000, 0xb4dab193 )
	ROM_LOAD( "showd6c.bin",   0x36000, 0x2000, 0xa1e6a2b3 )
	ROM_LOAD( "showd7c.bin",   0x38000, 0x2000, 0xbc1bea93 )
	ROM_LOAD( "showd8c.bin",   0x3a000, 0x2000, 0x337dd7fa )
	ROM_LOAD( "showd10c.bin",  0x3c000, 0x2000, 0x3ad32d71 )
	ROM_LOAD( "showd11c.bin",  0x3e000, 0x2000, 0x5fe91932 )
	ROM_LOAD( "showd1b.bin",   0x40000, 0x2000, 0x54ff987e )
	ROM_LOAD( "showd3b.bin",   0x42000, 0x2000, 0xe302e915 )
	ROM_LOAD( "showd4b.bin",   0x44000, 0x2000, 0x1b981516 )
	ROM_LOAD( "showd6b.bin",   0x46000, 0x2000, 0x4ee00996 )
	ROM_LOAD( "showd7b.bin",   0x48000, 0x2000, 0x018b7c00 )
	ROM_LOAD( "showd8b.bin",   0x4a000, 0x2000, 0x024fe6ee )
	ROM_LOAD( "showd10b.bin",  0x4c000, 0x2000, 0x0b318dfe )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )
	ROM_LOAD( "showd1h.bin",  0x0e000, 0x2000, 0x6a10ff47 )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 )
	ROM_LOAD( "showd2k.bin",   0x00000, 0x2000, 0x67a86f7f )
	ROM_LOAD( "showd2l.bin",   0x02000, 0x2000, 0x0bb8874b )
	ROM_LOAD( "showd2m.bin",   0x04000, 0x2000, 0x8b77eac8 )
	ROM_LOAD( "showd2n.bin",   0x06000, 0x2000, 0x78e6eed6 )
	ROM_LOAD( "showd2p.bin",   0x08000, 0x2000, 0x03a13435 )
	ROM_LOAD( "showd2r.bin",   0x0a000, 0x2000, 0x1b6b7eac )
	ROM_LOAD( "showd2s.bin",   0x0c000, 0x2000, 0xb88aeb82 )
	ROM_LOAD( "showd2t.bin",   0x0e000, 0x2000, 0x5c801f4d )
	ROM_LOAD( "showd1k.bin",   0x10000, 0x2000, 0x4e1f4f15 )
	ROM_LOAD( "showd1l.bin",   0x12000, 0x2000, 0x6779a745 )
	ROM_LOAD( "showd1m.bin",   0x14000, 0x2000, 0x9cebd8ea )
	ROM_LOAD( "showd1n.bin",   0x16000, 0x2000, 0x689d8a3f )
	ROM_LOAD( "showd1p.bin",   0x18000, 0x2000, 0x862b350d )
	ROM_LOAD( "showd1r.bin",   0x1a000, 0x2000, 0x95b099ed )
	ROM_LOAD( "showd1s.bin",   0x1c000, 0x2000, 0x8f230881 )
	ROM_LOAD( "showd1t.bin",   0x1e000, 0x2000, 0x70e724c7 )
ROM_END



/*************************************
 *
 *	Game drivers
 *
 *************************************/

GAME( 1983, crossbow, 0,        exidy440, crossbow, crossbow, ROT0, "Exidy", "Crossbow (version 2.0)" )
GAME( 1984, cheyenne, 0,        exidy440, cheyenne, cheyenne, ROT0, "Exidy", "Cheyenne (version 1.0)" )
GAME( 1985, combat,   0,        exidy440, combat,   combat,   ROT0, "Exidy", "Combat (version 3.0)" )
GAME( 1985, cracksht, 0,        exidy440, cracksht, cracksht, ROT0, "Exidy", "Crackshot (version 2.0)" )
GAME( 1986, claypign, 0,        exidy440, claypign, claypign, ROT0, "Exidy", "Clay Pigeon (version 2.0)" )
GAME( 1986, chiller,  0,        exidy440, chiller,  chiller,  ROT0, "Exidy", "Chiller (version 3.0)" )
GAME( 1986, topsecex, 0,        exidy440, topsecex, topsecex, ROT0, "Exidy", "Top Secret (Exidy) (version 1.0)" )
GAME( 1987, hitnmiss, 0,        exidy440, hitnmiss, hitnmiss, ROT0, "Exidy", "Hit 'n Miss (version 3.0)" )
GAME( 1987, hitnmis2, hitnmiss, exidy440, hitnmiss, hitnmiss, ROT0, "Exidy", "Hit 'n Miss (version 2.0)" )
GAME( 1988, whodunit, 0,        exidy440, whodunit, whodunit, ROT0, "Exidy", "Who Dunit (version 8.0)" )
GAME( 1988, showdown, 0,        exidy440, showdown, showdown, ROT0, "Exidy", "Showdown (version 5.0)" )
