/*****************************************************************************
 *
 *	 i8x41.c
 *	 Portable UPI-41/8041/8741/8042/8742 emulator V0.1
 *
 *	 Copyright (c) 1999 Juergen Buchmueller, all rights reserved.
 *
 *	 - This source code is released as freeware for non-commercial purposes.
 *	 - You are free to use and redistribute this code in modified or
 *	   unmodified form, provided you list me in the credits.
 *	 - If you modify this source code, you must add a notice to each modified
 *	   source file that it has been changed.  If you're a nice person, you
 *	   will clearly mark each change too.  :)
 *	 - If you wish to use this for commercial purposes, please contact me at
 *	   pullmoll@t-online.de
 *	 - The author of this copywritten work reserves the right to change the
 *	   terms of its usage and license at any time, including retroactively
 *	 - This entire notice must remain in the source code.
 *
 *	This work is solely based on the
 *	'Intel(tm) UPI(tm)-41AH/42AH Users Manual'
 *
 *****************************************************************************/

#include <stdio.h>
#include "driver.h"
#include "state.h"
#include "mamedbg.h"
#include "i8x41.h"

typedef struct {
	UINT16	ppc;
	UINT16	pc;
	UINT16	timer;
	UINT16	subtype;
	UINT8	a;
	UINT8	psw;
	UINT8	state;
	UINT8	tovf;
	UINT8	enable;
	UINT8	dbbi;
	UINT8	dbbo;
	UINT8	*ram;
	int 	(*irq_callback)(int irqline);
}	I8X41;

int i8x41_ICount;

static I8X41 i8x41;

/* Layout of the registers in the debugger */
static UINT8 i8x41_reg_layout[] = {
	I8X41_PC, I8X41_SP, I8X41_PSW, I8X41_A, I8X41_T, I8X41_DATA, I8X41_CMND, -1,
	I8X41_R0, I8X41_R1, I8X41_R2, I8X41_R3, I8X41_R4, I8X41_R5, I8X41_R6, I8X41_R7, 0
};

/* Layout of the debugger windows x,y,w,h */
static UINT8 i8x41_win_layout[] = {
	 0, 0,80, 2,	/* register window (top rows) */
	 0, 3,24,19,	/* disassembler window (left colums) */
	25, 3,55, 9,	/* memory #1 window (right, upper middle) */
	25,13,55, 9,	/* memory #2 window (right, lower middle) */
	 0,23,80, 1,	/* command line window (bottom rows) */
};

#define RM(a)	cpu_readmem16(a)
#define WM(a,v) cpu_writemem16(a,v)
#define ROP(pc) cpu_readop(pc)
#define ROP_ARG(pc) cpu_readop_arg(pc)

/* PC vectors */
#define V_RESET 0x000	/* power on address */
#define V_IBF	0x003	/* input buffer full interrupt vector */
#define V_TIMER 0x007	/* timer/counter interrupt vector */

/*
 * Memory locations
 * Note:
 * 000-3ff		internal ROM for 8x41 (1K)
 * 400-7ff		(more) internal for 8x42 type (2K)
 * 800-8ff		internal RAM
 */
#define M_IRAM	0x800	/* internal RAM is mapped here */
#define M_BANK0 0x800	/* register bank 0 (8 times 8 bits) */
#define M_STACK 0x808	/* stack (8 times 16 bits) */
#define M_BANK1 0x818	/* register bank 1 (8 times 8 bits) */
#define M_USER	0x820	/* user memory (224 times 8 bits) */

/* PSW flag bits */
#define FC		0x80	/* carry flag */
#define FA		0x40	/* auxiliary carry flag */
#define F0		0x20	/* flag 0 */
#define F1		0x10	/* flag 1 (also used as bank select!?) */
#define F3		0x08	/* unused */
#define SP		0x07	/* lower three bits are used as stack pointer */

/* STATE flag bits */
#define OBF 	0x01	/* output buffer full */
#define IBF 	0x02	/* input buffer full */
#define TEST0	0x04	/* test0 line */
#define TEST1	0x08	/* test1 line */

/* ENABLE flag bits */
#define IBFI	0x01	/* input buffer full interrupt */
#define TCNTI	0x02	/* timer/counter interrupt */
#define DMA 	0x04	/* DMA mode */
#define FLAGS	0x08	/* FLAGS mode */
#define T		0x10	/* timer */
#define CNT 	0x20	/* counter */

/* shorter names for the I8x41 structure elements */
#define PPC 	i8x41.ppc
#define PC		i8x41.pc
#define A		i8x41.a
#define PSW 	i8x41.psw
#define DBBI	i8x41.dbbi
#define DBBO	i8x41.dbbo
#define R(n)	i8x41.ram[((PSW & F1) ? M_BANK1:M_BANK0)+(n)]
#define STATE	i8x41.state
#define ENABLE	i8x41.enable
#define TIMER	i8x41.timer
#define TOVF	i8x41.tovf

static UINT8 i8x41_cycles[] = {
	1,1,1,2,2,1,1,1,2,2,2,2,2,2,2,2,
	1,1,2,2,2,1,2,1,1,1,1,1,1,1,1,1,
	1,1,1,2,2,1,2,1,1,1,1,1,1,1,1,1,
	1,1,2,1,2,1,2,1,2,2,2,2,2,2,2,2,
	1,1,1,2,2,1,2,1,1,1,1,1,1,1,1,1,
	1,1,2,2,2,1,2,1,1,1,1,1,1,1,1,1,
	1,1,1,1,2,1,1,1,1,1,1,1,1,1,1,1,
	1,1,2,1,2,1,2,1,1,1,1,1,1,1,1,1,
	1,1,1,2,2,1,2,1,2,2,2,2,2,2,2,2,
	1,1,2,2,1,1,2,1,2,2,2,2,2,2,2,2,
	1,1,1,2,2,1,1,1,1,1,1,1,1,1,1,1,
	2,2,2,2,2,1,2,1,2,2,2,2,2,2,2,2,
	1,1,1,1,2,1,2,1,1,1,1,1,1,1,1,1,
	1,1,2,1,2,1,2,1,1,1,1,1,1,1,1,1,
	1,1,1,2,2,1,2,1,1,1,1,1,1,1,1,1,
	1,1,2,1,2,1,2,1,2,2,2,2,2,2,2,2
};

/***********************************
 *	illegal opcodes
 ***********************************/
INLINE void illegal(void)
{
	logerror("i8x41 #%d: illegal opcode at 0x%03x: %02x\n", cpu_getactivecpu(), PC, ROP(PC));
}

/***********************************
 *	0110 1rrr *  ADD	 A,Rr
 ***********************************/
INLINE void add_r(int r)
{
	UINT8 res = A + R(r);
	if( res < A ) PSW |= FC;
	if( (res & 0x0f) < (A & 0x0f) ) PSW |= FA;
	A = res;
}

/***********************************
 *	0110 000r
 *	ADD 	A,@Rr
 ***********************************/
INLINE void add_rm(int r)
{
	UINT8 res = A + RM( M_IRAM + (R(r) & 0x3f) );
	if( res < A ) PSW |= FC;
	if( (res & 0x0f) < (A & 0x0f) ) PSW |= FA;
	A = res;
}

/***********************************
 *	0000 0011 7654 3210
 *	ADD 	A,#n
 ***********************************/
INLINE void add_i(void)
{
	UINT8 res = A + ROP_ARG(PC);
	PC++;
	if( res < A ) PSW |= FC;
	if( (res & 0x0f) < (A & 0x0f) ) PSW |= FA;
	A = res;
}

/***********************************
 *	0111 1rrr
 *	ADDC	A,Rr
 ***********************************/
INLINE void addc_r(int r)
{
	UINT8 res = A + R(r) + (PSW >> 7);
	if( res <= A ) PSW |= FC;
	if( (res & 0x0f) <= (A & 0x0f) ) PSW |= FA;
	A = res;
}

/***********************************
 *	0111 000r
 *	ADDC	A,@Rr
 ***********************************/
INLINE void addc_rm(int r)
{
	UINT8 res = A + RM( M_IRAM+ (R(r) & 0x3f) ) + (PSW >> 7);
	if( res <= A ) PSW |= FC;
	if( (res & 0x0f) <= (A & 0x0f) ) PSW |= FA;
	A = res;
}

/***********************************
 *	0001 0011 7654 3210
 *	ADDC	A,#n
 ***********************************/
INLINE void addc_i(void)
{
	UINT8 res = A + ROP_ARG(PC);
	PC++;
	if( res < A ) PSW |= FC;
	if( (res & 0x0f) < (A & 0x0f) ) PSW |= FA;
	A = res;
}

/***********************************
 *	0101 1rrr
 *	ANL 	A,Rr
 ***********************************/
INLINE void anl_r(int r)
{
	A = A & R(r);
}

/***********************************
 *	0101 000r
 *	ANL 	A,@Rr
 ***********************************/
INLINE void anl_rm(int r)
{
	A = A & RM( M_IRAM + (R(r) & 0x3f) );
}

/***********************************
 *	0101 0011 7654 3210
 *	ANL 	A,#n
 ***********************************/
INLINE void anl_i(void)
{
	A = A & ROP_ARG(PC);
	PC++;
}

/***********************************
 *	1001 10pp 7654 3210
 *	ANL 	Pp,#n
 ***********************************/
INLINE void anl_p_i(int p)
{
	UINT8 val = ROP_ARG(PC);
	PC++;
	val = val & cpu_readport16(p);
	cpu_writeport16(p, val);
}

/***********************************
 *	1001 11pp 7654 3210
 *	ANLD	Pp,A
 ***********************************/
INLINE void anld_p_a(int p)
{
	UINT8 val = A & 0x0f;
	val = (val | 0xf0) & cpu_readport16(p);
	cpu_writeport16(p, val);
}

/***********************************
 *	aaa1 0100 7654 3210
 *	CALL	addr
 ***********************************/
INLINE void call_i(int page)
{
	UINT8 adr = ROP_ARG(PC);
	PC++;
	WM( M_STACK + (PSW&SP) * 2 + 0, PC & 0xff);
	WM( M_STACK + (PSW&SP) * 2 + 1, ((PC >> 8) & 0x0f) | (PSW & 0xf0) );
	PSW = (PSW & ~SP) | ((PSW + 1) & SP);
	PC = page | adr;
}

/***********************************
 *	0010 0111
 *	CLR 	A
 ***********************************/
INLINE void clr_a(void)
{
	A = 0;
}

/***********************************
 *	1001 0111
 *	CLR 	C
 ***********************************/
INLINE void clr_c(void)
{
	PSW &= ~FC;
}

/***********************************
 *	1000 0101
 *	CLR 	F0
 ***********************************/
INLINE void clr_f0(void)
{
	PSW &= ~F0;
}

/***********************************
 *	1010 0101
 *	CLR 	F1
 ***********************************/
INLINE void clr_f1(void)
{
	PSW &= ~F1;
}

/***********************************
 *	0011 0111
 *	CPL 	A
 ***********************************/
INLINE void cpl_a(void)
{
	A = ~A;
}

/***********************************
 *	1010 0111
 *	CPL 	C
 ***********************************/
INLINE void cpl_c(void)
{
	PSW ^= FC;
}

/***********************************
 *	1001 0101
 *	CPL 	F0
 ***********************************/
INLINE void cpl_f0(void)
{
	PSW ^= F0;
}

/***********************************
 *	1011 0101
 *	CPL 	F1
 ***********************************/
INLINE void cpl_f1(void)
{
	PSW ^= F1;
}

/***********************************
 *	0101 0111
 *	DA		A
 ***********************************/
INLINE void da_a(void)
{
	UINT8 res = A + ((PSW & FA) || ((A & 0x0f) > 0x09)) ? 0x06 : 0x00;
	if( (PSW & FC) || ((res & 0xf0) > 0x90) )
		res += 0x60;
	if( res < A )
		PSW |= FC;
	else
		PSW &= ~FC;
	A = res;
}

/***********************************
 *	0000 0111
 *	DEC 	A
 ***********************************/
INLINE void dec_a(void)
{
	A -= 1;
}

/***********************************
 *	1100 1rrr
 *	DEC 	Rr
 ***********************************/
INLINE void dec_r(int r)
{
	R(r) -= 1;
}

/***********************************
 *	0001 0101
 *	DIS 	I
 ***********************************/
INLINE void dis_i(void)
{
	ENABLE &= ~IBFI;	/* disable input buffer full interrupt */
}

/***********************************
 *	0011 0101
 *	DIS 	TCNTI
 ***********************************/
INLINE void dis_tcnti(void)
{
	ENABLE &= ~TCNTI;	/* disable timer/counter interrupt */
}

/***********************************
 *	0111 1rrr 7654 3210
 *	DJNZ	Rr,addr
 ***********************************/
INLINE void djnz_r_i(int r)
{
	UINT8 adr = ROP_ARG(PC);
	PC++;
	R(r) -= 1;
	if( R(r) )
		PC = (PC & 0x700) | adr;
}

/***********************************
 *	1110 0101
 *	EN		DMA
 ***********************************/
INLINE void en_dma(void)
{
	ENABLE |= DMA;		/* enable DMA handshake lines */
}

/***********************************
 *	1111 0101
 *	EN		FLAGS
 ***********************************/
INLINE void en_flags(void)
{
	ENABLE |= FLAGS;	/* enable flags handshake lines P24 & P25 */
}

/***********************************
 *	0000 0101
 *	EN		I
 ***********************************/
INLINE void en_i(void)
{
	if (0 == (ENABLE & IBFI))
	{
		ENABLE |= IBFI; 	/* enable input buffer full interrupt */
		if (STATE & IBF)	/* already got data in the buffer? */
			i8x41_set_irq_line(I8X41_INT_IBF, HOLD_LINE);
	}
}

/***********************************
 *	0010 0101
 *	EN		TCNTI
 ***********************************/
INLINE void en_tcnti(void)
{
	ENABLE |= TCNTI;	/* enable timer/counter interrupt */
}

/***********************************
 *	0010 0010
 *	IN		A,DBB
 ***********************************/
INLINE void in_a_dbb(void)
{
	if (i8x41.irq_callback)
		(*i8x41.irq_callback)(I8X41_INT_IBF);	/* clear input buffer full flag */
	STATE &= ~IBF;
	A = DBBI;				/* DBB input buffer */
}

/***********************************
 *	0000 10pp
 *	IN		A,Pp
 ***********************************/
INLINE void in_a_p(int p)
{
	A = cpu_readport16(p);	/* should read port 0/3 be prevented? */
}

/***********************************
 *	0001 0111
 *	INC 	A
 ***********************************/
INLINE void inc_a(void)
{
	A += 1;
}

/***********************************
 *	0001 1rrr
 *	DEC 	Rr
 ***********************************/
INLINE void inc_r(int r)
{
	R(r) += 1;
}

/***********************************
 *	0001 000r
 *	INC  @	Rr
 ***********************************/
INLINE void inc_rm(int r)
{
	UINT8 addr = M_IRAM + (R(r) & 0x3f);
	WM( addr, RM(addr) + 1 );
}

/***********************************
 *	bbb1 0010
 *	JBb 	addr
 ***********************************/
INLINE void jbb_i(int bit)
{
	UINT8 adr = ROP_ARG(PC);
	PC += 1;
	if( A & (1 << bit) )
		PC = (PC & 0x700) | adr;
}

/***********************************
 *	1111 0110
 *	JC		addr
 ***********************************/
INLINE void jc_i(void)
{
	UINT8 adr = ROP_ARG(PC);
	PC += 1;
	if( PSW & FC )
		PC = (PC & 0x700) | adr;
}

/***********************************
 *	1011 0110
 *	JF0 	addr
 ***********************************/
INLINE void jf0_i(void)
{
	UINT8 adr = ROP_ARG(PC);
	PC += 1;
	if( PSW & F0 )
		PC = (PC & 0x700) | adr;
}

/***********************************
 *	0111 0110
 *	JF1 	addr
 ***********************************/
INLINE void jf1_i(void)
{
	UINT8 adr = ROP_ARG(PC);
	PC += 1;
	if( PSW & F1 )
		PC = (PC & 0x700) | adr;
}

/***********************************
 *	aaa0 0100
 *	JMP 	addr
 ***********************************/
INLINE void jmp_i(int page)
{
	/* err.. do we have 10 or 11 PC bits?
	 * CALL is said to use 0aa1 (4 pages)
	 * JMP is said to use aaa0 (8 pages)
	 */
	UINT8 adr = ROP_ARG(PC);
	PC += 1;
	PC = page | adr;
}

/***********************************
 *	1011 0011
 *	JMP  @	A
 ***********************************/
INLINE void jmpp_a(void)
{
	UINT16 adr = (PC & 0x700) | A;
	PC = (PC & 0x700) | RM(adr);
}

/***********************************
 *	1110 0110
 *	JNC 	addr
 ***********************************/
INLINE void jnc_i(void)
{
	UINT8 adr = ROP_ARG(PC);
	PC += 1;
	if( !(PSW & FC) )
		PC = (PC & 0x700) | adr;
}

/***********************************
 *	1101 0110
 *	JNIBF	addr
 ***********************************/
INLINE void jnibf_i(void)
{
	UINT8 adr = ROP_ARG(PC);
	PC += 1;
	if( 0 == (STATE & IBF) )
		PC = (PC & 0x700) | adr;
}

/***********************************
 *	0010 0110
 *	JNT0	addr
 ***********************************/
INLINE void jnt0_i(void)
{
	UINT8 adr = ROP_ARG(PC);
	PC += 1;
	if( !(STATE & TEST0) )
		PC = (PC & 0x700) | adr;
}

/***********************************
 *	0100 0110
 *	JNT1	addr
 ***********************************/
INLINE void jnt1_i(void)
{
	UINT8 adr = ROP_ARG(PC);
	PC += 1;
	if( !(STATE & TEST1) )
		PC = (PC & 0x700) | adr;
}

/***********************************
 *	1001 0110
 *	JNZ 	addr
 ***********************************/
INLINE void jnz_i(void)
{
	UINT8 adr = ROP_ARG(PC);
	PC += 1;
	if( A )
		PC = (PC & 0x700) | adr;
}

/***********************************
 *	1000 0110
 *	JOBF	addr
 ***********************************/
INLINE void jobf_i(void)
{
	UINT8 adr = ROP_ARG(PC);
	PC += 1;
	if( STATE & OBF )
		PC = (PC & 0x700) | adr;
}

/***********************************
 *	0001 0110
 *	JTF 	addr
 ***********************************/
INLINE void jtf_i(void)
{
	UINT8 adr = ROP_ARG(PC);
	PC += 1;
	if( TOVF )
		PC = (PC & 0x700) | adr;
	TOVF = 0;
}

/***********************************
 *	0011 0110
 *	JT0 	addr
 ***********************************/
INLINE void jt0_i(void)
{
	UINT8 adr = ROP_ARG(PC);
	PC += 1;
	if( STATE & TEST0 )
		PC = (PC & 0x700) | adr;
}

/***********************************
 *	0101 0110
 *	JT1 	addr
 ***********************************/
INLINE void jt1_i(void)
{
	UINT8 adr = ROP_ARG(PC);
	PC += 1;
	if( STATE & TEST1 )
		PC = (PC & 0x700) | adr;
}

/***********************************
 *	1100 0110
 *	JZ		addr
 ***********************************/
INLINE void jz_i(void)
{
	UINT8 adr = ROP_ARG(PC);
	PC += 1;
	if( !A )
		PC = (PC & 0x700) | adr;
}

/***********************************
 *	0010 0011
 *	MOV 	A,#n
 ***********************************/
INLINE void mov_a_i(void)
{
	A = ROP(PC);
	PC += 1;
}

/***********************************
 *	1100 0111
 *	MOV 	A,PSW
 ***********************************/
INLINE void mov_a_psw(void)
{
	A = PSW;
}

/***********************************
 *	1111 1rrr
 *	MOV 	A,Rr
 ***********************************/
INLINE void mov_a_r(int r)
{
	A = R(r);
}

/***********************************
 *	1111 000r
 *	MOV 	A,Rr
 ***********************************/
INLINE void mov_a_rm(int r)
{
	A = RM( M_IRAM + (R(r) & 0x3f) );
}

/***********************************
 *	0100 0010
 *	MOV 	A,T
 ***********************************/
INLINE void mov_a_t(void)
{
	A = (UINT8) (TIMER / 32);
}

/***********************************
 *	1101 0111
 *	MOV 	PSW,A
 ***********************************/
INLINE void mov_psw_a(void)
{
	PSW = A;
}

/***********************************
 *	1010 1rrr
 *	MOV 	Rr,A
 ***********************************/
INLINE void mov_r_a(int r)
{
	R(r) = A;
}

/***********************************
 *	1011 1rrr
 *	MOV 	Rr,#n
 ***********************************/
INLINE void mov_r_i(int r)
{
	UINT8 val = ROP_ARG(PC);
	PC += 1;
	R(r) = val;
}

/***********************************
 *	1010 000r
 *	MOV 	@Rr,A
 ***********************************/
INLINE void mov_rm_a(int r)
{
	WM( M_IRAM + (R(r) & 0x3f), A );
}

/***********************************
 *	1011 000r
 *	MOV 	@Rr,#n
 ***********************************/
INLINE void mov_rm_i(int r)
{
	UINT8 val = ROP_ARG(PC);
	PC += 1;
	WM( M_IRAM + (R(r) & 0x3f), val );
}

/***********************************
 *	1001 0000
 *	MOV 	STS,A
 ***********************************/
INLINE void mov_sts_a(void)
{
	STATE = (STATE & 0x0f) | (A & 0xf0);
}

/***********************************
 *	0110 0010
 *	MOV 	T,A
 ***********************************/
INLINE void mov_t_a(void)
{
	TIMER = A * 32;
}

/***********************************
 *	0000 11pp
 *	MOVD	A,Pp
 ***********************************/
INLINE void movd_a_p(int p)
{
	UINT8 val = cpu_readport16(p);
	A = val & 0x0f;
}

/***********************************
 *	0011 11pp
 *	MOVD	Pp,A
 ***********************************/
INLINE void movd_p_a(int p)
{
	cpu_writeport16(p, A & 0x0f);
}

/***********************************
 *	1010 0011
 *	MOVP	A,@A
 ***********************************/
INLINE void movp_a_am(void)
{
	UINT16 addr = (PC & 0x700) | A;
	A = RM(addr);
}

/***********************************
 *	1110 0011
 *	MOVP3	A,@A
 ***********************************/
INLINE void movp3_a_am(void)
{
	UINT16 addr = 0x300 | A;
	A = RM(addr);
}

/***********************************
 *	0000 0000
 *	NOP
 ***********************************/
INLINE void nop(void)
{
}

/***********************************
 *	0100 1rrr
 *	ORL 	A,Rr
 ***********************************/
INLINE void orl_r(int r)
{
	A = A | R(r);
}

/***********************************
 *	0100 000r
 *	ORL 	A,@Rr
 ***********************************/
INLINE void orl_rm(int r)
{
	A = A | RM( M_IRAM + (R(r) & 0x3f) );
}

/***********************************
 *	0100 0011 7654 3210
 *	ORL 	A,#n
 ***********************************/
INLINE void orl_i(void)
{
	UINT8 val = ROP_ARG(PC);
	PC++;
	A = A | val;
}

/***********************************
 *	1000 10pp 7654 3210
 *	ORL 	Pp,#n
 ***********************************/
INLINE void orl_p_i(int p)
{
	UINT8 val = ROP_ARG(PC);
	PC++;
	val = val | cpu_readport16(p);
	cpu_writeport16(p, val);
}

/***********************************
 *	1000 11pp 7654 3210
 *	ORLD	Pp,A
 ***********************************/
INLINE void orld_p_a(int p)
{
	UINT8 val = A & 0x0f;
	val = val | cpu_readport16(p);
	cpu_writeport16(p, val);
}

/***********************************
 *	0000 0010
 *	OUT 	DBB,A
 ***********************************/
INLINE void out_dbb_a(void)
{
	DBBO = A;			/* DBB output buffer */
	STATE |= OBF;		/* assert the output buffer full flag */
}

/***********************************
 *	0011 10pp
 *	OUT 	Pp,A
 ***********************************/
INLINE void out_p_a(int p)
{
	cpu_writeport16(p, A);
}

/***********************************
 *	1000 0011
 *	RET
 ***********************************/
INLINE void ret(void)
{
	UINT8 msb;
	PSW = (PSW & ~SP) | ((PSW - 1) & SP);
	msb = RM(M_STACK + (PSW&SP) * 2 + 1);
	PC = RM(M_STACK + (PSW&SP) * 2 + 0);
	PC |= (msb << 8) & 0x700;
}

/***********************************
 *	1001 0011
 *	RETR
 ***********************************/
INLINE void retr(void)
{
	UINT8 msb;
	PSW = (PSW & ~SP) | ((PSW - 1) & SP);
	msb = RM(M_STACK + (PSW&SP) * 2 + 1);
	PC = RM(M_STACK + (PSW&SP) * 2 + 0);
	PC |= (msb << 8) & 0x700;
	PSW = (PSW & 0x0f) | (msb & 0xf0);
}

/***********************************
 *	1110 0111
 *	RL		A
 ***********************************/
INLINE void rl_a(void)
{
	A = (A << 1) | (A >> 7);
}

/***********************************
 *	1111 0111
 *	RLC 	A
 ***********************************/
INLINE void rlc_a(void)
{
	UINT8 c = PSW >> 7;
	PSW = (PSW & ~FC) | (A >> 7);
	A = (A << 1) | c;
}

/***********************************
 *	0111 0111
 *	RR		A
 ***********************************/
INLINE void rr_a(void)
{
	A = (A >> 1) | (A << 7);
}

/***********************************
 *	0110 0111
 *	RRC 	A
 ***********************************/
INLINE void rrc_a(void)
{
	UINT8 c = PSW & 0x80;
	PSW = (PSW & ~FC) | (A << 7);
	A = (A >> 1) | c;
}

/***********************************
 *	1100 0101
 *	SEL 	RB0
 ***********************************/
INLINE void sel_rb0(void)
{
	PSW &= ~F1;
}

/***********************************
 *	1101 0101
 *	SEL 	RB1
 ***********************************/
INLINE void sel_rb1(void)
{
	PSW |= F1;
}

/***********************************
 *	0110 0101
 *	STOP	TCNT
 ***********************************/
INLINE void stop_tcnt(void)
{
	ENABLE &= ~(T|CNT);
}

/***********************************
 *	0100 0101
 *	STRT	CNT
 ***********************************/
INLINE void strt_cnt(void)
{
	ENABLE |= CNT;
}

/***********************************
 *	0101 0101
 *	STRT	T
 ***********************************/
INLINE void strt_t(void)
{
	ENABLE |= T;
}

/***********************************
 *	0100 0111
 *	SWAP	A
 ***********************************/
INLINE void swap_a(void)
{
	A = (A << 4) | (A >> 4);
}

/***********************************
 *	0010 1rrr
 *	XCH 	A,Rr
 ***********************************/
INLINE void xch_a_r(int r)
{
	UINT8 tmp = R(r);
	R(r) = A;
	A = tmp;
}

/***********************************
 *	0010 000r
 *	XCH 	A,@Rr
 ***********************************/
INLINE void xch_a_rm(int r)
{
	UINT8 addr = M_IRAM + (R(r) & 0x3f);
	UINT8 tmp = RM(addr);
	WM( addr, A );
	A = tmp;
}

/***********************************
 *	0011 000r
 *	XCHD	A,@Rr
 ***********************************/
INLINE void xchd_a_rm(int r)
{
	UINT8 addr = M_IRAM + (R(r) & 0x3f);
	UINT8 tmp = RM(addr);
	WM( addr, (tmp & 0xf0) | (A & 0x0f) );
	A = (A & 0xf0) | (tmp & 0x0f);
}

/***********************************
 *	1101 1rrr
 *	XRL 	A,Rr
 ***********************************/
INLINE void xrl_r(int r)
{
	A = A ^ R(r);
}

/***********************************
 *	1101 000r
 *	XRL 	A,@Rr
 ***********************************/
INLINE void xrl_rm(int r)
{
	A = A ^ RM( M_IRAM + (R(r) & 0x3f) );
}

/***********************************
 *	1101 0011 7654 3210
 *	XRL 	A,#n
 ***********************************/
INLINE void xrl_i(void)
{
	UINT8 val = ROP_ARG(PC);
	PC++;
	A = A ^ val;
}

void i8x41_init(void)
{
	int cpu = cpu_getactivecpu();
	state_save_register_UINT16("i8x41", cpu, "PPC",       &i8x41.ppc,    1);
	state_save_register_UINT16("i8x41", cpu, "PC",        &i8x41.pc,     1);
	state_save_register_UINT16("i8x41", cpu, "TIMER",     &i8x41.timer,  1);
	state_save_register_UINT16("i8x41", cpu, "SUBTYPE",   &i8x41.subtype,1);
	state_save_register_UINT8 ("i8x41", cpu, "A",         &i8x41.a,      1);
	state_save_register_UINT8 ("i8x41", cpu, "PSW",       &i8x41.psw,    1);
	state_save_register_UINT8 ("i8x41", cpu, "STATE",     &i8x41.state,  1);
	state_save_register_UINT8 ("i8x41", cpu, "TOVF",      &i8x41.tovf,   1);
	state_save_register_UINT8 ("i8x41", cpu, "ENABLE",    &i8x41.enable, 1);
	state_save_register_UINT8 ("i8x41", cpu, "DBBI",      &i8x41.dbbi,   1);
	state_save_register_UINT8 ("i8x41", cpu, "DBBO",      &i8x41.dbbo,   1);
}

/* Reset registers to the initial values */
void i8x41_reset(void *param)
{
	memset(&i8x41, 0, sizeof(I8X41));
	/* default to 8041 behaviour for DBBI/DBBO and extended commands */
	i8x41.subtype = 8041;
	/* ugly hack.. excuse my lazyness */
	i8x41.ram = memory_region(REGION_CPU1 + cpu_getactivecpu());
	ENABLE = IBFI | TCNTI;
	DBBI = 0xff;
	DBBO = 0xff;
}

/* Shut down CPU core */
void i8x41_exit(void)
{
	/* nothing to do */
}

/* Execute cycles - returns number of cycles actually run */
int i8x41_execute(int cycles)
{
	i8x41_ICount = cycles;

	do
	{
		UINT8 op = cpu_readop(PC);

		PPC = PC;

		CALL_MAME_DEBUG;

		PC += 1;
		i8x41_ICount -= i8x41_cycles[op];

		if( ENABLE & T )
			TIMER += i8x41_cycles[op];

		if( TIMER > 0x1fff )
		{
			TIMER &= 0x1fff;
			TOVF = 1;
			if( ENABLE & TCNTI )
			{
				WM( M_STACK + (PSW&SP) * 2 + 0, PC & 0xff);
				WM( M_STACK + (PSW&SP) * 2 + 1, ((PC >> 8) & 0x0f) | (PSW & 0xf0) );
				PSW = (PSW & ~SP) | ((PSW + 1) & SP);
				PC = V_TIMER;
			}
		}

		switch( op )
		{
		/* opcode cycles bitmask */
		case 0x00: /* 1: 0000 0000 */
			nop();
			break;
		case 0x01: /* 1: 0000 0001 */
			illegal();
			break;
		case 0x02: /* 1: 0000 0010 */
			out_dbb_a();
			break;
		case 0x03: /* 2: 0000 0011 */
			add_i();
			break;
		case 0x04: /* 2: aaa0 0100 */
			jmp_i(0x000);
			break;
		case 0x05: /* 1: 0000 0101 */
			en_i();
			break;
		case 0x06: /* 1: 0000 0110 */
			illegal();
			break;
		case 0x07: /* 1: 0000 0111 */
			dec_a();
			break;
		case 0x08: /* 2: 0000 10pp */
		case 0x09: /* 2: 0000 10pp */
		case 0x0a: /* 2: 0000 10pp */
		case 0x0b: /* 2: 0000 10pp */
			in_a_p(op & 3);
			break;
		case 0x0c: /* 2: 0000 11pp */
		case 0x0d: /* 2: 0000 11pp */
		case 0x0e: /* 2: 0000 11pp */
		case 0x0f: /* 2: 0000 11pp */
			movd_a_p(op & 3);
			break;
		case 0x10: /* 1: 0001 000r */
			inc_rm(0);
			break;
		case 0x11: /* 1: 0001 000r */
			inc_rm(1);
			break;
		case 0x12: /* 2: bbb1 0010 */
			jbb_i(0);
			break;
		case 0x13: /* 2: 0001 0011 */
			addc_i();
			break;
		case 0x14: /* 2: aaa1 0100 */
			call_i(0x000);
			break;
		case 0x15: /* 1: 0001 0101 */
			dis_i();
			break;
		case 0x16: /* 2: 0001 0110 */
			jtf_i();
			break;
		case 0x17: /* 1: 0001 0111 */
			inc_a();
			break;
		case 0x18: /* 1: 0001 1rrr */
		case 0x19: /* 1: 0001 1rrr */
		case 0x1a: /* 1: 0001 1rrr */
		case 0x1b: /* 1: 0001 1rrr */
		case 0x1c: /* 1: 0001 1rrr */
		case 0x1d: /* 1: 0001 1rrr */
		case 0x1e: /* 1: 0001 1rrr */
		case 0x1f: /* 1: 0001 1rrr */
			inc_r(op & 7);
			break;
		case 0x20: /* 1: 0010 000r */
			xch_a_rm(0);
			break;
		case 0x21: /* 1: 0010 000r */
			xch_a_rm(1);
			break;
		case 0x22: /* 1: 0010 0010 */
			in_a_dbb();
			break;
		case 0x23: /* 2: 0010 0011 */
			mov_a_i();
			break;
		case 0x24: /* 2: aaa0 0100 */
			jmp_i(0x100);
			break;
		case 0x25: /* 1: 0010 0101 */
			en_tcnti();
			break;
		case 0x26: /* 2: 0010 0110 */
			jnt0_i();
			break;
		case 0x27: /* 1: 0010 0111 */
			clr_a();
			break;
		case 0x28: /* 1: 0010 1rrr */
		case 0x29: /* 1: 0010 1rrr */
		case 0x2a: /* 1: 0010 1rrr */
		case 0x2b: /* 1: 0010 1rrr */
		case 0x2c: /* 1: 0010 1rrr */
		case 0x2d: /* 1: 0010 1rrr */
		case 0x2e: /* 1: 0010 1rrr */
		case 0x2f: /* 1: 0010 1rrr */
			xch_a_r(op & 7);
			break;
		case 0x30: /* 1: 0011 000r */
			xchd_a_rm(0);
			break;
		case 0x31: /* 1: 0011 000r */
			xchd_a_rm(1);
			break;
		case 0x32: /* 2: bbb1 0010 */
			jbb_i(1);
			break;
		case 0x33: /* 1: 0011 0101 */
			illegal();
			break;
		case 0x34: /* 2: aaa1 0100 */
			call_i(0x100);
			break;
		case 0x35: /* 1: 0000 0101 */
			dis_tcnti();
			break;
		case 0x36: /* 2: 0011 0110 */
			jt0_i();
			break;
		case 0x37: /* 1: 0011 0111 */
			cpl_a();
			break;
		case 0x38: /* 2: 0011 10pp */
		case 0x39: /* 2: 0011 10pp */
		case 0x3a: /* 2: 0011 10pp */
		case 0x3b: /* 2: 0011 10pp */
			out_p_a(op & 3);
			break;
		case 0x3c: /* 2: 0011 11pp */
		case 0x3d: /* 2: 0011 11pp */
		case 0x3e: /* 2: 0011 11pp */
		case 0x3f: /* 2: 0011 11pp */
			movd_p_a(op & 3);
			break;
		case 0x40: /* 1: 0100 000r */
			orl_rm(0);
			break;
		case 0x41: /* 1: 0100 000r */
			orl_rm(1);
			break;
		case 0x42: /* 1: 0100 0010 */
			mov_a_t();
			break;
		case 0x43: /* 2: 0100 0011 */
			orl_i();
			break;
		case 0x44: /* 2: aaa0 0100 */
			jmp_i(0x200);
			break;
		case 0x45: /* 1: 0100 0101 */
			strt_cnt();
			break;
		case 0x46: /* 2: 0100 0110 */
			jnt1_i();
			break;
		case 0x47: /* 1: 0100 0111 */
			swap_a();
			break;
		case 0x48: /* 1: 0100 1rrr */
		case 0x49: /* 1: 0100 1rrr */
		case 0x4a: /* 1: 0100 1rrr */
		case 0x4b: /* 1: 0100 1rrr */
		case 0x4c: /* 1: 0100 1rrr */
		case 0x4d: /* 1: 0100 1rrr */
		case 0x4e: /* 1: 0100 1rrr */
		case 0x4f: /* 1: 0100 1rrr */
			orl_r(op & 7);
			break;
		case 0x50: /* 1: 0101 000r */
			anl_rm(0);
			break;
		case 0x51: /* 1: 0101 000r */
			anl_rm(1);
			break;
		case 0x52: /* 2: bbb1 0010 */
			jbb_i(2);
			break;
		case 0x53: /* 2: 0101 0011 */
			anl_i();
			break;
		case 0x54: /* 2: aaa1 0100 */
			call_i(0x200);
			break;
		case 0x55: /* 1: 0101 0101 */
			strt_t();
			break;
		case 0x56: /* 2: 0101 0110 */
			jt1_i();
			break;
		case 0x57: /* 1: 0101 0111 */
			da_a();
			break;
		case 0x58: /* 1: 0101 1rrr */
		case 0x59: /* 1: 0101 1rrr */
		case 0x5a: /* 1: 0101 1rrr */
		case 0x5b: /* 1: 0101 1rrr */
		case 0x5c: /* 1: 0101 1rrr */
		case 0x5d: /* 1: 0101 1rrr */
		case 0x5e: /* 1: 0101 1rrr */
		case 0x5f: /* 1: 0101 1rrr */
			anl_r(op & 7);
			break;
		case 0x60: /* 1: 0110 000r */
			add_rm(0);
			break;
		case 0x61: /* 1: 0110 000r */
			add_rm(1);
			break;
		case 0x62: /* 1: 0110 0010 */
			mov_t_a();
			break;
		case 0x63: /* 1: 0110 0011 */
			illegal();
			break;
		case 0x64: /* 2: aaa0 0100 */
			jmp_i(0x300);
			break;
		case 0x65: /* 1: 0110 0101 */
			stop_tcnt();
			break;
		case 0x66: /* 1: 0110 0110 */
			illegal();
			break;
		case 0x67: /* 1: 0110 0111 */
			rrc_a();
			break;
		case 0x68: /* 1: 0110 1rrr */
		case 0x69: /* 1: 0110 1rrr */
		case 0x6a: /* 1: 0110 1rrr */
		case 0x6b: /* 1: 0110 1rrr */
		case 0x6c: /* 1: 0110 1rrr */
		case 0x6d: /* 1: 0110 1rrr */
		case 0x6e: /* 1: 0110 1rrr */
		case 0x6f: /* 1: 0110 1rrr */
			add_r(op & 7);
			break;
		case 0x70: /* 1: 0111 000r */
			addc_rm(0);
			break;
		case 0x71: /* 1: 0111 000r */
			addc_rm(1);
			break;
		case 0x72: /* 2: bbb1 0010 */
			jbb_i(3);
			break;
		case 0x73: /* 1: 0111 0011 */
			illegal();
			break;
		case 0x74: /* 2: aaa1 0100 */
			call_i(0x300);
			break;
		case 0x75: /* 1: 0111 0101 */
			illegal();
			break;
		case 0x76: /* 2: 0111 0110 */
			jf1_i();
			break;
		case 0x77: /* 1: 0111 0111 */
			rr_a();
			break;
		case 0x78: /* 1: 0111 1rrr */
		case 0x79: /* 1: 0111 1rrr */
		case 0x7a: /* 1: 0111 1rrr */
		case 0x7b: /* 1: 0111 1rrr */
		case 0x7c: /* 1: 0111 1rrr */
		case 0x7d: /* 1: 0111 1rrr */
		case 0x7e: /* 1: 0111 1rrr */
		case 0x7f: /* 1: 0111 1rrr */
			addc_r(op & 7);
			break;
		case 0x80: /* 1: 1000 0000 */
			illegal();
			break;
		case 0x81: /* 1: 1000 0001 */
			illegal();
			break;
		case 0x82: /* 1: 1000 0010 */
			illegal();
			break;
		case 0x83: /* 2: 1000 0011 */
			ret();
			break;
		case 0x84: /* 2: aaa0 0100 */
			jmp_i(0x400);
			break;
		case 0x85: /* 1: 1000 0101 */
			clr_f0();
			break;
		case 0x86: /* 2: 1000 0110 */
			jobf_i();
			break;
		case 0x87: /* 1: 1000 0111 */
			illegal();
			break;
		case 0x88: /* 2: 1000 10pp */
		case 0x89: /* 2: 1000 10pp */
		case 0x8a: /* 2: 1000 10pp */
		case 0x8b: /* 2: 1000 10pp */
			orl_p_i(op & 3);
			break;
		case 0x8c: /* 2: 1000 11pp */
		case 0x8d: /* 2: 1000 11pp */
		case 0x8e: /* 2: 1000 11pp */
		case 0x8f: /* 2: 1000 11pp */
			orld_p_a(op & 7);
			break;
		case 0x90: /* 1: 1001 0000 */
			mov_sts_a();
			break;
		case 0x91: /* 1: 1001 0001 */
			illegal();
			break;
		case 0x92: /* 2: bbb1 0010 */
			jbb_i(4);
			break;
		case 0x93: /* 2: 1001 0011 */
			retr();
			break;
		case 0x94: /* 1: aaa1 0100 */
			call_i(0x400);
			break;
		case 0x95: /* 1: 1001 0101 */
			cpl_f0();
			break;
		case 0x96: /* 2: 1001 0110 */
			jnz_i();
			break;
		case 0x97: /* 1: 1001 0111 */
			clr_c();
			break;
		case 0x98: /* 2: 1001 10pp */
		case 0x99: /* 2: 1001 10pp */
		case 0x9a: /* 2: 1001 10pp */
		case 0x9b: /* 2: 1001 10pp */
			anl_p_i(op & 3);
			break;
		case 0x9c: /* 2: 1001 11pp */
		case 0x9d: /* 2: 1001 11pp */
		case 0x9e: /* 2: 1001 11pp */
		case 0x9f: /* 2: 1001 11pp */
			anld_p_a(op & 7);
			break;
		case 0xa0: /* 1: 1010 000r */
			mov_rm_a(0);
			break;
		case 0xa1: /* 1: 1010 000r */
			mov_rm_a(1);
			break;
		case 0xa2: /* 1: 1010 0010 */
			illegal();
			break;
		case 0xa3: /* 2: 1010 0011 */
			movp_a_am();
			break;
		case 0xa4: /* 2: aaa0 0100 */
			jmp_i(0x500);
			break;
		case 0xa5: /* 1: 1010 0101 */
			clr_f1();
			break;
		case 0xa6: /* 1: 1010 0110 */
			illegal();
			break;
		case 0xa7: /* 1: 1010 0111 */
			cpl_c();
			break;
		case 0xa8: /* 1: 1010 1rrr */
		case 0xa9: /* 1: 1010 1rrr */
		case 0xaa: /* 1: 1010 1rrr */
		case 0xab: /* 1: 1010 1rrr */
		case 0xac: /* 1: 1010 1rrr */
		case 0xad: /* 1: 1010 1rrr */
		case 0xae: /* 1: 1010 1rrr */
		case 0xaf: /* 1: 1010 1rrr */
			mov_r_a(op & 7);
			break;
		case 0xb0: /* 2: 1011 000r */
			mov_rm_i(0);
			break;
		case 0xb1: /* 2: 1011 000r */
			mov_rm_i(1);
			break;
		case 0xb2: /* 2: bbb1 0010 */
			jbb_i(5);
			break;
		case 0xb3: /* 2: 1011 0011 */
			jmpp_a();
			break;
		case 0xb4: /* 2: aaa1 0100 */
			call_i(0x500);
			break;
		case 0xb5: /* 1: 1011 0101 */
			cpl_f1();
			break;
		case 0xb6: /* 2: 1011 0110 */
			jf0_i();
			break;
		case 0xb7: /* 1: 1011 0111 */
			illegal();
			break;
		case 0xb8: /* 2: 1011 1rrr */
		case 0xb9: /* 2: 1011 1rrr */
		case 0xba: /* 2: 1011 1rrr */
		case 0xbb: /* 2: 1011 1rrr */
		case 0xbc: /* 2: 1011 1rrr */
		case 0xbd: /* 2: 1011 1rrr */
		case 0xbe: /* 2: 1011 1rrr */
		case 0xbf: /* 2: 1011 1rrr */
			mov_r_i(op & 7);
			break;
		case 0xc0: /* 1: 1100 0000 */
			illegal();
			break;
		case 0xc1: /* 1: 1100 0001 */
			illegal();
			break;
		case 0xc2: /* 1: 1100 0010 */
			illegal();
			break;
		case 0xc3: /* 1: 1100 0011 */
			illegal();
			break;
		case 0xc4: /* 2: aaa0 0100 */
			jmp_i(0x600);
			break;
		case 0xc5: /* 1: 1100 0101 */
			sel_rb0();
			break;
		case 0xc6: /* 2: 1100 0110 */
			jz_i();
			break;
		case 0xc7: /* 1: 1100 0111 */
			mov_a_psw();
			break;
		case 0xc8: /* 1: 1100 1rrr */
		case 0xc9: /* 1: 1100 1rrr */
		case 0xca: /* 1: 1100 1rrr */
		case 0xcb: /* 1: 1100 1rrr */
		case 0xcc: /* 1: 1100 1rrr */
		case 0xcd: /* 1: 1100 1rrr */
		case 0xcf: /* 1: 1100 1rrr */
			dec_r(op & 7);
			break;
		case 0xd0: /* 1: 1101 000r */
			xrl_rm(0);
			break;
		case 0xd1: /* 1: 1101 000r */
			xrl_rm(1);
			break;
		case 0xd2: /* 2: bbb1 0010 */
			jbb_i(6);
			break;
		case 0xd3: /* 1: 1101 0011 */
			xrl_i();
			break;
		case 0xd4: /* 2: aaa1 0100 */
			call_i(0x600);
			break;
		case 0xd5: /* 1: 1101 0101 */
			sel_rb1();
			break;
		case 0xd6: /* 2: 1101 0110 */
			jnibf_i();
			break;
		case 0xd7: /* 1: 1101 0111 */
			mov_psw_a();
			break;
		case 0xd8: /* 1: 1101 1rrr */
		case 0xd9: /* 1: 1101 1rrr */
		case 0xda: /* 1: 1101 1rrr */
		case 0xdb: /* 1: 1101 1rrr */
		case 0xdc: /* 1: 1101 1rrr */
		case 0xdd: /* 1: 1101 1rrr */
		case 0xde: /* 1: 1101 1rrr */
		case 0xdf: /* 1: 1101 1rrr */
			xrl_r(op & 7);
			break;
		case 0xe0: /* 1: 1110 0000 */
			illegal();
			break;
		case 0xe1: /* 1: 1110 0001 */
			illegal();
			break;
		case 0xe2: /* 1: 1110 0010 */
			illegal();
			break;
		case 0xe3: /* 2: 1110 0011 */
			movp3_a_am();
			break;
		case 0xe4: /* 2: aaa0 0100 */
			jmp_i(0x700);
			break;
		case 0xe5: /* 1: 1110 0101 */
			en_dma();
			break;
		case 0xe6: /* 2: 1110 0110 */
			jnc_i();
			break;
		case 0xe7: /* 1: 1110 0111 */
			rl_a();
			break;
		case 0xe8: /* 2: 1111 1rrr */
		case 0xe9: /* 2: 1111 1rrr */
		case 0xea: /* 2: 1111 1rrr */
		case 0xeb: /* 2: 1111 1rrr */
		case 0xec: /* 2: 1111 1rrr */
		case 0xed: /* 2: 1111 1rrr */
		case 0xee: /* 2: 1111 1rrr */
		case 0xef: /* 2: 1111 1rrr */
			djnz_r_i(op & 7);
			break;
		case 0xf0: /* 1: 1111 000r */
			mov_a_rm(0);
			break;
		case 0xf1: /* 1: 1111 000r */
			mov_a_rm(1);
			break;
		case 0xf2: /* 2: bbb1 0010 */
			jbb_i(7);
			break;
		case 0xf3: /* 1: 1111 0011 */
			illegal();
			break;
		case 0xf4: /* 2: aaa1 0100 */
			call_i(0x700);
			break;
		case 0xf5: /* 1: 1111 0101 */
			en_flags();
			break;
		case 0xf6: /* 2: 1111 0110 */
			jc_i();
			break;
		case 0xf7: /* 1: 1111 0111 */
			rlc_a();
			break;
		case 0xf8: /* 1: 1110 1rrr */
		case 0xf9: /* 1: 1110 1rrr */
		case 0xfa: /* 1: 1110 1rrr */
		case 0xfb: /* 1: 1110 1rrr */
		case 0xfc: /* 1: 1110 1rrr */
		case 0xfd: /* 1: 1110 1rrr */
		case 0xfe: /* 1: 1110 1rrr */
		case 0xff: /* 1: 1110 1rrr */
			mov_a_r(op & 7);
			break;
		}
	} while( i8x41_ICount > 0 );

	return cycles - i8x41_ICount;
}

/* Get registers, return context size */
unsigned i8x41_get_context(void *dst)
{
	if( dst )
		memcpy(dst, &i8x41, sizeof(I8X41));
	return sizeof(I8X41);
}

/* Set registers */
void i8x41_set_context(void *src)
{
	if( src )
		memcpy(&i8x41, src, sizeof(I8X41));
}

/* Get program counter */
unsigned i8x41_get_pc(void)
{
	return PC;
}

/* Set program counter */
void i8x41_set_pc(unsigned val)
{
	PC = val & 0x7ff;
}

/* Get stack pointer */
unsigned i8x41_get_sp(void)
{
	return PSW & SP;
}

/* Set stack pointer */
void i8x41_set_sp(unsigned val)
{
	PSW = (PSW & ~SP) | (val & SP);
}

unsigned i8x41_get_reg(int regnum)
{
	switch( regnum )
	{
	case I8X41_PC:	return PC;
	case I8X41_SP:	return PSW & SP;
	case I8X41_PSW: return PSW;
	case I8X41_A:	return A;
	case I8X41_T:	return TIMER;
	case I8X41_R0:	return R(0);
	case I8X41_R1:	return R(1);
	case I8X41_R2:	return R(2);
	case I8X41_R3:	return R(3);
	case I8X41_R4:	return R(4);
	case I8X41_R5:	return R(5);
	case I8X41_R6:	return R(6);
	case I8X41_R7:	return R(7);
	case I8X41_DATA:
			STATE &= ~OBF;	/* reset the output buffer full flag */
			return DBBO;
	case I8X41_STAT:
			return STATE;
	case REG_PREVIOUSPC: return PPC;
	default:
		if( regnum <= REG_SP_CONTENTS )
		{
			unsigned offset = (PSW & SP) + (REG_SP_CONTENTS - regnum);
			if( offset < 8 )
				return RM( M_STACK + offset ) | ( RM( M_STACK + offset + 1 ) << 8 );
		}
	}
	return 0;
}

void i8x41_set_reg (int regnum, unsigned val)
{
	switch( regnum )
	{
	case I8X41_PC:	PC = val & 0x7ff;
	case I8X41_SP:	PSW = (PSW & ~SP) | (val & SP);
	case I8X41_PSW: PSW = val;
	case I8X41_A:	A = val;
	case I8X41_T:	TIMER = val & 0x1fff;
	case I8X41_R0:	R(0) = val; break;
	case I8X41_R1:	R(1) = val; break;
	case I8X41_R2:	R(2) = val; break;
	case I8X41_R3:	R(3) = val; break;
	case I8X41_R4:	R(4) = val; break;
	case I8X41_R5:	R(5) = val; break;
	case I8X41_R6:	R(6) = val; break;
	case I8X41_R7:	R(7) = val; break;
	case I8X41_DATA:
			PSW &= ~F1;
			DBBI = val;
			if (i8x41.subtype == 8041) /* plain 8041 had no split input/output DBB buffers */
				DBBO = val;
			if (ENABLE & IBFI)
				i8x41_set_irq_line(I8X41_INT_IBF, HOLD_LINE);
			else
				STATE |= IBF;
			break;
	case I8X41_CMND:
			PSW |= F1;
			DBBI = val;
			if (i8x41.subtype == 8041) /* plain 8041 had no split input/output DBB buffers */
				DBBO = val;
			if (ENABLE & IBFI)
				i8x41_set_irq_line(I8X41_INT_IBF, HOLD_LINE);
			else
				STATE |= IBF;
			break;
	case I8X41_STAT:
			/* writing status.. hmm, should we issue interrupts here too? */
			STATE = val;
			break;
	default:
		if( regnum <= REG_SP_CONTENTS )
		{
			unsigned offset = (PSW & SP) + (REG_SP_CONTENTS - regnum);
			if( offset < 8 )
			{
				WM( M_STACK + offset, val & 0xff );
				WM( M_STACK + offset + 1, (val >> 8) & 0xff );
			}
		}
	}
}

void i8x41_set_nmi_line(int state)
{
	/* not applicable */
}

void i8x41_set_irq_line(int irqline, int state)
{
	switch( irqline )
	{
	case I8X41_INT_IBF:
		if (state != CLEAR_LINE)
		{
			STATE |= IBF;
			if (ENABLE & IBFI)
			{
				WM( M_STACK + (PSW&SP) * 2 + 0, PC & 0xff);
				WM( M_STACK + (PSW&SP) * 2 + 1, ((PC >> 8) & 0x0f) | (PSW & 0xf0) );
				PSW = (PSW & ~SP) | ((PSW + 1) & SP);
				PC = V_IBF;
			}
		}
		else
		{
			STATE &= ~IBF;
		}
		break;

	case I8X41_INT_TEST0:
		if (state != CLEAR_LINE)
			STATE |= TEST0;
		else
			STATE &= ~TEST0;
		break;

	case I8X41_INT_TEST1:
		if (state != CLEAR_LINE)
		{
			STATE |= TEST1;
		}
		else
		{
			/* high to low transition? */
			if (STATE & TEST1)
			{
				/* counting enabled? */
				if (ENABLE & CNT)
				{
					if (++TIMER > 0x1fff)
					{
						TOVF = 1;
						if (ENABLE & TCNTI)
						{
							WM( M_STACK + (PSW&SP) * 2 + 0, PC & 0xff);
							WM( M_STACK + (PSW&SP) * 2 + 1, ((PC >> 8) & 0x0f) | (PSW & 0xf0) );
							PSW = (PSW & ~SP) | ((PSW + 1) & SP);
							PC = V_TIMER;
						}
					}
				}
			}
			STATE &= ~TEST1;
		}
		break;
	}
}

void i8x41_set_irq_callback(int (*callback)(int irqline))
{
	i8x41.irq_callback = callback;
}

void i8x41_state_save(void *file)
{
}

void i8x41_state_load(void *file)
{
}

const char *i8x41_info(void *context, int regnum)
{
	static char buffer[8][15+1];
	static int which = 0;
	I8X41 *r = context;

	which = ++which % 8;
	buffer[which][0] = '\0';
	if( !context )
		r = &i8x41;

	switch( regnum )
	{
		case CPU_INFO_REG+I8X41_PC: sprintf(buffer[which], "PC:%04X", r->pc); break;
		case CPU_INFO_REG+I8X41_SP: sprintf(buffer[which], "S:%X", r->psw & SP); break;
		case CPU_INFO_REG+I8X41_PSW:sprintf(buffer[which], "PSW:%02X", r->psw); break;
		case CPU_INFO_REG+I8X41_A:	sprintf(buffer[which], "A:%02X", r->a); break;
		case CPU_INFO_REG+I8X41_T:	sprintf(buffer[which], "T:%04X", r->timer); break;
		case CPU_INFO_REG+I8X41_R0: sprintf(buffer[which], "R0:%02X", i8x41.ram[((r->psw & F1) ? M_BANK1 : M_BANK0) + 0]); break;
		case CPU_INFO_REG+I8X41_R1: sprintf(buffer[which], "R1:%02X", i8x41.ram[((r->psw & F1) ? M_BANK1 : M_BANK0) + 1]); break;
		case CPU_INFO_REG+I8X41_R2: sprintf(buffer[which], "R2:%02X", i8x41.ram[((r->psw & F1) ? M_BANK1 : M_BANK0) + 2]); break;
		case CPU_INFO_REG+I8X41_R3: sprintf(buffer[which], "R3:%02X", i8x41.ram[((r->psw & F1) ? M_BANK1 : M_BANK0) + 3]); break;
		case CPU_INFO_REG+I8X41_R4: sprintf(buffer[which], "R4:%02X", i8x41.ram[((r->psw & F1) ? M_BANK1 : M_BANK0) + 4]); break;
		case CPU_INFO_REG+I8X41_R5: sprintf(buffer[which], "R5:%02X", i8x41.ram[((r->psw & F1) ? M_BANK1 : M_BANK0) + 5]); break;
		case CPU_INFO_REG+I8X41_R6: sprintf(buffer[which], "R6:%02X", i8x41.ram[((r->psw & F1) ? M_BANK1 : M_BANK0) + 6]); break;
		case CPU_INFO_REG+I8X41_R7: sprintf(buffer[which], "R7:%02X", i8x41.ram[((r->psw & F1) ? M_BANK1 : M_BANK0) + 7]); break;
		case CPU_INFO_REG+I8X41_DATA:sprintf(buffer[which], "DBBI:%02X", i8x41.dbbi); break;
		case CPU_INFO_REG+I8X41_CMND:sprintf(buffer[which], "DBBO:%02X", i8x41.dbbo); break;
		case CPU_INFO_REG+I8X41_STAT:sprintf(buffer[which], "STAT:%02X", i8x41.state); break;
		case CPU_INFO_FLAGS:
			sprintf(buffer[which], "%c%c%c%c%c%c%c%c",
				r->psw & 0x80 ? 'C':'.',
				r->psw & 0x40 ? 'A':'.',
				r->psw & 0x20 ? '0':'.',
				r->psw & 0x10 ? 'B':'.',
				r->psw & 0x08 ? '?':'.',
				r->psw & 0x04 ? 's':'.',
				r->psw & 0x02 ? 's':'.',
				r->psw & 0x01 ? 's':'.');
			break;
		case CPU_INFO_NAME: return "I8X41";
		case CPU_INFO_FAMILY: return "Intel 8x41";
		case CPU_INFO_VERSION: return "0.1";
		case CPU_INFO_FILE: return __FILE__;
		case CPU_INFO_CREDITS: return "Copyright (c) 1999 Juergen Buchmueller, all rights reserved.";
		case CPU_INFO_REG_LAYOUT: return (const char*)i8x41_reg_layout;
		case CPU_INFO_WIN_LAYOUT: return (const char*)i8x41_win_layout;
	}
	return buffer[which];
}

unsigned i8x41_dasm(char *buffer, unsigned pc)
{
#ifdef MAME_DEBUG
	return Dasm8x41( buffer, pc );
#else
	sprintf( buffer, "$%02X", cpu_readop(pc) );
	return 1;
#endif
}

