/********************************************************************/
/*                                                                  */
/*            L   I  QQ  U U I DD    W   W  A  RR    555            */
/*            L   I Q  Q U U I D D   W   W A A R R   5              */
/*            L   I Q  Q U U I D D   W W W AAA RR    55             */
/*            L   I Q Q  U U I D D   WW WW A A R R     5            */
/*            LLL I  Q Q  U  I DD    W   W A A R R   55             */
/*                                                                  */
/*                             b                                    */
/*                             bb  y y                              */
/*                             b b yyy                              */
/*                             bb    y                              */
/*                                 yy                               */
/*                                                                  */
/*                     U U       FFF  O   O  TTT                    */
/*                     U U       F   O O O O  T                     */
/*                     U U TIRET FF  O O O O  T                     */
/*                     U U       F   O O O O  T                     */
/*                      U        F    O   O   T                     */
/*                                                                  */
/********************************************************************/

/********************************************************************/
/* this software is protected by the GPL, see copying.txt           */
/********************************************************************/

/********************************************************************/
/* name          : sockw32.c                                        */
/* content       : simple wrappers on the winsock API               */
/* last update   : April 13th 2001                                  */
/********************************************************************/

/*==================================================================*/
/* includes                                                         */
/*==================================================================*/

#ifdef WIN32
#include <winsock.h>
#endif
#include <stdlib.h>
#include <time.h>
#include <string.h>

#include "sockgen.h"
#include "log.h"

/*==================================================================*/
/* defines                                                          */
/*==================================================================*/

#define LW_SOCK_NB_BACKLOG               10
#define LW_SOCK_ACCEPT_SEC               60
#define LW_SOCK_ACCEPT_USEC               0
#define LW_SOCK_RECV_SEC                 60
#define LW_SOCK_RECV_USEC                 0 

#define LW_DESIRED_WINSOCK_VERSION 0x0101 

/*==================================================================*/
/* macros                                                           */
/*==================================================================*/

#ifndef MIN
#define MIN(a, b)  (((a) < (b)) ? (a) : (b))
#endif
#ifndef MAX
#define MAX(a, b)  (((a) > (b)) ? (a) : (b))
#endif

/*==================================================================*/
/* globals                                                          */
/*==================================================================*/

int LW_SOCK_LOG=0;

/*==================================================================*/
/* fonctions                                                        */
/*==================================================================*/

/*------------------------------------------------------------------*/
/*
 * Useless on UNIX, this function is usefull on Microsh*t Windozs
 * where one has to "initialize" the winsock api 8-(
 */
int lw_sock_init()
{
  int result=0;
  int err;
  WSADATA wsadata;
	
  err = WSAStartup(LW_DESIRED_WINSOCK_VERSION, &wsadata);
	
  if (err != 0)
    {
      if (LW_SOCK_LOG)
	{
	  log_println_str("Unable to initialize winsock API");
	}
    }
  else
    {
      result=1;
    }
	
  return result;
}

/*------------------------------------------------------------------*/
/*
 * Useless on UNIX.
 */
int lw_sock_exit()
{
  int result=1;

  WSACleanup();

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Creates, binds and prepares a TCP socket for listening.
 * This is usefull on the server.
 */
int lw_sock_listen(int *sock,int port)
{
  int result=0;
  struct sockaddr_in name;
  int enable=1;

  *sock=socket(AF_INET, SOCK_STREAM, 0);
  if (*sock>=0)
    {
      setsockopt(*sock,SOL_SOCKET,SO_REUSEADDR,
		 (char *) &enable,sizeof(int));

      name.sin_family = AF_INET;
      name.sin_addr.s_addr = INADDR_ANY;
      name.sin_port = htons((short) port);
      if (bind(*sock, (struct sockaddr *) &name, sizeof name)>=0) 
	{
	  if (listen((*sock),LW_SOCK_NB_BACKLOG)>=0)
	    {
	      result=1;
	    }
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Accepts an incoming connexion from a program which has issued a
 * connect. Usefull on the server.
 */
int lw_sock_accept(int *new_sock,char *ip,int *port,int listening_sock)
{
  int result=0;
  struct sockaddr_in name;
  int namelen=sizeof(struct sockaddr_in);
  fd_set read;
  struct timeval tv;
  int res;
  int enable=1;
  int disable=0;
  struct linger li;

  FD_ZERO(&read);
  FD_SET(listening_sock,&read);
  tv.tv_sec=LW_SOCK_ACCEPT_SEC;
  tv.tv_usec=LW_SOCK_ACCEPT_USEC;
  res=select(listening_sock+1,&read,NULL,NULL,&tv);
  if (res>=1)
    {
      (*new_sock)=accept(listening_sock,(struct sockaddr *) &name,&namelen);
      if ((*new_sock)>=0)
	{
	  li.l_onoff=0;
	  li.l_linger=0;
	  setsockopt(*new_sock,SOL_SOCKET,SO_KEEPALIVE,
		     (char *) &enable,sizeof(int));
	  setsockopt(*new_sock,SOL_SOCKET,SO_OOBINLINE,
		     (char *) &disable,sizeof(int));
	  setsockopt(*new_sock,SOL_SOCKET,SO_LINGER,
		     (char *) &li,sizeof(struct linger));

	  ioctlsocket(*new_sock,FIONBIO,&disable);

	  strncpy(ip,inet_ntoa(name.sin_addr),LW_SOCK_IP_SIZE-1);
	  ip[LW_SOCK_IP_SIZE-1]=0;
	  (*port)=(int) ntohs(name.sin_port);

	  result=1;
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Establishes a connection. The remote must have called sock_listen
 * first and must confirm the acceptation with a sock_accept.
 * Usefull on the client. 
 */
int lw_sock_connect(int *sock,char *ip,int port)
{
  int result=0;
  struct sockaddr_in name;
  int enable=1;
  int disable=0;
  struct linger li;

  *sock=socket(AF_INET, SOCK_STREAM, 0);
  if (*sock>=0)
    {
      name.sin_family = AF_INET;
      name.sin_addr.s_addr = INADDR_ANY;
      name.sin_port = 0;
      if (bind(*sock, (struct sockaddr *) &name, sizeof name) >= 0) 
	{
	  name.sin_family = AF_INET;
	  name.sin_addr.s_addr = inet_addr(ip);
	  name.sin_port = htons((short) port);
	  if (connect((*sock),(struct sockaddr *) &name, sizeof name)>=0)
	    {
	      /*
	       * Added this code copied/paste from accept.
	       * don'tknow if it's usefull
	       */
	      li.l_onoff=0;
	      li.l_linger=0;
	      setsockopt(*sock,SOL_SOCKET,SO_KEEPALIVE,
			 (char *) &enable,sizeof(int));
	      setsockopt(*sock,SOL_SOCKET,SO_OOBINLINE,
			 (char *) &disable,sizeof(int));
	      setsockopt(*sock,SOL_SOCKET,SO_LINGER,
			 (char *) &li,sizeof(struct linger));
	      
	      ioctlsocket(*sock,FIONBIO,&disable);
	      
	      result=1;
	    }
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Sends a string on the network.
 * The advantage of this function over a raw "send" is that it does
 * a "strlen" automatically to know the length of the string, and
 * adds a tailing "\n" so that the message is "telnet compliant"
 */
int lw_sock_send_str_ex(int sock, char *str)
{
  int result=0;
  int len;
  char buffer[LW_SOCK_MESSAGE_SIZE];

  /*
   * We put the string in a buffer, since we'll probably have
   * to modify it (cut if it's too long, add a tailing '\n').
   */
  len=strlen(str);
  len=MIN(len,LW_SOCK_MESSAGE_SIZE-2);
  strncpy(buffer,str,len);
  buffer[len]='\n';
  ++len;
  buffer[len]='\0';  

  /*
   * Now we send the bytes on the network. We assume all the bytes will
   * be transmitted in one shot.
   */
  if (send(sock,buffer,len,0)==len)
    {
      result=1;
    }

  if (LW_SOCK_LOG)
    {
      if (result)
	{
	  log_print_int(sock);
	  log_print_str(" > \"");
	  log_print_str(str);
	  log_print_str("\"");
	  log_println();
	  log_flush();
	}
      else
	{
	  log_print_int(sock);
	  log_print_str(" > timeout!");
	  log_println();
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Receives some data from the network.
 * A tailing "\n" is expected. The routine handles both chr(10) and chr(13)
 * correctly so that the program can be used with telnet under UNIX or
 * windows without much trouble.
 * This tailing "\n" is removed, so the return string is just the exact
 * message which has been send with sock_send.
 * Buffer must be able to accept at least LW_SOCK_MESSAGE_SIZE chars.
 * Note that the function will not block forever, if there's no incoming
 * for more than a #define specified amount of time, it will return
 * an error.
 */
int lw_sock_recv_str_ex(int sock, char *str)
{
  int result=1;
  int pos,l;
  int cr_found;
  char *cr;
  fd_set read;
  struct timeval tv;
  int res;

  /*
   * We keep on trying to get data until
   * - we get a '\n' character
   * - the size of incoming data exceeds LW_SOCK_MESSAGE_SIZE
   * - there's a reception low level error
   */
  cr_found=0;
  pos=0;
  str[0]='\0';
  while (!cr_found && pos<LW_SOCK_MESSAGE_SIZE-1 && result>0)
    {
      FD_ZERO(&read);
      FD_SET(sock,&read);
      tv.tv_sec=LW_SOCK_RECV_SEC;
      tv.tv_usec=LW_SOCK_RECV_USEC;
      res=select(sock+1,&read,NULL,NULL,&tv);
      if (res<=0)
	{
	  result=0;
	}
      else
	{
	  /*
	   * We check that the event we just received concerns the socket
	   * we are polling. If the event is not for us, let's consider
	   * everything is fine...
	   */
	  if (!FD_ISSET(sock,&read))
	    {
	      result=1;
	    }
	  else
	    {
	      /*
	       * We get the caracters one by one. This is a performance
	       * killer but we don't care since this routine is only
	       * used at the very beginning of the game, when the players
	       * are connecting themselves. And it has the *big* advantage
	       * that "netkeys" are not eaten up by this routine. More
	       * precisely, there's used to be a bug because when reading
	       * the final "OK" message, the clients read some netkeys
	       * along with it, which caused some network inconsistency
	       * since the messages had "disappeared".
	       */
	      if ((l=recv(sock,str+pos,1,0))<=0)
		{
		  /*
		   * OK, now we have received a message on this socket
		   * but there's no data. In most of the cases this means
		   * the socket is dead, so we return -1 instead of 0
		   * so that the caller can make the difference between
		   * a timeout (0) and a dead socket (-1).
		   */ 
		  result=-1;
		}
	      else
		{
		  /*
		   * pos is an offset in the buffer. It is used to keep a
		   * trace of where new data should be appended in case we
		   * have to call recv several times to retrieve the whole
		   * message.
		   */
		  pos+=l;
		  /*
		   * We add a tailing '\0' so that the string is "C compliant"
		   */
		  str[pos]=0;
		  /*
		   * We seek for character 10, which should be '\n'
		   */
		  if ((cr=strchr(str,10))!=NULL)
		    {
		      cr_found=1;
		      /*
		       * We handle the special character '13' for very often,
		       * especially when using telnet, the strings come
		       * with char(13)chr(10) at the end. So if it happens
		       * that after removing the 10 there's still a 13, then
		       * we remove the 13 as well.
		       */
		      if ((cr-str)>=1 && (*(cr-1))==13)
			{
			  /*
			   * Let's cut this ugly "ascii 13" character
			   * along with "ascii 10"
			   */
			  (*(cr-1))=0;
			}
		      else
			{	
			  /*
			   * No "ascii 13" in sight, simply remove "ascii 10"
			   */
			  (*cr)=0;
			}
		    }
		  result=1;
		}
	    }
	}
    }

  if (LW_SOCK_LOG)
    {
      if (result)
	{
	  log_print_int(sock);
	  log_print_str(" < \"");
	  log_print_str(str);
	  log_print_str("\"");
	  log_println();
	  log_flush();
	}
      else
	{
	  log_print_int(sock);
	  log_print_str(" < timeout!");
	  log_println();
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Sends a buffer on the network.
 * Only a standard send wrapper
 */
int lw_sock_send_buffer_ex(int sock, char *buffer,int len)
{
  int result=0;
  char trace[LW_SOCK_MESSAGE_SIZE];

  len=MIN(len,LW_SOCK_MESSAGE_SIZE-1);
  if (send(sock,buffer,len,0)==len)
    {
      result=1;
    }

  if (LW_SOCK_LOG)
    {
      if (result)
	{
	  strncpy(trace,buffer,len);
	  trace[len]='\0';
	  log_print_int(sock);
	  log_print_str(" > [");
	  log_print_str(trace);
	  log_print_str("]");
	  log_println();
	  log_flush();
	}
      else
	{
	  log_print_int(sock);
	  log_print_str(" > timeout!");
	  log_println();
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Receives a buffer on the network.
 * Only a standard recv wrapper
 */
int lw_sock_recv_buffer_ex(int sock, char *buffer,int len)
{
  int result=0;
  char trace[LW_SOCK_MESSAGE_SIZE];
  int res;
  fd_set read;
  struct timeval tv;

  len=MIN(len,LW_SOCK_MESSAGE_SIZE-1);

  FD_ZERO(&read);
  FD_SET(sock,&read);
  tv.tv_sec=LW_SOCK_RECV_SEC;
  tv.tv_usec=LW_SOCK_RECV_USEC;
  res=select(sock+1,&read,NULL,NULL,&tv);
  if (res>0)
    {
      if (FD_ISSET(sock,&read))
	{
	  if (recv(sock,buffer,len,0)==len)
	    {
	      result=1;
	    }
	}
    }

  if (LW_SOCK_LOG)
    {
      if (result)
	{
	  strncpy(trace,buffer,len);
	  trace[len]='\0';
	  log_print_int(sock);
	  log_print_str(" < [");
	  log_print_str(trace);
	  log_print_str("]");
	  log_println();
	  log_flush();
	}
      else
	{
	  log_print_int(sock);
	  log_print_str(" < timeout!");
	  log_println();
	}
    }

  return result;
}

/*------------------------------------------------------------------*/
/*
 * Closes a socket for good.
 */
int lw_sock_close(int sock)
{
  int result=0;

  if (shutdown(sock,2)!=0)
    {
      /*
       * We could print a warning here.
       */
    }

  if (close(sock)!=0)
    {
      /*
       * Strange error, was the socket really opened?
       */
    }
  else
    {
      result=1;
    }

  return result;
}



