/* $Header: /fridge/cvs/xscorch/snet/snetcomm.c,v 1.3 2001/04/07 19:51:22 justins Exp $ */
/*

   xscorch - snetcomm.c       Copyright(c) 2001,2000 Jacob Luna Lundberg
                              Copyright(c) 2001,2000 Justin David Smith
   jacob(at)chaos2.org        http://chaos2.org/~jacob
   justins(at)chaos2.org      http://chaos2.org/

   Communication functions, packet setup

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#define __ALLOW_STRING_H__
#include <snetint.h>



#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <sutil/sslink.h>



bool sc_net_set_nonblocking(int socket) {

   /* Setup the socket as nonblocking */
   if(fcntl(socket, F_SETFL, O_NONBLOCK) != 0) {
      sc_net_set_error("set_nonblocking", strerror(errno));
      return(false);
   }
   return(true);

}



bool sc_net_shutdown(int *socket) {

   /* Close down a TCP socket */
   if(socket == NULL || *socket < 0) return(true);
   shutdown(*socket, 2);
   close(*socket);
   *socket = -1;
   return(true);

}



bool sc_net_ready_to_send(sc_connection *conn) {

   fd_set fdesc;        /* Descriptor */
   struct timeval t;    /* Do not delay on select() call */
   int selectresult;    /* Result of select call */

   /* If no socket, return "ready" to prevent lockup */
   if(conn == NULL) return(true);

   /* Setup descriptor list and timer */
   FD_ZERO(&fdesc);
   FD_SET(conn->socket, &fdesc);
   t.tv_sec  = 0; 
   t.tv_usec = 0;

   /* Check for ready to send */
   selectresult = select(conn->socket + 1, NULL, &fdesc, NULL, &t);
   if(selectresult < 0) {
      sc_net_set_error("ready_to_send", strerror(errno));
      SC_CONN_SET_FLAGS(*conn, SC_CONN_LOCAL_ERROR);
   }

   /* Return result */
   return(selectresult ? true : false);

}



bool sc_net_ready_to_recv_within(sc_connection *conn, struct timeval *tv) {

   fd_set fdesc;        /* Descriptor */
   struct timeval t;    /* Do not delay on select() call */
   int selectresult;    /* Result of select call */

   /* If no socket, return "ready" to prevent lockup */
   if(conn == NULL) return(true);

   /* Setup descriptor list and timer */
   FD_ZERO(&fdesc);   
   FD_SET(conn->socket, &fdesc);
   if(tv == NULL) {
      t.tv_sec  = 0;
      t.tv_usec = 0;
   } else {
      t.tv_sec  = tv->tv_sec;
      t.tv_usec = tv->tv_usec;
   }

   /* Check for ready to send */
   selectresult = select(conn->socket + 1, &fdesc, NULL, NULL, &t);
   if(selectresult < 0) {
      sc_net_set_error("ready_to_recv", strerror(errno));
      SC_CONN_SET_FLAGS(*conn, SC_CONN_LOCAL_ERROR);
   }

   /* Return result */
   return(selectresult ? true : false);

}



bool sc_net_ready_to_recv(sc_connection *conn) {
/* sc_net_ready_to_recv()
   Check if we're ready to pull data from the network immediately. */

   return(sc_net_ready_to_recv_within(conn, NULL));

}



bool sc_net_send_buffer(sc_connection *conn, const byte *buffer, int size) {

   int res;
   int sent;

   if(conn == NULL || buffer == NULL || size <= 0) return(true);

   sent = 0;
   while (sent < size) {
      while((res = sc_net_ready_to_send(conn)) == 0) /* Just loop */;
      if(res < 0) return(true);

      res = send(conn->socket, buffer+sent, size-sent, 0);
      if(res < 0 && errno != EAGAIN) {
         sc_net_set_error("send_buffer", strerror(errno));
         SC_CONN_SET_FLAGS(*conn, SC_CONN_LOCAL_ERROR);
         return(true);
      }
      if(res > 0)
         sent += res;
   }

   return(false);

}



sc_packet_queue *sc_net_queue_append(sc_packet_queue *head, sc_packet_queue *item) {
/* sc_net_queue_append()
   Initialize an sc_packet_queue item.
   Append the item to the queue (wrapper for slink_append()). */

   head = (sc_packet_queue *)slink_append((slinklist *)head, (slinklist *)item);
   item->packet = (sc_packet *)malloc(sizeof(sc_packet));
   sc_net_packet_init(item->packet, SC_NET_UNKNOWN, 0);
   item->valid = false;
   item->length = 0;
   return(head);

}



sc_packet_queue *sc_net_queue_delete(sc_packet_queue *head, sc_packet_queue *item) {
/* sc_net_queue_delete()
   Remove an item from a packet queue (slink list).
   First, purge references to it from the list.  Then delete it. */

   sc_packet_queue *temp = head;
   if(head == item) head = head->next;
   while(temp != NULL) {
      if(temp->next == item)
         temp->next = temp->next->next;
      else
         temp = temp->next;
   }
   sc_net_packet_release(item->packet);
   free(item);
   return(head);

}



/**//* TEMP - Restructuring Markup is currently In Progress */
bool sc_net_recv_packet(sc_connection *conn, sc_config *c, void *parm, packet_handler handler) {
/* sc_net_recv_packet()
   Pull packets off the network stream and tokenize them into the packet queue.
   Then return a single packet off the top of the queue. */

   /* TEMP: We need to implement various timeouts here. */

   char buf[SC_NET_BUFFER_SIZE];
   sc_packet_queue *queue_item;
   int bufferfill, minbuf;

   /* Pull any waiting network data into separate packets in the queue. */
   while(sc_net_ready_to_recv(conn)) {

/**//* SUB 1 - Data Acquisition */
      /* Acquire some data from the socket. */
      bufferfill = recv(conn->socket, buf, SC_NET_BUFFER_SIZE, 0);
      if(bufferfill < 0) {
         sc_net_set_error("recv_buffer", strerror(errno));
         SC_CONN_SET_FLAGS(*conn, SC_CONN_LOCAL_ERROR);
         return(false);
      } else if(bufferfill == 0) {
         sc_net_set_error("recv_buffer", "Expected data, received nothing");
         SC_CONN_SET_FLAGS(*conn, SC_CONN_LOCAL_ERROR);
         return(false);
      }
/**//* END SUB 1 */

      /* Put the code in a packet on the queue. */

/**//* SUB 2 - Queue Preprocess Integrity */
/**//*    Might want to move this out of the loop or even eliminate it eventually. */
      while(conn->recv_tail != NULL && !conn->recv_tail->valid && !conn->recv_tail->length) {
         /* We have created empty packet(s) on the queue somehow. */
         /* TEMP: We should dump something to the console here (BUG). */
         conn->recv_head = sc_net_queue_delete(conn->recv_head, conn->recv_tail);
         if(conn->recv_head == NULL) conn->recv_tail = NULL;
      }
/**//* END SUB 2 */

      if(conn->recv_tail != NULL && !conn->recv_tail->valid) {
         /* We have a partial packet and some more data to add to it. */
         queue_item = conn->recv_tail;

/**//* SUB 3 - Writing a Packet */
/**//*    Ok, so here's where the hard work is.  There are currently two SUB 3 marked. */
/**//*    Somehow they'll have to be combined despite obvious differences.  :)  */
         if(queue_item->length + bufferfill < SC_PACKET_HEADER_SIZE) {
            /* This is the highly unusual case where we're reading a couple bytes at a time. */
            memcpy(queue_item->packet + queue_item->length, buf, bufferfill);
            queue_item->length += bufferfill;
            bufferfill = 0;
         } else {
            if(queue_item->length < SC_PACKET_HEADER_SIZE) {
               /* We have to construct the header before we work on the packet. */
               memcpy(queue_item->packet + queue_item->length, buf, SC_PACKET_HEADER_SIZE - queue_item->length);
               queue_item->packet->msg_type = untohl(queue_item->packet->msg_type);
               queue_item->packet->data_size = untohl(queue_item->packet->data_size);
               queue_item->packet->pkt_id = untohl(queue_item->packet->pkt_id);
               bufferfill -= SC_PACKET_HEADER_SIZE - queue_item->length;
               memmove(buf, buf + SC_PACKET_HEADER_SIZE - queue_item->length, bufferfill);
               queue_item->length = SC_PACKET_HEADER_SIZE;
               if((udword)queue_item->packet->data_size > SC_PACKET_MAX_DATA_SIZE)
                  /* TEMP: Packet data is probably misaligned.  Big problem.  Do something. */
                  ;
            }
            minbuf = MIN(bufferfill, queue_item->packet->data_size - queue_item->length + SC_PACKET_HEADER_SIZE);
            queue_item->packet->data = (byte *)realloc(queue_item->packet->data, minbuf);
            memcpy(queue_item->packet->data, buf, minbuf);
            queue_item->length += minbuf;
            bufferfill -= minbuf;
            memmove(buf, buf + minbuf, bufferfill);
            if(queue_item->length == queue_item->packet->data_size + SC_PACKET_HEADER_SIZE) {
               /* TEMP: Ooo you bad boy, Jacob.  You should add a check for the EOP here... */
               if(queue_item->packet->pkt_id == ++conn->id_rd) {
                  queue_item->valid = true;
               } else {
                  --conn->id_rd;
                  conn->recv_head = sc_net_queue_delete(conn->recv_head, queue_item);
                  sc_net_set_error("recv_packet", "Packet recieved out of order.");
                  SC_CONN_SET_FLAGS(*conn, SC_CONN_LOCAL_ERROR);
               }
            }
         }
/**//* END SUB 3 */

      }
      while(bufferfill > 0) {
         /* We have data to start a new packet with. */
         queue_item = (sc_packet_queue *)malloc(sizeof(sc_packet_queue));
         if(queue_item == NULL) {
            sc_net_set_error("recv_packet", "malloc() failed");
            SC_CONN_SET_FLAGS(*conn, SC_CONN_LOCAL_ERROR);
            return(false);
         }
         queue_item->next = NULL;
         conn->recv_head = sc_net_queue_append(conn->recv_head, queue_item);
         conn->recv_tail = queue_item;

/**//* SUB 3 - Writing a Packet(s) */
         memcpy(queue_item->packet, buf, MIN(SC_PACKET_HEADER_SIZE, bufferfill));
         if(bufferfill < SC_PACKET_HEADER_SIZE) {
            queue_item->length = bufferfill;
            bufferfill = 0;
         } else {
            queue_item->packet->msg_type = untohl(queue_item->packet->msg_type);
            queue_item->packet->data_size = untohl(queue_item->packet->data_size);
            queue_item->packet->pkt_id = untohl(queue_item->packet->pkt_id);
            queue_item->length = SC_PACKET_HEADER_SIZE;
            bufferfill -= SC_PACKET_HEADER_SIZE;
            memmove(buf, buf + SC_PACKET_HEADER_SIZE, bufferfill);
            if((udword)queue_item->packet->data_size > SC_PACKET_MAX_DATA_SIZE)
               /* TEMP: Packet data is probably misaligned.  Big problem.  Do something. */
               ;
            minbuf = MIN(bufferfill, queue_item->packet->data_size);
            queue_item->packet->data = (byte *)realloc(queue_item->packet->data, minbuf);
            memcpy(queue_item->packet->data, buf, minbuf);
            queue_item->length += minbuf;
            bufferfill -= minbuf;
            memmove(buf, buf + minbuf, bufferfill);
            if(queue_item->length == queue_item->packet->data_size + SC_PACKET_HEADER_SIZE) {
               /* TEMP: Ooo you bad boy, Jacob.  You should add a check for the EOP here... */
               if(queue_item->packet->pkt_id == ++conn->id_rd) {
                  queue_item->valid = true;
               } else {
                  --conn->id_rd;
                  conn->recv_head = sc_net_queue_delete(conn->recv_head, queue_item);
                  sc_net_set_error("recv_packet", "Packet recieved out of order.");
                  SC_CONN_SET_FLAGS(*conn, SC_CONN_LOCAL_ERROR);
               }
            }
         }
/**//* END SUB 3 */

      }
   }

/**//* SUB 4 - Handle Packet */
   /*
    * Handle the topmost (oldest) packet in the queue.
    * We could handle them all but this way we are request driven.
    * When under load this should reduce the strain on the state machine.
    */
   if(conn->recv_head != NULL && conn->recv_head->valid) {
      handler(c, parm, conn->recv_head->packet);
      sc_net_packet_release(conn->recv_head->packet);
      conn->recv_head = sc_net_queue_delete(conn->recv_head, conn->recv_head);
      if(conn->recv_head == NULL) conn->recv_tail = NULL;
      /* Even if the queue is empty now, we still try once more, in case a packet completes. */
      return(true);
   }
/**//* END SUB 4 */

   /* If there was no valid packet, stop looping on packet read. */
   return(false);

}



bool sc_net_send_header(sc_connection *conn, udword type, udword size, udword pktid) {
/* sc_net_send_header()
   Transmit the header portion of a packet. */

   udword buffer[3];
   buffer[0] = uhtonl(type);
   buffer[1] = uhtonl(size);
   buffer[2] = uhtonl(pktid);
   return(sc_net_send_buffer(conn, (byte *)buffer, SC_PACKET_HEADER_SIZE));

}



bool sc_net_send_packet(sc_connection *conn, sc_packet *packet) {
/* sc_net_send_packet()
   Try to transmit a packet; true indicates success. */

   /*
    * Note the increment of the packet id counter in the call to send the header.
    * Might be good to reverse that on network errors if we ever implement recovery...
    */
   if(sc_net_send_header(conn, packet->msg_type, packet->data_size, ++conn->id_wr) < 0)
      return(false);
   else if(sc_net_send_buffer(conn, packet->data, packet->data_size) < 0)
      return(false);
   else
      return(true);

}



bool sc_net_send_message(sc_connection *conn, udword msg_type, const char *msg) {
/* sc_net_send_message()
   Wrapper for send_packet; the data payload will be a text message. */

   bool ret;
   sc_packet packet;

   sc_net_packet_init(&packet, msg_type, MIN(strlen(msg) + 1, SC_NET_BUFFER_SIZE));
   strncpy(packet.data, msg, packet.data_size);
   ret = sc_net_send_packet(conn, &packet);
   sc_net_packet_release(&packet);
   return(ret);

}


