/* $Header: /fridge/cvs/xscorch/sgtk/sinventory-gtk.c,v 1.7 2001/04/08 22:06:53 jacob Exp $ */
/*
   
   xscorch - sinventory-gtk.c Copyright(c) 2001,2000 Justin David Smith
                              Copyright(c) 2001      Jacob Luna Lundberg
   justins(at)chaos2.org      http://chaos2.org/
   jacob(at)chaos2.org        http://chaos2.org/~jacob
    
   Player configuration dialogue
    

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <swindow-gtk.h>
#include <sactiveconsole.h>
#include <gdk/gdkkeysyms.h>
#include <scolor-gtk.h>
#include <sutil/sstr.h>
#include <sgame/sinventory.h>
#include <sgame/sweapon.h>
#include <sgame/sconfig.h>
#include <sgame/splayer.h>
#include <sgame/sstate.h>
#include <sgame/sgame.h>

#include <sdisplay.h>
#include <sconsole.h>



/* Alter this constant at risk of total annihilation. */
#define  SC_MIN_WINDOW_SIZE      10



typedef struct _sc_inventory_gtk {
   sc_window_gtk *w;             /* Backpointer to main window */
   sc_player *p;                 /* Inventory is for this player */
   GtkWidget *infopane;          /* Top panel, which has name */
   GtkWidget *weaponpane;        /* Weapons panel */
   GtkWidget *accessorypane;     /* Accessories panel */
} sc_inventory_gtk;



static void _sc_inventory_info_paint_gtk(sc_inventory_gtk *inv) {
/* sc_inventory_info_paint_gtk
   Paints the inventory window with player name, budget, etc. */

   char buf[SC_GTK_STRING_BUFFER];
   snprintfn(buf, sizeof(buf), "Inventory for player %-16s  $%-9d    Round #%-2d (%-2d remaining)\n", 
             inv->p->name, inv->p->money, inv->w->c->curround + 2, 
             inv->w->c->numrounds - inv->w->c->curround - 1);
   sc_console_write_line(SC_CONSOLE(inv->infopane), 0, 0, buf);
   
}



static void _sc_inventory_weapon_paint_gtk(sc_inventory_gtk *inv) {
/* sc_inventory_weapon_paint_gtk
   Paint the weapons window pane.  */

   int count;                       /* Number of weapons to display */
   int index = 0;                   /* Current weapon index/iterator */
   sc_weapon_config *wc;            /* Weapon configuration data */
   const sc_weapon_info *info;      /* Data on the current weapon */
   char buf[SC_GTK_STRING_BUFFER];  /* Arbitrary text buffer */
   char less, great;                /* Can buy/sell indicators */

   /* Get the weapon lists */
   wc = inv->w->c->weapons;

   /* Deactivate any highlighting in the weapon panel */
   sc_console_highlight_detach_all(SC_CONSOLE(inv->weaponpane));

   /* Iterate through the list of weapons */
   /* TEMP 
      INDEX must refer to the current line in the display.
      
      sc_weapon_count() must refer the number of weapons
      TO BE DISPLAYED; if this function overestimates then
      inventory may crash.  
      
      */
   count = sc_weapon_count(wc, SC_WEAPON_LIMIT_ALL);
   info = sc_weapon_first(wc, SC_WEAPON_LIMIT_ALL);

   for(index = 0; index < count; ++index) {
      /* Get data about this weapon */
      less = (sc_inventory_can_sell_weapon(inv->p, info) ? '<' : ' ');
      great = (sc_inventory_can_buy_weapon(inv->p, info, SC_INVENTORY_INFINITE) ? '>' : ' ');
      snprintfn(buf, sizeof(buf), " %c %-16s %2d/$%-7d  %2d %c\n", 
                less, info->name, info->bundle, info->price, 
                info->inventories[inv->p->index], great);

      /* Display the weapon data. If the player's budget currently
         does not allow them to buy or sell, or the inventory constraints
         kick in, then disable the item. */
      sc_console_write_line(SC_CONSOLE(inv->weaponpane), 0, index, buf);
      if(less == ' ' && great == ' ') {
         sc_console_highlight_attach(SC_CONSOLE(inv->weaponpane), 
                                     &inv->w->colormap->gray, 
                                     &inv->w->colormap->black, 
                                     0, index, 
                                     sc_console_get_width(SC_CONSOLE(inv->weaponpane)), 1);
      } /* Disabling the pane? */
      info = sc_weapon_next(wc, info->ident, SC_WEAPON_LIMIT_ALL);
   } /* Iterate through weapons */

}



static void _sc_inventory_accessory_paint_gtk(sc_inventory_gtk *inv) {
/* sc_inventory_accessory_paint_gtk
   Paints the accessories pane. */

   int index;
   int *accinventory;
   sc_accessory_config *ac;
   const sc_accessory_info *info;
   char buf[SC_GTK_STRING_BUFFER];

   ac = inv->w->c->accessories;
   accinventory = inv->p->inventory->accessories;

   sc_console_highlight_detach_all(SC_CONSOLE(inv->accessorypane));

   for(index = 0; index < sc_accessory_count(ac); ++index) {
      info = sc_accessory_lookup(ac, index);
      snprintfn(buf, sizeof(buf), " %c %-16s %2d/$%-7d  %2d %c\n", 
                (sc_inventory_can_sell_accessory(ac, inv->p, index) ? '<' : ' '), 
                info->name, info->bundle, info->price, accinventory[index], 
                (sc_inventory_can_buy_accessory(ac, inv->p, index, SC_INVENTORY_INFINITE) ? '>' : ' '));
      sc_console_write_line(SC_CONSOLE(inv->accessorypane), 0, index, buf);
      if(!sc_inventory_can_buy_accessory(ac, inv->p, index, SC_INVENTORY_INFINITE) &&
        !sc_inventory_can_sell_accessory(ac, inv->p, index)) {
         sc_console_highlight_attach(SC_CONSOLE(inv->accessorypane), 
                                     &inv->w->colormap->gray, &inv->w->colormap->black, 
                                     0, index, sc_console_get_width(SC_CONSOLE(inv->accessorypane)), 1);
      }
   }

}



static gboolean _sc_inventory_weapon_key_gtk(ScActiveConsole *cons, ScActiveConsoleSpot *spot, GdkEventKey *event, sc_inventory_gtk *inv) {
/* sc_inventory_weapon_key_gtk
   User hit a key in the weapons panel; process it and return
   TRUE if the key has been processed by this function. */

   int count = spot->y;
   sc_weapon_config *wc = inv->w->c->weapons;
   const sc_weapon_info *info = sc_weapon_first(wc, SC_WEAPON_LIMIT_ALL);

   /* Find the weapon in question */
   for(; count > 0; --count)
      info = sc_weapon_next(wc, info->ident, SC_WEAPON_LIMIT_ALL);

   switch(event->keyval) {
      case GDK_Right:
      case GDK_KP_Right:
         sc_inventory_buy_weapon(inv->p, info);
         _sc_inventory_info_paint_gtk(inv);
         _sc_inventory_weapon_paint_gtk(inv);
         _sc_inventory_accessory_paint_gtk(inv);
         return(TRUE);

      case GDK_Left:
      case GDK_KP_Left:
         sc_inventory_sell_weapon(inv->p, info);
         _sc_inventory_info_paint_gtk(inv);
         _sc_inventory_weapon_paint_gtk(inv);
         _sc_inventory_accessory_paint_gtk(inv);
         return(TRUE);
   }

   return(FALSE);
   
}



static gboolean _sc_inventory_accessory_key_gtk(ScActiveConsole *cons, ScActiveConsoleSpot *spot, GdkEventKey *event, sc_inventory_gtk *inv) {
/* sc_inventory_accessory_key_gtk
   User hit a key in the accessory panel; process it and return
   TRUE if the key has been processed by this function. */

   sc_accessory_config *ac;
   ac = inv->w->c->accessories;

   switch(event->keyval) {
      case GDK_Right:
      case GDK_KP_Right:
         sc_inventory_buy_accessory(ac, inv->p, spot->y);
         _sc_inventory_info_paint_gtk(inv);
         _sc_inventory_weapon_paint_gtk(inv);
         _sc_inventory_accessory_paint_gtk(inv);
         return(TRUE);

      case GDK_Left:
      case GDK_KP_Left:
         sc_inventory_sell_accessory(ac, inv->p, spot->y);
         _sc_inventory_info_paint_gtk(inv);
         _sc_inventory_weapon_paint_gtk(inv);
         _sc_inventory_accessory_paint_gtk(inv);
         return(TRUE);
   }

   return(FALSE);
   
}



static gboolean _sc_inventory_weapon_button_gtk(ScActiveConsole *cons, ScActiveConsoleSpot *spot, GdkEventButton *event, sc_inventory_gtk *inv) {
/* sc_inventory_weapon_button_gtk
   User clicked in the weapon panel; process it and return
   TRUE if the key has been processed by this function. */

   int count = spot->y;
   sc_weapon_config *wc = inv->w->c->weapons;
   const sc_weapon_info *info = sc_weapon_first(wc, SC_WEAPON_LIMIT_ALL);

   /* Find the weapon in question */
   for(; count > 0; --count)
      info = sc_weapon_next(wc, info->ident, SC_WEAPON_LIMIT_ALL);

   switch(event->button) {
      case 1:  /* Left mouse */
         sc_inventory_buy_weapon(inv->p, info);
         _sc_inventory_info_paint_gtk(inv);
         _sc_inventory_weapon_paint_gtk(inv);
         _sc_inventory_accessory_paint_gtk(inv);
         return(TRUE);

      case 3:  /* Right mouse */
         sc_inventory_sell_weapon(inv->p, info);
         _sc_inventory_info_paint_gtk(inv);
         _sc_inventory_weapon_paint_gtk(inv);
         _sc_inventory_accessory_paint_gtk(inv);
         return(TRUE);
   }

   return(FALSE);
   
}



static gboolean _sc_inventory_accessory_button_gtk(ScActiveConsole *cons, ScActiveConsoleSpot *spot, GdkEventButton *event, sc_inventory_gtk *inv) {
/* sc_inventory_accessory_button_gtk
   User clicked in the accessory panel; process it and return
   TRUE if the key has been processed by this function. */

   sc_accessory_config *ac;
   ac = inv->w->c->accessories;

   switch(event->button) {
      case 1:  /* Left mouse */
         sc_inventory_buy_accessory(ac, inv->p, spot->y);
         _sc_inventory_info_paint_gtk(inv);
         _sc_inventory_weapon_paint_gtk(inv);
         _sc_inventory_accessory_paint_gtk(inv);
         return(TRUE);

      case 3:  /* Right mouse */
         sc_inventory_sell_accessory(ac, inv->p, spot->y);
         _sc_inventory_info_paint_gtk(inv);
         _sc_inventory_weapon_paint_gtk(inv);
         _sc_inventory_accessory_paint_gtk(inv);
         return(TRUE);
   }

   return(FALSE);
   
}



static gboolean _sc_inventory_continue_gtk(ScActiveConsole *cons, ScActiveConsoleSpot *spot, sc_inventory_gtk *inv) {
/* sc_inventory_continue_gtk
   User has closed the inventory window. */

   sc_display_console_detach_all(SC_DISPLAY(inv->w->screen));
   sc_game_set_state_now(inv->w->c, inv->w->c->game, SC_STATE_INVENTORY_PL_DONE);
   free(inv);
   return(TRUE);

}



static void _sc_inventory_create_gtk(sc_window_gtk *w, sc_player *p) {
/* sc_inventory_create_gtk 
   Construct the inventory window for given player. */

   sc_inventory_gtk *inv;
   GtkWidget *console;
   int wpcount, account;
   int windowsize;
   int i;

   /* Compute the window size */
   windowsize = w->c->fieldheight / sc_console_get_row_height(SC_CONSOLE(w->status)) - 15;
   if(windowsize < SC_MIN_WINDOW_SIZE) windowsize = SC_MIN_WINDOW_SIZE;
   
   /* How many weapons/accessories, so we can decide height of panes? */
   wpcount = sc_weapon_count(w->c->weapons, SC_WEAPON_LIMIT_ALL);
   account = sc_accessory_count(w->c->accessories);
   inv = (sc_inventory_gtk *)malloc(sizeof(sc_inventory_gtk));
   inv->w = w;
   inv->p = p;

   /* Construct the info, weapons, and accessories panels. */
   inv->infopane = sc_active_console_new(4, 2, 86, 2, CONSOLE_NORMAL);
   inv->weaponpane = sc_active_console_new(4, 7, 40, windowsize, CONSOLE_NORMAL);
   inv->accessorypane = sc_active_console_new(50, 7, 40, windowsize, CONSOLE_NORMAL);

   /* Weapons and accessories maintain offscreen buffers for scrolling */
   sc_console_buffer_size(SC_CONSOLE(inv->weaponpane), 40, wpcount);
   sc_console_buffer_size(SC_CONSOLE(inv->accessorypane), 40, account);

   /* Construct the bottom informational console. */
   console = sc_console_new(4, windowsize + 10, 86, 3, CONSOLE_NORMAL);
   sc_console_write_line(SC_CONSOLE(console), 0, 0, "[Left click] buys an item, [Right click] sells an item.  Or, you may");
   sc_console_write_line(SC_CONSOLE(console), 0, 1, "use the keyboard:  [Tab]/[Up]/[Down] navigates, [Left] buys, and [Right]");
   sc_console_write_line(SC_CONSOLE(console), 0, 2, "sells.  Press [Enter] when you are done.");
   sc_display_console_attach(SC_DISPLAY(w->screen), SC_CONSOLE(console));

   /* Hook up the various callbacks and add hotspots for 
      each weapon which is in the panel. */
   gtk_signal_connect_after(GTK_OBJECT(inv->weaponpane), "key_press_spot", GTK_SIGNAL_FUNC(_sc_inventory_weapon_key_gtk), inv);
   gtk_signal_connect_after(GTK_OBJECT(inv->weaponpane), "button_press_spot", GTK_SIGNAL_FUNC(_sc_inventory_weapon_button_gtk), inv);
   sc_display_console_attach(SC_DISPLAY(w->screen), SC_CONSOLE(inv->weaponpane));
   for(i = 0; i < wpcount; ++i) {
      sc_active_console_add_row_spot(SC_ACTIVE_CONSOLE(inv->weaponpane), i, NULL);
   }

   /* Hook up the various callbacks and add hotspots for 
      each accessory which is in the panel. */
   gtk_signal_connect_after(GTK_OBJECT(inv->accessorypane), "key_press_spot", GTK_SIGNAL_FUNC(_sc_inventory_accessory_key_gtk), inv);
   gtk_signal_connect_after(GTK_OBJECT(inv->accessorypane), "button_press_spot", GTK_SIGNAL_FUNC(_sc_inventory_accessory_button_gtk), inv);
   sc_display_console_attach(SC_DISPLAY(w->screen), SC_CONSOLE(inv->accessorypane));
   for(i = 0; i < account; ++i) {
      sc_active_console_add_row_spot(SC_ACTIVE_CONSOLE(inv->accessorypane), i, NULL);
   }

   /* Display the panes, and hookup miscellaneous signals. */
   sc_display_console_attach(SC_DISPLAY(w->screen), SC_CONSOLE(inv->infopane));
   sc_console_set_foreground(SC_CONSOLE(inv->infopane), &w->colormap->pcolors[p->index]);
   sc_console_write_line(SC_CONSOLE(inv->infopane), 70, 1, "< Continue >");
   sc_active_console_add_spot(SC_ACTIVE_CONSOLE(inv->infopane), 70, 1, 12, 1, NULL);
   gtk_signal_connect_after(GTK_OBJECT(inv->infopane), "select_spot", GTK_SIGNAL_FUNC(_sc_inventory_continue_gtk), inv);

   /* Paint every pane. */
   _sc_inventory_info_paint_gtk(inv);
   _sc_inventory_weapon_paint_gtk(inv);
   _sc_inventory_accessory_paint_gtk(inv);
   #if !SC_GTK_DEBUG_DISABLE_DEFAULT
      gtk_window_set_default(GTK_WINDOW(w->app), inv->infopane);
   #endif

}



void sc_window_inventory(sc_window *w_, sc_player *p) {
/* sc_window_inventory
   Interface function for obtaining inventory (purchases and sales)
   from the specified player.  Usually we do all players in order,
   at the beginning of every round. */

   sc_window_gtk *w = (sc_window_gtk *)w_;
   sc_display_console_detach_all(SC_DISPLAY(w->screen));
   _sc_inventory_create_gtk(w, p);

}
