/* $Header: /fridge/cvs/xscorch/sgame/sstate.c,v 1.4 2001/04/08 01:18:31 justins Exp $ */
/*
   
   xscorch - sstate.c         Copyright(c) 2000 Justin David Smith
   justins(at)chaos2.org      http://chaos2.org/

   Scorched game state machine
    

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <sgame.h>         /* Game state header */
#include <sstate.h>        /* Game state header */
#include <sphysics.h>      /* Need to do Wind updates */
#include <seconomy.h>      /* Need to do interest update */
#include <swindow.h>       /* We do much drawing from state machine */
#include <sconfig.h>       /* Config is dereferenced frequently */
#include <splayer.h>       /* Dereferenced in death check, etc. */
#include <sweapon.h>       /* We construct weapon chains here */
#include <strack.h>        /* Need for tracking, return codes */
#include <ssound/ssound.h> /* Sound hooks are present, here  */
#include <sland.h>         /* Need to drop the land after deton */
#include <sutil/srand.h>   /* Need a random number generator */
#include <sai/sai.h>       /* We are responsible for running AI */
#include <snet/snet.h>



static void _sc_state_prelude_begin(sc_config *c, sc_game *g) {

   /* Start sound, if applicable */
   g->musicid = SC_MUSIC_PRELUDE;
   sc_sound_start(c->sound, g->musicid);
            
   /* Setup screen/status */
   sc_status_message(c->window, "Welcome to XScorch!");
            
   /* Activate the main menu and wait */
   sc_window_main_menu(c->window);
   sc_game_set_state(g, SC_STATE_PRELUDE_IDLE, SC_DELAY_PROMPT);

}



static void _sc_state_game_end(sc_config *c, sc_game *g) {
   
   /* Start end-game sound */
   g->musicid = SC_MUSIC_ENDGAME;
   sc_sound_start(c->sound, g->musicid);

   /* Display end-game scores */
   sc_window_paint_end_game(c->window);
   sc_game_set_state(g, SC_STATE_GAME_END_IDLE, SC_DELAY_PROMPT);

}



static void _sc_state_inventory_begin(sc_config *c, sc_game *g) {

   /* Enable inventory music */
   g->musicid = SC_MUSIC_INVENTORY;
   sc_sound_start(c->sound, g->musicid);

   /* Starting with player 0... */
   g->curplayer = 0;
   #if USE_NETWORK
      if(c->client != NULL) sc_status_message(c->window, "Waiting for other players to begin round ...");
      sc_net_client_sync(c->client, SC_CONN_SYNC_INV, c->server != NULL);
   #endif /* Network? */
   sc_game_set_state_asap(g, SC_STATE_INVENTORY_PL_BEGIN);

}



static void _sc_state_inventory_pl_begin(sc_config *c, sc_game *g) {

   sc_player *p = c->players[g->curplayer];

   if(sc_ai_player_buy(c, p) == SC_AI_CONTINUE) {
      sc_status_message(c->window, "");
      sc_game_set_state_now(c, g, SC_STATE_INVENTORY_PL_DONE);
   } else {
      sc_status_player_message(c->window, p, "Inventory");
      sc_window_inventory(c->window, p);
      sc_game_set_state(g, SC_STATE_INVENTORY_PL_IDLE, SC_DELAY_PROMPT);
   }

}



static void _sc_state_inventory_pl_done(sc_config *c, sc_game *g) {

   #if USE_NETWORK
      sc_net_client_send_inventory(c, c->client, g->curplayer);
   #endif /* Networking? */
   
   ++g->curplayer;
   if(g->curplayer >= c->numplayers) {
      sc_game_set_state_now(c, g, SC_STATE_ROUND_BEGIN);
   } else {
      sc_game_set_state_now(c, g, SC_STATE_INVENTORY_PL_BEGIN);
   }
   
}



static void _sc_state_round_begin(sc_config *c, sc_game *g) {

   g->musicid = SC_MUSIC_ROUND;
   sc_sound_start(c->sound, g->musicid);
   sc_config_init_round(c);
   sc_window_paint(c->window, 0, 0, c->fieldwidth - 1, c->fieldheight - 1, SC_REGENERATE_LAND | SC_PAINT_EVERYTHING);
   #if USE_NETWORK
      if(c->client != NULL) sc_status_message(c->window, "Waiting for other players to close their inventory ...");
      sc_net_client_sync(c->client, SC_CONN_SYNC_ROUND, c->server != NULL);
   #endif /* Network? */
   sc_game_set_state_asap(g, SC_STATE_TURN_BEGIN);

}



static void _sc_state_round_end(sc_config *c, sc_game *g) {

   sc_game_set_victor(c);
   sc_economy_interest(c, c->economy);

   if(c->curround + 1 >= c->numrounds) {
      sc_game_set_state_now(c, g, SC_STATE_GAME_END);
   } else {
      g->musicid = SC_MUSIC_ENDROUND;
      sc_sound_start(c->sound, g->musicid);
      sc_window_paint_end_round(c->window);
      sc_game_set_state(g, SC_STATE_ROUND_END_IDLE, SC_DELAY_PROMPT);
   }

}



static void _sc_state_turn_begin(sc_config *c, sc_game *g) {

   sc_game_sync_timeout(g);
   sc_config_init_turn(c);
   sc_status_message(c->window, "");
   if(sc_game_victor(c)) {
      sc_game_set_state_now(c, g, SC_STATE_ROUND_END);
   } else {
      g->curplayer = 0;
      sc_game_set_state_now(c, g, SC_STATE_TURN_PL_BEGIN);
   }

}



static void _sc_state_turn_pl_begin(sc_config *c, sc_game *g) {

   sc_player *p = c->plorder[g->curplayer];

   if(p->dead || sc_ai_player_turn(c, p) == SC_AI_CONTINUE) {
      sc_status_message(c->window, "");
      sc_game_set_state(g, SC_STATE_TURN_PL_DONE, SC_DELAY_SHORT);
   } else {
      sc_status_update(c->window, p);
      sc_game_set_state(g, SC_STATE_TURN_PL_IDLE, SC_DELAY_PROMPT);
   }

}



static void _sc_state_turn_pl_done(sc_config *c, sc_game *g) {

   sc_player *p = c->plorder[g->curplayer];

   /* Arm the player's weapon */
   if(!p->dead) {
      p->armed = true;
      sc_window_redraw_tank(c->window, p);
   }

   sc_status_message(c->window, "");

   switch(c->options.mode) {
      case SC_CONFIG_MODE_SEQUENTIAL:
         #if USE_NETWORK
            sc_net_client_send_player_state(c, c->client);
            if(c->client != NULL) sc_status_message(c->window, "Waiting for other players to turn ...");
            sc_net_client_sync(c->client, SC_CONN_SYNC_TURN, c->server != NULL);
         #endif /* Network? */
         sc_game_set_state_asap(g, SC_STATE_RUN_TALK);
         break;
      case SC_CONFIG_MODE_SYNCHRONOUS:
         sc_game_set_state_now(c, g, SC_STATE_TURN_PL_NEXT);
         break;
   }

}



static void _sc_state_turn_pl_next(sc_config *c, sc_game *g) {

   ++g->curplayer;
   if(g->curplayer >= c->numplayers) switch(c->options.mode) {
      case SC_CONFIG_MODE_SEQUENTIAL:
         sc_game_set_state_now(c, g, SC_STATE_TURN_END);
         break;
      case SC_CONFIG_MODE_SYNCHRONOUS:
         g->curplayer = 0;
         #if USE_NETWORK
            sc_net_client_send_player_state(c, c->client);
            if(c->client != NULL) sc_status_message(c->window, "Waiting for other players to turn ...");
            sc_net_client_sync(c->client, SC_CONN_SYNC_TURN, c->server != NULL);
         #endif /* Network? */
         sc_game_set_state_asap(g, SC_STATE_RUN_TALK);
         break;
   } else {
      sc_game_set_state_now(c, g, SC_STATE_TURN_PL_BEGIN);
   }

}



static void _sc_state_run_talk(sc_config *c, sc_game *g) {

   const char *msg;
   sc_player *p = c->plorder[g->curplayer];
   int nextstate = 0;

   sc_status_message(c->window, "");

   switch(c->options.mode) {
      case SC_CONFIG_MODE_SEQUENTIAL:
         nextstate = SC_STATE_RUN_CREATION;
         break;
      case SC_CONFIG_MODE_SYNCHRONOUS:
         if(g->curplayer + 1 >= c->numplayers) {
            nextstate = SC_STATE_RUN_CREATION;
         } else {
            nextstate = SC_STATE_RUN_TALK;
         }
         break;
   }

   if(nextstate == SC_STATE_RUN_TALK) ++g->curplayer;

   if(!p->dead && p->armed) {
      msg = sc_player_talk(c, p);
      if(!SC_CONFIG_GFX_FAST(c) && msg != NULL) {
         sc_status_player_message(c->window, p, msg);
         sc_game_set_state(g, nextstate, SC_DELAY_TALK);
      }
   }
   
   sc_game_set_state_now(c, g, nextstate);

}



static void _sc_state_run_creation(sc_config *c, sc_game *g) {

   sc_weapon_create_all(c, &(g->explosions));
   sc_game_set_state_now(c, g, SC_STATE_RUN_WEAPONS);

}



static void _sc_state_run_weapons(sc_config *c, sc_game *g) {

   switch(sc_weapon_track_all(c, &g->explosions)) {
      case SC_WEAPON_TRACK_NO_ACTION:
         /* Weapons have all terminated */
         sc_game_set_state_now(c, g, SC_STATE_RUN_END);
         break;

      case SC_WEAPON_TRACK_NEED_RECURSE:
         /* Weapons are still flying, but no detonations yet */
         sc_game_reinstate(g, SC_CONFIG_GFX_FAST(c) ? 0 : SC_DELAY_SHORT);
         break;

      case SC_WEAPON_TRACK_DETONATE:
         /* Weapons may still be flying; a detonation occurred */
         g->curindex = 0;
         sc_game_set_state_now(c, g, SC_STATE_RUN_EXPLOSION);
         break;

      default:
         /* Oops */
   }

}



static void _sc_state_run_explosion(sc_config *c, sc_game *g) {

   /* What explosion should we work on now? */
   g->curexpl = sc_expl_index(g->explosions, g->curindex);

   if(g->curexpl == NULL) {
      /* No more explosions; delay then start clearing them */
      g->curindex = sc_expl_count(g->explosions) - 1;
      sc_game_set_state_now(c, g, SC_STATE_RUN_CLEAR_EXPLOSION);
   
   } else if(!sc_expl_annihilate(c, g->curexpl)) {
      /* This explosion terminated quickly; proceed */
      sc_game_sync_timeout(g);
      ++g->curindex;
      sc_game_set_state_allow_now(c, g, SC_STATE_RUN_EXPLOSION, SC_CONFIG_GFX_FAST(c) ? 0 : SC_DELAY_EXPLOSION);

   } else {
      /* This explosion is still executing */
      sc_game_set_state(g, SC_STATE_RUN_CONT_EXPLOSION, SC_DELAY_SHORT);
   } /* What's up with explosions lately? */

}



static void _sc_state_run_cont_explosion(sc_config *c, sc_game *g) {

   if(!sc_expl_annihilate_continue(c, g->curexpl)) {
      sc_game_sync_timeout(g);
      ++g->curindex;
      sc_game_set_state_allow_now(c, g, SC_STATE_RUN_EXPLOSION, SC_CONFIG_GFX_FAST(c) ? 0 : SC_DELAY_EXPLOSION);
   } else {
      sc_game_reinstate(g, SC_DELAY_SHORT);
   }

}



static void _sc_state_run_clear_explosion(sc_config *c, sc_game *g) {

   g->curexpl = sc_expl_index(g->explosions, g->curindex);
   if(g->curexpl == NULL) {
      g->curindex = 0;
      sc_game_set_state_allow_now(c, g, SC_STATE_RUN_LAND_DROP, SC_CONFIG_GFX_FAST(c) ? 0 : SC_DELAY_CLEAR);
   } else {
     /* Warning: damage routine must be called _before_ annihilate
         clear is called, as damage_all might need information that is
         deallocated on annihilate_clear().  */
      sc_player_damage_all(c, g->curexpl);

      /* Try to update the game status */
      if(!sc_expl_annihilate_clear(c, g->curexpl)) {
         /* We finished processing the clear explosion */
         sc_game_sync_timeout(g);
         --g->curindex;
         sc_game_set_state_allow_now(c, g, SC_STATE_RUN_CLEAR_EXPLOSION, SC_CONFIG_GFX_FAST(c) ? 0 : SC_DELAY_SHORT);
      } else {
         /* We are still processing the clear order */
         sc_game_set_state(g, SC_STATE_RUN_CLEAR_CONT_EXPL, SC_DELAY_SHORT);
      }
   }

}



static void _sc_state_run_clear_cont_explosion(sc_config *c, sc_game *g) {

   /* Try to update the game status */
   if(!sc_expl_annihilate_clear_continue(c, g->curexpl)) {
      /* We finally finished processing the clear explosion */
      sc_game_sync_timeout(g);
      --g->curindex;
      sc_game_set_state_allow_now(c, g, SC_STATE_RUN_CLEAR_EXPLOSION, SC_CONFIG_GFX_FAST(c) ? 0 : SC_DELAY_SHORT);
   } else {
      /* We are /still/ processing the clear order */
      sc_game_set_state(g, SC_STATE_RUN_CLEAR_CONT_EXPL, SC_DELAY_SHORT);
   }

}



static void _sc_state_run_land_drop(sc_config *c, sc_game *g) {

   g->curexpl = sc_expl_index(g->explosions, g->curindex);
   if(g->curexpl == NULL) {
      sc_game_set_state_now(c, g, SC_STATE_RUN_CLEAR_CHAIN);
   } else {
      sc_game_set_state_now(c, g, SC_STATE_RUN_LAND_DROP_STEP);
   }

}



static void _sc_state_run_land_drop_step(sc_config *c, sc_game *g) {

   if(sc_land_drop(c, c->land, g->curexpl->centerx, g->curexpl->radius)) {
      sc_game_reinstate_allow_now(c, g, SC_CONFIG_GFX_FAST(c) ? 0 : SC_DELAY_DROPPING);
   } else {
      ++g->curindex;
      sc_game_set_state_allow_now(c, g, SC_STATE_RUN_LAND_DROP, SC_CONFIG_GFX_FAST(c) ? 0 : SC_DELAY_DROPPING);
   }

}



static void _sc_state_run_player_drop(sc_config *c, sc_game *g) {

   if(sc_player_drop_all(c)) {
      sc_game_reinstate_allow_now(c, g, SC_CONFIG_GFX_FAST(c) ? 0 : SC_DELAY_DROPPING);
   } else {
      g->substate = 0;
      sc_game_set_state_now(c, g, SC_STATE_RUN_PLAYER_DEATH);
   }

}



static void _sc_state_run_player_death(sc_config *c, sc_game *g) {

   sc_player *p;
   const char *msg;

   sc_status_message(c->window, "");
   if(g->substate >= c->numplayers) {
      sc_game_set_state_now(c, g, SC_STATE_RUN_WEAPONS);
   } else {
      p = c->players[g->substate];
      if(!p->dead && p->life <= 0) {
         /* This player just died */
         msg = sc_player_death_talk(c, p);
         if(msg != NULL) {
            sc_status_player_message(c->window, p, msg);
         } else {
            sc_status_player_message(c->window, p, "");
         } /* Should player give farewell message? */
         sc_player_death(c, p, &g->explosions);
         sc_player_died(c, p);
         g->curindex = 0;
         sc_window_undraw_tank(c->window, p);
         sc_game_set_state_now(c, g, SC_STATE_RUN_EXPLOSION);
         ++g->substate;
      } else {
         ++g->substate;
         sc_game_reinstate_now(c, g);
      }
   }

}



static void _sc_state_run_end(sc_config *c, sc_game *g) {

   switch(c->options.mode) {
      case SC_CONFIG_MODE_SEQUENTIAL:
         sc_game_set_state_now(c, g, SC_STATE_TURN_PL_NEXT);
         break;
      case SC_CONFIG_MODE_SYNCHRONOUS:
         sc_game_set_state_now(c, g, SC_STATE_TURN_END);
         break;
   }

}



void sc_state_run(sc_config *c, sc_game *g) {

   struct timeval curtime;
   
   /* Check network activity */
   #if USE_NETWORK 
      sc_net_client_run(c, c->client);
      sc_net_server_run(c, c->server);
      if(sc_net_client_death(&c->client)) {
         sc_window_message(c->window, "Client Disconnected", "Client was disconnected from the network");
         sc_window_update(c->window);
      }
      if(c->client != NULL && !SC_CONN_IS_OKAY(c->client->server)) {
         /* We deferred this cycle */
         return;
      }
   #endif /* Network activity? */
   
   sc_game_time(&curtime);
   if(curtime.tv_sec > g->timeout.tv_sec || 
     (curtime.tv_sec == g->timeout.tv_sec && curtime.tv_usec >= g->timeout.tv_usec)) {

      /*printf("State %8x    ", g->state); 
      SC_PROFILE_BEGIN("game_run")*/
      
      switch(g->state) {



         /***  PRELUDE  ***/


         /* PRELUDE_BEGIN  
               -> PRELUDE_IDLE
            Initialise for a new game; display main menu. 
         */
         case SC_STATE_PRELUDE_BEGIN:
         case SC_STATE_GAME_END_DONE:
            _sc_state_prelude_begin(c, g);
            break;



         /***  GAME  ***/


         /* GAME_BEGIN     
               -> INVENTORY_BEGIN   (immediate)
            When user first presses Enter on the main menu.  Initialize
            configuration: Initialize config, player game data, and just
            about everything else relevant for game state.
         */
         case SC_STATE_GAME_BEGIN:
            /* Initialize configuration for a new GAME */
            sc_config_init_game(c);
            sc_status_message(c->window, "Starting a new game ...");
            #if USE_NETWORK
               if(c->client != NULL) sc_status_message(c->window, "Waiting for other players to begin game ...");
               sc_net_client_sync(c->client, SC_CONN_SYNC_GAME, c->server != NULL);
            #endif /* Network? */
            sc_game_set_state_asap(g, SC_STATE_INVENTORY_BEGIN);
            break;


         /* GAME_END
               -> GAME_END_IDLE
            All rounds have been played, display the final winners.  Pause
            for user input.
         */
         case SC_STATE_GAME_END:
            _sc_state_game_end(c, g);
            break;



         /***  INVENTORY   ***/
         
         
         /* INVENTORY_BEGIN
               -> INVENTORY_PL_BEGIN   (immediate)
            Begin inventory; prepare to take inventory for one player.
         */
         case SC_STATE_INVENTORY_BEGIN:
         case SC_STATE_ROUND_END_DONE:
            _sc_state_inventory_begin(c, g);
            break;


         /* INVENTORY_PL_BEGIN
             Human:
               -> INVENTORY_PL_IDLE
             AI:
               -> INVENTORY_PL_DONE
            If this state is entered, then curplayer must be a valid player.
            Purchases for the player indicated.  Loops back to self until all
            players have bought items.  If this is an AI, purchases are made
            now.
         */
         case SC_STATE_INVENTORY_PL_BEGIN:
            _sc_state_inventory_pl_begin(c, g);
            break;


         /* INVENTORY_PL_DONE
               -> INVENTORY_PL_BEGIN   (loop)
               t> ROUND_BEGIN          (terminating condition)
         */
         case SC_STATE_INVENTORY_PL_DONE:
            _sc_state_inventory_pl_done(c, g);
            break;
            

          
         /***  ROUND    ***/
         
         
         /* ROUND_BEGIN
               -> TURN_BEGIN     (immediate)
            Begin a new round:  Inventory has been taken; generate the land,
            place tanks as appropriate, and prepare for Scorch.
         */
         case SC_STATE_ROUND_BEGIN:
            _sc_state_round_begin(c, g);
            break;
            
            
         /* ROUND_END
               -> ROUND_END_IDLE (end of round, will loop)
               t> GAME_END       (end of all rounds) 
            Round has ended; advance round, and go to inventory or final
            winnings (latter occurs only if all rounds have been played). 
            In former, we identify the victor, do economics and final
            winning calculations, and then notify the user who won.
         */
         case SC_STATE_ROUND_END:
            _sc_state_round_end(c, g);
            break;
            


         /***  TURNS    ***/
         
         
         /* TURN_BEGIN
               -> TURN_PL_BEGIN  (normal progression)
               t> ROUND_END      (no one/1 person alive)
            Player turn begin: Setup for each player to take a turn.  Make
            sure to check if the game has already ended `tho (is there a
            victor?)  Make sure times are in sync, and init the turn.
         */
         case SC_STATE_TURN_BEGIN:
         case SC_STATE_TURN_END:
            _sc_state_turn_begin(c, g);
            break;


         /* TURN_PL_BEGIN
             Dead:
               -> TURN_PL_DONE   (cannot play)
             Human:
               -> TURN_PL_IDLE   (take orders)
             AI:
               -> TURN_PL_DONE   (AI actions taken)
            If we get here, we must have a valid curplayer value (it may be
            a dead player, however).  If dead, then this cycle is basically
            skipped; otherwise, humans are given the opportunity to enter
            orders, and AI actions are taken now.
         */
         case SC_STATE_TURN_PL_BEGIN:
            _sc_state_turn_pl_begin(c, g);
            break;
            
            
         /* TURN_PL_DONE
         
            ==Sequential Mode==
               -> RUN_TALK
            This player has given orders; execute those orders if the player
            is living.  If the player is dead, RUN_TALK will fall through
            and proceed to the next player.  The player's weapon is armed
            here.
               
            ==Synchronous Mode==
               -> TURN_PL_NEXT
            This player has given orders; take orders from the next player.
            If no more players, the next state will fall to RUN_TALK loop.
            The player's weapon is armed here.
         */
         case SC_STATE_TURN_PL_DONE:
            _sc_state_turn_pl_done(c, g);
            break;


         /* TURN_PL_NEXT
            
            ==Sequential Mode==
               -> TURN_PL_BEGIN     (immediate)
               t> TURN_END          (terminal, immediate)
               
            ==Synchronous Mode==
               -> TURN_PL_BEGIN     (immediate)
               t> RUN_TALK          (terminal, immediate)
         */
         case SC_STATE_TURN_PL_NEXT:
            _sc_state_turn_pl_next(c, g);
            break;



         /***  RUN   ***/


         /* RUN_TALK
            
            ==Sequential Mode==
               -> RUN_CREATION      (might be 2s delay)
            This state DOES NOT LOOP.  If this player is not dead and armed,
            then we might allow them to speak (2s delay) or might launch
            their weapon now (immediate).
               
            ==Synchronous Mode==
               -> RUN_TALK          (might delay, loops)
               t> RUN_CREATION      (out of players)
            This state loops through all players.  If the player is not dead
            and is armed, they may speak at this time (2s delay) or might
            not (loop is immediate).  Once complete, this will yield to the
            first weapons state.
         */            
         case SC_STATE_RUN_TALK:
            _sc_state_run_talk(c, g);
            break;


         /* RUN_CREATION
               -> RUN_WEAPONS       (immediate)
            Create all player weapons.
         */
         case SC_STATE_RUN_CREATION:
            _sc_state_run_creation(c, g);
            break;


         /* RUN_WEAPONS
               -> RUN_WEAPONS       (weapons still flying)
               -> RUN_EXPLOSION     (might be flying; detonation)
               t> RUN_END           (terminal case; no weapons)
            Simulate the weapons in-flight.
         */
         case SC_STATE_RUN_WEAPONS:
            _sc_state_run_weapons(c, g);
            break;


         /* RUN_EXPLOSION
               -> RUN_EXPLOSION        (this weapon done; goto next)
               -> RUN_CONT_EXPLOSION   (this weapon NOT done; wait)
               t> CLEAR_EXPLOSION      (terminal; no more explosions; immediate)
            Run an explosion in progress.  The weapon explosion might opt to
            end at this point, or it may decide to continue its trend of
            annihilation for a few more cycles.  This loop terminates when
            there are no explosions remaining to be drawn; then we will
            start clearing explosions after a respectable pause.
         */
         case SC_STATE_RUN_EXPLOSION:
            _sc_state_run_explosion(c, g);
            break;


         /* RUN_CONT_EXPLOSION
               -> RUN_CONT_EXPLOSION   (loop)
               t> RUN_EXPLOSION        (terminal)
            Run an explosion that was continued.
         */
         case SC_STATE_RUN_CONT_EXPLOSION:
            _sc_state_run_cont_explosion(c, g);
            break;


         /* RUN_CLEAR_EXPLOSION
               -> RUN_CLEAR_EXPLOSION
               -> RUN_CLEAR_CONT_EXPL  (this clear NOT done; continue)
               t> RUN_LAND_DROP        (immediate)
            Clear the explosion; simulate damage to each tank. 
         */
         case SC_STATE_RUN_CLEAR_EXPLOSION:
            _sc_state_run_clear_explosion(c, g);
            break;
            

         /* RUN_CLEAR_CONT_EXPL
               -> RUN_CLEAR_CONT_EXPL  (loop)
               t> RUN_LAND_DROP        (terminal)
            Clear the explosion; simulate damage to each tank. 
         */
         case SC_STATE_RUN_CLEAR_CONT_EXPL:
            _sc_state_run_clear_cont_explosion(c, g);
            break;
            

         /* RUN_LAND_DROP
               -> LAND_DROP_STEP    (immediate)
               t> CLEAR_CHAIN       (immediate)
         */
         case SC_STATE_RUN_LAND_DROP:
            _sc_state_run_land_drop(c, g);
            break;


         /* RUN_LAND_DROP_STEP
               -> LAND_DROP_STEP    (loop condition)
               t> LAND_DROP         (might be immed)
         */
         case SC_STATE_RUN_LAND_DROP_STEP:
            _sc_state_run_land_drop_step(c, g);
            break;


         /* RUN_CLEAR_CHAIN
               -> RUN_PLAYER_DROP   (immediate)
            Destroy the explosion chain.
         */
         case SC_STATE_RUN_CLEAR_CHAIN:
            sc_expl_free_chain(&g->explosions);
            sc_game_set_state_now(c, g, SC_STATE_RUN_PLAYER_DROP);
            break;


         /* RUN_PLAYER_DROP
               -> RUN_PLAYER_DROP   (loop)
               t> RUN_PLAYER_DEATH  (terminal)
         */
         case SC_STATE_RUN_PLAYER_DROP:
            _sc_state_run_player_drop(c, g);
            break;


         /* RUN_PLAYER_DEATH
               -> RUN_PLAYER_DEATH
               t> RUN_END
            Check if any players just died.
         */
         case SC_STATE_RUN_PLAYER_DEATH:
            _sc_state_run_player_death(c, g);
            break;
            
            
         /* RUN_END
            
            ==Sequential Mode==
               -> TURN_PL_NEXT      (immediate)
               
            ==Synchronous Mode==
               -> TURN_END          (immediate)
         */
         case SC_STATE_RUN_END:
            _sc_state_run_end(c, g);
            break;


         /*    End of (known) game states    */


         default:
            sc_game_reinstate(g, SC_DELAY_LONG);
            break;

      }

      /*SC_PROFILE_END*/
 
   }

}
