/* $Header: /fridge/cvs/xscorch/sgame/splayer.c,v 1.12 2001/04/12 06:51:28 jacob Exp $ */
/*
   
   xscorch - splayer.c        Copyright(c) 2001,2000 Justin David Smith
   justins(at)chaos2.org      http://chaos2.org/

   Scorched Player information
    

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdlib.h>
#include <stdio.h>
#include <sinventory.h>
#include <saccessory.h>
#include <sexplosion.h>
#include <stankpro.h>
#include <seconomy.h>
#include <splayer.h>
#include <swindow.h>
#include <sconfig.h>
#include <sweapon.h>
#include <sshield.h>
#include <sspill.h>
#include <sland.h>
#include <math.h>
#include <sutil/srand.h>
#include <sutil/sstr.h>
#include <snet/snet.h>



void sc_player_init_game(const sc_config *c, sc_player *p) {
/* Game initialization for this player */

   /* Initialise the player's currency and inventory */
   p->money = c->economy->initialcash;
   sc_inventory_init(c, p->inventory);
   
   /* No wins (yet) */
   p->numwins = 0;
   p->kills = 0;
   p->suicides = 0;
   p->ac_state = 0;
   p->turret = 0;
   p->power  = 0;
   
   /* Set player arms level */
   p->armslevel = c->weapons->armslevel;
   
   /* Initialise AI */
   sc_ai_init_game(c, p);

}



void sc_player_init_round(sc_config *c, sc_player *p) {

   /* Determine the tank's position on the field */
   p->field_index = game_lrand(c->numplayers);
   while(c->field_position[p->field_index] >= 0) {
      p->field_index = (p->field_index + 1) % c->numplayers;
   }
   c->field_position[p->field_index] = p->index;

   /* Initialise player X, Y coordinates -- we may need to modify
      the land a bit so the tank is initially on flat ground.  */
   p->x = rint((p->field_index + 1) * c->fieldwidth / (c->numplayers + 1.0));
   p->y = sc_land_avg_height_around(c->land, p->x, c->fieldheight, p->tank->radius);
   sc_land_level_around(c, c->land, p->x, p->tank->radius, p->y);

   /* Setup player turret and initial power */
   p->turret = game_lrand(181);
   p->power = SC_PLAYER_MAX_POWER / 5;
   
   /* Set initial life status */
   p->life = SC_PLAYER_MAX_LIFE;
   p->dead = false;
   p->armed = false;
   p->killedby = -1;
   
   /* Setup fuel level */
   p->fuel = 0;
   
   /* Clear contact triggers */
   p->contacttriggers = false;
   
   /* Load weapons and accessories */
   p->selweapon = SC_WEAPON_DEFAULT;
   sc_weapon_free_chain(&p->weapons);
   sc_shield_free(&p->shield);
   
   /* Initialise player shields if autodefence */
   sc_player_activate_auto_def(c, p);
   
   /* Money is as it was at beginning of the round */
   p->oldmoney = p->money;

   /* Initialise AI */
   sc_ai_init_round(c, p);

}



void sc_player_init_turn(const sc_config *c, sc_player *p) {

   /* Clear any land that got stuck on our profile (sanity check) */
   /* This sanity check must not be removed, as it is the only thing
      protecting tanks from dirt bombs that might PERMANENTLY lock the
      tank into position, as a consequence. */
   sc_land_clear_profile(c, c->land, p);

   /* make sure we aren't occupying someone else's space */
   if(!sc_player_passable(c, p, p->x, p->y)) {
      printf("warning: Player %d is LOCKED, cannot resolve   ** This is a bug **\n", p->index);
      printf("warning: ** If there are any levitating tanks, well, this is the cause.\n");
   } /* levitation check */

   /* make sure tanks aren't levitating */
   if(sc_player_passable(c, p, p->x, p->y - 1)) {
      printf("warning: Player %d is LEVITATING, cannot resolve   ** This is a bug **\n", p->index);
      printf("warning: ** If there are any levitating tanks, well, this is the cause.\n");
   } /* levitation check */
   if(sc_land_support(c, c->land, p->x, p->y, p->tank->radius, p->tank->shelfsize) != 0) {
      printf("warning: Player %d is UNSTABLE, cannot resolve   ** This is a bug **\n", p->index);
      printf("warning: ** If there are any levitating tanks, well, this is the cause.\n");
   } /* levitation check */

}



sc_player *sc_player_new(int index, const sc_tank_profile *tank) {
/* Create a new player */

   sc_player *p;

   /* Allocate player */
   p = (sc_player *)malloc(sizeof(sc_player));
   if(p == NULL) return(NULL);

   /* Initialize player name and ID number */
   snprintfn(p->name, sizeof(p->name), "Player %d", index + 1);
   p->name[SC_PLAYER_NAME_LENGTH - 1] = '\0';
   p->index  = index;
   p->aitype = SC_AI_HUMAN;
   p->tank   = tank;

   /* Allocate the inventory */
   p->inventory = sc_inventory_new();
   if(p->inventory == NULL) return(NULL);

   /* No weapons or accessories, by default */
   p->armed = false;
   p->weapons = NULL;
   p->shield = NULL;
   p->turret = 0;
   p->power  = 0;

   /* Setup AI state */
   p->ai = sc_ai_new();

   return(p);

}



void sc_player_free(sc_player **p) {
/* Remove a player from the game */

   if(p == NULL || *p == NULL) return;
   sc_inventory_free(&(*p)->inventory);
   sc_weapon_free_chain(&(*p)->weapons);
   sc_shield_free(&(*p)->shield);
   sc_ai_free(&(*p)->ai);
   free(*p);
   *p = NULL;

}



void sc_player_advance_power(const sc_config *c, sc_player *p, int delta) {

   /* Sanity checks */
   if(c == NULL || p == NULL) return;

   /* Update power */
   p->power += delta;
   if(p->power > SC_PLAYER_MAX_POWER) p->power = SC_PLAYER_MAX_POWER;
   else if(p->power < 0) p->power = 0;

   /* Update state */
   #if USE_NETWORK
      sc_net_client_send_orders(c, c->client, p->index);
   #endif
   sc_status_update(c->window, p);

}



void sc_player_advance_turret(const sc_config *c, sc_player *p, int delta) {

   /* Sanity checks */
   if(c == NULL || p == NULL) return;

   /* Update turret angle */
   p->turret += delta;
   while(p->turret > 180) p->turret -= 180;
   while(p->turret < 0) p->turret += 180;

   /* Update state */
   #if USE_NETWORK
      sc_net_client_send_orders(c, c->client, p->index);
   #endif
   sc_status_update(c->window, p);

}



void sc_player_advance_weapon(const sc_config *c, sc_player *p, int delta) {

   const sc_weapon_info *info;

   /* Sanity checks */
   if(c == NULL || p == NULL) return;
   
   info = sc_weapon_lookup(c->weapons, p->selweapon, SC_WEAPON_LIMIT_NONE);
   if(info == NULL) {
      printf("warning: player %i has selected invalid weapon, aborting search\n", p->index);
      return;
   }

   /* Cycle through the weapons */
   do {
      if(delta > 0) info = sc_weapon_next(c->weapons, info->ident, SC_WEAPON_LIMIT_ALL);
      else info = sc_weapon_prev(c->weapons, info->ident, SC_WEAPON_LIMIT_ALL);
      if(info == NULL) {
         printf("warning: player %i has no weapons (at all!), aborting search\n", p->index);
         break;
      }
      sc_player_set_weapon(c, p, info);
   } while(info->inventories[p->index] <= 0);

   /* Update state */
   #if USE_NETWORK
      sc_net_client_send_orders(c, c->client, p->index);
   #endif
   sc_status_update(c->window, p);

}



void sc_player_set_power(const sc_config *c, sc_player *p, int power) {

   /* Sanity checks */
   if(c == NULL || p == NULL) return;
   
   /* Update power; return if no change */
   if(power < 0) power = 0;
   if(power > SC_PLAYER_MAX_POWER) power = SC_PLAYER_MAX_POWER;
   if(p->power == power) return;
   p->power = power;

   /* Update state */
   #if USE_NETWORK
      sc_net_client_send_orders(c, c->client, p->index);
   #endif
   sc_status_update(c->window, p);

}



void sc_player_set_turret(const sc_config *c, sc_player *p, int turret) {

   /* Sanity checks */
   if(c == NULL || p == NULL) return;
   
   /* Update turret angle; return if no change */
   while(turret < 0) turret += 180;
   while(turret > 180) turret -= 180;
   if(p->turret == turret) return;
   p->turret = turret;

   /* Update state */
   #if USE_NETWORK
      sc_net_client_send_orders(c, c->client, p->index);
   #endif
   sc_status_update(c->window, p);

}



void sc_player_set_weapon(const sc_config *c, sc_player *p, const sc_weapon_info *info) {

   /* Sanity checks */
   if(c == NULL || p == NULL || info == NULL) return;

   /* Make sure index is sane, and player has that weapon */
   if(info->inventories[p->index] <= 0) return;
   p->selweapon = info->ident;

   /* Update state */
   #if USE_NETWORK
      sc_net_client_send_orders(c, c->client, p->index);
   #endif
   sc_status_update(c->window, p);

}



bool sc_player_activate_shield(const sc_config *c, sc_player *p) {
/* sc_player_activate_shield
   Activates the best player shield available.  If unable to activate,
   then false is returned.  Any existing shield will be destroyed, even
   if fully powered; so use with caution. */

   int i;            /* Iterator variable */
   
   /* Sanity check */
   if(c == NULL || p == NULL) return(false);

   /* Search for the best shield in inventory */
   i = sc_accessory_count(c->accessories) - 1;
   do {
      while(i >= 0 && !SC_ACCESSORY_IS_SHIELD(sc_accessory_lookup(c->accessories, i))) --i;
      if(i >= 0 && p->inventory->accessories[i] > 0) {
         /* Activate this shield; release any existing shields */
         sc_shield_free(&p->shield);
         --(p->inventory->accessories[i]);
         p->shield = sc_shield_new(sc_accessory_lookup(c->accessories, i));
         /* Update state */
         #if USE_NETWORK
            sc_net_client_send_shields(c, c->client, p->index);
         #endif
         sc_status_update(c->window, p);
         return(true);
      }
      --i;
   } while(i >= 0);
   
   /* Failure */
   return(false);

}



bool sc_player_activate_auto_def(const sc_config *c, sc_player *p) {
/* sc_player_activate_auto_def
   Activates player auto-defense, if available and if any shields are
   currently available. */

   int i;            /* Iterator variable */
   
   /* Sanity check */
   if(c == NULL || p == NULL) return(false);

   /* Search for the best shield in inventory */
   i = sc_accessory_count(c->accessories) - 1;
   do {
      while(i >= 0 && !SC_ACCESSORY_IS_AUTO_DEF(sc_accessory_lookup(c->accessories, i))) --i;
      if(i >= 0 && p->inventory->accessories[i] > 0) {
         /* Activate this auto-defense */
         if(sc_player_activate_shield(c, p)) {
            /* Success; this auto-defense was used. */
            --(p->inventory->accessories[i]);
            return(true);
         }
      }
      --i;
   } while(i >= 0);

   /* Failure */
   return(false);

}



void sc_player_set_contact_triggers(const sc_config *c, sc_player *p, bool flag) {
/* sc_player_set_contact_triggers
   Toggles the contact-trigger flag automagically.  */
   
   const sc_accessory_info *info;/* Acc. information */
   int i;                     /* Iterator */

   /* Sanity check */
   if(c == NULL || p == NULL) return;

   /* Update the flag */
   p->contacttriggers = flag;
   
   /* Check to see if we have any contact triggers to use. */
   if(p->contacttriggers) {
      i = sc_accessory_count(c->accessories) - 1;
      flag = false;
      while(i >= 0 && !flag) {
         /* Make sure the damn thing is a batch-o-contact-triggers */
         info = sc_accessory_lookup(c->accessories, i);
         if(SC_ACCESSORY_IS_TRIGGER(info)) {
            if(p->inventory->accessories[i] > 0) {
               /* We're okay */
               flag = true;
            }
         }
         --i;
      }
      p->contacttriggers = flag;
   }
   
   /* Make sure player flags are transmitted over network */
   #if USE_NETWORK
      sc_net_client_send_flags(c, c->client, p->index);
   #endif
   
   /* Update status window. */
   sc_status_update(c->window, p);

}



bool sc_player_use_contact_trigger(const sc_config *c, sc_player *p) {
/* sc_player_use_contact_trigger
   Try to use a contact trigger.  If successful, return true. */
   
   const sc_accessory_info *info;/* Acc. information */
   int i;                     /* Iterator */

   /* Sanity check */
   if(c == NULL || p == NULL) return(false);

   /* Are they enabled? */
   if(!p->contacttriggers) return(false);
   
   /* Check to see if we have any contact triggers to use. */
   i = sc_accessory_count(c->accessories) - 1;
   while(i >= 0) {
      /* Make sure the damn thing is a batch-o-contact-triggers */
      info = sc_accessory_lookup(c->accessories, i);
      if(SC_ACCESSORY_IS_TRIGGER(info)) {
         if(p->inventory->accessories[i] > 0) {
            /* We're okay; use one! */
            --p->inventory->accessories[i];
            sc_player_set_contact_triggers(c, p, true);
            return(true);
         }
      }
      --i;
   }
   
   /* Failure; sorry. */
   return(false);

}



void sc_player_set_position(const sc_config *c, sc_player *p, int x, int y) {

   /* Sanity checks */
   if(c == NULL || p == NULL) return;
   
   /* Make sure tank has actually moved */
   if(p->x == x && p->y == y) return;
   
   /* Update tank position on screen */
   sc_window_undraw_tank(c->window, p);
   p->x = x;
   p->y = y;
   sc_window_draw_tank(c->window, p);
   
   /* Transmit state by network */
   #if USE_NETWORK
      sc_net_client_send_orders(c, c->client, p->index);
   #endif /* Network? */

}



void sc_player_inc_wins(sc_config *c, sc_player *p) {

   ++p->numwins;
   p->money += c->economy->survivalbonus;

}



void sc_player_died(sc_config *c, sc_player *p) {

   p->dead = true;
   p->money -= c->economy->deathloss;

}



static bool _sc_player_drop(sc_config *c, sc_player *p, int falldist) {
/* Drop current player; return nonzero if still falling */

   int deltax;
   int dropdist;
   
   if(falldist <= 0) return(true);
   
   if(SC_CONFIG_NO_ANIM(c)) dropdist = c->fieldheight;
   else dropdist = falldist;

   if(!p->dead) {
      /* Get maximum height of land around and just below the tank */
      falldist = 0;
      while(falldist < dropdist && sc_player_passable(c, p, p->x, p->y - 1)) {
         /* Still falling vertically down */
         if(falldist == 0) sc_window_undraw_tank(c->window, p);
         ++falldist;
         --p->y;
      } /* Falling tank vertically down */
      if(falldist > 0) {
         sc_window_draw_tank(c->window, p);
         _sc_player_drop(c, p, dropdist - falldist);
         /* Assume need recurse, a tank may have been using us for support */
         return(true);
      } /* Haven't landed yet */

      /* We've landed */
      /* Might be sliding on the slope at this point */
      deltax = sc_land_support(c, c->land, p->x, p->y, p->tank->radius, p->tank->shelfsize);
      if(deltax != 0 && sc_player_passable(c, p, p->x + deltax, p->y - 1)) {

         /* We slid to one side (deltax > 0, slide right) */
         sc_window_undraw_tank(c->window, p);
         p->x += deltax;
         --p->y;
         sc_window_draw_tank(c->window, p);
         falldist = dropdist - 4 * abs(deltax);
         _sc_player_drop(c, p, falldist);
         /* Assume need recurse, a tank may have been using us for support */
         return(true);
      } /* Tank slid on a steep slope */
   } /* Tank isn't dead */

   /* Tank has settled onto the ground */
   return(false);

}



bool sc_player_drop_all(sc_config *c) {
/* Drop all players */

   bool needsrecurse;
   int i;

   needsrecurse = 0;
   i = c->numplayers - 1;
   while(i >= 0) {
      needsrecurse = _sc_player_drop(c, c->players[i], SC_TANK_MAX_DROP_PER_CYCLE) || needsrecurse;
      --i;
   }

   return(needsrecurse);

}



static void _sc_player_damage(sc_config *c, sc_player *p, const sc_explosion *e) {

   int damage;
   
   /* If player is already dead, then there's nothing to do */
   if(p->dead || p->life <= 0) return;

   /* How much damage was taken? */
   damage = sc_expl_damage_at_point(c->land, e, p->x, p->y);
   if(damage <= 0) return;
   
   /* Check if the shields absorbed the explosion for us. */
   damage = sc_shield_absorb_explosion(p, e, damage);
   
   /* Take any remaining damage ourselves */
   p->life -= damage;

   /* Tank took a direct or partial hit.  Damn. */
   p->money -= c->economy->damageloss;
   if(p->life <= 0) {
      p->life = 0;  /* We died */
      p->killedby = e->playerid;
      /* Who killed us? */
      if(e->playerid == p->index) {
         /* It was a suicide */
         ++c->players[e->playerid]->suicides;
         p->money -= c->economy->suicideloss;
      } else {
         /* Give opponent some points for killing us */
         ++c->players[e->playerid]->kills;
         c->players[e->playerid]->money += c->economy->killbonus;
      }
   }
   
   /* Give points to other player for damaging us */
   if(e->playerid != p->index) {
      c->players[e->playerid]->money += c->economy->damagebonus;
   } /* Make sure not self */

}



void sc_player_damage_all(sc_config *c, const sc_explosion *e) {

   int i;

   for(i = c->numplayers - 1; i >= 0; --i) {
      _sc_player_damage(c, c->players[i], e);
   }

}



sc_player **sc_player_random_order(sc_config *c, sc_player **playerlist) {

   int order[SC_MAX_PLAYERS];
   int i;
   int j;
   
   for(i = 0; i < c->numplayers; ++i) {
      order[i] = i;
   }
   
   for(i = 0; i < c->numplayers; ++i) {
      for(j = 0; j < c->numplayers; ++j) {
         if(i != j && game_lrand(100) < 50) {
            order[i] += order[j];
            order[j] = order[i] - order[j];
            order[i] = order[i] - order[j];
         }
      }
   }
   
   for(i = 0; i < c->numplayers; ++i) {
      playerlist[i] = c->players[order[i]];
   }
   
   return(playerlist);
   
}



sc_player **sc_player_winner_order(sc_config *c, sc_player **playerlist) {

   sc_player *tmp;
   int i;
   int j;

   sc_player_random_order(c, playerlist);
   
   for(i = 1; i < c->numplayers; ++i) {
      for(j = 0; j < i; ++j) {
         if(playerlist[i]->kills - playerlist[i]->suicides < playerlist[j]->kills - playerlist[j]->suicides) {
            tmp = playerlist[i];
            playerlist[i] = playerlist[j];
            playerlist[j] = tmp;
         }
      }
   }
   
   return(playerlist);
   
}



sc_player **sc_player_loser_order(sc_config *c, sc_player **playerlist) {

   sc_player *tmp;
   int i;
   int j;

   sc_player_random_order(c, playerlist);
   
   for(i = 1; i < c->numplayers; ++i) {
      for(j = 0; j < i; ++j) {
         if(playerlist[i]->kills - playerlist[i]->suicides > playerlist[j]->kills - playerlist[j]->suicides) {
            tmp = playerlist[i];
            playerlist[i] = playerlist[j];
            playerlist[j] = tmp;
         }
      }
   }
   
   return(playerlist);
   
}



int sc_player_total_fuel(const sc_accessory_config *ac, const sc_player *p) {

   const sc_accessory_info *a;
   int fuel;
   int i;

   if(p == NULL) return(0);

   fuel = p->fuel;
   for(i = 0; i < sc_accessory_count(ac); ++i) {
      a = sc_accessory_lookup(ac, i);
      if(SC_ACCESSORY_IS_FUEL(a)) {
         fuel += a->fuel * p->inventory->accessories[i];
      }
   }
   
   return(fuel);

}



static bool _sc_player_consume_fuel(const sc_accessory_config *ac, sc_player *p) {

   const sc_accessory_info *a;
   int i;

   if(p == NULL) return(false);

   for(i = 0; i < sc_accessory_count(ac) && p->fuel <= 0; ++i) {
      a = sc_accessory_lookup(ac, i);
      if(SC_ACCESSORY_IS_FUEL(a)) {
         while(p->inventory->accessories[i] > 0 && p->fuel <= 0) {
            p->fuel += a->fuel;
            --p->inventory->accessories[i];
         }
      }
   }
   
   if(p->fuel <= 0) return(false);
   --p->fuel;
   return(true);

}



bool sc_player_move(const sc_config *c, sc_player *p, int delta) {
/* sc_player_move
   Move the player's tank by means of consuming fuel cells.  This call
   is only used when moving by using fuel, and should not be used to
   set player position by ``forces''.  If the player does not have any
   fuel, then we will give up and not attempt the move at all.  */

   int y;      /* Height of the land at the destination */

   /* Sanity checks */
   if(c == NULL || p == NULL || delta == 0) return(false);

   /* Check that the height is not too high a climb for a heavy tank. */
   y = sc_land_height_around(c->land, p->x + delta, c->fieldheight, p->tank->radius);
   if(y > p->y + SC_TANK_CLIMB_HEIGHT) return(false);

   /* Check that the land is otherwise ``passable'', i.e. no walls
      or other tanks in our way.  */
   if(!sc_player_passable(c, p, p->x + delta, y)) return(false);

   /* Attempt to consume fuel. If this succeeds then the move goes thru */
   if(!_sc_player_consume_fuel(c->accessories, p)) return(false);
   sc_player_set_position(c, p, p->x + delta, y);
   return(true);

}



int sc_player_turret_x(const sc_player *p, int angle) {

   return(p->x + (p->tank->radius + 3) * cos(angle * M_PI / 180));

}



int sc_player_turret_y(const sc_player *p, int angle) {

   return(p->y + (p->tank->radius + 3) * sin(angle * M_PI / 180));

}



void sc_player_death(const sc_config *c, const sc_player *p, sc_explosion **e) {

   sc_explosion *expl;

   if(c == NULL || p == NULL || e == NULL) return;

   switch(game_lrand(10)) {

      case 0:
         /* Just say it with napalm ... */
         expl = sc_expl_new(p->x, p->y, c->weapons->scaling * SC_WEAPON_NAPALM_RADIUS, SC_WEAPON_NAPALM_FORCE, p->killedby, SC_EXPLOSION_NAPALM);
         expl->data = sc_spill_new(c, c->land, SC_WEAPON_NAPALM_LIQUID, expl->centerx, expl->centery);
         sc_expl_add(e, expl);
         break;
         
      default:
         /* Just the usual, 3-stage detonation */
         sc_expl_add(e, sc_expl_new(p->x, p->y, c->weapons->scaling * SC_WEAPON_SMALL_EXPLOSION, SC_WEAPON_SMALL_FORCE, p->killedby, SC_EXPLOSION_NORMAL));
         if(game_drand() < 0.5) {
            sc_expl_add(e, sc_expl_new(p->x, p->y, c->weapons->scaling * SC_WEAPON_MEDIUM_EXPLOSION, SC_WEAPON_MEDIUM_FORCE, p->killedby, SC_EXPLOSION_NORMAL));
            if(game_drand() < 0.1) {
               sc_expl_add(e, sc_expl_new(p->x, p->y, c->weapons->scaling * SC_WEAPON_LARGE_EXPLOSION, SC_WEAPON_LARGE_FORCE, p->killedby, SC_EXPLOSION_PLASMA));
            } /* Do third stage? */
         } /* Do second stage? */
         break;

   } /* End of switch */
   
}



bool sc_player_passable(const sc_config *c, const sc_player *p, int x, int y) {

   const char *data;
   int radius;
   int cx;
   int cy;

   if(c == NULL || p == NULL) return(false); 
     
   data = p->tank->data;
   radius = p->tank->radius;
   for(cy = radius; cy >= 0; --cy) {
      for(cx = radius; cx >= -radius; --cx, ++data) {
         if(*data != SC_TANK_PROFILE_CLEAR) {
            if(!sc_land_passable_point(c, p, x + cx, y + cy)) return(false);
         }
      }
   }

   return(true);

}



#define  SC_PLAYER_NUM_MESSAGES  36
static const char *_sc_player_messages[SC_PLAYER_NUM_MESSAGES + 1] = {
   "In times of trouble, go with what you know.",
   "Die!",
   "You're toast!",
   "Banzai!",
   "From Hell's heart I stab at thee...",
   "I didn't do it.  Nobody saw me do it.",
   "Make my day.",
   "Charge!",
   "Attack!",
   "You're outta here.",
   "Freeze, or I'll shoot!",
   "Ha ha ha.",
   "We come in peace - Shoot to kill!",
   "In your face!",
   "I love the smell of Napalm in the morning.",
   "Victory!",
   "Show some respect.",
   "Just who do you think you are?",
   "Look out below!",
   "Knock, Knock.",
   "Look over there.",
   "Guess what's coming for dinner?",
   "Merry Christmas.",
   "Open wide!",
   "Here goes nothing...",
   "Don't worry, it isn't a live round.",
   "I wonder what this button does?",
   "Don't take this personally.",
   "Would this make you mad?",
   "I could spare you, but why?",
   "My bomb is bigger than yours.",
   "Don't forget about me!",
   "Take this!",
   "This screen ain't big enough for the both of us.",
   "Say \"Arrgghhhhh....\"",
   "I shall oil my turret with your blood.",
   NULL
};   



#define  SC_PLAYER_NUM_DEATH_MESSAGES  28
static const char *_sc_player_death_messages[SC_PLAYER_NUM_DEATH_MESSAGES + 1] = {
   "Aargh!",
   "I hate it when that happens.",
   "One direct hit can ruin your whole day.",
   "Ouch.",
   "Oh no, not again.",
   "Another one bites the dust.",
   "Goodbye.",
   "Farewell, cruel world.",
   "Another day, another bomb.",
   "Why does everything happen to me?",
   "I've got a bad feeling about this.",
   "What was that noise?",
   "Mama said there'd be days like this.",
   "Its just one of those days...",
   "I see a bright light...",
   "I let you hit me!",
   "I didn't want to live anyway.",
   "Was that as close as I think it was?",
   "Join the army, see the world they said.",
   "I thought you liked me?",
   "Such senseless violence!  I don't understand it.",
   "I think this guy's a little crazy.",
   "Somehow I don't feel like killing anymore.",
   "Gee... thanks.",
   "I've fallen and I can't get up!",
   "I'll be back...",
   "Hey - I've got lawyers.",
   "Time to call 1-900-SUE-TANK.",
   NULL
};



const char *sc_player_talk(const sc_config *c, const sc_player *p) {

   switch(c->options.talk) {
      case SC_CONFIG_TALK_OFF:
         break;
      case SC_CONFIG_TALK_COMPUTERS:
         if(p->aitype == SC_AI_HUMAN) break;
      default:
         if(game_lrand(100) < c->options.talkprob) {
            return(_sc_player_messages[game_lrand(SC_PLAYER_NUM_MESSAGES)]);
         }
   }
   return(NULL);

}



const char *sc_player_death_talk(const sc_config *c, const sc_player *p) {

   switch(c->options.talk) {
      case SC_CONFIG_TALK_OFF:
         break;
      case SC_CONFIG_TALK_COMPUTERS:
         if(p->aitype == SC_AI_HUMAN) break;
      default:
         if(game_lrand(100) < c->options.talkprob) {
            return(_sc_player_death_messages[game_lrand(SC_PLAYER_NUM_DEATH_MESSAGES)]);
         }
   }
   return(NULL);

}



