/* $Header: /fridge/cvs/xscorch/sgame/sinventory.c,v 1.9 2001/04/09 04:48:32 jacob Exp $ */
/*

   xscorch - sinventory.c     Copyright(c) 2001,2000 Justin David Smith
                              Copyright(c) 2001      Jacob Luna Lundberg
   justins(at)chaos2.org      http://chaos2.org/
   jacob(at)chaos2.org        http://chaos2.org/~jacob

   Scorched Player inventory


   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <sinventory.h>
#include <seconomy.h>
#include <splayer.h>
#include <sconfig.h>
#include <sweapon.h>
#include <saccessory.h>



/* TEMP - this will need to die with the sc_inventory struct - JL */
void sc_inventory_init(const sc_config *c, sc_inventory *inv) {
/* sc_inventory_init
   Initialise all inventory entries to zero (except for infinite items).  */

   int i;         /* Iterator variable. */

   /* Allocate the accessory count list */
   inv->accessories = (int *)realloc(inv->accessories, sc_accessory_count(c->accessories) * sizeof(int));

   /* Initialise the accessory counts to zero. */
   for(i = 0; i < sc_accessory_count(c->accessories); ++i) {
      inv->accessories[i] = 0;
   } /* Initializing accessory counts. */

}



sc_inventory *sc_inventory_new(void) {
/* sc_inventory_new
   Create a new inventory list for this player.
   NOTE that this list will be unuseable until you call sc_inventory_init() */

   sc_inventory *inv;

   /* Allocate the inventory struct */
   inv = (sc_inventory *)malloc(sizeof(sc_inventory));
   if (inv == NULL)  return (NULL);

   inv->accessories = NULL;

   return(inv);

}



void sc_inventory_free(sc_inventory **inv) {
/* sc_inventory_free
   Releases the specified inventory list.  */

   if(inv == NULL || *inv == NULL) return;
   free(*inv);
   *inv = NULL;

}



static int _sc_inventory_quantity(int current, int bundlesize) {
/* sc_inventory_quantity
   Determine the number of items the user can purchase of this item. */

   if(current + bundlesize <= SC_INVENTORY_MAX_ITEMS) {
      /* User can buy an entire bundle. */
      return(bundlesize);
   } else {
      /* User can only purchase part of the bundle. */
      return(SC_INVENTORY_MAX_ITEMS - current);
   } /* Did we break a bundle? */

}



static int _sc_inventory_purchase_price(const sc_item_info *info, int quantity) {
/* sc_inventory_purchase_price
   Price required to purchase this item.  */

   int unitprice;       /* Cost of a single item. */

   if(quantity < info->bundle) {
      /* Bundle was split up; implement a markup. */
      unitprice = info->price / info->bundle;
      return(unitprice * quantity * SC_ECONOMY_SPLIT_MARKUP);
   } else {
      /* Bundle not split; return full price. */
      return(info->price);
   }

}



static int _sc_inventory_sale_price(const sc_item_info *info, int quantity) {
/* sc_inventory_sale_price
   Price required to sell this item.  */

   int unitprice;       /* Cost of a single item. */

   if(quantity < info->bundle) {
      /* Bundle is partial. */
      unitprice = info->price / info->bundle;
      return(unitprice * quantity * SC_ECONOMY_SELL_MARKUP);
   } else {
      /* Bundle complete; still markdown a bit */
      return(info->price * SC_ECONOMY_SELL_MARKUP);
   }

}



bool sc_inventory_can_buy_weapon(const sc_player *p, const sc_weapon_info *info, int budget) {
/* sc_inventory_can_buy_weapon
   Determine if this player can buy the specified weapon.  Returns true
   if the purchase can be made.  Budget is the amount of money allocated for
   the purchase, or may be -1 (player's total resources will be allowed into
   the calculation). */

   int count;        /* Number of weapons to buy */
   int cost;         /* Total cost to buy */

   /* Does the weapon exist? */
   if(info == NULL) return(false);

/* TEMP - This block isn't needed anymore, or so the theory goes... - JL */
#if 0
   /* Check arms level */
   if(p->armslevel < info->armslevel) return(false);
#endif

   /* Are we maxed out on inventory? */
   count = _sc_inventory_quantity(info->inventories[p->index], info->bundle);
   if(count == 0) return(false);

   /* Can we afford the weapon? */
   cost = _sc_inventory_purchase_price((sc_item_info *)info, count);
   if(cost > p->money) return(false);

   /* Is the weapon within our budget? */
   if(budget >= 0 && cost > budget) return(false);

   /* We can make this purchase. */
   return(true);

}



bool sc_inventory_buy_weapon(sc_player *p, const sc_weapon_info *info) {
/* sc_inventory_buy_weapon
   Buys the specified weapon.  Returns true if the purchase was successful. */

   int *current;     /* Pointer to current inventory element */
   int count;        /* Number of weapons to buy */
   int cost;         /* Total cost to buy */

   /* Can we buy this weapon? */
   if(!sc_inventory_can_buy_weapon(p, info, SC_INVENTORY_INFINITE)) return(false);

   /* Get weapon purchase information.. */
   current = (int *)&info->inventories[p->index];
   count = _sc_inventory_quantity(*current, info->bundle);
   cost = _sc_inventory_purchase_price((sc_item_info *)info, count);

   /* Make the purchase */
   *current += count;
   p->money -= cost;

   /* Return success. */
   return(true);

}



bool sc_inventory_can_buy_accessory(const sc_accessory_config *ac, const sc_player *p, int index, int budget) {
/* sc_inventory_can_buy_accessory
   Determine if this player can buy the specified accessory.  Returns true
   if the purchase can be made.  Budget is the amount of money allocated for
   the purchase, or may be -1 (player's total resources will be allowed into
   the calculation). */

   const sc_accessory_info *info;   /* accessory information */
   int count;        /* Number of accessories to buy */
   int cost;         /* Total cost to buy */

   /* Does the accessory exist? */
   info = sc_accessory_lookup(ac, index);
   if(info == NULL) return(false);

   /* Is the accessory a one-time buy we already own? */
   if(p->ac_state & info->state) return(false);

   /* Check arms level */
   if(p->armslevel < info->armslevel) return(false);

   /* Are we maxed out on inventory? */
   count = _sc_inventory_quantity(p->inventory->accessories[index], info->bundle);
   if(count == 0) return(false);

   /* Can we afford the accessory? */
   cost = _sc_inventory_purchase_price((sc_item_info *)info, count);
   if(cost > p->money) return(false);

   /* Is the accessory in our budget? */
   if(budget >= 0 && cost > budget) return(false);

   /* We can make this purchase. */
   return(true);

}



bool sc_inventory_buy_accessory(const sc_accessory_config *ac, sc_player *p, int index) {
/* sc_inventory_buy_accessory
   Buys the specified accessory.  Returns true if the purchase was successful. */

   const sc_accessory_info *info;   /* Weapon information */
   int *current;     /* Pointer to current inventory element */
   int count;        /* Number of accessories to buy */
   int cost;         /* Total cost to buy */

   /* Can we buy this accessory? */
   if(!sc_inventory_can_buy_accessory(ac, p, index, SC_INVENTORY_INFINITE)) return(false);

   /* Get accessory purchase information.. */
   info = sc_accessory_lookup(ac, index);
   current = &p->inventory->accessories[index];
   count = _sc_inventory_quantity(*current, info->bundle);
   cost = _sc_inventory_purchase_price((sc_item_info *)info, count);

   /* Make the purchase */
   *current += count;
   p->money -= cost;

   /* Add any attributes granted by the accessory. */
   p->ac_state |= info->state;

   /* Return success. */
   return(true);

}



bool sc_inventory_can_sell_weapon(const sc_player *p, const sc_weapon_info *info) {
/* sc_inventory_can_sell_weapon
   Determine if this player can sell the specified weapon.
   Returns true if the sale can be made. */

   /* Does the weapon exist? */
   if(info == NULL) return(false);

   /* Are we min'd out on inventory? */
   if(info->inventories[p->index] == 0)
      return(false);

   /* Is the weapon infinite? */
   if(SC_WEAPON_IS_INFINITE(info))
      return(false);

   /* We can make this transaction. */
   return(true);

}



bool sc_inventory_sell_weapon(sc_player *p, const sc_weapon_info *info) {
/* sc_inventory_sell_weapon
   Buys the specified weapon.  Returns true if the transaction was successful. */

   int *current;     /* Pointer to current inventory element */
   int count;        /* Number of weapons to buy */
   int cost;         /* Total cost to buy */

   /* Can we sell this weapon? */
   if(!sc_inventory_can_sell_weapon(p, info)) return(false);

   /* Get weapon purchase information.. */
   current = (int *)&info->inventories[p->index];
   count = *current;
   if(count > info->bundle) count = info->bundle;
   cost = _sc_inventory_sale_price((sc_item_info *)info, count);

   /* Sell the item */
   *current -= count;
   p->money += cost;

   /* Return success. */
   return(true);

}



bool sc_inventory_can_sell_accessory(const sc_accessory_config *ac, const sc_player *p, int index) {
/* sc_inventory_can_sell_accessory
   Determine if this player can sell the specified accessory.
   Returns true if the sale can be made. */

   const sc_accessory_info *info;   /* Accessory information */

   /* Does the accessory exist? */
   info = sc_accessory_lookup(ac, index);
   if(info == NULL) return(false);

   /* Are we min'd out on inventory? */
   if(p->inventory->accessories[index] == 0)
      return(false);

   /* Is the accessory infinite? */
   if(SC_ACCESSORY_IS_INFINITE(info))
      return(false);

   /* We can make this transaction. */
   return(true);

}



bool sc_inventory_sell_accessory(const sc_accessory_config *ac, sc_player *p, int index) {
/* sc_inventory_sell_accessory
   Buys the specified accessory.  Returns true if the transaction was successful. */

   const sc_accessory_info *info;   /* Accessory information */
   int *current;     /* Pointer to current inventory element */
   int count;        /* Number of accessories to buy */
   int cost;         /* Total cost to buy */

   /* Can we sell this accessory? */
   if(!sc_inventory_can_sell_accessory(ac, p, index)) return(false);

   /* Get accessory purchase information.. */
   info = sc_accessory_lookup(ac, index);
   current = &p->inventory->accessories[index];
   count = *current;
   if(count > info->bundle) count = info->bundle;
   cost = _sc_inventory_sale_price((sc_item_info *)info, count);

   /* Remove any attributes granted by the accessory. */
   p->ac_state &= ~info->state;

   /* Sell the item */
   *current -= count;
   p->money += cost;

   /* Return success. */
   return(true);

}
