/* $Header: /fridge/cvs/xscorch/sgame/sexplosion.c,v 1.4 2001/04/08 01:18:31 justins Exp $ */
/*
   
   xscorch - sexplosion.c     Copyright(c) 2001,2000 Justin David Smith
                              Copyright(c) 2001,2000 Jacob Luna Lundberg
   justins(at)chaos2.org      http://chaos2.org/
   jacob(at)chaos2.org        http://chaos2.org/~jacob

   Scorched zone explosions
    

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <sexplosion.h>    /* Explosion header */
#include <sphysics.h>      /* Need current physics model */
#include <swindow.h>       /* For expl_cache drawing functions */
#include <sconfig.h>       /* Need to get colormap, land */  
#include <scolor.h>        /* Need colormap information */
#include <sspill.h>        /* Constructs spillage/napalm */
#include <sland.h>         /* We clear the land */


sc_explosion *sc_expl_new(int cx, int cy, int r, int force, int playerid, sc_explosion_type type) {
/* sc_expl_new
   Create a new explosion.  */

   sc_explosion *e;        /* A newly created explosion */

   /* Make sure explosion has a radius */
   if(r <= 0) return(NULL);
   
   /* Allocate memory for a new explosion */
   e = (sc_explosion *)malloc(sizeof(sc_explosion));
   if(e == NULL) return(NULL);
   
   /* Initialise variables */
   e->centerx = cx;
   e->centery = cy;
   e->radius  = r;
   e->force   = force;
   e->playerid= playerid;
   e->type    = type;

   /* By default, no chain on this weapon */   
   e->chain = NULL;
   return(e);

}



sc_explosion *sc_expl_add(sc_explosion **e, sc_explosion *add) {
/* sc_expl_add
   Adds explosion add to the _end_ of e.  */

   sc_explosion *insert_after;
   
   if(e == NULL) return(NULL);

   insert_after = *e;
   if(insert_after == NULL) {
      *e = add;
   } else {
      while(insert_after->chain != NULL) insert_after = insert_after->chain;
      insert_after->chain = add;
   }

   return(add);

}



sc_explosion *sc_expl_index(sc_explosion *e, int index) {
/* sc_expl_index */

   if(e == NULL || index < 0) return(NULL);
   while(index > 0 && e != NULL) {
      e = e->chain;
      --index;
   }
   return(e);

}



int sc_expl_count(const sc_explosion *e) {
/* sc_expl_count */

   int count = 0;
   while(e != NULL) {
      ++count;
      e = e->chain;
   }
   return(count);

}



void sc_expl_free(sc_explosion **e) {
/* sc_expl_free
   Releases the explosion at the head of the list, 
   and sets *e to its original chain pointer.  */

   sc_explosion *del;      /* Explosion to be deleted */
   
   /* Sanity checking */
   if(e == NULL || *e == NULL) return;
   
   /* Get the explosion to be deleted, and reassign *e */
   del = *e;
   *e = del->chain;
   
   /* Delete this explosion */
   free(del);

}



void sc_expl_free_chain(sc_explosion **e) {
/* sc_expl_free_chain
   Releases all explosions on this chain, at once.  */

   /* Release the whole chain */
   if(e == NULL) return;
   while(*e != NULL) sc_expl_free(e);

}



typedef struct _sc_expl_cache {
   int cacheid;
   int eradius;
} sc_expl_cache;



static inline void _sc_expl_annihilate_fill_column(sc_config *c, sc_land *l, int x, int y1, int y2) {
/* sc_expl_annihilate_fill_column
   Fills a column with land.  This fills any clear tiles in column x,
   from y1 up to y2 (y1 < y2).  This is an internal function only.  */

   const int *gradient; /* Sky gradient */
   bool dither;         /* Enable dithering? */
   int *lp;             /* Pointer into land structure at (x, y) */
   int y;               /* Current Y coordinate */

   /* Get the sky gradient */
   gradient = c->colors->gradindex[SC_GRAD_GROUND];
   dither = c->graphics.gfxdither;

   /* Boundary checks */
   if(!sc_land_translate_x(l, &x)) return;
   if(y1 < 0) y1 = 0;
   if(y2 >= c->fieldheight) y2 = c->fieldheight - 1;

   /* Boundary checks have already been performed */
   y = y1;
   lp = SC_LAND_XY(l, x, y);
   while(y <= y2) {
      if(SC_LAND_IS_SKY(*lp)) {
         *lp = SC_LAND_GROUND | sc_color_gradient_index(dither, gradient, y);
      } /* Was the tile originally sky? */
      ++lp;
      ++y;
   }

}



static void _sc_expl_annihilate_fill(sc_config *c, const sc_explosion *e, int r) {
/* sc_expl_annihilate_fill
   Fills the area specified with land.  This regenerates the 
   land buffer but it will NOT update the display screen.  */

   int cx=e->centerx;/* Center X of explosion */
   int cy=e->centery;/* Center Y of explosion */
   int dx;           /* Delta X (distance away from cx) - iterator variable */
   int dy;           /* Delta Y (distance away from cy) for _edge_ of circle */
   int rad2;         /* Radius squared */
   int rad2major2;   /* Radius^2 + the major_distance^2 */
   int min2thresh;   /* Minimum threshold to avoid redrawing columns where dx>dy */

   /* DX = major axis, DY = minor axis */
   dx = 0;           /* DX starts at zero (iterator) */
   dy = r;           /* DY is one radius away (edge of circle at cx+dx) */
   rad2 = r * r;     /* Calculate Radius Squared */
   rad2major2 = rad2;/* Radius^2 + major^2, running total */
   min2thresh = rad2 - dy; /* Minimum threshold before need to redraw edges */
   
   /* Should know that, we are incrementing DX every time.  However,
      if we call the transpose method every time as well, then we will
      be filling parts of the circle multiple times.  Hence the 
      min2thresh variable. */
   do {
      _sc_expl_annihilate_fill_column(c, c->land, cx - dx, cy - dy, cy + dy);
      _sc_expl_annihilate_fill_column(c, c->land, cx + dx, cy - dy, cy + dy);
      ++dx;
      rad2major2 -= dx + dx - 1;
      if(rad2major2 <= min2thresh) {
         _sc_expl_annihilate_fill_column(c, c->land, cx - dy, cy - dx, cy + dx);
         _sc_expl_annihilate_fill_column(c, c->land, cx + dy, cy - dx, cy + dx);
         --dy;
         min2thresh -= dy + dy;
      }
   } while(dx <= dy);

   /* Repaint everything in the "zone" */
   sc_window_paint(c->window, cx - r, cy - r, cx + r, cy + r, SC_REGENERATE_LAND);

}



static inline void _sc_expl_annihilate_clear_column(sc_config *c, sc_land *l, int x, int y1, int y2) {
/* sc_expl_annihilate_clear_column
   Clears a column of anything with the explosion flag set.  This clears
   column x, from y1 up to y2 (y1 < y2).  This is an internal function 
   only.  */

   const int *gradient; /* Sky gradient */
   int gradientflag;    /* Sky gradient flag */
   bool dither;         /* Enable dithering? */
   int *lp;             /* Pointer into land structure at (x, y) */
   int y;               /* Current Y coordinate */

   /* Get the sky gradient */
   gradient = sc_land_sky_index(c);
   gradientflag = sc_land_sky_flag(c);
   dither = c->graphics.gfxdither;

   /* Boundary checks */
   if(!sc_land_translate_x(l, &x)) return;
   if(y1 < 0) y1 = 0;
   if(y2 >= c->fieldheight) y2 = c->fieldheight - 1;

   /* Boundary checks have already been performed */
   y = y1;
   lp = SC_LAND_XY(l, x, y);
   while(y <= y2) {
      *lp = gradientflag | sc_color_gradient_index(dither, gradient, y);   
      ++lp;
      ++y;
   }

}



static void _sc_expl_annihilate_clear(sc_config *c, const sc_explosion *e, int r) {
/* sc_expl_annihilate_clear
   Clears the explosion specified.  This updates the land 
   buffer but it will NOT update the display screen.  */

   int cx=e->centerx;/* Center X of explosion */
   int cy=e->centery;/* Center Y of explosion */
   int dx;           /* Delta X (distance away from cx) - iterator variable */
   int dy;           /* Delta Y (distance away from cy) for _edge_ of circle */
   int rad2;         /* Radius squared */
   int rad2major2;   /* Radius^2 + the major_distance^2 */
   int min2thresh;   /* Minimum threshold to avoid redrawing columns where dx>dy */

   /* DX = major axis, DY = minor axis */
   dx = 0;           /* DX starts at zero (iterator) */
   dy = r;           /* DY is one radius away (edge of circle at cx+dx) */
   rad2 = r * r;     /* Calculate Radius Squared */
   rad2major2 = rad2;/* Radius^2 + major^2, running total */
   min2thresh = rad2 - dy; /* Minimum threshold before need to redraw edges */
   
   /* Should know that, we are incrementing DX every time.  However,
      if we call the transpose method every time as well, then we will
      be filling parts of the circle multiple times.  Hence the 
      min2thresh variable. */
   do {
      _sc_expl_annihilate_clear_column(c, c->land, cx - dx, cy - dy, cy + dy);
      _sc_expl_annihilate_clear_column(c, c->land, cx + dx, cy - dy, cy + dy);
      ++dx;
      rad2major2 -= dx + dx - 1;
      if(rad2major2 <= min2thresh) {
         _sc_expl_annihilate_clear_column(c, c->land, cx - dy, cy - dx, cy + dx);
         _sc_expl_annihilate_clear_column(c, c->land, cx + dy, cy - dx, cy + dx);
         --dy;
         min2thresh -= dy + dy;
      }
   } while(dx <= dy);

   /* Update the land buffer */
   sc_window_paint(c->window, cx - r, cy - r, cx + r, cy + r, SC_REGENERATE_LAND);

}



bool sc_expl_annihilate(sc_config *c, sc_explosion *e) {
/* sc_expl_annihilate 
   Annihilate a section of the screen by drawing a huge explosion to it. */

   sc_expl_cache *ca;   /* Explosion cache (standard explosions) */
   sc_spill *sp;        /* Spill information (liquid explosions) */

   /* Sanity check */
   if(c == NULL || e == NULL) return(false);
   
   /* Action depends on weapon type */
   switch(e->type) {
      case SC_EXPLOSION_SPIDER:
         /* if there is an arc to draw, draw it */
         if(e->data != NULL)
            sc_window_draw_arc(c->window, e->data, e->playerid);
         else
            fprintf(stderr, "BUG: spider with null leg arc.\n");
         /* fall through to explosion drawing */

      case SC_EXPLOSION_NORMAL:
      case SC_EXPLOSION_PLASMA:
         /* Get a new explosion cache ID, and draw it */
         if(!SC_CONFIG_GFX_FAST(c)) {
            ca = (sc_expl_cache *)malloc(sizeof(sc_expl_cache));
            if(ca == NULL) return(false);
            if(SC_CONFIG_ANIM(c)) ca->eradius = 0;
            else ca->eradius = e->radius;
            ca->cacheid = sc_expl_cache_new(c->window, e->radius, e->type);
            e->cache = ca;
            return(true);
         } /* Only executed if not in fast mode */
         return(false);

      case SC_EXPLOSION_NAPALM:
         /* Retrieve the napalm spill */
         sp = (sc_spill *)e->data;
         if(!SC_CONFIG_GFX_FAST(c) && sp != NULL) {
            if(SC_CONFIG_ANIM(c)) {
               sc_window_draw_napalm_frame(c->window, 
                                           sp->spillx, sp->spilly, 
                                           MIN(sp->size, SC_EXPL_LIQUID_STEP));
               sp->index = SC_EXPL_LIQUID_STEP;
            } else {
               sc_window_draw_napalm_frame(c->window, sp->spillx, sp->spilly, sp->size);
               sp->index = sp->size + 1;
            }
            return(true);
         } /* Only attempt the draw if spill pointer valid */
         return(false);

      case SC_EXPLOSION_LIQ_DIRT:
         /* Retrieve the liquid dirt spill */
         sp = (sc_spill *)e->data;
         if(!SC_CONFIG_GFX_FAST(c) && sp != NULL) {
            if(SC_CONFIG_ANIM(c)) {
               sc_land_create_dirt(c, c->land, 
                                   sp->spillx, sp->spilly, 
                                   MIN(sp->size, SC_EXPL_LIQUID_STEP));
               sp->index = SC_EXPL_LIQUID_STEP;
            } else {
               sc_land_create_dirt(c, c->land, sp->spillx, sp->spilly, sp->size);
               sp->index = sp->size + 1;
            }
            return(true);
         } /* Only attempt the draw if spill pointer valid */
         return(false);

      case SC_EXPLOSION_DIRT:
         /* Animate a circular explosion that is filling up */
         _sc_expl_annihilate_fill(c, e, e->radius);
         if(!SC_CONFIG_GFX_FAST(c)) {
            /* Only do slow animation if not in fast mode */
            e->idraw = 0;
            return(true);
         } else {
            /* Just update everything at once */
            sc_window_paint(c->window, 
                            e->centerx - e->radius, 
                            e->centery - e->radius, 
                            e->centerx + e->radius, 
                            e->centery + e->radius, 
                            SC_PAINT_EVERYTHING);
         } /* Only executed if not in fast mode */
         break;

      case SC_EXPLOSION_RIOT:
         /* Animate a circular explosion that is clearing out */
         _sc_expl_annihilate_clear(c, e, e->radius);
         if(!SC_CONFIG_GFX_FAST(c)) {
            /* Only do slow animation if not in fast mode */
            e->idraw = 0;
            return(true);
         } else {
            /* Just update everything at once */
            sc_window_paint(c->window, 
                            e->centerx - e->radius, 
                            e->centery - e->radius, 
                            e->centerx + e->radius, 
                            e->centery + e->radius, 
                            SC_PAINT_EVERYTHING);
         } /* Only executed if not in fast mode */
         break;

   } /* End switch on explosion type */
   
   return(false);
   
}



bool sc_expl_annihilate_continue(sc_config *c, sc_explosion *e) {
/* sc_expl_annihilate_continue
   Annihilate a section of the screen by drawing a huge explosion to it. */

   sc_expl_cache *ca;   /* Explosion cache (standard explosions) */
   sc_spill *sp;        /* Spill information (liquid explosions) */

   /* Sanity check */
   if(c == NULL || e == NULL) return(false);
   
   /* Action depends on weapon type */
   switch(e->type) {
      case SC_EXPLOSION_SPIDER:
         /* nothing special to animate spider explosions */

      case SC_EXPLOSION_NORMAL:
      case SC_EXPLOSION_PLASMA:
         /* Get a new explosion cache ID, and draw it */
         ca = e->cache;
         if(ca != NULL) {
            ca->eradius += SC_EXPL_EXPLOSION_STEP;
            if(ca->eradius >= e->radius) ca->eradius = e->radius;
            sc_expl_cache_draw(c->window, ca->cacheid, 
                               e->centerx, e->centery, 
                               ca->eradius);
            if(c->physics->walls == SC_WALL_WRAP) {
               if(e->centerx + e->radius >= c->fieldwidth) {
                  sc_expl_cache_draw(c->window, ca->cacheid, 
                                     e->centerx - c->fieldwidth, 
                                     e->centery, ca->eradius);
               } else if(e->centerx - e->radius < 0) {
                  sc_expl_cache_draw(c->window, ca->cacheid, 
                                     e->centerx + c->fieldwidth,
                                     e->centery, ca->eradius);
               } /* Did the explosion wrap off-screen? */
            } /* Were boundaries wrap-around? */
            
            /* Done animating? */
            if(ca->eradius >= e->radius) {
               free(ca);
               return(false);
            }

            /* We still need to animate */            
            return(true);
         } /* Only executed if not in fast mode */
         return(false);
         
      case SC_EXPLOSION_NAPALM:
         /* Construct the napalm spill */
         if(SC_CONFIG_NO_ANIM(c)) return(false);
         sp = (sc_spill *)e->data;
         if(sp != NULL) {
            if(sp->count > 0) {  
               sc_window_draw_napalm_final(c->window, sp->spillx, sp->spilly, sp->size);
               ++sp->count;
               if(sp->count > SC_EXPL_NAPALM_FLAMES) return(false);
            } else if(sp->index < sp->size) {
               sc_window_draw_napalm_frame(c->window, 
                                           sp->spillx + sp->index,
                                           sp->spilly + sp->index,
                                           MIN(sp->size - sp->index, SC_EXPL_LIQUID_STEP));
               sp->index += SC_EXPL_LIQUID_STEP;
            } else {
               sp->count = 1;
            } /* Do we still have business to take care of? */
         } /* Only attempt the draw if spill pointer valid */
         return(true);
            
      case SC_EXPLOSION_LIQ_DIRT:
         /* Construct the liquid dirt spill */
         if(SC_CONFIG_NO_ANIM(c)) return(false);
         sp = (sc_spill *)e->data;
         if(sp != NULL) {
            if(sp->index < sp->size) {
               sc_land_create_dirt(c, c->land,
                                   sp->spillx + sp->index,
                                   sp->spilly + sp->index,
                                   MIN(sp->size - sp->index, SC_EXPL_LIQUID_STEP));
               sp->index += SC_EXPL_LIQUID_STEP;
            } else {
               return(false);
            } /* Do we still have business to take care of? */
         } /* Only attempt the draw if spill pointer valid */
         return(true);
            
      case SC_EXPLOSION_DIRT:
      case SC_EXPLOSION_RIOT:
         /* Continue to clear a circular explosion */
         e->idraw += SC_EXPL_EXPLOSION_STEP;
         if(e->idraw >= e->radius) {
            sc_window_paint(c->window, 
                            e->centerx - e->radius, 
                            e->centery - e->radius, 
                            e->centerx + e->radius, 
                            e->centery + e->radius, 
                            SC_PAINT_EVERYTHING);
            return(false);
         } else {
            sc_window_paint_circular(c->window, 
                                     e->centerx, e->centery, 
                                     e->idraw, 
                                     SC_PAINT_EVERYTHING);
            return(true);
         }
         break;
            
   } /* End switch on explosion type */
   
   return(false);
   
}



bool sc_expl_annihilate_clear(sc_config *c, sc_explosion *e) {
/* sc_expl_annihilate_clear 
   Clears someone else's explosions.  */

   sc_trajectory *tr;
   sc_spill *sp;        /* Spill information (liquid explosions) */

   /* Sanity check */
   if(c == NULL || e == NULL) return(false);

   /* Action depends on weapon type */
   switch(e->type) {
      case SC_EXPLOSION_SPIDER:
        /* if there is an arc to clear, clear it */
        if(e->data != NULL) {
           sc_window_clear_arc(c->window, e->data);
           tr = (sc_trajectory *)e->data;
           sc_traj_free(&tr);
        }
        /* fall through to explosion clearing */

      case SC_EXPLOSION_NORMAL:
      case SC_EXPLOSION_PLASMA:
         /* Clear the circular explosion */
         _sc_expl_annihilate_clear(c, e, e->radius);
         if(!SC_CONFIG_GFX_FAST(c)) {
            /* Only do slow animation if not in fast mode */
            e->idraw = 0;
            return(true);
         } else {
            /* Just update everything at once */
            sc_window_paint(c->window, 
                            e->centerx - e->radius, 
                            e->centery - e->radius, 
                            e->centerx + e->radius, 
                            e->centery + e->radius, 
                            SC_PAINT_EVERYTHING);
         } /* Only executed if not in fast mode */
         break;

      case SC_EXPLOSION_NAPALM:
         /* Clean up the napalm spill */
         sp = (sc_spill *)e->data;
         if(!SC_CONFIG_GFX_FAST(c)) {
            sc_window_clear_napalm(c->window, sp->spillx, sp->spilly, sp->size);
         }
         sc_spill_free((sc_spill **)&e->data);
         break;

      case SC_EXPLOSION_LIQ_DIRT:
      case SC_EXPLOSION_RIOT:
      case SC_EXPLOSION_DIRT:
         /* Nothing to do */
         break;

   } /* End switch on explosion type */
   
   return(false);

}



bool sc_expl_annihilate_clear_continue(sc_config *c, sc_explosion *e) {
/* sc_expl_annihilate_clear_continue
   Continue clearing someone else's explosions.  */

   /* Sanity check */
   if(c == NULL || e == NULL) return(false);

   /* Action depends on weapon type */
   switch(e->type) {
      case SC_EXPLOSION_SPIDER:
      case SC_EXPLOSION_NORMAL:
      case SC_EXPLOSION_PLASMA:
         /* Continue to clear the circular explosion */
         e->idraw += SC_EXPL_EXPLOSION_STEP;
         if(e->idraw >= e->radius) {
            sc_window_paint(c->window, 
                            e->centerx - e->radius, 
                            e->centery - e->radius, 
                            e->centerx + e->radius, 
                            e->centery + e->radius, 
                            SC_PAINT_EVERYTHING);
            return(false);
         } else {
            sc_window_paint_circular(c->window, 
                                     e->centerx, e->centery, 
                                     e->idraw, 
                                     SC_PAINT_EVERYTHING);
            return(true);
         }
         break;

      case SC_EXPLOSION_NAPALM:
      case SC_EXPLOSION_LIQ_DIRT:
      case SC_EXPLOSION_DIRT:
      case SC_EXPLOSION_RIOT:
         /* Nothing to do */
         break;

   } /* End switch on explosion type */
   
   return(false);

}



int sc_expl_damage_at_point(const sc_land *l, const sc_explosion *e, int x, int y) {
/* sc_expl_damage_at_point
   Returns the amount of damage (in units comparable to life), done by the
   specified explosion to an object centered at (x, y).  If no damage is
   done, zero is returned.  */

   sc_spill *sp;     /* Spill data (for napalm damages) */
   int distance;     /* Distance between explosion center and (x,y) (squared) */
   int damage;       /* Actual amount of damage done */
   int deltax;       /* Change in X */
   int deltay;       /* Change in Y */
   int i;            /* Iterator */

   /* Determine the damaging effect of the weapon */
   damage = 0;
   switch(e->type) {
      case SC_EXPLOSION_SPIDER:
      case SC_EXPLOSION_NORMAL:
      case SC_EXPLOSION_PLASMA:
         /* Calculate distance */
         sc_land_calculate_deltas(l, &deltax, &deltay, x, y, e->centerx, e->centery);
         distance = SQR(deltax) + SQR(deltay);

         /* Calculate the overall damage */
         damage = rint(e->force * (1 - distance / (double)SQR(e->radius)));
         break;
         
      case SC_EXPLOSION_NAPALM:
         /* Integrate over all points... */
         sp = (sc_spill *)e->data;
         for(i = 0; i < sp->size; ++i) {
            sc_land_calculate_deltas(l, &deltax, &deltay, sp->spillx[i], sp->spilly[i], x, y);
            distance = SQR(deltax) + SQR(deltay);
            distance = rint(e->force * (1 - distance / (double)SQR(e->radius)));
            if(distance < 0) distance = 0;
            damage += distance;
         } /* Iterate through all points in damage zone */
         break;

      case SC_EXPLOSION_LIQ_DIRT:
      case SC_EXPLOSION_DIRT:
      case SC_EXPLOSION_RIOT:
         /* Dirt balls/riots do no damage to tanks */
         break;

   } /* End switch to select radius */

   if(damage < 0) damage = 0;
   /*printf("   damage assessed = %d\n", damage);*/
   return(damage);

}
