/* $Header: /fridge/cvs/xscorch/sgame/saddconf.c,v 1.14 2001/04/14 00:31:00 jacob Exp $ */
/*
   
   xscorch - saddconf.c       Copyright(c) 2001 Jacob Luna Lundberg
   jacob(at)chaos2.org        http://chaos2.org/~jacob

   Scorched read/append config (weapons and accessories)
    

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <sweapon.h>
#include <sphoenix.h>
#include <saddconf.h>
#include <saccessory.h>
#include <sinventory.h>
#include <sutil/sreg.h>
#include <sutil/sstr.h>
#include <sutil/shash.h>
#include <sutil/sslink.h>



static const reg_class_data _reg_weapons_class[] = {
   { "weaponName",              REG_STRING,     NULL },
   { "armsLevel",               REG_INTEGER,    NULL },
   { "price",                   REG_INTEGER,    NULL },
   { "bundle",                  REG_INTEGER,    NULL },
   { "radius",                  REG_INTEGER,    NULL },
   { "force",                   REG_INTEGER,    NULL },
   { "liquid",                  REG_INTEGER,    NULL },
   { "scatter",                 REG_INTEGER,    NULL },
   { "children",                REG_INTEGER,    NULL },
   { "useless",                 REG_BOOLEAN,    NULL },
   { "indirect",                REG_BOOLEAN,    NULL },
   { "stateFlags",              REG_STRING,     NULL },  /* bit field */
   { "phoenixFlags",            REG_STRING,     NULL },  /* bit field */
   { "phoenixChild",            REG_STRING,     NULL },
   { 0, 0, 0 }
};

static const reg_class_list _reg_class_dataes[] = {
   { "weapons_class",           _reg_weapons_class },
   { 0, 0 }
};



sc_weapon_info *_sc_addconf_append_item(sc_weapon_list *wl, char *name) {
/* _sc_addconf_append_item
   Local method to tack a new weapon def onto the weapon_list in wl. */

   int hashindex;

   /* Add the def to the dll */
   if(wl->defs_head == NULL) {
      /* Empty list */
      wl->defs_tail = wl->defs_head = sc_weapon_list_item_new();
      if(wl->defs_head == NULL) return(NULL);
      wl->defs_head->item->ident = 0;
   } else {
      /* Append to list */
      wl->defs_tail->next = sc_weapon_list_item_new();
      if(wl->defs_tail->next == NULL) return(NULL);
      wl->defs_tail->next->item->ident = wl->defs_tail->item->ident + 1;
      wl->defs_tail->next->prev = wl->defs_tail;
      wl->defs_tail = wl->defs_tail->next;
   }

   /* Add the def to the hash chains */
   hashindex = shash(SC_WEAPON_HASH_BITS, wl->defs_tail->item->ident);
   if(wl->defs_hash[hashindex] == NULL) {
      /* Nothing on hash chain */
      wl->defs_hash[hashindex] = wl->defs_tail;
   } else {
      /* Already an item on the hash chain */
      wl->defs_tail->next = wl->defs_hash[hashindex];
      while(wl->defs_tail->next->chain != NULL)
         wl->defs_tail->next = wl->defs_tail->next->chain;
      wl->defs_tail->next->chain = wl->defs_tail;
      wl->defs_tail->next = NULL;
   }

   /* Set in the new weapon's name */
   wl->defs_tail->item->name = name;

   /* Only now are we truely the winners */
   return(wl->defs_tail->item);

}



bool sc_weapon_append_file(sc_weapon_config *wc, const char *filename) {
/* sc_weapon_append_file
   Append weapons to the weapons registries.  Returns true on success. */

   char *name;
   int iterator;
   int count = 0;
   sc_weapon_info *info;
   const sc_weapon_info *child;
   sc_weapon_list *wl = wc->weaponlist;
   char childname[SC_WEAPON_MAX_NAME_LEN];
   reg_var *weapon;
   reg_var *top;
   reg *readr;

   /* Read the file into the registry */
   readr = reg_new(filename);
   if(readr == NULL) return(false);
   reg_class_register_list(readr, _reg_class_dataes);
   if(!reg_load(readr)) {
      /* Failed to read file; signal error */
      printf("saddconf - aborting, weapons def file unreadable\n");
      reg_free(&readr);
      return(false);
   }

   /* Read the variables in order */
   top = reg_get_top(readr);
   weapon = reg_get_block_head(top);
   while(weapon != NULL) {
      /* Check the class name */
      if(strcmpn(reg_get_var_class(weapon), "weapons_class")) {
         /* Make some space for a new weapon name string */
         name = (char *)malloc(SC_WEAPON_MAX_NAME_LEN * sizeof(char));
         if(name == NULL) {
            printf("saddconf - aborting, failed to allocate memory for weapon name\n");
            break;
         }
         /* Get the weapon's name string. */
         reg_get_string(readr, weapon, "weaponName", name, SC_WEAPON_MAX_NAME_LEN);
         /* Check the name for duplication */
         if(sc_weapon_lookup_by_name(wc, name, SC_WEAPON_LIMIT_NONE)) {
            printf("saddconf - ignoring attempt to add weapon \"%s\" - duplicate by name\n", name);
            free(name);
            continue;
         }
         /* addconf_append will set the weapon's ident for us */
         info = _sc_addconf_append_item(wl, name);
         if(info == NULL) {
            printf("saddconf - aborting, weapon list append failed, likely malloc error\n");
            free(name);
            break;
         }

         /* Set in the other fields in the weapon_info struct */
         reg_get_integer(readr, weapon, "armsLevel", &info->armslevel);
         reg_get_integer(readr, weapon, "price",     &info->price);
         reg_get_integer(readr, weapon, "bundle",    &info->bundle);
         reg_get_integer(readr, weapon, "radius",    &info->radius);
         reg_get_integer(readr, weapon, "force",     &info->force);
         reg_get_integer(readr, weapon, "liquid",    &info->liquid);
         reg_get_integer(readr, weapon, "scatter",   &info->scatter);
         reg_get_integer(readr, weapon, "children",  &info->children);

         reg_get_boolean(readr, weapon, "useless",   &info->useless);
         reg_get_boolean(readr, weapon, "indirect",  &info->indirect);

         info->state = reg_get_bitmask_from_values(readr, weapon, "stateFlags",   0,
                                                   sc_weapon_state_bit_names(),
                                                   sc_weapon_state_bit_items());
         info->ph_fl = reg_get_bitmask_from_values(readr, weapon, "phoenixFlags", 0,
                                                   sc_phoenix_flags_bit_names(),
                                                   sc_phoenix_flags_bit_items());

         /* Set the child if there is one */
         childname[0] = '\0';
         reg_get_string(readr, weapon, "phoenixChild", childname, SC_WEAPON_MAX_NAME_LEN);
         if(childname[0] != '\0') {
            child = sc_weapon_lookup_by_name(wc, childname, SC_WEAPON_LIMIT_NONE);
            if(child == NULL) {
               printf("saddconf - warning, \"%s\" has missing child \"%s\"\n", name, childname);
            } else {
               info->ph_ch = child->ident;
            }
         }
         

         /* Test for weapon acceptance */
         if(SC_WEAPON_IS_PHOENIX(info) && !sc_phoenix_verify(wc, info)) {
            printf("saddconf - \"%s\" is an invalid phoenix weapon, killing it\n", name);
            /* TEMP - Well, ok, so we don't kill it yet; there's no list removal - JL */
            info->indirect = true;
            continue;
         }

         /* Clean up minor details such as player inventories. */
         for(iterator = 0; iterator < SC_MAX_PLAYERS; ++iterator) {
            if(SC_WEAPON_IS_INFINITE(info))
               info->inventories[iterator] = SC_INVENTORY_MAX_ITEMS;
            else
               info->inventories[iterator] = 0;
         }

         /* Signal we've completed an addition. */
         ++count;
      } else {
         /* Bad class given; signal the error. */
         printf("saddconf - reading weapon def file, expected weapons_class but didn't get it\n");
      }

      /* Get the next variable */
      weapon = reg_get_next_var(weapon);
   }

   /* Release the registry */   
   reg_free(&readr);

   /* Let the next layer up know whether we found some weapons */
   if(count)
      return(true);
   else
      return(false);

}
