/* $Header: /fridge/cvs/xscorch/sai/saibuy.c,v 1.8 2001/04/08 09:33:35 jacob Exp $ */
/*
   
   xscorch - sai.c            Copyright(c) 2001,2000 Justin David Smith
   justins(at)chaos2.org      http://chaos2.org/
    
   Main file for AI code
    

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <saiint.h>              /* AI internal header */
#include <sgame/seconomy.h>      /* Need current economical state */
#include <sgame/sinventory.h>    /* Need to get player inventory */
#include <sgame/splayer.h>       /* Need player data */
#include <sgame/sconfig.h>       /* Need basic configuration */
#include <sgame/sweapon.h>       /* Yield calculations */



static inline int _sc_ai_budget(const sc_config *c, const sc_player *p) {
/* sc_ai_budget
   Calculate the amount of money the AI is permitted to spend this turn */

   const sc_economy *ec;   /* Economy data */
   double percent;         /* Percentage of rounds played */

   /* Any budget constraints? */
   if(c->aicontrol->nobudget) {
      return(p->money * 0.75);
   }
   
   ec = c->economy;
   if(ec->computersbuy) {
      /* Only buy based on current interest rate, or the default
         conservative budget, whichever is larger */
      if(ec->computersaggressive) {
         percent = SC_AI_AGGRESSIVE_BUDGET;
      } else {
         percent = SC_AI_CONSERVATIVE_BUDGET;
      } /* What rate should we set as the "minimum"? */
      if(ec->interestrate > percent) {
         percent = ec->currentinterest;
      } /* Take maximum of "percent" and current interest rate */

      /* Also factor in the number of rounds remaining :) */
      percent += ((double)(c->curround + 1) / c->numrounds) * (1 - percent);
      if(percent > 1) percent = 1;

      #if SC_AI_DEBUG_BUY
         printf("AI_budget:   %s, %s has budget %d\n", sc_ai_name(p->aitype), p->name, (int)(p->money * percent));
      #endif /* Debug? */
      
      /* That ought to be sufficiently devastating. */
      return(p->money * percent);
   } /* Can computers buy? */
   
   /* Computers cannot buy. */
   return(0);

}



static void _sc_ai_buy_last_weapons(const sc_config *c, sc_player *p, int *budget) {
/* sc_ai_buy_last_weapons
   Buy the last weapons on the list that we can afford.  This is a rather
   naive approach, in assuming that weapons at the end of the list must
   surely be better?  */

   const sc_weapon_info *info;  /* Weapon information */
   int j;                       /* Iterator */

   j = SC_AI_BUY_MAX_OF_WEAPON;
   while(j > 0) {
      info = sc_weapon_last(c->weapons, SC_WEAPON_LIMIT_ALL);
      while(info != NULL) {
         if(sc_inventory_can_buy_weapon(p, info, *budget) && sc_weapon_yield(c->weapons, info) > 0) {
            /* Buy some of this weapon. */
            sc_inventory_buy_weapon(p, info);
            *budget -= info->price;
            #if SC_AI_DEBUG_BUY
               printf("   AI_buy:   %s, %s bought %s.  $%d\n", sc_ai_name(p->aitype), p->name, info->name, *budget);
            #endif /* Debug? */
         } /* Can we buy this weapon? */
         info = sc_weapon_prev(c->weapons, info->ident, SC_WEAPON_LIMIT_ALL);
      } /* Iteration on info lists */
      --j;
   } /* Iterate multiples of weapons */

}



static void _sc_ai_sorted_weapon_scores(sc_weapon_config *wc, int *ordering, double *scores) {
/* sc_ai_sorted_weapon_scores
   Impose an ordering on the weapons, based on scores loaded into *scores.  */
   
   int i, j;                  /* Iterator variables... */
   int count = sc_weapon_count(wc, SC_WEAPON_LIMIT_ALL);

   /* Bubble-sort the scores */   
   for(i = 0; i < count; ++i) {
      for(j = 0; j < i; ++j) {
         if(scores[j] > scores[i]) {
            scores[j] += scores[i];
            scores[i] = scores[j] - scores[i];
            scores[j] = scores[j] - scores[i];
            ordering[j] += ordering[i];
            ordering[i] = ordering[j] - ordering[i];
            ordering[j] = ordering[j] - ordering[i];
         }
      }
   }

}



static void _sc_ai_buy_weapons_from_list(const sc_config *c, sc_player *p, int *budget, const int *ordering) {
/* sc_ai_buy_weapons_from_list
   Buys weapons from a presorted list, beginning at the _end_ of the list.  */

   const sc_weapon_info *info;   /* Weapon information. */
   int i;                        /* iterator */
   int j;                        /* iterator */
   int count = sc_weapon_count(c->weapons, SC_WEAPON_LIMIT_ALL);

   /* Iterate through, beginning with the "best" weapons. */
   j = SC_AI_BUY_MAX_OF_WEAPON;
   while(j > 0) {
      i = count - 1;
      while(i >= 0) {
         info = sc_weapon_lookup(c->weapons, ordering[i], SC_WEAPON_LIMIT_ALL);
         if(sc_inventory_can_buy_weapon(p, info, *budget)) {
            /* buy some of this weapon. */
            sc_inventory_buy_weapon(p, info);
            *budget -= info->price;
            #if SC_AI_DEBUG_BUY
               printf("   AI_buy:   %s, %s bought %s.  $%d\n", sc_ai_name(p->aitype), p->name, info->name, *budget);
            #endif /* Debug? */
         } /* Can we buy this weapon? */
         --i;
      } /* Buying weapons */
      --j;
   } /* Iterate multiples of weapons */

}



static void _sc_ai_buy_weapons_by_score(const sc_config *c, sc_player *p, int *budget) {
/* sc_ai_buy_weapons_by_score
   Buy the weapons yielding the best destructive power.  This is a great
   buying strategy for players that just want to annihilate the entire
   playing field.  Coupled with agressive, this can be quite deadly!  */

   const sc_weapon_info *info;
   int count = sc_weapon_count(c->weapons, SC_WEAPON_LIMIT_ALL);
   /* Sorted indices, based on score. */
   int *ordering = (int *)malloc(sizeof(int) * count);
   /* Sorted scores corresponding to ordering */
   double *scores = (double *)malloc(sizeof(double) * count);
   int i; /* Iterator */

   if(ordering == NULL || scores == NULL) {
      fprintf(stderr, "Malloc error in sc_ai_buy_weapons_by_score!\n");
      return;
   }

   /* Calculate scores, and set initial ordering */
   info = sc_weapon_first(c->weapons, SC_WEAPON_LIMIT_ALL);
   for(i = 0; i < count; ++i) {
      ordering[i] = info->ident;
      scores[i] = sc_weapon_yield(c->weapons, info);
      info = sc_weapon_next(c->weapons, info->ident, SC_WEAPON_LIMIT_ALL);
   }

   /* Bubble-sort the scores */   
   _sc_ai_sorted_weapon_scores(c->weapons, ordering, scores);

   /* Iterate through, beginning with the "best" weapons. */
   _sc_ai_buy_weapons_from_list(c, p, budget, ordering);

   free(ordering);
   free(scores);

}



static void _sc_ai_buy_weapons_by_weighted_score(const sc_config *c, sc_player *p, int *budget) {
/* sc_ai_buy_weapons_by_weighted_score
   Buy the weapons yielding the best destructive power per price of
   unit.  This is the "efficient" way to buy, but there are times when
   a good screen-destroying weapon is better than a weapon that will
   take us 100 turns to wipe out the opponent...  */

   const sc_weapon_info *info;
   int count = sc_weapon_count(c->weapons, SC_WEAPON_LIMIT_ALL);
   /* Sorted indices, based on score. */
   int *ordering = (int *)malloc(sizeof(int) * count);
   /* Sorted scores corresponding to ordering */
   double *scores = (double *)malloc(sizeof(double) * count);
   int i; /* Iterator */

   if(ordering == NULL || scores == NULL) {
      fprintf(stderr, "Malloc error in sc_ai_buy_weapons_by_weighted_score!\n");
      return;
   }

   /* Calculate scores, and set initial ordering */
   info = sc_weapon_first(c->weapons, SC_WEAPON_LIMIT_ALL);
   for(i = 0; i < count; ++i) {
      ordering[i] = info->ident;
      scores[i] = sc_weapon_weighted_yield(c->weapons, info);
      info = sc_weapon_next(c->weapons, info->ident, SC_WEAPON_LIMIT_ALL);
   }

   /* Bubble-sort the scores */   
   _sc_ai_sorted_weapon_scores(c->weapons, ordering, scores);

   /* Iterate through, beginning with the "best" weapons. */
   _sc_ai_buy_weapons_from_list(c, p, budget, ordering);

   free(ordering);
   free(scores);

}



/* These days, there's not much to designate a ``precise'' weapon. */
#define  _sc_ai_buy_precision_weapons(c, p, budget)   _sc_ai_buy_weapons_by_score(c, p, budget)



static void _sc_ai_buy_auto_defense(const sc_config *c, sc_player *p, int *budget) {
/* sc_ai_buy_auto_defense
   Buy auto-defense, only if this is a sequential game. */

   const sc_accessory_info *info;/* Item information */
   int i;                     /* Iterator */

   /* Check if this is a sequential game. */
   switch(c->options.mode) {
      case SC_CONFIG_MODE_SEQUENTIAL:  break;
      case SC_CONFIG_MODE_SYNCHRONOUS: return;
   }

   /* If this is a sequential game, make damn sure we buy autodefense */
   i = sc_accessory_count(c->accessories) - 1;
   while(i >= 0) {
      if(sc_inventory_can_buy_accessory(c->accessories, p, i, *budget)) {
         /* Make sure the damn thing is auto-defence. */
         info = sc_accessory_lookup(c->accessories, i);
         if(SC_ACCESSORY_IS_AUTO_DEF(info)) {
            /* Buy some of this autodefense */
            sc_inventory_buy_accessory(c->accessories, p, i);
            *budget -= info->price;
            #if SC_AI_DEBUG_BUY
               printf("   AI_buy:   %s, %s bought %s.  $%d\n", sc_ai_name(p->aitype), p->name, info->name, *budget);
            #endif /* Debug? */
            return;
         } /* Is accessory auto-defense? */
      } /* Can we buy? */
      --i;
   } /* Iterate through */
   
}



static void _sc_ai_buy_contact_triggers(const sc_config *c, sc_player *p, int *budget) {
/* sc_ai_buy_contact_triggers
   Buy contact triggers if tunneling is allowed. */

   const sc_accessory_info *info;/* Item information */
   int i;                     /* Iterator */

   /* Only buy if tunneling is allowed. */
   if(!c->weapons->tunneling) return;

   /* If this is a sequential game, make damn sure we buy autodefense */
   i = sc_accessory_count(c->accessories) - 1;
   while(i >= 0) {
      /* Make sure the damn thing is a contact trigger. */
      info = sc_accessory_lookup(c->accessories, i);
      if(SC_ACCESSORY_IS_TRIGGER(info)) {
         while(sc_inventory_can_buy_accessory(c->accessories, p, i, *budget)) {
            /* Buy some of these contact triggers */
            sc_inventory_buy_accessory(c->accessories, p, i);
            *budget -= info->price;
            #if SC_AI_DEBUG_BUY
               printf("   AI_buy:   %s, %s bought %s.  $%d\n", sc_ai_name(p->aitype), p->name, info->name, *budget);
            #endif /* Debug? */
            return;
         } /* Can we buy? */
      } /* Is accessory a contact trigger? */
      --i;
   } /* Iterate through */
   
}



static void _sc_ai_buy_last_shields(const sc_config *c, sc_player *p, int *budget) {
/* sc_ai_buy_last_shields
   Buy the last shields on the list that we can afford.  This is a rather
   naive approach, in assuming that shields at the end of the list must
   surely be better?  */

   const sc_accessory_info *info;/* Shield information */
   int i;                     /* Iterator */

   /* Buy some shielding */
   i = sc_accessory_count(c->accessories) - 1;
   while(i >= 0) {
      if(sc_inventory_can_buy_accessory(c->accessories, p, i, *budget)) {
         /* Make sure the damn thing is a shield. */
         info = sc_accessory_lookup(c->accessories, i);
         if(SC_ACCESSORY_IS_SHIELD(info)) {
            /* Buy some of this shield */
            sc_inventory_buy_accessory(c->accessories, p, i);
            *budget -= info->price;
            #if SC_AI_DEBUG_BUY
               printf("   AI_buy:   %s, %s bought %s.  $%d\n", sc_ai_name(p->aitype), p->name, info->name, *budget);
            #endif /* Debug? */
            return;  /* That's all, for now */
         } /* Is accessory a shield? */
      } /* Can we buy? */
      --i;
   } /* Iterate through */

}



static void _sc_ai_sorted_accessory_scores(sc_accessory_config *wc, int *ordering, int *scores) {
/* sc_ai_sorted_accessory_scores
   Impose an ordering on the accessorys, based on scores loaded into *scores.  */
   
   int i, j;                  /* Iterator variables... */

   /* Bubble-sort the scores */   
   for(i = 0; i < sc_accessory_count(wc); ++i) {
      for(j = 0; j < i; ++j) {
         if(scores[j] > scores[i]) {
            scores[j] += scores[i];
            scores[i] = scores[j] - scores[i];
            scores[j] = scores[j] - scores[i];
            ordering[j] += ordering[i];
            ordering[i] = ordering[j] - ordering[i];
            ordering[j] = ordering[j] - ordering[i];
         }
      }
   }

}






static void _sc_ai_buy_shields_from_list(const sc_config *c, sc_player *p, int *budget, const int *ordering) {
/* sc_ai_buy_shields_from_list
   Buy shields from the ordered list given; keep trying to
   buy until we find a set we can afford, then stop there.  */

   const sc_accessory_info *info;/* Accessory information. */
   int i;                        /* Iterator */

   /* Iterate through, beginning with the "best" shields. */
   i = sc_accessory_count(c->accessories) - 1;
   while(i >= 0) {
      /* Make sure the damn thing is a shield. */
      info = sc_accessory_lookup(c->accessories, i);
      if(SC_ACCESSORY_IS_SHIELD(info)) {
         /* Can we afford this set of shields? */
         if(sc_inventory_can_buy_accessory(c->accessories, p, ordering[i], *budget)) {
            /* buy some of this shielding. */
            sc_inventory_buy_accessory(c->accessories, p, i);
            *budget -= info->price;
            #if SC_AI_DEBUG_BUY
               printf("   AI_buy:   %s, %s bought %s.  $%d\n", sc_ai_name(p->aitype), p->name, info->name, *budget);
            #endif /* Debug? */
            return;  /* That's all, for now */
         } /* Can we buy these shields? */
      } /* Is this accessory a shield? */
      --i;
   }

}



static void _sc_ai_buy_best_shields(const sc_config *c, sc_player *p, int *budget) {
/* sc_ai_buy_best_shields
   Buy the best shields on the list that we can afford.  We will
   want to sort the shields out before deciding which to buy.  */

   /* Sorted indices, based on score. */
   int *ordering = (int *)malloc(sizeof(int) * sc_accessory_count(c->accessories));
   /* Sorted scores cooresponding to ordering */
   int *scores = (int *)malloc(sizeof(int) * sc_accessory_count(c->accessories));
   const sc_accessory_info *info;
   int i; /* Iterator */
   
   if(ordering == NULL || scores == NULL) {
      fprintf(stderr, "Malloc error in sc_ai_buy_best_shields!\n");
      return;
   }

   /* Calculate scores, and set initial ordering */
   for(i = 0; i < sc_accessory_count(c->accessories); ++i) {
      ordering[i] = i;
      info = sc_accessory_lookup(c->accessories, i);
      scores[i] = info->shield;
   }

   /* Bubble-sort the scores */   
   _sc_ai_sorted_accessory_scores(c->accessories, ordering, scores);

   /* Iterate through, beginning with the "best" shields. */
   _sc_ai_buy_shields_from_list(c, p, budget, ordering);

   free(ordering);
   free(scores);

}



sc_ai_result sc_ai_player_buy(const sc_config *c, sc_player *p) {
/* sc_ai_player_buy
   Buy some weapons.  */
   
   int budget;    /* Amount of money allocated to spend */

   budget = _sc_ai_budget(c, p);

   switch(p->ai->realaitype) {
      case SC_AI_HUMAN:
         return(SC_AI_NO_ACTION);
         break;

      case SC_AI_NETWORK:
      case SC_AI_RANDOM:
         /* No-ops */
         break;

      case SC_AI_MORON:
         _sc_ai_buy_contact_triggers(c, p, &budget);
         _sc_ai_buy_auto_defense(c, p, &budget);
         _sc_ai_buy_last_shields(c, p, &budget);
         _sc_ai_buy_last_weapons(c, p, &budget);
         break;

      case SC_AI_SHOOTER:
         _sc_ai_buy_contact_triggers(c, p, &budget);
         _sc_ai_buy_auto_defense(c, p, &budget);
         _sc_ai_buy_best_shields(c, p, &budget);
         _sc_ai_buy_weapons_by_weighted_score(c, p, &budget);
         break;

      case SC_AI_SPREADER:
         _sc_ai_buy_contact_triggers(c, p, &budget);
         _sc_ai_buy_auto_defense(c, p, &budget);
         _sc_ai_buy_best_shields(c, p, &budget);
         _sc_ai_buy_weapons_by_score(c, p, &budget);
         break;

      case SC_AI_CHOOSER:
         _sc_ai_buy_contact_triggers(c, p, &budget);
         _sc_ai_buy_auto_defense(c, p, &budget);
         _sc_ai_buy_best_shields(c, p, &budget);
         _sc_ai_buy_precision_weapons(c, p, &budget);
         break;

      case SC_AI_CALCULATER:
         _sc_ai_buy_contact_triggers(c, p, &budget);
         _sc_ai_buy_auto_defense(c, p, &budget);
         _sc_ai_buy_best_shields(c, p, &budget);
         _sc_ai_buy_precision_weapons(c, p, &budget);
         break;

      case SC_AI_ANNIHILATER:
      case SC_AI_INSANITY:
         budget = p->money * 0.75;
         if(budget > p->money) budget = p->money;
         _sc_ai_buy_contact_triggers(c, p, &budget);
         _sc_ai_buy_weapons_by_score(c, p, &budget);
         break;
         
   }
   
   return(SC_AI_CONTINUE);

}
