/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 and 2000 by the Free Software Foundation.      *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#ifndef _DFA_H_
#define _DFA_H_

#include <stdio.h>
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <assert.h>
#include <errno.h>
#include <string.h>

/********************************
 *         parameters           *
 ********************************/

/* #define DFA_TRACE define this to trace the program */
/* #define DFA_PARANOIAC define this to activate a lot of assert() */
#define DFA_MAX_BOARD 19
#define MAX_ORDER DFA_MAX_BOARD*DFA_MAX_BOARD*4
#define DFA_RESIZE_STEP 0x20000
#define DFA_INIT_SIZE 0X100
#ifndef EMPTY
#define EMPTY     0		/* . */
#define WHITE     1		/* O */
#define BLACK     2		/* X */
#endif
#define OUT_BOARD 3		/* # */

/********************************
 *    data types definition     *
 ********************************/

/* intersections */

typedef unsigned short Intersection_t;

/* attribute list */

typedef struct attrib
{
  int val;
  int next;
}
attrib_t;


/* dfa state */

typedef struct state
{
  int att;
  int next[4];
}
state_t;


/* dfa */

typedef struct dfa
{
  /* file header */
  char head[20];
  char name[80];
  char copyleft[100];
  int  all_trans; /* if set to 1 the dfa already contain all transformations */
  
  /* transition graph */
  state_t *states;
  int maxStates;
  int lastState;

  /* attributes sets */
  attrib_t *indexes;
  int maxIndexes;
  int lastIndex;

  /* scanning state */
  int actual; /* actual state */
  int actual_index; /* actual index or zero if not reading indexes */
  int trans;  /* transformation */
  int m; /* anchor position */
  int n;
  int row; /* row in the scanned string */
}
dfa_t;


/* scan order */

typedef struct
{
  int i;
  int j;
}
order_t;


/********************************
 *    functions declaration     *
 ********************************/

extern void dfa_init (void);	/* Every call to a fda function must be done */
extern void dfa_end (void);	/* between calls of those 2 functions. */

/* basic dfa manipulation */
extern void new_dfa (dfa_t * pdfa, const char *name);
extern void copy_dfa (dfa_t * p_to, dfa_t * p_from);
extern void kill_dfa (dfa_t * pdfa);
extern int  dfa_size (dfa_t * pdfa);	/* in Kb */
extern void save_dfa (const char *f_name, dfa_t * pdfa);
extern dfa_t* load_dfa (const char *f_path, const char *f_name, dfa_t * pdfa);
extern void dump_dfa (FILE * f, dfa_t * pdfa);

struct pattern;

/* conversion between a gnugo pattern struct into a dfa string. */
extern void pattern_2_string (struct pattern *pat, char *str,
			      int trans, int ci, int cj);

/* add all the transformations of a gnugo pattern into a dfa, 
 * removing automatically the invariants.
 * FIXME: This make the :8 value obsolete in db files. 
 */
extern int 
dfa_add_pattern (struct pattern *pat, dfa_t *pdfa, int ci,
		 int cj, int board_size, int index,
		 int  stats[5][board_size][board_size]);

/* add a string with attribute att_val into a dfa if
 * it size is not too big. Return the increase in Kbytes or -1
 * if the pattern is rejected */
extern int dfa_add_string (dfa_t * pdfa, const char *str, int pattern_index,
			   int max_Kb_size, int max_Kb_increase);

/* must be called before/after using dfa_add_string 
 * if we want to do some stats about patterns.
 */
extern void reset_pattern_stats(int board_size, 
				int stats[5][board_size][board_size]);
extern void display_pattern_stats(int board_size, 
				  int stats[5][board_size][board_size], int n);

#ifdef DFA_MATCHER
/* prepare the board for the pattern matching, must be called one time
 * by position */
extern void dfa_compile_for_match(int color);
#endif

/* reset the dfa for a given position */
extern void dfa_restart (dfa_t *pdfa, int trans, int m, int n);

/* get the index of next matched pattern or -1 if no more 
 * must be initialized by dfa_restart befor each match. */
extern int dfa_scan (dfa_t *pdfa);

/* loading of dfa files */
extern void dfa_match_init(void);

/* conversion macros */
extern int dfa_asc2val[];
extern char dfa_val2asc[];
#define ASC2VAL(c) (c<90?dfa_asc2val[(int)c]:3)
#define VAL2ASC(n) (n<4?dfa_val2asc[n]:'!')

#define DFA_TRANSFORM(i,j,ti,tj,trans) \
do { \
  *ti = dfa_transformations[trans][0][0] * (i) +\
 dfa_transformations[trans][0][1] * (j); \
  *tj = dfa_transformations[trans][1][0] * (i) +\
 dfa_transformations[trans][1][1] * (j); \
} while(0)

/********************************
 *    global variables          *
 ********************************/

extern int dfa_verbose;		/* the verbose level */

#endif /* _DFA_H_ */









