/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 and 2000 by the Free Software Foundation.      *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * * * * * * fast pattern matching with DFA  version 2.9 * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "patterns.h"
#include "dfa.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

/*********************
 *   Public data     *
 *********************/

/* If > 0 more detailled infomation is given */
int dfa_verbose = 0;

/* conversion array (stupid hack) */
int dfa_asc2val[90] = {
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
  3, 3, 3, 3, 3, 3, 0, 3, 3, 3,	/* '.' == 46 */
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
  3, 3, 3, 3, 3, 3, 3, 3, 3, 3,
  3, 3, 3, 3, 3, 3, 3, 3, 3, 1,	/* 'O' == 79 */
  3, 3, 3, 3, 3, 3, 3, 3, 2, 3	/* 'X' == 88 */
};

char dfa_val2asc[4] = {
    '.', 'O', 'X', '#' };


/*********************
 *  Private data     *
 *********************/

/* the private board */
int dfa_board_size = DFA_MAX_BOARD;
int dfa_p[DFA_MAX_BOARD * 4][DFA_MAX_BOARD * 4];
extern int board_size;

/* auxiliary dfa's for high level functions */
static dfa_t aux_dfa1;		/* used to store strings */
static dfa_t aux_dfa2;		/* used to store the result of dfa's products*/

/* To be sure that everything was well initialized */
static int dfa_was_initialized = 0;


/* table for use by DFA_TRANSFORM() (patterns.h) */
/* FIXME : copied from matchpat.c : 
 * please put it somewhere so we can share one copy */

static const int dfa_transformations[8][2][2] = {
  {{1, 0}, {0, 1}},		/* a - identity transformation matrix */
  {{0, 1}, {-1, 0}},		/* g - rotate 270 counter-clockwise */
  {{-1, 0}, {0, -1}},		/* d - rotate 180 */
  {{0, -1}, {1, 0}},		/* f - rotate 90 counter-clockwise */
  {{0, -1}, {-1, 0}},		/* h - rotate 90 and invert */
  {{-1, 0}, {0, 1}},		/* b - flip left */
  {{0, 1}, {1, 0}},		/* e - rotate 90 and flip left */
  {{1, 0}, {0, -1}}		/* c - invert */
};


/* convert is a table to convert the colors */
static const int convert[3][4] = {
  {-1, -1, -1, -1},		/* not used */
  {EMPTY, WHITE, BLACK, OUT_BOARD},	/* WHITE */
  {EMPTY, BLACK, WHITE, OUT_BOARD}	/* BLACK */

};

#define EXPECTED_COLOR(player_c, position_c) convert[player_c][position_c]


/* convert ATT_* values to the corresponding expected values on the board */
static const char att2val[8] =
  { '.', 'X', 'O', 'x', 'o', ',', 'a', '!' };

#define EXPECTED_VAL(att_val) att2val[att_val]


/************************************************
 *   forward declaration of private functions   *
 ************************************************/
static void buildSpiralOrder (order_t order[8][MAX_ORDER]);
#ifdef DFA_MATCHER
static int  read_board (int ll, int m, int n, int row);
void dfa_compile_for_match(int color);
#endif
static void clean_dfa (dfa_t * pdfa);
static void resize_dfa (dfa_t * pdfa, int maxStates, int maxIndexes);
static void create_dfa (dfa_t * pdfa, const char *str, int att_val);
static void do_sync_product (int l, int r);
static void sync_product (dfa_t * pout, dfa_t * pleft, dfa_t * pright);

static int board_to_string(int edges, int to_test, int board_size, 
			   char board[board_size][board_size], 
			   char* str, int m, int n);
static int range_check_ok(struct pattern *pat, int board_size, 
			  int m, int n, int trans,
			  int ci,int cj);
static void show_pattern_instance(int board_size, 
				  char board[board_size][board_size]);
static int update_pattern_stats(int board_size, 
				char board[board_size][board_size], 
				int  stats[5][board_size][board_size], 
				int m, int n);


/********************************
 *   manipulating scan orders   *
 ********************************/

/* The spiral order is the way we scan the board,
 * we begin on the anchor and we 
 * progressively scan all its neigbouring intersections,
 * collecting all the known patterns we meet on our way:
 *
 *                  4      4      4
 * 1    1     13    13    513    513  ... and so on until we reach
 *      2     2     2      2     827      a stopping state in the
 *                                6        dfa.
 */

static order_t spiral[8][MAX_ORDER];

/*
 * Build the spiral order for each
 * transformation: instead of changing the board
 * or changing the patterns, we only change the order,
 * for eg. the same dfa can perform the pattern matching
 *
 * that way for identity:
 *
 *     765                                            567
 *     814F      and this way for mirror symetry:    F418
 *     923E                                          E329
 *     CABD                                          DBAC
 *
 * Anther possibility is to generate one string by pattern and by
 * transformation in mkpat to avoid any runtime transformation 
 * but it may increase the size of the dfa.
 * 
 */

static const order_t generator[4] =
  { { 1, 0}, { 0,  1}, {-1, 0}, {0, -1}  };

void
buildSpiralOrder (order_t order[8][MAX_ORDER])
{
  int Mark[DFA_MAX_BOARD * 4][DFA_MAX_BOARD * 4];
  order_t fifo[8 * MAX_ORDER];
  int top = 0, end = 0;
  int i, j, k, ll;
  int di, dj;

  if (dfa_verbose > 1)
    fprintf (stderr, "Building spiral order\n");

  /* First we build the basic (pseudo)spiral order */

  /* initialization */
  memset (Mark, 1, sizeof (Mark));
  for (i = DFA_MAX_BOARD; i != DFA_MAX_BOARD * 3; i++)
    for (j = DFA_MAX_BOARD; j != DFA_MAX_BOARD * 3; j++)
      Mark[i][j] = 0;

  end = 0;
  top = 1;
  fifo[end].i = DFA_MAX_BOARD * 2;
  fifo[end].j = DFA_MAX_BOARD * 2;
  Mark[fifo[end].i][fifo[end].j] = 1;

  /* generation */
  while (end < MAX_ORDER)
    {
      i = fifo[end].i;
      j = fifo[end].j;
      order[0][end].i = i - DFA_MAX_BOARD * 2;
      order[0][end].j = j - DFA_MAX_BOARD * 2;
      end++;

      for (k = 0; k != 4; k++)
	{
	  di = generator[k].i;
	  dj = generator[k].j;

	  if (!Mark[i + di][j + dj])
	    {
	      fifo[top].i = i + di;
	      fifo[top].j = j + dj;
	      Mark[i + di][j + dj] = 1;
	      top++;
	    }
	}
    }

  /* Then we compute all the geometric transformations
     on this order */
  for (ll = 1; ll != 8; ll++)
    for (k = 0; k != MAX_ORDER; k++)
      DFA_TRANSFORM (order[0][k].i, order[0][k].j,
		 &(order[ll][k].i), &(order[ll][k].j), ll);

}

/*
 * A function to read the board's values folowing the order
 * spiral[ll] from the anchor (m,n).
 * 
 */
#ifdef DFA_MATCHER

static int
read_board (int ll, int m, int n, int row)
{
  int i2, j2;

  i2 = DFA_MAX_BOARD + m + spiral[ll][row].i;
  j2 = DFA_MAX_BOARD + n + spiral[ll][row].j;

#ifdef DFA_TRACE
  assert(row < MAX_ORDER);
  assert(i2 < DFA_MAX_BOARD*4 && j2 < DFA_MAX_BOARD*4);
  assert(i2 >= 0 && j2 >= 0);
  fprintf(stderr,"%c",VAL2ASC(dfa_p[i2][j2]));
#endif

  return dfa_p[i2][j2];
}
#endif

/********************************
 * manipulating attributes list *
 ********************************/

/*
 * Test if val is member of the attributes set att
 */

static int
member_att (dfa_t * pdfa, int att, int val)
{
  int res;

  res = 0;
  while (!res && att != 0)
    {
      res = (pdfa->indexes[att].val == val);
      att = pdfa->indexes[att].next;
    }
  return res;
}

/* 
 * return the union of two attribute sets att1 & att2
 * repectively from dfa1 and dfa2 into
 * att in dfa.
 */

static int
union_att (dfa_t * pdfa, dfa_t * pdfa1, int att1, dfa_t * pdfa2,
	   int att2)
{
  int att;
  int att_aux;

  /* copy att1 in att */
  att = 0;
  while (att1 != 0)
    {
      pdfa->lastIndex++;
      if (pdfa->lastIndex >= pdfa->maxIndexes)
	resize_dfa (pdfa, pdfa->maxStates,
		    pdfa->maxIndexes + DFA_RESIZE_STEP);
      att_aux = pdfa->lastIndex;

      pdfa->indexes[att_aux].val = pdfa1->indexes[att1].val;
      pdfa->indexes[att_aux].next = att;
      att = att_aux;
      att1 = pdfa1->indexes[att1].next;
    }

  /* add to att the new elements of att2 */
  while (att2 != 0)
    {
      if (!member_att (pdfa, att, pdfa2->indexes[att2].val))
	{
	  pdfa->lastIndex++;
	  if (pdfa->lastIndex >= pdfa->maxIndexes)
	    resize_dfa (pdfa, pdfa->maxStates,
			pdfa->maxIndexes + DFA_RESIZE_STEP);
	  att_aux = pdfa->lastIndex;

	  pdfa->indexes[att_aux].val = pdfa2->indexes[att2].val;
	  pdfa->indexes[att_aux].next = att;
	  att = att_aux;
	}
      att2 = pdfa2->indexes[att2].next;
    }

  return att;
}

/**********************
 * manipulating dfa's *
 **********************/


/*
 * return the effective size of a dfa in Kb.
 */

int
dfa_size (dfa_t * pdfa)
{
  int states_size, indexes_size;

  states_size = (pdfa->lastState + 1) * sizeof (state_t);
  indexes_size = (pdfa->lastIndex + 1) * sizeof (attrib_t);

  return (states_size + indexes_size + sizeof (dfa_t)) / 1024;
}


/* 
 * resize memory for a dfa 
 */

void
resize_dfa (dfa_t * pdfa, int maxStates, int maxIndexes)
{
  state_t *pBuf;
  attrib_t *pBuf2;
  int i;

  if (dfa_verbose > 1)
    fprintf (stderr, "Resizing dfa %s\n", pdfa->name);

  assert (pdfa->lastState <= pdfa->maxStates);
  assert (pdfa->lastIndex <= pdfa->maxIndexes);

  pBuf = realloc (pdfa->states, maxStates * sizeof (state_t));
  pBuf2 = realloc (pdfa->indexes, maxIndexes * sizeof (attrib_t));
  if (pBuf == NULL || pBuf2 == NULL)
    {
      fprintf (stderr, "No memory left for dfa: %s", pdfa->name);
      exit (1);
    }

  for (i = pdfa->maxStates; i < maxStates; i++)
    memset (pBuf + i, 0, sizeof (state_t));
  for (i = pdfa->maxIndexes; i < maxIndexes; i++)
    memset (pBuf2 + i, 0, sizeof (attrib_t));

  pdfa->states = pBuf;
  pdfa->maxStates = maxStates;
  pdfa->indexes = pBuf2;
  pdfa->maxIndexes = maxIndexes;

}



/* 
 * dump a dfa (debugging purpose).
 */

static const char *line =
  "----------------------------------------------------\n";

void
dump_dfa (FILE * f, dfa_t * pdfa)
{
  int i;
  int att, k;

  fprintf (f, line);
  fprintf (f, " name : %s\n", pdfa->name);
  fprintf (f, " Actual state : %d : %d %d %d %d --> %d\n", pdfa->actual,
	   pdfa->states[pdfa->actual].next[0],
	   pdfa->states[pdfa->actual].next[1],
	   pdfa->states[pdfa->actual].next[2],
	   pdfa->states[pdfa->actual].next[3],
	   pdfa->states[pdfa->actual].att);
  fprintf (f, " Nb states :  %7d, max= %d\n", pdfa->lastState + 1,
	   pdfa->maxStates);
  fprintf (f, " Nb Indexes : %7d, max= %d\n", pdfa->lastIndex,
	   pdfa->maxIndexes);
  fprintf (f, " memory needed : %d Mb\n", dfa_size (pdfa) / 1024);
  fprintf (f, line);

  if (dfa_size (pdfa) > 2)	/* change this value if needed */
    return;
  fprintf (f, " state  |   .    |   O    |   X    |   #    |  att \n");
  fprintf (f, line);
  for (i = 1; i != pdfa->lastState + 1; i++)
    {
      int *pnext = pdfa->states[i].next;
      fprintf (f, " %6d |", i);
      fprintf (f, " %6d | %6d | %6d |", pnext[0], pnext[1], pnext[2]);
      fprintf (f, " %6d |", pnext[OUT_BOARD]);
      att = pdfa->states[i].att;
      k = 0;
      fprintf (f, " %5d:", att);
      while (att != 0 && k < 10)
	{
	  fprintf (f, " %4d", pdfa->indexes[att].val);
	  att = pdfa->indexes[att].next;
	  k++;
	}
      if (att != 0)
	fprintf (f, " ...");
      fprintf (f, "\n");
    }
  fprintf (f, line);
  fflush (f);
}


/*
 * Reset a dfa
 */

static void
clean_dfa (dfa_t * pdfa)
{
  memset (pdfa->states, 0, pdfa->maxStates * sizeof (state_t));
  memset (pdfa->indexes, 0, pdfa->maxIndexes * sizeof (attrib_t));
  pdfa->lastState = 1;		/* initial state */
  pdfa->lastIndex = 0;
  pdfa->indexes[0].val = -1;
}


/* 
 * allocate memory for a new dfa 
 */

void
new_dfa (dfa_t * pdfa, const char *name)
{
  memset (pdfa, 0, sizeof (dfa_t));
  resize_dfa (pdfa, DFA_INIT_SIZE, DFA_INIT_SIZE);
  clean_dfa (pdfa);
  if (name != NULL)
    strcpy (pdfa->name, name);
  else
    strcpy (pdfa->name, "noname ");

  if (dfa_verbose > 1)
    fprintf (stderr, "dfa %s is born :)\n", pdfa->name);

}

/*
 * free memory used by a dfa
 */

void
kill_dfa (dfa_t * pdfa)
{
  free (pdfa->states);
  free (pdfa->indexes);
  if (dfa_verbose > 1)
    fprintf (stderr, "dfa %s is dead :(\n", pdfa->name);

  memset (pdfa, 0, sizeof (dfa_t));
}


/*
 * Copy a dfa.
 * and rezise the destination dfa if necessary.
 */

void
copy_dfa (dfa_t * p_to, dfa_t * p_from)
{
  assert (p_to != p_from);

  if (p_to->maxStates < p_from->lastState)
    resize_dfa (p_to, p_from->maxStates, p_to->maxIndexes);

  if (p_to->maxIndexes < p_from->lastIndex)
    resize_dfa (p_to, p_to->maxStates, p_from->maxIndexes);

  clean_dfa (p_to);

  memcpy (p_to->states, p_from->states,
	  sizeof (state_t) * (p_from->lastState + 1));
  memcpy (p_to->indexes, p_from->indexes,
	  sizeof (attrib_t) * (p_from->lastIndex + 1));

  p_to->lastState = p_from->lastState;
  p_to->lastIndex = p_from->lastIndex;
}



/*
 * save_dfa:
 * Save the dfa in a .dfa file The file format is:
 *
 * 1) The Header: exactly the dfa_t structure
 * with states and indexes set to NULL.
 * 2) The pattern list (i.e. a set of (lastIndex+1) attributes.
 * 3) The dfa (i.e. a set of (lastStates+1) x states).
 *
 * FIXME: The variable size of int may cause trouble to
 * portability of dfa files between different boxes.
 */

void
save_dfa (const char *f_name, dfa_t * pdfa)
{
  int message;
  dfa_t old_dfa;
  FILE *f;

  assert (pdfa->lastIndex < pdfa->maxIndexes);
  assert (pdfa->lastState < pdfa->maxStates);

  if (dfa_verbose > 0)
    fprintf (stderr, "saving dfa %s info file %s\n", pdfa->name,
	     f_name);

  f = fopen (f_name, "w");
  if (f == NULL)
    {
      perror (f_name);
      exit (1);
    }

  /* save the old dfa */
  memcpy (&old_dfa, pdfa, sizeof (dfa_t));

  /* initialize header */
  pdfa->indexes = NULL;
  pdfa->states = NULL;
  pdfa->maxIndexes = pdfa->lastIndex + 1;
  pdfa->maxStates = pdfa->lastState + 1;
  strcpy (pdfa->head, "gnugo dfa file    \n");
  if (pdfa->name[0] == 0)
    strcpy (pdfa->name, "noname\n                \n");
  strcpy (pdfa->copyleft, "Free Software Foundation\n");

  /* save */
  fwrite (pdfa, sizeof (dfa_t), 1, f);
  /* restore the old dfa */
  memcpy (pdfa, &old_dfa, sizeof (dfa_t));

  message =
    fwrite (pdfa->indexes, sizeof (attrib_t), pdfa->lastIndex + 1, f);
  if (pdfa->lastIndex + 1 != message)
    {
      perror (f_name);
      exit (1);
    }

  message =
    fwrite (pdfa->states, sizeof (state_t), pdfa->lastState + 1, f);
  if (pdfa->lastState + 1 != message)
    {
      perror ("f_name");
      exit (1);
    }

  fclose (f);

}


/*
 * load_dfa:
 * Load a dfa from a *.dfa file and allocate
 * memory for it.
 * return pdfa if no error occured or NULL 
 * if the file could not be loaded.
 */

dfa_t*
load_dfa (const char *f_path, const char *f_name, dfa_t * pdfa)
{
  int message, error;
  char * full_path;
  FILE *f;

  
  assert(f_path!=NULL && f_name!=NULL);
  full_path = calloc(2+strlen(f_name)+strlen(f_path),sizeof(char));
  strcpy(full_path, f_path);
  strcat(full_path, f_name);

  f = fopen (full_path, "r");
  error = (f == NULL);
  if (error)
    return NULL;

  if(pdfa == NULL)
      pdfa=malloc(sizeof(dfa_t));
  error|=(pdfa == NULL);

  fprintf (stderr, "loading dfa file %s\n", full_path);

  /* First we do some security check */
  error |= fread (pdfa, sizeof (dfa_t), 1, f) != 1;
  error |= (strcmp (pdfa->head, "gnugo dfa file    \n") != 0);
  error |= (strcmp (pdfa->copyleft, "Free Software Foundation\n") != 0);
  error |= (pdfa->lastIndex + 1 != pdfa->maxIndexes);
  error |= (pdfa->lastState + 1 != pdfa->maxStates);

  if (error)
    {
      /* FIXME: problem if builded on another platform */
      fprintf (stderr, "corrupted dfa file: %s rebuild it.",full_path);
      exit (1);
    }

  /* Second we allocate memory */
  resize_dfa (pdfa, pdfa->maxStates, pdfa->maxIndexes);

  /* Then we read the indexes */
  message =
    fread (pdfa->indexes, sizeof (attrib_t), pdfa->lastIndex + 1, f);
  if (pdfa->lastIndex + 1 != message)
    {
      perror ("f_name");
      exit (1);
    }

  /* And the states */
  message =
    fread (pdfa->states, sizeof (state_t), pdfa->lastState + 1, f);
  if (pdfa->lastState + 1 != message)
    {
      perror ("f_name");
      exit (1);
    }
  fclose (f);
  free(full_path);

  return pdfa;
}


/*
 * Create a linear dfa from a string and an attributes value
 * and resize the dfa if needed.
 *
 * For eg.
 * create_dfa (pdfa, "Oo?.", 2001)
 * gives:
 *
 *           1              0,1            0,1,2             0
 * (1,{}) -------> (2,{}) -------> (3,{}) -------> (4,{}) ------> (5,{2001})
 *                                                  
 * An empty string force a junk pattern : The scanner will always 
 * consider this pattern as active.
 *
 * The possible input symbols are :
 * 
 * '.', ',', '*', '!' for EMPTY expected.
 * 'X'                for BLACK expected.
 * 'O'                for WHITE expected.
 * 'x'                for BLACK|EMPTY expected.
 * 'o'                for WHITE|EMPTY expected.
 * '#', '+', '-', '|' for OUT_BOARD expected.
 * '?'                for EMPTY|BLACK|WHITE expected.
 * '$'                for EMPTY|BLACK|WHITE|OUT_BOARD expected.
 */

static void
create_dfa (dfa_t * pdfa, const char *str, int att_val)
{
  int new_state;

  if (dfa_verbose > 1)
    fprintf (stderr, "linear dfa in %s with string\n%s\n", pdfa->name,
	     str);

  assert (str != NULL);
  assert (pdfa->maxStates > 1);
  assert (pdfa->maxIndexes > 1);

  clean_dfa (pdfa);
  new_state = 1;
  for (; *str != '\0' && strchr ("$#+-|OoXx.?,!a*", *str); str++)
    {
      memset (pdfa->states[new_state].next, 0, 4 * sizeof (int));
      if (strchr ("$?.ox,a!*", *str))
	pdfa->states[new_state].next[0] = new_state + 1;
      if (strchr ("$?Oo", *str))
	pdfa->states[new_state].next[1] = new_state + 1;
      if (strchr ("$?Xx", *str))
	pdfa->states[new_state].next[2] = new_state + 1;
      if (strchr ("$#+-|", *str))
	pdfa->states[new_state].next[OUT_BOARD] = new_state + 1;
      new_state++;
      if (new_state >= pdfa->maxStates)
	resize_dfa (pdfa, pdfa->maxStates + DFA_RESIZE_STEP,
		    pdfa->maxIndexes);
    }
  memset (pdfa->states[new_state].next, 0, 4 * sizeof (int));

  pdfa->lastIndex++;
  if (pdfa->lastIndex >= pdfa->maxIndexes)
    resize_dfa (pdfa, pdfa->maxStates,
		pdfa->maxIndexes + DFA_RESIZE_STEP);

  memset (&(pdfa->indexes[pdfa->lastIndex]), 0, sizeof (attrib_t));
  pdfa->states[new_state].att = pdfa->lastIndex;

  pdfa->indexes[pdfa->states[new_state].att].val = att_val;
  pdfa->indexes[pdfa->states[new_state].att].next = 0;
  pdfa->lastState = new_state;
}



/* 
 * Synchronization product between two automata.
 *
 * L(A) is the set of patterns recognized by the automaton A.
 *
 * A syncronized product betwenn two acyclic deterministic automata
 * A1 and A2 is an acyclic deterministic classifier A1xA2 that 
 * recognize and classify the languages 
 * L(A1), L(A2), L(A1 Union A2) and L(A1 Inter A2).
 *
 * This algorithm do the product and the reduction at the same time.
 *
 * See Hopcroft & Ullman "The design and analysis of computer algorithms"
 * Ed. Addison-Wesley, Reading MA, 1974
 * For the theorical aspects.
 */

/* globals used to improve readability */
static dfa_t *gpout, *gpleft, *gpright;

/* array used to test if a state has been already 
   visited and then give its position in the new automaton. */
static int *gptest, goffset;

static void
do_sync_product (int l, int r)
{
  int c;
  int nextl, nextr;
  int actual;

  actual = gpout->lastState;

  /* unify the attributes of states l and r */
  gpout->states[actual].att =
    union_att (gpout, gpleft, gpleft->states[l].att,
	       gpright, gpright->states[r].att);

  /* scan each possible out-transition */
  for (c = 0; c != 4; c++)
    {
      nextl = gpleft->states[l].next[c];
      nextr = gpright->states[r].next[c];
      assert (nextl < gpleft->lastState + 1);
      assert (nextr < gpright->lastState + 1);

      /* transition to (0,0) mean no transition at all */
      if (nextl != 0 || nextr != 0)
	{
	  /* if the out-state doesn't already exist */
	  if (gptest[goffset * nextl + nextr] == 0)
	    {
	      /* create it! */
	      gpout->lastState++;
	      if (gpout->lastState >= gpout->maxStates)
		resize_dfa (gpout, gpout->maxStates + DFA_RESIZE_STEP,
			    gpout->maxIndexes);

	      gptest[goffset * nextl + nextr] = gpout->lastState;

	      /* link it */
	      gpout->states[actual].next[c] = gpout->lastState;

	      /* create also its sub-automaton */
	      do_sync_product (nextl, nextr);
	    }
	  else
	    {
	      /* link it */
	      assert (goffset * nextl + nextr <
		      (gpleft->lastState + 1) * (gpright->lastState +
						 1));
	      gpout->states[actual].next[c] =
		gptest[goffset * nextl + nextr];
	    }
	}
      else
	{
	  /* no output by c from the actual state */
	  gpout->states[actual].next[c] = 0;
	}
    }
}

static void
sync_product (dfa_t * pout, dfa_t * pleft, dfa_t * pright)
{
  int *test_array = NULL;

  pout->lastIndex = 0;

  if (dfa_verbose > 2)
    {
      fprintf (stderr, "Product between %s and %s\n", pleft->name,
	       pright->name);
      fprintf (stderr, "result in %s\n", pout->name);
    }

  test_array =
    malloc (sizeof (int) * (pleft->lastState + 1) *
	    (pright->lastState + 1));
  if (test_array == NULL)
    {
      perror ("sync_product");
      exit (1);
    }
  memset (test_array, 0,
	  sizeof (int) * (pleft->lastState + 1) * (pright->lastState +
						   1));

  gpout = pout;
  gpleft = pleft;
  gpright = pright;
  gptest = test_array;
  goffset = pright->lastState + 1;
  test_array[goffset * 1 + 1] = 1;
  pout->lastState = 1;
  do_sync_product (1, 1);
  free (gptest);
  gptest = NULL;

}

/* 
 * Init/end functions
 */

void
dfa_init (void)
{
  int i,j;

  if (dfa_verbose > 1)
    fprintf (stderr, "dfa: init\n");
  dfa_was_initialized++;
  buildSpiralOrder (spiral);
  new_dfa (&aux_dfa1, "stringAux ");
  new_dfa (&aux_dfa2, "copyAux ");

  /* set the private board to OUT_BOARD */
  for(i =0; i!= DFA_MAX_BOARD*4; i++)
    for(j =0; j!= DFA_MAX_BOARD*4; j++)
	dfa_p[i][j] = OUT_BOARD;
}

void
dfa_end (void)
{
  if (dfa_verbose > 1)
    fprintf (stderr, "dfa: end\n");

  kill_dfa (&aux_dfa1);
  kill_dfa (&aux_dfa2);
  dfa_was_initialized--;
}


/*
 * Add a new string with attribute att_val into the dfa.
 * if the new size of the dfa respect some size conditions
 * return increase in Kb or -1 if the pattern was rejected.
 * This function never rejects string of length <= 1.
 */

int
dfa_add_string (dfa_t * pdfa, const char *str, int pattern_index,
		int max_Kb_size, int max_Kb_increase)
{
  int old_size, new_size;	/* in Kbytes */

  if (dfa_verbose > 1)
    fprintf (stderr, "adding %s\n to dfa %s\n", str, pdfa->name);

  assert (dfa_was_initialized > 0);
  assert (pdfa != NULL);

  old_size = dfa_size (pdfa);
  if (old_size > max_Kb_size && strlen (str) > 1)
    return -1;

  /* We first create a dfa in aux_dfa1 from the string. */
  create_dfa (&aux_dfa1, str, pattern_index);

  /* then we do the synchronization product with dfa */
  sync_product (&aux_dfa2, &aux_dfa1, pdfa);

  /* FIXME: I do not like this cook */
  /* if the size is correct or if str is a very short string */
  new_size = dfa_size (&aux_dfa2);
  if ((new_size < max_Kb_size && new_size - old_size < max_Kb_increase)
      || strlen (str) <= 1)
    {
      /* the result is copied in dfa */
      copy_dfa (pdfa, &aux_dfa2);
      return new_size - old_size;
    }

  return -1;
}


/*
 * Build a pattern string from a pattern.
 * str must refer a buffer of size greater than MAX_ORDER.
 */
void
pattern_2_string (struct pattern *pat, char *str, int trans, int ci,
		  int cj)
{
  char work_space[DFA_MAX_BOARD * 4][DFA_MAX_BOARD * 4];
  int m, n;			/* anchor position */
  int edges, borders, to_test;
  int i, j, k;
  char c;

  m = DFA_MAX_BOARD * 2 + ci;
  n = DFA_MAX_BOARD * 2 + cj;	/* position of the anchor */

  assert (dfa_was_initialized);
  memset (str, 0, MAX_ORDER);
  memset (work_space, '#', sizeof (work_space));

  if (dfa_verbose > 0)
    fprintf (stderr, "converting pattern into string.\n");

  /* basic edge constraints */
  for (i = DFA_MAX_BOARD; i != DFA_MAX_BOARD * 3; i++)
    for (j = DFA_MAX_BOARD; j != DFA_MAX_BOARD * 3; j++)
      work_space[i][j] = '$';

  /* pattern mask */
  for (i = pat->mini + m; i != pat->maxi + m + 1; i++)
    for (j = pat->minj + n; j != pat->maxj + n + 1; j++)
      work_space[i][j] = '?';

  /* more advanced edge constraints */

  /* South constraint */
  if (pat->edge_constraints & SOUTH)
    {
      for (i = m + pat->maxi + 1; i != DFA_MAX_BOARD * 3; i++)
	for (j = 0; j != DFA_MAX_BOARD * 3; j++)
	  work_space[i][j] = '-';
    }

  /* East constraint */
  if (pat->edge_constraints & EAST)
    {
      for (i = 0; i != DFA_MAX_BOARD * 3; i++)
	for (j = n + pat->maxj + 1; j != DFA_MAX_BOARD * 3; j++)
	  work_space[i][j] = '|';
    }

  /* North constraint */
  if (pat->edge_constraints & NORTH)
    {
      for (i = 0; i != m + pat->mini; i++)
	for (j = 0; j != DFA_MAX_BOARD * 4; j++)
	  work_space[i][j] = '-';
    }

  /* West constraint */
  if (pat->edge_constraints & WEST)
    {
      /* we take taking care not to erase the south edge constraint */
      for (i = 0; i != m + pat->maxi + 1; i++)
	for (j = 0; j != n + pat->minj; j++)
	  work_space[i][j] = '|';

      /* We complete the last corner only if necessary */
      if (!(pat->edge_constraints & SOUTH))
	{
	  for (i = m + pat->maxi + 1; i != DFA_MAX_BOARD * 3; i++)
	    for (j = 0; j != n + pat->minj; j++)
	      work_space[i][j] = '|';
	}

    }

  /* dump */
  if (dfa_verbose > 4)
    {
      for (i = DFA_MAX_BOARD - 1; i != DFA_MAX_BOARD * 3 + 1; i++)
	{
	  for (j = DFA_MAX_BOARD - 1; j != DFA_MAX_BOARD * 3 + 1; j++)
	    {
	      if (i == m && j == n)
		fprintf (stderr, "s");	/* mark the anchor */
	      else
		fprintf (stderr, "%c", work_space[i][j]);
	    }
	  fprintf (stderr, "\n");
	}
      fprintf (stderr, "\n");
    }

  /* pattern representation on the work space */
  for (k = 0; k != pat->patlen; k++)
    {
      c = EXPECTED_VAL (pat->patn[k].att);
      assert (work_space[m + pat->patn[k].x - ci]
	      [n + pat->patn[k].y - cj] == '?');
      work_space[m + pat->patn[k].x - ci][n + pat->patn[k].y - cj] = c;
    }

  /* dump */
  if (dfa_verbose > 3)
    {
      for (i = DFA_MAX_BOARD - 1; i != DFA_MAX_BOARD * 3 + 1; i++)
	{
	  for (j = DFA_MAX_BOARD - 1; j != DFA_MAX_BOARD * 3 + 1; j++)
	    {
	      if (i == m && j == n)
		fprintf (stderr, "s");	/* mark the anchor */
	      else
		fprintf (stderr, "%c", work_space[i][j]);
	    }
	  fprintf (stderr, "\n");
	}
      fprintf (stderr, "\n");
    }

  /* Now we can build the smaller pattern string as possible 
   * from the anchor */

  to_test = pat->patlen;	/* How many positions left to test ? */
  edges = pat->edge_constraints;	/* how many constraint tested ? */
  borders = 0xF; 
  /* we must test at least one intersection by border for 
     patterns like
     
     ???
     O.O
     ???
     
     To ensure edge position.
  */

  for (k = 0; (k != MAX_ORDER - 1) && 
	 ((borders > 0) || edges || to_test > 0); k++)
    {
      i = spiral[trans][k].i;
      j = spiral[trans][k].j;

      if (i == pat->maxi)
	borders &= ~SOUTH;
      if (i == pat->mini)
	borders &= ~NORTH;
      if (j == pat->maxj)
	borders &= ~EAST;
      if (j == pat->minj)
	borders &= ~WEST;

      assert (m + i < DFA_MAX_BOARD * 3 && m + i < DFA_MAX_BOARD * 3);
      str[k] = work_space[m + i][n + j];
      assert (strchr ("XOxo.,a!?$#|-+", str[k]));

      if (strchr ("XOxo.,a!", str[k]))
	to_test--;
      if (strchr ("#|-+", str[k]))
	{
	  if (i > pat->maxi)
	    edges &= ~SOUTH;
	  if (i < pat->mini)
	    edges &= ~NORTH;
	  if (j > pat->maxj)
	    edges &= ~EAST;
	  if (j < pat->minj)
	    edges &= ~WEST;
	}
    }

  assert (k < MAX_ORDER);
  str[k] = '\0';		/* end of string */
}



/* 
 * Associate one permutation by transformation 
 */

static int edge_perm[8][4] =
{
    { NORTH, SOUTH,  EAST,  WEST},
    {  EAST,  WEST, SOUTH, NORTH},
    { SOUTH, NORTH,  WEST,  EAST},
    {  WEST,  EAST, NORTH, SOUTH},
    {  EAST,  WEST, NORTH, SOUTH},
    { SOUTH, NORTH,  EAST,  WEST},
    {  WEST,  EAST, SOUTH, NORTH},
    { NORTH, SOUTH,  WEST,  EAST},
};

static int 
permut(int trans, int edge)
{
    int i;
    int new_edge=0;

    for(i=0; i!=4; i++)
	if((1<<i) & edge)
	    new_edge |= edge_perm[trans][i];

    return new_edge;
}

/* 
 * The range check applies the edge
 * constraints implicitly (used by dfa_add_pattern).
 */

static int
range_check_ok(struct pattern *pat, int board_size, int m, int n, int trans,
	       int ci,int cj)
{
  int mini, minj, maxi, maxj;
  int tmini, tminj, tmaxi, tmaxj;
  int edges;
  int ok = 1;

  /* first we copy the window */
  mini=pat->mini;
  maxi=pat->maxi;
  minj=pat->minj;
  maxj=pat->maxj;

  /* then we test if the pattern fit on the board */
  if(1+maxi-mini >= board_size || 1+maxj-minj >= board_size)
    return 0;

  /* We extend it to deal with edges constraint */
  /* FIXME: the old scheme in mkpat.c line is wrong and never used.
   * We should remove this dead code. */
  edges = pat->edge_constraints;
  if (edges & NORTH)
    maxi = board_size - 1 -ci;
  if (edges & SOUTH) 
    mini = maxi+1 - board_size;
  if (edges & WEST)
    maxj = board_size - 1 -cj;
  if (edges & EAST)
    minj = maxj+1 - board_size;

  /* we transform the extended window */
  DFA_TRANSFORM(mini, minj, &tmini, &tminj, trans);
  DFA_TRANSFORM(maxi, maxj, &tmaxi, &tmaxj, trans);

  /* and we check if the pattern fit on the board */
  ok = ok && m+tmini < board_size && m+tmini >=0;
  ok = ok && m+tmaxi < board_size && m+tmaxi >=0;
  ok = ok && n+tminj < board_size && n+tminj >=0;
  ok = ok && n+tmaxj < board_size && n+tmaxj >=0;

  return ok;
}


/*
 * Show a pattern at a given position and a given transformation
 * (used to debug add_pattern).
 */
void 
show_pattern_instance(int board_size, char board[board_size][board_size]) 
{
  int i,j;
  /* we watch to see if all is allright */
  for (i = 0; i != board_size; i++)
    {
      for (j = 0; j != board_size; j++)
	    fprintf (stderr, "%c", board[i][j]);
      fprintf (stderr, "\n");
    }
  fprintf (stderr, "\n");
}


/* 
 * Perform some statistics about patterns morphology.
 */
void
reset_pattern_stats(int board_size, int stats[5][board_size][board_size])
{
    memset(stats,0,sizeof(stats));
}

static int
update_pattern_stats(int board_size, 
	   char board[board_size][board_size], 
	   int  stats[5][board_size][board_size], 
	   int m, int n)
{
  int i,j;

  for(i= 0; i!= board_size ; i++)
    for(j=0; j!= board_size ; j++)
	{
	    if(strchr (".,a!", board[i][j]))
		stats[0][i][j]++; /* count empty */
	    if(strchr ("Oo", board[i][j]))
	        stats[1][i][j]++; /* count black */
	    if(strchr ("Xx", board[i][j]))
	        stats[2][i][j]++; /* count white */
	    if(strchr ("?", board[i][j]))
		stats[3][i][j]++; /* count wildcard */
	}
  stats[4][m][n]++; /* count anchor points */

  return 0;
}


/* 
 * Display pattern statistics.
 */
void
display_pattern_stats(int board_size, 
		      int stats[5][board_size][board_size], int n)
{
    int i,j,k;
    float sum;
    FILE* output = stderr;
    
    for(i=0; i!= board_size; i++)
      {
	for(j=0; j!= board_size ; j++)
	    {
		sum = 0.0;
		for(k=0; k!=4; k++) sum+=(float) stats[k][i][j];
		if(n==4)
		    fprintf(output,"%5d",stats[n][i][j]);
		else
		    fprintf(output,"%5.1f",(100.0*(float) stats[n][i][j])/sum);
	    }
	fprintf(output,"\n");
      }
}

/* 
 * build a pattern string from a board position starting from 
 * a given intersection (m,n) and scanning it following spiral order.
 * (used by dfa_add_pattern).
 */
int
board_to_string(int edges, int to_test, int board_size, 
		char board[board_size][board_size], 
		char* str, int m, int n)
{
  int i,j,k;

  memset(str,'\0',MAX_ORDER+1);

  if(dfa_verbose > 2)
      fprintf(stderr, "To test: %d ,edges: 0x%x\n",to_test,edges);

  /* We scan the pattern's position until all edges constraints
   * and all intersections to be tested are in the string.
   */
  for (k = 0; (k != MAX_ORDER) && (edges || (to_test > 0)); k++)
    {
	
      i = m + spiral[0][k].i;
      j = n + spiral[0][k].j;

      str[k]='?';
      if(i<0 && edges & NORTH) 
	  edges &= ~NORTH;
      if(i>=board_size && edges & SOUTH) 
	  edges &= ~SOUTH;
      if(j<0 && edges & WEST) 
	  edges &= ~WEST;
      if(j>=board_size && edges & EAST) 
	  edges &= ~EAST;
      if(i<0) 
	  str[k]='#';
      if(i>=board_size) 
	  str[k]='#';
      if(j<0) 
	  str[k]='#';
      if(j>=board_size) 
	  str[k]='#';
      
      if (str[k]=='?' && strchr ("XOxo.,a!", board[i][j]))
	{
	  to_test--;
	  str[k]=board[i][j];
	}

    } /* loop over rows */

  return 0;
}

/*
 * Build a pattern string from a pattern.
 *
 * Nota: The pattern values in pat->patn[] must be relative to the
 * top-left point of the pattern, not to the anchor position, that is 
 * why we need ci and cj as arguments.
 * the maxi,mini... values ARE relatives to anchor.
 *
 * The pattern is placed at all its possible positions for all 
 * its transformations. All the positions are stored in a stack
 * and compared with the preceeding to find symetry and other 
 * invariant transformations.
 */

int
dfa_add_pattern (struct pattern *pat, dfa_t *pdfa, int ci,
		 int cj, int board_size, int index,
		 int  stats[5][board_size][board_size])
{
  char board[board_size][board_size]; /* work space */
  int trans;                          /* transformation */
  int old_trans;                      /* the old transformation */
  int m, n;			      /* match position */
  char c;                             /* temporary value variable */
  int i,j;                            /* temporary position variables */
  int k;                              /* iterator */
  char str[MAX_ORDER+1];              /* a string */
  char old_str[MAX_ORDER+1];          /* a copy of this string */
  int edges;                          /* transformed edge constraints */
  int old_size,old_size2;                       

  old_size = dfa_size(pdfa);
  old_str[0] = 'Z'; old_str[1] = '\0';
  old_trans = -1;

  /* for each transformation */
  for(trans = 0; trans != 8 ; trans++)
    {
      /* and each position compatible with edges constraints */
      for(m=0; m!=board_size; m++)
	for(n=0; n!=board_size; n++)
	  if(range_check_ok(pat, board_size, m, n, trans, ci, cj))
	    {
	      /* we first fill the boards with wildcards */
	      memset (board, '?', sizeof (board));

	      /* we draw the pattern on the board */
	      for (k = 0; k != pat->patlen; k++)
		{
		  c = EXPECTED_VAL (pat->patn[k].att);
		  DFA_TRANSFORM(pat->patn[k].x -ci, pat->patn[k].y -cj,
				&i,&j,trans);

		  assert(m+i<board_size && m+i>=0);
		  assert(n+j<board_size && n+j>=0);
		  assert (board[m + i][n + j] == '?');
		  board[m + i][n + j] = c;
		}
	        
	      /* compute the right edges constraints 
               * after transformation of pattern */
	      edges = permut(trans, pat->edge_constraints);

	      /* do some stats about patterns */
	      update_pattern_stats(board_size, board, stats, m, n);

	      /* generate the string */
	      if(old_trans != trans || strcmp(old_str,str))
		  {
		      strcpy(old_str,str);
		      old_trans = trans;
		      board_to_string(edges, pat->patlen, board_size, 
				      board, str, m, n);

		      old_size2 = dfa_size(pdfa);
		      /* add it to the dfa */
		      if(1)
			  dfa_add_string (pdfa, str, index*8+trans, 
					  0xFFFFFF, 0xFFFFFF);

		      if(dfa_verbose > 3)
			  {
			      dump_dfa(stderr, pdfa);
			      show_pattern_instance ( board_size, board);
			      fprintf(stderr,"generated string: %s\n",str);
			      fprintf(stderr,"index: %d, size:+%dKb\n\n",
				      index, dfa_size(pdfa)-old_size2);
			      sleep(1);
			  }

		  }
	    }
    }
  return dfa_size(pdfa)-old_size;
}

/*
 * copy the board p into the dfa private board
 * changing the color if needed.
 */

#ifdef DFA_MATCHER
extern Intersection p[21][21];  /* FIXME: ugly */

void
dfa_compile_for_match(int color)
{
    int i,j;

    for(i = 0; i!= board_size; i++)
	for(j = 0; j!= board_size; j++)
	    dfa_p[DFA_MAX_BOARD+i][DFA_MAX_BOARD+j] = 
		EXPECTED_COLOR(color, p[i][j]);
}


/*
 * set a dfa to start position.
 */
void 
dfa_restart (dfa_t *pdfa, int trans, int m, int n)
{
  pdfa->actual = 1;
  pdfa->actual_index = 0;
  pdfa->trans = trans;
  pdfa->m = m;
  pdfa->n = n;
  pdfa->row = 0;
}


/*
 * Scan the (private) board and return the values found on the way.
 * return -1 if no more pattern can match here.
 */

int
dfa_scan (dfa_t *pdfa)
{
  int value;
  int rd;

  /* if we are already reading a list of indexes */
  if(pdfa->actual_index)
    {
      /* we take the actual value */
      value = pdfa->indexes[pdfa->actual_index].val;
      /* and we seek the following value */
      pdfa->actual_index = pdfa->indexes[pdfa->actual_index].next;
      return value;
    }

  /* if we are not on the error state */
  if(pdfa->actual)
    {
      /* while no list of indexes is attached to actual state */
      while(pdfa->actual && !pdfa->states[pdfa->actual].att)
	{
	  /* we go to next state */
	  rd=read_board(pdfa->trans,pdfa->m,pdfa->n,pdfa->row);
	  pdfa->row++;
	  pdfa->actual = pdfa->states[pdfa->actual].next[rd];
	}

      /* we have reached a new index list so we start reading it */
      pdfa->actual_index = pdfa->states[pdfa->actual].att;
      value = pdfa->indexes[pdfa->actual_index].val;
      pdfa->actual_index = pdfa->indexes[pdfa->actual_index].next;

      /* we already go to next state */
      rd = read_board ( pdfa->trans, pdfa->m, pdfa->n, pdfa->row);
      pdfa->row++;
      pdfa->actual = pdfa->states[pdfa->actual].next[rd];
      return value;
    }

  return -1;
}

#endif /* DFA_MATCHER */















