/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 and 2000 by the Free Software Foundation.      *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <math.h>

#include "interface.h"

#include "sgftree.h"
#include "analyze.h"
#include "gnugo.h"
#include "random.h"
#include "gg_utils.h"


void
play_solo(Gameinfo *gameinfo, int moves)
{
  int passes = 0; /* num. consecutive passes */
  int move_val;
  double t1, t2;
  int save_moves = moves;
  int boardsize = gameinfo->position.boardsize;
 
  /* It tends not to be very imaginative in the opening,
   * so we scatter a few stones randomly to start with.
   * We add two random numbers to reduce the probability
   * of playing stones near the edge.
   */
  
  int n = 6 + 2*gg_rand()%5;
  int i, j;

  sgffile_write_gameinfo(gameinfo, "solo");
 
  /* Generate some random moves. */
  if (boardsize > 6) {
    do {
      do {
	i = (gg_rand() % 4) + (gg_rand() % (boardsize - 4));
	j = (gg_rand() % 4) + (gg_rand() % (boardsize - 4));
      } while (!gnugo_is_legal(&gameinfo->position, i, j, 
			       gameinfo->to_move));
      
      gameinfo_play_move(gameinfo, i, j, gameinfo->to_move);
    } while (--n > 0);
  }
  
  t1 = gg_gettimeofday();
  while (passes < 2 && --moves >= 0 && !time_to_die) {
    move_val = gnugo_genmove(&gameinfo->position, &i, &j, 
			     gameinfo->to_move,
			     gameinfo->move_number);
    gameinfo_play_move(gameinfo, i, j, gameinfo->to_move);

    if (move_val < 0) {
      ++passes;
      printf("%s(%d): Pass\n", gameinfo->to_move==BLACK ? "Black" : "White",
	     gameinfo->move_number);
    } else {
      passes = 0;
      gprintf("%s(%d): %m\n", gameinfo->to_move==BLACK ? "Black" : "White",
	      gameinfo->move_number, i, j);
    }
  }
  t2 = gg_gettimeofday();
  
  /* Two passes and it's over. (EMPTY == BOTH) */
  gnugo_who_wins(&gameinfo->position, EMPTY, gameinfo->komi, stdout);
  if (t2 == t1) {
    printf("%.3f moves played\n", (double) (save_moves-moves));
  } else {
    printf("%.3f moves/sec\n", (save_moves-moves)/(t2-t1));
  }
}


/* ================================================================ */


/* FIXME PRE3.0: this should be in a separate source file, but it
 * is easier to send diffs when there are no new source
 * files to add.
 */


/*
 * Load SGF file and run genmove().
 */

void 
load_and_analyze_sgf_file(SGFNode *head, Gameinfo *gameinfo, 
			  const char *untilstr, int benchmark)
{
  int i, j;
  int next;
  int r;
  
  /* We load the header to get correct boardsize, komi, and handicap
   * for writing.
   */
  load_sgf_header(head, gameinfo); 
  sgffile_write_gameinfo(gameinfo, "load and analyze");
  next = play_sgf_tree(head, gameinfo, untilstr);
  
  if (benchmark) {
    for (r=0; r < benchmark; ++r) {
      genmove(&i, &j, next);
      next = OTHER_COLOR(next);
    }
  }
  else {
    genmove(&i, &j, next);
    
    if (is_pass(i, j)) {
      gprintf("%s move: PASS!\n", next == WHITE ? "white (o)" : "black (X)");
      sgffile_move_made(i, j, next, 0);
    }
    else {
      gprintf("%s move %m\n", next == WHITE ? "white (o)" : "black (X)", i,j);
      gnugo_play_move(&gameinfo->position, i, j, next);
      sgffile_move_made(i, j, next, 0);
    }
  }
}


/*
 * Load SGF file and score the game
 * untilstr:
 * end  - finish the game by selfplaying from the end of the file until 
 *        two passes
 * last - estimate territorial balance at the end of the of the file
 * move - load file until move is reached and estimate territorial balance
 */

void 
load_and_score_sgf_file(SGFTree *tree, Gameinfo *gameinfo, 
			const char *untilstr)
{
  int i, j, move_val, pass;
  int until;
  float score;
  float komi;
  float result;
  char *tempc = NULL;
  char dummy;
  char text[250];
  char winner;
  int next;
  pass = 0;

  assert(tree);
  load_sgf_header(tree->root, gameinfo);
  sgffile_write_gameinfo(gameinfo, "load and score");
  next = play_sgf_tree(tree->root, gameinfo, untilstr);
  komi = gameinfo->komi;
  
  until = atoi(untilstr);
  if (!strcmp(untilstr, "end")) {
    until = 9999;
    do {
      move_val = genmove(&i, &j, next);
      play_move(i, j, next);
      if (move_val >= 0) {
	pass = 0;
	gprintf("%d %s move %m\n", gameinfo->move_number,
		next == WHITE ? "white (o)" : "black (X)", i, j);
      } else {
	++pass;
	gprintf("%d %s move : PASS!\n", gameinfo->move_number, 
		next == WHITE ? "white (o)" : "black (X)");
      }
      sgffile_move_made(i, j, next, move_val);
      sgftreeAddPlay(tree, 0, next, i, j);
      sgfAnalyzer(tree->lastnode, move_val);
      gameinfo->move_number++;
      next = OTHER_COLOR(next);
    } while ((gameinfo->move_number <= until) && (pass < 2));

    if (pass >= 2) {
      /* Calculate the score */
      score = terri_eval[BLACK] - terri_eval[WHITE] - komi;

      if (score>0) {
	sprintf(text, "Black wins by %1.1f points\n", score);
	winner = 'B';
      }
      else if (score<0) {
	sprintf(text, "White wins by %1.1f points\n", -1.0 * score);
	winner = 'W';
      }
      else {
	sprintf(text, "Tie\n");
	winner = '0';
      }
      fputs(text, stdout);
      sgfShowTerritory(tree->lastnode, 0);
      sgfAddComment(tree->lastnode, text);
      sgffile_write_comment(text);
      if (sgfGetCharProperty(tree->root, "RE", &tempc)) {
	if (sscanf(tempc, "%1c%f", &dummy, &result) == 2) {
	  fprintf(stdout, "Result from file: %1.1f\n", result);
	  fputs("GNU Go result and result from file are ", stdout);
	  if ((result==fabs(score)) && (winner==dummy))
	    fputs("identical\n", stdout);
	  else
	    fputs("different\n", stdout);
	      
	} else {
	  if (tempc[2] == 'R') {
	    fprintf(stdout, "Result from file: Resign\n");
	    fputs("GNU Go result and result from file are ", stdout);
	    if (tempc[0] == winner)
	      fputs("identical\n", stdout);
	    else
	      fputs("different\n", stdout);
	  }
	}
      }
      sgfWriteResult(tree->root, score, 1);
    }
  }

  move_val = genmove(&i, &j, next);
  sgfAnalyzer(tree->lastnode, move_val);
  score = terri_eval[BLACK] - terri_eval[WHITE] - komi;
  sprintf(text,
"Black moyo: %3i\n\
White moyo: %3i\n\
Black territory: %3i\n\
White territory: %3i\n\
Dead white stones: %3i\n\
Dead black stones: %3i\n\
%s seems to win by %1.1f points\n",
	  moyo_eval[BLACK], moyo_eval[WHITE],
	  terri_eval[BLACK], terri_eval[WHITE],
	  gameinfo->position.white_captured,
	  gameinfo->position.black_captured,
	  (score>0) ? ("BLACK") : ((score<0) ? ("WHITE") : ("NOBODY")),
	  (score<0) ? (-1*score) : (score));
  if (strcmp(untilstr, "last"))
      sgftreeSetLastNode(tree, sgftreeStartVariant(tree, 0));
  sgfShowMoyo(tree->lastnode, SGF_SHOW_COLOR, 0);
  sgfAddComment(tree->lastnode, text);
  sgffile_write_comment(text);
  fputs(text, stdout);
}


/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
