/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 and 2000 by the Free Software Foundation.      *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <assert.h>
#include <ctype.h>

#include "gnugo.h"
#include "interface.h"
#include "liberty.h"
#include "gtp.h"
#include "random.h"

/* Internal state that's not part of the engine. */
int color_to_move;
float komi;
int handicap;

static void
print_influence(float white_influence[MAX_BOARD][MAX_BOARD],
		float black_influence[MAX_BOARD][MAX_BOARD],
		int influence_regions[MAX_BOARD][MAX_BOARD]);

#define DECLARE(func) static int func(char *s, int id)

DECLARE(gtp_set_boardsize);
DECLARE(gtp_set_komi);
DECLARE(gtp_playblack);
DECLARE(gtp_playwhite);
DECLARE(gtp_is_legal);
DECLARE(gtp_all_legal);
DECLARE(gtp_prisoners);
DECLARE(gtp_what_color);
DECLARE(gtp_countlib);
DECLARE(gtp_findlib);
DECLARE(gtp_quit);
DECLARE(gtp_loadsgf);
DECLARE(gtp_showboard);
DECLARE(gtp_trymove);
DECLARE(gtp_dump_stack);
DECLARE(gtp_popgo);
DECLARE(gtp_increase_depths);
DECLARE(gtp_decrease_depths);
DECLARE(gtp_attack);
DECLARE(gtp_defend);
DECLARE(gtp_owl_attack);
DECLARE(gtp_owl_defend);
DECLARE(gtp_eval_eye);
DECLARE(gtp_dragon_status);
DECLARE(gtp_same_dragon);
DECLARE(gtp_combination_attack);
DECLARE(gtp_reset_life_node_counter);
DECLARE(gtp_get_life_node_counter);
DECLARE(gtp_reset_owl_node_counter);
DECLARE(gtp_get_owl_node_counter);
DECLARE(gtp_reset_reading_node_counter);
DECLARE(gtp_get_reading_node_counter);
DECLARE(gtp_reset_trymove_counter);
DECLARE(gtp_get_trymove_counter);
DECLARE(gtp_genmove_black);
DECLARE(gtp_genmove_white);
DECLARE(gtp_debug_influence);
DECLARE(gtp_debug_move_influence);
DECLARE(gtp_influence);
DECLARE(gtp_move_influence);
DECLARE(gtp_worm_data);
DECLARE(gtp_dragon_data);
DECLARE(gtp_genmove);
DECLARE(gtp_tune_move_ordering);

/* List of known commands. */
static struct gtp_command commands[] = {
  {"quit",             	      gtp_quit},
  {"boardsize",        	      gtp_set_boardsize},
  {"komi",        	      gtp_set_komi},
  {"black",            	      gtp_playblack},
  {"white",            	      gtp_playwhite},
  {"color",            	      gtp_what_color},
  {"countlib",         	      gtp_countlib},
  {"findlib",          	      gtp_findlib},
  {"is_legal",         	      gtp_is_legal},
  {"all_legal",        	      gtp_all_legal},
  {"prisoners",        	      gtp_prisoners},
  {"loadsgf",          	      gtp_loadsgf},
  {"showboard",        	      gtp_showboard},
  {"trymove",          	      gtp_trymove},
  {"dump_stack",       	      gtp_dump_stack},
  {"popgo",            	      gtp_popgo},
  {"increase_depths",  	      gtp_increase_depths},
  {"decrease_depths",  	      gtp_decrease_depths},
  {"attack",           	      gtp_attack},
  {"defend",           	      gtp_defend},
  {"owl_attack",     	      gtp_owl_attack},
  {"owl_defend",     	      gtp_owl_defend},
  {"eval_eye",         	      gtp_eval_eye},
  {"dragon_status",    	      gtp_dragon_status},
  {"same_dragon",    	      gtp_same_dragon},
  {"combination_attack",      gtp_combination_attack},
  {"reset_life_node_counter", gtp_reset_life_node_counter},
  {"get_life_node_counter",   gtp_get_life_node_counter},
  {"reset_owl_node_counter",  gtp_reset_owl_node_counter},
  {"get_owl_node_counter",    gtp_get_owl_node_counter},
  {"reset_reading_node_counter", gtp_reset_reading_node_counter},
  {"get_reading_node_counter",   gtp_get_reading_node_counter},
  {"reset_trymove_counter",   gtp_reset_trymove_counter},
  {"get_trymove_counter",     gtp_get_trymove_counter},
  {"genmove_black",           gtp_genmove_black},
  {"genmove_white",           gtp_genmove_white},
  {"gg_genmove",              gtp_genmove},
  {"debug_influence",         gtp_debug_influence},
  {"debug_move_influence",    gtp_debug_move_influence},
  {"influence",               gtp_influence},
  {"move_influence",          gtp_move_influence},
  {"worm_data",               gtp_worm_data},
  {"dragon_data",             gtp_dragon_data},
  {"tune_move_ordering",      gtp_tune_move_ordering},
  {NULL,                      NULL}
};


/* Start playing using the Go Text Protocol. */
void
play_gtp()
{
  /* Try to make sure that we have a useful level of buffering of stdout. */
#ifdef HAVE_SETLINEBUF
  setlinebuf(stdout);
#else
  setbuf(stdout, NULL);
#endif

  /* Inform the GTP utility functions about the board size. */
  gtp_internal_set_boardsize(19);
  
  /* Prepare pattern matcher and reading code. */
  reset_engine();

  gtp_main_loop(commands);
}


/****************************
 * Administrative commands. *
 ****************************/

/* Function:  Quit
 * Arguments: none
 * Fails:     never
 * Returns:   nothing
 */
static int
gtp_quit(char *s, int id)
{
  UNUSED(s);
  gtp_success(id, "");
  return GTP_QUIT;
}


/***************************
 * Setting the board size. *
 ***************************/

/* Function:  Set the board size to NxN and clear the board.
 * Arguments: integer
 * Fails:     board size outside engine's limits
 * Returns:   nothing
 */
static int
gtp_set_boardsize(char *s, int id)
{
  int boardsize;
  if (sscanf(s, "%d", &boardsize) < 1)
    return gtp_failure(id, "boardsize not an integer");
  
  if (boardsize < MIN_BOARD || boardsize > MAX_BOARD)
    return gtp_failure(id, "unacceptable boardsize");

  board_size = boardsize;
  clear_board();
  gtp_internal_set_boardsize(boardsize);
  return gtp_success(id, "");
}


/***************************
 * Setting komi.           *
 ***************************/

/* Function:  Set the komi.
 * Arguments: float
 * Fails:     incorrect argument
 * Returns:   nothing
 */
static int
gtp_set_komi(char *s, int id)
{
  if (sscanf(s, "%f", &komi) < 1)
    return gtp_failure(id, "komi not a float");
  
  return gtp_success(id, "");
}


/******************
 * Playing moves. *
 ******************/

/* Function:  Play a black stone at the given vertex.
 * Arguments: vertex
 * Fails:     invalid vertex
 * Returns:   nothing
 */
static int
gtp_playblack(char *s, int id)
{
  int i, j;
  if (!gtp_decode_coord(s, &i, &j))
    return gtp_failure(id, "invalid coordinate");

  play_move(i, j, BLACK);
  return gtp_success(id, "");
}


/* Function:  Play a white stone at the given vertex.
 * Arguments: vertex
 * Fails:     invalid vertex
 * Returns:   nothing
 */
static int
gtp_playwhite(char *s, int id)
{
  int i, j;
  if (!gtp_decode_coord(s, &i, &j))
    return gtp_failure(id, "invalid coordinate");

  play_move(i, j, WHITE);
  return gtp_success(id, "");
}


/* Function:  Load an sgf file, possibly up to a move number or the first
 *            occurence of a move.           
 * Arguments: filename + move number, vertex, or nothing
 * Fails:     missing filename or failure to open or parse file
 * Returns:   color to play
 */
static int
gtp_loadsgf(char *s, int id)
{
  char filename[GTP_BUFSIZE];
  SGFNode *sgf;
  Gameinfo gameinfo;
  int n;
  
  if (sscanf(s, "%s %n", filename, &n) != 1)
    return gtp_failure(id, "missing filename");
  
  if ((sgf = readsgffile(filename)) == NULL)
    return gtp_failure(id, "cannot open or parse '%s'", filename);

  gameinfo_clear(&gameinfo, 19); /* Probably unnecessary. */
  load_sgf_header(sgf, &gameinfo);
  color_to_move = play_sgf_tree(sgf, &gameinfo, s+n);
  gnugo_force_to_globals(&gameinfo.position);
  movenum = gameinfo.move_number;
  komi = gameinfo.komi;
  handicap = gameinfo.handicap;
  gtp_internal_set_boardsize(gameinfo.position.boardsize);
  reset_engine();

  gtp_printid(id, GTP_SUCCESS);
  gtp_mprintf("%C", color_to_move);
  return gtp_finish_response();
}


/*****************
 * Board status. *
 *****************/

/* Function:  Return the color at a vertex.
 * Arguments: vertex
 * Fails:     invalid vertex
 * Returns:   "black", "white", or "empty"
 */
static int
gtp_what_color(char *s, int id)
{
  int i, j;
  if (!gtp_decode_coord(s, &i, &j))
    return gtp_failure(id, "invalid coordinate");
  
  return gtp_success(id, color_to_string(p[i][j]));
}


/* Function:  Count number of liberties for the string at a vertex.
 * Arguments: vertex
 * Fails:     invalid vertex, empty vertex
 * Returns:   Number of liberties.
 */
static int
gtp_countlib(char *s, int id)
{
  int i, j;
  if (!gtp_decode_coord(s, &i, &j))
    return gtp_failure(id, "invalid coordinate");

  if (p[i][j] == EMPTY)
    return gtp_failure(id, "vertex must not be empty");

  return gtp_success(id, "%d", countlib(i, j));
}


/* Function:  Return the positions of the liberties for the string at a vertex.
 * Arguments: vertex
 * Fails:     invalid vertex, empty vertex
 * Returns:   Sorted space separated list of vertices.
 */
static int
gtp_findlib(char *s, int id)
{
  int i, j;
  int libi[MAXLIBS];
  int libj[MAXLIBS];
  int libs;
  
  if (!gtp_decode_coord(s, &i, &j))
    return gtp_failure(id, "invalid coordinate");

  if (p[i][j] == EMPTY)
    return gtp_failure(id, "vertex must not be empty");

  libs = findlib(i, j, MAXLIBS, libi, libj);
  gtp_printid(id, GTP_SUCCESS);
  gtp_print_vertices(libs, libi, libj);
  return gtp_finish_response();
}


/* Function:  Tell whether a move is legal.
 * Arguments: move
 * Fails:     invalid move
 * Returns:   1 if the move is legal, 0 if it is not.
 */
static int
gtp_is_legal(char *s, int id)
{
  int i, j;
  int color;
  
  if (!gtp_decode_move(s, &color, &i, &j))
    return gtp_failure(id, "invalid color or coordinate");

  return gtp_success(id, "%d", is_legal(i, j, color));
}


/* Function:  List all legal moves for either color.
 * Arguments: color
 * Fails:     invalid color
 * Returns:   Sorted space separated list of vertices.
 */
static int
gtp_all_legal(char *s, int id)
{
  int i, j;
  int color;
  int movei[MAXLIBS];
  int movej[MAXLIBS];
  int moves = 0;
  
  if (!gtp_decode_color(s, &color))
    return gtp_failure(id, "invalid color");

  for (i=0; i<board_size; i++)
    for (j=0; j<board_size; j++)
      if (p[i][j] == EMPTY && is_legal(i, j, color)) {
	movei[moves] = i;
	movej[moves++] = j;
      }

  gtp_printid(id, GTP_SUCCESS);
  gtp_print_vertices(moves, movei, movej);
  return gtp_finish_response();
}


/* Function:  List the number of prisoners taken by either color.
 * Arguments: color
 * Fails:     invalid color
 * Returns:   Number of prisoners.
 */
static int
gtp_prisoners(char *s, int id)
{
  int color;
  
  if (!gtp_decode_color(s, &color))
    return gtp_failure(id, "invalid color");

  if (color == BLACK)
    return gtp_success(id, "%d", white_captured);
  else
    return gtp_success(id, "%d", black_captured);
}


/**********************
 * Retractable moves. *
 **********************/

/* Function:  Play a stone of the given color at the given vertex.
 * Arguments: move (color + vertex)
 * Fails:     invalid color, invalid vertex, illegal move
 * Returns:   nothing
 */
static int
gtp_trymove(char *s, int id)
{
  int i, j;
  int color;
  if (!gtp_decode_move(s, &color, &i, &j))
    return gtp_failure(id, "invalid color or coordinate");

  if (!trymove(i, j, color, "gtp_trymove", -1, -1))
    return gtp_failure(id, "illegal move");

  return gtp_success(id, "");
}


/* Function:  Undo a trymove.
 * Arguments: none
 * Fails:     stack empty
 * Returns:   nothing
 */
static int
gtp_popgo(char *s, int id)
{
  UNUSED(s);

  if (stackp == 0)
    return gtp_failure(id, "Stack empty.\n");

  popgo();
  return gtp_success(id, "");
}


/*********************
 * Tactical reading. *
 *********************/

/* Function:  Try to attack a string.
 * Arguments: vertex
 * Fails:     invalid vertex, empty vertex
 * Returns:   attack code followed by attack point if attack code nonzero.
 */
static int
gtp_attack(char *s, int id)
{
  int i, j;
  int ai, aj;
  int attack_code;
  
  if (!gtp_decode_coord(s, &i, &j))
    return gtp_failure(id, "invalid coordinate");

  if (p[i][j] == EMPTY)
    return gtp_failure(id, "vertex must not be empty");

  attack_code = attack(i, j, &ai, &aj);
  gtp_printid(id, GTP_SUCCESS);
  gtp_printf("%d", attack_code);
  if (attack_code > 0) {
    gtp_printf(" ");
    gtp_print_vertex(ai, aj);
  }
  return gtp_finish_response();
}  


/* Function:  Try to defend a string.
 * Arguments: vertex
 * Fails:     invalid vertex, empty vertex
 * Returns:   defense code followed by defense point if defense code nonzero.
 */
static int
gtp_defend(char *s, int id)
{
  int i, j;
  int di, dj;
  int defend_code;
  
  if (!gtp_decode_coord(s, &i, &j))
    return gtp_failure(id, "invalid coordinate");

  if (p[i][j] == EMPTY)
    return gtp_failure(id, "vertex must not be empty");

  defend_code = find_defense(i, j, &di, &dj);
  gtp_printid(id, GTP_SUCCESS);
  gtp_printf("%d", defend_code);
  if (defend_code > 0) {
    gtp_printf(" ");
    gtp_print_vertex(di, dj);
  }
  return gtp_finish_response();
}  


/* Function:  Increase depth values by one.
 * Arguments: none
 * Fails:     never
 * Returns:   nothing
 */
static int
gtp_increase_depths(char *s, int id)
{
  UNUSED(s);
  increase_depth_values();
  return gtp_success(id, "");
}  


/* Function:  Decrease depth values by one.
 * Arguments: none
 * Fails:     never
 * Returns:   nothing
 */
static int
gtp_decrease_depths(char *s, int id)
{
  UNUSED(s);
  decrease_depth_values();
  return gtp_success(id, "");
}  


/******************
 * owl reading. *
 ******************/

/* Function:  Try to attack a dragon.
 * Arguments: vertex
 * Fails:     invalid vertex, empty vertex
 * Returns:   attack code followed by attack point if attack code nonzero.
 */
static int
gtp_owl_attack(char *s, int id)
{
  int i, j;
  int ai, aj;
  int attack_code;
  int save_verbose = verbose;
  if (!gtp_decode_coord(s, &i, &j))
    return gtp_failure(id, "invalid coordinate");

  if (p[i][j] == EMPTY)
    return gtp_failure(id, "vertex must not be empty");

  verbose=0;
  examine_position(p[i][j], EXAMINE_DRAGONS_WITHOUT_OWL);
  verbose=save_verbose;
  
  attack_code = owl_attack(i, j, &ai, &aj);
  gtp_printid(id, GTP_SUCCESS);
  gtp_printf("%d", attack_code);
  if (attack_code > 0) {
    gtp_printf(" ");
    gtp_print_vertex(ai, aj);
  }
  return gtp_finish_response();
}  


/* Function:  Try to defend a dragon.
 * Arguments: vertex
 * Fails:     invalid vertex, empty vertex
 * Returns:   defense code followed by defense point if defense code nonzero.
 */
static int
gtp_owl_defend(char *s, int id)
{
  int i, j;
  int di, dj;
  int defend_code;
  int save_verbose = verbose;
  
  if (!gtp_decode_coord(s, &i, &j))
    return gtp_failure(id, "invalid coordinate");

  if (p[i][j] == EMPTY)
    return gtp_failure(id, "vertex must not be empty");

  verbose=0;
  examine_position(p[i][j], EXAMINE_DRAGONS_WITHOUT_OWL);
  verbose=save_verbose;

  defend_code = owl_defend(i, j, &di, &dj);
  gtp_printid(id, GTP_SUCCESS);
  gtp_printf("%d", defend_code);
  if (defend_code > 0) {
    gtp_printf(" ");
    gtp_print_vertex(di, dj);
  }
  return gtp_finish_response();
}  


/********
 * eyes *
 ********/

/* Function:  Evaluate an eye space
 * Arguments: vertex
 * Fails:     invalid vertex
 * Returns:   Minimum and maximum number of eyes. If these differ an
 *            attack and a defense point are additionally returned.
 *            If the vertex is not an eye space or not of unique color,
 *            a single -1 is returned.
 */

static int
gtp_eval_eye(char *s, int id)
{
  int m, n;
  int max, min;
  int attacki, attackj;
  int defendi, defendj;
  int i, j;
  int save_verbose = verbose;

  if (!gtp_decode_coord(s, &m, &n))
    return gtp_failure(id, "invalid coordinate");

  verbose = 0;
  examine_position(BLACK, EXAMINE_DRAGONS_WITHOUT_OWL);
  verbose = save_verbose;
  
  if (black_eye[m][n].color == BLACK_BORDER) {
    i = black_eye[m][n].origini;
    j = black_eye[m][n].originj;
    compute_eyes(i, j, &max, &min, &attacki, &attackj, &defendi, &defendj,
		 black_eye, half_eye, 0, EMPTY);
  }
  else if (white_eye[m][n].color == WHITE_BORDER) {
    i = white_eye[m][n].origini;
    j = white_eye[m][n].originj;
    compute_eyes(i, j, &max, &min, &attacki, &attackj, &defendi, &defendj,
		 white_eye, half_eye, 0, EMPTY);
  }
  else
    /* Not an eye or not of unique color. */
    return gtp_success(id, "-1");

  gtp_printid(id, GTP_SUCCESS);
  gtp_printf("%d %d", min, max);
  if (max != min) {
    gtp_printf(" ");
    gtp_print_vertex(attacki, attackj);
    gtp_printf(" ");
    gtp_print_vertex(defendi, defendj);
  }
  return gtp_finish_response();
}


/*****************
 * dragon status *
 *****************/

/* Function:  Determine status of a dragon.
 * Arguments: vertex
 * Fails:     invalid vertex, empty vertex
 * Returns:   status ("alive", "critical", "dead", or "unknown"),
 *            attack point, defense point. Points of attack and
 *            defense are only given if the status is critical and the
 *            owl code is enabled.
 *
 * FIXME PRE/POST3.0: Should be able to distinguish between life in seki
 *        and life with territory. Should also be able to identify ko.
 */

static int
gtp_dragon_status(char *s, int id)
{
  int i, j;
  int save_verbose = verbose;

  if (!gtp_decode_coord(s, &i, &j))
    return gtp_failure(id, "invalid coordinate");

  if (p[i][j] == EMPTY)
    return gtp_failure(id, "vertex must not be empty");

  verbose = 0;
  examine_position(BLACK, EXAMINE_DRAGONS);
  verbose = save_verbose;
  
  /* FIXME POST3.0: We should also call the semeai module. */

  if (dragon[i][j].owl_status == UNCHECKED) {
    if (dragon[i][j].status == ALIVE)
      return gtp_success(id, "alive");
  
    if (dragon[i][j].status == DEAD)
      return gtp_success(id, "dead");
  
    if (dragon[i][j].status == UNKNOWN)
      return gtp_success(id, "unknown");

    assert(dragon[i][j].status == CRITICAL); /* Only remaining possibility. */
    return gtp_success(id, "critical");
  }

  /* Owl code active. */
  if (dragon[i][j].owl_status == ALIVE)
    return gtp_success(id, "alive");
  
  if (dragon[i][j].owl_status == DEAD)
    return gtp_success(id, "dead");
  
  if (dragon[i][j].owl_status == UNKNOWN)
    return gtp_success(id, "unknown");
  
  assert(dragon[i][j].owl_status == CRITICAL);
  /* Status critical, need to return attack and defense point as well. */
  gtp_printid(id, GTP_SUCCESS);
  gtp_printf("critical ");
  gtp_print_vertex(dragon[i][j].owl_attacki, dragon[i][j].owl_attackj);
  gtp_printf(" ");
  gtp_print_vertex(dragon[i][j].owl_defendi, dragon[i][j].owl_defendj);
  return gtp_finish_response();
}


/* Function:  Determine whether two stones belong to the same dragon.
 * Arguments: vertex, vertex
 * Fails:     invalid vertex, empty vertex
 * Returns:   1 if the vertices belong to the same dragon, 0 otherwise
 */

static int
gtp_same_dragon(char *s, int id)
{
  int ai, aj;
  int bi, bj;
  int save_verbose = verbose;
  int n;

  n = gtp_decode_coord(s, &ai, &aj);
  if (n == 0)
    return gtp_failure(id, "invalid coordinate");

  if (!gtp_decode_coord(s + n, &bi, &bj))
    return gtp_failure(id, "invalid coordinate");

  if (p[ai][aj] == EMPTY || p[bi][bj] == EMPTY)
    return gtp_failure(id, "vertex must not be empty");

  verbose = 0;
  examine_position(BLACK, EXAMINE_DRAGONS);
  verbose = save_verbose;
  
  return gtp_success(id, "%d", dragon[ai][aj].id == dragon[bi][bj].id);
}


/***********************
 * combination attacks *
 ***********************/

/* Function:  Find a move by color capturing something through a
 *            combination attack.
 * Arguments: color
 * Fails:     invalid color
 * Returns:   Recommended move, PASS if no move found
 */

static int
gtp_combination_attack(char *s, int id)
{
  int color;
  int i, j;
  int save_verbose = verbose;
  int n;

  n = gtp_decode_color(s, &color);
  if (!n)
    return gtp_failure(id, "invalid color");

  verbose = 0;
  examine_position(BLACK, EXAMINE_ALL);
  verbose = save_verbose;

  if (!atari_atari(color, &i, &j, verbose)) {
    i = -1;
    j = -1;
  }
  
  gtp_printid(id, GTP_SUCCESS);
  gtp_print_vertex(i, j);
  return gtp_finish_response();
}


/********************
 * generating moves *
 ********************/

/* Function:  Generate and play the supposedly best black move.
 * Arguments: none
 * Fails:     never
 * Returns:   a move coordinate (or "PASS")
 */
static int
gtp_genmove_black(char *s, int id)
{
  int i, j;
  UNUSED(s);
  
  if (genmove(&i, &j, BLACK) >= 0)
    play_move(i, j, BLACK);

  gtp_printid(id, GTP_SUCCESS);
  gtp_print_vertex(i, j);
  return gtp_finish_response();
}

/* Function:  Generate and play the supposedly best white move.
 * Arguments: none
 * Fails:     never
 * Returns:   a move coordinate (or "PASS")
 */
static int
gtp_genmove_white(char *s, int id)
{
  int i, j;
  UNUSED(s);
  if (genmove(&i, &j, WHITE) >= 0)
    play_move(i, j, WHITE);

  gtp_printid(id, GTP_SUCCESS);
  gtp_print_vertex(i, j);
  return gtp_finish_response();
}

/* Function:  Generate the supposedly best move for either color.
 * Arguments: color to move, optionally a random seed
 * Fails:     invalid color
 * Returns:   a move coordinate (or "PASS")
 */
static int
gtp_genmove(char *s, int id)
{
  int i, j;
  int color;
  int n;
  unsigned int seed;

  n = gtp_decode_color(s, &color);
  if (!n)
    return gtp_failure(id, "invalid color");

  /* This is intended for regression purposes and should therefore be
   * deterministic. The best way to ensure this is to reset the random
   * number generator before calling genmove(). By default it is
   * seeded with 0, but if an optional unsigned integer is given in
   * the command after the color, this is used as seed instead.
   */
  seed = 0;
  sscanf(s+n, "%u", &seed);
  gg_srand(seed);
  
  genmove(&i, &j, color);

  gtp_printid(id, GTP_SUCCESS);
  gtp_print_vertex(i, j);
  return gtp_finish_response();
}


/**************
 * statistics *
 **************/

/* Function:  Reset the count of life nodes.
 * Arguments: none
 * Fails:     never
 * Returns:   nothing
 */
static int
gtp_reset_life_node_counter(char *s, int id)
{
  UNUSED(s);
  reset_life_node_counter();
  return gtp_success(id, "");
}


/* Function:  Retrieve the count of life nodes.
 * Arguments: none
 * Fails:     never
 * Returns:   number of life nodes
 */
static int
gtp_get_life_node_counter(char *s, int id)
{
  int nodes = get_life_node_counter();
  UNUSED(s);
  return gtp_success(id, "%d", nodes);
}


/* Function:  Reset the count of owl nodes.
 * Arguments: none
 * Fails:     never
 * Returns:   nothing
 */
static int
gtp_reset_owl_node_counter(char *s, int id)
{
  UNUSED(s);
  reset_owl_node_counter();
  return gtp_success(id, "");
}


/* Function:  Retrieve the count of owl nodes.
 * Arguments: none
 * Fails:     never
 * Returns:   number of owl nodes
 */
static int
gtp_get_owl_node_counter(char *s, int id)
{
  int nodes = get_owl_node_counter();
  UNUSED(s);
  return gtp_success(id, "%d", nodes);
}


/* Function:  Reset the count of reading nodes.
 * Arguments: none
 * Fails:     never
 * Returns:   nothing
 */
static int
gtp_reset_reading_node_counter(char *s, int id)
{
  UNUSED(s);
  reset_reading_node_counter();
  return gtp_success(id, "");
}


/* Function:  Retrieve the count of reading nodes.
 * Arguments: none
 * Fails:     never
 * Returns:   number of reading nodes
 */
static int
gtp_get_reading_node_counter(char *s, int id)
{
  int nodes = get_reading_node_counter();
  UNUSED(s);
  return gtp_success(id, "%d", nodes);
}


/* Function:  Reset the count of trymoves/trykos.
 * Arguments: none
 * Fails:     never
 * Returns:   nothing
 */
static int
gtp_reset_trymove_counter(char *s, int id)
{
  UNUSED(s);
  reset_trymove_counter();
  return gtp_success(id, "");
}


/* Function:  Retrieve the count of trymoves/trykos.
 * Arguments: none
 * Fails:     never
 * Returns:   number of trymoves/trykos
 */
static int
gtp_get_trymove_counter(char *s, int id)
{
  int nodes = get_trymove_counter();
  UNUSED(s);
  return gtp_success(id, "%d", nodes);
}



/*********
 * debug *
 *********/

/* Function:  Write the position to stderr.
 * Arguments: none
 * Fails:     never
 * Returns:   nothing
 */
static int
gtp_showboard(char *s, int id)
{
  UNUSED(s);
  showboard(0);
  return gtp_success(id, "");
}


/* Function:  Dump stack to stderr.
 * Arguments: none
 * Fails:     never
 * Returns:   nothing
 */
static int
gtp_dump_stack(char *s, int id)
{
  UNUSED(s);
  dump_stack();
  return gtp_success(id, "");
}


/* Function:  Write information about the influence function to stderr.
 * Arguments: color to move, optionally a list of what to show
 * Fails:     invalid color
 * Returns:   nothing
 */
static int
gtp_debug_influence(char *s, int id)
{
  int save_verbose   = verbose;
  int save_debug     = debug;
  int save_printmoyo = printmoyo;
  int color;

  if (!gtp_decode_color(s, &color))
    return gtp_failure(id, "invalid color");

  verbose   = 0;
  debug     = 0;
  printmoyo = 0;
  examine_position(color, EXAMINE_ALL);
  verbose   = save_verbose;
  debug     = save_debug;
  printmoyo = save_printmoyo;

  /* FIXME PRE/POST3.0: Decide the choice of information from the command. */
  printmoyo |= (PRINTMOYO_NUMERIC_INFLUENCE | PRINTMOYO_PRINT_INFLUENCE
		| PRINTMOYO_PERMEABILITY | PRINTMOYO_STRENGTH
		| PRINTMOYO_ATTENUATION);
  print_initial_influence(color, 1);
  printmoyo = save_printmoyo;
  
  return gtp_success(id, "");
}


/* Function:  Write information about the influence function after making
 *            a move to stderr.
 * Arguments: move, optionally a list of what to show
 * Fails:     never
 * Returns:   nothing
 */
static int
gtp_debug_move_influence(char *s, int id)
{
  int save_verbose   = verbose;
  int save_debug     = debug;
  int save_printmoyo = printmoyo;
  int color;
  int i, j;
  char saved_stones[MAX_BOARD][MAX_BOARD];
  
  if (!gtp_decode_move(s, &color, &i, &j))
    return gtp_failure(id, "invalid color or coordinate");

  verbose   = 0;
  debug     = 0;
  printmoyo = 0;
  examine_position(color, EXAMINE_ALL);
  verbose   = save_verbose;
  debug     = save_debug;
  printmoyo = save_printmoyo;

  find_stones_saved_by_move(i, j, color, saved_stones);
  
  /* FIXME PRE/POST3.0: Decide the choice of information from the command. */
  printmoyo |= (PRINTMOYO_NUMERIC_INFLUENCE | PRINTMOYO_PRINT_INFLUENCE
		| PRINTMOYO_PERMEABILITY | PRINTMOYO_STRENGTH);
  print_move_influence(i, j, color, saved_stones);
  printmoyo = save_printmoyo;

  return gtp_success(id, "");
}

/* Function:  Return information about the influence function.
 * Arguments: color to move
 * Fails:     never
 * Returns:   Influence data formatted like:
 *
 * white:
 *   0.51   1.34   3.20   6.60   9.09   8.06   1.96   0.00   0.00 
 *   0.45   1.65   4.92  12.19  17.47  15.92   4.03   0.00   0.00 
 *                   .
 *                   .
 *                   .
 *   0.00   0.00   0.00   0.00   0.00 100.00  75.53  41.47  23.41
 * black:
 *   1.57   2.51   4.10   3.10   3.60   4.54   8.32   4.15   2.71 
 *   2.96   4.62   9.18   5.47   5.89  10.88  20.54  10.19   4.08 
 *                   .
 *                   .
 *                   .
 * 100.00 139.39 100.00 139.39 100.00   0.00   0.00   0.00   0.00
 * regions:
 * -1  0  0  1  1  0 -1 -3 -3
 *              .
 *              .
 *              .
 * -3 -3 -3 -3 -3  3  3  3  3
 *
 * The encoding of the regions is as follows:
 *  3 white territory
 *  2 white moyo
 *  1 white area
 *  0 neutral
 * -1 black area
 * -2 black moyo
 * -3 black territory
 */
static int
gtp_influence(char *s, int id)
{
  int save_verbose   = verbose;
  int save_debug     = debug;
  int save_printmoyo = printmoyo;
  int color;
  float white_influence[MAX_BOARD][MAX_BOARD];
  float black_influence[MAX_BOARD][MAX_BOARD];
  int influence_regions[MAX_BOARD][MAX_BOARD];


  
  if (!gtp_decode_color(s, &color))
    return gtp_failure(id, "invalid color");

  verbose   = 0;
  debug     = 0;
  printmoyo = 0;
  examine_position(color, EXAMINE_ALL);
  verbose   = save_verbose;
  debug     = save_debug;
  printmoyo = save_printmoyo;

  gtp_printid(id, GTP_SUCCESS);
  get_initial_influence(color, 1, white_influence,
			black_influence, influence_regions);
  print_influence(white_influence, black_influence, influence_regions);
  /* We already have one newline and thus can't use gtp_finish_response(). */
  gtp_printf("\n");
  return GTP_OK;
}

static void
print_influence(float white_influence[MAX_BOARD][MAX_BOARD],
		float black_influence[MAX_BOARD][MAX_BOARD],
		int influence_regions[MAX_BOARD][MAX_BOARD])
{
  int m, n;
  gtp_printf("white:\n");
  for (m=0; m<board_size; m++) {
    for (n=0; n<board_size; n++) {
      gtp_printf("%6.2f ", white_influence[m][n]);
    }
    gtp_printf("\n");
  }

  gtp_printf("black:\n");
  for (m=0; m<board_size; m++) {
    for (n=0; n<board_size; n++) {
      gtp_printf("%6.2f ", black_influence[m][n]);
    }
    gtp_printf("\n");
  }

  gtp_printf("regions:\n");
  for (m=0; m<board_size; m++) {
    for (n=0; n<board_size; n++) {
      gtp_printf("%2d ", influence_regions[m][n]);
    }
    gtp_printf("\n");
  }
}


/* Function:  Return information about the influence function after a move
 * Arguments: move
 * Fails:     never
 * Returns:   Influence data formatted in the same way as for gtp_influence.
 */
static int
gtp_move_influence(char *s, int id)
{
  int save_verbose   = verbose;
  int save_debug     = debug;
  int save_printmoyo = printmoyo;
  int color;
  int i, j;
  char saved_stones[MAX_BOARD][MAX_BOARD];
  float white_influence[MAX_BOARD][MAX_BOARD];
  float black_influence[MAX_BOARD][MAX_BOARD];
  int influence_regions[MAX_BOARD][MAX_BOARD];

  if (!gtp_decode_move(s, &color, &i, &j))
    return gtp_failure(id, "invalid color or coordinate");

  verbose   = 0;
  debug     = 0;
  printmoyo = 0;
  examine_position(color, EXAMINE_ALL);
  verbose   = save_verbose;
  debug     = save_debug;
  printmoyo = save_printmoyo;

  find_stones_saved_by_move(i, j, color, saved_stones);
  
  gtp_printid(id, GTP_SUCCESS);
  get_move_influence(i, j, color, saved_stones, white_influence,
		     black_influence, influence_regions);
  print_influence(white_influence, black_influence, influence_regions);
  /* We already have one newline and thus can't use gtp_finish_response(). */
  gtp_printf("\n");
  return GTP_OK;
}

/* Function:  Return the information in the worm data structure.
 * Arguments: optional vertex
 * Fails:     never
 * Returns:   Worm data formatted like:
 *
 * A19:
 * color           black
 * size            10
 * effective_size  17.83
 * origin          A19
 * liberties       8
 * liberties2      15
 * liberties3      10
 * liberties4      8
 * attack          PASS
 * attack_code     0
 * lunch           B19
 * defend          PASS
 * defend_code     0
 * cutstone        2
 * cutstone2       0
 * genus           0
 * ko              0
 * inessential     0
 * B19:
 * color           white
 * .
 * .
 * .
 * inessential     0
 * C19:
 * ...
 *
 * If an intersection is specified, only data for this one will be returned.
 */
static int
gtp_worm_data(char *s, int id)
{
  int i = -1;
  int j = -1;
  int m, n;

  if (sscanf(s, " %*c") >= 0 && !gtp_decode_coord(s, &i, &j))
    return gtp_failure(id, "invalid color or coordinate");

  examine_position(EMPTY, EXAMINE_WORMS);

  gtp_printid(id, GTP_SUCCESS);
  
  for (m=0; m<board_size; m++)
    for (n=0; n<board_size; n++)
      if (i == -1 || (m == i && n == j)) {
	struct worm_data *w = &worm[m][n];
	gtp_print_vertex(m, n);
	gtp_printf(":\n");
	gtp_mprintf("color           %C\n",  w->color);
	gtp_printf("size            %d\n",   w->size);
	gtp_printf("effective_size  %.2f\n", w->effective_size);
	gtp_mprintf("origin          %m\n",  w->origini, w->originj);
	gtp_printf("liberties       %d\n",   w->liberties);
	gtp_printf("liberties2      %d\n",   w->liberties2);
	gtp_printf("liberties3      %d\n",   w->liberties3);
	gtp_printf("liberties4      %d\n",   w->liberties4);
	gtp_mprintf("attack          %m\n",  w->attacki, w->attackj);
	gtp_printf("attack_code     %d\n",   w->attack_code);
	gtp_mprintf("lunch           %m\n",  w->lunchi, w->lunchj);
	gtp_mprintf("defend          %m\n",  w->defendi, w->defendj);
	gtp_printf("defend_code     %d\n",   w->defend_code);
	gtp_printf("cutstone        %d\n",   w->cutstone);
	gtp_printf("cutstone2       %d\n",   w->cutstone2);
	gtp_printf("genus           %d\n",   w->genus);
	gtp_printf("ko              %d\n",   w->ko);
	gtp_printf("inessential     %d\n",   w->inessential);
      }
  
  gtp_printf("\n");
  return GTP_OK;
}

/* Function:  Return the information in the dragon data structure.
 * Arguments: optional intersection
 * Fails:     never
 * Returns:   Dragon data formatted in the corresponding way to gtp_worm_data.
 * FIXME PRE3.0: This is not up to date.
 */
static int
gtp_dragon_data(char *s, int id)
{
  int i = -1;
  int j = -1;
  int m, n;

  if (sscanf(s, " %*c") >= 0 && !gtp_decode_coord(s, &i, &j))
    return gtp_failure(id, "invalid color or coordinate");

  examine_position(EMPTY, EXAMINE_DRAGONS);

  gtp_printid(id, GTP_SUCCESS);
  
  for (m=0; m<board_size; m++)
    for (n=0; n<board_size; n++)
      if (i == -1 || (m == i && n == j)) {
	struct dragon_data *d = &dragon[m][n];
	gtp_print_vertex(m, n);
	gtp_printf(":\n");
	gtp_printf("color           %s\n",   color_to_string(d->color));
	gtp_mprintf("origin          %m\n",  d->origini, d->originj);
	gtp_mprintf("border          %m\n",  d->borderi, d->borderj);
	gtp_printf("size            %d\n",   d->size);
	gtp_printf("effective_size  %.2f\n", d->effective_size);
	gtp_printf("heyes           %d\n",   d->heyes);
	gtp_mprintf("heye            %m\n",  d->heyei, d->heyej);
	gtp_printf("genus           %d\n",   d->genus);
	gtp_printf("escape_route    %d\n",   d->escape_route);
	gtp_mprintf("lunch           %m\n",  d->lunchi, d->lunchj);
	gtp_printf("status          %d\n",   d->status);
	gtp_printf("owl_status      %d\n",   d->owl_status);
	gtp_mprintf("owl_attack      %m\n",  d->owl_attacki,
		                             d->owl_attackj);
	gtp_mprintf("owl_defend      %m\n",  d->owl_defendi,
		                             d->owl_defendj);
	gtp_printf("semeai          %d\n",   d->semeai);
      }
  
  gtp_printf("\n");
  return GTP_OK;
}

/* Function:  Tune the parameters for the move ordering in the tactical
 *            reading.
 * Arguments: MOVE_ORDERING_PARAMETERS integers
 * Fails:     incorrect arguments
 * Returns:   nothing
 */
static int
gtp_tune_move_ordering(char *s, int id)
{
  int params[MOVE_ORDERING_PARAMETERS];
  int k;
  int p;
  int n;

  for (k=0; k<MOVE_ORDERING_PARAMETERS; k++) {
    if (sscanf(s, "%d%n", &p, &n) == 0)
      return gtp_failure(id, "incorrect arguments, expected %d integers",
			 MOVE_ORDERING_PARAMETERS);
    params[k] = p;
    s += n;
  }

  tune_move_ordering(params);
  return gtp_success(id, "");
}

/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
