/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 and 2000 by the Free Software Foundation.      *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */



#include <stdio.h>
#include <stdlib.h>

#include "interface.h"
#include "gnugo.h"
#include "gmp.h"
#include "sgftree.h"
#include "analyze.h"

/* --------------------------------------------------------------*/
/* Play a game against a go-modem-protocol (GMP) client */
/* --------------------------------------------------------------*/
void play_gmp(int boardsize)
{
  Gameinfo  gameinfo;
  SGFTree   sgftree;

  Gmp *ge;
  GmpResult message;
  const char *error;
  
  int i, j;
  int moveval;
  int passes = 0; /* two passes and its over */
  int to_move;  /* who's turn is next ? */

  int mycolor;  /* who has which color */
  int yourcolor;
  float score;
  SGFNode *curnode = NULL; /* current SGFNode */
  int white_terri;
  int black_terri;
  int white_dead;
  int black_dead;
  
  gameinfo_clear(&gameinfo, boardsize);

  mycolor  = gameinfo.computer_player;

  sgftree_clear(&sgftree);
  sgftreeCreateHeaderNode(&sgftree, boardsize, gameinfo.komi);

  yourcolor = OTHER_COLOR(mycolor);

  ge = gmp_create(0, 1);
  TRACE("board size=%d\n", boardsize);

  curnode = sgftree.root;
  
  /* Leave all the -1's so the client can negotiate the game parameters. */
  gmp_startGame(ge, -1, -1, 5.5, 0, -1);
  
  do {
    message = gmp_check(ge, 1, NULL, NULL, &error);
  } while (!time_to_die
	   && ((message == gmp_nothing) 
	       || (message == gmp_reset)));
  
  if (message == gmp_err)  {
    fprintf(stderr, "gnugo-gmp: Error \"%s\" occurred.\n", error);
    exit(1);
  } else if (message != gmp_newGame)  {
    fprintf(stderr, "gnugo-gmp: Expecting a newGame, got %s\n",
	    gmp_resultString(message));
    exit(1);
  }

  gameinfo.handicap = gmp_handicap(ge);
  gameinfo.position.boardsize = gmp_size(ge);
#if 0
  /* This should work, but doesn't: problem is in gmp.c
   * query_komi is not defined in go modem protocol ?!
   */
  gameinfo.komi = gmp_komi(ge);
#endif
  if (gameinfo.handicap > 1)
    gameinfo.komi = 0.5;

  sgfOverwritePropertyInt(sgftree.root, "SZ", gameinfo.position.boardsize);

  TRACE("size=%d, handicap=%d\n, komi=%d", 
	gameinfo.position.boardsize, gameinfo.handicap, gameinfo.komi);

  sgffile_write_gameinfo(&gameinfo, "gmp");
  gameinfo.handicap = gnugo_sethand(&gameinfo.position, gameinfo.handicap,
				    sgftree.root);
  sgfOverwritePropertyInt(sgftree.root, "HA", gameinfo.handicap);

  if (gameinfo.handicap)
    to_move = WHITE;
  else
    to_move = BLACK;

  if (gmp_iAmWhite(ge)) {
    mycolor = WHITE;     /* computer white */
    yourcolor = BLACK;   /* human black */
  }
  else {
    mycolor = BLACK;
    yourcolor = WHITE;
  }

  /* main GMP loop */
  while (passes < 2 && !time_to_die) {

    if (to_move == yourcolor) {
      moveval = 0;
      /* Get opponent's move from gmp client. */
      message = gmp_check(ge, 1, &j, &i, &error);

      if (message == gmp_err) {
	fprintf(stderr, "GNU Go: Sorry, error from gmp client\n");
	sgffile_close_file();
	return;
      }

      if (message == gmp_undo) {
	int movenumber = gameinfo.move_number;
	curnode = sgftree.root;
	movenumber -= j-1;
	if (movenumber < 0) {
	  fprintf(stderr,
		  "GNU Go: %s UNDO: already at the beginning of game tree\n",
		  "play_gmp");
	  continue;
	}
	to_move = sgfPlayTree(&curnode, &gameinfo.position, &movenumber);
	gameinfo.move_number = movenumber-1;
	continue;
      }

      if (message == gmp_pass) {
	++passes;
        curnode = sgfAddPlay(curnode, to_move, -1, -1);
	gnugo_play_move(&gameinfo.position, -1, -1, yourcolor);
	sgffile_move_made(-1, -1, to_move, moveval);
	gameinfo.move_number++;
      } else {
	/* not pass */
	passes = 0;
        curnode = sgfAddPlay(curnode, to_move, i, j);
	TRACE("\nyour move: %m\n\n", i, j);
	gnugo_play_move(&gameinfo.position, i, j, yourcolor);
	sgffile_move_made(i, j, to_move, moveval);
	gameinfo.move_number++;
      }

    } else {
      /* Generate my next move. */
      moveval = gnugo_genmove(&gameinfo.position, &i, &j, mycolor,
			      gameinfo.move_number);
      gameinfo.move_number++;
      gnugo_play_move(&gameinfo.position, i, j, mycolor);
      
      if (moveval < 0) {
	/* pass */
        curnode = sgfAddPlay(curnode, to_move, -1, -1);
	gmp_sendPass(ge);
	sgffile_move_made(-1, -1, to_move, moveval);
	++passes;
      } else {
	/* not pass */
        curnode = sgfAddPlay(curnode, to_move, i, j);
	gmp_sendMove(ge, j, i);
	sgffile_move_made(i, j, to_move, moveval);
	passes = 0;
	TRACE("\nmy move: %m\n\n", i, j);
      }
    }
    
    to_move = OTHER_COLOR(to_move);
  }
  
  /* two passes: game over */
  gmp_sendPass(ge);   
  
  /* We hang around here until cgoban asks us to go, since
   * sometimes cgoban crashes if we exit first.
   */
  
  if (!quiet)
    fprintf(stderr, "Game over - waiting for client to shut us down\n");
  sgffile_close_file();
  who_wins(mycolor, gameinfo.komi, stderr);
  
  
  /* play_gmp() does not return to main(), therefore the analyzerfile
   * writing code is here.
   */
  evaluate_territory(&white_terri, &black_terri, &white_dead, &black_dead);
  score = (black_terri - white_terri
	   - black_dead + white_dead
	   - gameinfo.position.black_captured
	   + gameinfo.position.white_captured - gameinfo.komi);
  sgfWriteResult(sgftree.root, score, 1);
  sgfShowTerritory(curnode, 0);
  if (analyzerfile && sgftree.root) {
    sgf_write_header(sgftree.root, analyzerflag & ANALYZE_OVERWRITE, 
		     gameinfo.seed, gameinfo.komi);
    writesgf(sgftree.root, analyzerfile);
  }
  
  while (!time_to_die) {
    message = gmp_check(ge, 1, &j, &i, &error);
    if (!quiet)
      fprintf(stderr, "Message %d from gmp\n", message);
    if (message == gmp_err)
      break;
  }
  if (!quiet)
    fprintf(stderr, "gnugo going down\n");
}

/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
