/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 and 2000 by the Free Software Foundation.      *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#else
#include <io.h>
#endif

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#include <signal.h>

#include "gg-getopt.h"

#include "interface.h"
#include "gmp.h"
#include "sgftree.h"
#include "analyze.h"
#include "gnugo.h"
#include "random.h"

static void show_copyright(void);
static void show_version(void);
static void show_help(void);
static void show_debug_help(void);
static int get_coordinate(int boardsize, char *str, int *m, int *n);


/* long options which have no short form */
enum {OPT_BOARDSIZE=2,
      OPT_HANDICAPSTONES,
      OPT_COLOR,
      OPT_KOMI,
      OPT_MODE,
      OPT_NOINHIBIT,
      OPT_INFILE,
      OPT_OUTFILE, 
      OPT_QUIET,
      OPT_SHOWCOPYRIGHT,
      OPT_TESTMODE,
      OPT_DECIDE_STRING,
      OPT_DECIDE_DRAGON,
      OPT_DECIDE_SEMEAI,
      OPT_DECIDE_POSITION,
      OPT_DECIDE_EYE,
      OPT_BRANCH_DEPTH,
      OPT_BACKFILL2_DEPTH,
      OPT_SUPERSTRING_DEPTH,
      OPT_AA_DEPTH,
      OPT_OWL_DISTRUST,
      OPT_OWL_BRANCH,
      OPT_OWL_READING,
      OPT_OWL_NODE_LIMIT,
      OPT_LIFE,
      OPT_LIFE_EYESIZE,
      OPT_NOFUSEKIDB,
      OPT_NOJOSEKIDB,
      OPT_LEVEL,
      OPT_SHOWTIME,
      OPT_DEBUG_INFLUENCE,
      OPT_SCORE,
      OPT_PRINTSGF,
      OPT_PROFILE_PATTERNS,
      OPT_ANALYZER_FILE,
      OPT_ANALYZE,
      OPT_HELPANALYZE
};

/* names of playing modes */

enum mode {
  MODE_UNKNOWN=0,
  MODE_ASCII,
  MODE_ASCII_EMACS,
  MODE_GTP,
  MODE_GMP,
  MODE_SGF,
  MODE_LOAD_AND_ANALYZE,
  MODE_LOAD_AND_SCORE,
  MODE_LOAD_AND_PRINT,
  MODE_SOLO,
  MODE_TEST,
  MODE_DECIDE_STRING,
  MODE_DECIDE_DRAGON,
  MODE_DECIDE_SEMEAI,
  MODE_DECIDE_POSITION,
  MODE_DECIDE_EYE
};


/* Definitions of the --long options. Final column is
 * either an OPT_ as defined in the enum above, or it
 * is the equivalent single-letter option.
 * It is useful to keep them in the same order as the
 * help string, for maintenance purposes only.
 */

static struct option const long_options[] =
{
  {"mode",           required_argument, 0, OPT_MODE},
  {"testmode",       required_argument, 0, OPT_TESTMODE},
  {"quiet",          no_argument,       0, OPT_QUIET},

  {"infile",         required_argument, 0, 'l'},
  {"until",          required_argument, 0, 'L'},
  {"outfile",        required_argument, 0, 'o'},

  {"boardsize",      required_argument, 0, OPT_BOARDSIZE},
  {"color",          required_argument, 0, OPT_COLOR},
  {"handicap",       required_argument, 0, OPT_HANDICAPSTONES},
  {"komi",           required_argument, 0, OPT_KOMI},

  {"help",           optional_argument, 0, 'h'},
  {"helpanalyze",    no_argument,       0, OPT_HELPANALYZE },
  {"copyright",      no_argument,       0, OPT_SHOWCOPYRIGHT},
  {"version",        no_argument,       0, 'v'},

  {"allpats",        no_argument,       0, 'a'},
  {"printboard",     no_argument,       0, 'T'},
  {"debug",          required_argument, 0, 'd'},
  {"depth",          required_argument, 0, 'D'},
  {"backfill_depth", required_argument, 0, 'B'},
  {"branch_depth",   required_argument, 0, OPT_BRANCH_DEPTH},
  {"backfill2_depth",required_argument, 0, OPT_BACKFILL2_DEPTH},
  {"superstring_depth",required_argument, 0,SUPERSTRING_DEPTH},
  {"fourlib_depth",  required_argument, 0, 'F'},
  {"ko_depth",       required_argument, 0, 'K'},
  {"aa_depth",       required_argument, 0, OPT_AA_DEPTH},
  {"owl_distrust",   required_argument, 0, OPT_OWL_DISTRUST},
  {"owl_branch",     required_argument, 0, OPT_OWL_BRANCH},
  {"owl_reading",    required_argument, 0, OPT_OWL_READING},
  {"owl_node_limit", required_argument, 0, OPT_OWL_NODE_LIMIT},
  {"level",          required_argument, 0, OPT_LEVEL},
  {"noinhibit",      no_argument,       0, OPT_NOINHIBIT},

  {"memory",         required_argument, 0, 'M'},
  {"hash",           required_argument, 0, 'H'},
  
  {"worms",          no_argument,       0, 'w'},
  {"moyo",           required_argument, 0, 'm'},
  {"benchmark",      required_argument, 0, 'b'},
  {"stack",          no_argument,       0, 's'},
  {"statistics",     no_argument,       0, 'S'},
  {"trace",          no_argument,       0, 't'},
  {"seed",           required_argument, 0, 'r'},
  {"decidestring",   required_argument, 0, OPT_DECIDE_STRING},
  {"decidedragon",   required_argument, 0, OPT_DECIDE_DRAGON},
  {"decidesemeai",   required_argument, 0, OPT_DECIDE_SEMEAI},
  {"decideposition", no_argument,       0, OPT_DECIDE_POSITION},
  {"decideeye",      required_argument, 0, OPT_DECIDE_EYE},
  {"life",           no_argument,       0, OPT_LIFE},
  {"life_eyesize",   required_argument, 0, OPT_LIFE_EYESIZE},
  {"nofusekidb",     no_argument,       0, OPT_NOFUSEKIDB},
  {"nojosekidb",     no_argument,       0, OPT_NOJOSEKIDB},
  {"debuginfluence", required_argument, 0, OPT_DEBUG_INFLUENCE},
  {"showtime",       no_argument,       0, OPT_SHOWTIME},
  {"score",          required_argument, 0, OPT_SCORE},
  {"printsgf",       required_argument, 0, OPT_PRINTSGF},
  {"profile_patterns", no_argument,     0, OPT_PROFILE_PATTERNS},
  {"analyzerfile",   required_argument, 0, OPT_ANALYZER_FILE},
  {"analyze",        required_argument, 0, OPT_ANALYZE},
  {NULL, 0, NULL, 0}
};

float memory = (float) DEFAULT_MEMORY;	  /* Megabytes used for hash table. */

static void sigterm_handler(int);


int
main(int argc, char *argv[])
{
  Gameinfo  gameinfo;
  SGFTree   sgftree;
  int       boardsize = 19;

  int i, umove;
  enum mode playmode = MODE_UNKNOWN;
  enum testmode testmode = UNKNOWN_TESTMODE;
  
  char *infilename = NULL;
  char *untilstring = NULL;

  char *outfile = NULL;
  
  char *printsgffile = NULL;
  
  char decide_this[8];
  char * decide_that = NULL;
  char debuginfluence_move[4] = "\0";
  
  int benchmark = 0;  /* benchmarking mode (-b) */
  float komi = 0.0;
  
  int seed = 0;      /* If seed is zero, GNU Go will play a different game 
			each time. If it is set using -r, GNU Go will play the
			same game each time. (Change seed to get a different
			game). */
  
  /* set SIGTERM handler*/
  signal(SIGTERM, sigterm_handler);

  /* FIXME PRE3.0: Want to do this somewhere more structured. */
  nominal_depth               = DEPTH;
  nominal_backfill_depth      = BACKFILL_DEPTH;
  nominal_backfill2_depth     = BACKFILL2_DEPTH;
  nominal_superstring_depth   = SUPERSTRING_DEPTH;
  nominal_fourlib_depth       = FOURLIB_DEPTH;
  nominal_ko_depth            = KO_DEPTH;
  nominal_branch_depth        = BRANCH_DEPTH;
  nominal_owl_distrust_depth  = OWL_DISTRUST_DEPTH;
  nominal_owl_branch_depth    = OWL_BRANCH_DEPTH;
  nominal_owl_reading_depth   = OWL_READING_DEPTH;
  nominal_owl_node_limit      = OWL_NODE_LIMIT;
  nominal_aa_depth            = AA_DEPTH;
  debug = 0;
  level = 10;
  noinhibit = 0;
  life  = 0;
  life_eyesize = LIFE_EYESIZE;
  fusekidb = 1;
  josekidb = 1;
  
  sgftree_clear(&sgftree);
  gameinfo_clear(&gameinfo, boardsize);
  
  /* Set some standard options. */
  umove = BLACK;
  
  /* Now weed through all of the command line options. */
  while ((i=getopt_long(argc, argv, 
			"-ab:B:d:D:EF:gh::H:K:l:L:M:m:o:p:r:fsStTvw",
			long_options, NULL)) != EOF)
    {
      switch (i) {
      case 'T': printboard++; break;
      case 't': ++verbose; break;
      case 'a': allpats = 1; break;

      case  1 :
      case 'l': infilename = optarg; 
	break;
	
      case 'b': benchmark = atoi(optarg); playmode = MODE_SOLO; break;
      case 'r': seed = atoi(optarg); break;
      case 's': showstack = 1; break;
      case 'S': showstatistics = 1; break;
      case 'w': printworms = 1; break;
      case 'm': printmoyo = strtol(optarg, NULL, 0);  /* allows 0x... */ break;
      case 'd': debug ^= strtol(optarg, NULL, 0);  /* allows 0x... */ break;
      case 'D': nominal_depth = atoi(optarg); break;

      case 'M': memory = atof(optarg); break; /* floating point number */
      case 'H': hashflags = strtol(optarg, NULL, 0);  /* allows 0x... */ break;

      case 'E': printboard = 2; break;
      case 'B': nominal_backfill_depth = atoi(optarg); break;
      case 'F': nominal_fourlib_depth = atoi(optarg); break;
      case 'K': nominal_ko_depth = atoi(optarg); break;

      case 'L':
	untilstring = optarg;
	break;
	
      case 'o':
	outfile = optarg;
	strcpy(gameinfo.outfilename, optarg);
	break;
	
      case OPT_QUIET:
	quiet = 1;
	break;
	
      case OPT_SHOWTIME:
	showtime = 1;
	break;
	
      case OPT_HANDICAPSTONES:
	{
	  int handicap = atoi(optarg);
	  
	  if (handicap < 0 || handicap > MAX_HANDICAP) {
	    fprintf(stderr, "Illegal handicap: %d.\n", handicap);
	    fprintf(stderr, "Try `gnugo --help' for more information.\n");
	    exit(EXIT_FAILURE);
	  }
	  gameinfo.handicap = handicap;
	}
        break;
      
      case OPT_BOARDSIZE: 
	boardsize = atoi(optarg);
	  
	if (boardsize < MIN_BOARD || boardsize > MAX_BOARD) {
	  fprintf(stderr, "Illegal board size: %d.\n", boardsize);
	  fprintf(stderr, "Try `gnugo --help' for more information.\n");
	  exit(EXIT_FAILURE);
	}
	gnugo_clear_position(&gameinfo.position, boardsize);
	break;
	
      case OPT_KOMI: 
	if (sscanf(optarg, "%f", &komi) !=1 ) {
	  fprintf(stderr, "Invalid komi selection: %s\n", optarg);
	  fprintf(stderr, "Try `gnugo --help' for more information.\n");
	  exit(EXIT_FAILURE);
	}
	gameinfo.komi = komi;
	break;
	
      case OPT_MODE: 
	if (strcmp(optarg, "ascii") == 0) 
	  playmode = MODE_ASCII;
	else if (strcmp(optarg, "emacs") == 0)
	  playmode = MODE_ASCII_EMACS;
	else if (strcmp(optarg, "gtp") == 0)
	  playmode = MODE_GTP;
	else if (strcmp(optarg, "gmp") == 0)
	  playmode = MODE_GMP;
	else if (strcmp(optarg,"test") == 0)
	  playmode = MODE_TEST;
	else {
	  fprintf(stderr, "Invalid mode selection: %s\n", optarg);
	  fprintf(stderr, "Try `gnugo --help' for more information.\n");
	  
	  exit(EXIT_FAILURE);
	}
	break;
	
      case OPT_DECIDE_STRING:
	if (strlen(optarg) > 3) {
	  fprintf(stderr, "Invalid board coordinate: %s\n", optarg);
	  exit(EXIT_FAILURE);
	}
	strcpy(decide_this, optarg);
	playmode = MODE_DECIDE_STRING;
	break;
	
      case OPT_DECIDE_DRAGON:
	if (strlen(optarg) > 3) {
	  fprintf(stderr, "Invalid board coordinate: %s\n", optarg);
	  exit(EXIT_FAILURE);
	}
	strcpy(decide_this, optarg);
	playmode = MODE_DECIDE_DRAGON;
	break;
	
      case OPT_DECIDE_SEMEAI:
	if (strlen(optarg) > 7) {
	  fprintf(stderr, "usage: --decidedragon [first dragon]/[second dragon]\n");
	  return (EXIT_FAILURE);
	}
	strcpy(decide_this, optarg);
	strtok(decide_this, "/");
	decide_that = strtok(NULL, "/");
	if (!decide_that) {
	  fprintf(stderr, "usage: --decidedragon [first dragon]/[second dragon]\n");
	  return (EXIT_FAILURE);
	}

	playmode = MODE_DECIDE_SEMEAI;
	break;
	
      case OPT_DECIDE_POSITION:
	playmode = MODE_DECIDE_POSITION;
	break;
	
      case OPT_DECIDE_EYE:
	if (strlen(optarg) > 3) {
	  fprintf(stderr, "Invalid board coordinate: %s\n", optarg);
	  exit(EXIT_FAILURE);
	}
	strcpy(decide_this, optarg);
	playmode = MODE_DECIDE_EYE;
	break;
	
      case OPT_BRANCH_DEPTH:
	nominal_branch_depth = atoi(optarg);
	break;
	
      case OPT_BACKFILL2_DEPTH:
	nominal_backfill2_depth = atoi(optarg);
	break;
	
      case OPT_SUPERSTRING_DEPTH:
	nominal_superstring_depth = atoi(optarg);
	break;
	
      case OPT_AA_DEPTH:
	nominal_aa_depth = atoi(optarg);

      case OPT_OWL_DISTRUST:
	nominal_owl_distrust_depth = atoi(optarg);
	break;
	
      case OPT_OWL_BRANCH:
	nominal_owl_branch_depth = atoi(optarg);
	break;
	
      case OPT_OWL_READING:
	nominal_owl_reading_depth = atoi(optarg);
	break;
	
      case OPT_OWL_NODE_LIMIT:
	nominal_owl_node_limit = atoi(optarg);
	break;
	
      case OPT_LIFE:
 	life = 1;
	break;

      case OPT_LIFE_EYESIZE:
	life = 1;
	life_eyesize = atoi(optarg);
	break;
	
      case OPT_NOFUSEKIDB:
	fusekidb = 0;
	break;
	
      case OPT_NOJOSEKIDB:
	josekidb = 0;
	break;
	
      case OPT_LEVEL:
	level = atoi(optarg);
	break;
	
      case OPT_NOINHIBIT:
	noinhibit = 1;
	break;
	
      case OPT_DEBUG_INFLUENCE:
	if (strlen(optarg) > 3) {
	  fprintf(stderr, "Invalid board coordinate: %s\n", optarg);
	  exit(EXIT_FAILURE);
	}
	strcpy(debuginfluence_move, optarg);
        break;
	
      case OPT_TESTMODE: 
	playmode = MODE_TEST;
	if (strcmp(optarg, "move") == 0)
	  testmode = MOVE_ONLY;
	else if (strcmp(optarg, "annotation") == 0)
	  testmode = ANNOTATION_ONLY;
	else if (strcmp(optarg, "both") == 0) {
	  testmode = BOTH;
	  printf("Testmode: 'both' doesn't currently work \n");
	  exit(EXIT_FAILURE);
	}
	else if (strcmp(optarg,"game") == 0)
	  testmode = GAME;
	else {
	  fprintf(stderr, "Invalid test mode selection: %s\n", optarg);
	  fprintf(stderr, "Try `gnugo --help' for more information.\n");
	  exit(EXIT_FAILURE);
	}
	break;
	
      case OPT_SCORE:
	/* FIXME PRE3.0: Don't reuse untilstring */
	untilstring = optarg;
	if (playmode == MODE_UNKNOWN)
	  playmode = MODE_LOAD_AND_SCORE;
	break;
	
      case OPT_PRINTSGF:
	playmode = MODE_LOAD_AND_PRINT;
	printsgffile = optarg;
	break;
	
      case OPT_PROFILE_PATTERNS:
	profile_patterns = 1;
	prepare_pattern_profiling();
	break;
	
      case OPT_ANALYZER_FILE:
	analyzerfile = optarg;
	break;
	
      case OPT_ANALYZE:
	parse_analyzer_options(optarg);
	break;
	
      case OPT_HELPANALYZE:
	sgfHelp();
	return EXIT_SUCCESS;
	break;
	
      case OPT_COLOR: 
	if (strcmp(optarg, "white") == 0)
	  umove = WHITE;
	else if (strcmp(optarg, "black") == 0) 
	  umove = BLACK;
	else {
	  fprintf(stderr, "Invalid color selection: %s\n", optarg);
	  fprintf(stderr, "Try `gnugo --help' for more information.\n");
	  exit(EXIT_FAILURE);
	}
	break;
	
      case OPT_SHOWCOPYRIGHT: 
	show_copyright();
	return EXIT_SUCCESS;
	break;
	
      case 'v':
	show_version();
	return EXIT_SUCCESS;
	break;
	
      case 'h': 
	show_version();
	if (optarg) {
	  /* In the default behavior of getopt_long with optional args
	   * you need to type "-hdebug"
	   * I can't get long options "--helpdebug" to work at all
	   */
	  if (strncmp(optarg, "analyze", 7) == 0)
	    sgfHelp();
	  if (strncmp(optarg, "debug", 5) == 0)
	    show_debug_help();
	} else {
	  /* This is the trick to get "--help debug" and "-h debug" to work*/
	  if (gg_optind < argc) {
	    if (strncmp(argv[gg_optind], "analyze", 7) == 0)
	      sgfHelp();
	    if (strncmp(argv[gg_optind], "debug", 5) == 0)
	      show_debug_help();
	  } else
	    show_help();
	}
	return EXIT_SUCCESS;
	break;
	
	/* NOTE: getopt returns '?' if an illegal option is supplied. */
      case '?':
      default:
	fprintf(stderr, "Try `gnugo --help' for more information.\n");
	exit(EXIT_FAILURE);
      }
    }

  /* Read the infile if there is one. */
  if (infilename) {
    if (!sgftree_readfile(&sgftree, infilename)) {
      fprintf(stderr, "Cannot open or parse '%s'\n", infilename);
      exit(EXIT_FAILURE);
    }
  }
  
  /* Need to know the board size before we can do this.
   * FIXME POST3.0: This has been broken at some time. Currently we
   * don't have the right boardsize in the relevant case. There is
   * an ugly workaround in influence.c for this.
   */
  if (debuginfluence_move[0]) {
    int m, n;
    get_coordinate(boardsize, debuginfluence_move, &m, &n);
    debug_influence_move(m, n);
  }

  /* Figure out a default mode if there was no explicit one. */
  if (playmode == MODE_UNKNOWN) {
    if (infilename)
      playmode = MODE_LOAD_AND_ANALYZE;
    else
      playmode = (isatty(0)) ? MODE_ASCII : MODE_GMP;
  }
  
  /* start random number seed */
  if (!seed)
    seed = time(0);
  gg_srand(seed);
  gameinfo.seed = seed;
  
  /* Initialize the GNU go engine. */
  init_gnugo(memory);
  
  if (!quiet) {
    fprintf(stderr, "\n");
    show_version();
    show_copyright();
  }
  
  if (outfile) {
    if (playmode != MODE_DECIDE_STRING
	&& playmode != MODE_DECIDE_DRAGON
	&& playmode != MODE_DECIDE_POSITION
	&& playmode != MODE_DECIDE_SEMEAI
	&& playmode != MODE_DECIDE_EYE)
      if (!sgffile_open_file(outfile)) {
	fprintf(stderr, "Error: could not open '%s'\n", optarg);
	exit(EXIT_FAILURE);
      }
  }
  
  switch (playmode) {
  case MODE_GMP:     
    if (!sgftree.root)
      sgftreeCreateHeaderNode(&sgftree, boardsize, komi);
    play_gmp(boardsize);
    break;
    
  case MODE_SOLO:
    if (!sgftree.root)
      sgftreeCreateHeaderNode(&sgftree, boardsize, komi);
    gameinfo.moves = sgftree;	/* FIXME: A kludge. Remove sgftree completely */
    play_solo(&gameinfo, benchmark);
    break;
    
  case MODE_TEST:    
    if (!sgftree.root) {
      fprintf(stderr, "You must use -l infile with test mode.\n");
      exit(EXIT_FAILURE);
    }
    play_test(sgftree.root, testmode);
    break;
    
  case MODE_LOAD_AND_ANALYZE:
    if (!sgftree.root) {
      fprintf(stderr, "You must use -l infile with load and analyze mode.\n");
      exit(EXIT_FAILURE);
    }
    load_and_analyze_sgf_file(sgftree.root, &gameinfo, untilstring, benchmark);
    break;
    
  case MODE_LOAD_AND_SCORE:
    if (!sgftree.root) {
      fprintf(stderr, "gnugo: --score must be used with -l\n");
      exit(EXIT_FAILURE);
    }
    load_and_score_sgf_file(&sgftree, &gameinfo, untilstring);
    break;
    
  case MODE_LOAD_AND_PRINT:
    if (!sgftree.root) {
      fprintf(stderr, "gnugo: --printsgf must be used with -l\n");
      exit(EXIT_FAILURE);
    } else {
      int to_move;

      load_sgf_header(sgftree.root, &gameinfo);
      sgffile_write_gameinfo(&gameinfo, "load and print"); 
      to_move = play_sgf_tree(sgftree.root, &gameinfo, untilstring);
      sgffile_close_file();
      sgffile_open_file(printsgffile);
      sgffile_write_gameinfo(&gameinfo, "load and print"); 
      sgffile_printboard(to_move);
    }
    break;
    
  case MODE_DECIDE_STRING:
    {
      int m, n;
      
      play_sgf_tree(sgftree.root, &gameinfo, untilstring);
      boardsize = gameinfo.position.boardsize;
      
      if (!get_coordinate(boardsize, decide_this, &m, &n)) {
	fprintf(stderr, "gnugo: --decidestring: strange coordinate \n");
	return (EXIT_FAILURE);
      }

      if (!infilename) {
	fprintf(stderr, "gnugo: --decidestring must be used with -l\n");
	return (EXIT_FAILURE);
      }
      decidestring(m, n, outfile);
    }
  break;
  
  case MODE_DECIDE_DRAGON:
    {
      int m, n;
      
      play_sgf_tree(sgftree.root, &gameinfo, untilstring);
      boardsize = gameinfo.position.boardsize;
      if (!get_coordinate(boardsize, decide_this, &m, &n)) {
	fprintf(stderr, "gnugo: --decidedragon: strange coordinate \n");
	return (EXIT_FAILURE);
      }
      if (!infilename) {
	fprintf(stderr, "gnugo: --decidedragon must be used with -l\n");
	return (EXIT_FAILURE);
      }
      decidedragon(m, n, outfile);
    }
    break;
  
  case MODE_DECIDE_SEMEAI:
    {
      int ai, aj, bi, bj;
      
      play_sgf_tree(sgftree.root, &gameinfo, untilstring);
      boardsize = gameinfo.position.boardsize;
      if (!get_coordinate(boardsize, decide_this, &ai, &aj)) {
	fprintf(stderr, "usage: --decidedragon [first dragon]/[second dragon]\n");
	return (EXIT_FAILURE);
      }
      if (!get_coordinate(boardsize, decide_that, &bi, &bj)) {
	fprintf(stderr, "usage: --decidedragon [first dragon]/[second dragon]\n");
	return (EXIT_FAILURE);
      }
      if (!infilename) {
	fprintf(stderr, "gnugo: --decidesemeai must be used with -l\n");
	return (EXIT_FAILURE);
      }
      decidesemeai(ai, aj, bi, bj, outfile);
    }
    break;
    

  case MODE_DECIDE_POSITION:
    {
      int color;
      if (!infilename) {
	fprintf(stderr, "gnugo: --decideposition must be used with -l\n");
	return (EXIT_FAILURE);
      }
      color=play_sgf_tree(sgftree.root, &gameinfo, untilstring);
      decideposition(color, outfile);
    }
    break;
    
  case MODE_DECIDE_EYE:
    {
      int m, n;
      
      play_sgf_tree(sgftree.root, &gameinfo, untilstring);
      boardsize = gameinfo.position.boardsize;
      
      if (!get_coordinate(boardsize, decide_this, &m, &n)) {
	fprintf(stderr, "gnugo: --decideeye: strange coordinate \n");
	return (EXIT_FAILURE);
      }
      
      if (!infilename) {
	fprintf(stderr, "gnugo: --decideeye must be used with -l\n");
	return (EXIT_FAILURE);
      }
      decideeye(m, n, outfile);
    }
    break;
  
  case MODE_GTP:  
    if (!sgftree.root)
      sgftreeCreateHeaderNode(&sgftree, boardsize, komi);
    play_gtp();
    break;

  case MODE_ASCII_EMACS:  
    gameinfo.computer_player = OTHER_COLOR(umove);
    if (!sgftree.root)
      sgftreeCreateHeaderNode(&sgftree, boardsize, komi);
    play_ascii_emacs(&sgftree, &gameinfo, infilename, untilstring);
    break;

  case MODE_ASCII:  
  default:     
    gameinfo.computer_player = OTHER_COLOR(umove);
    if (!sgftree.root)
      sgftreeCreateHeaderNode(&sgftree, boardsize, komi);
    play_ascii(&sgftree, &gameinfo, infilename, untilstring);
    break;
  }
  
  sgffile_close_file();
  
  if (analyzerfile && sgftree.root) {
    sgf_write_header(sgftree.root, analyzerflag & ANALYZE_OVERWRITE, 
		     seed, komi);
    writesgf(sgftree.root, analyzerfile);
  }

  if (profile_patterns)
    report_pattern_profiling();
  
  return 0;
}  /* end main */



/* 
 * Cgoban sends us a sigterm when it wants us to die. But it doesn't
 * close the pipe, so we cannot rely on gmp to pick up an error.
 * 
 * We want to keep control so we can close the output sgf file
 * properly, so we trap the signal.
 */

volatile int  time_to_die = 0;   /* set by signal handlers */

static void 
sigterm_handler(int sig)
{
  time_to_die = 1;
  if (!quiet)
    write(2, "*SIGTERM*\n", 10);  /* bad to use stdio in a signal handler */

  close(0);                  /* this forces gmp.c to return an gmp_err */

  /* I thought signal handlers were one-shot, yet on my linux box it is kept.
   * Restore the default behaviour so that a second signal has the
   * original effect - in case we really are stuck in a loop.
   */
  signal(sig, SIG_DFL);

  /* schedule a SIGALRM in 5 seconds, in case we haven't cleaned up by then
   * - cgoban sends the SIGTERM only once 
   */
#ifdef HAVE_ALARM
  alarm(5);
#endif
}


static void
show_version(void)
{
  fprintf(stderr, "GNU Go Version %s\n", VERSION);
}


/* 
 * This string is modelled after the GNU tar --help output.
 */


#define USAGE "\n\
Usage: gnugo [-opts]\n\
\n\
Main Options:\n\
       --mode <mode>     Force the playing mode ('ascii', 'test' or 'gmp').\n\
                         Default is ASCII.\n\
                         If no terminal is detected GMP (Go Modem Protocol)\n\
                         will be assumed.\n\
       --quiet           Don't print copyright and informational messages\n\
   -l, --infile <file>   Load name sgf file\n\
   -L, --until <move>    Stop loading just before move is played. <move>\n\
                         can be the move number or location (eg L10).\n\
   -o, --outfile file    Write sgf output to file\n\
   --printsgf <file>     Write position as a diagram to file (use with -l)\n\
\n\
Options that affect strength (higher=stronger, slower):\n\
   -D, --depth <depth>          deep reading cutoff (default %d)\n\
   -B, --backfill_depth <depth> deep reading cutoff (default %d)\n\
   -F, --fourlib_depth <depth>  deep reading cutoff (default %d)\n\
   -K, --ko_depth <depth>       deep reading cutoff (default %d)\n\
   --branch_depth <depth>       deep reading cutoff (default %d)\n\
   --backfill2_depth <depth>    deep reading cutoff (default %d)\n\
   --superstring_depth <depth>  deep reading cutoff (default %d)\n\
   --aa_depth <depth>           deep reading cutoff (default %d)\n\
   --owl_distrust <depth>       owl distrust depth (default %d)\n\
   --owl_branch <depth>         owl branching depth (default %d)\n\
   --owl_reading <depth>        owl reading depth (default %d)\n\
   --owl_node_limit <limit>     max nodes for owl reading (default %d)\n\
   --level <amount>             amount = 1, 2 or more:\n\
                                play faster and weaker (default 0)\n\
\n\
Experimental options:\n\
   --life                       use life code\n\
   --life_eyesize <n>           use life code, life_eyesize=n (default %d)\n\n\
   --nofusekidb                 turn off fuseki database\n\
   --nojosekidb                 turn off joseki database\n\
\n\
Option that affects speed (higher=faster, more memory usage):\n\
   -M, --memory <megabytes>     hash memory (default %4.1f)\n\n\
Game Options: (--mode ascii)\n\
       --boardsize num   Set the board size to use (%d--%d)\n\
       --color <color>   Choose your color ('black' or 'white')\n\
       --handicap <num>  Set the number of handicap stones (0--%d)\n\
       --komi <num>      Set the komi\n\
\n\
Informative Output:\n\
   -v, --version         Display the version of GNU Go\n\
   -h, --help            Display this help message\n\
       --help analyze    Display help about analyzer options\n\
       --help debug      Display help about debugging options\n\
       --copyright       Display copyright notice\n\
\n\
"


#define COPYRIGHT "\n\n\
This is GNU GO, a Go program. Contact gnugo@gnu.org, or see\n\
http://www.gnu.org/software/gnugo/ for more information.\n\n\
Copyright 1999 and 2000 by the Free Software Foundation.\n\n\
This program is free software; you can redistribute it and/or\n\
modify it under the terms of the GNU General Public License\n\
as published by the Free Software Foundation - version 2.\n\n\
This program is distributed in the hope that it will be\n\
useful, but WITHOUT ANY WARRANTY; without even the implied\n\
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR\n\
PURPOSE.  See the GNU General Public License in file COPYING\n\
for more details.\n\n\
You should have received a copy of the GNU General Public\n\
License along with this program; if not, write to the Free\n\
Software Foundation, Inc., 59 Temple Place - Suite 330,\n\
Boston, MA 02111, USA\n\n\
"
#define USAGE_DEBUG "\n\
Debugging Options:\n\
\n\
       --testmode <mode> set the test mode to one of the following:\n\
                         (requires -l, implies --mode test)\n\
                     move: test at move node only\n\
                     annotation: test at annotation node only\n\
                     both: test at move and annotation nodes\n\
                     game: test to see if gnugo considered each move made\n\
                 This overrides a testmode=... comment in the sgf file\n\
   -a, --allpats                test all patterns\n\
   -T, --printboard             colored display of dragons\n\
   -E                           colored display of eye spaces\n\
   -d, --debug <level>          debugging output (see gnugo.h for bits)\n\
   -H, --hash <level>           hash (see gnugo.h for bits)\n\
   -w, --worms                  print worm and dragon data and move reasons\n\
   -m, --moyo <level>           moyo debugging, show moyo board\n\
   -b, --benchmark num          benchmarking mode - can be used with -l\n\
   -s, --stack                  stack trace (for debugging purposes)\n\n\
   -S, --statistics             print statistics (for debugging purposes)\n\n\
   -t, --trace                  verbose tracing (use twice or more to trace reading)\n\
   --showtime                   print the amount of time genmove takes to find move\n\
   --noinhibit                  estimate moyos boldly\n\
   -r, --seed number            set random number seed\n\
       --decidestring <string>  can this string live? (try with -o)\n\
       --decidedragon <dragon>  can this dragon live? (try with -o or -t)\n\
       --decideposition         evaluate all dragons (try with -o or -t)\n\
       --decideeye string       evaluate the eye?\n\
       --life <eyesize>         use eye reading code\n\
       --nofusekidb             turn off fuseki database\n\
       --nojosekidb             turn off joseki database\n\
       --debuginfluence <move>  print influence map after making a move\n\
       --score <end|last|move>  count or estimate territory\n\
       --profile_patterns       print statistics for pattern usage.\n\
"
    


static void
show_help(void)
{
  fprintf(stderr, USAGE, 
	  DEPTH, BACKFILL_DEPTH, FOURLIB_DEPTH, KO_DEPTH, BRANCH_DEPTH,
	  BACKFILL2_DEPTH, SUPERSTRING_DEPTH, AA_DEPTH, 
	  OWL_DISTRUST_DEPTH, OWL_BRANCH_DEPTH,
	  OWL_READING_DEPTH, OWL_NODE_LIMIT, LIFE_EYESIZE,
	  (float) DEFAULT_MEMORY, MIN_BOARD, MAX_BOARD, MAX_HANDICAP);
}


static void
show_debug_help(void)
{
  fprintf(stderr, USAGE_DEBUG);
}


static void
show_copyright(void)
{
  fputs(COPYRIGHT, stderr);
}

/*
 * Get the (m, n) coordinates in the standard GNU Go coordinate system
 * from the string STR.  This means that m is the nth row from the top
 * and n is the column. Both coordinates are between 0 and boardsize-1,
 * inclusive.
 *
 * Return 1 if ok, otherwise return 0;
 */

static int
get_coordinate(int boardsize, char *str, int *m, int *n)
{
  if (*str == '\0')
    return 0;

  if (!isalpha((int) *str))
    return 0;
  *n = tolower((int) *str) - 'a';
  if (tolower((int) *str) >= 'i')
    --*n;
  if (*n < 0 || *n > boardsize - 1)
    return 0;

  if (!isdigit((int) *(str+1)))
    return 0;
  *m = boardsize - atoi(str + 1);
  if (*m < 0 || *m > boardsize - 1)
    return 0;

  return 1;
}





/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
