/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 and 2000 by the Free Software Foundation.      *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <assert.h>

#include "liberty.h"
#include "sgftree.h"
#include "analyze.h"

#include "gg_utils.h"


/*
 * Change the status of all the stones in the dragon at (x,y).
 */

void
change_dragon_status(int x, int y, int status)
{
  int i,j;
  int origini = dragon[x][y].origini;
  int originj = dragon[x][y].originj;

  for (i=0; i<board_size; i++) 
    for (j=0; j<board_size; j++) {
      if (dragon[i][j].origini == origini
	  && dragon[i][j].originj == originj)
	dragon[i][j].status = status;
    }
}


/* 
 * Measure territory.
 */

void
count_territory(int *white, int *black)
{
  int i, j;

  *white = 0;
  *black = 0;
  make_worms();

  for (i=0; i<board_size; i++)
    for (j=0; j<board_size; j++)
      if (p[i][j] == EMPTY) {
	if (worm[i][j].color == BLACK_BORDER)
	  ++*black;
	else if (worm[i][j].color==WHITE_BORDER)
	  ++*white;
      }
}


/*
 * Evaluate the territory for both sides.
 * Removes dead dragons before counting.
 *
 * NOTE: The position cannot be reused after this operation.
 */

void
evaluate_territory(int *white_terri, int *black_terri, 
		   int *white_dead, int *black_dead)
{
  int orig_white_captured;
  int orig_black_captured;
  int i, j;

  *white_terri = 0;
  *black_terri = 0;

  orig_white_captured = white_captured;
  orig_black_captured = black_captured;

  /* First we remove all dead dragons. */
  for (i=0; i<board_size; i++)
    for (j=0; j<board_size; j++) {
      if (dragon[i][j].status == DEAD
	  && p[i][j] != EMPTY)
	remove_string(i, j);
    }

  *white_dead = white_captured - orig_white_captured;
  *black_dead = black_captured - orig_black_captured;

  /* Make hashdata correct after removals. */
  hashdata_recalc(&hashdata, p, ko_i, ko_j);

  make_worms();
  make_dragons(EMPTY, 0);
  for (i=0; i<board_size; i++)
    for (j=0; j<board_size; j++)
        switch(p[i][j]) {
        case EMPTY:
	  if (dragon[i][j].color == BLACK_BORDER)
	    ++*black_terri;
	  else if (dragon[i][j].color == WHITE_BORDER)
	    ++*white_terri;
	  break;
        case BLACK:
        case WHITE:
	  break;
      }
}


/* 
 * Change_defense(ai, aj, ti, tj, dcode) moves the point of defense of the
 * worm at (ai, aj) to (ti, tj), and sets worm[a].defend_code to dcode.
 *
 * FIXME PRE3.0: I think this function is obsolete in the new move generation.
 */

void
change_defense(int ai, int aj, int ti, int tj, int dcode)
{
  int origini = worm[ai][aj].origini;
  int originj = worm[ai][aj].originj;
  int si = worm[ai][aj].defendi;  /* Old defense point. */
  int sj = worm[ai][aj].defendj;
  
  assert (stackp==0);

  if (ti == -1)
    TRACE("Removed defense of %m (was %m).\n", origini, originj, si, sj);
  else if (si == -1)
    TRACE("Setting defense of %m to %m.\n", origini, originj, ti, tj);
  else
    TRACE("Moved defense of %m from %m to %m.\n", origini, originj,
	  si, sj, ti, tj);
  
  assert (ti == -1 || p[ti][tj] == EMPTY);
  assert (p[ai][aj] != EMPTY);

  if ((worm[ai][aj].attack_code != 0)
      && (worm[ai][aj].defend_code != 0)
      && (ti == -1)) 
  {
    int m, n;

    for (m=0; m<board_size; m++)
      for (n=0; n<board_size; n++)
	if ((dragon[m][n].origini == dragon[ai][aj].origini) 
	    && (dragon[m][n].originj == dragon[ai][aj].originj))
	  dragon[m][n].status = DEAD;
  }
  worm[origini][originj].defendi = ti;
  worm[origini][originj].defendj = tj;
  worm[origini][originj].defend_code = dcode;
  propagate_worm(origini, originj);

  if (ti != -1)
    add_defense_move(ti, tj, origini, originj);
  else if (si != -1)
    remove_defense_move(si, sj, origini, originj);
}


/*
 * change_attack(ai, aj, ti, tj, acode) moves the point of attack of the
 * worm at (ai, aj) to (ti, tj), and sets worm[a].attack_code to acode.
 */

void
change_attack(int ai, int aj, int ti, int tj, int acode)
{
  int origini = worm[ai][aj].origini;
  int originj = worm[ai][aj].originj;
  int si = worm[ai][aj].attacki;  /* Old attack point. */
  int sj = worm[ai][aj].attackj;
  
  assert(stackp==0);

  if ((ti == -1) && (si != -1)) 
    TRACE("Removed attack of %m (was %m).\n", origini, originj, si, sj);
  else if (si == -1)
    TRACE("Setting attack of %m to %m.\n", origini, originj, ti, tj);
  else
    TRACE("Moved attack of %m from %m to %m.\n", origini, originj,
	  si, sj, ti, tj);

  assert (ti == -1 || p[ti][tj] == EMPTY);
  assert (p[ai][aj] != EMPTY);

  worm[origini][originj].attacki = ti;
  worm[origini][originj].attackj = tj;
  worm[origini][originj].attack_code = acode;
  propagate_worm(origini, originj);

  if (ti != -1)
    add_attack_move(ti, tj, origini, originj);
  else if (si != -1)
    remove_attack_move(si, sj, origini, originj);
}


/*
 * Check whether a move at (ti,tj) stops the enemy from playing at (ai,aj).
 */

int
defend_against(int ti, int tj, int color, int ai, int aj)
{
  if (trymove(ti, tj, color, "defend_against", -1, -1)) {
    if (safe_move(ai, aj, OTHER_COLOR(color)) == 0) {
      popgo();
      return 1;
    }
    popgo();
  }
  return 0;
}


/* 
 * Returns true if color can cut at (i,j). This information is
 * collected by find_cuts(), using the B patterns in the connections
 * database.
 */

int
cut_possible(int i, int j, int color)
{
  if (color == WHITE)
    return black_eye[i][j].cut;
  else
    return white_eye[i][j].cut;
}


/*
 * does_attack(ti, tj, ai, aj) returns true if the move at (ti, tj)
 * attacks (ai, aj). This means that it captures the string, and that
 * (ai, aj) is not already dead.
 */

int
does_attack(int ti, int tj, int ai, int aj)
{
  int color = p[ai][aj];
  int other = OTHER_COLOR(color);
  int result = 0;
  int si = -1;
  int sj = -1;
  
  if (stackp == 0) {
    if (worm[ai][aj].attack_code != 0 && worm[ai][aj].defend_code == 0)
      return 0;
    si = worm[ai][aj].defendi;
    sj = worm[ai][aj].defendj;
  }
  else if (attack(ai, aj, NULL, NULL) && !find_defense(ai, aj, &si, &sj))
    return 0;
  
  if (trymove(ti, tj, other, "does_attack-A", ai, aj)) {
    if (!p[ai][aj] || !find_defense(ai, aj, NULL, NULL)) {
      result = 1;
      increase_depth_values();
      if (si != -1 && trymove(si, sj, color, "does_attack-B", ai, aj)) {
	if (p[ai][aj] && !attack(ai, aj, NULL, NULL))
	  result = 0;
	popgo();
      }
      decrease_depth_values();
    }
    popgo();
  }

  return result;
}


/*
 * does_defend(ti, tj, ai, aj) returns true if the move at (ti, tj)
 * defends (ai, aj). This means that it defends the string, and that
 * (ai, aj) can be captured if no defense is made.
 */

int
does_defend(int ti, int tj, int ai, int aj)
{
  int color = p[ai][aj];
  int other = OTHER_COLOR(color);
  int result = 0;
  int si = -1;
  int sj = -1;

  if (stackp == 0) {
    if (worm[ai][aj].attack_code == 0)
      return 0;
    else {
      si = worm[ai][aj].attacki;
      sj = worm[ai][aj].attackj;
    }
  }
  else if (!attack(ai, aj, &si, &sj))
    return 0;

  assert(si != -1 && sj != -1);
  
  if (trymove(ti, tj, color, "does_defend-A", ai, aj)) {
    if (!attack(ai, aj, NULL, NULL)) {
      result = 1;
      increase_depth_values();
      if (trymove(si, sj, other, "does_defend-B", ai, aj)) {
	if (!p[ai][aj] || !find_defense(ai, aj, NULL, NULL))
	  result = 0;
	popgo();
      }
      decrease_depth_values();
    }
    popgo();
  }

  return result;
}


/* 
 * Example: somewhere(WHITE, 2, ai, aj, bi, bj, ci, cj).
 * 
 * Returns true if one of the vertices listed
 * satisfies p[i][j]==color. Here last_move is the
 * number of moves minus one.
 */

int
somewhere(int color, int last_move, ...)
{
  va_list ap;
  int k;
  
  va_start(ap, last_move);
  for (k = 0; k <= last_move; k++) {
    int ai, aj;

    ai = va_arg(ap, int);
    aj = va_arg(ap, int);

    if (p[ai][aj] == color && dragon[ai][aj].matcher_status != DEAD)
      return 1;
  }

  return 0;
}


/* The function play_break_through_n() plays a sequence of moves,
 * alternating between the players and starting with color. After
 * having played through the sequence, the three last coordinate pairs
 * gives a position to be analyzed by break_through(), to see whether
 * either color has managed to enclose some stones and/or connected
 * his own stones. If any of the three last positions is empty, it's
 * assumed that the enclosure has failed, as well as the attempt to
 * connect.
 *
 * If one or more of the moves to play turns out to be illegal for
 * some reason, the rest of the sequence is played anyway, and
 * break_through() is called as if nothing special happened.
 *
 * Like break_through(), this function returns 1 if the attempt to
 * break through was succesful and 2 if it only managed to cut
 * through.
 */
   
int
play_break_through_n(int color, int num_moves, ...)
{
  va_list ap;
  int mcolor = color;
  int success = 0;
  int i;
  int played_moves = 0;
  int xi, xj;
  int yi, yj;
  int zi, zj;
  
  va_start(ap, num_moves);

  /* Do all the moves with alternating colors. */
  for (i = 0; i < num_moves; i++) {
    int ai, aj;

    ai = va_arg(ap, int);
    aj = va_arg(ap, int);

    if (ai != -1 && trymove(ai, aj, mcolor, "play_break_through_n", -1, -1))
      played_moves++;
    mcolor = OTHER_COLOR(mcolor);
  }

  /* Now do the real work. */
  xi = va_arg(ap, int);
  xj = va_arg(ap, int);
  yi = va_arg(ap, int);
  yj = va_arg(ap, int);
  zi = va_arg(ap, int);
  zj = va_arg(ap, int);
    
  /* Temporarily increase the depth values with the number of explicitly
   * placed stones.
   */
#if 0
  modify_depth_values(played_moves);
#endif
  
  if (!p[xi][xj] || !p[yi][yj] || !p[zi][zj])
    success = 1;
  else
    success = break_through(xi, xj, yi, yj, zi, zj);

#if 0
  modify_depth_values(-played_moves);
#endif
  
  /* Pop all the moves we could successfully play. */
  for (i = 0; i < played_moves; i++)
    popgo();

  va_end(ap);
  return success;
}


/* The function play_attack_defend_n() plays a sequence of moves,
 * alternating between the players and starting with color. After
 * having played through the sequence, the last coordinate pair gives
 * a target to attack or defend, depending on the value of do_attack.
 * If there is no stone present to attack or defend, it is assumed
 * that it has already been captured. If one or more of the moves to
 * play turns out to be illegal for some reason, the rest of the
 * sequence is played anyway, and attack/defense is tested as if
 * nothing special happened.
 *
 * A typical use for these functions is to set up a ladder in an
 * autohelper and see whether it works or not.
 */
   
int
play_attack_defend_n(int color, int do_attack, int num_moves, ...)
{
  va_list ap;
  int mcolor = color;
  int success = 0;
  int i;
  int played_moves = 0;
  int zi, zj;
  
  va_start(ap, num_moves);

  /* Do all the moves with alternating colors. */
  for (i = 0; i < num_moves; i++) {
    int ai, aj;

    ai = va_arg(ap, int);
    aj = va_arg(ap, int);

    if (ai != -1 && trymove(ai, aj, mcolor, "play_attack_defend_n", -1, -1))
      played_moves++;
    mcolor = OTHER_COLOR(mcolor);
  }

  /* Now do the real work. */
  zi = va_arg(ap, int);
  zj = va_arg(ap, int);

  /* Temporarily increase the depth values with the number of explicitly
   * placed stones.
   */
#if 0
  /* FIXME PRE3.0: I think that this should not be commented out.
                   Why is it?  Check all the other places also. */
  modify_depth_values(played_moves);
#endif
  
  if (do_attack) {
    if (!p[zi][zj])
      success = 1;
    else
      success = attack(zi, zj, NULL, NULL);
  } else {
    if (!p[zi][zj])
      success = 0;
    else {
      int dcode = find_defense(zi, zj, NULL, NULL);
      if (dcode == 0 && !attack(zi, zj, NULL, NULL))
	success = 1;
      else
	success = dcode;
    }
  }

#if 0
  modify_depth_values(-played_moves);
#endif
  
  /* Pop all the moves we could successfully play. */
  for (i = 0; i < played_moves; i++)
    popgo();

  va_end(ap);
  return success;
}


/* The function play_attack_defend2_n() plays a sequence of moves,
 * alternating between the players and starting with color. After
 * having played through the sequence, the two last coordinate pairs
 * give two targets to simultaneously attack or defend, depending on
 * the value of do_attack. If there is no stone present to attack or
 * defend, it is assumed that it has already been captured. If one or
 * more of the moves to play turns out to be illegal for some reason,
 * the rest of the sequence is played anyway, and attack/defense is
 * tested as if nothing special happened.
 *
 * A typical use for these functions is to set up a crosscut in an
 * autohelper and see whether at least one cutting stone can be
 * captured.
 */
   
int
play_attack_defend2_n(int color, int do_attack, int num_moves, ...)
{
  va_list ap;
  int mcolor = color;
  int success = 0;
  int i;
  int played_moves = 0;
  int yi, yj;
  int zi, zj;
  
  va_start(ap, num_moves);

  /* Do all the moves with alternating colors. */
  for (i = 0; i < num_moves; i++) {
    int ai, aj;

    ai = va_arg(ap, int);
    aj = va_arg(ap, int);

    if (ai != -1 && trymove(ai, aj, mcolor, "play_attack_defend2_n", -1, -1))
      played_moves++;
    mcolor = OTHER_COLOR(mcolor);
  }

  /* Now do the real work. */
  yi = va_arg(ap, int);
  yj = va_arg(ap, int);
  zi = va_arg(ap, int);
  zj = va_arg(ap, int);

  /* Temporarily increase the depth values with the number of explicitly
   * placed stones.
   */
#if 0
  modify_depth_values(played_moves);
#endif
  
  if (do_attack) {
    if (!p[yi][yj] || !p[zi][zj] || attack_either(yi, yj, zi, zj))
      success = 1;
  } else {
    if (p[yi][yj] && p[zi][zj] && defend_both(yi, yj, zi, zj))
      success = 1;
  }

#if 0
  modify_depth_values(-played_moves);
#endif
  
  /* Pop all the moves we could successfully play. */
  for (i = 0; i < played_moves; i++)
    popgo();

  va_end(ap);
  return success;
}


/* find_lunch(m, n, *wi, *wj, *ai, *aj) looks for a worm adjoining the
 * string at (m,n) which can be easily captured. Whether or not it can
 * be defended doesn't matter.
 *
 * Returns the location of the string in (*wi, *wj), and the location
 * of the attacking move in (*ai, *aj).
 */
	
int
find_lunch(int m, int n, int *wi, int *wj, int *ai, int *aj)
{
  int i, j, vi, vj, ki, kj;

  ASSERT(p[m][n] != 0, m, n);
  ASSERT(stackp==0, m, n);

  vi = -1;
  vj = -1;
  for (i=0; i<board_size; i++)
    for (j=0; j<board_size; j++)
      if (p[i][j]==OTHER_COLOR(p[m][n]))
	if ((   i > 0
		&& worm[i-1][j].origini == worm[m][n].origini 
		&& worm[i-1][j].originj == worm[m][n].originj)
	    || (i < board_size-1
		&& worm[i+1][j].origini == worm[m][n].origini 
		&& worm[i+1][j].originj == worm[m][n].originj)
	    || (j > 0
		&& worm[i][j-1].origini == worm[m][n].origini
		&& worm[i][j-1].originj == worm[m][n].originj)
	    || (j < board_size-1
		&& worm[i][j+1].origini == worm[m][n].origini
		&& worm[i][j+1].originj == worm[m][n].originj)
	    || (i > 0 && j > 0
		&& worm[i-1][j-1].origini == worm[m][n].origini
		&& worm[i-1][j-1].originj == worm[m][n].originj)
	    || (i > 0 && j < board_size-1
		&& worm[i-1][j+1].origini == worm[m][n].origini
		&& worm[i-1][j+1].originj == worm[m][n].originj)
	    || (i < board_size-1 && j > 0
		&& worm[i+1][j-1].origini == worm[m][n].origini
		&& worm[i+1][j-1].originj == worm[m][n].originj)
	    || (i < board_size-1 && j < board_size-1
		&& worm[i+1][j+1].origini == worm[m][n].origini
		&& worm[i+1][j+1].originj == worm[m][n].originj)
	  )
	  if (((stackp==0) && (worm[i][j].attack_code != 0)
	       && (!worm[i][j].ko))
	      || ((stackp>0) && attack(i, j, &ki, &kj))) {
	    /*
	     * If several adjacent lunches are found, we pick the 
	     * juiciest. First maximize cutstone, then minimize liberties. 
	     * We can only do this if the worm data is available, 
	     * i.e. if stackp==0.
	     */
	    if ((vi) == -1 || 
		(stackp==0
		 && (worm[i][j].cutstone > worm[vi][vj].cutstone 
		     || (worm[i][j].cutstone == worm[vi][vj].cutstone 
			 && worm[i][j].liberties < worm[vi][vj].liberties)))) {
	      vi = worm[i][j].origini;
	      vj = worm[i][j].originj;
	      if (stackp>0) {
		if (ai) *ai = ki;
		if (aj) *aj = kj;
	      }
	      else {
		if (ai) *ai=worm[i][j].attacki;
		if (aj) *aj=worm[i][j].attackj;
	      }
	    }
	  }

  if (vi != -1) {
    if (wi) *wi=vi;
    if (wj) *wj=vj;
    return 1;
  }
  else
    return 0;
}


/*
 * Modify the various tactical reading depth parameters. This is
 * typically used to avoid horizon effects. By temporarily increasing
 * the depth values when trying some move, one can avoid that an
 * irrelevant move seems effective just because the reading hits a
 * depth limit earlier than it did when reading only on relevant
 * moves.
 */

void
modify_depth_values(int n)
{
  depth              += n;
  backfill_depth     += n;
  backfill2_depth    += n;
  superstring_depth  += n;
  branch_depth       += n;
  fourlib_depth      += n;
  ko_depth           += n;
}

void
increase_depth_values(void)
{
  modify_depth_values(1);
}

void
decrease_depth_values(void)
{
  modify_depth_values(-1);
}

/* These functions allow more drastic temporary modifications of the
 * depth values. Typical use is to turn certain depth values way down
 * for reading where speed is more important than accuracy, e.g. for
 * the influence function.
 */

static int save_depth;
static int save_backfill_depth;
static int save_backfill2_depth;
static int save_superstring_depth;
static int save_branch_depth;
static int save_fourlib_depth;
static int save_ko_depth;

/* Currently this function is never called. */

void
set_temporary_depth_values(int d, int b, int f, int k, int br, int b2, int ss)
{
  save_depth             = depth;
  save_backfill_depth    = backfill_depth;
  save_backfill2_depth   = backfill2_depth;
  save_superstring_depth = superstring_depth;
  save_branch_depth      = branch_depth;
  save_fourlib_depth     = fourlib_depth;
  save_ko_depth          = ko_depth;

  depth             = d;
  backfill_depth    = b;
  backfill2_depth   = b2;
  superstring_depth = ss;
  branch_depth      = br;
  fourlib_depth     = f;
  ko_depth          = k;
}

void
restore_depth_values()
{
  depth             = save_depth;
  backfill_depth    = save_backfill_depth;
  backfill2_depth   = save_backfill2_depth;
  superstring_depth = save_superstring_depth;
  branch_depth      = save_branch_depth;
  fourlib_depth     = save_fourlib_depth;
  ko_depth          = save_ko_depth;
}


/* 
 * Test whether two dragons are the same. Used by autohelpers and elsewhere.
 */

int
same_dragon(int ai, int aj, int bi, int bj)
{
  if (ai == -1 || aj == -1 || bi == -1 || bj == -1)
    return (ai == bi && aj == bj);
  
  return (dragon[ai][aj].origini == dragon[bi][bj].origini
	  && dragon[ai][aj].originj == dragon[bi][bj].originj);
}


/*
 * Test whether the origin of the worm at (wi, wj) is (i, j).
 */

int
is_worm_origin(int wi, int wj, int i, int j)
{
  return (worm[wi][wj].origini == i
	  && worm[wi][wj].originj == j);
}


/* Play a stone at (m, n) and count the number of liberties for the
 * resulting string. This requires (m, n) to be empty.
 *
 * This function differs from approxlib() by the fact that it removes
 * captured stones before counting the liberties.
 */

int
accurate_approxlib(int m, int n, int color, int maxlib, int *libi, int *libj)
{
  int libs = 0;

  assert(p[m][n] == EMPTY);
  assert(color != EMPTY);

  /* Use tryko() since we don't care whether the move would violate
   * the ko rule.
   */
  if (tryko(m, n, color, "accurate approxlib")) {
    if (libi != NULL)
      libs = findlib(m, n, maxlib, libi, libj);
    else
      libs = countlib(m, n);
    popgo();
  }
  
  return libs;
}


/* This function will detect some blunders. If the move reduces the
 * number of liberties of an adjacent friendly string, there is a
 * danger that the move could backfire, so the function checks that no
 * friendly worm which was formerly not attackable becomes attackable,
 * and it checks that no opposing worm which was not defendable
 * becomes defendable. Only worms with worm.size>size are checked.
 *
 * For use when called from fill_liberty, this function may optionally
 * return a point of defense, which, if taken, will presumably make
 * the move at (i, j) safe on a subsequent turn.
 */

int
confirm_safety(int i, int j, int color, int size, int *di, int *dj)
{
  int libi[5], libj[5];
  int libs = accurate_approxlib(i, j, color, 5, libi, libj);
  int other = OTHER_COLOR(color);
  int issafe = 1;
  int m, n;
  
  if (di) *di = -1;
  if (dj) *dj = -1;

  TRACE("Checking safety of a %s move at %m\n", color_to_string(color), i, j);
  if (libs>4)
    return 1;

  /* Need to increase the depth values during this reading to avoid
   * horizon effects.
   */
  increase_depth_values();
  
  if ((    (i>0)
	   && (p[i-1][j] == color)
	   && (libs < worm[i-1][j].liberties))
       || ((i<board_size-1)
	   && (p[i+1][j] == color)
	   && (libs < worm[i+1][j].liberties))
       || ((j>0)
	   && (p[i][j-1] == color)
	   && (libs < worm[i][j-1].liberties))
       || ((j<board_size-1)
	   && (p[i][j+1] == color)
	   && (libs < worm[i][j+1].liberties))) {
    if (trymove(i, j, color, NULL, -1, -1)) {
      for (m=0; m<board_size; m++)
	for (n=0; n<board_size; n++)
	  if (issafe
	      && p[m][n]
	      && (worm[m][n].origini==m)
	      && (worm[m][n].originj==n)) {
	    if ((p[m][n]==color)
		&& (worm[m][n].attack_code == 0) 
		&& (worm[m][n].size > size)
		&& attack(m, n, NULL, NULL)) {
	      issafe = 0;
	      TRACE("Worm at %m becomes attackable.\n", m, n);
	    }
	    else if ((p[m][n]==other)
		     && (worm[m][n].attack_code != 0)
		     && (worm[m][n].defend_code == 0)
		     && (worm[m][n].size > size)
		     && find_defense(m, n, NULL, NULL)) {
	      issafe = 0;
	      TRACE("Worm at %m becomes defendable.\n", m, n);
	    }
	  }
      if (libs==2) {
	if (double_atari(libi[0], libj[0], other)) {
	  if (di && safe_move(libi[0], libj[0], color) == 1) {
	    *di = libi[0];
	    *dj = libj[0];
	  }
	  issafe = 0;
	  TRACE("Double threat appears at %m.\n", libi[0], libj[0]);
	}
	else if (double_atari(libi[1], libj[1], other)) {
	  if (di && safe_move(libi[1], libj[1], color) == 1) {
	    *di = libi[1];
	    *dj = libj[1];
	  }
	  issafe = 0;
	  TRACE("Double threat appears at %m.\n", libi[1], libj[1]);
	}
      }
      popgo();
    }
  }
  
  /* Reset the depth values. */
  decrease_depth_values();
  
  return issafe;
}


/* Returns true if a move by (color) fits the following shape:
 * 
 *     .
 *    X*.       (O=color)
 *    OX
 * 
 * capturing one of the two X strings. The name is a slight
 * misnomer since this includes attacks which are not necessarily
 * double ataris, though the common double atari is the most
 * important special case.
 */

int
double_atari(int m, int n, int color)
{
  int other=OTHER_COLOR(color);
  int k;

  if ((m==0) || (m==board_size-1) || (n==0) || (n==board_size-1))
    return 0;

  /* Loop over the diagonal directions. */
  for (k=4; k<8; k++) {
    int dm = deltai[k];
    int dn = deltaj[k];
    
    if ((p[m+dm][n+dn]==color)
	&& (p[m][n+dn]==other)
	&& (p[m+dm][n]==other)
	&& (p[m-dm][n]==EMPTY)
	&& (p[m][n-dn]==EMPTY)
	&& trymove(m, n, color, "double_atari", -1, -1)) {
      if (!defend_both(m, n+dn, m+dm, n)) {
	popgo();
	return 1;
      }
      popgo();
    }
  }
  
  return 0;
}
    

/* Find those worms of the given color that can never be captured,
 * even if the opponent is allowed an arbitrary number of consecutive
 * moves. The coordinates of the origins of these worms are written to
 * the wormi, wormj arrays and the number of non-capturable worms is
 * returned.
 *
 * The algorithm is to cycle through the worms until none remains or
 * no more can be captured. A worm is removed when it is found to be
 * capturable, by letting the opponent try to play on all its
 * liberties. If the attack fails, the moves are undone.
 *
 */

int
unconditional_life(int wormi[MAX_STRINGS], int wormj[MAX_STRINGS], int color)
{
  int something_captured = 1; /* To get into the first turn of the loop. */
  int moves_played = 0;
  int save_moves;
  int m, n;
  int k;
  int libi[MAXLIBS];
  int libj[MAXLIBS];
  int libs;
  int other = OTHER_COLOR(color);
  int number_of_strings;
  
  while (something_captured) {
    /* Nothing captured so far in this turn of the loop. */
    something_captured = 0;

    /* Visit all friendly strings on the board. */
    for (m=0; m<board_size; m++)
      for (n=0; n<board_size; n++) {
	if (p[m][n] != color || !is_worm_origin(m, n, m, n))
	  continue;
	
	/* Try to capture the worm at (m, n). */
	libs = findlib(m, n, MAXLIBS, libi, libj);
	save_moves = moves_played;
	for (k=0; k<libs; k++) {
	  if (trymove(libi[k], libj[k], other, "unconditional_life", m, n))
	    moves_played++;
	}

	/* Successful if already captured or a single liberty remains.
	 * Otherwise we must rewind and take back the last batch of moves.
	 */
	if (p[m][n] == EMPTY)
	  something_captured = 1;
	else if (findlib(m, n, 2, libi, libj) == 1) {
	  /* Need to use tryko as a defense against the extreme case
           * when the only opponent liberty that is not suicide is an
           * illegal ko capture, like in this 5x5 position:
	   * +-----+
	   * |.XO.O|
	   * |XXOO.|
	   * |X.XOO|
	   * |XXOO.|
	   * |.XO.O|
	   * +-----+
	   */
	  int success = tryko(libi[0], libj[0], other, "unconditional_life");
	  assert(success);
	  moves_played++;
	  something_captured++;
	}
	else
	  while (moves_played > save_moves) {
	    popgo();
	    moves_played--;
	  }
      }
  }

  /* The strings still remaining are uncapturable. */
  number_of_strings = 0;
  for (m=0; m<board_size; m++)
    for (n=0; n<board_size; n++)
      if (p[m][n] == color && is_worm_origin(m, n, m, n)) {
	wormi[number_of_strings] = m;
	wormj[number_of_strings] = n;
	number_of_strings++;
      }

  /* Take back all moves. */
  while (moves_played > 0) {
    popgo();
    moves_played--;
  }

  return number_of_strings;
}


/* Find the stones of an extended string, where the extensions are
 * through the following kinds of connections:
 *
 * 1. Solid connections (just like ordinary string).
 *
 *    OO
 *
 * 2. Diagonal connection or one space jump through an intersection
 *    where an opponent move would be suicide or self-atari.
 *
 *    ...
 *    O.O
 *    XOX
 *    X.X
 *
 * 3. Bamboo joint.
 *
 *    OO
 *    ..
 *    OO
 *
 * 4. Diagonal connection where both adjacent intersections are empty.
 *
 *    .O
 *    O.
 *
 * 5. Connection through adjacent or diagonal tactically captured stones.
 *    Connections of this type are omitted when the superstring code is
 *    called from reading.c, but included when the superstring code is
 *    called from owl.c
 */

static void superstring_recurse(int m, int n,
				char mx[MAX_BOARD][MAX_BOARD],
				int *items, int *itemi, int *itemj, 
				int type, int oi, int oj, int liberty_cap);

void
find_superstring(int m, int n, int *stones, int *stonei, int *stonej)
{
  char mx[MAX_BOARD][MAX_BOARD];

  memset(mx, 0, sizeof(mx));
  *stones = 0;
  superstring_recurse(m, n, mx, stones, stonei, stonej, 1, -1, -1, 0);
}


/* This function computes the superstring at (m, n) as described above,
 * but omitting connections of type 5. Then it constructs a list of
 * liberties of the superstring which are not already liberties of
 * (m, n).
 *
 * If liberty_cap is nonzero, only liberties of substrings of the
 * superstring which have fewer than liberty_cap liberties are
 * generated.
 */

void
find_superstring_liberties(int m, int n, 
			   int *libs, int *libi, int *libj, int liberty_cap)
{
  char mx[MAX_BOARD][MAX_BOARD];

  memset(mx, 0, sizeof(mx));
  *libs = 0;
  superstring_recurse(m, n, mx, libs, libi, libj, 0, -1, -1, liberty_cap);
}

/* This function is the same as find_superstring_liberties, but it
 * omits those liberties of the string (m,n), presumably since
 * those have already treated elsewhere.
 *
 * If liberty_cap is nonzero, only liberties of substrings of the
 * superstring which have at most liberty_cap liberties are
 * generated.
 */

void
find_proper_superstring_liberties(int m, int n, 
				  int *libs, int *libi, int *libj, 
				  int liberty_cap)
{
  char mx[MAX_BOARD][MAX_BOARD];

  memset(mx, 0, sizeof(mx));
  *libs = 0; 
  superstring_recurse(m, n, mx, libs, libi, libj, 0, m, n, liberty_cap);
}

/* analogous to chainlinks, this function finds boundary chains of the
 * superstring at (m, n), including those which are boundary chains of
 * (m, n) itself. If liberty_cap != 0, only those boundary chains with
 * <= liberty_cap liberties are reported.
 */

void
superstring_chainlinks(int m, int n, 
		       int *adj, int adji[MAXCHAIN], int adjj[MAXCHAIN],
		       int liberty_cap)
{
  char mx[MAX_BOARD][MAX_BOARD];

  memset(mx, 0, sizeof(mx));
  *adj = 0; 
  superstring_recurse(m, n, mx, adj, adji, adjj, 2, -1, -1, liberty_cap);
}


/* analogous to chainlinks, this function finds boundary chains of the
 * superstring at (m, n), omitting those which are boundary chains of
 * (m, n) itself. If liberty_cap != 0, only those boundary chains with
 * <= liberty_cap liberties are reported.
 */

void
proper_superstring_chainlinks(int m, int n, int *adj,
			      int adji[MAXCHAIN], int adjj[MAXCHAIN],
			      int liberty_cap)
{
  char mx[MAX_BOARD][MAX_BOARD];

  memset(mx, 0, sizeof(mx));
  *adj = 0; 
  superstring_recurse(m, n, mx, adj, adji, adjj, 2, m, n, liberty_cap);
}



/* Recursive function called by the superstring functions.
 * 
 * If type is 0, the list of items is a list of liberties.
 * If type is 1, the list of items is a list of stones;
 * If type is 2, the list of items is a list of origins of boundary chains.
 * It the second case, we omit connections of type 5 in
 * computing the superstring. 
 * 
 * Type 0 details: 
 * If (oi,oj) != (-1,-1), liberties of the string (oi,oj) are omitted.  
 * If liberty_cap != 0, only liberties of substrings with <= liberty_cap
 * liberties are reported.
 *
 * Type 2 details:
 * If (oi,oj) != (-1,-1), boundary chains which are adjacent to (oi, oj)
 * are omitted.
 */

static void
superstring_recurse(int m, int n, char mx[MAX_BOARD][MAX_BOARD],
		    int *items, int *itemi, int *itemj, int type, 
		    int oi, int oj, int liberty_cap)
{
  int k, l;
  int color = p[m][n];
  int other = OTHER_COLOR(color);

  ASSERT(ON_BOARD(m, n), m, n);
  
  mx[m][n] = 1;

  if (type == 0 && (*items) < MAX_LIBERTIES
      && (liberty_cap == 0 || countlib(m, n) <= liberty_cap)) {
    if (m > 0 && p[m-1][n] == EMPTY && !mx[m-1][n] 
	&& (oi == -1 || !liberty_of(m-1, n, oi, oj))) {
      mx[m-1][n] = 1;
      itemi[*items] = m-1;
      itemj[*items] = n;
      (*items)++;
    }
    if (m < board_size-1 && p[m+1][n] == EMPTY && !mx[m+1][n] 
	&& (oi == -1 || !liberty_of(m+1, n, oi, oj))) {
      mx[m+1][n] = 1;
      itemi[*items] = m+1;
      itemj[*items] = n;
      (*items)++;
    }
    if (n > 0 && p[m][n-1] == EMPTY && !mx[m][n-1] 
	&& (oi == -1 || !liberty_of(m, n-1, oi, oj))) {
      mx[m][n-1] = 1;
      itemi[*items] = m;
      itemj[*items] = n-1;
      (*items)++;
    }
    if (n < board_size-1 && p[m][n+1] == EMPTY && !mx[m][n+1] 
	&& (oi == -1 || !liberty_of(m, n+1, oi, oj))) {
      mx[m][n+1] = 1;
      itemi[*items] = m;
      itemj[*items] = n+1;
      (*items)++;
    }
  }
  else if (type == 1) {
    itemi[*items] = m;
    itemj[*items] = n;
    (*items)++;
  }
  else if (type == 2 && (*items) < MAXCHAIN) {
    int bi, bj;
    if (m > 0 && p[m-1][n] == other
	&& (oi == -1 || !neighbor_of(m-1, n, oi, oj))) {
      find_origin(m-1, n, &bi, &bj);
      if (!mx[bi][bj]) {
	mx[bi][bj] = 1;
	itemi[*items] = bi;
	itemj[*items] = bj;
	(*items)++;
      }
    }
    if (m < board_size-1 && p[m+1][n] == other
	&& (oi == -1 || !neighbor_of(m+1, n, oi, oj))) {
      find_origin(m+1, n, &bi, &bj);
      if (!mx[bi][bj]) {
	mx[bi][bj] = 1;
	itemi[*items] = bi;
	itemj[*items] = bj;
	(*items)++;
      }
    }
    if (n > 0 && p[m][n-1] == other
	&& (oi == -1 || !neighbor_of(m, n-1, oi, oj))) {
      find_origin(m, n-1, &bi, &bj);
      if (!mx[bi][bj]) {      
	mx[bi][bj] = 1;
	itemi[*items] = bi;
	itemj[*items] = bj;
	(*items)++;
      }
    }
    if (n < board_size-1 && p[m][n+1] == other
	&& (oi == -1 || !neighbor_of(m, n+1, oi, oj))) {
      find_origin(m, n+1, &bi, &bj);
      if (!mx[bi][bj]) {
	itemi[*items] = bi;
	itemj[*items] = bj;
	(*items)++;
      }
    }
  }

  for (k=0; k<4; k++) {
    /* List of relative coordinates. (m, n) is marked by *.
     *
     *  ef.
     *  gb.
     *  *ac
     *  .d.
     *
     */
    int normali = -deltaj[k];
    int normalj = deltai[k];
    
    int ai = m + deltai[k];
    int aj = n + deltaj[k];
    int bi = m + deltai[k] + normali;
    int bj = n + deltaj[k] + normalj;
    int ci = m + 2*deltai[k];
    int cj = n + 2*deltaj[k];
    int di = m + deltai[k] - normali;
    int dj = n + deltaj[k] - normalj;
    int ei = m + 2*normali;
    int ej = n + 2*normalj;
    int fi = m + deltai[k] + 2*normali;
    int fj = n + deltaj[k] + 2*normalj;
    int gi = m + normali;
    int gj = n + normalj;
    int unsafe_move;
    
    if (!ON_BOARD(ai, aj))
      continue;

    /* Case 1. */
    if (p[ai][aj] == color && !mx[ai][aj])
      superstring_recurse(ai, aj, mx, items, itemi, itemj, 
			  type, oi, oj, liberty_cap);

    /* Case 2. */
    if (p[ai][aj] == EMPTY) {
      if (type == 2)
	unsafe_move = (approxlib(ai, aj, other, 2, NULL, NULL) < 2);
      else
	unsafe_move = is_self_atari(ai, aj, other);
      
      if (unsafe_move) {
	if (ON_BOARD(bi, bj) && p[bi][bj] == color && !mx[bi][bj])
	  superstring_recurse(bi, bj, mx, items, itemi, itemj, 
			      type, oi, oj, liberty_cap);
	if (ON_BOARD(ci, cj) && p[ci][cj] == color && !mx[ci][cj])
	  superstring_recurse(ci, cj, mx, items, itemi, itemj, 
			      type, oi, oj, liberty_cap);
	if (ON_BOARD(di, dj) && p[di][dj] == color && !mx[di][dj])
	  superstring_recurse(di, dj, mx, items, itemi, itemj, 
			      type, oi, oj, liberty_cap);
      }
    }

    /* Case 3. */
    if (ON_BOARD(fi, fj) && !mx[ei][ej] && p[ai][aj] == color
	&& p[ei][ej] == color && p[fi][fj] == color
	&& p[bi][bj] == EMPTY && p[gi][gj] == EMPTY)
      superstring_recurse(ei, ej, mx, items, itemi, itemj, 
			  type, oi, oj, liberty_cap);
    /* Don't bother with f, it will already have been found when the
     * recursion returns here.
     */

    /* Case 4. */
    if (ON_BOARD(bi, bj) && !mx[bi][bj] && p[bi][bj] == color
	&& p[ai][aj] == EMPTY && p[gi][gj] == EMPTY)
      superstring_recurse(bi, bj, mx, items, itemi, itemj, 
			  type, oi, oj, liberty_cap);
    /* Case 5. */
    if (type == 1)
      for (l = 0; l < 2; l++) {
	int ui, uj;
	
	if (l == 0) {
	  /* adjacent lunch */
	  ui = ai;
	  uj = aj;
	}
	else {
	  /* diagonal lunch */
	  ui = bi;
	  uj = bj;
	}
	
	if (!ON_BOARD(ui, uj))
	  continue;
	
	if (p[ui][uj] != other)
	  continue;
	
	find_origin(ui, uj, &ui, &uj);
	
	/* Only do the reading once. */
	if (mx[ui][uj] == 1)
	  continue;
	
	mx[ui][uj] = 1;
	
	if (attack(ui, uj, NULL, NULL)
	    && !find_defense(ui, uj, NULL, NULL)) {
	  int lunch_stonei[MAX_BOARD*MAX_BOARD];
	  int lunch_stonej[MAX_BOARD*MAX_BOARD];
	  int lunch_stones = findstones(ui, uj, MAX_BOARD*MAX_BOARD,
					lunch_stonei, lunch_stonej);
	  int r, s;
	  for (r=0; r<lunch_stones; r++)
	    for (s=0; s<8; s++) {
	      int vi = lunch_stonei[r] + deltai[s];
	      int vj = lunch_stonej[r] + deltaj[s];
	      if (ON_BOARD(vi, vj) && p[vi][vj] == color && !mx[vi][vj])
		superstring_recurse(vi, vj, mx, items, itemi, itemj, 
				    type, oi, oj, liberty_cap);
	    }
	}
      }
  }
}


#define ENQUEUE(i, j) (queue_i[queue_end] = (i),\
		       queue_j[queue_end++] = (j),\
		       mx[i][j] = 1)


/* This is a generic function for walking through a number of intersections
 * and perform some action at each intersection.
 * The walk starts at (m, n) and for each adjacent intersection for which
 * predicate() returns non-zero, the action() is performed.
 *
 * Return value is number of intersection for which action() returned non-zero.
 */

static int
generic_walk(int m, int n, void *data,
	     int (*predicate)(int i, int j, void *data),
	     int (*action)(int i, int j, void *data))
{
  int  queue_start;
  int  queue_end;
  int  queue_i[MAX_BOARD * MAX_BOARD];
  int  queue_j[MAX_BOARD * MAX_BOARD];
  int  mx[MAX_BOARD][MAX_BOARD];
  int  retval;
  int  k;
  int  i, j;

  /* Initialize the search. */
  queue_start = 0;
  queue_end = 0;
  memset(mx, 0, sizeof(mx));
  retval = 0;

  ENQUEUE(m, n);
  while (queue_start != queue_end) {
    /* Dequeue (m, n). */
    m = queue_i[queue_start];
    n = queue_j[queue_start];
    queue_start++;

    if ((*action)(m, n, data))
      retval++;

    for (k = 0; k < 4; ++k) {
      i = m + deltai[k];
      j = n + deltaj[k];
      if (ON_BOARD(i, j) && (*predicate)(i, j, data))
	ENQUEUE(i, j);
    }
  }

  return 0;
}



/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
