/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 and 2000 by the Free Software Foundation.      *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/*
 * sgfutils.c
 *
 * This file connects the functionality of the engine with 
 * the SGF tree routines in sgfnode.c and sgftree.c
 */


#include <stdlib.h>

#include "sgftree.h"
#include "liberty.h"


/*
 * Reads header info from sgf structure and sets the appropriate variables.
 */

void
load_sgf_header(SGFNode *head, Gameinfo *gameinfo)
{
  int bsize;
  int handicap;
  float komi;
  
  if (sgfGetIntProperty(head, "SZ", &bsize))
    gnugo_clear_position(&gameinfo->position, bsize);
  else
    gnugo_clear_position(&gameinfo->position, 19);
  
  if (sgfGetIntProperty(head, "HA", &handicap) && handicap > 0)
    /* Handicap stones should appear as AW,AB properties in the sgf file. */
    gameinfo->handicap = handicap;

  if (sgfGetFloatProperty(head, "KM", &komi))
    gameinfo->komi = komi;
}


/*
 * Play the moves in an SGF tree. Walk the main variation, actioning
 * the properties into the playing board.
 *
 * Returns the color of the next move to be made.
 *
 * Head is a sgf tree. 
 * Untilstr is an optional string of the form either 'L12' or '120'
 * which tells it to stop playing at that move or move-number.
 * When debugging, this will probably be a particularly bad move,
 * and we want to know why.
 */

int
play_sgf_tree(SGFNode *head, Gameinfo *gameinfo, const char *untilstr)
{
  int bs, handicap;
  int next = BLACK;
  
  int untilm = -1, untiln = -1;
  int until = 9999;
  int addstone = 0;          /* handicap stone detector */
  
  SGFNode *node;
  
  /* Start by retreiving the board size. */
  if (sgfGetIntProperty(head, "SZ", &bs))
    gnugo_clear_position(&gameinfo->position, bs);
  else
    gnugo_clear_position(&gameinfo->position, 19);

  /* Now we can safely parse the until string (which depends on board size). */
  if (untilstr) {
    if (*untilstr > '0' && *untilstr <= '9') {
      until = atoi(untilstr);
      DEBUG(DEBUG_LOADSGF, "Loading until move %d\n", until);
    }
    else {
      untiln = *untilstr - 'A';
      if (*untilstr >= 'I')
	--untiln;
	  
      untilm = gameinfo->position.boardsize - atoi(untilstr+1);
      DEBUG(DEBUG_LOADSGF, "Loading until move at %d,%d (%m)\n", 
	    untilm, untiln, untilm, untiln);
    }
  }
  
  if (sgfGetIntProperty(head, "HA", &handicap) && handicap > 0) {
    gameinfo->handicap = handicap;
    next = WHITE;
  }
  
  /* Finally, we iterate over all the properties of all the
   * nodes, actioning them. We follow only the 'child' pointers,
   * as we have no interest in variations.
   *
   * The sgf routines map AB[aa][bb][cc] into AB[aa]AB[bb]AB[cc]
   */
  for (node=head; node; node=node->child) {
    SGFProperty *prop;
    int i, j;
      
    for (prop=node->props; prop; prop=prop->next) {
      DEBUG(DEBUG_LOADSGF, "%c%c[%s]\n", 
	    prop->name & 0xff, (prop->name >> 8), prop->value);
      switch(prop->name) {
      case SGFAB:
	get_moveXY(prop, &i, &j, gameinfo->position.boardsize);
	gnugo_add_stone(&gameinfo->position, i, j, BLACK);
	sgffile_put_stone(i, j, BLACK);
	addstone = 1;
	break;
	      
      case SGFAW:
	get_moveXY(prop, &i, &j, gameinfo->position.boardsize);
	gnugo_add_stone(&gameinfo->position, i, j, WHITE);
	sgffile_put_stone(i, j, WHITE);
	addstone = 1;
	break;
	      
      case SGFPL:
	/* following really should not be needed for proper sgf file */
	if (!gameinfo->move_number && !addstone) {
	  gnugo_sethand(&gameinfo->position, gameinfo->handicap, 0);
	  sgfOverwritePropertyInt(head, "HA", handicap);
	}

	if (prop->value[0] == 'w' || prop->value[0] == 'W')
	  next = WHITE;
	else
	  next = BLACK;
	break;
	      
      case SGFW:
      case SGFB:
	next = prop->name == SGFW ? WHITE : BLACK;
	/* following really should not be needed for proper sgf file */
	if (!gameinfo->move_number && !addstone) {
	  gnugo_sethand(&gameinfo->position, gameinfo->handicap, 0);
	  sgfOverwritePropertyInt(head, "HA", handicap);
	}

	gameinfo->move_number++;
	if (gameinfo->move_number == until)
	  return next;
	      
	if (get_moveXY(prop, &i, &j, gameinfo->position.boardsize)) {
	  if (i == untilm && j == untiln)
	    return next;
		  
	  gnugo_play_move(&gameinfo->position, i, j, next);
	  sgffile_move_made(i, j, next, 0);
	  next = OTHER_COLOR(next);
	}
	      
	break;
      }
    }
  }

  return next;
}


/*
 * Perform the moves and place the stones from the SGF Tree on the 
 * board.  Return whose turn it is to move (WHITE or BLACK).
 */

int
sgfPlayNode(SGFNode *node, Gameinfo *gameinfo, int to_move)
{
  int i,j;
  SGFProperty *prop;

  for (prop=node->props; prop; prop=prop->next) {
    switch(prop->name) {
    case SGFAB:
      /* A black stone. */
      get_moveXY(prop, &i, &j, gameinfo->position.boardsize);
      gnugo_add_stone(&gameinfo->position, i, j, BLACK);
      break;

    case SGFAW:
      /* A white stone. */
      get_moveXY(prop, &i, &j, gameinfo->position.boardsize);
      gnugo_add_stone(&gameinfo->position, i, j, WHITE);
      break;

    case SGFPL:
      /* Player property - who is next to move? */
      if (prop->value[0] == 'w' || prop->value[0] == 'W')
	to_move = WHITE;
      else
	to_move = BLACK;
      break;

    case SGFW:
    case SGFB:
      /* An ordinary move. */
      to_move = (prop->name == SGFW) ? WHITE : BLACK;
      get_moveXY(prop, &i, &j, gameinfo->position.boardsize);
      gameinfo->move_number++;
      gnugo_play_move(&gameinfo->position, i, j, to_move);
      to_move=OTHER_COLOR(to_move);
      break;
    }
  }

  return to_move;
}


/*
 * Play the moves in ROOT UNTIL movenumber is reached.
 */
/* FIXME PRE3.0: Move to another file since they use Position instead of just 
          handling SGF trees? */
int
sgfPlayTree(SGFNode **root, Position *position, int *until)
{
  SGFNode *node;
  int next=BLACK;
  SGFProperty *prop;
  int i,j;
  int movenumber=0;

  /* Start from the empty board. */
  gnugo_clear_position(position, position->boardsize);

  for (node=*root; node; node=node->child) {
    for (prop=node->props; prop; prop=prop->next) {
      switch(prop->name) {
      case SGFAB:
	get_moveXY(prop, &i, &j, position->boardsize);
	gnugo_add_stone(position, i, j, BLACK);
	break;

      case SGFAW:
	get_moveXY(prop, &i, &j, position->boardsize);
	gnugo_add_stone(position, i, j, WHITE);
	break;

      case SGFPL:
	if (prop->value[0] == 'w' || prop->value[0] == 'W')
	  next = WHITE;
	else
	  next = BLACK;
	break;

      case SGFW:
      case SGFB:
	next = prop->name == SGFW ? WHITE : BLACK;
	get_moveXY(prop, &i, &j, position->boardsize);
	movenumber++;
	if (movenumber == *until) {
	  *root=node->parent;
	  return next;
	}
	gnugo_play_move(position, i, j, next);
	next=OTHER_COLOR(next);
	break;
      }
    }
  }
  *until=movenumber;

  return next;
}



/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
