/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 and 2000 by the Free Software Foundation.      *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <assert.h>

#include "liberty.h"
#include "cache.h"

/* Size of array where candidate moves are stored. */
#define MAX_MOVES 50

/* FIXME PRE3.0: Test whether it pays off to check for duplicate moves.
 *        Currently it probably does not.
 */
#define ADD_CANDIDATE_MOVE(ti, tj, movei, movej, moves)\
  do {\
    int l;\
    for (l=0; l<moves; l++)\
      if (movei[l] == (ti) && movej[l] == (tj))\
	break;\
    if ((l == moves) && (moves < MAX_MOVES)) {\
      movei[moves] = ti;\
      movej[moves] = tj;\
      (moves)++;\
    }\
  } while(0) \

#define REMOVE_CANDIDATE_MOVE(ti, tj, movei, movej, moves)\
  do {\
    int k, l;\
    for (k=0; k<moves; k++) {\
      if (movei[k] == (ti) && movej[k] == (tj)) {\
        for (l=k; l<moves-1; l++) {\
	  movei[l]=movei[l+1];\
	  movej[l]=movej[l+1];\
	}\
        (moves)--;\
	break;\
      }\
    }\
  } while(0) \


/*
 * The functions in reading.c are used to read whether groups 
 * can be captured or not. See the Texinfo documentation 
 * (Reading) for more information.
 *
 * NULL POINTERS: Many functions in this file can use pointers
 * to return the locations of recommended plays. These can be
 * set NULL in which case these values are not returned.
 */

static int do_find_defense(int si, int sj, int *i, int *j, int komaster);
static int defend1(int si, int sj, int *i, int *j, int komaster);
static int restricted_defend1(int si, int sj, int *i, int *j, int komaster,
		   int forbidden_moves, 
		   int *forbidden_movei, int *forbidden_movej);
static int defend2(int si, int sj, int *i, int *j, int komaster);
static int defend3(int si, int sj, int *i, int *j, int komaster);
static int defend4(int si, int sj, int *i, int *j, int komaster);
static int special_rescue(int si, int sj, int ai, int aj, 
			  int *ti, int *tj, int komaster);
static int special_rescue2(int si, int sj, int libi[2], int libj[2], 
			   int *ti, int *tj, int komaster);
static int special_rescue3(int si, int sj, int libi[3], int libj[3],
			   int *ti, int *tj, int komaster);
static int do_attack(int si, int sj, int *i, int *j, int komaster);
static int attack1(int si, int sj, int *i, int *j, int komaster);
static int attack2(int si, int sj, int *i, int *j, int komaster);
static int restricted_attack2(int si, int sj, int *i, int *j, int komaster,
			      int forbidden_moves, 
			      int *forbidden_movei, int *forbidden_movej);
static int attack3(int si, int sj, int *i, int *j, int komaster);
static int attack4(int si, int sj, int *i, int *j, int komaster);
static int find_cap2(int si, int sj, int ai, int aj, int bi, int bj,
		     int *i, int *j, int komaster);
static int find_cap3(int si, int sj, int *i, int *j, int komaster);
static int draw_back(int si, int sj, int *ti, int *tj, int komaster);

static int special_attack2(int si, int sj, int libi[2], int libj[2], 
			   int *ti, int *tj, int komaster);
static int special_attack3(int si, int sj, int libi[2], int libj[2], 
			   int *ti, int *tj, int komaster);
static int special_attack4(int si, int sj, int libi[2], int libj[2],
			   int *ti, int *tj, int komaster);
static void propose_edge_moves(int si, int sj, int *libi, int *libj, int libs,
			       int movei[MAX_MOVES], int movej[MAX_MOVES],
			       int *moves, int color);
static void break_chain_moves(int si, int sj,
			      int movei[MAX_MOVES], int movej[MAX_MOVES],
			      int *moves);
static void break_chain2_efficient_moves(int si, int sj,
					 int movei[MAX_MOVES],
					 int movej[MAX_MOVES],
					 int *moves);
static void break_chain2_moves(int si, int sj,
			       int movei[MAX_MOVES], int movej[MAX_MOVES],
			       int *moves, int require_safe);
static int break_chain2(int si, int sj, int *i, int *j, int komaster);
static int break_chain3(int si, int sj, int *i, int *j, int komaster);
static int superstring_breakchain(int si, int sj, int *i, int *j, 
				  int liberty_cap);
static void double_atari_chain2(int si, int sj,
			       int movei[MAX_MOVES], int movej[MAX_MOVES],
			       int *moves);
static int relative_break_chain(int si, int sj, int *i, int *j, int ti, int tj,
				int komaster);
static void order_moves(int si, int sj, int num_moves, int *movei, int *movej,
			int color);
static int naive_ladder(int si, int sj, int *i, int *j);
static int naive_ladder_defense(int si, int sj, int ai, int aj,
				int bi, int bj, int color, int other);
static int naive_ladder_break_through(int si, int sj, int ai, int aj,
				      int color, int other);
static int in_list(int m, int n, int moves, int *movei, int *movej);


/* Statistics. */
static int reading_node_counter = 0;

/* ================================================================ */  
/*                          Goal functions                          */
/* ================================================================ */


/*
 * These functions define goals for the reading process.  They use 
 * the rest of the reading machinery to evaluate whether the goal
 * is fullfillable.
 *
 * The simplest goals are defined by attack() and find_defense(),
 * namely to see if it is possible to capture or defend a single
 * string.  More complex goals are defined by e.g. attack_either()
 * or defend_both().
 *
 * The functions in this section and the next are the only ones which are
 * callable from outside this file.  
 */


/* attack(si, sj, *i, *j) determines if the string at (m, n) can be 
 * captured, and if so, (*i, *j) returns the attacking move, unless
 * (*i, *j) are null pointers. Use null pointers if you are interested
 * in the result of the attack but not the attacking move itself.
 *
 * Return 1 if the attack succeeds unconditionally, 0 if it doesn't.
 * Returns 2 or 3 if the result depends on ko: 
 *   - Returns 2 if the attack succeeds provided attacker is willing to
 *     ignore any ko threat (the attacker makes the first ko capture).
 *   - Returns 3 if attack succeeds provided attacker has a ko threat
 *     which must be answered (the defender makes the first ko capture).
 */

#define MIN_NODES_TO_REPORT 10000

int
attack(int si, int sj, int *i, int *j)
{
  int nodes_when_called = reading_node_counter;
  int result = do_attack(si, sj, i, j, EMPTY);
  if (debug & DEBUG_READING_PERFORMANCE) {
    int oi, oj;
    find_origin(si, sj, &oi, &oj);
    if (reading_node_counter - nodes_when_called >= MIN_NODES_TO_REPORT) {
      gprintf("%oattack %m(%m) = %d, %d nodes ", si, sj, oi, oj, result,
	      reading_node_counter - nodes_when_called);
      dump_stack();
    }
  }
  return result;
}


/* find_defense(m, n, *i, *j) attempts to find a move that will save
 * the string at (m,n). It returns 1 if such a move is found, with
 * (*i, *j) the location of the saving move, unless (*i, *j) are
 * null pointers. It is not checked that tenuki defends, so this may 
 * give an erroneous answer if !attack(m,n).
 * 
 * Returns 2 or 3 if the result depends on ko. Returns 2 if the
 * string can be defended provided the defender is willing to ignore
 * any ko threat. Returns 3 if the defender wins by having a ko threat
 * which must be answered.
 */

int 
find_defense(int si, int sj, int *i, int *j)
{
  int nodes_when_called = reading_node_counter;
  int result = do_find_defense(si, sj, i, j, EMPTY);
  if (debug & DEBUG_READING_PERFORMANCE) {
    int oi, oj;
    find_origin(si, sj, &oi, &oj);
    if (reading_node_counter - nodes_when_called >= 10000) {
      gprintf("%odefend %m(%m) = %d, %d nodes ", si, sj, oi, oj, result,
	      reading_node_counter - nodes_when_called);
      dump_stack();
    }
  }
  return result;
}


/*
 * attack_either(ai, aj, bi, bj) returns true if there is a move which
 * guarantees that at least one of the strings (ai, aj) and (bi, bj)
 * can be captured. A typical application for this is in connection
 * patterns, where after a cut it suffices to capture one of the cutting
 * stones.
 *
 * FIXME PRE/POST3.0: The current implementation only looks for uncoordinated
 *        attacks. This is insufficient to find double ataris or 
 *        moves such as 'a' in positions like
 *
 *        XOOOOOOOX
 *        XOXXOXXOX
 *        XX..a..XX
 *        ---------
 *
 *        where neither of the threatened X stones can be captured right
 *        out. Still either can be captured by a move down to a.
 */

int
attack_either(int ai, int aj, int bi, int bj)
{
  int color = p[ai][aj];
  ASSERT(color != EMPTY , ai, aj);
  ASSERT(color == p[bi][bj], bi, bj);

  /* Start by attacking the string with the fewest liberties. On
   * average this seems to be slightly more efficient.
   */
  if (countlib(ai, aj) <= countlib(bi, bj))
    return attack(ai, aj, NULL, NULL) || attack(bi, bj, NULL, NULL);
  else
    return attack(bi, bj, NULL, NULL) || attack(ai, aj, NULL, NULL);
}


/*
 * defend_both(ai, aj, bi, bj) returns true if both the strings (ai, aj)
 * and (bi, bj) can be defended simultaneously or if there is no attack.
 * A typical application for this is in connection patterns, where
 * after a cut it's necessary to defend both cutting stones.
 *
 * FIXME PRE/POST3.0: The current implementation only looks at uncoordinated
 * defense.  A proper implementation would require some serious reading.
 */

int
defend_both(int ai, int aj, int bi, int bj)
{
  int a_threatened = 0;
  int b_threatened = 0;
  int ci, cj, di, dj;
  
  int color = p[ai][aj];
  ASSERT(color != EMPTY , ai, aj);
  ASSERT(color == p[bi][bj], bi, bj);

  if (attack(ai, aj, NULL, NULL)) {
    a_threatened = 1;
    if (!find_defense(ai, aj, &ci, &cj))
      return 0; /* (ai, aj) already lost */
  }
  
  if (attack(bi, bj, NULL, NULL)) {
    b_threatened = 1;
    if (!find_defense(bi, bj, &di, &dj))
      return 0; /* (bi, bj) already lost */
  }

  /* Neither string can be attacked or only one of them, in which case
   * we have time to save it.
   */
  if (!a_threatened || !b_threatened)
    return 1;
  
  /* If both strings are threatened we assume that one will become lost,
   * unless find_defense() happened to return the same defense point for
   * both (which e.g. may happen if they are in fact the same string).
   * This is still a bit too pessimistic, as there may be one move which
   * saves both strings. To do this right we should try each move which
   * defends either string and see if it also defends the other string.
   */

  if (ci == di && cj == dj)
    return 1; /* Both strings can be attacked but also defended by one move. */

  /* We also try each of the returned defense points and see whether
   * the other string can still be attacked. This still gives a
   * somewhat pessimistic estimation.
   */

  if (trymove(ci, cj, color, "defend_both-A", ai, aj)) {
    if (p[bi][bj] && !attack(bi, bj, NULL, NULL)) {
      popgo();
      return 1;
    }
    popgo();
  }
  
  if (trymove(di, dj, color, "defend_both-B", bi, bj)) {
    if (p[ai][aj] && !attack(ai, aj, NULL, NULL)) {
      popgo();
      return 1;
    }
    popgo();
  }
  
  /* Both strings can be attacked but we have only time to defend one. */
  return 0;
}


/*
 * break_through(ai, aj, bi, bj, ci, cj) returns 1 if a position can
 * succesfully be broken through and 2 if it can be cut. The position
 * is assumed to have the shape (the colors may be reversed)
 *
 * .O.       dbe
 * OXO       aFc
 *
 * It is X to move and try to capture at least one of a, b, and c. If
 * this succeeds, X is said to have broken through the position.
 * Otherwise X may try to cut through the position, which means
 * keeping F safe and getting a tactically safe string at either d or
 * e.
 *
 * Important notice: a, b, and c must be given in the correct order.
 *
 * FIXME POST3.0: The reading involved here can most likely be improved.
 *
 * FIXME POST3.0: We need to take ko results properly into account.
 */

static int
break_through_helper(int ai, int aj, int bi, int bj, int ci, int cj,
		     int di, int dj, int ei, int ej, int Fi, int Fj,
		     int color, int other);

int
break_through(int ai, int aj, int bi, int bj, int ci, int cj)
{
  int di, dj;
  int ei, ej;
  int Fi, Fj;
  int gi, gj;
  
  int color = p[ai][aj];
  int other = OTHER_COLOR(color);

  int success = 0;
  int success2 = 0;
  
  /* Basic sanity checking. */
  ASSERT(color != EMPTY , ai, aj);
  ASSERT(color == p[bi][bj], bi, bj);
  ASSERT(color == p[ci][cj], ci, cj);

  /* Construct the rest of the points in the pattern. */
  Fi = (ai + ci) / 2;    /* F midpoint between a and c. */
  Fj = (aj + cj) / 2;
  di = ai + bi - Fi;     /* Use diagonal relation a+b = d+F. */
  dj = aj + bj - Fj;
  ei = bi + ci - Fi;     /* Use diagonal relation b+c = e+F. */
  ej = bj + cj - Fj;

  /* More sanity checking. */
  ASSERT(p[di][dj] == EMPTY , di, dj);
  ASSERT(p[ei][ej] == EMPTY , ei, ej);

  /* F might already have been captured. (play_break_through_n() can't
   * detect this.
   */
  if (p[Fi][Fj] == EMPTY)
    return 0;
  
  ASSERT(p[Fi][Fj] == other , Fi, Fj);

  /* First X tries to play at d. */
  success = break_through_helper(ai, aj, bi, bj, ci, cj,
				 di, dj, ei, ej, Fi, Fj, color, other);
  if (success == 1)
    return 1;
  
  success2 = break_through_helper(ci, cj, bi, bj, ai, aj,
				  ei, ej, di, dj, Fi, Fj, color, other);

  if (success2 == 1)
    return 1;

  if (success2 == 2)
    success = 2;

  /* If we haven't been lucky yet, we might need to start by
   * defending F.
   *
   * FIXME PRE/POST3.0: The function would probably be considerably faster if we
   * start by checking whether F needs defense. Beware of ko potential
   * though.
   */
  success2 = 0;
  if (attack(Fi, Fj, NULL, NULL) && find_defense(Fi, Fj, &gi, &gj)) {
    if (trymove(gi, gj, other, "break_through-A", Fi, Fj)) {
      /* Now we let O defend his position by playing either d or e.
       * FIXME POST3.0: There may be other plausible moves too.
       */
      if (trymove(di, dj, color, "break_through-B", Fi, Fj)) {
	/* O connects at d, so X cuts at e. */
	if (safe_move(ei, ej, other)) {
	  success2 = 2;
	  if (!p[ci][cj] || attack(ci, cj, NULL, NULL))
	    success2 = 1;
	}
	popgo();
      }

      if (success2 > 0 && trymove(ei, ej, color, "break_through-C", Fi, Fj)) {
	/* O connects at e, so X cuts at d. */
	if (safe_move(di, dj, other)) {
	  /* success2 is already 1 or 2. */
	  if (p[ai][aj] && !attack(ai, aj, NULL, NULL))
	    success2 = 2;
	}
	else
	  success2 = 0;
	popgo();
      }
      popgo();
    }
  }
    
  if (success2 >= 1)
    return success2;

  return success;
}

/* Helper function for break_through(). Since we can symmetrically
 * start by cutting at d or e, we use the same code for both attacks,
 * simply switching positions between the two calls.
 */
static int
break_through_helper(int ai, int aj, int bi, int bj, int ci, int cj,
		     int di, int dj, int ei, int ej, int Fi, int Fj,
		     int color, int other)
{
  int success = 0;
  int gi, gj;

  if (trymove(di, dj, other, "break_through_helper-A", Fi, Fj)) {
    /* If F can be attacked we can't start in this way. */
    if (!attack(Fi, Fj, NULL, NULL)) {
      /* If d is safe too, we have at least managed to break through. */
      if (!attack(di, dj, &gi, &gj)) {
	success = 2;
	/* So now we can try to capture something. */
	if (!p[ai][aj] || !p[bi][bj] || !defend_both(ai, aj, bi, bj))
	  success = 1;
	else {
	  /* Both a and b could be defended, or didn't need to be.
	   * Let's see if a move at e is sufficient for O.
	   */
	  int attack_on_b = 0;
	  int attack_on_a = 0;

	  if (trymove(ei, ej, color, "break_through_helper-B", Fi, Fj)) {
	    if (attack(bi, bj, NULL, NULL))
	      attack_on_b = 1;
	    else if (attack(ai, aj, NULL, NULL))
	      attack_on_a = 1;
	    popgo();
	  }

	  /* Let O find a defense and play it. */
	  if (attack_on_a || attack_on_b) {
	    int hi = -1;
	    int hj = -1;
	    if (attack_on_b)
	      find_defense(ai, aj, &hi, &hj);
	    else
	      find_defense(bi, bj, &hi, &hj);
	    
	    if (hi != -1
		&& trymove(hi, hj, color, "break_through_helper-C", Fi, Fj)) {
	      /* Now we make a second cut at e, trying to capture
	       * either b or c.
	       */
	      if (trymove(ei, ej, other, "break_through_helper-D", Fi, Fj)) {
		if (!p[bi][bj] || !p[ci][cj] || !defend_both(bi, bj, ci, cj))
		  success = 1;
		popgo();
	      }
	      popgo();
	    }
	    else
	      success = 1; /* This should have been covered by
			    * defend_both(), so probably unnecessary. */
	  }
	}
      }

      /* Too bad, d could be attacked. We let O play the attack and
       * then try to make a second cut at e. But first we must test if
       * O at e is sufficient to capture d.
       */
      else {
	if (trymove(ei, ej, color, "break_through_helper-E", Fi, Fj)) {
	  if (!p[di][dj] || !find_defense(di, dj, NULL, NULL)) {
	    popgo();
	    popgo();
	    return 0;
	  }
	  popgo();
	}
	
	if (gi == ei && gj == ej) {
	  popgo();
	  return 0;
	}

	if (trymove(gi, gj, color, "break_through_helper-F", Fi, Fj)) {
	  if (trymove(ei, ej, other, "break_through_helper-G", Fi, Fj)) {
	    if (!attack(ei, ej, NULL, NULL)) {
	      success = 2;
	      if (!p[bi][bj] || !p[ci][cj] || !defend_both(bi, bj, ci, cj))
		success = 1;
	    }
	    popgo();
	  }
	  popgo();
	}
      }
	
    }
    popgo();
  }

  return success;
}

/* ================================================================ */
/*                     Global reading functions                     */
/* ================================================================ */

/* atari_atari(color, *i, *j) looks for a series of ataris on
 * strings of the other color culminating in the capture of
 * a string which is thought to be invulnerable by the reading
 * code. Such a move can be missed since it may be that each
 * string involved individually can be rescued, but nevertheless
 * one of them can be caught. The simplest example is a double
 * atari. The return value is the size of the smallest opponent
 * worm. 
 *
 * One danger with this scheme is that the first atari
 * tried might be irrelevant to the actual combination.
 * To detect this possibility, once we've found a combination,
 * we mark that first move as forbidden, then try again. If
 * no combination of the same size or larger turns up, then
 * the first move was indeed essential.
 *
 * For the purpose of the move generation, returns the
 * size of the smallest of the worms under attack.
 */

static int aa_status[MAX_BOARD][MAX_BOARD]; /* ALIVE, DEAD or CRITICAL */
static int forbidden[MAX_BOARD][MAX_BOARD];
static int do_atari_atari(int color, int *i, int *j, int ci, int cj,
			  int save_verbose);

int
atari_atari(int color, int *i, int *j, int save_verbose)
{
  int m, n;
  int fi, fj;
  int aa_val;

  /* Collect worm statuses of opponent's worms. We need to
   * know this because we only want to report unexpected
   * results. For example, we do not want to report success
   * if we find we can kill a worm which is already dead.
   */
  if (aa_depth < 2)
    return 0;
  memset(aa_status, 0, sizeof(aa_status));
  memset(forbidden, 0, sizeof(aa_status));
  for (m = 0; m < board_size; m++)
    for (n = 0; n < board_size; n++)
      if (p[m][n] == OTHER_COLOR(color)) {
	if (dragon[m][n].status == DEAD)
	  aa_status[m][n] = DEAD;
	else if (worm[m][n].attack_code != 0) {
	  if (worm[m][n].defend_code != 0)
	    aa_status[m][n] = CRITICAL;
	  else
	    aa_status[m][n] = DEAD;
	}
	else (aa_status[m][n] = ALIVE);
      }
  /* reclassify a worm with 2 liberties as INSUBSTANTIAL if capturing
   * it does not result in a live group.
   */
  for (m = 0; m < board_size; m++)
    for (n = 0; n < board_size; n++) 
      if ((p[m][n] == OTHER_COLOR(color))
	  && worm[m][n].origini == m
	  && worm[m][n].originj == n
	  && worm[m][n].liberties == 2
	  && !owl_substantial(m, n)) {
	int ti, tj;
	for (ti = 0; ti < board_size; ti++)
	  for (tj = 0; tj < board_size; tj++)
	    if ((worm[ti][tj].origini == m)
		&& (worm[ti][tj].originj == n))
	      aa_status[ti][tj] = INSUBSTANTIAL;
      }

  /* We try excluding the first atari found and see if the
   * combination still works.
   */
  aa_val = do_atari_atari(color, &fi, &fj, -1, -1, save_verbose);
  if (aa_val > 0) {
    forbidden[fi][fj] = 1;
    /* try dropping moves from the combination and see
     * if it still works 
     */
    while(do_atari_atari(color, &fi, &fj, -1, -1, save_verbose) >= aa_val) {
      forbidden[fi][fj] = 1;
    }
    /* The last do_atari_atari call fails. When do_atari_atari fails,
     * it does not change the value of (fi, fj), so these correspond
     * to a move that works and is necessary.
     */
    if (i) *i = fi;
    if (j) *j = fj;
    return aa_val;
  }
  else return 0;
}

/* Helper function for atari_atari. Here worms is the number of
 * opponent worms involved in the combination, and (ci, cj) is
 * the location of the last friendly move played. Moves marked
 * with the forbidden array are not tried. If no move is found,
 * the values of *i and *j are not changed.
 */

static int
do_atari_atari(int color, int *i, int *j, int ci, int cj, int save_verbose)
{
  int m, n;
  int k;
  
  int other = OTHER_COLOR(color);

  /* First look for strings adjacent to the last friendly move played
   * which can be unexpectedly attacked.
   */
  if (ci != -1)
    for (m = 0; m < board_size; m++)
      for (n = 0; n < board_size; n++) {
	int om, on;
	int ai, aj;

	if ((p[m][n] != other) 
	    || (aa_status[m][n] != ALIVE)
	    || (!neighbor_of(ci, cj, m, n)))
	  continue;
	
	find_origin(m, n, &om, &on);
	if ((m != om) || (n != on))
	  continue;
	if (attack(m, n, &ai, &aj)) {
	  if (save_verbose) {
	    gprintf("%oThe worm %m can be attacked at %m after ", m, n,
		    ai, aj);
	    dump_stack();
	  }	  
	  if (i) *i = ai;
	  if (j) *j = ai;
	  return countstones(m, n);
	}
      }

  if (stackp > AA_DEPTH)
    return 0;

  /* Next look for a string, not insubstantial, having exactly two
   * liberties and no boundary stones in atari. Atari, fill, then try
   * again. If that doesn't work, try the other atari.
   */
  for (m = 0; m < board_size; m++)
    for (n = 0; n < board_size; n++) {
      int om, on;
      int libi[2], libj[2];
#if 0
      int adj, adji[MAXCHAIN], adjj[MAXCHAIN];
#endif

      if (p[m][n] != other) 
	continue;
      find_origin(m, n, &om, &on);
      if ((m != om) || (n != on))
	continue;
      if (aa_status[m][n] == INSUBSTANTIAL)
	continue;
      if (findlib(m, n, 3, libi, libj) != 2)
	continue;
#if 0
      chainlinks2(m, n, &adj, adji, adjj, 1);
      if (adj > 0)
	continue;
#endif
      for (k = 0; k < 2; k++) {
	int ai = libi[k];
	int aj = libj[k];
	int bi, bj;
	if ((accurate_approxlib(ai, aj, color, 2, NULL, NULL) > 1) 
	    && !forbidden[ai][aj]) {
	  if (trymove(ai, aj, color, "do_atari_atari-A", m, n)) {
	    /* try to defend the stone (m,n) which is in atari */
	    if (find_defense(m, n, &bi, &bj)
		&& trymove(bi, bj, other, "do_atari_atari-B", m, n)) {
	      int aa_val;
	      /* These moves may have been irrelevant for later
               * reading, so in order to avoid horizon problems, we
               * need to temporarily increase the depth values. */
	      increase_depth_values();
	      increase_depth_values();
	      aa_val = do_atari_atari(color, NULL, NULL, ai, aj,
				      save_verbose);
	      decrease_depth_values();
	      decrease_depth_values();
	      if (aa_val) {
		if (i) *i = ai;
		if (j) *j = aj;
		popgo();
		popgo();
		return min(aa_val, countstones(m,n));
	      }
	      popgo();
	    }
	    popgo();
	  }
	}
      }
    }
  return 0;
}
  

/* ================================================================ */  
/*                       Defensive functions                        */
/* ================================================================ */


/* Like find_defense, but takes the komaster argument. If the
 * opponent is komaster, reading functions will not try
 * to take ko.
 */

static int
do_find_defense(int si, int sj, int *i, int *j, int komaster)
{
  int xi, xj;
  int dcode = 0;
  int liberties;
  int found_read_result;
  Read_result *read_result;
  
  SETUP_TRACE_INFO("find_defense", si, sj);
  
  RTRACE("Can we rescue %m?\n", si, sj);

  /* We first check if the number of liberties is larger than four. In
   * that case we don't cache the result and to avoid needlessly
   * storing the position in the hash table, we must do this test
   * before we look for cached results.
   */
  liberties = countlib(si, sj);
  
  if (liberties > 4
      || (liberties == 4 && stackp > depth)
      || (liberties == 3 && stackp > depth)) {
    /* No need to cache the result in these cases. */
    SGFTRACE(-1, -1, 1, "too many liberties or stackp > depth");
    if (i) *i = -1;
    if (j) *j = -1;
    return 1;
  }

  if ((stackp <= depth) && (hashflags & HASH_FIND_DEFENSE)) {
    found_read_result = get_read_result(FIND_DEFENSE, &si, &sj, &read_result);
    if (found_read_result) {
      TRACE_CACHED_RESULT(*read_result);
      if (rr_get_result(*read_result) != 0) {
	if (i) *i = rr_get_result_i(*read_result);
	if (j) *j = rr_get_result_j(*read_result);
      }

      SGFTRACE(rr_get_result_i(*read_result), rr_get_result_j(*read_result),
	       rr_get_result(*read_result), "cached");
      return rr_get_result(*read_result);
    }

    /* This data should always be recorded. */
    if (read_result) {
      rr_set_routine_i_j_stackp(*read_result, FIND_DEFENSE, si, sj, stackp);
    }
  } else
    read_result = NULL;

  if (liberties == 1)
    dcode = defend1(si, sj, &xi, &xj, komaster);
  else if (liberties == 2)
    dcode = defend2(si, sj, &xi, &xj, komaster);
  else if (liberties == 3)
    dcode = defend3(si, sj, &xi, &xj, komaster);
  else if (liberties == 4)
    dcode = defend4(si, sj, &xi, &xj, komaster);

  if (dcode) {
    RTRACE("saving move for %m found at %m!\n", si, sj, xi, xj);
    READ_RETURN(read_result, i, j, xi, xj, dcode);
  }
    
  READ_RETURN0(read_result);
}


/* If si, sj points to a string with exactly one liberty, defend1 
 * determines whether it can be saved by extending or capturing
 * a boundary chain having one liberty. The function returns 1 if the string
 * can be saved, otherwise 0. It returns 2 or 3 if it can be saved, conditioned
 * on ko. Returns 2 if it can be saved provided (color) is willing to
 * ignore any ko threat. Returns 3 if it can be saved if (color) has
 * a ko threat which must be answered.
 *
 * The pair defend1-attack2 call each other recursively to
 * read situations such as ladders. They read all ladders to the end.
 * If the reading ply (stackp) is deeper than the deep-reading cutoff
 * parameter depth, whose default value DEPTH is defined in gnugo.h, then a
 * string is assumed alive if it can get 3 liberties. When
 * fourlib_depth < stackp < depth, a string is considered alive if it can get
 * four liberties. When stackp < fourlib_depth, it is considered alive
 * if it can get 5 liberties.
 *
 */

static int
defend1(int si, int sj, int *i, int *j, int komaster)
{
  int color, other;
  int xi, xj;
  int libi, libj;
  int movei[MAX_MOVES], movej[MAX_MOVES];
  int moves = 0;
  int savei = -1, savej = -1;
  int savecode = 0;
  int liberties;
  int k;
  int found_read_result;
  Read_result *read_result;

  SETUP_TRACE_INFO("defend1", si, sj);
  reading_node_counter++;
  
  assert(p[si][sj] != EMPTY);
  ASSERT(countlib(si, sj) == 1, si, sj);

  RTRACE("try to escape atari on %m.\n",  si, sj);

  color = p[si][sj];
  other = OTHER_COLOR(color);

  if ((stackp <= depth) && (hashflags & HASH_DEFEND1)) {
  
    found_read_result = get_read_result(DEFEND1, &si, &sj, &read_result);
    if (found_read_result) {
      TRACE_CACHED_RESULT(*read_result);
      if (rr_get_result(*read_result) != 0) {
	if (i) *i = rr_get_result_i(*read_result);
	if (j) *j = rr_get_result_j(*read_result);
      }

      SGFTRACE(rr_get_result_i(*read_result), rr_get_result_j(*read_result),
	       rr_get_result(*read_result), "cached");
      return rr_get_result(*read_result);
    }

    /* This data should always be recorded. */
    if (read_result) {
      rr_set_routine_i_j_stackp(*read_result, DEFEND1, si, sj, stackp);
    }
  } else
    read_result = NULL;

  /* (libi, libj) will be the liberty of the string. */
  liberties = findlib(si, sj, 1, &libi, &libj);
  ASSERT(liberties == 1, si, sj);

  /* Collect moves to try in the first batch.
   * 1. First order liberty.
   * 2. Chain breaking moves.
   */
  movei[0] = libi;
  movej[0] = libj;
  moves = 1;
  
  break_chain_moves(si, sj, movei, movej, &moves);

  order_moves(si, sj, moves, movei, movej, color);

  for (k=0; k<moves; k++) {
    xi = movei[k];
    xj = movej[k];
    
    if (trymove(xi, xj, color, "defend1-A", si, sj)) {
      int acode = do_attack(si, sj, NULL, NULL, komaster);
      popgo();
      if (acode == 0) {
	SGFTRACE(xi, xj, 1, "defense effective - A");
	READ_RETURN(read_result, i, j, xi, xj, 1);
      }
      /* if the move works with ko we save it, then look for something
       * better.
       */
      UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, xi, xj);
    }
    else {
      if (stackp <= ko_depth
	  && savecode == 0 
	  && is_ko(xi, xj, color)
	  && komaster != other
	  && tryko(xi, xj, color, "defend1-B")) {
	if (do_attack(si, sj, NULL, NULL, color) != 1) {
	  savei = xi;
	  savej = xj;
	  savecode = 3;
	}
	popgo();
      }
    }
  }

  /* If the string is a single stone and a capture would give a ko,
   * try to defend it with ko by backfilling.
   */
  if (stackp <= backfill_depth
      && countstones(si, sj) == 1
      && is_ko(libi, libj, other)) {
    int lib2i[6];
    int lib2j[6];
    liberties = approxlib(libi, libj, color, 6, lib2i, lib2j);
    if (liberties <= 5) {
      for (k=0; k<liberties; k++) {
	int ai = lib2i[k];
	int aj = lib2j[k];
	if ((liberties == 1 || !is_self_atari(ai, aj, other))
	    && trymove(ai, aj, color, "attack1-C", si, sj)) {
	  int acode = do_attack(si, sj, NULL, NULL, komaster);
	  if (acode == 0) {
	    popgo();
	    SGFTRACE(ai, aj, 1, "backfilling");
	    READ_RETURN(read_result, i, j, ai, aj, 1);
	  }
	  UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, ai, aj);
	  popgo();
	}
      }
    }
  }
  
  if (savecode != 0) {
    SGFTRACE(savei, savej, savecode, "saved move");
    READ_RETURN(read_result, i, j, savei, savej, savecode);
  }
  
  SGFTRACE(-1, -1, 0, NULL);
  READ_RETURN0(read_result);
}



/* If si, sj points to a group with two liberties, defend2 determines
 * whether the group can be saved by extending, or by capturing part of
 * its surrounding chain. A group is considered safe if either part of
 * the surrounding chain may be captured, or if it can get 3
 * liberties. It is presumed that the opponent could kill if tenuki.
 * If both extensions work, it prefers the one which maximizes 
 * liberties.
 *
 * i and j return the move to save the stones. Can be NULL if caller
 * is not interested in the details.
*/

static int 
defend2(int si, int sj, int *i, int *j, int komaster)
{
  int color, other;
  int xi, xj;
  int liberties;
  int libi[2], libj[2];
  int liberties2;
  int lib2i[6], lib2j[6];
  int movei[MAX_MOVES], movej[MAX_MOVES];
  int moves = 0;
  int savei = -1, savej = -1;
  int savecode = 0;
  int bc;
  int k;
  int r;
  int s;
  int found_read_result;
  Read_result *read_result;

  SETUP_TRACE_INFO("defend2", si, sj);
  reading_node_counter++;
  
  RTRACE("trying to rescue %m\n",  si, sj);
  color = p[si][sj];
  other = OTHER_COLOR(color);

  assert(p[si][sj]!=EMPTY);
  assert(countlib(si, sj) == 2);

  if ((stackp <= depth) && (hashflags & HASH_DEFEND2)) {
  
    found_read_result = get_read_result(DEFEND2, &si, &sj, &read_result);
    if (found_read_result) {
      TRACE_CACHED_RESULT(*read_result);
      if (rr_get_result(*read_result) != 0) {
	if (i) *i = rr_get_result_i(*read_result);
	if (j) *j = rr_get_result_j(*read_result);
      }

      SGFTRACE(rr_get_result_i(*read_result), rr_get_result_j(*read_result),
	       rr_get_result(*read_result), "cached");
      return rr_get_result(*read_result);
    }

    /* This data should always be recorded. */
    if (read_result) {
      rr_set_routine_i_j_stackp(*read_result, DEFEND2, si, sj, stackp);
    }
  } else
    read_result = NULL;

  liberties = findlib(si, sj, 2, libi, libj);
  ASSERT(liberties == 2, si, sj);

  /* Collect moves to try in the first batch.
   * 1. First order liberties.
   * 2. Chain breaking moves.
   * 3. Second order liberties moving up from first line to second.
   */
  for (k = 0; k < liberties; k++) {
    movei[k] = libi[k];
    movej[k] = libj[k];
  }
  moves = liberties;
  
  break_chain_moves(si, sj, movei, movej, &moves);
  break_chain2_efficient_moves(si, sj, movei, movej, &moves);
  propose_edge_moves(si, sj, libi, libj, liberties, movei, movej, &moves, 
		     color);

  order_moves(si, sj, moves, movei, movej, color);

  for (k = 0; k < moves; k++) {
    xi = movei[k];
    xj = movej[k];
    
    if (trymove(xi, xj, color, "defend2-A", si, sj)) {
      int acode = do_attack(si, sj, NULL, NULL, komaster);
      popgo();
      if (acode == 0) {
	SGFTRACE(xi, xj, 1, "defense effective - A");
	READ_RETURN(read_result, i, j, xi, xj, 1);
      }
      /* if the move works with ko we save it, then look for something
       * better.
       */
      UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, xi, xj);
    }
    else {
      if (stackp <= ko_depth
	  && savecode == 0 
	  && is_ko(xi, xj, color)
	  && komaster != other
	  && tryko(xi, xj, color, "defend2-B")) {
	if (do_attack(si, sj, NULL, NULL, color) != 1) {
	  savei = xi;
	  savej = xj;
	  savecode = 3;
	}
	popgo();
      }
    }
  }

  /* Look for backfilling moves. */
  for (k = 0; k < liberties; k++) {
    if (is_self_atari(libi[k], libj[k], other)) {
      liberties2 = approxlib(libi[k], libj[k], color, 6, lib2i, lib2j);
      for (r=0; r<liberties2; r++) {
	xi = lib2i[r];
	xj = lib2j[r];
	/* Don't reconsider previously tested moves. */
	for (s = 0; s < moves; s++)
	  if (xi == movei[s] && xj == movej[s])
	    break;
	if (s < moves)
	  continue;

	if (trymove(xi, xj, color, "defend2-C", si, sj)) {
	  int acode = do_attack(si, sj, NULL, NULL, komaster);
	  popgo();
	  if (acode == 0) {
	    SGFTRACE(xi, xj, 1, "backfill effective");
	    READ_RETURN(read_result, i, j, xi, xj, 1);
	  }
	  UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, xi, xj);
	}
      }
    }
    else if (approxlib(libi[k], libj[k], other, 3, lib2i, lib2j) == 2) {
      for (r = 0; r < 2; r++) {
	xi = lib2i[r];
	xj = lib2j[r];
	/* Don't reconsider previously tested moves. */
	for (s = 0; s < moves; s++)
	  if (xi == movei[s] && xj == movej[s])
	    break;
	if (s < moves)
	  continue;

	if (!is_self_atari(xi, xj, color)
	    && trymove(xi, xj, color, "defend2-D", si, sj)) {
	  int acode = do_attack(si, sj, NULL, NULL, komaster);
	  popgo();
	  if (acode == 0) {
	    SGFTRACE(xi, xj, 1, "backfill effective");
	    READ_RETURN(read_result, i, j, xi, xj, 1);
	  }
	  UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, xi, xj);
	}
      }      
    }
  }

  if (stackp <= depth) {
    for (k = 0; k < liberties; ++k) {
      int dcode = special_rescue(si, sj, libi[k], libj[k], &xi, &xj, komaster);
      if (dcode == 1) {
	SGFTRACE(xi, xj, 1, "special rescue");
	READ_RETURN(read_result, i, j, xi, xj, 1);
      }
      UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, dcode, xi, xj);
    }
  }
  
  if (stackp <= backfill_depth) {
    int dcode = special_rescue2(si, sj, libi, libj, &xi, &xj, komaster);
    if (dcode == 1) {
      SGFTRACE(xi, xj, 1, "special rescue2");
      READ_RETURN(read_result, i, j, xi, xj, 1);
    }
    UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, dcode, xi, xj);
  }
  
  if (level >= 10 && stackp <= superstring_depth) {
    int dcode = superstring_breakchain(si, sj, &xi, &xj, 4);
    if (dcode == 1) {
      SGFTRACE(xi, xj, 1, "superstring_breakchain");
      READ_RETURN(read_result, i, j, xi, xj, 1);
    }
    UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, dcode, xi, xj);
  }

  /* If nothing else works, we try playing a liberty of the
   * super_string.
   */
  if (level >= 10 && stackp <= superstring_depth) {
    int libs;
    int libi[MAX_LIBERTIES + 4], libj[MAX_LIBERTIES + 4];

    find_superstring_liberties(si, sj, &libs, libi, libj, 3);
    for (k = 0; k < libs; k++) {
      int ai = libi[k];
      int aj = libj[k];
	
      if (liberty_of(ai, aj, si, sj))
	continue;
      if (trymove(ai, aj, color, "defend2-E", si, sj)) {
	int acode = do_attack(si, sj, NULL, NULL, komaster);
	popgo();
	if (acode == 0) {
	  SGFTRACE(ai, aj, 1, "superstring liberty");
	  READ_RETURN(read_result, i, j, ai, aj, 1);
	}
	UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, ai, aj);
      }
    }
    
    /* Now we are truly desperate. Try playing second order liberties of
     * the superstring.
     */
    for (k = 0; k < libs; k++) {
      int ai = libi[k];
      int aj = libj[k];
      int dcode;
	
      if (liberty_of(ai, aj, si, sj))
	continue;
      
      dcode = special_rescue(si, sj, ai, aj, &xi, &xj, komaster);
      if (dcode == 1) {
	SGFTRACE(xi, xj, 1, "special rescue");
	READ_RETURN(read_result, i, j, xi, xj, 1);
      }
      UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, dcode,
					xi, xj);
    }
  }

  /* We place the more speculative moves trying to break chain links
   * with 2 or 3 liberties last, because these moves often are not
   * really relevant.
   */
  
  bc = break_chain2(si, sj, &xi, &xj, komaster);
  if (bc == 1) {
    SGFTRACE(xi, xj, bc, "break chain2");
    READ_RETURN(read_result, i, j, xi, xj, bc);
  }
  UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, bc, xi, xj);
  
  if (stackp <= backfill2_depth) {
    bc = break_chain3(si, sj, &xi, &xj, komaster);
    if (bc == 1) {
      SGFTRACE(xi, xj, bc, "break chain3");
      READ_RETURN(read_result, i, j, xi, xj, bc);
    }
    UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, bc, xi, xj);
  }
  
  if (savecode != 0) {
    SGFTRACE(savei, savej, savecode, "saved move");
    READ_RETURN(read_result, i, j, savei, savej, savecode);
  }

  RTRACE("failed to find rescuing move.\n");
  SGFTRACE(-1, -1, 0, NULL);
  READ_RETURN0(read_result);
}


/* defend3(si, sj, *i, *j) attempts to find a move rescuing the 
 * string at (si, sj) with 3 liberties.  If such a move can be found,
 * it returns true, and if the pointers i, j are not NULL, 
 * then it returns the saving move in (*i, *j).
 */

static int 
defend3(int si, int sj, int *i, int *j, int komaster)
{
  int color, other;
  int xi, xj;
  int liberties;
  int libi[3], libj[3];
#if 0
  int liberties2;
  int lib2i[6], lib2j[6];
#endif
  int movei[MAX_MOVES], movej[MAX_MOVES];
  int moves = 0;
  int savei = -1, savej = -1;
  int savecode = 0;
  int bc;
  int k;
#if 0
  int r;
  int s;
#endif
  int found_read_result;
  Read_result *read_result;

  SETUP_TRACE_INFO("defend3", si, sj);
  reading_node_counter++;

  RTRACE("trying to rescue %m\n",  si, sj);
  color = p[si][sj];
  other = OTHER_COLOR(color);

  assert(p[si][sj] != EMPTY);
  assert(countlib(si, sj) == 3);

  if ((stackp <= depth) && (hashflags & HASH_DEFEND3)) {
    found_read_result = get_read_result(DEFEND3, &si, &sj, &read_result);
    if (found_read_result) {
      TRACE_CACHED_RESULT(*read_result);
      if (rr_get_result(*read_result) != 0) {
	if (i) *i = rr_get_result_i(*read_result);
	if (j) *j = rr_get_result_j(*read_result);
      }

      SGFTRACE(rr_get_result_i(*read_result), rr_get_result_j(*read_result),
	       rr_get_result(*read_result), "cached");
      return rr_get_result(*read_result);
    }

    /* This data should always be recorded. */
    if (read_result) {
      rr_set_routine_i_j_stackp(*read_result, DEFEND3, si, sj, stackp);
    }
  } else
    read_result = NULL;

  liberties = findlib(si, sj, 3, libi, libj);
  ASSERT(liberties == 3, si, sj);

  /* Collect moves to try in the first batch.
   * 1. First order liberties.
   * 2. Chain breaking moves.
   * 3. Second order liberties moving up from first line to second.
   */
  for (k=0; k<liberties; k++) {
    movei[k] = libi[k];
    movej[k] = libj[k];
  }
  moves = liberties;
  
  break_chain_moves(si, sj, movei, movej, &moves);
  break_chain2_efficient_moves(si, sj, movei, movej, &moves);
  propose_edge_moves(si, sj, libi, libj, liberties, movei, movej, &moves,
		     color);

  order_moves(si, sj, moves, movei, movej, color);

  for (k=0; k<moves; k++) {
    if (stackp >= branch_depth && k > 0)
      break;
    
    xi = movei[k];
    xj = movej[k];
    
    if (trymove(xi, xj, color, "defend3-A", si, sj)) {
      int acode = do_attack(si, sj, NULL, NULL, komaster);
      popgo();
      if (acode == 0) {
	SGFTRACE(xi, xj, 1, "defense effective - A");
	READ_RETURN(read_result, i, j, xi, xj, 1);
      }
      /* if the move works with ko we save it, then look for something
       * better.
       */
      UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, xi, xj);
    }
    else {
      if (stackp <= ko_depth
	  && savecode == 0 
	  && is_ko(xi, xj, color)
	  && komaster != other
	  && tryko(xi, xj, color, "defend3-B")) {
	if (do_attack(si, sj, NULL, NULL, color) != 1) {
	  savei = xi;
	  savej = xj;
	  savecode = 3;
	}
	popgo();
      }
    }
  }

#if 0
  if (stackp <= backfill_depth) {
    bc = break_chain2(si, sj, &xi, &xj, komaster);
    if (bc == 1) {
      SGFTRACE(xi, xj, bc, "break chain2");
      READ_RETURN(read_result, i, j, xi, xj, bc);
    }
    UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, bc, xi, xj);
  }
  
  if (stackp <= backfill2_depth) {
    bc = break_chain3(si, sj, &xi, &xj, komaster);
    if (bc == 1) {
      SGFTRACE(xi, xj, bc, "break chain3");
      READ_RETURN(read_result, i, j, xi, xj, bc);
    }
    UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, bc, xi, xj);
  }
#endif
  
  /* This looks a little too expensive. */
#if 0
  /* Look for backfilling moves. */
  if (stackp <= backfill_depth) {
    for (k=0; k<liberties; k++) {
      if (is_self_atari(libi[k], libj[k], other)) {
	liberties2 = approxlib(libi[k], libj[k], color, 6, lib2i, lib2j);
	for (r=0; r<liberties2; r++) {
	  xi = lib2i[r];
	  xj = lib2j[r];
	  /* Don't reconsider previously tested moves. */
	  for (s=0; s<moves; s++)
	    if (xi == movei[s] && xj == movej[s])
	      break;
	  if (s < moves)
	    continue;
	  
	  if (trymove(xi, xj, color, "defend2-C", si, sj)) {
	    int acode = do_attack(si, sj, NULL, NULL, komaster);
	    popgo();
	    if (acode == 0) {
	      SGFTRACE(xi, xj, 1, "backfill effective");
	      READ_RETURN(read_result, i, j, xi, xj, 1);
	    }
	    UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, xi, xj);
	  }
	}
      }
      else {
	liberties2 = approxlib(libi[k], libj[k], other, 3, lib2i, lib2j);
	if (liberties2 <= 3) {
	  for (r=0; r<liberties2; r++) {
	    xi = lib2i[r];
	    xj = lib2j[r];
	    /* Don't reconsider previously tested moves. */
	    for (s=0; s<moves; s++)
	      if (xi == movei[s] && xj == movej[s])
		break;
	    if (s < moves)
	      continue;
	    
	    if (!is_self_atari(xi, xj, color)
		&& trymove(xi, xj, color, "defend2-D", si, sj)) {
	      int acode = do_attack(si, sj, NULL, NULL, komaster);
	      popgo();
	      if (acode == 0) {
		SGFTRACE(xi, xj, 1, "backfill effective");
		READ_RETURN(read_result, i, j, xi, xj, 1);
	      }
	      UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, xi, xj);
	    }
	  }
	}
      }
    }
  }
#endif
  
  /* If nothing else works, try to defend with second order liberties. */
  if (stackp <= depth) {
    for (k = 0; k < liberties; ++k) {
      int dcode = special_rescue(si, sj, libi[k], libj[k], &xi, &xj, komaster);
      if (dcode == 1) {
	SGFTRACE(xi, xj, 1, "special rescue");
	READ_RETURN(read_result, i, j, xi, xj, 1);
      }
      UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, dcode, xi, xj);
    }
  }

  if (stackp <= backfill_depth) {
    int dcode = special_rescue3(si, sj, libi, libj, &xi, &xj, komaster);
    if (dcode == 1) {
      SGFTRACE(xi, xj, 1, "special rescue2");
      READ_RETURN(read_result, i, j, xi, xj, 1);
    }
    UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, dcode, xi, xj);
  }
    
  if (level >= 10 && stackp <= backfill2_depth) {
    int dcode = superstring_breakchain(si, sj, &xi, &xj, 4);
    if (dcode == 1) {
      SGFTRACE(xi, xj, 1, "superstring_breakchain");
      READ_RETURN(read_result, i, j, xi, xj, 1);
    }
    UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, dcode, xi, xj);
  }

  /* If nothing else works, we try playing a liberty of the
   * super_string.
   */
  if (level >= 10 && stackp <= backfill2_depth) {
    int libs;
    int libi[MAX_LIBERTIES + 4], libj[MAX_LIBERTIES + 4];

    find_superstring_liberties(si, sj, &libs, libi, libj, 3);
    for (k = 0; k < libs; k++) {
      int ai = libi[k];
      int aj = libj[k];
	
      if (liberty_of(ai, aj, si, sj))
	continue;
      if (trymove(ai, aj, color, "defend3-C", si, sj)) {
	int acode = do_attack(si, sj, NULL, NULL, komaster);
	popgo();
	if (acode == 0) {
	  SGFTRACE(ai, aj, 1, "superstring liberty");
	  READ_RETURN(read_result, i, j, ai, aj, 1);
	}
	UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, ai, aj);
      }
    }

    /* Now we are truly desperate. Try playing second order liberties of
     * the superstring.
     */
    for (k = 0; k < libs; k++) {
      int ai = libi[k];
      int aj = libj[k];
      int dcode;
	
      if (liberty_of(ai, aj, si, sj))
	continue;
      
      dcode = special_rescue(si, sj, ai, aj, &xi, &xj, komaster);
      if (dcode == 1) {
	SGFTRACE(xi, xj, 1, "special rescue");
	READ_RETURN(read_result, i, j, xi, xj, 1);
      }
      UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, dcode,
					xi, xj);
    }
  }

  /* We place the more speculative moves trying to break chain links
   * with 2 or 3 liberties last, because these moves often are not
   * really relevant.
   */
#if 1
  if (stackp <= backfill2_depth) {
    bc = break_chain2(si, sj, &xi, &xj, komaster);
    if (bc == 1) {
      SGFTRACE(xi, xj, bc, "break chain2");
      READ_RETURN(read_result, i, j, xi, xj, bc);
    }
    UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, bc, xi, xj);
  }
  
  if (stackp <= backfill2_depth) {
    bc = break_chain3(si, sj, &xi, &xj, komaster);
    if (bc == 1) {
      SGFTRACE(xi, xj, bc, "break chain3");
      READ_RETURN(read_result, i, j, xi, xj, bc);
    }
    UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, bc, xi, xj);
  }
#endif
  
  if (savecode != 0) {
    SGFTRACE(savei, savej, savecode, "saved move");
    READ_RETURN(read_result, i, j, savei, savej, savecode);
  }

  RTRACE("failed to find rescuing move.\n");
  SGFTRACE(-1, -1, 0, NULL);
  READ_RETURN0(read_result);
}


/* defend4(si, sj, *i, *j) attempts to find a move rescuing the 
 * string at (si, sj) with 4 liberties.  If such a move can be found,
 * it returns true, and if the pointers i, j are not NULL, 
 * then it returns the saving move in (*i, *j).
 */

static int 
defend4(int si, int sj, int *i, int *j, int komaster)
{
  int color, other;
  int xi, xj;
  int liberties;
  int libi[4], libj[4];
  int movei[MAX_MOVES], movej[MAX_MOVES];
  int moves = 0;
  int savei = -1, savej = -1;
  int savecode = 0;
  int k;
  int found_read_result;
  Read_result *read_result;
  
  SETUP_TRACE_INFO("defend4", si, sj);
  reading_node_counter++;

  RTRACE("trying to rescue %m\n",  si, sj);
  color = p[si][sj];
  other = OTHER_COLOR(color);

  assert(p[si][sj] != EMPTY);
  assert(countlib(si, sj) == 4);

  if ((stackp <= depth) && (hashflags & HASH_DEFEND4)) {
    found_read_result = get_read_result(DEFEND4, &si, &sj, &read_result);
    if (found_read_result) {
      TRACE_CACHED_RESULT(*read_result);
      if (rr_get_result(*read_result) != 0) {
	if (i) *i = rr_get_result_i(*read_result);
	if (j) *j = rr_get_result_j(*read_result);
      }

      SGFTRACE(rr_get_result_i(*read_result), rr_get_result_j(*read_result),
	       rr_get_result(*read_result), "cached");
      return rr_get_result(*read_result);
    }

    /* This data should always be recorded. */
    if (read_result) {
      rr_set_routine_i_j_stackp(*read_result, DEFEND4, si, sj, stackp);
    }
  } else
    read_result = NULL;

  liberties = findlib(si, sj, 4, libi, libj);
  ASSERT(liberties == 4, si, sj);

  /* Collect moves to try in the first batch.
   * 1. First order liberties.
   * 2. Chain breaking moves.
   */
  for (k=0; k<liberties; k++) {
    movei[k] = libi[k];
    movej[k] = libj[k];
  }
  moves = liberties;
  
  break_chain_moves(si, sj, movei, movej, &moves);
  break_chain2_efficient_moves(si, sj, movei, movej, &moves);

  order_moves(si, sj, moves, movei, movej, color);

  for (k=0; k<moves; k++) {
    if (stackp >= branch_depth && k > 0)
      break;
    
    xi = movei[k];
    xj = movej[k];
    
    if (trymove(xi, xj, color, "defend4-A", si, sj)) {
      int acode = do_attack(si, sj, NULL, NULL, komaster);
      popgo();
      if (acode == 0) {
	SGFTRACE(xi, xj, 1, "defense effective - A");
	READ_RETURN(read_result, i, j, xi, xj, 1);
      }
      /* if the move works with ko we save it, then look for something
       * better.
       */
      UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, xi, xj);
    }
    else {
      if (stackp <= ko_depth
	  && savecode == 0 
	  && is_ko(xi, xj, color)
	  && komaster != other
	  && tryko(xi, xj, color, "defend4-B")) {
	if (do_attack(si, sj, NULL, NULL, color) != 1) {
	  savei = xi;
	  savej = xj;
	  savecode = 3;
	}
	popgo();
      }
    }
  }

  if (stackp <= backfill_depth) {
    int bc = break_chain2(si, sj, &xi, &xj, komaster);
    if (bc == 1) {
      SGFTRACE(xi, xj, 1, "break chain2");
      READ_RETURN(read_result, i, j, xi, xj, 1);
    }
    UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, bc, xi, xj);
  }

  if (savecode != 0) {
    SGFTRACE(savei, savej, savecode, "saved move");
    READ_RETURN(read_result, i, j, savei, savej, savecode);
  }

  RTRACE("failed to find rescuing move.\n");
  SGFTRACE(-1, -1, 0, NULL);
  READ_RETURN0(read_result);
}


/*
 * special_rescue(si, sj, ai, aj, *ti, *tj) is called with (si, sj) a
 * string having a liberty at (ai, aj). The saving move is returned
 * in (*ti, *tj).
 *
 * This checks whether a move on a second order liberty is a rescuing
 * move, e.g. in shapes like:
 *
 *   .        O        O       X.XXO
 *  O.*  or  ..*  or  O.*  or  XOOXO
 *   O        O        O       ...*.
 *                             -----
 *
 * This will occasionally save a string where no other move will. To
 * reduce the branching caused by these moves, we require that the
 * opponent can be trivially captured when trying to intercept on the
 * corresponding first order liberty.
 */

static int
special_rescue(int si, int sj, int ai, int aj, int *ti, int *tj, int komaster)
{
  int other = OTHER_COLOR(p[si][sj]);
  int k;
  int savei = -1, savej = -1;
  int savecode = 0;

  /* Loop over the four neighbours of the liberty, (ai+di, aj+dj). */
  for (k=0; k<4; k++) {
    int di = deltai[k];
    int dj = deltaj[k];
    if (ON_BOARD(ai+di, aj+dj)
	&& p[ai+di][aj+dj] == EMPTY) {
      /* Use approxlib() to test for trivial capture. */
      if (approxlib(ai, aj, other, 3, NULL, NULL) > 2)
	continue;

      if (trymove(ai+di, aj+dj, p[si][sj], "special_rescue", si, sj)) {
	int acode = do_attack(si, sj, NULL, NULL, komaster);
	if (acode == 0) {
	  popgo();
	  if (ti) *ti = ai+di;
	  if (tj) *tj = aj+dj;
	  return 1;
	}
	UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, ai+di, aj+dj);
	popgo();
      }
    }
  }

  if (savecode != 0) {
    if (ti) *ti = savei;
    if (tj) *tj = savej;
    return savecode;
  }
  
  return 0;
}


/* In a situation like this:
 *       
 *   OOXXXX     the following code can find the
 *   .OXOOX     defensive move at 'c'.
 *   .cO.OX
 *   .X.OOX
 *   ------
 */
static int
special_rescue2(int si, int sj, int libi[2], int libj[2], 
		int *ti, int *tj, int komaster)
{
  int color = p[si][sj];
  int other = OTHER_COLOR(color);
  int newlibi[3];
  int newlibj[3];
  int xi, xj;
  int savei = -1, savej = -1;
  int savecode = 0;
  int k;

  for (k = 0; k < 2; ++k) {
    /* Let (ai, aj) and (bi, bj) be the two liberties. Reverse the
     * order during the second pass through the loop.
     */
    int ai = libi[k];
    int aj = libj[k];
    int bi = libi[1-k];
    int bj = libj[1-k];
    if (is_suicide(ai, aj, other) 
	&& (approxlib(ai, aj, color, 3, newlibi, newlibj)==2)) {
      if ((newlibi[0] != bi) || (newlibj[0] != bj)) {
	xi = newlibi[0];
	xj = newlibj[0];
      } else {
	xi = newlibi[1];
	xj = newlibj[1];
      }

      if (trymove(xi, xj, color, "special_rescue2", si, sj)) {
	int acode = do_attack(si, sj, NULL, NULL, komaster);
	if (acode != 1) {
	  if (acode == 0) {
	    popgo();
	    if (ti) *ti = xi;
	    if (tj) *tj = xj;
	    return 1;
	  }
	  UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, xi, xj);
	}
	popgo();
      }
    }
  }

  if (savecode) {
    if (ti) *ti = savei;
    if (tj) *tj = savej;
    return savecode;
  }

  return 0;
}


/* In a situation like this:
 *
 *   ...X.XXO
 *   .XXXOOXO
 *   XXOO.OXO     the following code can find the
 *   .O..X.*.     defensive move at '*'.
 *   --------
 *
 *   OXO   cde
 *   .*.   afg
 *   ---   b--
 */
static int
special_rescue3(int si, int sj, int libi[3], int libj[3], 
		int *ti, int *tj, int komaster)
{
  int color = p[si][sj];
  int other = OTHER_COLOR(color);
  int ai, aj, bi, bj, ci, cj, di, dj, ei, ej, fi, fj, gi, gj, oi, oj;
  int savei = -1, savej = -1;
  int savecode = 0;
  int k, l, r;

  ASSERT(countlib(si, sj) == 3, si, sj);
  
  for (r = 0; r < 3; r++) {
    /* Let (ai, aj) be one of the three liberties. */
    ai = libi[r];
    aj = libj[r];
    /* Try to find the configuration above. */
    for (k = 0; k < 4; k++) {
      bi = ai + deltai[k];
      bj = aj + deltaj[k];
      if (ON_BOARD(bi, bj))
	continue;

      ci = ai - deltai[k];
      cj = aj - deltaj[k];
      if (p[ci][cj] != color)
	continue;
      
      find_origin(ci, cj, &oi, &oj);
      if (oi != si || oj != sj)
	continue;

      for (l = 0; l < 2; l++) {
	int normali = deltaj[k];
	int normalj = -deltai[k];
	if (l == 1) {
	  normali = -normali;
	  normalj = -normalj;
	}
	
	di = ci + normali;
	dj = cj + normalj;
	if (!ON_BOARD(di, dj) || p[di][dj] != other)
	  continue;

	ei = di + normali;
	ej = dj + normalj;
	if (!ON_BOARD(ei, ej) || p[ei][ej] != color)
	  continue;

	fi = ai + normali;
	fj = aj + normalj;
	if (!ON_BOARD(fi, fj) || p[fi][fj] != EMPTY)
	  continue;

	gi = fi + normali;
	gj = fj + normalj;
	if (!ON_BOARD(gi, gj) || p[gi][gj] != EMPTY)	
	  continue;

	/* Configuration found. Now require an X move at 'a' not
	 * getting too many liberties
	 */

	if (approxlib(ai, aj, other, 4, NULL, NULL) > 3)
	  continue;
	
	/* Try to play at (fi, fj). */
	if (trymove(fi, fj, color, "special_rescue3", si, sj)) {
	  int acode = do_attack(si, sj, NULL, NULL, komaster);
	  if (acode != 1) {
	    if (acode == 0) {
	      popgo();
	      if (ti) *ti = fi;
	      if (tj) *tj = fj;
	      return 1;
	    }
	    UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, fi, fj);
	  }
	  popgo();
	}
      }
    }
  }

  if (savecode) {
    if (ti) *ti = savei;
    if (tj) *tj = savej;
    return savecode;
  }
  
  return 0;
}


/* 
 * This function handles some special cases on the edge.
 *
 * 1. If (si, sj) points to a string and 'a' an edge liberty of it,
 *    there is no point of trying to defend the string by crawling 
 *    along the edge if there is no hope of ever getting more liberties.
 *    This is of course if the blocking enemy group has enough liberties
 *    of its own.
 *
 *      XX       XX
 *      O.       Oa
 *      --       --
 *
 *    This function searches the edge towards the corner and sees if there
 *    is a friendly stone on one of the two first lines. If not, the move
 *    is removed from the  list of moves (movei[], movej[]).
 *
 * 2. If (si, sj) points to a string and 'a' an edge liberty of it,
 *    the drawing back/climbing up move 'b' is often correct attack or 
 *    defense. Another good move to try is 'c'.
 * 
 *      X.?        Xbc
 *      O..        Oa.
 *      ---        ---
 *
 *    This function adds the points configured like 'b' and 'c' relative to
 *    (si, sj) to the list of moves (movei[], movej[]).
 *
 * color is the color to move.
 */

static void
propose_edge_moves(int si, int sj, int *libi, int *libj, int libs,
		   int movei[MAX_MOVES], int movej[MAX_MOVES], int *moves,
		   int to_move)
{
  int color = p[si][sj];
  int other = OTHER_COLOR(color);
  int rightdir = 0;
  int righti, rightj;
  int updir = 0;
  int upi, upj;
  int found_it;
  int ai, aj;
  int k;

  for (k = 0; k < libs; k++) {
    ai = libi[k];
    aj = libj[k];
    
    /* Find out some directions.  We call the directions `rightdir and
     * `updir', as in the figure below, but in reality, they can have
     * any one of the standard eight directons through rotation.
     * rightdir and updir are indices into (deltai[],deltaj[]).
     *
     *     X.?        Xbc
     *     O..        Oa.
     *     ---        ---
     *
     * deltai, deltaj store directions in the order SOUTH, WEST, NORTH, EAST.
     */
    found_it = 0;
    if (ai == 0) {
      /* Top edge */
      if (aj > 0 && p[ai][aj-1] == color) {
	/* Group to the left */
	found_it = 1;
	updir = 0;
	rightdir = 3;
      }
      else if (aj < board_size - 1 && p[ai][aj+1] == color) {
	/* Group to the left */
	found_it = 1;
	updir = 0;
	rightdir = 1;
      }
    }
    else if (ai == board_size - 1) {
      /* Bottom edge */
      if (aj > 0 && p[ai][aj-1] == color) {
	/* Group to the left */
	found_it = 1;
	updir = 2;
	rightdir = 3;
      }
      else if (aj < board_size - 1 && p[ai][aj+1] == color) {
	/* Group to the left */
	found_it = 1;
	updir = 2;
	rightdir = 1;
      }
    }

    if (aj == 0) {
      /* Left edge */
      if (ai > 0 && p[ai-1][aj] == color) {
	/* Group to the top */
	found_it = 1;
	updir = 3;
	rightdir = 0;
      }
      else if (ai < board_size - 1 && p[ai+1][aj] == color) {
	/* Group to the bottom */
	found_it = 1;
	updir = 3;
	rightdir = 2;
      }
    }
    else if (aj == board_size - 1) {
      /* Right edge */
      if (ai > 0 && p[ai-1][aj] == color) {
	/* Group to the top */
	found_it = 1;
	updir = 1;
	rightdir = 0;
      }
      else if (ai < board_size - 1 && p[ai+1][aj] == color) {
	/* Group to the bottom */
	found_it = 1;
	updir = 1;
	rightdir = 2;
      }
    }

    if (!found_it)
      continue;

    upi = deltai[updir];
    upj = deltaj[updir];
    righti = deltai[rightdir];
    rightj = deltaj[rightdir];

    if (p[ai+upi][aj+upj] == other	   /* other on top of liberty */
	&& countlib(ai+upi, aj+upj) > 4    /* blocking group must be secure */
	&& p[si][sj] == to_move)           /* only applicable as defense */
    {
      /* Case 1: other above the liberty (crawl along the edge). */
      int  xi, xj;
	
      xi = ai;
      xj = aj;
      while (ON_BOARD(xi, xj)) {
	if (p[xi][xj] == color
	    || p[xi+upi][xj+upj] == color)
	  break;

	xi += righti;
	xj += rightj;
      }

      /* If no friendly stone found, then it is pointless and we
       * can just as well remove the move. */
      if (!ON_BOARD(xi, xj)) {
	REMOVE_CANDIDATE_MOVE(ai, aj, movei, movej, *moves);
      }
    }
    else if (p[ai+upi][aj+upj] == EMPTY           /* empty above the liberty */
	     && p[ai-righti+upi][aj-rightj+upj] == other
	     && ON_BOARD(ai+righti, aj+rightj)    /* c is on board */
	     && p[ai+righti][aj+rightj] == EMPTY) /* empty to the right */
    {
      /* Case 2: Try to escape or contain. */

      /* Add b */
      ADD_CANDIDATE_MOVE(ai+upi, aj+upj, movei, movej, *moves);

      /* Add c if empty */
      if (p[ai+righti+upi][aj+rightj+upj] == EMPTY)
	ADD_CANDIDATE_MOVE(ai+righti+upi, aj+rightj+upj, 
			   movei, movej, *moves);
    }
  }
}


/* ================================================================ */  
/*                       Attacking functions                        */
/* ================================================================ */


/* Like attack, but takes the komaster argument. If the
 * opponent is komaster, reading functions will not try
 * to take ko.
 */

static int 
do_attack(int si, int sj, int *i, int *j, int komaster)
{
  int color = p[si][sj];
  int xi, xj;
  int libs;
  int result = 0;
  int found_read_result;
  Read_result *read_result;

  SETUP_TRACE_INFO("attack", si, sj);

  ASSERT(color != 0, si, sj);

  if (color == 0)      /* if assertions are turned off, silently fails */
    return 0;

  libs = countlib(si, sj);

  if (libs > 4
      || (libs == 4 && stackp > fourlib_depth)) {
    /* No need to cache the result in these cases. */
    SGFTRACE(0, 0, 0, "too many liberties");
    return 0;
  }

  if ((stackp <= depth) && (hashflags & HASH_ATTACK)) {
    found_read_result = get_read_result(ATTACK, &si, &sj, &read_result);
    if (found_read_result) {
      TRACE_CACHED_RESULT(*read_result);
      if (rr_get_result(*read_result) != 0) {
	if (i) *i = rr_get_result_i(*read_result);
	if (j) *j = rr_get_result_j(*read_result);
      }

      SGFTRACE(rr_get_result_i(*read_result), rr_get_result_j(*read_result),
	       rr_get_result(*read_result), "cached");
      return rr_get_result(*read_result);
    }

    /* This data should always be recorded. */
    if (read_result) {
      rr_set_routine_i_j_stackp(*read_result, ATTACK, si, sj, stackp);
    }
  } else
    read_result = NULL;

  /* Treat the attack differently depending on how many liberties the 
     string at (si, sj) has. */
  if (libs == 1)
    result = attack1(si, sj, &xi, &xj, komaster);
  else if (libs == 2)
    result = attack2(si, sj, &xi, &xj, komaster);
  else if (libs == 3)
    result = attack3(si, sj, &xi, &xj, komaster);
  else if (libs == 4)
    result = attack4(si, sj, &xi, &xj, komaster);

  ASSERT(result >= 0 && result <= 3, si, sj);
  
  if (result)
    READ_RETURN(read_result, i, j, xi, xj, result);

  READ_RETURN0(read_result);
}


/* If (si, sj) points to a group with exactly one liberty, attack1
 * determines whether it can be captured by playing at this liberty.
 * If yes, (*i, *j) is the killing move. i & j may be NULL if caller is
 * only interested in whether it can be captured.
 *
 * The attack may fail for two different reasons. The first one is
 * that the attack may be an illegal ko capture, in this case 3 is
 * returned (need to play a ko threat before the attack can be
 * fulfilled).
 *
 * The second cause for failure is that the attack is caught in a
 * snapback. We must require that it is a proper snapback, though. By
 * proper snapback we mean a position like
 *
 *  XXXXO
 *  XO.XO
 *  XOXOO
 *  -----
 *
 * where capture by O and recapture by X leaves the X stone intact
 * with at least two liberties:
 *
 *  XXXXO
 *  X..XO
 *  X.XOO
 *  -----
 *
 * There are a number of different kinds of improper snapbacks, which
 * have in common that the attacked string ends up captured. We don't
 * consider these as failures to attack. Three examples are given below.
 *
 *   XXOOOOO     (X can recapture but loses most of the string.)
 *   X.XXXXO
 *   -------
 *
 *   XXXOOOOOOOO (Like the previous example, except O loses one more stone)
 *   XO*XXXXXXXO
 *   -----------
 *
 *   XXXOO       (After three captures, the lone X stone is gone.)
 *   XO.XO
 *   -----
 *
 * This function is fast and never branches. There's little point in
 * caching the result.
 */

static int
attack1(int si, int sj, int *i, int *j, int komaster)
{
  int xi, xj;
  int color = p[si][sj];
  int other = OTHER_COLOR(color);
  int result = -1;
  
  SETUP_TRACE_INFO("attack1", si, sj);
  reading_node_counter++;
  
  /* Pick up the position of the liberty. */
  findlib(si, sj, 1, &xi, &xj);

  /* If the attacked string consists of more than one stone, the
   * attack never fails. (This assumes simple ko rule. With superko
   * rule it could still be a ko violation.)
   */
  if (countstones(si, sj) > 1)
    result = 1;
  
  /* Try to play on the liberty. This fails if and only if it is an
   * illegal ko capture.
   */
  else if (trymove(xi, xj, other, "attack1-A", si, sj)) {
    /* Is the attacker in atari? If not the attack was successful. */
    if (countlib(xi, xj) > 1)
      result = 1;

    /* If the attacking string is also a single stone, a possible
     * recapture would be a ko violation, so the defender has to make
     * a ko threat first.
     */
    else if (countstones(xi, xj) == 1) {
      /* If the defender is allowed to take the ko at all the result is 2. */
      if (komaster != other)
	result = 2;
      else 
	/* But if the attacker is komaster, the attack was successful. */
	result = 1;
    }
      
    /* Otherwise, do recapture. Notice that the liberty must be
     * at (si, sj) since we have already established that this string
     * was a single stone.
     */
    else if (trymove(si, sj, color, "attack1-B", si, sj)) {
      /* If this was a proper snapback, (si, sj) will now have more
       * than one liberty.
       */
      if (countlib(si, sj) > 1) {
	/* Proper snapback, attack fails. */
	RTRACE("SNAPBACK at %m!\n", i, j);
	result = 0;
      }
      else
	result = 1;
      popgo();
    }
    popgo();
  }
  else {/* Illegal ko capture. */
    if (komaster != color) 
      result = 3;
    else
      result = 0;
  }

  /* If not yet successful, try backfilling.
   * FIXME PRE3.0: Maybe only meaningful to do this in positions involving ko.
   */
  if (result != 1) {
    int liberties;
    int libi[6];
    int libj[6];
    int k;
    liberties = approxlib(xi, xj, color, 6, libi, libj);
    if (liberties <= 5)
      for (k=0; k<liberties; k++) {
	int ai = libi[k];
	int aj = libj[k];
	if (!is_self_atari(ai, aj, other)
	    && trymove(ai, aj, other, "attack1-C", si, sj)) {
	  int dcode = do_find_defense(si, sj, NULL, NULL, komaster);
	  if (dcode != 1 && do_attack(si, sj, NULL, NULL, komaster)) {
	    if (dcode == 0) {
	      popgo();
	      SGFTRACE(ai, aj, 1, "backfilling");
	      if (i) *i = ai;
	      if (j) *j = aj;
	      return 1;
	    }
	    UPDATE_SAVED_KO_RESULT(result, xi, xj, dcode, ai, aj);
	  }
	  popgo();
	}
      }
  }
  
  if (result > 0) {
    if (i) *i = xi;
    if (j) *j = xj;
    SGFTRACE(xi, xj, result, NULL);
  }
  else {
    SGFTRACE(-1, -1, 0, NULL);
  }
  
  return result;
}


/* If si, sj points to a group with exactly two liberties
 * attack2 determines whether it can be captured in ladder or net.
 * If yes, *i, *j is the killing move. i & j may be null if caller 
 * is only interested in whether it can be captured.
 *  
 * Returns 2 or 3 if it can be killed conditioned on ko. Returns
 * 2 if it can be killed provided (other) is willing to ignore
 * any ko threat. Returns 3 if (other) wins provided he has a
 * ko threat which must be answered. Can give a return code 3 
 * yet (*i,*j)=(-1,-1) if the winning move is an illegal
 * ko capture. In this case, making a ko threat and having it
 * answered should transform the position to one where the return
 * code is 2. 
 *
 * See the comment before defend1 about ladders and reading depth.  */

static int 
attack2(int si, int sj, int *i, int *j, int komaster)
{
  int color, other;
  int ai, aj;
  int hi, hj;
  int xi, xj;
  int liberties, r;
  int libi[2], libj[2];
  int lib2i[2], lib2j[2];
  int adj, adji[MAXCHAIN], adjj[MAXCHAIN];
  int savei = -1, savej = -1;
  int savecode = 0;
  int acode;
  int dcode;
  int k;
  int atari_possible = 0;
  int movei[MAX_MOVES], movej[MAX_MOVES];
  int moves = 0;
  int found_read_result;
  Read_result *read_result;

  SETUP_TRACE_INFO("attack2", si, sj);
  reading_node_counter++;

  find_origin(si, sj, &si, &sj);
  assert(p[si][sj]!=EMPTY);
  assert(countlib(si, sj) == 2);

  RTRACE("checking attack on %m with 2 liberties\n", si, sj);

  color = p[si][sj];
  other = OTHER_COLOR(color);

  if ((stackp <= depth) && (hashflags & HASH_ATTACK2)) {
  
    found_read_result = get_read_result(ATTACK2, &si, &sj, &read_result);
    if (found_read_result) {
      TRACE_CACHED_RESULT(*read_result);
      if (rr_get_result(*read_result) != 0) {
	if (i) *i = rr_get_result_i(*read_result);
	if (j) *j = rr_get_result_j(*read_result);
      }

      SGFTRACE(rr_get_result_i(*read_result), rr_get_result_j(*read_result),
	       rr_get_result(*read_result), "cached");
      return rr_get_result(*read_result);
    }

    /* This data should always be recorded. */
    if (read_result) {
      rr_set_routine_i_j_stackp(*read_result, ATTACK2, si, sj, stackp);
    }
  } else
    read_result = NULL;

  /* The attack may fail if a boundary string is in atari and cannot 
   * be defended.  First we must try defending such a string. 
   *
   * We start by trying to defend the boundary string by looking for an
   * adjacent string which is in atari. 
   */
  chainlinks2(si, sj, &adj, adji, adjj, 1);
  for (r = 0; r < adj; r++) {
    /* if stackp > depth and any boundary chain is in atari, assume safe.
     * However, if the captured chain is only of size 1, there can still
     * be a working ladder, so continue if that is the case.
     */
    if (stackp > depth && countstones(adji[r], adjj[r]) > 1) {
      SGFTRACE(-1, -1, 0, "boundary in atari");
      READ_RETURN0(read_result);
    }

    /* Pick up moves breaking the second order chain. */
    if (stackp <= depth)
      break_chain_moves(adji[r], adjj[r], movei, movej, &moves);
    
    findlib(adji[r], adjj[r], 1, &hi, &hj);
    ADD_CANDIDATE_MOVE(hi, hj, movei, movej, moves);
  }

  /* Get the two liberties of (si, sj). */
  liberties = findlib(si, sj, 2, libi, libj);
  ASSERT(liberties == 2, si, sj);

  for (k = 0; k < 2; k++) {
    ai = libi[k];
    aj = libj[k];
    if (!is_self_atari(ai, aj, other))
      atari_possible = 1;
    /* we only want to consider the move at (ai,aj) if:
     * stackp <= backfill_depth
     * -or-  stackp <= depth and it is an isolated stone
     * -or-  it is not in immediate atari
     */
    if ((stackp <= backfill_depth) 
	|| ((stackp <= depth) 
	    && ((ai == 0) || (p[ai-1][aj] != other))
	    && ((ai == board_size-1) || (p[ai+1][aj] != other))
	    && ((aj == 0) || (p[ai][aj-1] != other))
	    && ((aj == board_size-1) || (p[ai][aj+1] != other)))
	|| !is_self_atari(ai, aj, other))
      ADD_CANDIDATE_MOVE(ai, aj, movei, movej, moves);

    /* Try backfilling if atari is impossible. */
    if (stackp <= backfill_depth
	&& approxlib(ai, aj, other, 2, lib2i, lib2j) == 1) {
      int ui = lib2i[0];
      int uj = lib2j[0];
      ADD_CANDIDATE_MOVE(ui, uj, movei, movej, moves);
    }
  }

  /* If one of the surrounding chains have only two liberties, which
   * coincide with the liberties of the attacked string, we try to
   * backcapture.
   */
  
  chainlinks2(si, sj, &adj, adji, adjj, 2);
  for (r = 0; r<adj; r++) {
    ai = adji[r];
    aj = adjj[r];
    if (liberty_of(libi[0], libj[0], ai, aj)
	&& liberty_of(libi[1], libj[1], ai, aj))
      break_chain_moves(ai, aj, movei, movej, &moves);
  }
  
  propose_edge_moves(si, sj, libi, libj, liberties, movei, movej, &moves,
		     other);
  order_moves(si, sj, moves, movei, movej, other);

  for (k = 0; k < moves; k++) {
    ai = movei[k];
    aj = movej[k];
    if (trymove(ai, aj, other, "attack2-A", si, sj)) {
      dcode = do_find_defense(si, sj, NULL, NULL, komaster);
      if (dcode != 1 && do_attack(si, sj, NULL, NULL, komaster)) {
	if (dcode == 0) {
	  popgo();
	  SGFTRACE(ai, aj, 1, "attack effective");
	  READ_RETURN(read_result, i, j, ai, aj, 1);
	}
	UPDATE_SAVED_KO_RESULT(savecode, savei, savej, dcode, ai, aj);
      }
      popgo();
    }
    else if (savecode == 0
	     && komaster != color
	     && is_ko(ai, aj, other)
	     && stackp <= ko_depth
	     && tryko(ai, aj, other, "attack2-B")) {
      if (do_find_defense(si, sj, NULL, NULL, other) != 1
	  && do_attack(si, sj, NULL, NULL, komaster) != 0) {
	savei = ai;
	savej = aj;
	savecode = 3;
      }
      popgo();
    }
  }
  
  /* The simple ataris didn't work. Try something more fancy. */
  acode = find_cap2(si, sj, libi[0], libj[0], libi[1], libj[1], &xi, &xj,
		    komaster);
  if (acode == 1) {
    SGFTRACE(xi, xj, 1, "find cap2");
    READ_RETURN(read_result, i, j, xi, xj, 1);
  }
  UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, acode, xi, xj);

  if (stackp <= backfill_depth) {
    acode = special_attack2(si, sj, libi, libj, &xi, &xj, komaster);
    if (acode == 1) {
      SGFTRACE(xi, xj, 1, "special attack2");
      READ_RETURN(read_result, i, j, xi, xj, 1);
    }
    UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, acode, xi, xj);
  }

  if (stackp <= backfill_depth) {
    acode = special_attack3(si, sj, libi, libj, &xi, &xj, komaster);
    if (acode == 1) {
      SGFTRACE(xi, xj, 1, "special attack3");
      READ_RETURN(read_result, i, j, xi, xj, 1);
    }
    UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, acode, xi, xj);
  }

  if (stackp <= backfill_depth) {
    acode = special_attack4(si, sj, libi, libj, &xi, &xj, komaster);
    if (acode == 1) {
      SGFTRACE(xi, xj, 1, "special attack4");
      READ_RETURN(read_result, i, j, xi, xj, 1);
    }
    UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, acode, xi, xj);
  }

  /* If it is not possible to make a direct atari, we try filling
   * a liberty of the superstring.
   */
  if (level >= 10
      && stackp <= backfill_depth
      && (stackp <= superstring_depth || !atari_possible)) {
    int libs;
    int libi[MAX_LIBERTIES + 4], libj[MAX_LIBERTIES + 4];
    int liberty_cap = 2;

    if (stackp <= backfill2_depth)
      liberty_cap = 3;
    
    find_superstring_liberties(si, sj, &libs, libi, libj, liberty_cap);
    if (libs <= 5) {
      for (k = 0; k < libs; k++) {
	int ai = libi[k];
	int aj = libj[k];
	
	if (liberty_of(ai, aj, si, sj))
	  continue;
	if (trymove(ai, aj, other, "attack2-C", si, sj)) {
	  if (countlib(ai, aj) == 1) {
	    /* can't atari, try backfilling */
	    if (restricted_defend1(ai, aj, &xi, &xj, komaster,
				   libs, libi, libj)) {
	      popgo();
	      if (trymove(xi, xj, other, "attack2-D", si, sj)) {
		dcode = do_find_defense(si, sj, NULL, NULL, komaster);
		if (dcode != 1 && do_attack(si, sj, NULL, NULL, komaster)) {
		  if (dcode == 0) {
		    popgo();
		    SGFTRACE(xi, xj, 1, "attack effective");
		    READ_RETURN(read_result, i, j, xi, xj, 1);
		  }
		  UPDATE_SAVED_KO_RESULT(savecode, savei, savej, dcode,
					 xi, xj);
		}
		popgo();
	      }
	    }
	    else popgo();
	  }
	  else {
	    dcode = do_find_defense(si, sj, NULL, NULL, komaster);
	    if (dcode != 1 && do_attack(si, sj, NULL, NULL, komaster)) {
	      if (dcode == 0) {
		popgo();
		SGFTRACE(ai, aj, 1, "attack effective");
		READ_RETURN(read_result, i, j, ai, aj, 1);
	      }
	      UPDATE_SAVED_KO_RESULT(savecode, savei, savej, dcode, ai, aj);
	    }
	    popgo();
	  }
	}
      }
    }
  }

  if (savecode == 0) {
    RTRACE("ALIVE!!\n");
    SGFTRACE(-1, -1, 0, NULL);
    READ_RETURN0(read_result);
  }

  SGFTRACE(savei, savej, savecode, "saved move");
  READ_RETURN(read_result, i, j, savei, savej, savecode);
}



/* attack3(si, sj, *i, *j) is used when (si, sj) points to a group with
 * three liberties. It returns true if it finds a way to kill the group.
 *
 * Return code is 2 if the group can be killed if the attacker is 
 * willing to ignore any ko threat.
 *
 * Return code is 3 if the group can be killed if the attacker is 
 * able to find a ko threat which must be answered.
 *
 * If non-NULL (*i, *j) will be set to the move which makes the
 * attack succeed.
 */

static int 
attack3(int si, int sj, int *i, int *j, int komaster)
{
  int color = p[si][sj];
  int other = OTHER_COLOR(color);
  int adj, adji[MAXCHAIN], adjj[MAXCHAIN];
  int xi, xj;
  int liberties;
  int libi[3], libj[3];
  int r;
  int dcode = 0;
  int k;
  int movei[MAX_MOVES], movej[MAX_MOVES];
  int moves = 0;
  int savei = -1, savej = -1;
  int savecode = 0;
  int found_read_result;
  Read_result *read_result;

  SETUP_TRACE_INFO("attack3", si, sj);
  reading_node_counter++;

  assert(p[si][sj] != EMPTY);

  if ((stackp <= depth) && (hashflags & HASH_ATTACK3)) {
    found_read_result = get_read_result(ATTACK3, &si, &sj, &read_result);
    if (found_read_result) {
      TRACE_CACHED_RESULT(*read_result);
      if (rr_get_result(*read_result) != 0) {
	if (i) *i = rr_get_result_i(*read_result);
	if (j) *j = rr_get_result_j(*read_result);
      }

      SGFTRACE(rr_get_result_i(*read_result), rr_get_result_j(*read_result),
	       rr_get_result(*read_result), "cached");
      return rr_get_result(*read_result);
    }

    /* This data should always be recorded. */
    if (read_result) {
      rr_set_routine_i_j_stackp(*read_result, ATTACK3, si, sj, stackp);
    }
  } else
    read_result = NULL;

  if (stackp > depth) {
    SGFTRACE(-1, -1, 0, "stackp > depth");
    READ_RETURN0(read_result);
  }

  chainlinks2(si, sj, &adj, adji, adjj, 1);
  for (r = 0; r < adj; r++) {
    int hi, hj;
    break_chain_moves(adji[r], adjj[r], movei, movej, &moves);
    
    findlib(adji[r], adjj[r], 1, &hi, &hj);
    ADD_CANDIDATE_MOVE(hi, hj, movei, movej, moves);
  }

  /* Defend against double atari in the surrounding chain early. */
  double_atari_chain2(si, sj, movei, movej, &moves);
  
  /* Get the three liberties of (si, sj). */
  liberties = findlib(si, sj, 3, libi, libj);
  ASSERT(liberties == 3, si, sj);

  for (k=0; k<3; k++) {
#if 0
    int lib2i[2];
    int lib2j[2];
#endif
    int ai = libi[k];
    int aj = libj[k];
    /* we only want to consider the move at (ai,aj) if:
     * stackp <= backfill_depth
     * -or-  stackp <= depth and it is an isolated stone
     * -or-  it is not in immediate atari
     */
    if ((stackp <= backfill_depth) 
	|| ((stackp <= depth) 
	    && ((ai == 0) || (p[ai-1][aj] != other))
	    && ((ai == board_size-1) || (p[ai+1][aj] != other))
	    && ((aj == 0) || (p[ai][aj-1] != other))
	    && ((aj == board_size-1) || (p[ai][aj+1] != other)))
	|| !is_self_atari(ai, aj, other))
      ADD_CANDIDATE_MOVE(ai, aj, movei, movej, moves);

#if 0
    /* Try backfilling if atari is impossible. */
    if (stackp <= backfill_depth
	&& approxlib(ai, aj, other, 2, lib2i, lib2j) == 1) {
      int ui = lib2i[0];
      int uj = lib2j[0];
      ADD_CANDIDATE_MOVE(ui, uj, movei, movej, moves);
    }
#endif
  }

  /* Pick up some edge moves. */
  propose_edge_moves(si, sj, libi, libj, liberties, movei, movej, &moves,
		     other);
  order_moves(si, sj, moves, movei, movej, other);

  /* Try the moves collected so far. */
  for (k=0; k<moves; k++) {
    if (stackp >= branch_depth && k > 0)
      break;
    xi = movei[k];
    xj = movej[k];
    if (trymove(xi, xj, other, "attack3-A", si, sj)) {
      dcode = do_find_defense(si, sj, NULL, NULL, komaster);
      if (dcode != 1 && do_attack(si, sj, NULL, NULL, komaster)) {
	if (dcode == 0) {
	  popgo();
	  SGFTRACE(xi, xj, 1, "attack effective");
	  READ_RETURN(read_result, i, j, xi, xj, 1);
	}
	UPDATE_SAVED_KO_RESULT(savecode, savei, savej, dcode, xi, xj);
      }
      popgo();
    }
    else if (savecode == 0
	     && komaster != color
	     && is_ko(xi, xj, other)
	     && stackp <= ko_depth
	     && tryko(xi, xj, other, "attack3-B")) {
      if (do_find_defense(si, sj, NULL, NULL, other) != 1
	  && do_attack(si, sj, NULL, NULL, komaster) != 0) {
	savei = xi;
	savej = xj;
	savecode = 3;
      }
      popgo();
    }
  }
  
  /* The simple ataris didn't work. Try something more fancy. */
  if (stackp <= backfill_depth) {
    int acode = find_cap3(si, sj, &xi, &xj, komaster);
    if (acode == 1) {
      SGFTRACE(xi, xj, 1, "find cap3");
      READ_RETURN(read_result, i, j, xi, xj, 1);
    }
    UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, acode, xi, xj);
  }

  if (stackp <= fourlib_depth) {
    int acode = draw_back(si, sj, &xi, &xj, komaster);
    if (acode == 1) {
      SGFTRACE(xi, xj, 1, "draw back");
      READ_RETURN(read_result, i, j, xi, xj, 1);
    }
    UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, acode, xi, xj);
  }

  /* Try to defend chain links with two liberties. */
  if (stackp <= backfill2_depth) {
    int saved_moves = moves;
    chainlinks2(si, sj, &adj, adji, adjj, 2);
    for (r=0; r<adj; r++) {
      int lib2i[2], lib2j[2];
      break_chain_moves(adji[r], adjj[r], movei, movej, &moves);
      break_chain2_moves(adji[r], adjj[r], movei, movej, &moves, 1);
      findlib(adji[r], adjj[r], 2, lib2i, lib2j);
      for (k=0; k<2; k++)
	ADD_CANDIDATE_MOVE(lib2i[k], lib2j[k], movei, movej, moves);
    }
    /* Only order and test the new set of moves. */
    order_moves(si, sj, moves-saved_moves,
		&(movei[saved_moves]), &(movej[saved_moves]), other);
    for (k=saved_moves; k<moves; k++) {
      if (stackp >= branch_depth && k > 0)
	break;
      xi = movei[k];
      xj = movej[k];
      
      if (trymove(xi, xj, other, "attack3-C", si, sj)) {
	dcode = do_find_defense(si, sj, NULL, NULL, komaster);
	if (dcode != 1 && do_attack(si, sj, NULL, NULL, komaster)) {
	  if (dcode == 0) {
	    popgo();
	    SGFTRACE(xi, xj, 1, "attack effective");
	    READ_RETURN(read_result, i, j, xi, xj, 1);
	  }
	  UPDATE_SAVED_KO_RESULT(savecode, savei, savej, dcode, xi, xj);
	}
	popgo();
      }
      else if (savecode == 0
	       && komaster != color
	       && is_ko(xi, xj, other)
	       && stackp <= ko_depth
	       && tryko(xi, xj, other, "attack3-D")) {
	if (do_find_defense(si, sj, NULL, NULL, other) != 1
	    && do_attack(si, sj, NULL, NULL, komaster) != 0) {
	  savei = xi;
	  savej = xj;
	  savecode = 3;
	}
	popgo();
      }
    }
  }
    
  /* If nothing else works, we try filling a liberty of the
   * super_string.
   */
  if (level >= 10 && stackp <= backfill2_depth) {
    int libs;
    int libi[MAX_LIBERTIES + 4], libj[MAX_LIBERTIES + 4];

    find_superstring_liberties(si, sj, &libs, libi, libj, 3);
    if (libs <= 5) {
      for (k = 0; k < libs; k++) {
	int ai = libi[k];
	int aj = libj[k];
	
	if (liberty_of(ai, aj, si, sj))
	  continue;
	if (trymove(ai, aj, other, "attack3-E", si, sj)) {
	  if (countlib(ai, aj) == 1) {
	    /* can't atari, try backfilling */
	    if (restricted_defend1(ai, aj, &xi, &xj, komaster,
				   libs, libi, libj)) {
	      popgo();
	      if (trymove(xi, xj, other, "attack3-F", si, sj)) {
		dcode = do_find_defense(si, sj, NULL, NULL, komaster);
		if (dcode != 1 && do_attack(si, sj, NULL, NULL, komaster)) {
		  if (dcode == 0) {
		    popgo();
		    SGFTRACE(xi, xj, 1, "attack effective");
		    READ_RETURN(read_result, i, j, xi, xj, 1);
		  }
		  UPDATE_SAVED_KO_RESULT(savecode, savei, savej, dcode,
					 xi, xj);
		}
		popgo();
	      }
	    }
	    else popgo();
	  }
	  else {
	    dcode = do_find_defense(si, sj, NULL, NULL, komaster);
	    if (dcode != 1 && do_attack(si, sj, NULL, NULL, komaster)) {
	      if (dcode == 0) {
		popgo();
		SGFTRACE(ai, aj, 1, "attack effective");
		READ_RETURN(read_result, i, j, ai, aj, 1);
	      }
	      UPDATE_SAVED_KO_RESULT(savecode, savei, savej, dcode, ai, aj);
	    }
	    popgo();
	  }
	}
      }
    }
  }

  if (savecode != 0) {
    SGFTRACE(savei, savej, savecode, "saved move");
    READ_RETURN(read_result, i, j, savei, savej, savecode);
  }
  
  SGFTRACE(-1, -1, 0, NULL);
  READ_RETURN0(read_result);
}


/* attack4 tries to capture a string with 4 liberties. This function
 * is not cached.
 */

static int 
attack4(int si, int sj, int *i, int *j, int komaster)
{
  int color = p[si][sj];
  int other = OTHER_COLOR(color);
  int ai, aj;
  int xi, xj;
  int r;
  int k;
  int liberties;
  int libi[4], libj[4];
  int mx[MAX_BOARD][MAX_BOARD];  
  int adj, adji[MAXCHAIN], adjj[MAXCHAIN];
  int savecode = 0;
  int savei = -1, savej = -1;
  int bcode;

  SETUP_TRACE_INFO("attack4", si, sj);
  
  assert(p[si][sj]!=EMPTY);
  reading_node_counter++;
  
  if (stackp > depth) {
    SGFTRACE(-1, -1, 0, "stackp > depth");
    return 0;
  }
  
 /* The attack may fail if a boundary string is in atari and cannot be
  * defended. First we must try defending such a string.
  */
  
  memset(mx, 0, sizeof(mx));

  chainlinks2(si, sj, &adj, adji, adjj, 1);
  for (r=0; r<adj; r++) {
    findlib(adji[r], adjj[r], 1, &xi, &xj);

    /* If moving out results in more than one liberty, we resume the attack. */
    if (!mx[xi][xj] && !is_self_atari(xi, xj, other)) {
      if (trymove(xi, xj, other, "attack4-A", si, sj)) {
	int dcode;
	mx[xi][xj] = 1;
	dcode = do_find_defense(si, sj, NULL, NULL, komaster);
	if (dcode != 1 && do_attack(si, sj, NULL, NULL, komaster) != 0) {
	  if (dcode == 0) {
	    popgo();
	    SGFTRACE(xi, xj, 1, "defend boundary");
	    if (i) *i = xi;
	    if (j) *j = xj;
	    return 1;
	  }
	  UPDATE_SAVED_KO_RESULT(savecode, savei, savej, dcode, xi, xj);
	}
	popgo();
      }
    }
    bcode = relative_break_chain(adji[r], adjj[r], &xi, &xj, si, sj,
				 komaster);
    if (bcode == 1) {
      SGFTRACE(xi, xj, 1, "relative break chain");
      if (i) *i = xi;
      if (j) *j = xj;
      return 1;
    }
    UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, bcode, xi, xj);
  }

  RTRACE("checking attack on %m with 4 liberties\n", si, sj);
  liberties = findlib(si, sj, 4, libi, libj);
  ASSERT (liberties == 4, si, sj);
  order_moves(si, sj, liberties, libi, libj, other);

  /* If find_defense was fully reliable for strings with 3 liberties
   * we would not need the inner test (if !attack( ...,x)
   * ||!does_defend(x, ...)).  Currently it seems that the
   * performance penalty of making defend3 try attacking boundary
   * string with 3 liberties is too great. Instead, we try here to
   * attack the stone just played, and if it cannot be captured, or if
   * attacking it does not defend (si,sj), the attack succeeds.
   *
   */

  for (k = 0; k < liberties; ++k) {

    if (stackp >= branch_depth && k > 0)
      break;
    ai = libi[k];
    aj = libj[k];
    if (trymove(ai, aj, other, "attack4-B", si, sj)) {
      int dcode;
      RTRACE("try attacking at %m ...\n", ai, aj);
      ASSERT(countlib(si, sj) == 3, si, sj);
      dcode = do_find_defense(si, sj, NULL, NULL, komaster);
      if ((dcode != 1)
	  && do_attack(si, sj, NULL, NULL, komaster)) {
	if (1 || !do_attack(ai, aj, &xi, &xj, komaster) ||
	    !does_defend(xi, xj, si, sj)) {
	  if (dcode == 0) {
	    popgo();
	    SGFTRACE(ai, aj, 1, "attack just played stone");
	    if (i) *i = ai;
	    if (j) *j = aj;
	    return 1;
	  }
	  else
	    UPDATE_SAVED_KO_RESULT(savecode, savei, savej, dcode, ai, aj);
	}
      }
      popgo();
    }
  }

  if (savecode != 0) {
    SGFTRACE(savei, savej, savecode, "saved move");
    if (i) *i = savei;
    if (j) *j = savej;
    return savecode;
  }

  SGFTRACE(-1, -1, 0, NULL);
  return 0;
}


/* If (si, sj) points to a string with 2 or 3 liberties,
 * find_cap2(si, sj, ai, aj, bi, bj, &i, &j, komaster)
 * looks for a configuration of the following type:
 *
 *  X.
 *  .*
 *
 * where X is an element of the string in question. It tries the
 * move at * and returns true this move captures the string, leaving
 * (i, j) pointing to *. 
 */

static int
find_cap2(int si, int sj, int ai, int aj, int bi, int bj, int *i, int *j,
	  int komaster)
{
  int ti = -1, tj = -1;

  /* Check if the two liberties are located like the figure above. */
  if (((ai == bi+1) || (ai == bi-1)) 
      && ((aj == bj+1) || (aj == bj-1))) 
  {
    /* Which of the two corner points should we use? One of them is 
     * always occupied by the string at (m, n), the other one is either
     * free or occupied by something else.
     */
    if (p[bi][aj] == EMPTY) {
      ti = bi;
      tj = aj;
    }
    if (p[ai][bj] == EMPTY) {
      ti = ai;
      tj = bj;
    }

    /* If we didn't find a free intersection, we couldn't make the move. */
    if (ti == -1)
      return 0;

    /* Ok, we found the spot. Now see if the move works. */
    RTRACE("trying to capture %m with capping move at %m\n", si, sj, ti, tj);
    if (trymove(ti, tj, OTHER_COLOR(p[si][sj]),"find_cap2", si, sj)) {
      int dcode = do_find_defense(si, sj, NULL, NULL, komaster);
      popgo();
      if (dcode != 1) {
	if (i) *i = ti;
	if (j) *j = tj;
      }
      switch (dcode) {
      case 0:
	RTRACE("cap2 succeeded!\n");
	return 1;
	break;
      case 1:
	RTRACE("cap2 failed!\n");
	return 0;
	break;
      case 2:
	RTRACE("cap2 succeeded with ko return code 3\n");
	return 3;
	break;
      case 3:
	RTRACE("cap2 succeeded with ko return code 2\n");
	return 2;
	break;
      }
    }
  }

  return 0;
}    


/* If (si,sj) points to a string with 3 liberties, find_cap3(si,sj,&i,&j)
 * looks for a configuration of the following type:
 *
 *  XXa
 *  cb*
 *
 * where X are elements of the string in question and a, b and c are
 * its liberties. It tries the move at * and returns true this move
 * captures the string, leaving (i,j) pointing to *.
 */

static int
find_cap3(int si, int sj, int *i, int *j, int komaster)
{
  int ai, aj, bi, bj;
  int libi[3], libj[3];
  int xi = -1, xj = -1;
  int k;
  int savecode = 0;
  int savei = -1;
  int savej = -1;
  int acode;

  if (findlib(si, sj, 3, libi, libj) != 3)
    return 0;

  for (k = 0; k < 3; ++k) {
    /* k and k+1 mod 3 will be (0,1), (1,2) and (2,0); These are the 
     * three combinations of indices that we have to send to find_cap2.
     */
    ai = libi[k];
    aj = libj[k];
    bi = libi[(k+1)%3];
    bj = libj[(k+1)%3];

    acode = find_cap2(si, sj, ai, aj, bi, bj, &xi, &xj, komaster);
    if (acode == 1) {
      if (i) *i = xi;
      if (j) *j = xj;
      return 1;
    }
    UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej, acode, xi, xj);
  }

  if (savecode != 0) {
    if (i) *i = savei;
    if (j) *j = savej;
  }
  return savecode;
}


/* In a situation like this:
 *       
 * -----        the code that
 * cO.OX        follows can find
 * XXOOX        the attacking move
 * XO.OX        at 'c=(xi,xj)'.
 * XOOOX
 * XXXXX
 *
 * The name of the function corresponds to special_rescue2, which is
 * fairly similar to this situation.
 */

static int
special_attack2(int si, int sj, int libi[2], int libj[2], 
		int *ti, int *tj, int komaster)
{
  int color = p[si][sj];
  int other = OTHER_COLOR(color);
  int savecode = 0;
  int newlibi[3], newlibj[3];
  int xi, xj;
  int savei = -1, savej = -1;
  int k;

  for (k = 0; k < 2; ++k) {
    if (is_suicide(libi[k], libj[k], other) 
	&& (approxlib(libi[k], libj[k], color, 3, newlibi, newlibj)==2)) {
      if ((newlibi[0] != libi[1-k]) || (newlibj[0] != libj[1-k])) {
	xi = newlibi[0];
	xj = newlibj[0];
      } else {
	xi = newlibi[1];
	xj = newlibj[1];
      }

      if (!is_self_atari(xi, xj, other)
	  && trymove(xi, xj, other, "special_attack2", si, sj)) {
	int dcode = do_find_defense(si, sj, NULL, NULL, komaster);
	if (dcode != 1 && do_attack(si, sj, NULL, NULL, komaster)) {
	  if (dcode == 0) {
	    popgo();
	    if (ti) *ti = xi;
	    if (tj) *tj = xj;
	    return 1;
	  }
	  UPDATE_SAVED_KO_RESULT(savecode, savei, savej, dcode, xi, xj);
	}
	popgo();
      }
    }
  }

  if (savecode) {
    if (ti) *ti = savei;
    if (tj) *tj = savej;
    return savecode;
  }

  return 0;
}


/* In situations like these:
 *
 * ..XXX..   ...XX
 * .XX.XX.   .cO.X
 * XXOOOXX   ....X
 * XO.O.OX   XOOXX
 * XO.c.OX   XXXX.
 * -------
 *
 * the code that follows can find the attacking move at 'c=(xi,xj)'.
 */

static int
special_attack3(int si, int sj, int libi[2], int libj[2], 
		int *ti, int *tj, int komaster)
{
  int color = p[si][sj];
  int other = OTHER_COLOR(color);
  int acode;
  int dcode;
  int savecode = 0;
  int savei = -1, savej = -1;
  int newlibi[2], newlibj[2];
  int xi, xj;
  int yi, yj;
  int ai, aj;
  int bi, bj;
  int k;

  assert(countlib(si, sj) == 2);

  for (k = 0; k < 2; ++k) {
    ai = libi[k];
    aj = libj[k];
    bi = libi[1-k];
    bj = libj[1-k];

    if (ai == bi && (aj == bj+1 || aj == bj-1)) {
      xj = aj;
      if (ai < board_size-1 && p[ai+1][aj] == EMPTY)
	xi = ai+1;
      else if (ai > 0 && p[ai-1][aj] == EMPTY)
	xi = ai-1;
      else
	continue;
    }
    else if (aj == bj && (ai == bi+1 || ai == bi-1)) {
      xi = ai;
      if (aj < board_size-1 && p[ai][aj+1] == EMPTY)
	xj = aj+1;
      else if (aj > 0 && p[ai][aj-1] == EMPTY)
	xj = aj-1;
      else
	continue;
    }
    else
      return 0; /* Incorrect configuration, give up. */
      
    if (is_self_atari(xi, xj, other)
	|| !trymove(xi, xj, other, "special_attack3-A", si, sj))
      continue;
    
    if (countlib(xi, xj) == 2) {
      findlib(xi, xj, 2, newlibi, newlibj);
      if (newlibi[0] == ai && newlibj[0] == aj) {
	yi = newlibi[1];
	yj = newlibj[1];
      }
      else {
	yi = newlibi[0];
	yj = newlibj[0];
      }
      if (!is_self_atari(yi, yj, color)
	  && trymove(yi, yj, color, "special_attack3-B", si, sj)) {
	acode = do_attack(si, sj, NULL, NULL, komaster);
	if (acode == 0) {
	  popgo();
	  popgo();
	  continue;
	}
	UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej,
					  acode, xi, xj);
	popgo();
      }
    }
    
    dcode = do_find_defense(si, sj, NULL, NULL, komaster);
    if (dcode != 1 && do_attack(si, sj, NULL, NULL, komaster)) {
      if (dcode == 0) {
	popgo();
	if (ti) *ti = xi;
	if (tj) *tj = xj;
	return 1;
      }
      UPDATE_SAVED_KO_RESULT(savecode, savei, savej, dcode, xi, xj);
    }
    popgo();
  }

  if (savecode) {
    if (ti) *ti = savei;
    if (tj) *tj = savej;
    return savecode;
  }

  return 0;
}


/* In situations like these:
 *
 * ...O.O...   ...O.O...
 * XXXXOOXXX   XXXXOOXXX
 * XOOOXXO*.   Xsssbbcd.
 * .X.O.....   .X.sa.e..
 * ---------   ---------
 *
 * the code that follows can find the attacking move at *.
 */

static int
special_attack4(int si, int sj, int libi[2], int libj[2],
		int *ti, int *tj, int komaster)
{
  int color = p[si][sj];
  int other = OTHER_COLOR(color);
  int dcode;
  int savecode = 0;
  int savei = -1, savej = -1;
  int adj, adji[MAXCHAIN], adjj[MAXCHAIN];
  int adj2, adj2i[MAXCHAIN], adj2j[MAXCHAIN];
  int lib2i[2], lib2j[2];
  int ai, aj;
  int bi = -1, bj = -1;
  int ci, cj;
  int di, dj;
  int ei, ej;
  int k, s, t;

  assert(countlib(si, sj) == 2);

  /* To avoid making this too general, we require that both
   * liberties are self ataris for X.
   */
  if (!is_self_atari(libi[0], libj[0], other) 
      || !is_self_atari(libi[1], libj[1], other))
    return 0;

  /* Pick up chain links with 2 liberties. */
  chainlinks2(si, sj, &adj, adji, adjj, 2);
  
  for (k = 0; k < 2; ++k) {
    ai = libi[k];
    aj = libj[k];

    /* Check that (ai, aj) also is a liberty of one of the two liberty
     * chain links.
     */
    for (s=0; s<adj; s++)
      if (liberty_of(ai, aj, adji[s], adjj[s])) {
	bi = adji[s];
	bj = adjj[s];
	break;
      }

    /* Nothing found. */
    if (s == adj)
      continue;

    /* Now require that (bi, bj) has a chain link, different from (si,
     * sj), also with two liberties.
     */
    chainlinks2(bi, bj, &adj2, adj2i, adj2j, 2);

    for (s=0; s<adj2; s++) {
      find_origin(adj2i[s], adj2j[s], &ci, &cj);
      if (ci == si && cj == sj)
	continue;
      
      /* Pick up the liberties of (ci, cj). */
      findlib(ci, cj, 2, lib2i, lib2j);

      /* Try playing at a liberty. Before doing this, verify that (ci,
       * cj) cannot get more than two liberties by answering on the
       * other liberty and that we are not putting ourselves in atari.
       */
      for (t=0; t<2; t++) {
	di = lib2i[t];
	dj = lib2j[t];
	ei = lib2i[1-t];
	ej = lib2j[1-t];
	if (is_self_atari(di, dj, other))
	  break;
	if (approxlib(ei, ej, color, 4, NULL, NULL) > 3)
	  break;

	if (trymove(di, dj, other, "special_attack4", si, sj)) {
	  dcode = do_find_defense(si, sj, NULL, NULL, komaster);
	  if (dcode != 1 && do_attack(si, sj, NULL, NULL, komaster)) {
	    if (dcode == 0) {
	      popgo();
	      if (ti) *ti = di;
	      if (tj) *tj = dj;
	      return 1;
	    }
	    UPDATE_SAVED_KO_RESULT(savecode, savei, savej, dcode, di, dj);
	  }
	  popgo();
	}
      }
    }
  }

  if (savecode) {
    if (ti) *ti = savei;
    if (tj) *tj = savej;
    return savecode;
  }

  return 0;
}


/* 
 * If (si, sj) points to a string, draw_back(si, sj, &ti, &tj, komaster)
 * looks for a move in the following configuration which attacks
 * the string:
 *
 *      X*            X=attacker, O=defender
 *      O.
 *
 * In the initial implementation we consider cases 
 * where X has exactly 2 liberties. 
 *
 */

static int
draw_back(int si, int sj, int *ti, int *tj, int komaster)
{
  int r, k;
  int adj, adji[MAXCHAIN], adjj[MAXCHAIN];
  int xi[2], xj[2];
  int savei = -1, savej = -1;
  int savecode = 0;

  chainlinks2(si, sj, &adj, adji, adjj, 2);
  for (r=0; r<adj; r++) {
    findlib(adji[r], adjj[r], 2, xi, xj);
    for (k=0; k<2; k++) {
      if (!liberty_of(xi[k], xj[k], si, sj) &&
	  ((   xi[k]>0            && liberty_of(xi[k]-1, xj[k],   si, sj))
	   || (xi[k]<board_size-1 && liberty_of(xi[k]+1, xj[k],   si, sj))
	   || (xj[k]>0            && liberty_of(xi[k]  , xj[k]-1, si, sj))
	   || (xj[k]<board_size-1 && liberty_of(xi[k]  , xj[k]+1, si, sj)))) {
	if (trymove(xi[k], xj[k], OTHER_COLOR(p[si][sj]),
		    "draw_back", si, sj)) {
	  int dcode = do_find_defense(si, sj, NULL, NULL, komaster);
	  if (dcode != 1 && do_attack(si, sj, NULL, NULL, komaster)) {
	    if (dcode == 0) {
	      popgo();
	      if (ti) *ti = xi[k];
	      if (tj) *tj = xj[k];
	      return 1;
	    }
	    UPDATE_SAVED_KO_RESULT(savecode, savei, savej,
				   dcode, xi[k], xj[k]);
	  }
	  popgo();
	}
      }
    }
  }

  if (savecode) {
    if (ti) *ti = savei;
    if (tj) *tj = savej;
    return savecode;
  }

  return 0;
}

/* ================================================================ */
/*            Defending by attacking surrounding strings            */
/* ================================================================ */

/* Add the chainbreaking moves relative to the string (si, sj) to the
 * (movei[], movej[]) arrays.
 */
static void
break_chain_moves(int si, int sj, int movei[MAX_MOVES], int movej[MAX_MOVES],
		  int *moves)
{
  int r;
  int xi, xj;
  int adj, adji[MAXCHAIN], adjj[MAXCHAIN];
  
  /* Find links in atari. */
  chainlinks2(si, sj, &adj, adji, adjj, 1);
  
  for (r=0; r<adj; r++) {
    findlib(adji[r], adjj[r], 1, &xi, &xj);
    ADD_CANDIDATE_MOVE(xi, xj, movei, movej, *moves);
  }
}


/*
 * Find moves which immediately capture chain links with 2
 * liberties, in the sense that the links cannot escape atari.
 *
 * The used heuristics are slightly sloppy, so useless moves may
 * appear occasionally. This should, however, only lead to slightly
 * worse performance but not to incorrect results.
 */
static void
break_chain2_efficient_moves(int si, int sj,
			     int movei[MAX_MOVES], int movej[MAX_MOVES],
			     int *moves)
{
  int r;
  int adj, adji[MAXCHAIN], adjj[MAXCHAIN];
  int adj2, adj2i[MAXCHAIN], adj2j[MAXCHAIN];
  int libi[2], libj[2];
  
  /* Find links with 2 liberties. */
  chainlinks2(si, sj, &adj, adji, adjj, 2);
  
  for (r=0; r<adj; r++) {
    chainlinks2(si, sj, &adj2, adj2i, adj2j, 1);
    if (adj2 > 0)
      continue;
    findlib(adji[r], adjj[r], 2, libi, libj);
    if (approxlib(libi[0], libj[0], p[adji[r]][adjj[r]], 3, NULL, NULL) <= 2)
      ADD_CANDIDATE_MOVE(libi[1], libj[1], movei, movej, *moves);
    if (approxlib(libi[1], libj[1], p[adji[r]][adjj[r]], 3, NULL, NULL) <= 2)
      ADD_CANDIDATE_MOVE(libi[0], libj[0], movei, movej, *moves);
  }
}

static void
break_chain2_moves(int si, int sj, int movei[MAX_MOVES], int movej[MAX_MOVES],
		   int *moves, int require_safe)
{
  int r;
  int k;
  int ai, aj;
  int adj;
  int adji[MAXCHAIN], adjj[MAXCHAIN];
  int libi[2], libj[2];
  int color = p[si][sj];
  int other = OTHER_COLOR(color);

  chainlinks2(si, sj, &adj, adji, adjj, 2);
  
  for (r=0; r<adj; r++) {
    ai = adji[r];
    aj = adjj[r];

    findlib(ai, aj, 2, libi, libj);
    for (k = 0; k < 2; ++k) {
      int unsafe = is_self_atari(libi[k], libj[k], color);
      if (!unsafe
	  || (!require_safe
	      && approxlib(libi[k], libj[k], other, 5, NULL, NULL) < 5))
	ADD_CANDIDATE_MOVE(libi[k], libj[k], movei, movej, *moves);
    }

    if (stackp <= backfill2_depth)
      break_chain_moves(ai, aj, movei, movej, moves);
  }
}

/*
 * (si,sj) points to a group. break_chain2(si, sj, *i, *j)
 * returns 1 if there is a string in the surrounding chain having
 * exactly two liberties whose attack leads to the rescue of
 * (si, sj). Then (*i, *j) points to the location of the attacking move.
 * 
 * Returns 2 if the saving move depends on ignoring a ko threat;
 * 
 * Returns 3 if the saving move requires making a ko threat and winning
 *   the ko.
 */

static int 
break_chain2(int si, int sj, int *i, int *j, int komaster)
{
  int color, other;
  int v;
  int savei = -1, savej = -1;
  int libi[2], libj[2];
  int liberties;
  int ti[MAXCHAIN], tj[MAXCHAIN];
  int moves = 0;
  int mw[MAX_BOARD][MAX_BOARD];
  int savecode = 0;

  SETUP_TRACE_INFO("break_chain2", si, sj);
  
  memset(mw, 0, sizeof(mw));

  RTRACE("in break_chain2 at %m\n", si, sj);
  color = p[si][sj];
  other = OTHER_COLOR(color);
  
  break_chain2_moves(si, sj, ti, tj, &moves, 0);
  order_moves(si, sj, moves, ti, tj, color);

  /* We do not wish to consider the move if it can be 
   * immediately recaptured, unless stackp <= backfill_depth.
   *
   * With the incremental board code it's not much point in using
   * approxlib() as heuristic whether we want to play the move. Better
   * and more accurate to really play the move and then use findlib() to
   * see how many liberties we obtained.
   */

  for (v=0; v<moves; v++) {
    if (!trymove(ti[v], tj[v], color, "break_chain2-A", si, sj)) {
      /* If trymove failed, check whether it was an illegal ko capture. */
      if (stackp <= ko_depth
	  && komaster != other
	  && savecode == 0
	  && is_ko(ti[v], tj[v], color) 
	  && tryko(ti[v], tj[v], color, "break_chain2-B")) {
	if (do_attack(si, sj, NULL, NULL, color) != 1) {
	  savecode = 3;
	  savei = ti[v];
	  savej = tj[v];
	}
	popgo();
      }
      continue;
    }
    
    liberties = findlib(ti[v], tj[v], 2, libi, libj);
    if (liberties > 1) {
      int acode = do_attack(si, sj, i, j, komaster);
      if (acode == 0) {
	if (i) *i = ti[v];
	if (j) *j = tj[v];
	popgo();
	SGFTRACE(ti[v], tj[v], 1, "attack defended-A");
	return 1;
      }
      UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, ti[v], tj[v]);
    }
    else if (stackp <= backfill_depth) {
      int bi = libi[0];
      int bj = libj[0];
      int try_harder = 0;

      assert(liberties == 1);

      if (trymove(bi, bj, other, "break_chain2-C", si, sj)) {
	if (countlib(bi, bj) == 1)
	  try_harder = 1;
	if (p[si][sj] != EMPTY) {
	  int dcode = do_find_defense(si, sj, NULL, NULL, komaster);
	  if (dcode == 1 && !try_harder) {
	    if (i) *i = ti[v];
	    if (j) *j = tj[v];
	    popgo();
	    popgo();
	    SGFTRACE(ti[v], tj[v], 1, "attack defended-B");
	    return 1;
	  }
	  /* FIXME PRE3.0: Possibly the ko result codes are not handled
           * correctly in the presence of two trymove().
	   */
	  UPDATE_SAVED_KO_RESULT_UNREVERSED(savecode, savei, savej,
					      dcode, ti[v], tj[v]);
	}
	popgo();
      }
      else {
	try_harder = 1;
	if (savecode == 0
	    && depth <= ko_depth
	    && komaster != color
	    && is_ko(ti[v], tj[v], other)
	    && tryko(ti[v], tj[v], other, "break_chain2-D")) {
	  assert(p[si][sj] != EMPTY);
	  if (do_find_defense(si, sj, NULL, NULL, other) != 0) {
	    savecode = 3;
	    savei = ti[v];
	    savej = tj[v];
	  }
	  popgo();
	}
      }
      if (try_harder) {
	int acode = do_attack(si, sj, i, j, komaster);
	if (acode == 0) {
	  if (i) *i = ti[v];
	  if (j) *j = tj[v];
	  popgo();
	  SGFTRACE(ti[v], tj[v], 1, "attack defended-C");
	  return 1;
	}
	UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, ti[v], tj[v]);
      }
    }
    popgo(); /* (ti[v], tj[v]) */
  }

  if (savecode != 0) {
    if (i) *i = savei;
    if (j) *j = savej;
    SGFTRACE(savei, savej, savecode, "saved move");
    return savecode;
  }

  SGFTRACE(-1, -1, 0, NULL);
  return 0;
}


/*
 * (si,sj) points to a group. break_chain3(si, sj, *i, *j)
 * returns 1 if there is a string in the surrounding chain having
 * exactly three liberties whose attack leads to the rescue of
 * (si, sj). Then (*i, *j) points to the location of the attacking move.
 * 
 * Returns 2 if the saving move depends on ignoring a ko threat;
 * 
 * Returns 3 if the saving move requires making a ko threat and winning
 *   the ko.
 */

static int 
break_chain3(int si, int sj, int *i, int *j, int komaster)
{
  int color, other;
  int r;
  int k;
  int u=0, v;
  int ai, aj;
  int savei = -1, savej = -1;
  int adj;
  int adji[MAXCHAIN], adjj[MAXCHAIN];
  int libi[3], libj[3];
  int ti[MAXCHAIN], tj[MAXCHAIN];
  int mw[MAX_BOARD][MAX_BOARD];
  int savecode = 0;
  int liberties = countlib(si, sj);

  SETUP_TRACE_INFO("break_chain3", si, sj);

  memset(mw, 0, sizeof(mw));
  
  RTRACE("in break_chain3 at %m\n", si, sj);
  color = p[si][sj];
  other = OTHER_COLOR(color);
  chainlinks2(si, sj, &adj, adji, adjj, 3);
  for (r=0;r<adj;r++) {
    ai = adji[r];
    aj = adjj[r];

    /* We make a list in the (adji, adjj) array of the liberties
     * of boundary strings having exactly three liberties. We mark
     * each liberty in the mw array so that we do not list any
     * more than once.
     */
    findlib(ai, aj, 3, libi, libj);
    for (k = 0; k < 3; ++k) {
      if (!mw[libi[k]][libj[k]]) {
	mw[libi[k]][libj[k]] = 1;
	ti[u] = libi[k];
	tj[u] = libj[k];
	u++;
      }
    }
  }
  
  /* We do not wish to consider the move if it can be 
   * immediately recaptured, unless stackp <= backfill2_depth.
   */

  for (v=0; v<u; v++) {
    if (!trymove(ti[v], tj[v], color, "break_chain3-A", si, sj))
      continue;

    if (countlib(ti[v], tj[v]) == 1 && stackp > backfill2_depth) {
      popgo();
      continue;
    }
    
    /* If we just filled our own liberty we back out now */
    if (countlib(si, sj) >= liberties) {
      int acode = do_attack(si, sj, i, j, komaster);
      if (acode == 0) {
	if (i) *i = ti[v];
	if (j) *j = tj[v];
	popgo();
	SGFTRACE(ti[v], tj[v], 1, "attack defended");
	return 1;
      }
      UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, ti[v], tj[v]);
    }
    popgo(); /* (ti[v], tj[v]) */
  }

  if (savecode != 0) {
    if (i) *i = savei;
    if (j) *j = savej;
    SGFTRACE(savei, savej, savecode, "saved move");
    return savecode;
  }

  SGFTRACE(-1, -1, 0, NULL);
  return 0;
}

/* This function looks for moves attacking those components
 * of the surrounding chain of the superstring (see find_superstring
 * for the definition which have fewer than liberty_cap liberties,
 * and which are not adjacent to the string itself, since those
 * are tested by break_chain. If such a boundary chain can be
 * attacked, and if attacking the boundary results in saving
 * the (si,sj) string, then success is reported.
 */
/* FIXME: Consider ko captures */
int
superstring_breakchain(int si, int sj, int *i, int *j, int liberty_cap)
{
  int adj;
  int adji[MAXCHAIN], adjj[MAXCHAIN];
  int k, ai, aj;
  int color = p[si][sj];
  int savei = -1;
  int savej = -1;
  int savecode = 0;

  SETUP_TRACE_INFO("superstring_breakchain", si, sj);

  proper_superstring_chainlinks(si, sj, &adj, adji, adjj, liberty_cap);
  for (k = 0; k < adj; k++) {
    if (countlib(adji[k], adjj[k]) > 1)
      continue;
    findlib(adji[k], adjj[k], 1, &ai, &aj);
    if (trymove(ai, aj, color, "superstring_break_chain", si, sj)) {
      int acode = do_attack(si, sj, NULL, NULL, EMPTY);
      
      if (acode == 0) {
	popgo();
	if (i) *i = ai;
	if (j) *j = aj;
	SGFTRACE(ai, aj, 1, "attack defended");
	return 1;
      }
      else if (acode != 1) {
	UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, 
			       ai, aj);
      }
      popgo();
    }
  }
  
  if (savecode) {
    if (i) *i = savei;
    if (j) *j = savej;
    SGFTRACE(savei, savej, savecode, "saved move");
    return savecode;
  }

  SGFTRACE(-1, -1, 0, NULL);
  return 0;
}

/*
 * If si, sj points to a group, double_atari_chain2() adds all moves
 * which make a double atari on some strings in the surrounding chain
 * to the (movei[], movej[]) arrays.
 */

static void
double_atari_chain2(int si, int sj,
		    int movei[MAX_MOVES], int movej[MAX_MOVES], int *moves)
{
  int r, k;
  int ai, aj;
  int adj;
  int adji[MAXCHAIN], adjj[MAXCHAIN];
  int libi[2], libj[2];
  int mw[MAX_BOARD][MAX_BOARD];

  memset(mw, 0, sizeof(mw));

  chainlinks2(si, sj, &adj, adji, adjj, 2);
  for (r=0; r<adj; r++) {
    ai = adji[r];
    aj = adjj[r];
    findlib(ai, aj, 2, libi, libj);
    for (k = 0; k < 2; ++k) {
      mw[libi[k]][libj[k]]++;
      if (mw[libi[k]][libj[k]] == 2) {
	/* found a double atari */
	ADD_CANDIDATE_MOVE(libi[k], libj[k], movei, movej, *moves);
      }
    }
  }
}


/* 
 * relative_break_chain(si, sj, *i, *j, ti, tj) is a variant of
 * break_chain. The strings (si, sj) and (ti, tj) are of
 * opposite color, and (ti, tj) is under attack. This function
 * looks for a boundary string to (si, sj) which is in atari,
 * and asks whether capturing it will result in the capture
 * of (ti, tj).
 *
 * Returns 2 if the saving move depends on ignoring a ko threat;
 * 
 * Returns 3 if the saving move requires making a ko threat and winning
 *   the ko.
 */

static int
relative_break_chain(int si, int sj, int *i, int *j, int ti, int tj,
		     int komaster)
{
  int color = p[si][sj];
  int other = OTHER_COLOR(color);
  int r;
  int ai, aj;
  int xi, xj;
  int adj, adji[MAXCHAIN], adjj[MAXCHAIN];
  int savecode = 0;
  int savei = -1, savej = -1;
  
  SETUP_TRACE_INFO("relative_break_chain", si, sj);
  
  if (stackp > depth) {
    SGFTRACE(-1, -1, 0, "stackp > depth");
    return 0;
  }

  RTRACE("in relative_break_chain at %m (%m)\n", si, sj, ti, tj);
  chainlinks2(si, sj, &adj, adji, adjj, 1);

  for (r=0; r<adj; r++) {
    ai = adji[r];
    aj = adjj[r];
    findlib(ai, aj, 1, &xi, &xj);
    if (trymove(xi, xj, color, "relative_break_chain-A", si, sj)) {
      int dcode = do_find_defense(ti, tj, NULL, NULL, komaster);
      if (dcode != 1 && do_attack(ti, tj, NULL, NULL, komaster) != 0) {
	if (dcode == 0) {
	  RTRACE("%m found to attack %m by defending %m\n", 
		 xi, xj, ti, tj, si, sj);
	  if (i) *i = xi;
	  if (j) *j = xj;
	  popgo();
	  SGFTRACE(xi, xj, 1, "attack defended");
	  return 1;
	}
	UPDATE_SAVED_KO_RESULT(savecode, savei, savej, dcode, xi, xj);
      }
      popgo();
    }
    else if (savecode == 0
	     && komaster != other
	     && stackp <= ko_depth
	     && is_ko(xi, xj, color)
	     && tryko(xi, xj, color, "relative_break_chain-B")) {
      if (do_find_defense(ti, tj, NULL, NULL, color) != 1
	  && do_attack(ti, tj, NULL, NULL, color) != 0) {
	savecode = 3;
	savei = xi;
	savej = xj;
      }
      popgo();
    }
  }

  if (savecode != 0) {
    if (i) *i = savei;
    if (j) *j = savej;
    SGFTRACE(savei, savej, savecode, "saved move");
    return savecode;
  }
  
  SGFTRACE(-1, -1, 0, NULL);
  return 0;
}


/* ================================================================ */
/*                Restricted Attack and Defense                     */
/* ================================================================ */

/* These functions try to attack and defend a string, avoiding moves
 * from a certain set. It is assumed that as soon as the string gets
 * three liberties, it is alive.
 * 
 * These functions can be used to generate backfilling moves as
 * follows: Suppose that we would like to make atari on a
 * string, but the atari is not safe until we make a backfilling
 * move. To find the backfilling move, we make a list of the
 * liberties of the string under attack, declaring these moves
 * forbidden. Neither player will play them while the restricted
 * functions are in effect. We fill the liberty, creating a
 * string which is under attack, and look for a defensive move
 * which avoids the forbidden moves. This is the backfilling
 * move.
 *
 * These are minimalist functions capable of reading a ladder
 * and not much more.
 */

/* Given a list of moves, restricted_defend1 tries to find a 
 * move that defends the string (si, sj) with one liberty,
 * not considering moves from the list.
 */
static int
restricted_defend1(int si, int sj, int *i, int *j, int komaster,
		   int forbidden_moves, 
		   int *forbidden_movei, int *forbidden_movej)
{
  int color, other;
  int xi, xj;
  int libi, libj;
  int movei[MAX_MOVES], movej[MAX_MOVES];
  int moves = 0;
  int savei = -1, savej = -1;
  int savecode = 0;
  int liberties;
  int k;

  SETUP_TRACE_INFO("restricted_defend1", si, sj);
  reading_node_counter++;
  
  assert(p[si][sj] != EMPTY);
  ASSERT(countlib(si, sj) == 1, si, sj);

  RTRACE("try to escape atari on %m.\n",  si, sj);

  color = p[si][sj];
  other = OTHER_COLOR(color);

  /* (libi, libj) will be the liberty of the string. */
  liberties = findlib(si, sj, 1, &libi, &libj);
  ASSERT(liberties == 1, si, sj);

  /* Collect moves to try in the first batch.
   * 1. First order liberty.
   * 2. Chain breaking moves.
   */
  movei[0] = libi;
  movej[0] = libj;
  moves = 1;
  
  break_chain_moves(si, sj, movei, movej, &moves);
  order_moves(si, sj, moves, movei, movej, color);
  for (k = 0; k < moves; k++) {
    xi = movei[k];
    xj = movej[k];
    if (in_list(xi, xj, forbidden_moves, forbidden_movei, forbidden_movej))
      continue;
    /* To avoid loops with double ko, we do not allow any ko captures,
     * even legal ones, if the opponent is komaster.
     */
    if ((komaster != other || !is_ko(xi, xj, color))
	&& trymove(xi, xj, color, "defend1-A", si, sj)) {
      int libs = countlib(si, sj);
      if (libs > 2) {
	popgo();
	SGFTRACE(xi, xj, 1, "defense effective");
	if (i) *i = xi;
	if (j) *j = xj;
	return 1;
      }
      if (libs == 2) {
	int acode = restricted_attack2(si, sj, NULL, NULL, komaster,
				       forbidden_moves,
				       forbidden_movei, forbidden_movej);
	popgo();
	if (acode == 0) {
	  SGFTRACE(xi, xj, 1, "defense effective");
	  if (i) *i = xi;
	  if (j) *j = xj;
	  return 1;
	}
	/* if the move works with ko we save it, then look for something
	 * better.
	 */
	UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, xi, xj);
      }
      else popgo();
    }
    else {
      if (stackp <= ko_depth
	  && savecode == 0 
	  && is_ko(xi, xj, color)
	  && komaster != other
	  && tryko(xi, xj, color, "restricted_defend1-B")) {
	int libs = countlib(si, sj);
	if (libs > 2) {
	  popgo();
	  UPDATE_SAVED_KO_RESULT(savecode, savei, savej, 2, xi, xj);
	}
	else if (libs == 2) {
	  int acode = restricted_attack2(si, sj, NULL, NULL, color,
					 forbidden_moves,
					 forbidden_movei, forbidden_movej);
	  popgo();
	  UPDATE_SAVED_KO_RESULT(savecode, savei, savej, acode, xi, xj);
	}
	else popgo();
      }
    }
  }
  if (savecode != 0) {
    SGFTRACE(savei, savej, savecode, "saved move");
    if (i) *i = savei;
    if (j) *j = savej;
    return savecode;
  }
  SGFTRACE(-1, -1, 0, NULL);
  return 0;
}


/* Given a list of moves, restricted_attack2 tries to find a 
 * move that attacks the string (si, sj) with two liberties,
 * not considering moves from the list.
 */
static int
restricted_attack2(int si, int sj, int *i, int *j, int komaster,
		   int forbidden_moves, 
		   int *forbidden_movei, int *forbidden_movej)
{
  int color, other;
  int ai, aj;
  int liberties;
  int libi[2], libj[2];
  int savei = -1, savej = -1;
  int savecode = 0;
  int k;

  SETUP_TRACE_INFO("restricted_attack2", si, sj);
  reading_node_counter++;

  find_origin(si, sj, &si, &sj);
  assert(p[si][sj] != EMPTY);
  assert(countlib(si, sj) == 2);

  RTRACE("restricted attack on %m with 2 liberties\n", si, sj);

  color = p[si][sj];
  other = OTHER_COLOR(color);

  /* The attack may fail if a boundary string is in atari and cannot 
   * be defended.  First we must try defending such a string. 
   */
  /* Get the two liberties of (si, sj). */
  liberties = findlib(si, sj, 2, libi, libj);
  ASSERT(liberties == 2, si, sj);

  for (k = 0; k < 2; k++) {
    ai = libi[k];
    aj = libj[k];
    if (in_list(ai, aj, forbidden_moves, forbidden_movei, forbidden_movej))
      continue;
    /* To avoid loops with double ko, we do not allow any ko captures,
     * even legal ones, if the opponent is komaster. 
     */
    if ((komaster != color || !is_ko(ai, aj, other))
	&& trymove(ai, aj, other, "restricted_attack2", si, sj)) {
      if (!restricted_defend1(si, sj, NULL, NULL, komaster,
			      forbidden_moves,
			      forbidden_movei, forbidden_movej)) {
	popgo();
	SGFTRACE(ai, aj, 1, "attack effective");
	if (i) *i = ai;
	if (j) *j = aj;
	return 1;
      }
      popgo();
    }
    else if (savecode == 0
	     && komaster != color 
	     && tryko(ai, aj, other, "restricted_attack2")) {
      if (!restricted_defend1(si, sj, NULL, NULL, other,
			      forbidden_moves,
			      forbidden_movei, forbidden_movej)) {
	popgo();
	savecode = 3;
	savei = ai;
	savej = aj;
      }
      else popgo();
    }
  }
  if (savecode) {
    SGFTRACE(savei, savej, savecode, "saved move");
    if (i) *i = savei;
    if (j) *j = savej;
    return savecode;
  }
  SGFTRACE(-1, -1, 0, NULL);
  return 0;
}

/* Returns true if the move at (m, n) is in a given list of moves */

static int
in_list(int m, int n, int moves, int *movei, int *movej)
{
  int k;

  for (k = 0; k < moves; k++)
    if (movei[k] == m && movej[k] == n)
      return 1;
  return 0;
}


/* ================================================================ */
/*                          Move ordering                           */
/* ================================================================ */

/* Parameters used in the ordering of moves to try in the tactical
 * reading.
 */

/*                                        0   1   2   3   4  >4  */
static int defend_lib_score[6]        = {-5, -4,  0,  2,  5, 20};
static int defend_capture_score[6]    = { 0,  6,  9, 13, 18, 24};
static int defend_open_score[5]       = { 0,  1,  2,  3,  4};
static int attack_own_lib_score[5]    = {10, -4,  2,  3,  4};
static int attack_string_lib_score[6] = { 0,  8, 12, 16, 20, 25};
static int attack_capture_score[6]    = {-4,  4, 10, 15, 20, 25};
static int attack_save_score[6]       = { 0,  8, 15, 18, 21, 24};
static int attack_open_score[5]       = { 0,  1,  2,  3,  4};
static int defend_not_edge_score      = 5;
static int attack_not_edge_score      = 1;


/* The string at (si, sj) is under attack. The num_moves moves in
 * (movei[], movej[]) for color color have been deemed interesting in
 * the attack or defense of the group. Most of these moves will be
 * immediate liberties of the group.
 *
 * This function orders the moves in the order where the move most
 * likely to succeed to attack or defend the string will be first and
 * so on.
 *
 * Currently, this is defined as:
 * 1) Moves which let the defending string get more liberties are more
 *    interesting.
 * 2) Moves adjacent to the most open liberties are more 
 *    interesting than those with fewer open liberties.
 * 3) Moves on the edge are less interesting.
 */

#define NO_UNROLL 0

static void
order_moves(int si, int sj, int num_moves, int *movei, int *movej, int color)
{
  int  origini, originj;
  int  scores[MAX_BOARD * MAX_BOARD];
  int  move;
  int  i, j;
  int  maxscore;
  int  max_at;
  int  string_color = p[si][sj];
#if NO_UNROLL == 1
  int  k;
#endif

  find_origin(si, sj, &origini, &originj);

  /* Each move will have a score. First, we clear the score array. */
  memset(scores, 0, num_moves * sizeof(int));

  /* Assign a score to each move. */
  for (move = 0; move < num_moves; ++move) {
    int mi = movei[move];
    int mj = movej[move];

    /* Look at the neighbors of this move and count the things we
     * find. Friendly and opponent stones are related to color, i.e.
     * the player to move, not to the color of the string.
     *
     * If this piece of code should turn out to be a bottleneck it can
     * be speeded up by loop unrolling and/or moving it into the
     * incremental board code.
     */
    int number_edges       = 0; /* outside board */
    int number_same_string = 0; /* the string being attacked */
    int number_own         = 0; /* friendly stone */
    int number_opponent    = 0; /* opponent stone */
    int captured_stones    = 0; /* number of stones captured by this move */
    int saved_stones       = 0; /* number of stones in atari saved */
    int number_open        = 0; /* empty intersection */
#if NO_UNROLL == 1
    for (k=0; k<4; k++) {
      i = mi + deltai[k];
      j = mj + deltaj[k];
      if (!ON_BOARD(i, j))
	number_edges++;
      else if (p[i][j] == EMPTY)
	number_open++;
      else {
	if (p[i][j] == string_color) {
	  int ai, aj;
	  find_origin(i, j, &ai, &aj);
	  if (ai == origini && aj == originj)
	    number_same_string++;
	}
	
	if (p[i][j] == color) {
	  number_own++;
	  if (countlib(i, j) == 1)
	    saved_stones += countstones(i, j);
	}
	else {
	  number_opponent++;
	  if (countlib(i, j) == 1)
	    captured_stones += countstones(i, j);
	}
      }
    }

#else
#define code1(argi,argj) \
      if (p[argi][argj] == EMPTY) \
	number_open++; \
      else { \
	if (p[argi][argj] == string_color) { \
	  int ai, aj; \
	  find_origin(argi, argj, &ai, &aj); \
	  if (ai == origini && aj == originj) \
	    number_same_string++; \
	} \
	if (p[argi][argj] == color) { \
	  number_own++; \
	  if (countlib(argi, argj) == 1) \
	    saved_stones += countstones(argi, argj); \
	} \
	else { \
	  number_opponent++; \
	  if (countlib(argi, argj) == 1) \
	    captured_stones += countstones(argi, argj); \
	} \
      }

      if (mi > 0) {
        code1(mi-1, mj);
      } else {
	number_edges++;
      }
      if (mi < board_size-1) {
        code1(mi+1, mj);
      } else {
	number_edges++;
      }
      if (mj > 0) {
        code1(mi, mj-1);
      } else {
	number_edges++;
      }
      if (mj < board_size-1) {
        code1(mi, mj+1);
      } else {
	number_edges++;
      }
#endif

    /* Different score strategies depending on whether the move is
     * attacking or defending the string.
     */
    if (color == string_color) {
      /* Defense move.
       *
       * 1) Add twice the number of liberties the group receives by
       *    extending to the intersection of the move, if more than one.
       *    Only applicable if the move is adjacent to the group.
       */
      
      if (number_same_string > 0) {
	int liberties = approxlib(mi, mj, string_color, 4, NULL, NULL);
	if (liberties > 5 || (liberties == 4 && stackp > fourlib_depth))
	  liberties = 5;
	scores[move] += defend_lib_score[liberties];
      }
      
      /* 2) Add the number of open liberties near the move to its score. */
      assert(number_open <= 4);
      scores[move] += defend_open_score[number_open];
      
      /* 3) Add 5 if the move is not on the edge. 
       */
      if (number_edges == 0)
	scores[move] += defend_not_edge_score;
      
      /* 4) Add thrice the number of captured stones. */
      if (captured_stones <= 5)
	scores[move] += defend_capture_score[captured_stones];
      else
	scores[move] += defend_capture_score[5] + captured_stones;
    }
    else {
      /* Attack move.
       *
       * 1) Add the number of liberties the attacker gets when playing
       *    there, but never more than four.
       */
      int libs = approxlib(mi, mj, color, 4, NULL, NULL);
      if (libs > 4)
	libs = 4;
      scores[move] += attack_own_lib_score[libs];

      /* 2) If the move is not a self atari and adjacent to the
       *    string, add the number of liberties the opponent would gain
       *    by playing there.
       */
      if (libs + captured_stones > 1 && number_same_string > 0) {
	int liberties = approxlib(mi, mj, string_color, 4, NULL, NULL);
	if (liberties > 5 || (liberties == 4 && stackp > fourlib_depth))
	  liberties = 5;
	scores[move] += attack_string_lib_score[liberties];
      }
      
      /* 3) Add the number of open liberties near the move to its score. */
      assert(number_open <= 4);
      scores[move] += attack_open_score[number_open];
      
      /* 4) Add 5 if the move is not on the edge. */
      if (number_edges == 0)
	scores[move] += attack_not_edge_score;
      
      /* 5) Add twice the number of captured stones. */
      if (captured_stones <= 5)
	scores[move] += attack_capture_score[captured_stones];
      else
	scores[move] += attack_capture_score[5];

      /* 6) Add a bonus for saved stones. */
      if (saved_stones <= 5)
	scores[move] += attack_save_score[saved_stones];
      else
	scores[move] += attack_save_score[5];
    }
  }
  
  /* Now sort the moves.  We use selection sort since this array will
   * probably never be more than 10 moves long.  In this case, the
   * overhead imposed by quicksort will probably overshadow the gains
   * given by the O(n*log(n)) behaviour over the O(n^2) behaviour of
   * selection sort.
   */
  for (i = 0; i < num_moves-1; ++i) {

    /* Find the move with the biggest score. */
    maxscore = scores[i];
    max_at = i;
    for (j = i+1; j < num_moves; ++j) {
      if (scores[j] > maxscore) {
	maxscore = scores[j];
	max_at = j;
      }
    }

    /* Now exchange the move at i with the move at max_at.
     * Don't forget to exchange the scores as well.
     */
    if (max_at != i) {
      int tempi = movei[i];
      int tempj = movej[i];
      int tempmax = scores[i];

      movei[i] = movei[max_at];
      movej[i] = movej[max_at];
      scores[i] = scores[max_at];

      movei[max_at] = tempi;
      movej[max_at] = tempj;
      scores[max_at] = tempmax;
    }
  }
}


/* Set new values for the move ordering parameters. */
void
tune_move_ordering(int params[MOVE_ORDERING_PARAMETERS])
{
  int k;
  for (k=0; k<6; k++) {
    defend_lib_score[k]        = params[k];
    defend_capture_score[k]    = params[k + 6];
    if (k < 5) {
      defend_open_score[k]     = params[k + 12];
      attack_own_lib_score[k]  = params[k + 17];
    }
    attack_string_lib_score[k] = params[k + 22];
    attack_capture_score[k]    = params[k + 28];
    attack_save_score[k]       = params[k + 34];
    if (k < 5)
      attack_open_score[k]     = params[k + 40];
  }
  defend_not_edge_score        = params[45];
  attack_not_edge_score        = params[46];
}


/* ================================================================ */
/*                         Reading utilities                        */
/* ================================================================ */


static int safe_move_cache[MAX_BOARD][MAX_BOARD][2];
static int safe_move_cache_when[MAX_BOARD][MAX_BOARD][2];
static void clear_safe_move_cache(void);

static void
clear_safe_move_cache(void)
{
  int i,j;

  for (i=0;i<MAX_BOARD;i++)
    for (j=0;j<MAX_BOARD;j++) {
      safe_move_cache_when[i][j][0] = -1;
      safe_move_cache_when[i][j][1] = -1;
    }
}

/* safe_move(i, j, color) checks whether a move at (i, j) is illegal
 * or can immediately be captured. If stackp==0 the result is cached.
 * If the move only can be captured by a ko, it's considered safe.
 * This may or may not be a good convention.
 *
 * For performance reasons, the result of this function is cached.
 */

int 
safe_move(int i, int j, int color)
{
  int safe = 0;
  static int initialized = 0;

  if (!initialized) {
    clear_safe_move_cache();
    initialized = 1;
  }

  /* If we have this position cached, use the previous value. */
  if (stackp == 0
      && safe_move_cache_when[i][j][color==BLACK] == position_number)
    return safe_move_cache[i][j][color==BLACK];

  /* Otherwise calculate the value... */
  if (trymove(i, j, color, "safe_move-A", -1, -1)) {
    int acode = attack(i, j, NULL, NULL);
    if (acode == 0)
      safe = 1;
    else if (acode == 1)
      safe = 0;
    else if (acode == 2)
      safe = 3;
    else if (acode == 3)
      safe = 2;
    popgo();
  }
#if 1
  else if (is_ko(i, j, color)
	   && tryko(i, j, color, "safe_move-B")) {
    if (do_attack(i, j, NULL, NULL, color) != 1)
      safe = 3;
    else
      safe = 0;
    popgo();
  }
#endif  
  /* ...and store it in the cache.
   * FIXME PRE3.0: Only store result in cache when we're working at full depth.
   */
  if (stackp == 0) {
    if (0)
      gprintf("Safe move at %m for %s cached when depth=%d, position number=%d\n",
	      i, j, color_to_string(color), depth, position_number);
    safe_move_cache_when[i][j][color==BLACK] = position_number;
    safe_move_cache[i][j][color==BLACK] = safe;
  }

  return safe;
}


/* ===================== Statistics  ============================= */


/* Clear statistics. */
void
reset_reading_node_counter()
{
  reading_node_counter = 0;
}


/* Retrieve statistics. */
int
get_reading_node_counter()
{
  return reading_node_counter;
}



/* ================================================================ */
/*              Experimental code, not currently in use.            */
/* ================================================================ */


/* naive_ladder(si, sj, &i, &j) tries to capture a string (si, sj)
 * with exactly two liberties under simplified assumptions, which are
 * adequate in a ladder. The rules are as follows:
 *
 * 1. The attacker is allowed to play at each of the two liberties.
 *    If the move was legal, the string now has exactly one
 *    liberty.
 * 2. Define the last stone of the string to be the stone of the
 *    string adjacent to the last liberty. The defender is allowed to
 *    try the following moves:
 *    - extend the string by playing on the liberty
 *    - try to capture the last stone played by the attacker
 *    - try to capture any stone that was put into atari by the
 *      previous move. It is conjectured that it's sufficient to look
 *      for such stones at a distance two from the last liberty.
 *    We only consider captures that can be done immediately.
 * 3. Depending on the resulting number of liberties of the string, we
 *    value each node as follows:
 *    3 or more liberties: the attack has failed
 *    2 liberties:         recurse
 *    1 liberty:           the attack has succeeded
 *    illegal move for the defender: successful attack
 *    illegal move for the attacker: failed attack
 *
 * Return codes are as usual 0 for failure and 1 for success. If the
 * attack was successful, (*i, *j) contains the attacking move, unless
 * i and j are null pointers.
 *
 * The differences compared to the attack2()/defend1() combination for
 * reading ladders is that this one really always reads them to the
 * very end and that it is faster (because it doesn't call
 * break_chain()). In contrast to attack2() though, this function can
 * only be used for capturing in ladders.
 *
 * FIXME POST3.0: The ladder capture may depend on ko. Need to add the ko
 *        return codes.
 *
 * FIXME PRE3.0: Do we really want these functions?
 */

static int
naive_ladder(int si, int sj, int *i, int *j)
{
  int color, other;
  int ai, aj, bi, bj;
  int acount = 0, bcount = 0;
  int liberties;
  int libi[2], libj[2];
  
  assert(p[si][sj] != EMPTY);
  assert(countlib(si, sj) == 2);
  DEBUG(DEBUG_READING, "naive_ladder(%m)\n", si, sj);

  RTRACE("checking ladder attack on %m with 2 liberties\n", si, sj);

  color = p[si][sj];
  other = OTHER_COLOR(color);

  /* Get the two liberties of (si, sj) into (ai, aj) and (bi, bj). */ 
  liberties = findlib(si, sj, 2, libi, libj);
  ASSERT(liberties==2, si, sj);
  ai = libi[0];
  aj = libj[0];
  bi = libi[1];
  bj = libj[1];

  /* if (bi, bj) looks more promising we wish to switch the two liberties.
   * We check whether (bi, bj) is adjacent to more open liberties than
   * (ai, aj).
   *
   * FIXME POST3.0: This is taken from attack2(). Not sure whether it is good
   * here too.
   */

  if ((ai>0) && p[ai-1][aj]==EMPTY)
    acount++;
  if ((ai<board_size-1) && p[ai+1][aj]==EMPTY)
    acount++;
  if ((aj>0) && p[ai][aj-1]==EMPTY)
    acount++;
  if ((aj<board_size-1) && p[ai][aj+1]==EMPTY)
    acount++;
  if ((bi>0) && p[bi-1][bj]==EMPTY)
    bcount++;
  if ((bi<board_size-1) && p[bi+1][bj]==EMPTY)
    bcount++;
  if ((bj>0) && p[bi][bj-1]==EMPTY)
    bcount++;
  if ((bj<board_size-1) && p[bi][bj+1]==EMPTY)
    bcount++;

  if (bcount>acount) {
    ai = libi[1];
    aj = libj[1];
    bi = libi[0];
    bj = libj[0];
  }

  RTRACE("considering atari at %m\n", ai, aj);
  
  if (trymove(ai, aj, other, "naive_ladder-A", si, sj)) {
    if (!naive_ladder_defense(si, sj, ai, aj, bi, bj, color, other)) {
      popgo();
      if (i) *i = ai;
      if (j) *j = aj;
      return 1;
    }
    popgo();
  }
	  
  if (trymove(bi, bj, other, "naive_ladder-B", si, sj)) {
    if (!naive_ladder_defense(si, sj, bi, bj, ai, aj, color, other)) {
      popgo();
      if (i) *i = bi;
      if (j) *j = bj;
      return 1;
    }
    popgo();
  }

  /* Neither move worked. */
  return 0;
}


/* Try to save the one-liberty string (si, sj) from being caught in a
 * ladder. (ai, aj) is the last played attacking stone and (bi, bj) is
 * the last remaining liberty.
 */

static int
naive_ladder_defense(int si, int sj, int ai, int aj, int bi, int bj,
		     int color, int other) {
  int liberties;
  
  /* Try to capture the just played stone. */
  if (naive_ladder_break_through(si, sj, ai, aj, color, other))
    return 1;

  /* Try to run away by extending on the last liberty. */
  if (trymove(bi, bj, color, "naive_ladder_defense", si, sj)) {
    liberties = countlib(si, sj);
    if (liberties >= 3
	|| (liberties == 2
	    && !naive_ladder(si, sj, NULL, NULL))) {
      popgo();
      return 1;
    }
    popgo();
  }
  
  /* Try to capture a string at distance two (Manhattan metric) from
   * the last liberty.
   */
  if (naive_ladder_break_through(si, sj, bi-2, bj, color, other))
    return 1;

  if (naive_ladder_break_through(si, sj, bi-1, bj-1, color, other))
    return 1;
  
  if (naive_ladder_break_through(si, sj, bi, bj-2, color, other))
    return 1;
  
  if (naive_ladder_break_through(si, sj, bi+1, bj-1, color, other))
    return 1;
  
  if (naive_ladder_break_through(si, sj, bi+2, bj, color, other))
    return 1;
  
  if (naive_ladder_break_through(si, sj, bi+1, bj+1, color, other))
    return 1;
  
  if (naive_ladder_break_through(si, sj, bi, bj+2, color, other))
    return 1;
  
  if (naive_ladder_break_through(si, sj, bi-1, bj+1, color, other))
    return 1;
  
  /* Nothing worked. */
  return 0;
}


/* Try to break out of the ladder by capturing (ai, aj). We must first
 * verify that there is an opponent stone there and that it is in
 * atari so we can capture it immediately. After the capture we count
 * liberties for (si, sj) to see if the ladder is decided yet.
 */
static int
naive_ladder_break_through(int si, int sj, int ai, int aj,
			   int color, int other)
{
  int liberties;
  int bi, bj;
  
  if (ai < 0
      || ai >= board_size
      || aj < 0
      || aj >= board_size)
    return 0;

  if (p[ai][aj] != other)
    return 0;
  
  if (findlib(ai, aj, 1, &bi, &bj) != 1)
    return 0;

  if (trymove(bi, bj, color, "naive_ladder_break_through", si, sj)) {
    liberties = countlib(si, sj);
    if (liberties >= 3
	|| (liberties == 2
	    && !naive_ladder(si, sj, NULL, NULL))) {
      popgo();
      return 1;
    }
    popgo();
  }
  
  return 0;
}




/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
