/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 and 2000 by the Free Software Foundation.      *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "liberty.h"
#include "patterns.h"

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * * * * * * fast pattern matching with DFA  version 2.9 * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define DFA_MATCHER

/* If DFA_SORT is set, all the matched pattern are sorted
 * and checked in the same order as in the old scheme, this allow
 * to check if the two matcher do the same work */
#define DFA_SORT 1 

#include "dfa.c"

/* Forward declaration. */
static void do_dfa_matchpat(dfa_t *pdfa,
		int m, int n, matchpat_callback_fn_ptr callback,
		int color, struct pattern *database,
		void *callback_data, char goal[MAX_BOARD][MAX_BOARD]);

static void do_matchpat(int m, int n, matchpat_callback_fn_ptr callback,
			int color, struct pattern *database,
			void *callback_data, char goal[MAX_BOARD][MAX_BOARD]);

static void check_pattern (int m, int n, matchpat_callback_fn_ptr callback,
			   int color, struct pattern *pattern, int ll,
			   void *callback_data,
			   char goal[MAX_BOARD][MAX_BOARD]);

static void check_pattern_light (int m, int n, 
			   matchpat_callback_fn_ptr callback,
			   int color, struct pattern *pattern, int ll,
			   void *callback_data,
			   char goal[MAX_BOARD][MAX_BOARD]);


#if DFA_SORT
static int compare_int (const void *a, const void *b);
#endif

/* define this to see how each phase of pattern rejection is performing */
/* #define PROFILE_MATCHER */

/* We keep a board hash for the position which was last compiled for
 * hashing, so we can be sure it's up to date when used.
 */

Hash_data compilation_hashdata;


/* In the current implementation, the edge constraints depend on
 * the board size, because we pad width or height out to the
 * board size. (This is because it is easy to find the corners
 * of the rotated pattern, but it is harder to transform the
 * bitmask of edge constraints.)
 *
 * But since version 1.103, board size is variable. Thus we
 * make a first pass through the table once we know the board
 * size.
 *
 * This should be called once for each pattern database.
 */

static void
fixup_patterns_for_board_size(struct pattern *pattern)
{
  int i,j;

  for ( ; pattern->patn; ++pattern )
    if (pattern->edge_constraints != 0) {

      /* If the patterns have been fixed up for a different board size
       * earlier, we need to undo the modifications that were done
       * below before we do them anew. The first time this function is
       * called, this step is effectively a no-op.
       */
      
      if (pattern->edge_constraints & NORTH)
	pattern->maxi = pattern->mini + pattern->height;
	
      if (pattern->edge_constraints & SOUTH)
	pattern->mini = pattern->maxi - pattern->height;
	
      if (pattern->edge_constraints & WEST)
	pattern->maxj = pattern->minj + pattern->width;
      
      if (pattern->edge_constraints & EAST)
	pattern->minj = pattern->maxj - pattern->width;
      
      /* we extend the pattern in the direction opposite the constraint,
       * such that maxi (+ve) - mini (-ve) = board_size-1
       * Note : the pattern may be wider than the board, so
       * we need to be a bit careful !
       */
      
      if (pattern->edge_constraints & NORTH)
	if (pattern->maxi < (board_size-1) + pattern->mini)
	  pattern->maxi = (board_size-1) + pattern->mini;
      
      if (pattern->edge_constraints & SOUTH)
	if (pattern->mini > pattern->maxi - (board_size-1))
	  pattern->mini = pattern->maxi - (board_size-1);
      
      if (pattern->edge_constraints & WEST)
	if (pattern->maxj <  (board_size-1) + pattern->minj)
	  pattern->maxj = (board_size-1) + pattern->minj;
      
      if (pattern->edge_constraints & EAST)
	if (pattern->minj > pattern->maxj - (board_size-1))
	  pattern->minj = pattern->maxj - (board_size-1);
    }


  /* clean up the dfa board */
  for( i = 0; i != DFA_MAX_BOARD*4 ; i++)
    for( j = 0; j != DFA_MAX_BOARD*4 ; j++)
      dfa_p[i][j]= OUT_BOARD;
 
}  


#ifdef PROFILE_MATCHER
static int totals[6];
#endif


/* [i][j] contains merged entries from p[][] around i,j */
static uint32 merged_board[2][MAX_BOARD][MAX_BOARD];

/* This combines the values in p[][] around each point, to allow 
 * rapid rejection of patterns. Must be called once per board position,
 * before matchpat is invoked.
 */

static void 
compile_for_match(void)
{
  int i, j, color;

  hashdata_recalc(&compilation_hashdata, p, ko_i, ko_j);

#ifdef PROFILE_MATCHER
  fprintf(stderr, 
	  "total %d, anchor=%d, grid=%d, edge=%d, matched=%d, accepted=%d\n",
	  totals[0], totals[1], totals[2], totals[3], totals[4], totals[5]);
#endif


#if GRID_OPT > 0
  memset(merged_board, 0, sizeof(merged_board));
  for (color=0; color<2; ++color) {
    for (i=0; i<board_size; ++i) {
      for (j=0; j<board_size; ++j) {
	uint32 this = p[i][j];  /* colour here */
	int ii;
	int shift = 0;
	
	if (!p[i][j])
	  continue;
	
	/* mkpat prepares all the grid entries of the pattern assuming
	 * color==1. If this is not true, we swap all the colors on the
	 * board as we merge.
	 */
	if (this != EMPTY && color != 0)
	  this = OTHER_COLOR(this);
	
	for (shift=0, ii = i-2; ii <= i+1 ; ++ii, shift += 8) {
	  if (ii < 0 || ii >= board_size)
	    continue;
	  
	  /* Add this one into all the nearby merged_board[][] elements. */
	  if (j > 1)  merged_board[color][ii][j-2] |= this << (shift+0);
	  if (j > 0)  merged_board[color][ii][j-1] |= this << (shift+2);
	  merged_board[color][ii][j]               |= this << (shift+4);
	  if (j < board_size-1) 
	    merged_board[color][ii][j+1] |= this << (shift+6);
	}
      }
    }
  
    /* Now go over the positions near the edge, and write in illegal
     * 'color' %11 for each location off the board. This stops a pattern
     * requiring spaces from matching near the edge.
     */

    for (i=0; i<board_size; ++i) {
      merged_board[color][0][i]            |= 0xff000000;
      merged_board[color][board_size-2][i] |= 0x000000ff;
      merged_board[color][board_size-1][i] |= 0x0000ffff;
      
      merged_board[color][i][0]  |= 0xc0c0c0c0;
      merged_board[color][i][board_size-2] |= 0x03030303;
      merged_board[color][i][board_size-1] |= 0x0f0f0f0f;
    }
  }
  
#endif
}



/* Compute the transform of (i,j) under transformation number trans.
 * *ti and *tj point to the transformed coordinates.
 * ORDER MATTERS : see texinfo documentation for details
 *
 * There is a copy of this table in mkpat.c
 */

const int transformations[8][2][2] = {
  {{ 1,  0}, { 0,  1}}, /* a - identity transformation matrix */
  {{ 0,  1}, {-1,  0}}, /* g - rotate 270 counter-clockwise */
  {{-1,  0}, { 0, -1}}, /* d - rotate 180 */
  {{ 0, -1}, { 1,  0}}, /* f - rotate 90 counter-clockwise */
  {{ 0, -1}, {-1,  0}}, /* h - rotate 90 and invert */
  {{-1,  0}, { 0,  1}}, /* b - flip left */
  {{ 0,  1}, { 1,  0}}, /* e - rotate 90 and flip left */
  {{ 1,  0}, { 0, -1}}  /* c - invert */
};


/* Functional version for completeness. Prefer the TRANSFORM macro
 * in patterns.h.
 */

void 
transform(int i, int j, int *ti, int *tj, int trans)
{
  TRANSFORM(i, j, ti, tj, trans);
}


/* Compute the point offset by (i,j), relative to a base point (basei,basej), 
 * taking into account a transformation.
 */

void 
offset(int i, int j, int basei, int basej, int *ti, int *tj, int trans)
{
  int ui, uj;
  TRANSFORM(i, j, &ui, &uj, trans);
  (*ti) = basei + ui;
  (*tj) = basej + uj;
}

/* Check the pattern database has been prepared for matching on a
 * board of this size, then forward the call to the internal
 * do_matchpat().
 */
void
matchpat(int m, int n, matchpat_callback_fn_ptr callback, int color,
	 struct pattern_db *pdb, void *callback_data,
	 char goal[MAX_BOARD][MAX_BOARD]) 
{
  if (pdb->fixed_for_size != board_size) {
    fixup_patterns_for_board_size(pdb->patterns);
    pdb->fixed_for_size = board_size;
  }

#if DFA_ENABLED > 0
  if(pdb->pdfa != NULL)
    {
      do_dfa_matchpat(pdb->pdfa,m, n, callback, color, pdb->patterns, 
		callback_data, goal);
      return;
    }
#endif
  do_matchpat(m, n, callback, color, pdb->patterns, callback_data, goal);
}

/* Precomputed tables to allow rapid checks on the piece at
 * the board. This table relies on the fact that color is
 * 1 or 2.
 *
 * For pattern element i,  require  (p[m][n] & andmask[i]) == valmask[i]
 *
 * .XO) For i=0,1,2,  p[m][n] & 3 is a no-op, so we check p[][] == valmask
 * x)   For i=3, we are checking that p[][] is not color, so AND color and
 *      we get 0 for either empty or OTHER_COLOR, but color if it contains
 *      color
 * o)   Works the other way round for checking it is not X.
 *
 *
 *  gcc allows the entries to be computed at run-time, but that is not ANSI.
 */
 
static const int and_mask[2][8] = {
  /*  .      X      O     x      o      ,      a      !         color */ 
  {   3,     3,     3,  WHITE, BLACK,   3,     3,     3   }, /* BLACK */
  {   3,     3,     3,  BLACK, WHITE,   3,     3,     3   }  /* WHITE */
};

static const int val_mask[2][8] = {
  { EMPTY, BLACK, WHITE,  0,     0,   EMPTY, EMPTY, EMPTY},  /* BLACK */ 
  { EMPTY, WHITE, BLACK,  0,     0,   EMPTY, EMPTY, EMPTY}   /* WHITE */
};


/* and a table for checking classes quickly
 * class_mask[status][color] contains the mask to look for in class.
 * ie. if  pat[r].class & class_mask[dragon[x][y].status][p[x][y]]
 * is not zero then we reject it
 * Most elements if class_mask[] are zero - it is a sparse
 * matrix containing
 *  CLASS_O in [DEAD][color]
 *  CLASS_O in [CRITICAL][color]
 *  CLASS_o in [ALIVE][color]
 *  CLASS_X in [DEAD][other]
 *  CLASS_x in [ALIVE][other]
 *
 * so eg. if we have a dead white dragon, and we
 * are checking a pattern for black, then
 *  class_mask[DEAD][other]  will contain CLASS_X
 * Then we reject any patterns which have CLASS_X
 * set in the class bits.
 *
 * Making it static guarantees that all fields are
 * initially set to 0, and we overwrite the ones
 * we care about each time.
 */
  
static int class_mask[MAX_DRAGON_STATUS][3];

/*
 * Try all the patterns in the given array at (m,n). Invoke the
 * callback for any that matches. Classes X,O,x,o are checked here. It
 * is up to the callback to process the other classes, and any helper
 * or autohelper functions.
 *
 * If the support of goal[MAX_BOARD][MAX_BOARD] is a subset of the board,
 * patterns are rejected which do not involve this dragon. If goal is a 
 * null pointer, this parameter is ignored.
 */

static void
do_matchpat(int m, int n, matchpat_callback_fn_ptr callback, int color,
	    struct pattern *pattern, void *callback_data,
	    char goal[MAX_BOARD][MAX_BOARD]) 
{
  int other = OTHER_COLOR(color);
  int ll;   /* Iterate over transformations (rotations or reflections)  */
  
  class_mask[DEAD][color]     = CLASS_O;
  class_mask[DEAD][other]     = CLASS_X;
  class_mask[CRITICAL][color] = CLASS_O;
  class_mask[CRITICAL][other] = 0;       /* Need to reset this. */
  class_mask[ALIVE][color]    = CLASS_o;
  class_mask[ALIVE][other]    = CLASS_x;

  /* Basic sanity checks. */
  assert(color != EMPTY);
  assert(m>=0 && m<board_size && n>=0 && n<board_size);

  /* Verify that this is the board we have prepared for matching. This
   * test is not guaranteed to always work (the hash values may
   * coincide although the boards don't), but it doesn't really have
   * to since a systematic error would show up in almost every game.
   */
  if (hashdata_compare(&compilation_hashdata, &hashdata) != 0)
    compile_for_match();
  assert(hashdata_diff_dump(&compilation_hashdata, &hashdata) == 0);

  
  /* Try each pattern - NULL pattern marks end of list. */
  for ( ; pattern->patn; ++pattern) { 

    /*
     * These days we always match all patterns.
     */
    if (1) {
      int start_transformation = 0;
      int end_transformation = pattern->trfno;
      
      /* Ugly trick for dealing with 'O' symmetry. */
      if (pattern->trfno == 5) {
	start_transformation = 2;
	end_transformation = 6;
      }
      
#ifdef PROFILE_MATCHER
      totals[0] += end_transformation - start_transformation;
#endif

      /* We can check the color of the anchor stone now.
       * Roughly half the patterns are anchored at each
       * color, and since the anchor stone is invariant under
       * rotation, we can reject all rotations of a wrongly-anchored
       * pattern in one go.
       *
       * Patterns are always drawn from O perspective in .db,
       * so p[m][n] is 'color' if the pattern is anchored
       * at O, or 'other' for X.
       * Since we require that this flag contains 3 for
       * anchored_at_X, we can check that
       *   p[m][n] == (color ^ anchored_at_X)
       * which is equivalent to
       *           == anchored_at_X ? other : color
       */

      if (p[m][n] != (pattern->anchored_at_X ^ color) )
	continue;  /* does not match the anchor */
    
      /* try each orientation transformation */
      for (ll = start_transformation; ll < end_transformation; ll++) 
	check_pattern (m, n, callback, color, pattern, ll,
		       callback_data, goal);
    } /* if not rejected by maxwt */
  } /* loop over patterns */
}


/*
 * Do the pattern matching for a given pattern and a given 
 * transformation ll.
 */

static void
check_pattern(int m, int n, matchpat_callback_fn_ptr callback, int color,
	      struct pattern *pattern, int ll, void *callback_data,
	      char goal[MAX_BOARD][MAX_BOARD])
{
  int k;			/* Iterate over elements of pattern */
  int found_goal, found_nongoal;
  
#if PROFILE_PATTERNS
  int nodes_before;
#endif
  
#ifdef PROFILE_MATCHER
  ++totals[1];
#endif
  
#if GRID_OPT == 1
  
  /* We first perform the grid check : this checks up to 16
   * elements in one go, and allows us to rapidly reject
   * patterns which do not match.  While this check invokes a
   * necessary condition, it is not a sufficient test, so more
   * careful checks are still required, but this allows rapid
   * rejection. merged_board[][] should contain a combination of
   * 16 board positions around m,n.  The colours have been fixed
   * up so that stones which are 'O' in the pattern are
   * bit-pattern %01.  
   */
  if ((merged_board[color - 1][m][n] & pattern->and_mask[ll])
      != pattern->val_mask[ll])
    return;			/* large-scale match failed */
  
#endif /* GRID_OPT == 1 */
  

#ifdef PROFILE_MATCHER
  ++totals[2];
#endif
  
  /* Next, we do the range check. This applies the edge
   * constraints implicitly.
   */
  {
    int mi, mj, xi, xj;
    
    TRANSFORM(pattern->mini, pattern->minj, &mi, &mj, ll);
    TRANSFORM(pattern->maxi, pattern->maxj, &xi, &xj, ll);
    
    /* transformed {m,x}{i,j} are arbitrary corners - 
       Find top-left and bot-right. */
    if (xi < mi) {
      int xx = mi;
      mi = xi;
      xi = xx;
    }
    if (xj < mj) {
      int xx = mj;
      mj = xj;
      xj = xx;
    }
    
    DEBUG(DEBUG_MATCHER, 
	  "---\nconsidering pattern '%s', rotation %d at %m. Range %d,%d -> %d,%d\n",
	  pattern->name, ll, m, n, mi, mj, xi, xj);
    
    /* now do the range-check */
    if (m + mi < 0 || m + xi > board_size - 1
	|| n + mj < 0 || n + xj > board_size - 1)
      return;			/* out of range */
  }

#ifdef PROFILE_MATCHER
  ++totals[3];
#endif
  
  /* Now iterate over the elements of the pattern. */
  found_goal = 0;
  found_nongoal = 0;
  for (k = 0; k < pattern->patlen; ++k) {
  				/* match each point */
    int x, y;			/* absolute (board) co-ords of 
  				   (transformed) pattern element */
    int att = pattern->patn[k].att;	/* what we are looking for */
    
    
    /* Work out the position on the board of this pattern element. */
    
    /* transform pattern real coordinate... */
    TRANSFORM(pattern->patn[k].x, pattern->patn[k].y, &x, &y, ll);
    x += m;
    y += n;
  
    assert (x >= 0 && x < board_size && y >= 0 && y < board_size);

    /* ...and check that p[x][y] matches (see above). */
    if ((p[x][y] & and_mask[color - 1][att]) != val_mask[color - 1][att])
      goto match_failed;
    
    if (goal != NULL) {
      if (goal[x][y])
	found_goal = 1;
      else if (p[x][y] == color)
	found_nongoal = 1;
    }
    
    /* Check out the class_X, class_O, class_x, class_o
     * attributes - see patterns.db and above.
     */
    if ((pattern->class
	 & class_mask[dragon[x][y].matcher_status][p[x][y]]) != 0)
      goto match_failed;
    
  }				/* loop over elements */
  
  /* Make it here ==> We have matched all the elements to the board. */
  if ((goal != NULL) && !found_goal)
    goto match_failed;
  if ((goal != NULL) && ((pattern->class) & CLASS_C) && !found_nongoal)
    goto match_failed;

#ifdef PROFILE_MATCHER
  ++totals[4];
#endif

#if GRID_OPT == 2

  /* Make sure the grid optimisation wouldn't have rejected this pattern */
  ASSERT((merged_board[color - 1][m][n] & pattern->and_mask[ll]) ==
	 pattern->val_mask[ll], m, n);
  
#endif /* we don't trust the grid optimisation */

#ifdef PROFILE_MATCHER
  ++totals[5];
#endif
  
#if PROFILE_PATTERNS
  pattern->hits++;
  nodes_before = stats.nodes;
#endif
  
  /* A match!  - Call back to the invoker to let it know. */
  callback(m, n, color, pattern, ll, callback_data);
  
#if PROFILE_PATTERNS
  pattern->reading_nodes += stats.nodes - nodes_before;
#endif
  
  /* We jump to here as soon as we discover a pattern has failed. */
 match_failed:
  DEBUG (DEBUG_MATCHER,
	 "end of pattern '%s', rotation %d at %m\n---\n",
	 pattern->name, ll, m, n);
  
} /* check_pattern */

/*
 * Do the pattern matching for a given pattern and a given 
 * transformation ll. (light version)
 */

static void
check_pattern_light(int m, int n, matchpat_callback_fn_ptr callback, int color,
	      struct pattern *pattern, int ll, void *callback_data,
	      char goal[MAX_BOARD][MAX_BOARD])
{
  int k;			/* Iterate over elements of pattern */
  int found_goal, found_nongoal;
  
#if PROFILE_PATTERNS
  int nodes_before;
#endif
  
#ifdef PROFILE_MATCHER
  ++totals[1];
#endif
 
  /* Now iterate over the elements of the pattern. */
  found_goal = 0;
  found_nongoal = 0;
  for (k = 0; k < pattern->patlen; ++k) {
  				/* match each point */
    int x, y;			/* absolute (board) co-ords of 
  				   (transformed) pattern element */

    /* transform pattern real coordinate... */
    TRANSFORM(pattern->patn[k].x, pattern->patn[k].y, &x, &y, ll);
    x += m;
    y += n;
    assert (x >= 0 && x < board_size && y >= 0 && y < board_size);

    /* goal check */
    if (goal != NULL) {
      if (goal[x][y])
	found_goal = 1;
      else if (p[x][y] == color)
	found_nongoal = 1;
    }
    
    /* class check */
    if ((pattern->class
	 & class_mask[dragon[x][y].matcher_status][p[x][y]]) != 0)
      goto match_failed;
    
  }/* loop over elements */
  
  /* Make it here ==> We have matched all the elements to the board. */
  if ((goal != NULL) && !found_goal)
    goto match_failed;
  if ((goal != NULL) && ((pattern->class) & CLASS_C) && !found_nongoal)
    goto match_failed;

#ifdef PROFILE_MATCHER
  ++totals[4];
#endif

#if PROFILE_PATTERNS
  pattern->hits++;
  nodes_before = stats.nodes;
#endif
  
  /* A match!  - Call back to the invoker to let it know. */
  callback(m, n, color, pattern, ll, callback_data);
  
#if PROFILE_PATTERNS
  pattern->reading_nodes += stats.nodes - nodes_before;
#endif
  
  /* We jump to here as soon as we discover a pattern has failed. */
 match_failed:
  DEBUG (DEBUG_MATCHER,
	 "end of pattern '%s', rotation %d at %m\n---\n",
	 pattern->name, ll, m, n);
  
} /* check_pattern_light */


/* A dedicated matcher which can only do fullboard matching on
 * odd-sized boards, optimized for fuseki patterns.
 */
void
fullboard_matchpat(fullboard_matchpat_callback_fn_ptr callback, int color,
		   struct fullboard_pattern *pattern)
{
  int other = OTHER_COLOR(color);
  int ll;   /* Iterate over transformations (rotations or reflections)  */
  int k;    /* Iterate over elements of pattern */
  int mid = (board_size-1)/2; /* We transform around the center point. */
  int m, n;
  int number_of_stones_on_board = 0;
  
  /* Basic sanity check. */
  assert(color != EMPTY);
  assert(board_size%2 == 1);
  
  /* Count the number of stones on the board. */
  for (m=0; m<board_size; m++)
    for (n=0; n<board_size; n++)
      if (p[m][n] != EMPTY)
	number_of_stones_on_board++;
  
  /* Try each pattern - NULL pattern marks end of list. */
  for ( ; pattern->patn; ++pattern) { 
    /* The number of stones on the board must be right. This is not
     * only an optimization because we never even look at the
     * intersections which are empty in the pattern.
     */
    if (pattern->patlen != number_of_stones_on_board)
      continue;
    
    /* try each orientation transformation */
    for (ll = 0; ll < 8; ll++) {
      /* Now iterate over the elements of the pattern. */
      for (k = 0; k < pattern->patlen; ++k) { /* match each point */
	int x, y; /* board co-ords of transformed pattern element */
	int att = pattern->patn[k].att;  /* what we are looking for */
	
	/* Work out the position on the board of this pattern element. */
	TRANSFORM(pattern->patn[k].x, pattern->patn[k].y, &x, &y, ll);
	x += mid;
	y += mid;
	
	assert(x>=0 && x < board_size && y >= 0 && y < board_size);

	if ((att == ATT_O && p[x][y] != color)
	    || (att == ATT_X && p[x][y] != other))
	  break;
	
      } /* loop over elements */
	
      if (k == pattern->patlen) {
	/* A match!  - Call back to the invoker to let it know. */
	int x, y;
	TRANSFORM(pattern->movei, pattern->movej, &x, &y, ll);
	x += mid;
	y += mid;
	callback(x, y, pattern, ll);
      }
    }
  }
}

void
dfa_match_init(void)
{
  const char *const_path = NULL;
  char *path = NULL;

  /* get the path where to find dfa files */
  const_path = getenv ("GNUGO_PATH");
  if(const_path == NULL)
    {
      fprintf(stderr," Could not find $GNUGO_PATH variable\n");
      path = calloc(3,sizeof(char)); 
      strcpy(path,"./");
    }
  else
    {
      path = calloc(strlen(const_path)+200,sizeof(char)); 
      strcpy(path, const_path);
      strcat(path,"/patterns/");
    }

  dfa_init();
  
  pat_db.pdfa = load_dfa(path,"pat.dfa",NULL);
  joseki_db.pdfa = load_dfa(path,"joseki.dfa",NULL);
  owl_attackpat_db.pdfa = load_dfa(path,"owl_attackpat.dfa",NULL);
  owl_defendpat_db.pdfa = load_dfa(path,"owl_defendpat.dfa",NULL);
  conn_db.pdfa = load_dfa(path,"conn.dfa",NULL);
  attpat_db.pdfa = load_dfa(path,"attpat.dfa",NULL);
  defpat_db.pdfa = load_dfa(path,"defpat.dfa",NULL);
  owl_vital_apat_db.pdfa = load_dfa(path,"owl_vital_apats.dfa",NULL);
  endpat_db.pdfa = load_dfa(path,"endpat.dfa",NULL);
  influencepat_db.pdfa = load_dfa(path,"influencepat.dfa",NULL);
  barrierspat_db.pdfa = load_dfa(path,"barrierspat.dfa",NULL);

  free(path);
}


#if DFA_SORT
static int
compare_int (const void *a, const void *b)
{
  const int *da = (const int *) a;
  const int *db = (const int *) b;
     
  return (*da > *db) - (*da < *db);
}

#endif

/* perform the pattern matching with a dfa filtering */
static void 
do_dfa_matchpat(dfa_t *pdfa,
		int m, int n, matchpat_callback_fn_ptr callback,
		int color, struct pattern *database,
		void *callback_data, char goal[MAX_BOARD][MAX_BOARD])
{
  int other = OTHER_COLOR(color);
  int ll;      /* Iterate over transformations (rotations or reflections)  */
  int matched; /* index in database[] of the matched pattern */

#if DFA_SORT
  int reorder[10000];
  int maxr =0,k;
#endif

  class_mask[DEAD][color]     = CLASS_O;
  class_mask[DEAD][other]     = CLASS_X;
  class_mask[CRITICAL][color] = CLASS_O;
  class_mask[CRITICAL][other] = 0;       /* Need to reset this. */
  class_mask[ALIVE][color]    = CLASS_o;
  class_mask[ALIVE][other]    = CLASS_x;

  /* Basic sanity checks. */
  assert(color != EMPTY);
  assert(m>=0 && m<board_size && n>=0 && n<board_size);

  /* Verify that this is the board we have prepared for matching. This
   * test is not guaranteed to always work (the hash values may
   * coincide although the boards don't), but it doesn't really have
   * to since a systematic error would show up in almost every game.
   */
  if (hashdata_compare(&compilation_hashdata, &hashdata) != 0)
    compile_for_match();
  assert(hashdata_diff_dump(&compilation_hashdata, &hashdata) == 0);

  /* FIXME: should be with compile_for_match? */
  dfa_compile_for_match(color); 

  /* for the moment we perform one test by transformation,
   * later we will add one pattern by transformation in the
   * dfa to speedup the matching by 8.
   */
  for(ll = 0; ll != 8; ll++)
    {
      dfa_restart(pdfa, ll,m,n); 
      matched = dfa_scan(pdfa);
      while(matched != -1)
	{
	  int start_transformation = 0;
	  int end_transformation = database[matched].trfno;
      
	  /* Ugly trick for dealing with 'O' symmetry. */
	  if (database[matched].trfno == 5) {
	    start_transformation = 2;
	    end_transformation = 6;
	  }
	  
	  if((ll >= start_transformation) && (ll < end_transformation))
	    {
#if PROFILE_PATTERNS
	      database[matched].dfa_hits++;
#endif
#if DFA_SORT
	      reorder[maxr++]= matched*8+ll;
#else
	      check_pattern_light (m, n, callback, color, database+matched, 
			     ll, callback_data, goal);
#endif

	    }
	  matched = dfa_scan(pdfa);

	  
	}
    }

#if DFA_SORT
  qsort(reorder,maxr,sizeof(int),compare_int);
  for(k=0; k!= maxr ; k++)
    check_pattern_light (m, n, callback, color, database+(reorder[k]/8), 
		   reorder[k]%8, callback_data, goal);
#endif

}


#if PROFILE_PATTERNS
/* Initialize pattern profiling fields in one pattern struct array. */
static void
clear_pattern_profile_data(struct pattern *pattern)
{
  for (; pattern->patn; ++pattern) {
    pattern->hits = 0;
    pattern->reading_nodes = 0;
  }
}


/* Print profiling information for one pattern struct array. */
static void
print_pattern_profile_data(struct pattern *pattern, int *total_hits,
			   int *total_nodes, int *total_dfa_hits)
{
  for (; pattern->patn; ++pattern)
    if (pattern->hits > 0) {
      *total_dfa_hits += pattern->dfa_hits;
      *total_hits += pattern->hits;
      *total_nodes += pattern->reading_nodes;
      fprintf(stderr, "%6d %6d %9d %8.1f %s\n", pattern->dfa_hits,
	      pattern->hits,
	      pattern->reading_nodes,
	      pattern->reading_nodes / (float) pattern->hits, 
	      pattern->name);
    }
}
#endif


/* Initialize pattern profiling fields in pattern struct arrays. */
void
prepare_pattern_profiling()
{
#if PROFILE_PATTERNS
  clear_pattern_profile_data(pat_db.patterns);
  clear_pattern_profile_data(attpat_db.patterns);
  clear_pattern_profile_data(defpat_db.patterns);
  clear_pattern_profile_data(endpat_db.patterns);
  clear_pattern_profile_data(conn_db.patterns);
  clear_pattern_profile_data(influencepat_db.patterns);
  clear_pattern_profile_data(barrierspat_db.patterns);
  clear_pattern_profile_data(owl_attackpat_db.patterns);
  clear_pattern_profile_data(owl_vital_apat_db.patterns);
  clear_pattern_profile_data(owl_defendpat_db.patterns);
#else
  fprintf(stderr,
	  "Warning, no support for pattern profiling in this binary.\n");
#endif
}


/* Report result of pattern profiling. Only patterns with at least one
 * match are listed.
 */
void
report_pattern_profiling()
{
#if PROFILE_PATTERNS
  int hits = 0;
  int dfa_hits = 0;
  int nodes = 0;
  print_pattern_profile_data(pat_db.patterns, 
			     &hits, &nodes, &dfa_hits);
  print_pattern_profile_data(attpat_db.patterns, 
			     &hits, &nodes, &dfa_hits);
  print_pattern_profile_data(defpat_db.patterns, 
			     &hits, &nodes, &dfa_hits);
  print_pattern_profile_data(endpat_db.patterns, 
			     &hits, &nodes, &dfa_hits);
  print_pattern_profile_data(conn_db.patterns, 
			     &hits, &nodes, &dfa_hits);
  print_pattern_profile_data(influencepat_db.patterns, 
			     &hits, &nodes, &dfa_hits);
  print_pattern_profile_data(barrierspat_db.patterns, 
			     &hits, &nodes, &dfa_hits);
  print_pattern_profile_data(owl_attackpat_db.patterns, 
			     &hits, &nodes, &dfa_hits);
  print_pattern_profile_data(owl_vital_apat_db.patterns, 
			     &hits, &nodes, &dfa_hits);
  print_pattern_profile_data(owl_defendpat_db.patterns, 
			     &hits, &nodes, &dfa_hits);
  fprintf(stderr, "------ ---------\n");
  fprintf(stderr, "%6d, %6d %9d\n", dfa_hits, hits, nodes);
#endif
}

/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
