/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 and 2000 by the Free Software Foundation.      *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <assert.h>

#include "sgftree.h"
#include "liberty.h"
#include "analyze.h"


/*
 * Initialize the gnugo engine. This needs to be called 
 * once only.
 */

#if DFA_ENABLED
extern void dfa_match_init(void);
#endif

void
init_gnugo(float memory)
{
  reading_cache_init(memory * 1024 * 1024);
  clear_board();

#if DFA_ENABLED
  dfa_match_init();
#endif
}


/*
 * Put the values in POS into the global variables which is the
 * equivalent of the position.  
 */

static void
position_to_globals(Position *pos)
{
  board_size = pos->boardsize;
  setup_board(pos->board, pos->ko_i, pos->ko_j,
	      pos->white_captured, pos->black_captured);
}


/*
 * Get the values in POS from the global variables which is the
 * equivalent of the position.
 */

static void
globals_to_position(Position *pos)
{
  pos->boardsize = board_size;
  memcpy(pos->board, p, sizeof(p));
  pos->ko_i = ko_i;
  pos->ko_j = ko_j;

  pos->white_captured = white_captured;
  pos->black_captured = black_captured;
}


/* ---------------------------------------------------------------- */


/*
 * Clear a position.
 */

void
gnugo_clear_position(Position *pos, int boardsize)
{
  assert(MIN_BOARD <= boardsize && boardsize <= MAX_BOARD);
  pos->boardsize = boardsize;
  memset(pos->board, EMPTY, sizeof(pos->board));
  pos->ko_i = -1;
  pos->ko_j = -1;

  pos->white_captured = 0;
  pos->black_captured = 0;
}


/* Copy a position */
void
gnugo_copy_position(Position *to, Position *from)
{
  memcpy(to, from, sizeof(Position));
}

/* Wrapper for position_to_globals */
void
gnugo_force_to_globals(Position *pos)
{
  position_to_globals(pos);
}

/* Interface to add_stone */
void
gnugo_add_stone(Position *pos, int i, int j, int color)
{
  position_to_globals(pos);
  add_stone(i, j, color);
  globals_to_position(pos);
}


/* Interface to remove_stone */
void
gnugo_remove_stone(Position *pos, int i, int j)
{
  position_to_globals(pos);
  remove_stone(i, j);
  globals_to_position(pos);
}


/* Interface to play_move */
void
gnugo_play_move(Position *pos, int i, int j, int color)
{
  position_to_globals(pos);
  play_move(i, j, color);
  globals_to_position(pos);
}


/* Interface to is_legal. */
int
gnugo_is_legal(Position *pos, int i, int j, int color)
{
  int  retval;

  position_to_globals(pos);
  retval = is_legal(i, j, color);

  return retval;
}


/* Interface to is_suicide. */
int
gnugo_is_suicide(Position *pos, int i, int j, int color)
{
  int  retval;

  position_to_globals(pos);
  retval = is_suicide(i, j, color);

  return retval;
}


/* Interface to placehand. Sets up handicap pieces and
 * returns the number of placed handicap stones. */
int
gnugo_placehand(Position *pos, int handicap)
{
  int  retval;

  position_to_globals(pos);
  retval = placehand(handicap);
  globals_to_position(pos);

  return retval;
}


/* Interface to sgffile_recordboard */
void
gnugo_recordboard(Position *pos, SGFNode *root)
{
  position_to_globals(pos);
  sgffile_recordboard(root);
}

/* Interface to placehand. Sets up handicap pieces and
 * returns the number of placed handicap stones, updating the sgf file
 */
int
gnugo_sethand(Position *pos, int handicap, SGFNode *node)
{
  int  stones;

  position_to_globals(pos);
  stones = placehand(handicap);
  sgffile_recordboard(node);
  globals_to_position(pos);

  return stones;
}


/* Interface to genmove */
int
gnugo_genmove(Position *pos, int *i, int *j, int color, int move_number)
{
  position_to_globals(pos);
  movenum = move_number;	/* Another global */
  return genmove(i, j, color);
}

/* Interface to attack*/
int
gnugo_attack(Position *pos, int m, int n, int *i, int *j)
{
  position_to_globals(pos);
  return attack(m, n, i, j);
}


/* Interface to find_defense */
int
gnugo_find_defense(Position *pos, int m, int n, int *i, int *j)
{
  position_to_globals(pos);
  return find_defense(m, n, i, j);
}


/* Interface to who_wins */
void
gnugo_who_wins(Position *pos, int color, float komi, FILE *outfile)
{
  position_to_globals(pos);
  who_wins(color, komi, outfile);
}


void
gnugo_evaluate_territory(Position *pos, int *white_terri, int *black_terri,
			 int *white_dead, int *black_dead)
{
  position_to_globals(pos);
  evaluate_territory(white_terri, black_terri, white_dead, black_dead);
}


void
gnugo_examine_position(Position *pos, int color, int how_much)
{
  position_to_globals(pos);
  examine_position(color, how_much);
}


/* ================================================================ */
/*                             Gameinfo                             */
/* ================================================================ */


/*
 * Initialize the structure.
 */

void
gameinfo_clear(Gameinfo *ginfo, int boardsize)
{
  ginfo->handicap = 0;
  ginfo->komi = 5.5;
  
  ginfo->move_number = 0;
  gnugo_clear_position(&(ginfo->position), boardsize);
  ginfo->to_move = BLACK;
  sgftree_clear(&ginfo->moves);

  /* Info relevant to the computer player. */
  ginfo->seed = 0.0;
  ginfo->computer_player = WHITE; /* Make an assumtion. */

  ginfo->outfilename[0] = '\0';
  ginfo->outfile = NULL;
}


/*
 * Print a gameinfo.
 */

void
gameinfo_print(Gameinfo *ginfo)
{
  printf("Board Size:   %d\n", ginfo->position.boardsize);
  printf("Handicap      %d\n", ginfo->handicap);
  printf("Komi:         %.1f\n", ginfo->komi);
  printf("Move Number:  %d\n", ginfo->move_number);

  printf("To Move:      %s\n", color_to_string(ginfo->to_move));

  printf("Computer player: ");
  if (ginfo->computer_player == WHITE)
    printf("White\n");
  else if (ginfo->computer_player == BLACK)
    printf("Black\n");
  else if (ginfo->computer_player == EMPTY)
    printf("Both (solo)\n");
  else
    printf("Nobody\n");
}


/*
 * Make a move in the game.  Return 1 if the move was legal. In that
 * case the move is actually done. Otherwise return 0.
 */

void
gameinfo_play_move(Gameinfo *ginfo, int i, int j, int color)
{
  gnugo_play_move(&ginfo->position, i, j, color);
  sgftreeAddPlay(&ginfo->moves, 0, color, i, j);
  sgfAnalyzer(ginfo->moves.lastnode, 0);

  sgffile_move_made(i, j, color, 0);

  ginfo->move_number++;
  ginfo->to_move = OTHER_COLOR(color);
}


/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
