/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 by the Free Software Foundation.               *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
 \* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */



/* The functions in this file are incremental versions of the
 * functions in board.c. See the Texinfo documentation
 * (Utility Functions: Incremental Board) for an introduction
 * to this file.
 */

#include <stdio.h>
#include <string.h>
#include "incremental_board.h"

/* Main array of string information. */
static struct string_data string[MAX_STRINGS];

/* Stacks and stack pointers. */
struct change_stack_entry change_stack[STACK_SIZE];
struct vertex_stack_entry vertex_stack[STACK_SIZE];
struct change_stack_entry *change_stack_pointer;
struct vertex_stack_entry *vertex_stack_pointer;

/* Index into list of strings. The index is only valid if there is a
 * stone at the vertex.
 */
static int string_number[MAX_BOARD][MAX_BOARD];

/* The stones in a string are linked together in a cyclic list. These
 * are the coordinates to the next stone in the string.
 */
static int next_stonei[MAX_BOARD][MAX_BOARD];
static int next_stonej[MAX_BOARD][MAX_BOARD];

/* Macros to traverse the stones of a string. */
#define FIRST_STONE(s, i, j) (i = string[s].origini, j = string[s].originj)
#define NEXT_STONE(i, j) do {\
    int nexti = next_stonei[i][j];\
    j = next_stonej[i][j];\
    i = nexti;\
  } while (0)
#define BACK_TO_FIRST_STONE(s, i, j) (i == string[s].origini &&\
				      j == string[s].originj)

/* Assorted useful macros. Some of them could have been functions but
 * are implemented as macros for speed.
 */
#define LIBERTY(i, j) (p[i][j] == EMPTY)
#define UNMARKED_LIBERTY(i, j) (p[i][j] == EMPTY && ml[i][j] != liberty_mark)
#define MARK_LIBERTY(i, j) ml[i][j] = liberty_mark

#define UNMARKED_STRING(i, j) (string[string_number[i][j]].mark != string_mark)
#define UNMARKED_OPPONENT_STRING(s, i, j)\
  (p[i][j] == OTHER_COLOR(string[s].color)\
   && string[string_number[i][j]].mark != string_mark)
#define UNMARKED_OWN_STRING(s, i, j)\
  (p[i][j] == string[s].color\
   && string[string_number[i][j]].mark != string_mark)
#define MARK_STRING(i, j) string[string_number[i][j]].mark = string_mark
#define STRING_AT_VERTEX(i, j, s)\
  (p[i][j] != EMPTY && string_number[i][j] == (s))
  
#define LIBERTIES(i, j) string[string_number[i][j]].liberties
#define ADD_LIBERTY(s, i, j) do {\
    if (string[s].liberties < MAX_LIBERTIES) {\
      string[s].libi[string[s].liberties] = i;\
      string[s].libj[string[s].liberties] = j;\
    }\
    string[s].liberties++;\
  } while (0)
#define ADD_AND_MARK_LIBERTY(s, i, j) do {\
    if (string[s].liberties < MAX_LIBERTIES) {\
      string[s].libi[string[s].liberties] = i;\
      string[s].libj[string[s].liberties] = j;\
    }\
    string[s].liberties++;\
    ml[i][j] = liberty_mark;\
  } while (0)
#define ADD_NEIGHBOR(s, i, j)\
  string[s].neighborlist[string[s].neighbors++] = string_number[i][j]

#define DO_ADD_STONE(i, j, color) do {\
    PUSH_VERTEX(p[i][j]);\
    p[i][j] = color;\
    hashdata_invert_stone(&hashdata, i, j, color);\
  } while (0)
#define DO_REMOVE_STONE(i, j) do {\
    PUSH_VERTEX(p[i][j]);\
    hashdata_invert_stone(&hashdata, i, j, p[i][j]);\
    p[i][j] = EMPTY;\
  } while (0)


/* Are the data structures up to date for the current board position. */
static int strings_initialized = 0;

/* Number of the next free string. */
static int next_string;

/* For marking purposes. */
static int ml[MAX_BOARD][MAX_BOARD];
static int liberty_mark;
static int string_mark;


#if 0
/* Debug function. Dump all string information. */
static void
dump_incremental_board(void)
{
  int i, j, s;
  
  for (i=0; i<board_size; i++) {
    for (j=0; j<board_size; j++)
      if (p[i][j] == EMPTY)
	fprintf(stderr, " . ");
      else
	fprintf(stderr, "%2d ", string_number[i][j]);
    fprintf(stderr, "\n");
  }

  for (s=0; s<next_string; s++) {
    if (p[string[s].origini][string[s].originj] == EMPTY)
      continue;
    
    gprintf("%o%d %s %m size %d, %d liberties, %d neighbors\n", s,
	    color_to_string(string[s].color),
	    string[s].origini, string[s].originj, string[s].size,
	    string[s].liberties, string[s].neighbors);
    gprintf("%ostones:");

    FIRST_STONE(s, i, j);
    do {
      gprintf("%o %m", i, j);
      NEXT_STONE(i, j);
    } while (!BACK_TO_FIRST_STONE(s, i, j));
    
    gprintf("%o\nliberties:");
    for (i=0; i<string[s].liberties; i++)
      gprintf("%o %m", string[s].libi[i], string[s].libj[i]);
    
    gprintf("%o\nneighbors:");
    for (i=0; i<string[s].neighbors; i++)
      gprintf("%o %d(%m)", string[s].neighborlist[i],
	      string[string[s].neighborlist[i]].origini,
	      string[string[s].neighborlist[i]].originj);
    gprintf("%o\n\n");
  }
}
#endif


/* Build a string and its cyclic list representation from scratch.
 * propagate_string(i, j, m, n) adds the stone (i, j) to the string
 * (m, n) and recursively continues with not already included friendly
 * neighbors. To start a new string at (i, j), use
 * propagate_string(i, j, i, j). The size of the string is returned.
 */

static int
propagate_string(int i, int j, int m, int n)
{
  int size = 1;
  int k;
  
  if (i == m && j == n) {
    /* Start a new string. */
    next_stonei[i][j] = i;
    next_stonej[i][j] = j;
  }
  else {
    /* Link the stone at (i, j) to the string including (m, n) */
    string_number[i][j] = string_number[m][n];
    next_stonei[i][j] = next_stonei[m][n];
    next_stonej[i][j] = next_stonej[m][n];
    next_stonei[m][n] = i;
    next_stonej[m][n] = j;
  }

  /* Look in all four directions for more stones to add. */
  for (k=0; k<4; k++) {
    int di = deltai[k];
    int dj = deltaj[k];
    if (ON_BOARD(i+di, j+dj)
	&& p[i+di][j+dj] == p[i][j]
	&& string_number[i+di][j+dj] == -1)
      size += propagate_string(i+di, j+dj, m, n);
  }
  
  return size;
}


/* Build the lists of liberties and neighbors of a string from
 * scratch. No information is pushed onto the stack by this function.
 */

static void
find_liberties_and_neighbors(int s)
{
  int i, j;

  /* Clear the marks. */
  liberty_mark++;
  string_mark++;

  /* Traverse the stones of the string, by following the cyclic chain. */
  FIRST_STONE(s, i, j);
  do {
    /* Look in each direction for new liberties or new neighbors. Mark
     * already visited liberties and neighbors.
     */
    if (i > 0) {
      if (UNMARKED_LIBERTY(i-1, j)) {
	ADD_AND_MARK_LIBERTY(s, i-1, j);
      }
      else if (UNMARKED_OPPONENT_STRING(s, i-1, j)) {
	ADD_NEIGHBOR(s, i-1, j);
	MARK_STRING(i-1, j);
      }
    }
    
    if (i < board_size-1) {
      if (UNMARKED_LIBERTY(i+1, j)) {
	ADD_AND_MARK_LIBERTY(s, i+1, j);
      }
      else if (UNMARKED_OPPONENT_STRING(s, i+1, j)) {
	ADD_NEIGHBOR(s, i+1, j);
	MARK_STRING(i+1, j);
      }
    }
    
    if (j > 0) {
      if (UNMARKED_LIBERTY(i, j-1)) {
	ADD_AND_MARK_LIBERTY(s, i, j-1);
      }
      else if (UNMARKED_OPPONENT_STRING(s, i, j-1)) {
	ADD_NEIGHBOR(s, i, j-1);
	MARK_STRING(i, j-1);
      }
    }
    
    if (j < board_size-1) {
      if (UNMARKED_LIBERTY(i, j+1)) {
	ADD_AND_MARK_LIBERTY(s, i, j+1);
      }
      else if (UNMARKED_OPPONENT_STRING(s, i, j+1)) {
	ADD_NEIGHBOR(s, i, j+1);
	MARK_STRING(i, j+1);
      }
    }

    NEXT_STONE(i, j);
  } while (!BACK_TO_FIRST_STONE(s, i, j));
}


/* Update the liberties of a string from scratch, first pushing the
 * old information.
 */

static void
update_liberties(int s)
{
  int i, j, k;

  /* Push the old information. */
  PUSH_VALUE(string[s].liberties);
  for (k=0; k<string[s].liberties && k<MAX_LIBERTIES; k++) {
    PUSH_VALUE(string[s].libi[k]);
    PUSH_VALUE(string[s].libj[k]);
  }
  string[s].liberties = 0;

  /* Clear the liberty mark. */
  liberty_mark++;

  /* Traverse the stones of the string, by following the cyclic chain. */
  FIRST_STONE(s, i, j);
  do {
    /* Look in each direction for new liberties. Mark already visited
     * liberties. 
     */
    if (i > 0 && UNMARKED_LIBERTY(i-1, j)) {
      ADD_AND_MARK_LIBERTY(s, i-1, j);
    }
    
    if (i < board_size-1 && UNMARKED_LIBERTY(i+1, j)) {
      ADD_AND_MARK_LIBERTY(s, i+1, j);
    }
    
    if (j > 0 && UNMARKED_LIBERTY(i, j-1)) {
      ADD_AND_MARK_LIBERTY(s, i, j-1);
    }
    
    if (j < board_size-1 && UNMARKED_LIBERTY(i, j+1)) {
      ADD_AND_MARK_LIBERTY(s, i, j+1);
    }

    NEXT_STONE(i, j);
  } while (!BACK_TO_FIRST_STONE(s, i, j));
}


/* Set up incremental board structures and populate them with the
 * strings available in the position given by p[][]. Clear the stacks
 * and start the mark numbers from zero. All undo information is lost
 * by calling this function.
 */

void
incremental_board_init()
{
  int i, j, s;
  next_string = 0;
  liberty_mark = 0;
  string_mark = 0;
  CLEAR_STACKS();

  memset(string, 0, sizeof(string));
  memset(ml, 0, sizeof(ml));

  /* propagate_string relies on non-assigned stones to have
   * string_number -1.
   */
  for (i=0; i<board_size; i++)
    for (j=0; j<board_size; j++)
      string_number[i][j] = -1;

  /* Find the existing strings. */
  for (i=0; i<board_size; i++)
    for (j=0; j<board_size; j++)
      if (p[i][j] != EMPTY && string_number[i][j] == -1) {
	string_number[i][j] = next_string;
	string[next_string].size = propagate_string(i, j, i, j);
	string[next_string].color = p[i][j];
	string[next_string].origini = i;
	string[next_string].originj = j;
	string[next_string].mark = 0;
	next_string++;
      }
  
  /* Fill in liberty and neighbor info. */
  for (s=0; s<next_string; s++) {
    find_liberties_and_neighbors(s);
  }

  /* Now we can trust the information. */
  strings_initialized = 1;
}


/* Don't trust the incremental string data until it's reinitialized.
 *
 * This function should be called if the board is modified by other
 * means than incremental_play_move() or incremental_undo_move().
 * It's also useful to force a recomputation of the strings if we
 * don't have any immediate plans to undo the move, because it recovers
 * undo stack space and holes in the 'string' array.
 */

void
incremental_invalidate_strings()
{
  strings_initialized = 0;
}


/* Remove a string from the list of neighbors and push the changed
 * information.
 */

static void
remove_neighbor(int str, int n)
{
  int k;
  int done=0;
  struct string_data *s = &string[str];
  for (k=0; k<s->neighbors; k++)
    if (s->neighborlist[k] == n) {
      /* We need to push the last entry too because it may become
       * destroyed later.
       */
      PUSH_VALUE(s->neighborlist[s->neighbors - 1]);
      PUSH_VALUE(s->neighborlist[k]);
      PUSH_VALUE(s->neighbors);
      s->neighborlist[k] = s->neighborlist[s->neighbors - 1];
      s->neighbors--;
      done = 1;
      break;
    }
  assert(done);
}


/* Remove one liberty from the list of liberties, pushing changed
 * information. If the string had more liberties than the size of the
 * list, rebuild the list from scratch.
 */

static void
remove_liberty(int str, int i, int j)
{
  int k;
  struct string_data *s = &string[str];
  
  if (s->liberties > MAX_LIBERTIES)
    update_liberties(str);
  else {
    for (k=0; k<s->liberties; k++)
      if (s->libi[k] == i && s->libj[k] == j) {
	/* We need to push the last entry too because it may become
	 * destroyed later.
	 */
	PUSH_VALUE(s->libi[s->liberties - 1]);
	PUSH_VALUE(s->libj[s->liberties - 1]);
	PUSH_VALUE(s->libi[k]);
	PUSH_VALUE(s->libj[k]);
	PUSH_VALUE(s->liberties);
	s->libi[k] = s->libi[s->liberties - 1];
	s->libj[k] = s->libj[s->liberties - 1];
	s->liberties--;
	break;
      }
  }
}


/* Remove a string from the board, pushing necessary information to
 * restore it. Return the number of removed stones.
 */

static int
incremental_remove_string(int s)
{
  int i, j, k;

  /* Traverse the stones of the string, by following the cyclic chain. */
  FIRST_STONE(s, i, j);
  do {
    /* Push color, string number and cyclic chain pointers. */
    PUSH_VALUE(string_number[i][j]);
    PUSH_VALUE(next_stonei[i][j]);
    PUSH_VALUE(next_stonej[i][j]);
    DO_REMOVE_STONE(i, j);
    NEXT_STONE(i, j);
  } while (!BACK_TO_FIRST_STONE(s, i, j));

  /* The neighboring strings have obtained some new liberties and lost
   * a neighbor.
   */
  for (k=0; k<string[s].neighbors; k++) {
    remove_neighbor(string[s].neighborlist[k], s);
    update_liberties(string[s].neighborlist[k]);
  }

  /* Update the number of captured stones. These are assumed to
   * already have been pushed.
   */
  if (string[s].color == WHITE)
    white_captured += string[s].size;
  else
    black_captured += string[s].size;
    
  return string[s].size;
}


/* Test whether a move would be a suicide.
 *
 * This is the case if
 *
 * 1. There is no neighboring empty intersection.
 * 2. There is no neighboring opponent string with exactly one liberty.
 * 3. There is no neighboring friendly string with more than one liberty.
 *
 */

int
incremental_is_suicide(int i, int j, int color)
{
  if (!strings_initialized)
    incremental_board_init();
  
  if (i > 0
      && (LIBERTY(i-1, j)
	  || ((p[i-1][j] == color) ^ (LIBERTIES(i-1, j) == 1))))
    return 0;

  if (i < board_size-1
      && (LIBERTY(i+1, j)
	  || ((p[i+1][j] == color) ^ (LIBERTIES(i+1, j) == 1))))
    return 0;

  if (j > 0
      && (LIBERTY(i, j-1)
	  || ((p[i][j-1] == color) ^ (LIBERTIES(i, j-1) == 1))))
    return 0;

  if (j < board_size-1
      && (LIBERTY(i, j+1)
	  || ((p[i][j+1] == color) ^ (LIBERTIES(i, j+1) == 1))))
    return 0;

  return 1;
}


/* We have played an isolated new stone and need to create a new
 * string for it.
 */
static void
create_new_string(int i, int j)
{
  int s;
  int color = p[i][j];

  /* Get the next free string number. */
  PUSH_VALUE(next_string);
  s = next_string++;
  string_number[i][j] = s;
  /* Set up a size one cycle for the string. */
  next_stonei[i][j] = i;
  next_stonej[i][j] = j;

  /* Set trivially known values and initialize the rest to zero. */
  string[s].color = color;
  string[s].size = 1;
  string[s].origini = i;
  string[s].originj = j;
  string[s].liberties = 0;
  string[s].neighbors = 0;
  string[s].mark = 0;

  /* Clear the string mark. */
  string_mark++;

  /* In each direction, look for a liberty or a nonmarked opponent
   * neighbor. Mark visited neighbors. There is no need to mark the
   * liberties since we can't find them twice. */
  if (i > 0) {
    if (LIBERTY(i-1, j)) {
      ADD_LIBERTY(s, i-1, j);
    }
    else if (UNMARKED_OPPONENT_STRING(s, i-1, j)) {
      int s2 = string_number[i-1][j];
      /* Add the neighbor to our list. */
      ADD_NEIGHBOR(s, i-1, j);
      /* Add us to our neighbor's list. */
      PUSH_VALUE(string[s2].neighbors);
      ADD_NEIGHBOR(s2, i, j);
      MARK_STRING(i-1, j);
    }
  }
  
  if (i < board_size-1) {
    if (LIBERTY(i+1, j)) {
      ADD_LIBERTY(s, i+1, j);
    }
    else if (UNMARKED_OPPONENT_STRING(s, i+1, j)) {
      int s2 = string_number[i+1][j];
      ADD_NEIGHBOR(s, i+1, j);
      PUSH_VALUE(string[s2].neighbors);
      ADD_NEIGHBOR(s2, i, j);
      MARK_STRING(i+1, j);
    }
  }
  
  if (j > 0) {
    if (LIBERTY(i, j-1)) {
      ADD_LIBERTY(s, i, j-1);
    }
    else if (UNMARKED_OPPONENT_STRING(s, i, j-1)) {
      int s2 = string_number[i][j-1];
      ADD_NEIGHBOR(s, i, j-1);
      PUSH_VALUE(string[s2].neighbors);
      ADD_NEIGHBOR(s2, i, j);
      MARK_STRING(i, j-1);
    }
  }
  
  if (j < board_size-1) {
    if (LIBERTY(i, j+1)) {
      ADD_LIBERTY(s, i, j+1);
    }
    else if (UNMARKED_OPPONENT_STRING(s, i, j+1)) {
      int s2 = string_number[i][j+1];
      ADD_NEIGHBOR(s, i, j+1);
      PUSH_VALUE(string[s2].neighbors);
      ADD_NEIGHBOR(s2, i, j);
      /* No need to mark since no visits left. */
/*      MARK_STRING(i, j+1);*/
    }
  }
}


/* We have played a stone with exactly one friendly neighbor. Add the
 * new stone to that string.
 */
static void
extend_neighbor_string(int i, int j, int s)
{
  int k;

  /* Link in the stone in the cyclic list. */
  int m = string[s].origini;
  int n = string[s].originj;
  next_stonei[i][j] = next_stonei[m][n];
  next_stonej[i][j] = next_stonej[m][n];
  PUSH_VALUE(next_stonei[m][n]);
  PUSH_VALUE(next_stonej[m][n]);
  next_stonei[m][n] = i;
  next_stonej[m][n] = j;
  
  /* Do we need to update the origin? */
  if (i < m || (i == m && j < n)) {
    PUSH_VALUE(string[s].origini);
    PUSH_VALUE(string[s].originj);
    string[s].origini = i;
    string[s].originj = j;
  }
  
  string_number[i][j] = s;

  /* The size of the string has increased by one. */
  PUSH_VALUE(string[s].size);
  string[s].size++;

  /* The place of the new stone is no longer a liberty. */
  remove_liberty(s, i, j);

  /* Mark old neighbors of the string. */
  string_mark++;
  for (k=0; k<string[s].neighbors; k++)
    string[string[s].neighborlist[k]].mark = string_mark;

  /* Look at the neighbor locations of (i, j) for new liberties and/or
   * neighbor strings.
   */
  if (i > 0) {
    /* If we find a liberty, look two steps away to determine whether
     * this already is a liberty of s.
     */
    if (LIBERTY(i-1, j)) {
      if (!((i > 1 && STRING_AT_VERTEX(i-2, j, s))
	    || (j > 0 && STRING_AT_VERTEX(i-1, j-1, s))
	    || (j < board_size-1 && STRING_AT_VERTEX(i-1, j+1, s))))
	ADD_LIBERTY(s, i-1, j);
    }
    else if (UNMARKED_OPPONENT_STRING(s, i-1, j)) {
      int s2 = string_number[i-1][j];
      PUSH_VALUE(string[s].neighbors);
      ADD_NEIGHBOR(s, i-1, j);
      PUSH_VALUE(string[s2].neighbors);
      ADD_NEIGHBOR(s2, i, j);
      MARK_STRING(i-1, j);
    }
  }
  
  if (i < board_size-1) {
    if (LIBERTY(i+1, j)) {
      if (!((i < board_size-2 && STRING_AT_VERTEX(i+2, j, s))
	    || (j > 0 && STRING_AT_VERTEX(i+1, j-1, s))
	    || (j < board_size-1 && STRING_AT_VERTEX(i+1, j+1, s))))
	ADD_LIBERTY(s, i+1, j);
    }
    else if (UNMARKED_OPPONENT_STRING(s, i+1, j)) {
      int s2 = string_number[i+1][j];
      PUSH_VALUE(string[s].neighbors);
      ADD_NEIGHBOR(s, i+1, j);
      PUSH_VALUE(string[s2].neighbors);
      ADD_NEIGHBOR(s2, i, j);
      MARK_STRING(i+1, j);
    }
  }
  
  if (j > 0) {
    if (LIBERTY(i, j-1)) {
      if (!((j > 1 && STRING_AT_VERTEX(i, j-2, s))
	    || (i > 0 && STRING_AT_VERTEX(i-1, j-1, s))
	    || (i < board_size-1 && STRING_AT_VERTEX(i+1, j-1, s))))
	ADD_LIBERTY(s, i, j-1);
    }
    else if (UNMARKED_OPPONENT_STRING(s, i, j-1)) {
      int s2 = string_number[i][j-1];
      PUSH_VALUE(string[s].neighbors);
      ADD_NEIGHBOR(s, i, j-1);
      PUSH_VALUE(string[s2].neighbors);
      ADD_NEIGHBOR(s2, i, j);
      MARK_STRING(i, j-1);
    }
  }
  
  if (j < board_size-1) {
    if (LIBERTY(i, j+1)) {
      if (!((j < board_size-2 && STRING_AT_VERTEX(i, j+2, s))
	    || (i > 0 && STRING_AT_VERTEX(i-1, j+1, s))
	    || (i < board_size-1 && STRING_AT_VERTEX(i+1, j+1, s))))
	ADD_LIBERTY(s, i, j+1);
    }
    else if (UNMARKED_OPPONENT_STRING(s, i, j+1)) {
      int s2 = string_number[i][j+1];
      PUSH_VALUE(string[s].neighbors);
      ADD_NEIGHBOR(s, i, j+1);
      PUSH_VALUE(string[s2].neighbors);
      ADD_NEIGHBOR(s2, i, j);
/*      MARK_STRING(i, j+1);*/
    }
  }
}


/* Incorporate the string at (i, j) with the string s.
 */

static void
assimilate_string(int s, int i, int j)
{
  int k;
  int lasti, lastj;
  int s2 = string_number[i][j];
  string[s].size += string[s2].size;

  /* Walk through the s2 stones and change string number. Also pick up
   * the last stone in the cycle for later use.
   */
  FIRST_STONE(s2, i, j);
  do {
    PUSH_VALUE(string_number[i][j]);
    string_number[i][j] = s;
    lasti = i;
    lastj = j;
    NEXT_STONE(i, j);
  } while(!BACK_TO_FIRST_STONE(s2, i, j));

  /* Link the two cycles together. */
  {
    int m = string[s].origini;
    int n = string[s].originj;
    PUSH_VALUE(next_stonei[lasti][lastj]);
    PUSH_VALUE(next_stonej[lasti][lastj]);
    PUSH_VALUE(next_stonei[m][n]);
    PUSH_VALUE(next_stonej[m][n]);
    next_stonei[lasti][lastj] = next_stonei[m][n];
    next_stonej[lasti][lastj] = next_stonej[m][n];
    next_stonei[m][n] = string[s2].origini;
    next_stonej[m][n] = string[s2].originj;
    
    /* Do we need to update the origin? */
    if (string[s2].origini < m || (string[s2].origini == m
				   && string[s2].originj < n)) {
      string[s].origini = string[s2].origini;
      string[s].originj = string[s2].originj;
    }
  }

  /* Pick up the liberties of s2 that we don't already have.
   * It is assumed that the liberties of s have been marked before
   * this function is called.
   */
  if (string[s2].liberties <= MAX_LIBERTIES) {
    for (k=0; k<string[s2].liberties; k++) {
      int m = string[s2].libi[k];
      int n = string[s2].libj[k];
      if (UNMARKED_LIBERTY(m, n)) {
	ADD_AND_MARK_LIBERTY(s, m, n);
      }
    }
  }
  else {
    /* If s2 had too many liberties the above strategy wouldn't be
     * effective, since not all liberties are listed in
     * (libi[], libj[]). Moreover, the chain of stones for s2 is no
     * longer available (it has already been merged with s) so we
     * can't reconstruct the s2 liberties. Instead we capitulate and
     * rebuild the list of liberties for s (including the neighbor
     * strings assimilated so far) from scratch.
     */
    liberty_mark++;          /* Reset the mark. */
    string[s].liberties = 0; /* To avoid pushing the current list. */
    update_liberties(s);
  }

  /* Remove s2 as neighbor to the neighbors of s2 and instead add s if
   * they don't already have added it. Also add the neighbors of s2 as
   * neighbors of s, unless they already have been added. The already
   * known neighbors of s are assumed to have been marked before this
   * function is called.
   */
  for (k=0; k<string[s2].neighbors; k++) {
    int t = string[s2].neighborlist[k];
    remove_neighbor(t, s2);
    if (string[t].mark != string_mark) {
      PUSH_VALUE(string[t].neighbors);
      string[t].neighborlist[string[t].neighbors++] = s;
      string[s].neighborlist[string[s].neighbors++] = t;
      string[t].mark = string_mark;
    }
  }
}


/* Create a new string for the stone at (i, j) and assimilate all
 * friendly neighbor strings.
 */

static void
assimilate_neighbor_strings(int i, int j)
{
  int s;
  int color = p[i][j];

  /* Get the next free string number. */
  PUSH_VALUE(next_string);
  s = next_string++;
  string_number[i][j] = s;
  /* Set up a size one cycle for the string. */
  next_stonei[i][j] = i;
  next_stonej[i][j] = j;
  
  /* Set trivially known values and initialize the rest to zero. */
  string[s].color = color;
  string[s].size = 1;
  string[s].origini = i;
  string[s].originj = j;
  string[s].liberties = 0;
  string[s].neighbors = 0;

  /* Clear the marks. */
  liberty_mark++;
  string_mark++;

  /* Mark ourselves. */
  string[s].mark = string_mark;

  /* Look in each direction for
   *
   * 1. liberty: Add if not already visited.
   * 2. opponent string: Add it among our neighbors and us among its
   *    neighbors, unless already visited.
   * 3. friendly string: Assimilate.
   */
  if (i > 0) {
    if (UNMARKED_LIBERTY(i-1, j)) {
      ADD_AND_MARK_LIBERTY(s, i-1, j);
    }
    else if (UNMARKED_OPPONENT_STRING(s, i-1, j)) {
      ADD_NEIGHBOR(s, i-1, j);
      PUSH_VALUE(string[string_number[i-1][j]].neighbors);
      ADD_NEIGHBOR(string_number[i-1][j], i, j);
      MARK_STRING(i-1, j);
    }
    else if (UNMARKED_OWN_STRING(s, i-1, j)) {
      assimilate_string(s, i-1, j);
    }
  }

  if (i < board_size-1) {
    if (UNMARKED_LIBERTY(i+1, j)) {
      ADD_AND_MARK_LIBERTY(s, i+1, j);
    }
    else if (UNMARKED_OPPONENT_STRING(s, i+1, j)) {
      ADD_NEIGHBOR(s, i+1, j);
      PUSH_VALUE(string[string_number[i+1][j]].neighbors);
      ADD_NEIGHBOR(string_number[i+1][j], i, j);
      MARK_STRING(i+1, j);
    }
    else if (UNMARKED_OWN_STRING(s, i+1, j)) {
      assimilate_string(s, i+1, j);
    }
  }

  if (j > 0) {
    if (UNMARKED_LIBERTY(i, j-1)) {
      ADD_AND_MARK_LIBERTY(s, i, j-1);
    }
    else if (UNMARKED_OPPONENT_STRING(s, i, j-1)) {
      ADD_NEIGHBOR(s, i, j-1);
      PUSH_VALUE(string[string_number[i][j-1]].neighbors);
      ADD_NEIGHBOR(string_number[i][j-1], i, j);
      MARK_STRING(i, j-1);
    }
    else if (UNMARKED_OWN_STRING(s, i, j-1)) {
      assimilate_string(s, i, j-1);
    }
  }

  if (j < board_size-1) {
    if (UNMARKED_LIBERTY(i, j+1)) {
      ADD_AND_MARK_LIBERTY(s, i, j+1);
    }
    else if (UNMARKED_OPPONENT_STRING(s, i, j+1)) {
      ADD_NEIGHBOR(s, i, j+1);
      PUSH_VALUE(string[string_number[i][j+1]].neighbors);
      ADD_NEIGHBOR(string_number[i][j+1], i, j);
      MARK_STRING(i, j+1);
    }
    else if (UNMARKED_OWN_STRING(s, i, j+1)) {
      assimilate_string(s, i, j+1);
    }
  }
}


/* Suicide at (i, j). Remove the neighboring friendly strings.
 */

static void
incremental_commit_suicide(int i, int j, int color)
{
  if (i > 0 && p[i-1][j] == color)
    incremental_remove_string(string_number[i-1][j]);

  if (i < board_size-1 && p[i+1][j] == color)
    incremental_remove_string(string_number[i+1][j]);

  if (j > 0 && p[i][j-1] == color)
    incremental_remove_string(string_number[i][j-1]);

  if (j < board_size-1 && p[i][j+1] == color)
    incremental_remove_string(string_number[i][j+1]);
}


/* Play a move without legality checking. Suicide is allowed.
 */

void
incremental_play_move(int i, int j, int color)
{
  int other = OTHER_COLOR(color);
  int captured_stones = 0;
  int neighbor_allies = 0;
  int have_liberties = 0;
  int s = -1;
  
  if (!strings_initialized)
    incremental_board_init();
    
  /* Remove captured stones and check for suicide.*/
  if (i > 0) {
    if (p[i-1][j] == other && LIBERTIES(i-1, j) == 1)
      captured_stones += incremental_remove_string(string_number[i-1][j]);
    else if (LIBERTY(i-1, j) || (p[i-1][j] == color && LIBERTIES(i-1, j) > 1))
      have_liberties = 1;
  }

  if (i < board_size-1) {
    if (p[i+1][j] == other && LIBERTIES(i+1, j) == 1)
      captured_stones += incremental_remove_string(string_number[i+1][j]);
    else if (LIBERTY(i+1, j) || (p[i+1][j] == color && LIBERTIES(i+1, j) > 1))
      have_liberties = 1;
  }

  if (j > 0) {
    if (p[i][j-1] == other && LIBERTIES(i, j-1) == 1)
      captured_stones += incremental_remove_string(string_number[i][j-1]);
    else if (LIBERTY(i, j-1) || (p[i][j-1] == color && LIBERTIES(i, j-1) > 1))
      have_liberties = 1;
  }

  if (j < board_size-1) {
    if (p[i][j+1] == other && LIBERTIES(i, j+1) == 1)
      captured_stones += incremental_remove_string(string_number[i][j+1]);
    else if (LIBERTY(i, j+1) || (p[i][j+1] == color && LIBERTIES(i, j+1) > 1))
      have_liberties = 1;
  }

  /* No captures and no liberties -> suicide. */
  if (have_liberties == 0 && captured_stones == 0) {
    incremental_commit_suicide(i, j, color);
    return;
  }
  
  /* Put down the stone. */
  DO_ADD_STONE(i, j, color);

  /* Count the number of adjacent strings of my color and remove
   * (i, j) as liberty for the adjacent opponent strings.
   */
  string_mark++;
  if (i > 0 && p[i-1][j] != EMPTY && UNMARKED_STRING(i-1, j)) {
    if (p[i-1][j] == color) {
      neighbor_allies++;
      s = string_number[i-1][j];
    }
    else
      remove_liberty(string_number[i-1][j], i, j);
    MARK_STRING(i-1, j);
  }
  
  if (i < board_size-1 && p[i+1][j] != EMPTY && UNMARKED_STRING(i+1, j)) {
    if (p[i+1][j] == color) {
      neighbor_allies++;
      s = string_number[i+1][j];
    }
    else
      remove_liberty(string_number[i+1][j], i, j);
    MARK_STRING(i+1, j);
  }
  
  if (j > 0 && p[i][j-1] != EMPTY && UNMARKED_STRING(i, j-1)) {
    if (p[i][j-1] == color) {
      neighbor_allies++;
      s = string_number[i][j-1];
    }
    else
      remove_liberty(string_number[i][j-1], i, j);
    MARK_STRING(i, j-1);
  }
  
  if (j < board_size-1 && p[i][j+1] != EMPTY && UNMARKED_STRING(i, j+1)) {
    if (p[i][j+1] == color) {
      neighbor_allies++;
      s = string_number[i][j+1];
    }
    else
      remove_liberty(string_number[i][j+1], i, j);
/*    MARK_STRING(i, j+1);*/
  }

  /* Choose strategy depending on the number of friendly neighbors. */
  if (neighbor_allies == 0)
    create_new_string(i, j);
  else if (neighbor_allies == 1) {
    assert(s >= 0);
    extend_neighbor_string(i, j, s);
  }
  else
    assimilate_neighbor_strings(i, j);

  /* Check whether this move was a ko capture and if so set (ko_i, ko_j).
   * No need to push (ko_i, ko_j) on the stack, this has been done
   * much earlier.
   */
  s = string_number[i][j];
  if (string[s].liberties == 1
      && string[s].size == 1
      && captured_stones == 1) {
    ko_i = string[s].libi[0];
    ko_j = string[s].libj[0];
    hashdata_set_ko(&hashdata, ko_i, ko_j);
  }
}


/* Restore board state to the position before the last move. This is
 * accomplished by popping everything that was stored on the stacks
 * since the last BEGIN_CHANGE_RECORD().
 */

void
incremental_undo_move()
{
  assert(strings_initialized);
  assert(change_stack_pointer - change_stack <= STACK_SIZE);
  if (0)
    gprintf("Change stack size = %d\n", change_stack_pointer - change_stack);
  POP_MOVE();
  POP_VERTICES();
}


/* Find the liberties a move of the given color at (i, j) would have,
 * excluding possible captures, by traversing all adjacent friendly
 * strings. This is a fallback used by incremental_approxlib() when a
 * faster algorithm can't be used.
 */

static int
slow_approxlib(int i, int j, int color, int maxlib, int *libi, int *libj)
{
  int libs = 0;
  int k;

  liberty_mark++;
  MARK_LIBERTY(i, j);
  string_mark++;
  for (k=0; k<4; k++) {
    int di = deltai[k];
    int dj = deltaj[k];
    if (ON_BOARD(i+di, j+dj)) {
      if (UNMARKED_LIBERTY(i+di, j+dj)) {
	if (libi) {
	  libi[libs] = i+di;
	  libj[libs] = j+dj;
	}
	libs++;
	if (libs == maxlib)
	  return libs;
	MARK_LIBERTY(i+di, j+dj);
      }
      else if (p[i+di][j+dj] == color
	       && UNMARKED_STRING(i+di, j+dj)) {
	int s = string_number[i+di][j+dj];
	int m, n;
	FIRST_STONE(s, m, n);
	do {
	  int l;
	  for (l=0; l<4; l++) {
	    int dm = deltai[l];
	    int dn = deltaj[l];
	    if (ON_BOARD(m+dm, n+dn) && UNMARKED_LIBERTY(m+dm, n+dn)) {
	      if (libi) {
		libi[libs] = m+dm;
		libj[libs] = n+dn;
	      }
	      libs++;
	      if (libs == maxlib)
		return libs;
	      MARK_LIBERTY(m+dm, n+dn);
	    }
	  }
	  
	  NEXT_STONE(m, n);
	} while (!BACK_TO_FIRST_STONE(s, m, n));
	MARK_STRING(i+di, j+dj);
      }
    }
  }
  return libs;
}


/* Count and return the number of liberties for the string at (i, j).
 * (i, j) must not be empty.
 */

int
incremental_countlib(int i, int j)
{
  assert(p[i][j] != EMPTY);
  
  if (!strings_initialized)
    incremental_board_init();

  /* We already know the number of liberties. Just look it up. */
  return string[string_number[i][j]].liberties;
}


/* Find the liberties of the string at (i, j). (i, j) must not be
 * empty. Found liberties are put in the (libi[], libj[]) arrays, but
 * never more than maxlib of them. The total number of liberties is
 * returned.
 */

int
incremental_findlib(int i, int j, int maxlib, int *libi, int *libj)
{
  int k;
  int libs;
  int s;
  
  assert(p[i][j] != EMPTY);
  assert(libi != NULL && libj != NULL);
  
  if (!strings_initialized)
    incremental_board_init();

  /* We already have the list of liberties and only need to copy it to
   * (libi[], libj[]).
   *
   * However, if the string has more than MAX_LIBERTIES liberties the
   * list is truncated and if maxlib is also larger than MAX_LIBERTIES
   * we have to traverse the stones in the string in order to find
   * where the liberties are.
   */
  s = string_number[i][j];
  libs = string[s].liberties;
  if (libs <= MAX_LIBERTIES || maxlib <= MAX_LIBERTIES) {
    /* The easy case, it suffices to copy liberty locations from the
     * incrementally updated list.
     */
    for (k=0; k<maxlib && k<libs; k++) {
      libi[k] = string[s].libi[k];
      libj[k] = string[s].libj[k];
    }
  }
  else {
    /* The harder case, where we have to traverse the stones in the
     * string. We don't have to check explicitly if we are back to
     * the start of the chain since we will run out of liberties
     * before that happens.
     */
    int m, n;
    liberty_mark++;
    for (k=0, FIRST_STONE(s, m, n); k<maxlib && k<libs; ) {
      if (m > 0 && UNMARKED_LIBERTY(m-1, n)) {
	libi[k] = m-1;
	libj[k++] = n;
	MARK_LIBERTY(m-1, n);
	if (k >= maxlib)
	  break;
      }
      
      if (m < board_size-1 && UNMARKED_LIBERTY(m+1, n)) {
	libi[k] = m+1;
	libj[k++] = n;
	MARK_LIBERTY(m+1, n);
	if (k >= maxlib)
	  break;
      }
      
      if (n > 0 && UNMARKED_LIBERTY(m, n-1)) {
	libi[k] = m;
	libj[k++] = n-1;
	MARK_LIBERTY(m, n-1);
	if (k >= maxlib)
	  break;
      }
      
      if (n < board_size-1 && UNMARKED_LIBERTY(m, n+1)) {
	libi[k] = m;
	libj[k++] = n+1;
	MARK_LIBERTY(m, n+1);
	if (k >= maxlib)
	  break;
      }
      /* We can't use this macro within the for statement, so we have
       * to put it at the end of the loop instead.
       */
      NEXT_STONE(m, n);
    }
  }
      
  return libs;
}


/* Find the liberties a stone of the given color would get if played
 * at (m, n), ignoring possible captures of opponent stones. (m, n)
 * must be empty. If libi!=NULL, the locations of up to maxlib
 * liberties are written into (libi[], libj[]). The number of
 * liberties found is returned. Liberties in excess of maxlib may or
 * may not be included in the returned number.
 */

int
incremental_approxlib(int i, int j, int color, int maxlib,
		      int *libi, int *libj)
{
  int k;
  int libs = 0;

  /* Either both NULL or neither NULL. */
  assert(((libi != NULL) ^ (libj != NULL)) == 0);
  
  if (!strings_initialized)
    incremental_board_init();

  /* Look for empty neighbors and the liberties of the adjacent
   * strings of the given color. The algorithm below won't work
   * correctly if any of the adjacent strings have more than
   * MAX_LIBERTIES liberties AND maxlib is larger than MAX_LIBERTIES.
   * If this might be the case, we use a more robust fallback.
   */
  if (maxlib > MAX_LIBERTIES)
    return slow_approxlib(i, j, color, maxlib, libi, libj);
  
  /* Start by marking (i, j) itself so it isn't counted among its own
   * liberties.
   */
  liberty_mark++;
  MARK_LIBERTY(i, j);
    
  if (i > 0) {
    if (UNMARKED_LIBERTY(i-1, j)) {
      if (libi != NULL && libs < maxlib) {
	libi[libs] = i-1;
	libj[libs] = j;
      }
      MARK_LIBERTY(i-1, j);
      libs++;
      /* Stop counting if we reach maxlib. */
      if (libs >= maxlib)
	return libs;
    }
    else if (p[i-1][j] == color) {
      int s = string_number[i-1][j];
      for (k=0; k<string[s].liberties; k++) {
	int ai = string[s].libi[k];
	int aj = string[s].libj[k];
	if (UNMARKED_LIBERTY(ai, aj)) {
	  if (libi != NULL && libs < maxlib) {
	    libi[libs] = ai;
	    libj[libs] = aj;
	  }
	  MARK_LIBERTY(ai, aj);
	  libs++;
	  if (libs >= maxlib)
	    return libs;
	}
      }
    }
  }
  
  if (i < board_size-1) {
    if (UNMARKED_LIBERTY(i+1, j)) {
      if (libi != NULL && libs < maxlib) {
	libi[libs] = i+1;
	libj[libs] = j;
      }
      MARK_LIBERTY(i+1, j);
      libs++;
      if (libs >= maxlib)
	return libs;
    }
    else if (p[i+1][j] == color) {
      int s = string_number[i+1][j];
      for (k=0; k<string[s].liberties; k++) {
	int ai = string[s].libi[k];
	int aj = string[s].libj[k];
	if (UNMARKED_LIBERTY(ai, aj)) {
	  if (libi != NULL && libs < maxlib) {
	    libi[libs] = ai;
	    libj[libs] = aj;
	  }
	  MARK_LIBERTY(ai, aj);
	  libs++;
	  if (libs >= maxlib)
	    return libs;
	}
      }
    }
  }
  
  if (j > 0) {
    if (UNMARKED_LIBERTY(i, j-1)) {
      if (libi != NULL && libs < maxlib) {
	libi[libs] = i;
	libj[libs] = j-1;
      }
      MARK_LIBERTY(i, j-1);
      libs++;
      if (libs >= maxlib)
	return libs;
    }
    else if (p[i][j-1] == color) {
      int s = string_number[i][j-1];
      for (k=0; k<string[s].liberties; k++) {
	int ai = string[s].libi[k];
	int aj = string[s].libj[k];
	if (UNMARKED_LIBERTY(ai, aj)) {
	  if (libi != NULL && libs < maxlib) {
	    libi[libs] = ai;
	    libj[libs] = aj;
	  }
	  MARK_LIBERTY(ai, aj);
	  libs++;
	  if (libs >= maxlib)
	    return libs;
	}
      }
    }
  }
  
  if (j < board_size-1) {
    if (UNMARKED_LIBERTY(i, j+1)) {
      if (libi != NULL && libs < maxlib) {
	libi[libs] = i;
	libj[libs] = j+1;
      }
      MARK_LIBERTY(i, j+1);
      libs++;
      if (libs >= maxlib)
	return libs;
    }
    else if (p[i][j+1] == color) {
      int s = string_number[i][j+1];
      for (k=0; k<string[s].liberties; k++) {
	int ai = string[s].libi[k];
	int aj = string[s].libj[k];
	if (UNMARKED_LIBERTY(ai, aj)) {
	  if (libi != NULL && libs < maxlib) {
	    libi[libs] = ai;
	    libj[libs] = aj;
	  }
	  MARK_LIBERTY(ai, aj);
	  libs++;
	  if (libs >= maxlib)
	    return libs;
	}
      }
    }
  }

  return libs;
}


/* Determine whether a move by color at (m, n) might be a self atari.
 * This function is sloppy in that it only does a quick check for two
 * liberties and might miss certain cases. The only guarantee is that
 * if it returns zero, it cannot be a self atari.
 */

int
incremental_sloppy_self_atari(int m, int n, int color)
{
  /* number of empty neighbors */
  int trivial_liberties = 0;
  /* number of captured opponent strings */
  int captures = 0;
  /* Whether there is a friendly neighbor with a spare liberty. If it
   * has more than one spare liberty we immediately return 0.
   */
  int far_liberties = 0;

  /* Clear string mark. */
  string_mark++;
  
  if (m > 0) {
    if (p[m-1][n] == EMPTY)
      trivial_liberties++;
    else if (p[m-1][n] == color) {
      if (LIBERTIES(m-1, n) > 2)
	return 0;
      if (LIBERTIES(m-1, n) == 2)
	far_liberties = 1;
    }
    else if (LIBERTIES(m-1, n) == 1 && UNMARKED_STRING(m-1, n)) {
      captures++;
      MARK_STRING(m-1, n);
    }
  }

  if (m < board_size-1) {
    if (p[m+1][n] == EMPTY)
      trivial_liberties++;
    else if (p[m+1][n] == color) {
      if (LIBERTIES(m+1, n) > 2)
	return 0;
      if (LIBERTIES(m+1, n) == 2)
	far_liberties = 1;
    }
    else if (LIBERTIES(m+1, n) == 1 && UNMARKED_STRING(m+1, n)) {
      captures++;
      MARK_STRING(m+1, n);
    }
  }

  if (n > 0) {
    if (p[m][n-1] == EMPTY)
      trivial_liberties++;
    else if (p[m][n-1] == color) {
      if (LIBERTIES(m, n-1) > 2)
	return 0;
      if (LIBERTIES(m, n-1) == 2)
	far_liberties = 1;
    }
    else if (LIBERTIES(m, n-1) == 1 && UNMARKED_STRING(m, n-1)) {
      captures++;
      MARK_STRING(m, n-1);
    }
  }

  if (n < board_size-1) {
    if (p[m][n+1] == EMPTY)
      trivial_liberties++;
    else if (p[m][n+1] == color) {
      if (LIBERTIES(m, n+1) > 2)
	return 0;
      if (LIBERTIES(m, n+1) == 2)
	far_liberties = 1;
    }
    else if (LIBERTIES(m, n+1) == 1 && UNMARKED_STRING(m, n+1)) {
      captures++;
      MARK_STRING(m, n+1);
    }
  }

  /* Each captured string is guaranteed to produce at least one
   * liberty. These are disjoint from both trivial liberties and far
   * liberties. The two latter may however coincide.
   */
  
  if (trivial_liberties + captures >= 2)
    return 0;

  if (far_liberties + captures >= 2)
    return 0;

  return 1;
}


/* Report the number of stones in the string at (i, j).
 */

int
incremental_countstones(int i, int j)
{
  if (!strings_initialized)
    incremental_board_init();

  return string[string_number[i][j]].size;
}


/* Find the stones of the string at (m, n). (m, n) must not be
 * empty. The locations of up to maxstones stones are written into
 * (stonei[], stonej[]). The full number of stones is returned.
 */

int
incremental_findstones(int m, int n, int maxstones, int *stonei, int *stonej)
{
  int s;
  int size;
  int i, j;
  int k;
  
  if (!strings_initialized)
    incremental_board_init();

  s = string_number[m][n];
  size = string[s].size;
  
  /* Traverse the stones of the string, by following the cyclic chain. */
  FIRST_STONE(s, i, j);
  for (k=0; k<maxstones && k<size; k++) {
    stonei[k] = i;
    stonej[k] = j;
    NEXT_STONE(i, j);
  }

  return size;
}


  
/* Find the neighbors of the string at (i, j). Return the number of
 * them in *adj, their coordinates in (adji[], adjj[]), their sizes in
 * adjsize[], and their number of liberties in adjlib[].
 */

void 
incremental_chainlinks(int i, int j, int *adj, int adji[MAXCHAIN],
		       int adjj[MAXCHAIN], int adjsize[MAXCHAIN],
		       int adjlib[MAXCHAIN])
{
  struct string_data *s, *t;
  int k;

  ASSERT(p[i][j] != EMPTY, i, j);

  if (!strings_initialized)
    incremental_board_init();

  /* We already have the list ready, just copy it and fill in the
   * desired information.
   */
  s = &string[string_number[i][j]];
  for (k=0; k<s->neighbors; k++) {
    t = &string[s->neighborlist[k]];
    adji[k] = t->origini;
    adjj[k] = t->originj;
    adjsize[k] = t->size;
    adjlib[k] = t->liberties;
  }
  *adj = s->neighbors;
}


/* Find the neighbors of the string at (i, j), which have exactly lib
 * liberties. Return the number of them in *adj and their coordinates
 * in (adji[], adjj[]).
 */

void 
incremental_chainlinks2(int i, int j, int *adj, int adji[MAXCHAIN],
			int adjj[MAXCHAIN], int lib)
{
  struct string_data *s, *t;
  int k;

  ASSERT(p[i][j] != EMPTY, i, j);

  if (!strings_initialized)
    incremental_board_init();

  /* We already have the list ready, just copy the strings with the
   * right number of liberties.
   */
  *adj = 0;
  s = &string[string_number[i][j]];
  for (k=0; k<s->neighbors; k++) {
    t = &string[s->neighborlist[k]];
    if (t->liberties == lib) {
      adji[*adj] = t->origini;
      adjj[*adj] = t->originj;
      (*adj)++;
    }
  }
}


/* Return the origin of the string at (i, j) as (*origini, *originj).
 */

void
incremental_find_origin(int i, int j, int *origini, int *originj)
{
  struct string_data *s;

  assert(p[i][j] != EMPTY);

  if (!strings_initialized)
    incremental_board_init();
  
  s = &string[string_number[i][j]];
  *origini = s->origini;
  *originj = s->originj;
}


/*
 * Returns true if (ai, aj) is next to a stone in the string at (si, sj).
 */

int
incremental_neighbor_of(int ai, int aj, int si, int sj)
{
  int s;

  assert(p[si][sj] != EMPTY);

  if (!strings_initialized)
    incremental_board_init();

  s = string_number[si][sj];
  
  if (ai > 0
      && p[ai-1][aj] == p[si][sj]
      && string_number[ai-1][aj] == s)
    return 1;

  if (ai < board_size-1
      && p[ai+1][aj] == p[si][sj]
      && string_number[ai+1][aj] == s)
    return 1;

  if (aj > 0
      && p[ai][aj-1] == p[si][sj]
      && string_number[ai][aj-1] == s)
    return 1;

  if (aj < board_size-1
      && p[ai][aj+1] == p[si][sj]
      && string_number[ai][aj+1] == s)
    return 1;
  
  return 0;
}


/*
 * Return true if the move (i,j) by (color) is a ko capture
 * (whether capture is legal on this move or not).
 *
 * A move is a ko capture iff
 * 1. All neighbors are opponent stones.
 * 2. The number of captured stones is exactly one.
 */

int
incremental_is_ko(int i, int j, int color)
{
  int other = OTHER_COLOR(color);
  int captures = 0;
  
  if (!strings_initialized)
    incremental_board_init();
  
  if (i > 0) {
    if (p[i-1][j] != other)
      return 0;
    else if (LIBERTIES(i-1, j) == 1) {
      captures += string[string_number[i-1][j]].size;
      if (captures > 1)
	return 0;
    }
  }
  
  if (i < board_size-1) {
    if (p[i+1][j] != other)
      return 0;
    else if (LIBERTIES(i+1, j) == 1) {
      captures += string[string_number[i+1][j]].size;
      if (captures > 1)
	return 0;
    }
  }
  
  if (j > 0) {
    if (p[i][j-1] != other)
      return 0;
    else if (LIBERTIES(i, j-1) == 1) {
      captures += string[string_number[i][j-1]].size;
      if (captures > 1)
	return 0;
    }
  }
  
  if (j < board_size-1) {
    if (p[i][j+1] != other)
      return 0;
    else if (LIBERTIES(i, j+1) == 1) {
      captures += string[string_number[i][j+1]].size;
      if (captures > 1)
	return 0;
    }
  }

  return (captures == 1);
}


/* Sets the array mx equal to mark for every element of the string at (i,j). */

void
incremental_mark_string(int m, int n, char mx[MAX_BOARD][MAX_BOARD], char mark)
{
  int i = m;
  int j = n;
  do {
    mx[i][j] = mark;
    NEXT_STONE(i, j);
  } while (i != m || j != n);
}

/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
