/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 and 2000 by the Free Software Foundation.      *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* ---------------------------------------------------------------- *
 * gnugo.h
 *	This file contains the public interface to the GNU Go engine.
 * ---------------------------------------------------------------- */


#ifndef _GNUGO_H_
#define _GNUGO_H_


#include <stdio.h>

#include "sgftree.h"


/* interface.c */
/* Initialize the whole thing. Should be called once. */
void init_gnugo(float memory);


/* ================================================================ */
/*                some public macros used everywhere                */
/* ================================================================ */


/* Colors */
#define EMPTY        0
#define WHITE        1
#define BLACK        2
#define GRAY_BORDER  3
#define WHITE_BORDER 4
#define BLACK_BORDER 5

#define OTHER_COLOR(color)  (WHITE+BLACK-(color))


/* Group statuses */
#define DEAD        0
#define ALIVE       1
#define CRITICAL    2 
#define UNKNOWN     3
#define UNCHECKED   4
#define MAX_DRAGON_STATUS 4	/* used to size an array in matchpat.c */

#define CAN_THREATEN_ATTACK  1
#define CAN_THREATEN_DEFENSE 2

/* Dragon safety values. DEAD, ALIVE, and CRITICAL are reused. */
#define INESSENTIAL     3
#define TACTICALLY_DEAD 4
#define WEAK            5
#define WEAKLY_ALIVE    6
#define ALIVE_IN_SEKI   7
#define STRONGLY_ALIVE  8
#define INVINCIBLE      9
#define INSUBSTANTIAL   10

/* 
 * It is assumed in reading a ladder if stackp >= depth that
 * as soon as a bounding stone is in atari, the string is safe.
 * It is used similarly at other places in reading.c to implement simplifying
 * assumptions when stackp is large. DEPTH is the default value of depth.
 *
 * Unfortunately any such scheme invites the ``horizon effect.'' Increasing
 * DEPTH will make the program stronger and slower.
 *
 */

/* Tactical reading using C functions */
#define DEPTH                16
#define BRANCH_DEPTH         13
#define BACKFILL_DEPTH       12
#define BACKFILL2_DEPTH       5
#define SUPERSTRING_DEPTH     7
#define FOURLIB_DEPTH         7

/* Pattern based reading */
#define AA_DEPTH              6
#define KO_DEPTH              8
#define OWL_DISTRUST_DEPTH    6
#define OWL_BRANCH_DEPTH      8
#define OWL_READING_DEPTH    20
#define OWL_NODE_LIMIT      600

/* Eye reading */
#define LIFE_EYESIZE         10


/* ================================================================ */
/*                        Board manipulation                        */
/* ================================================================ */


/* Board sizes */
#define MIN_BOARD     3		/* minimum supported board size */
#define MAX_BOARD    21         /* maximum supported board size */


#define MAX_HANDICAP  9		/* maximum supported handicap   */


/* This type is used to store each piece on the board.
 *
 * On a 486, char is best, since the time taken to push and pop
 * becomes significant otherwise. On other platforms, an int may
 * be better, e.g. if memcpy() is particularly fast, or if
 * character access is very slow.
 */

typedef unsigned char Intersection;


typedef struct {
  int          boardsize;
  Intersection board[MAX_BOARD][MAX_BOARD];
  int          ko_i;
  int          ko_j;

  int          white_captured;
  int          black_captured;
} Position;

void gnugo_clear_position(Position *pos, int boardsize);
void gnugo_copy_position(Position *to, Position *from);
void gnugo_add_stone(Position *pos, int i, int j, int color);
void gnugo_remove_stone(Position *pos, int i, int j);
void gnugo_play_move(Position *pos, int i, int j, int color);
int  gnugo_is_legal(Position *pos, int i, int j, int color);
int  gnugo_is_suicide(Position *pos, int i, int j, int color);

int  gnugo_placehand(Position *pos, int handicap);
int  gnugo_sethand(Position *pos, int handicap, SGFNode *root);
void gnugo_recordboard(Position *pos, SGFNode *node);

int  gnugo_genmove(Position *pos, int *i, int *j, int color,
		   int move_number);

int  gnugo_attack(Position *pos, int m, int n, int *i, int *j);
int  gnugo_find_defense(Position *pos, int m, int n, int *i, int *j);

void gnugo_who_wins(Position *pos, int color, float komi, FILE *outfile);
void gnugo_evaluate_territory(Position *pos, 
			      int *white_terri, int *black_terri,
			      int *white_dead, int *black_dead);

void gnugo_force_to_globals(Position *pos);
void gnugo_examine_position(Position *pos, int color, int how_much);


/* ================================================================ */
/*                           Game handling                          */
/* ================================================================ */


typedef struct {
  int       handicap;
  float     komi;

  Position  position;
  int       move_number;
  int       to_move;		/* whose move it currently is */
  SGFTree   moves;		/* The moves in the game. */

  int       seed;		/* random seed */
  int       computer_player;	/* BLACK, WHITE, or EMPTY (used as BOTH) */

  char      outfilename[128];	/* Trickle file */
  FILE      *outfile;
} Gameinfo;

void gameinfo_clear(Gameinfo *ginfo, int boardsize);
void gameinfo_print(Gameinfo *ginfo);
void gameinfo_play_move(Gameinfo *ginfo, int i, int j, int color);


/* ================================================================ */
/*                           global variables                       */
/* ================================================================ */


/* Miscellaneous debug options. */
extern int quiet;		/* Minimal output. */
extern int verbose;		/* Bore the opponent. */
extern int allpats;		/* generate all patterns, even small ones */
extern int printworms;		/* print full data on each string */
extern int printmoyo;		/* print moyo board each move */
extern int printdragons;	/* print full data on each dragon */
extern int printboard;		/* print board each move */
extern int showstack;		/* debug stack pointer */
extern int showstatistics;	/* print statistics */
extern int profile_patterns;	/* print statistics of pattern usage */


/* debug flag bits */
/* NOTE : can specify -d0x... */
#define DEBUG_INFLUENCE           0x0001
#define DEBUG_EYES                0x0002
#define DEBUG_OWL                 0x0004
#define DEBUG_ESCAPE              0x0008
#define DEBUG_MATCHER             0x0010
#define DEBUG_DRAGONS             0x0020
#define DEBUG_SEMEAI              0x0040
#define DEBUG_LOADSGF             0x0080
#define DEBUG_HELPER              0x0100
#define DEBUG_READING             0x0200
#define DEBUG_WORMS               0x0400 /* unused */
#define DEBUG_MOVE_REASONS        0x0800
#define DEBUG_OWL_PERFORMANCE     0x1000
#define DEBUG_LIFE                0x2000
#define DEBUG_FILLLIB             0x4000
#define DEBUG_READING_PERFORMANCE 0x8000

/* hash flag bits 
 *
 * Regarding HASH_DEFAULT:
 * Hashing all functions saves time, but wastes table space, which is
 * bad when the reading is complicated. HASH_DEFAULT is a compromise. 
 */

#define HASH_FIND_DEFENSE 0x0001  /* NOTE : can specify -d0x... */
#define HASH_DEFEND1      0x0002
#define HASH_DEFEND2      0x0004
#define HASH_DEFEND3      0x0008
#define HASH_DEFEND4      0x0010
#define HASH_ATTACK       0x0020
#define HASH_ATTACK2      0x0040
#define HASH_ATTACK3      0x0080
#define HASH_OWL_ATTACK   0x0100
#define HASH_OWL_DEFEND   0x0200
#define HASH_NOTHING      0
#define HASH_ALL          0xffff
#define HASH_DEFAULT      (HASH_ATTACK | HASH_FIND_DEFENSE\
			   | HASH_OWL_ATTACK | HASH_OWL_DEFEND)

extern int debug;		/* debug flags */
extern int hashflags;		/* hash flags */
extern int life;                /* use "life" reading code */
extern int life_eyesize;        /* max eye size for life code */
extern int fusekidb;            /* use fuseki database */
extern int josekidb;            /* use joseki database */
extern int level;		/* controls depth of reading */
extern int urgent;              /* urgent move on board */
extern int kothreat_needed;     /* ko threat needed */
extern int noinhibit;		/* estimate moyos */
extern int showtime;		/* print genmove time */
extern int analyzerflag;

/* Reading parameters */
extern int nominal_depth;	      /* deep reading cutoff, nominal value */
extern int nominal_backfill_depth;    /* deep reading cutoff, nominal value */
extern int nominal_backfill2_depth;   /* deep reading cutoff, nominal value */
extern int nominal_superstring_depth; /* deep reading cutoff, nominal value */
extern int nominal_fourlib_depth;   /* deep reading cutoff, nominal value */
extern int nominal_ko_depth;	    /* deep reading cutoff, nominal value */
extern int nominal_branch_depth;    /* deep reading cutoff, nominal value */
extern int nominal_aa_depth;
extern int nominal_owl_distrust_depth;  
extern int nominal_owl_branch_depth;  
extern int nominal_owl_reading_depth; 
extern int nominal_owl_node_limit;    

extern int potential_moves[MAX_BOARD][MAX_BOARD];

extern int terri_eval[3];
extern int moyo_eval[3];

extern char *analyzerfile;

extern volatile int time_to_die;   /* set by signal handlers */


/* ================================================================ */
/*                 tracing and debugging functions                  */
/* ================================================================ */


/* Colors. */
#define GG_COLOR_BLACK   0
#define GG_COLOR_RED     1
#define GG_COLOR_GREEN   2
#define GG_COLOR_YELLOW  3
#define GG_COLOR_BLUE    4
#define GG_COLOR_MAGENTA 5
#define GG_COLOR_CYAN    6
#define GG_COLOR_WHITE   7

/* showbord.c */
void start_draw_board(void);
void draw_color_char(int m, int n, int c, int color);
void draw_char(int m, int n, int c);
void end_draw_board(void);
void showboard(int xo);  /* ascii rep. of board to stdout */

/* printutils.c */
void gprintf(const char *fmt, ...);
void mprintf(const char *fmt, ...);
const char *color_to_string(int color);
const char *location_to_string(int i, int j);
const char *status_to_string(int status);
double gg_gettimeofday(void);


/* influence.c */
void debug_influence_move(int i, int j);

#ifdef __GNUC__

/* gnuc allows variadic macros, so the tests can be done inline */
#define TRACE(fmt, args...) \
    do { if (verbose) gprintf(fmt, ##args); } while(0)
#define RTRACE(fmt, args...) \
    do { if (verbose >= 3) gprintf(fmt, ##args); } while(0)
#define VTRACE(fmt, args...) \
    do { if (verbose >= 4) gprintf(fmt, ##args); } while(0)
#define DEBUG(level, fmt, args...) \
    do { if ((debug & (level))) gprintf(fmt, ##args); } while(0)

#else

/* else we need fns to do these */
void TRACE(const char *fmt, ...);
void RTRACE(const char *fmt, ...);
void VTRACE(const char *fmt, ...);
void DEBUG(int level, const char *fmt, ...);

#endif

/* genmove.c */
#define EXAMINE_WORMS               1
#define EXAMINE_INITIAL_INFLUENCE   2
#define EXAMINE_DRAGONS_WITHOUT_OWL 3
#define EXAMINE_DRAGONS             4
#define EXAMINE_OWL_REASONS         5
#define EXAMINE_MOYO                6
#define EXAMINE_ESCAPE_POTENTIAL    7
#define EXAMINE_INITIAL_INFLUENCE2  8

#define EXAMINE_ALL                 99

void reset_engine(void);
void examine_position(int color, int how_much);


/* ================================================================ */
/*                         statistics functions                     */
/* ================================================================ */


/* These are mostly used for GTP examination. */
void reset_life_node_counter(void);
int  get_life_node_counter(void);
void reset_owl_node_counter(void);
int  get_owl_node_counter(void);
void reset_reading_node_counter(void);
int  get_reading_node_counter(void);
void reset_trymove_counter(void);
int  get_trymove_counter(void);


/* ================================================================ */
/*                         Low level functions                      */
/* ================================================================ */


/* board.c */
/* General board handling. */
void clear_board(void);
void setup_board(Intersection new_p[MAX_BOARD][MAX_BOARD], int koi, int koj,
		 int w_captured, int b_captured);

/* Putting stones on the board.. */
void add_stone(int i, int j, int color);
void remove_stone(int i, int j);
void remove_string(int i, int j);
void play_move(int i, int j, int color);
int is_legal(int i, int j, int color);
int is_suicide(int i, int j, int color);
int trymove(int i, int j, int color, const char *message, int k, int l);
int tryko(int i, int j, int color, const char *message);
void popgo(void);

/* utils.c */
void change_dragon_status(int x, int y, int status);
void count_territory( int *white, int *black);
void evaluate_territory(int *white_terri, int *black_terri, 
			int *white_dead, int *black_dead);
int same_dragon(int ai, int aj, int bi, int bj);
int is_worm_origin(int wi, int wj, int i, int j);


/* high-level routine to generate the best move for the given color */
int genmove(int *i, int *j, int color);

/* Basic information gathering. */
/* worms.c */
void build_worms(void);
void make_worms(void);
int is_same_worm(int ai, int aj, int bi, int bj);

void make_dragons(int color, int stop_before_owl);
void show_dragons(void);
int dragon_status(int i, int j);
int matcher_status(int i, int j);

/* data concerning moyo */
extern int terri_test[3];
extern int moyo_test[3];
extern int very_big_move[3];

/* moyo functions */
int make_moyo(int color);
void print_moyo(int color);

/* FIXME PRE3.0: This is in moyo.c, but should be somewhere else. */
void who_wins(int color, float fkomi, FILE* stdwhat);

/* debugging functions */
void prepare_pattern_profiling(void);
void report_pattern_profiling(void);

/* sgffile.c */
void sgffile_move_made(int i, int j, int color, int value);
void sgffile_put_stone(int i, int j, int color);

int  sgffile_open_file(const char *);
int  sgffile_close_file(void);
int  sgffile_write_gameinfo(Gameinfo *gameinfo, const char *);
void sgffile_write_comment(const char *comment);
void sgffile_printboard(int next);
void sgffile_recordboard(SGFNode *node);

void decidestring(int i, int j, const char *sgf_output);
void decidedragon(int i, int j, const char *sgf_output);
void decidesemeai(int ai, int aj, int bi, int bj, const char *sgf_output);
void decideposition(int color, const char *sgf_output);
void decideeye(int m, int n, const char *sgf_output);

void load_sgf_header(SGFNode *head, Gameinfo *gameinfo);
int play_sgf_tree(SGFNode *head, Gameinfo *gameinfo, const char *untilstr);

/* FIXME PRE3.0: Better name for these? Also: See if play_sgf_tree makes them obsolete. */
int sgfPlayNode(SGFNode *node, Gameinfo *gameinfo, int to_move);
int sgfPlayTree(SGFNode **root, Position *position, int *until);


#endif  /* _GNUGO_H_ */


/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
