/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 and 2000 by the Free Software Foundation.      *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <math.h>

#include "liberty.h"
#include "sgftree.h"
#include "analyze.h"


#define MAXTRY 400

/* Return one if x doesn't equal position_number and 0 otherwise.
 * After using this macro x will always have the value
 * position_number.
 */
#define NEEDS_UPDATE(x) (x != position_number ? (x = position_number, 1) : 0)

static int get_level(int *level);
static void look_for_kos(int color);

void sgfShowConsideredMoves(void);


/* Reset some things in the engine. 
 *
 * This prepares the hash table for the reading code for use.  It
 * should be called when we start examine a new position.  
 */

void
reset_engine()
{
  static int node_limits[10] = {500, 450, 400, 325, 275,
				200, 150, 100, 75, 50};
  /* Initialize things for hashing of positions. */
  reading_cache_clear();
  hashdata_recalc(&hashdata, p, ko_i, ko_j);

  /* Prepare our table of move reasons. */
  clear_move_reasons();

  /* Set up depth values. Level 10 should be more or less the same as level 9
   * except that the superstring technology is used in the reading code. This
   * makes us about 10% slower.
   */
  if (level < 10) {
    depth               = max(6, nominal_depth - 9 + level);
    ko_depth            = max(1, nominal_ko_depth - 9 + level);
    backfill_depth      = max(2, nominal_backfill_depth - 9 + level);
    backfill2_depth     = max(1, nominal_backfill2_depth - 9 + level);
    superstring_depth   = 0 ;
    branch_depth        = max(3, nominal_branch_depth - 9 + level);
    if (level < 9)
      fourlib_depth     = max(1, nominal_fourlib_depth - 8 + level);
    else
      fourlib_depth     = max(1, nominal_fourlib_depth - 9 + level);
    aa_depth            = max(0, nominal_aa_depth - 10 + level);
    owl_distrust_depth  = max(1, nominal_owl_distrust_depth - 5 + (level+1)/2);
    owl_branch_depth    = max(2, nominal_owl_branch_depth - 5 + (level+1)/2);
    owl_reading_depth   = max(5, nominal_owl_reading_depth - 5 + (level+1)/2);
    owl_node_limit      = (nominal_owl_node_limit * node_limits[9 - level] /
			   node_limits[0]);
    owl_node_limit      = max(20, owl_node_limit);
    urgent              = 0;
    kothreat_needed     = 0;
  }
  else {
    depth               = max(6, nominal_depth - 10 + level);
    ko_depth            = max(1, nominal_ko_depth - 10 + level);
    backfill_depth      = max(2, nominal_backfill_depth - 10 + level);
    backfill2_depth     = max(1, nominal_backfill2_depth - 10 + level);
    superstring_depth   = max(1, nominal_superstring_depth - 10 + level);
    branch_depth        = max(3, nominal_branch_depth - 10 + level);
    fourlib_depth       = max(1, nominal_fourlib_depth - 10 + level);
    aa_depth            = max(0, nominal_aa_depth - 10 + level);
    owl_distrust_depth  = max(1, nominal_owl_distrust_depth - 5 + level/2);
    owl_branch_depth    = max(2, nominal_owl_branch_depth - 5 + level/2);
    owl_reading_depth   = max(5, nominal_owl_reading_depth - 5 + level/2);
    if (level == 10)
      owl_node_limit    = nominal_owl_node_limit;
    else
      owl_node_limit    = nominal_owl_node_limit * pow(1.5, -10 + level);
    urgent              = 0;
    kothreat_needed     = 0;
  }
}

/*
 * Examine the position and try to gather as much information as possible.
 * This is used mainly for move generation, but could also be called
 * for debugging purposes (decidestring, etc).
 *
 * The parameter how_much tells us how much of the work we have to do.
 * For move generation we have to do it all.  For debugging we can 
 * sometimes stop a little earlier.
 */

void
examine_position(int color, int how_much)
{
  int save_verbose;
  int m, n;

  /* Position numbers for which various examinations were last made. */
  static int worms_examined = -1;
  static int initial_influence_examined = -1;
  static int dragons_examined_without_owl = -1;
  static int dragons_examined = -1;
  static int owl_reasons_examined = -1;
  static int moyo_examined = -1;
  static int initial_influence2_examined = -1;
  
  /* Don't print reading traces during make_worms and make_dragons unless 
   * the user really wants it (verbose == 3). 
   */
  save_verbose = verbose;
  if ((verbose == 1) || (verbose == 2))
    --verbose;

  if (NEEDS_UPDATE(worms_examined))
    make_worms();
  if (how_much == EXAMINE_WORMS) {
    verbose = save_verbose;
    return;
  }

  if (stones_on_board(BLACK | WHITE) != 0) {
    if (NEEDS_UPDATE(initial_influence_examined))
      compute_initial_influence(color, 0);
    if (how_much == EXAMINE_INITIAL_INFLUENCE) {
      verbose = save_verbose;
      return;
    }

    if (how_much == EXAMINE_DRAGONS_WITHOUT_OWL) {
      if (NEEDS_UPDATE(dragons_examined_without_owl))
	make_dragons(color, 1);
      verbose = save_verbose;
      return;
    }
    
    if (NEEDS_UPDATE(dragons_examined))
      make_dragons(color, 0);
    if (how_much == EXAMINE_DRAGONS) {
      verbose = save_verbose;
      return;
    }

    if (NEEDS_UPDATE(owl_reasons_examined))
      owl_reasons(color);
    if (how_much == EXAMINE_OWL_REASONS) {
      verbose = save_verbose;
      return;
    }
  }
  else if (how_much == EXAMINE_INITIAL_INFLUENCE
	   || how_much == EXAMINE_DRAGONS
	   || how_much == EXAMINE_OWL_REASONS) {
    verbose = save_verbose;
    return;
  }
  
  verbose = save_verbose;

  /* Map the moyos for both sides.
   * FIXME POST3.0: Probably the make_moyo() call should go into
   * make_dragons() at some point.
   */
  if (NEEDS_UPDATE(moyo_examined))
    make_moyo(color);
  if (how_much == EXAMINE_MOYO)
    return;

  if (printworms) {
    show_dragons();
  }
  else if (verbose) {
    for (m=0;m<board_size;m++)
      for (n=0;n<board_size;n++)
	if ((p[m][n])
	    && (worm[m][n].origini==m) 
	    && (worm[m][n].originj==n) 
	    && (worm[m][n].cutstone)) 
        {
	  TRACE("%scutting stones found at %m\n",
		worm[m][n].cutstone==2 ? "potential " : "", m,n); 
	}
  }	  

  if (NEEDS_UPDATE(initial_influence2_examined))
    compute_initial_influence(color, 1);
  if (how_much == EXAMINE_INITIAL_INFLUENCE2)
    return;
}


/* 
 * Generate computer move for COLOR.
 *
 * Return the generated move in (*i, *j).
 */
  
int
genmove(int *i, int *j, int color)
{
  float val;
  double t1=0., t2=0.;

  if (showtime)
    t1 = gg_gettimeofday();

  /* Prepare our table of moves considered. */
  memset(potential_moves, 0, sizeof(potential_moves));

  /* Reset all the statistics for each move. */
  stats.nodes = 0;
  stats.position_entered    = 0;
  stats.position_hits       = 0;
  stats.read_result_entered = 0;
  stats.read_result_hits    = 0;
  stats.hash_collisions     = 0;

  /* no move is found yet. */
  *i = -1;  
  *j = -1;  
  val = -1; 
  if (get_level(&level))
    fprintf(stderr, "level = %d\n", level);

  /* Prepare pattern matcher and reading code. */
  reset_engine();
  
  /* Find out information about the worms and dragons. */
  examine_position(color, EXAMINE_ALL);

  /*
   * Print some of the information if the user wants to.
   */
  /* compute_influence(); */
  if (printmoyo)
    print_moyo(color);
   
  if (printboard) {
    if (printboard == 1)
      fprintf(stderr,"\n          dragon_status display:\n\n");
    if (printboard == 2)
      fprintf(stderr,"\n          eye display:\n\n");
    showboard(printboard); 
    if (printboard == 1) {
      fprintf(stderr,"\n           owl_status display:\n\n");      
      showboard(3);
    }
  }

  assert(stackp == 0);

  /*
   * Ok, information gathering is complete. Now start to find some moves!
   */

  /* Try to find empty corner moves. */
  fuseki(color);
  assert(stackp == 0);

  /* pattern matcher */
  shapes(color);
  assert(stackp == 0);
  look_for_kos(color);
  {
    int ai, aj;
    int other = OTHER_COLOR(color);
    int aa_val;

    int save_verbose = verbose;
    if (verbose > 0)
      verbose--;

    if (save_verbose)
      gprintf("\nlooking for combination attacks ...\n");
    aa_val = atari_atari(color, &ai, &aj, save_verbose);
    if (aa_val)
      add_my_atari_atari_move(ai, aj, aa_val);
    aa_val = atari_atari(other, &ai, &aj, save_verbose);
    if (aa_val && safe_move(ai, aj, color))
      add_your_atari_atari_move(ai, aj, aa_val);
    verbose = save_verbose;
  }

  /* Review the move reasons and estimate move values. */
  if (review_move_reasons(i, j, &val, color)) {
    TRACE("Move generation likes %m with value %f\n", *i, *j, val);
    if (analyzerflag & ANALYZE_RECOMMENDED)
      sgfBoardText(0, *i, *j, "P");
  }
  assert(stackp == 0);
 
  /* If the move value is 6 or lower, we look for endgame patterns too. */
  if (val <= 6.0) {
    endgame_shapes(color);
    assert(stackp == 0);
    if (review_move_reasons(i, j, &val, color)) {
      TRACE("Move generation likes %m with value %f\n", *i, *j, val);
      if (analyzerflag & ANALYZE_RECOMMENDED)
	sgfBoardText(0, *i, *j, "P");
    }
    assert(stackp == 0);
  }
 
  
  /* If no move found yet, revisit any semeai and change the
   * status of the opponent group from DEAD to UNKNOWN, then 
   * run shapes and endgame_shapes again. This may turn up a move.
   */
  if (val < 0.0) {
    if (revise_semeai(color)) {
      shapes(color);
      endgame_shapes(color);
      if (review_move_reasons(i, j, &val, color)) {
	TRACE("Upon reconsideration move generation likes %m with value %f\n",
	      *i, *j, val); 
	if (analyzerflag & ANALYZE_RECOMMENDED)
	  sgfBoardText(0,*i,*j,"p");
      }
    }
  }
  
  if ((val < 0.0) && fill_liberty(i, j, color)) {
    val = 1.0;
    TRACE("Filling a liberty at %m\n", *i, *j);
    if (analyzerflag & ANALYZE_RECOMMENDED)
      sgfBoardText(0, *i, *j, "L");
  }

  /* If no move is found then pass. */
  if (val < 0) {
    TRACE("I pass.\n");
    *i = -1;
    *j = -1;
  } else
    TRACE("genmove() recommends %m with value %f\n", *i, *j, val);
  
  /* If statistics is turned on, this is the place to show it. */
  if (showstatistics) {
    /* FIXME: Is this needed? */
#if 0
    char text[120];
    sprintf(text, "Nodes:                %d\n", stats.nodes);
    sgfAddComment(0,text);
#endif

    printf("Nodes:                %d\n", stats.nodes);
    printf("Positions entered:    %d\n", stats.position_entered);
    printf("Position hits:        %d\n", stats.position_hits);
    printf("Read results entered: %d\n", stats.read_result_entered);
    printf("Read result hits:     %d\n", stats.read_result_hits);
    printf("Hash collisions:      %d\n", stats.hash_collisions);
  }

  if (showtime) {
    t2 = gg_gettimeofday();
    fprintf(stderr, "TIME to generate move: %f\n", t2-t1);
  }
  
  return val;
  
}  /* end genmove */




/* This is called for each move which has been considered. For
 * debugging purposes, we keep a table of all the moves we
 * have considered.
 */

void 
move_considered(int i, int j, int val)
{
  if ((val > potential_moves[i][j])) {
    potential_moves[i][j] = val;
  }
}


/* If there is a file with the name "level", reads it
 * each move and corrects the value of level.
 */

static int
get_level(int *level)
{
  char buffer[128];
  FILE *fp;
  const char filename[]="level";

  if ((fp=fopen(filename,"r"))==NULL)
    return 0;

  if (fgets(buffer, 128, fp)) {
    if (sscanf(buffer, "%d", level))
      return 1;
    else
      return 0;
  }
  else
    return 0;
}
  
  
/* See if there is a ko threat needed */

static void
look_for_kos(int color)
{
  int m, n;
  for (m=0; m<board_size-1; m++)
    for (n=0; n<board_size-1; n++)
      if ((worm[m][n].origini == m)
	  && (worm[m][n].originj == n)
	  && (worm[m][n].attacki != -1)
	  && (worm[m][n].defendi != -1))
	if (((worm[m][n].color == color)
	    && (worm[m][n].defend_code == 3)
	     && !is_legal(worm[m][n].defendi, worm[m][n].defendj, color))
	    || ((worm[m][n].color == OTHER_COLOR(color))
		&& (worm[m][n].attack_code == 3)
		&& !is_legal(worm[m][n].attacki, worm[m][n].attackj, color)))
	  if ((int) worm[m][n].effective_size > kothreat_needed)
	    kothreat_needed = (int) 2*worm[m][n].effective_size;
}


/* ================================================================ */
/*       Set up fixed placement handicap pieces for black side      */
/* ================================================================ */


/* Handicap pieces are set up according to the following diagrams:
 *  
 * 2 stones:                    3 stones:           
 *
 *   A B C D E F G H J	  	  A B C D E F G H J  
 * 9 . . . . . . . . . 9  	9 . . . . . . . . . 9
 * 8 . . . . . . . . . 8  	8 . . . . . . . . . 8
 * 7 . . + . . . X . . 7  	7 . . X . . . X . . 7
 * 6 . . . . . . . . . 6  	6 . . . . . . . . . 6
 * 5 . . . . + . . . . 5  	5 . . . . + . . . . 5
 * 4 . . . . . . . . . 4  	4 . . . . . . . . . 4
 * 3 . . X . . . + . . 3  	3 . . X . . . + . . 3
 * 2 . . . . . . . . . 2  	2 . . . . . . . . . 2
 * 1 . . . . . . . . . 1  	1 . . . . . . . . . 1
 *   A B C D E F G H J	  	  A B C D E F G H J  
 *   
 * 4 stones:                    5 stones:           
 *						     
 *   A B C D E F G H J	          A B C D E F G H J  
 * 9 . . . . . . . . . 9 	9 . . . . . . . . . 9
 * 8 . . . . . . . . . 8 	8 . . . . . . . . . 8
 * 7 . . X . . . X . . 7 	7 . . X . . . X . . 7
 * 6 . . . . . . . . . 6 	6 . . . . . . . . . 6
 * 5 . . . . + . . . . 5 	5 . . . . X . . . . 5
 * 4 . . . . . . . . . 4 	4 . . . . . . . . . 4
 * 3 . . X . . . X . . 3 	3 . . X . . . X . . 3
 * 2 . . . . . . . . . 2 	2 . . . . . . . . . 2
 * 1 . . . . . . . . . 1 	1 . . . . . . . . . 1
 *   A B C D E F G H J	          A B C D E F G H J  
 *  
 * 6 stones:                    7 stones:           
 *						     
 *   A B C D E F G H J	          A B C D E F G H J  
 * 9 . . . . . . . . . 9 	9 . . . . . . . . . 9
 * 8 . . . . . . . . . 8 	8 . . . . . . . . . 8
 * 7 . . X . . . X . . 7 	7 . . X . . . X . . 7
 * 6 . . . . . . . . . 6 	6 . . . . . . . . . 6
 * 5 . . X . + . X . . 5 	5 . . X . X . X . . 5
 * 4 . . . . . . . . . 4 	4 . . . . . . . . . 4
 * 3 . . X . . . X . . 3 	3 . . X . . . X . . 3
 * 2 . . . . . . . . . 2 	2 . . . . . . . . . 2
 * 1 . . . . . . . . . 1 	1 . . . . . . . . . 1
 *   A B C D E F G H J	          A B C D E F G H J  
 *  
 * 8 stones:                    9 stones:           
 *						     
 *   A B C D E F G H J	          A B C D E F G H J  
 * 9 . . . . . . . . . 9   	9 . . . . . . . . . 9
 * 8 . . . . . . . . . 8   	8 . . . . . . . . . 8
 * 7 . . X . X . X . . 7   	7 . . X . X . X . . 7
 * 6 . . . . . . . . . 6   	6 . . . . . . . . . 6
 * 5 . . X . + . X . . 5   	5 . . X . X . X . . 5
 * 4 . . . . . . . . . 4   	4 . . . . . . . . . 4
 * 3 . . X . X . X . . 3   	3 . . X . X . X . . 3
 * 2 . . . . . . . . . 2   	2 . . . . . . . . . 2
 * 1 . . . . . . . . . 1   	1 . . . . . . . . . 1
 *   A B C D E F G H J	          A B C D E F G H J  
 *  
 * For odd-sized boards larger than 9x9, the same pattern is followed,
 * except that the edge stones are moved to the fourth line for 13x13
 * boards and larger.
 *
 * For even-sized boards at least 8x8, only the four first diagrams
 * are used, because there is no way to place the center stones
 * symmetrically. As for odd-sized boards, the edge stones are moved
 * to the fourth line for boards larger than 11x11.
 *
 * At most four stones are placed on 7x7 boards too (this size may or
 * may not be supported by the rest of the engine). No handicap stones
 * are ever placed on smaller boards.
 *
 * Notice that this function only deals with fixed handicap placement.
 * Larger handicaps can be added by free placement if the used
 * interface supports it. */


/* This table contains the (coded) positions of the stones.
 *  2 maps to 2 or 3, depending on board size
 *  0 maps to center
 * -ve numbers map to  board_size - number
 *
 * The stones are placed in this order, *except* if there are
 * 5 or 7 stones, in which case center ( {0,0} ) is placed, and
 * then as for 4 or 6.
 */

static const int places[][2] = {

  {2,-2}, {-2,2}, {2,2}, {-2,-2},  /* first 4 are easy */
                                   /* for 5, {0,0} is explicitly placed */
  
  {0,2},  {0,-2},                  /* for 6 these two are placed */
                                   /* for 7, {0,0} is explicitly placed */
  
  {2,0}, {-2,0},                   /* for 8, these two are placed */

  {0,0},                           /* finally tengen for 9 */
};


/*
 * Sets up handicap stones, returning the number of placed handicap stones.
 */

int
placehand(int handicap)
{
  int x;
  int maxhand;
  int three = board_size > 11 ? 3 : 2;
  int mid = board_size/2;
  int retval = handicap;

  if ((board_size % 2 == 1) && (board_size >= 9))
    maxhand = 9;
  else if (board_size >= 7)
    maxhand = 4;
  else
    maxhand = 0;

  /* It's up to the caller of this function to notice if the handicap
   * was too large for fixed placement and act upon that.
   */
  if (handicap > maxhand) {
    handicap = maxhand;
    retval = maxhand;
  }

  /* special cases: 5 and 7 */
  if (handicap == 5 || handicap == 7) {
    add_stone(mid, mid, BLACK);
    handicap--;
  }

  for (x=0; x<handicap; ++x) {
    int i = places[x][0];
    int j = places[x][1];

    /* translate the encoded values to board co-ordinates */
    if (i == 2)  i = three;	/* 2 or 3 */
    if (i == -2) i = -three;

    if (j == 2)  j = three;
    if (j == -2) j = -three;

    if (i == 0) i = mid;
    if (j == 0) j = mid;

    if ( i < 0) i += board_size-1;
    if ( j < 0) j += board_size-1;

    add_stone(i, j, BLACK);
  }

  return retval;
}


/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */

