/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 and 2000 by the Free Software Foundation.      *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <assert.h>

#include "liberty.h"
#include "hash.h"
#include "incremental_board.h"
#include "sgftree.h"
#include "gg_utils.h"

/* Forward declaration. */
static void new_position(void);

/* Statistics. */
static int trymove_counter = 0;


/* ================================================================ */
/*                 Board navigation                                 */
/* ================================================================ */


/* Coordinates for the eight directions, ordered
 * south, west, north, east, southwest, northwest, northeast, southeast.
 */
int deltai[8] = { 1,  0, -1,  0,  1, -1, -1, 1};
int deltaj[8] = { 0, -1,  0,  1, -1, -1,  1, 1};


/* ================================================================ */
/*                 Pushing and popping of boards                    */
/* ================================================================ */

/* Stack of trial moves to get to current position 
 * position and which color made them. Perhaps 
 * this should be one array of a structure 
 */
static int      stacki[MAXSTACK];
static int      stackj[MAXSTACK];
static int      move_color[MAXSTACK];

static Hash_data  hashdata_stack[MAXSTACK];


/*
 * Do the main work of trymove() and tryko(), i.e. the common parts.
 * The ignore_ko flag tells whether an illegal ko capture may be done.
 * Return 1 if the move was valid, otherwise 0.
 */

static int 
do_trymove(int i, int j, int color, int ignore_ko)
{
  /* 1. The move must be inside the board and the color must be BLACK
   * or WHITE.
   */
  assert(i>=0 && i<board_size && j>=0 && j<board_size);
  assert(color == BLACK || color == WHITE);
  
  /* 2. The location must be empty. */
  if (p[i][j] != EMPTY)
    return 0;

  /* 3. The location must not be the ko point, unless ignore_ko==1. */
  if (!ignore_ko
      && i == ko_i && j == ko_j) {
    if (((i>0) && (p[i-1][j] != color))
	|| ((i==0) && (p[i+1][j] != color))) {
      RTRACE("%m would violate the ko rule\n", i, j);
      return 0;
    }
  }

  /* 4. Test for suicide. */
  if (incremental_is_suicide(i, j, color)) {
    RTRACE("%m would be suicide\n", i, j);
    return 0;
  }
  
  /* Check for stack overflow. */
  if (stackp >= MAXSTACK-2) {
    fprintf(stderr, 
	    "gnugo: Truncating search. This is beyond my reading ability!\n");
    return 0;
  }

  /* Only count trymove when we do create a new position. */
  trymove_counter++;
  
  /* So far, so good. Now push the move on the move stack. These are
   * needed for dump_stack().
   */
  stacki[stackp] = i;
  stackj[stackp] = j;
  move_color[stackp] = color;

  /*
   * FIXME POST3.0: Do we have to store hashdata in a stack?
   *
   * Answer: No, we don't.  But for every stone that we add
   *         or remove, we must call hashdata_invert_stone(). This is
   *         not difficult per se, but the whole incremental_board.c 
   *         will have to be checked, and there is lots of room
   *         for mistakes.
   *
   *         At the same time, profiling shows that storing the
   *         hashdata in a stack doesn't take a lot of time, so I am
   *         reclassifying this FIXME as POST 3.0.  /iw
   */
  BEGIN_CHANGE_RECORD();
  PUSH_VALUE(ko_i);
  PUSH_VALUE(ko_j);
  memcpy(&hashdata_stack[stackp], &hashdata, sizeof(hashdata));

  ko_i = -1;
  ko_j = -1;
  hashdata_remove_ko(&hashdata);
  
  PUSH_VALUE(black_captured);
  PUSH_VALUE(white_captured);

  if (showstack)
    gprintf("        *** STACK before push: %d\n", stackp);
  ++stackp;

  if (verbose == 4)
    dump_stack();

  incremental_play_move(i, j, color);

  return 1;
}


/*
 * trymove pushes the position onto the stack, and makes a move
 * at (i, j) of color. Returns one if the move is legal. The
 * stack pointer is only incremented if the move is legal.
 *
 * The way to use this is:
 *
 *   if (trymove(i, j, color, [message], k, l)) {
 *      ...
 *      popgo();
 *   }   
 *
 * The message can be written as a comment to an sgf file using 
 * sgfdump().  (k, l) can be -1 if they are not needed but if they are 
 * not the location of (k, l) is included in the comment.
 */

int 
trymove(int i, int j, int color, const char *message, int k, int l)
{
  /* Do the real work elsewhere. */
  if (!do_trymove(i, j, color, 0))
    return 0;
  
  /* Store the move in the sgf file if there is a dump file. */
  if (sgf_dump) {
    if (k == -1) {
      if (color == BLACK)
	sgffile_write_line("\n(;B[%c%c]C[%s (variation %d, hash %lx)\n]", 
			   'a'+j, 'a'+i, message, count_variations,
			   hashdata.hashval);
      if (color == WHITE)
	sgffile_write_line("\n(;W[%c%c]C[%s (variation %d, hash %lx)\n]",
			   'a'+j, 'a'+i, message, count_variations,
			   hashdata.hashval);
    } else {
      char movename[4];
	
      if (l < 8)
	*movename = l+65;
      else
	*movename = l+66;
      sprintf(movename+1, "%d", board_size-k);
	
      if (color == BLACK)
	sgffile_write_line("\n(;B[%c%c]C[%s at %s (variation %d, hash %lx)\n]",
			   'a'+j, 'a'+i, message, movename, count_variations,
			   hashdata.hashval);
      if (color == WHITE)
	sgffile_write_line("\n(;W[%c%c]C[%s at %s (variation %d, hash %lx)\n]",
			   'a'+j, 'a'+i, message, movename, count_variations,
			   hashdata.hashval);
    }
  }
  else if (sgf_dumptree) {
    char buf[100];
    if (k == -1)
      gg_snprintf(buf, 100, "%s (variation %d, hash %lx)", message, 
		  count_variations, hashdata.hashval);
    else
      gg_snprintf(buf, 100, "%s at %c%d (variation %d, hash %lx)", message,
		  l + 'A' + (l>=8), board_size - k, count_variations,
		  hashdata.hashval);
    sgftreeAddPlayLast(sgf_dumptree, NULL, color, i, j);
    sgftreeAddComment(sgf_dumptree, NULL, buf);
  }

  if (count_variations)
    count_variations++;
  stats.nodes++;

  return 1;
}


/*
 * tryko pushes the position onto the stack, and makes a move
 * at (i, j) of color. The move is allowed even if it is an
 * illegal ko capture. It is to be imagined that (color) has
 * made an intervening ko threat which was answered and now
 * the continuation is to be explored.
 *
 * Return 1 if the move is legal with the above caveat. Returns
 * zero if it is not legal because of suicide.
 */

int 
tryko(int i, int j, int color, const char *message)
{
  /* Do the real work elsewhere. */
  if (!do_trymove(i, j, color, 1))
    return 0;

  if (sgf_dump) {
    if (color == BLACK)
      sgffile_write_line("\n(;B[tt]C[tenuki (ko threat)]\n\
;W[tt]C[tenuki (answers ko threat)]\n;B[%c%c]C[%s (variation %d, hash %lx)\n]", 
			 'a'+j, 'a'+i, message, count_variations,
			 hashdata.hashval);
    else if (color == WHITE)
      sgffile_write_line("\n(;W[tt]C[tenuki (ko threat)]\n\
;B[tt]C[tenuki (answers ko threat)]\n;W[%c%c]C[%s (variation %d, hash %lx)\n]",
			 'a'+j, 'a'+i, message, count_variations,
			 hashdata.hashval);
  }
  else if (sgf_dumptree) {
    char buf[100];
    if (!message)
      message = "???";
    gg_snprintf(buf, 100, "%s (%d, %lx)", message,
		  count_variations-1,
		  hashdata.hashval);
    sgftreeAddPlayLast(sgf_dumptree, NULL, color, -1, -1);
    sgftreeAddComment(sgf_dumptree, NULL, "tenuki (ko threat)");
    sgftreeAddPlayLast(sgf_dumptree, NULL, OTHER_COLOR(color), -1, -1);
    sgftreeAddComment(sgf_dumptree, NULL, "tenuki (answers ko threat)");
    sgftreeAddPlayLast(sgf_dumptree, NULL, color, i, j);
    sgftreeAddComment(sgf_dumptree, NULL, buf);
  }
  
  if (count_variations)
    count_variations++;
  stats.nodes++;

  return 1;
}


/*
 * popgo pops the position from the stack.
 */

void
popgo()
{
  stackp--;
  if (showstack)
    gprintf("<=    *** STACK  after pop: %d\n", stackp);
  
  incremental_undo_move();
  
  memcpy(&hashdata, &(hashdata_stack[stackp]), sizeof(hashdata));
  if (sgf_dump) 
    sgffile_write_line(")\n");
  else if (sgf_dumptree) {
    sgf_dumptree->lastnode = sgf_dumptree->lastnode->parent;
    /* After tryko() we need to undo two pass nodes too. Since we have
     * no other way to identify ko moves, we skip all pass nodes.
     */
    while (is_pass_node(sgf_dumptree->lastnode, board_size))
      sgf_dumptree->lastnode = sgf_dumptree->lastnode->parent;
  }
}


/* Silent version of popgo(), suitable for use if you have called
 * do_trymove() without passing through trymove() or tryko().
 */

static void
silent_popgo(void)
{
  stackp--;
  incremental_undo_move();
  memcpy(&hashdata, &(hashdata_stack[stackp]), sizeof(hashdata));
}


/*
 * dump_stack() for use under gdb prints the move stack. 
 */

void
dump_stack(void)
{
  int n;

  for (n=0; n<stackp; n++)
    gprintf("%o%s:%m ", move_color[n] == BLACK ? "B" : "W",
	    stacki[n], stackj[n]);
  
#if !TRACE_READ_RESULTS
  if (count_variations)
    gprintf("%o (variation %d)", count_variations-1);
#else
  gprintf("%o (%d)", hashdata.hashval);
#endif

  gprintf("%o\n");
}


/* ================================================================ */
/*                Non-invertable board manipulation                 */
/* ================================================================ */


/*
 * do_add_stone and do_remove_stone are defined as macros for 
 * efficiency reasons.
 */

#define do_add_stone(i, j, color) \
  do { \
    p[i][j]=color; \
    hashdata_invert_stone(&hashdata, i, j, color); \
  } while (0)

#define do_remove_stone(i, j) \
  do { \
    hashdata_invert_stone(&hashdata, i, j, p[i][j]); \
    p[i][j] = EMPTY; \
  } while (0)


/*
 * And now the functions, accessible from outside this file.
 */

/* place a stone on the board and update the hashdata. */

void
add_stone(int i, int j, int color)
{
  ASSERT(stackp == 0, i, j);
  ASSERT(ON_BOARD(i, j), i, j);
  ASSERT(p[i][j] == EMPTY, i, j);
  do_add_stone(i, j, color);
  new_position();
}


/* remove a stone from the board and update the hashdata. */

void
remove_stone(int i, int j)
{
  ASSERT(stackp == 0, i, j);
  ASSERT(ON_BOARD(i, j), i, j);
  ASSERT(p[i][j] != EMPTY, i, j);
  do_remove_stone(i, j);
  new_position();
}


/*
 * Remove the string at (i,j), treating it as captured.
 */

void
remove_string(int i, int j)
{
  int color=p[i][j];

  assert(stackp == 0);
  assert(color != EMPTY);

  do_remove_stone(i, j);
  if (color==BLACK)
    black_captured++;
  else
    white_captured++;
  
  if ((i>0) && (p[i-1][j]==color))
    remove_string(i-1, j);
  if ((i<board_size-1) && (p[i+1][j]==color))
    remove_string(i+1, j);
  if ((j>0) && (p[i][j-1]==color))
    remove_string(i, j-1);
  if ((j<board_size-1) && (p[i][j+1]==color))
    remove_string(i, j+1);

  new_position();
}



/* ---------------------------------------------------------------- */


/*
 * is_legal(i, j, color) determines whether the move (color) at
 * (i, j) is legal.
 */

int 
is_legal(int i, int j, int color)
{
  /* 0. A pass move is always legal. */
  if (i == -1 && j == -1)
    return 1;

  /* 1. The move must be inside the board. */
  assert(i>=0 && i<board_size && j>=0 && j<board_size);

  /* 2. The location must be empty. */
  if (p[i][j]!=EMPTY) 
    return 0;

  /* 3. The location must not be the ko point. */
  if (i == ko_i && j == ko_j)
    if (((i>0) && (p[i-1][j] != color))
	|| ((i==0) && (p[i+1][j] != color))) {
      RTRACE("%m would violate the ko rule\n", i, j);
      return 0;
    }

  /* Check for stack overflow. */
  if (stackp >= MAXSTACK-2) {
    fprintf(stderr, 
	    "gnugo: Truncating search. This is beyond my reading ability!\n");
    return 0;
  }

  /* Check for suicide. */
  if (incremental_is_suicide(i, j, color)) {
    RTRACE("%m would be suicide\n", i, j);
    return 0;
  }
  
  return 1;
}


/*
 * is_suicide(i, j, color) determines whether the move (color) at
 * (i, j) is suicide.
 */
int 
is_suicide(int i, int j, int color)
{
  assert(i>=0 && i<board_size && j>=0 && j<board_size);
  assert(p[i][j] == EMPTY);

  /* Check for suicide. */
  return incremental_is_suicide(i, j, color);
}



/* Play a move. If you want to test for legality you should first call
 * is_legal(). This function strictly follows the algorithm: 
 * 1. Place a stone of given color on the board.
 * 2. If there are any adjacent opponent strings without liberties,
 *    remove them and increase the prisoner count. 
 * 3. If the newly placed stone is part of a string without liberties,
 *    remove it and increase the prisoner count.
 *
 * In contrast to a move played by trymove() or tryko(), this move
 * can't be unplayed.
 */
void
play_move(int i, int j, int color)
{
#if CHECK_HASHING
  Hash_data oldkey;

  /* Check the hash table to see if it corresponds to the cumulative one. */
  hashdata_recalc(&oldkey, p, ko_i, ko_j);
  assert(hashdata_diff_dump(&oldkey, &hashdata) == 0);
#endif
  
  assert(stackp == 0);
  
  ko_i = -1;
  ko_j = -1;
  hashdata_remove_ko(&hashdata);

  /* If the move is a pass, we are done here. */
  if (i == -1 && j == -1)
    return;

  assert(i>=0 && i<board_size && j>=0 && j<board_size);
  ASSERT(p[i][j] == EMPTY, i, j);

  /* Do play the move. */
  incremental_play_move(i, j, color);

#if CHECK_HASHING
  /* Check the hash table to see if it equals the previous one. */
  hashdata_recalc(&oldkey, p, ko_i, ko_j);
  assert(hashdata_diff_dump(&oldkey, &hashdata) == 0);
#endif

  movenum++;
  new_position();
}


/* ---------------------------------------------------------------- */

/* We have reached a new position. Increase the position counter and
 * invalidate the incremental strings.
 */

static void
new_position(void)
{
  incremental_invalidate_strings();
  position_number++;
}


/*
 * Set up an entirely new position.
 */

void
setup_board(Intersection new_p[MAX_BOARD][MAX_BOARD], int koi, int koj,
	    int w_captured, int b_captured)
{
  memcpy(p, new_p, sizeof(p));
  ko_i = koi;
  ko_j = koj;

  white_captured = w_captured;
  black_captured = b_captured;

  hashdata_recalc(&hashdata, p, ko_i, ko_j);
  new_position();
}


/*
 * Clear the internal board.
 */

void
clear_board(void)
{
  assert(board_size > 0 && board_size <= MAX_BOARD);
  
  memset(p, EMPTY, sizeof(p));
  ko_i = -1;
  ko_j = -1;

  white_captured = 0;
  black_captured = 0;

  hashdata_recalc(&hashdata, p, ko_i, ko_j);
  new_position();

  movenum = 0;
}



/* Count the number of liberties of the string at (m, n). (m, n) must
 * not be empty.
 */

int
countlib(int m, int n)
{
  ASSERT(m >= 0 && m < board_size && n >= 0 && n < board_size, m, n);
  ASSERT(p[m][n] != EMPTY, m, n);
  
  return incremental_countlib(m, n);
}


/* Find the liberties of the string at (m, n). (m, n) must not be
 * empty. The locations of up to maxlib liberties are written into
 * (libi[], libj[]). The full number of liberties is returned.
 *
 * If you want the locations of all liberties, whatever their number,
 * you should pass MAXLIBS as the value for maxlib and allocate space
 * for libi[], libj[] accordingly.
 */

int
findlib(int m, int n, int maxlib, int *libi, int *libj)
{
  ASSERT(m >= 0 && m < board_size && n >= 0 && n < board_size, m, n);
  ASSERT(p[m][n] != EMPTY, m, n);
  ASSERT(libi != NULL && libj != NULL, m, n);
  
  return incremental_findlib(m, n, maxlib, libi, libj);
}


/* Find the liberties a stone of the given color would get if played
 * at (m, n), ignoring possible captures of opponent stones. (m, n)
 * must be empty. If libi!=NULL, the locations of up to maxlib
 * liberties are written into (libi[], libj[]). The counting of
 * liberties may or may not be halted when maxlib is reached. The
 * number of liberties found is returned.
 *
 * If you want the number or the locations of all liberties, however
 * many they are, you should pass MAXLIBS as the value for maxlib and
 * allocate space for libi[], libj[] accordingly.
 */

int
approxlib(int m, int n, int color, int maxlib, int *libi, int *libj)
{
  ASSERT(m >= 0 && m < board_size && n >= 0 && n < board_size, m, n);
  ASSERT(p[m][n] == EMPTY, m,n);
  ASSERT(color != EMPTY, m, n);
  /* Either both NULL or neither NULL. */
  ASSERT(((libi != NULL) ^ (libj != NULL)) == 0, m, n);
  
  return incremental_approxlib(m, n, color, maxlib, libi, libj);
}


/* Determine whether a move by color at (m, n) would be a self atari,
 * i.e. whether it would get more than one liberty. This function
 * returns true also for the case of a suicide move.
 */

int
is_self_atari(int m, int n, int color)
{
  int liberties;
  
  ASSERT(m >= 0 && m < board_size && n >= 0 && n < board_size, m, n);
  ASSERT(p[m][n] == EMPTY, m, n);
  ASSERT(color != EMPTY, m, n);

  if (!incremental_sloppy_self_atari(m, n, color))
    return 0;

  if (!do_trymove(m, n, color, 1))
    return 1;
  liberties = incremental_countlib(m, n);
  silent_popgo();
  
  return liberties <= 1;
}


/* Count the number of stones in a string. */
int
countstones(int m, int n)
{
  ASSERT(m>=0 && m < board_size && n >= 0 && n < board_size, m, n);
  ASSERT(p[m][n] != EMPTY, m, n);

  return incremental_countstones(m, n);
}


/* Find the stones of the string at (m, n). (m, n) must not be
 * empty. The locations of up to maxstones stones are written into
 * (stonei[], stonej[]). The full number of stones is returned.
 */

int
findstones(int m, int n, int maxstones, int *stonei, int *stonej)
{
  ASSERT(m>=0 && m < board_size && n >= 0 && n < board_size, m, n);
  ASSERT(p[m][n] != EMPTY, m, n);

  return incremental_findstones(m, n, maxstones, stonei, stonej);
}


/* chainlinks returns (in adji, adjj arrays) the chains surrounding
 * the string at (m, n).
 */

void 
chainlinks(int m, int n, int *adj, int adji[MAXCHAIN], int adjj[MAXCHAIN],
	   int adjsize[MAXCHAIN], int adjlib[MAXCHAIN])
{
  incremental_chainlinks(m, n, adj, adji, adjj, adjsize, adjlib);
}


/* chainlinks2 returns (in adji, adjj arrays) the chains surrounding
 * the string at (m, n), which have exactly lib liberties.
 */

void 
chainlinks2(int m, int n, int *adj, int adji[MAXCHAIN], int adjj[MAXCHAIN],
	   int lib)
{
  incremental_chainlinks2(m, n, adj, adji, adjj, lib);
}


/*
 * Find the origin of a worm or a cavity, i.e. the point with smallest
 * i coordinate and in the case of a tie with smallest j coordinate.
 * The idea is to have a canonical reference point for a string.
 */

void
find_origin(int i, int j, int *origini, int *originj)
{
  incremental_find_origin(i, j, origini, originj);
}


/*
 * Returns true if (ai, aj) is a liberty of the string at (si, sj).
 */

int
liberty_of(int ai, int aj, int si, int sj)
{
  if (p[ai][aj] != EMPTY)
    return 0;

  return incremental_neighbor_of(ai, aj, si, sj);
}


/*
 * Returns true if (ai, aj) is adjacent to the string at (si, sj).
 */

int
neighbor_of(int ai, int aj, int si, int sj)
{
  return incremental_neighbor_of(ai, aj, si, sj);
}


/*
 * Return true if the move (i,j) by (color) is a ko capture
 * (whether capture is legal on this move or not).
 */

int
is_ko(int i, int j, int color)
{
  ASSERT_ON_BOARD(i, j);
  ASSERT(color == WHITE || color == BLACK, i, j);
  return incremental_is_ko(i, j, color);
}


/* For each stone in the string at (i, j), set mx to value mark. If
 * some of the stones in the string are marked prior to calling this
 * function, only the connected unmarked stones starting from (i, j)
 * are guaranteed to become marked. The rest of the string may or may
 * not become marked. (In the current implementation, it will.)
 */
void
mark_string(int i, int j, char mx[MAX_BOARD][MAX_BOARD], char mark)
{
  assert(p[i][j] != EMPTY);
  incremental_mark_string(i, j, mx, mark);
}


/* Returns true if at least one move been played at @code{(m, n)}
 * at deeper than level 'cutoff' in the reading tree?
 */
int
move_in_stack(int m, int n, int cutoff)
{
  int k;
  for (k=cutoff; k<stackp; k++)
    if (stacki[k] == m && stackj[k] == n)
      return 1;
  
  return 0;
}


/* Return the number of stones of the indicated color(s) on the board.
 * This only count stones in the permanent position, not stones placed
 * by trymove() or tryko(). Use stones_on_board(BLACK | WHITE) to get
 * the total number of stones on the board.
 */
int
stones_on_board(int color)
{
  static int stone_count_for_position = -1;
  static int white_stones = 0;
  static int black_stones = 0;
  
  if (stone_count_for_position != position_number) {
    int m, n;
    white_stones = 0;
    black_stones = 0;
    for (m=0; m<board_size; m++)
      for (n=0; n<board_size; n++) {
	if (p[m][n] == WHITE)
	  white_stones++;
	else if (p[m][n] == BLACK)
	  black_stones++;
      }
    
    stone_count_for_position = position_number;
  }

  return ((color & BLACK ? black_stones : 0) +
	  (color & WHITE ? white_stones : 0));
}


/* ===================== Statistics  ============================= */

/* Clear statistics. */
void
reset_trymove_counter()
{
  trymove_counter = 0;
}


/* Retrieve statistics. */
int
get_trymove_counter()
{
  return trymove_counter;
}

/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
