/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 and 2000 by the Free Software Foundation.      *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef HAVE___FUNCTION__
#define __FUNCTION__ ""
#endif

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <assert.h>

#include "sgftree.h"
#include "analyze.h"
#include "liberty.h"
#include "gg_utils.h"


#define USAGE "\n\
Usage : gnugo [-opts]\n\
\n\
Use --analyze with --analyzerfile and --testmode, --score or --benchmark.\n\
See the Texinfo documentation (User Guide/Invoking GNU Go) for more\n\
information.\n\n\
Analyzer Options:\n\
       --analyzerfile <name>    filename for analyzer output\n\
       --analyze <options>      where <options> can be:\n\
              areacolor         show color of influence \n\
              capture           show capturing moves\n\
              considered        show considered moves\n\
              defense           show defending moves\n\
              dragoninfo        print info about all dragons\n\
              dragonstatus      show dragonstatus\n\
              eyeinfo           print info about all eyes\n\
              eyes              show eyes and their vital points\n\
              moyocolor         show color of moyos\n\
              moyovalue         show moyo values \n\
              overwrite         overwrites game info from the inputfile\n\
              recommended       show moves recommend by modules\n\
              terricolor        show color of territories (moyo)\n\
              territory         show territory (worms)\n\
              terrivalue        show territory values (moyo)\n\
              worminfo          print info about all worms\n\
              wormliberties     show liberties of the worms\n\
You may use \"option1 option2 ...\" or option1,option2,... \n\
to specify more than one option for --analyze\n\
"

/* same order as ANALYZE_.. in sgfana.h */
const char *analyzer_options[]={
  "overwrite",
  "dragoninfo",
  "dragonstatus",
  "considered",
  "wormliberties",
  "eyes",
  "worminfo",
  "eyeinfo",
  "gameinfo",
  "moyocolor",
  "recommended",
  "capture",
  "moyovalue",
  "defense",
  "terricolor",
  "terrivalue",
  "territory",
  "areacolor",
  "areavalue",
};


void
sgfNextString(char *s)
{
  int i;

  i = strlen(s)-1;
  s[i]++;
  while(i >= 0  && s[i] > 'Z') {
    s[i] = 'A';
    i--;
    if(i>=0) s[i]++;
  }

  if (i == -1) {
    i = strlen(s);
    if (i<3) {
      s[i] = 'A';
      s[i+1] = '\0';
    }
  }
}


void
sgfPrevString(char *s)
{
  int i;

  i = strlen(s)-1;
  s[i]--;
  while (i >= 0  && s[i] < 'a') {
    s[i] = 'z';
    i--;
    if (i>=0) s[i]--;
  }

  if (i == -1) {
    i = strlen(s);
    if (i<3) {
      s[i] = 'z';
      s[i+1] = '\0';
    }
  }
}


/*
 * Mark all parts of a worm with mark.
 */

void
sgfMarkWorm(SGFNode *node, int x, int y, const char *mark)
{
  int i,j;
  char text[3];

  for(i=0;i<board_size;i++)
    for(j=0;j<board_size;j++) {
      if ((worm[i][j].origini==x) && (worm[i][j].originj==y)) {
	gg_snprintf(text, 3, "%c%c", j+'a', i+'a');
	sgfAddProperty(node, mark, text);
      }
    }
}


/*
 * Place a territory mark for color on the board at position (i,j).
 */

SGFNode *
sgfTerritory(SGFNode *node,int i,int j,int color)
{
  char text[3];

  gg_snprintf(text, 3, "%c%c", j+'a', i+'a');
  if (color==BLACK)
    sgfAddProperty(node, "TB", text);
  else
    sgfAddProperty(node, "TW", text);

  return node;
}


/*
 * Parse the options string and set the corresponding flags.
 */

void
parse_analyzer_options(char *optarg)
{
  char *str;
  unsigned int i;
  unsigned int n_options=sizeof(analyzer_options)/sizeof(analyzer_options[0]);

  str=strtok(optarg, " ,");
  do {
    i=0;
    if (!strcmp(str,"all")) {
      analyzerflag=-1;
      break;
    }

    while(i< n_options) {
      if (!strcmp( str,analyzer_options[i]))
	break;
      else
	i++;
    }

    if (i>=n_options) {
      fprintf(stderr,"gnugo: unknown option --analyze %s\n",str);
      exit(EXIT_FAILURE);
    }
    else
      analyzerflag|=1<<i;
  } while ((str=strtok(NULL, " ,")));
}


/* ================================================================ */
/*                 The analyzer functions themselves                */
/* ================================================================ */


static void sgfShowConsideredMoves(SGFNode *node);
static void sgfShowDragons(SGFNode *node, int mode);
static void sgfShowWorms(SGFNode *node, int mode);
static void sgfShowEyes(SGFNode *node, int mode);
static void sgfShowEyeInfo(SGFNode *node, int mode);
static void sgfShowDragonInfo(SGFNode *node, int mode);
static void sgfShowWormInfo(SGFNode *node, int mode);
static void sgfPrintGameInfo(SGFNode *node, int value);
static void sgfShowCapturingMoves(SGFNode *node);
static void sgfShowDefendingMoves(SGFNode *node);


/*
 * Calls the different output functions according to analyzerflag.
 */

void
sgfAnalyzer(SGFNode *node, int value)
{
  if (!analyzerflag)
    return;
    
  if (analyzerflag & ANALYZE_CONSIDERED)
    sgfShowConsideredMoves(node);
  if (analyzerflag & ANALYZE_DRAGONSTATUS)
    sgfShowDragons(node, SGF_SHOW_STATUS);
  if (analyzerflag & ANALYZE_WORMS)
    sgfShowWorms(node, SGF_SHOW_LIBERTIES);
  if (analyzerflag & ANALYZE_EYES)
    sgfShowEyes(node, 0);
  if (analyzerflag & ANALYZE_EYEINFO)
    sgfShowEyeInfo(node, 0);
  if (analyzerflag & ANALYZE_DRAGONINFO)
    sgfShowDragonInfo(node, 0);
  if (analyzerflag & ANALYZE_WORMINFO)
    sgfShowWormInfo(node, 0);
  if (analyzerflag & ANALYZE_GAMEINFO)
    sgfPrintGameInfo(node, value);
  if (analyzerflag & ANALYZE_MOYOCOLOR)
    sgfShowMoyo(node, SGF_SHOW_COLOR, 1);
  if (analyzerflag & ANALYZE_MOYOVALUE)
    sgfShowMoyo(node, SGF_SHOW_VALUE, 1);
  if (analyzerflag & ANALYZE_TERRICOLOR)
    sgfShowTerri(node, SGF_SHOW_COLOR, 1);
  if (analyzerflag & ANALYZE_TERRIVALUE)
    sgfShowTerri(node, SGF_SHOW_VALUE, 1);
  if (analyzerflag & ANALYZE_TERRITORY)
    sgfShowTerritory(node, 1);
  if (analyzerflag & ANALYZE_AREACOLOR)
    sgfShowArea(node, SGF_SHOW_COLOR,1);
  if (analyzerflag & ANALYZE_AREAVALUE)
    sgfShowArea(node, SGF_SHOW_VALUE,1);
  if (analyzerflag & ANALYZE_CAPTURING)
    sgfShowCapturingMoves(node);
  if (analyzerflag & ANALYZE_DEFENSE)
    sgfShowDefendingMoves(node);
}


static void
sgfPrintGameInfo(SGFNode *node, int value)
{
  char text[128];

  gg_snprintf(text, 128, "Move value: %i", value);
  node=sgfAddComment(node, text);
  gg_snprintf(text, 128, "W captured %i", black_captured);
  sgfAddComment(node, text);
  gg_snprintf(text, 128, "B captured %i", white_captured);
  sgfAddComment(node, text);
  gg_snprintf(text, 128, "W territory %i", moyo_eval[WHITE]);
  sgfAddComment(node, text);
  gg_snprintf(text, 128, "B territory %i", moyo_eval[BLACK]);
  sgfAddComment(node,text);
}


/*
 * Prints all considered moves and their values into the analyzer file.
 */

static void
sgfShowConsideredMoves(SGFNode *node)
{
  int i,j;

  assert(node);

  node=sgfStartVariant(node);
  sgfAddComment(node, "Considered Moves");
  for(i=0;i<board_size;i++)
    for(j=0;j<board_size;j++) {
      if (potential_moves[i][j]>0)
	sgfBoardNumber(node, i, j, potential_moves[i][j]);
    }
}

 
/*
 * Shows dragons
 */

static void
sgfShowDragons(SGFNode *node, int mode)
{
  struct dragon_data *d;
  int i,j;

  assert(node);

  node=sgfStartVariant(node);
  node=sgfAddComment(node,"Dragon Status");
  for(i=0;i<board_size;i++) {
    for(j=0;j<board_size;j++) {
      d=&(dragon[i][j]);
      if ((d->color==WHITE) || (d->color==BLACK)) {
	switch(mode) {
	case SGF_SHOW_STATUS:
	  switch(d->status) {
	  case ALIVE:    sgfBoardText(node, i, j, "L"); break;
	  case CRITICAL: sgfBoardText(node, i, j, "C"); break;
	  case DEAD:     sgfBoardText(node, i, j, "D"); break;
	  case UNKNOWN:  sgfBoardText(node, i, j, "U"); break;
	  default:
	    fprintf(stderr, "%s: illegal status %i at %i %i\n",
		    __FUNCTION__, d->color, i, j);
	    exit(-1);
	  }
	  break;
	default:
	  fprintf(stderr, "%s: illegal mode\n", __FUNCTION__);
	  exit(-1);
	}
      }
      else {
	switch(d->color) {
	case BLACK_BORDER: sgfTerritory(node, i, j, BLACK); break;
	case WHITE_BORDER: sgfTerritory(node, i, j, WHITE); break;
	case GRAY_BORDER:  ; break;
	case EMPTY:        ; break;
	default:
	  fprintf(stderr, "%s: illegal color %i at %i %i\n",
		  __FUNCTION__, d->color, i, j);
	  exit(-1);
	}
      }
    }
  }
}


static void
sgfShowWorms(SGFNode *node, int mode)
{
  struct worm_data *w;
  int i,j;

  assert(node);

  node=sgfStartVariant(node);
  node=sgfAddComment(node,"Worm Liberties");
  for(i=0;i<board_size;i++) {
    for(j=0;j<board_size;j++) {
      w=&(worm[i][j]);
      if ((w->color==WHITE) || (w->color==BLACK)) {
	switch(mode) {
	case SGF_SHOW_LIBERTIES:
	  sgfBoardNumber(node, i, j, w->liberties);
	  break;

	default:
	  fprintf(stderr, "%s: illegal mode\n", __FUNCTION__);
	  exit(-1);
	}
      }
      else {
	switch(w->color) {
	case BLACK_BORDER: sgfTerritory(node, i, j, BLACK); break;
	case WHITE_BORDER: sgfTerritory(node, i, j, WHITE); break;
	case GRAY_BORDER:  ; break;
	case EMPTY:        ; break;
	default:
	  fprintf(stderr, "%s: illegal color %i at %i %i\n",
		  __FUNCTION__, w->color, i, j);
	  exit(-1);
	}
      }
    }
  }
}


static void
sgfShowEyes(SGFNode *node, int mode)
{
  struct eye_data *e;
  int i,j;

  i=mode; /*avoid warning */
  assert(node);

  node=sgfStartVariant(node);
  node=sgfAddComment(node, "Vital Points of Eyes");
  for(i=0;i<board_size;i++) {
    for(j=0;j<board_size;j++) {
      if (p[i][j]==EMPTY) {
	e=&(white_eye[i][j]);
	switch(e->color) {
	case WHITE_BORDER: sgfTerritory(node, i, j, WHITE); break;
	case BLACK_BORDER: break;
	case GRAY_BORDER:
	case EMPTY:        break;

	default:
	  fprintf(stderr, "%s: illegal color %i at %i %i\n",
		  __FUNCTION__, e->color, i, j);
	  exit(-1);
	}

	if ((i==e->attacki)&&(j==e->attackj))
	  sgfMark(node, i, j);
            
	e=&(black_eye[i][j]);
	switch(e->color) {
	case BLACK_BORDER: sgfTerritory(node, i, j, BLACK); break;
	case WHITE_BORDER: break;
	case GRAY_BORDER:  break;
	case EMPTY:        break;
	default:
	  fprintf(stderr, "%s: illegal color %i at %i %i\n",
		  __FUNCTION__, e->color, i, j);
	  exit(-1);
	}
	if ((i==e->attacki)&&(j==e->attackj))
	  sgfMark(node, i, j);
      }
    }
  }
}


/*
 * Shows territory information from worms.
 */

void
sgfShowTerritory(SGFNode *node, int as_variant)
{
  struct worm_data *w;
  int i,j;

  assert(node);

  if (as_variant) {
    node=sgfStartVariant(node);
    node=sgfAddComment(node, "Territory");
  }

  for(i=0;i<board_size;i++) {
    for(j=0;j<board_size;j++) {
      w=&(worm[i][j]);
      if (!((w->color==WHITE) || (w->color==BLACK))) {
	switch(w->color) {
	case BLACK_BORDER: sgfTerritory(node, i, j, BLACK); break;
	case WHITE_BORDER: sgfTerritory(node, i, j, WHITE); break;
	case GRAY_BORDER:  ; break;
	default:
	  fprintf(stderr, "%s: illegal color %i at %i %i\n", 
		  __FUNCTION__, w->color, i, j);
	}
      }
    }
  }
}


/*
 * Prints the info of the eye at i,j with the name c.
 */

void
sgfPrintSingleEyeInfo(SGFNode *pr,int i, int j, char *c)
{
  char text[128];
  struct eye_data *e;

  if (white_eye[i][j].color==WHITE_BORDER)
    e=&(white_eye[i][j]);
  else if (black_eye[i][j].color==BLACK_BORDER)
    e=&(black_eye[i][j]);
  else
    return;

  gg_snprintf(text, 128, "%s esize=%i msize=%i mineye=%i maxeye=%i", c,
	   e->esize, e->msize, e->mineye, e->maxeye);
  sgfAddComment(pr, text);
}


static void
sgfShowEyeInfo(SGFNode *node, int mode)
{
  struct eye_data *e;
  int i,j;
  char next_black[4], next_white[4];
  char names[MAX_BOARD][MAX_BOARD][4];

  /* count down from 'z' to minimise confusion. 
     Avoid % operator on -ve quantities */
  strncpy(next_white, "z", 4);
  strncpy(next_black, "A", 4); /* avoid 0, which means unsigned */
  for(i=0;i<board_size;i++)
    for(j=0;j<board_size;j++)
      names[i][j][0]='\0';

  if (mode<0)
    return;

  assert(node);

  node=sgfStartVariant(node);
  node=sgfAddComment(node,"Eye Info\n");
  for(i=0;i<board_size;i++) {
    for(j=0;j<board_size;j++) {
      e=&(black_eye[i][j]);
      switch(e->color) {
      case BLACK_BORDER:
	if (!names[e->origini][e->originj][0]) {
	  strcpy(names[e->origini][e->originj],next_black);
	  sgfNextString(next_black);
	  sgfPrintSingleEyeInfo(node, e->origini, e->originj,
				names[e->origini][e->originj]);
	}
	sgfBoardText(node, i, j, names[e->origini][e->originj]);
	break;
      case WHITE_BORDER:
      case GRAY_BORDER:
      case BLACK:
      case WHITE:
      case EMPTY:
	break;

      default:
	fprintf(stderr, "%s: illegal color %i at %i %i\n", 
		__FUNCTION__, e->color, i, j);
	exit(-1);
      }

      e=&(white_eye[i][j]);
      switch(e->color) {
      case WHITE_BORDER:
	if (!names[e->origini][e->originj][0]) {
	  strcpy(names[e->origini][e->originj],next_white);
	  sgfPrevString(next_white);
	  sgfPrintSingleEyeInfo(node, e->origini, e->originj,
				names[e->origini][e->originj]);
	}
              
	sgfBoardText(node, i, j, names[e->origini][e->originj]);
	break;
      case GRAY_BORDER:
      case BLACK:
      case WHITE:
      case EMPTY:
      case BLACK_BORDER:
	break;
      default:
	fprintf(stderr, "%s: illegal color %i at %i %i\n", 
		__FUNCTION__, e->color, i, j);
	exit(-1);
      }
    }
  }
}


/*
 * Prints the info of the dragon at (i,j) with the name c.
 */

void
sgfPrintSingleDragonInfo(SGFNode *pr,int i, int j, char *c)
{
  char text[128];
  const char *status;

  switch (dragon[i][j].status) {
  case ALIVE:    status="alive"; break;
  case DEAD:     status="dead";  break;
  case CRITICAL: status="crititcal"; break;
  case UNKNOWN:  status="unknown";break;
  default:
    fprintf(stderr, "%s: illegal status %i at %i %i\n",
	    __FUNCTION__, dragon[i][j].status, i, j);
    exit(-1);
  }
  gg_snprintf(text, 128, "%s %s size=%i genus=%i", c, status,
	   dragon[i][j].size, dragon[i][j].genus);
  sgfAddComment(pr,text);
}


static void
sgfShowDragonInfo(SGFNode *node, int mode)
{
  struct dragon_data *d=&dragon[0][0];
  int i,j=0;
  char next_black[4], next_white[4];
  char names[MAX_BOARD][MAX_BOARD][4];

 /* count down from 'z' to minimise confusion. 
    Avoid % operator on -ve quantities */
  strncpy(next_white,"z",4);
  strncpy(next_black,"A",4); /* avoid 0, which means unsigned */
  for(i=0;i<board_size;i++)
    for(j=0;j<board_size;j++)
      names[i][j][0]='\0';

  if (mode<0)
    return;

  assert(node);
  node=sgfStartVariant(node);
  node=sgfAddComment(node,"Dragon Info\n");
  for(i=0;i<board_size;i++) {
    for(j=0;j<board_size;j++) {
      d=&(dragon[i][j]);
      switch(d->color) {
      case BLACK:
	if (!names[d->origini][d->originj][0]) {
	  strcpy(names[d->origini][d->originj], next_black);
	  sgfNextString(next_black);
	  sgfPrintSingleDragonInfo(node, d->origini, d->originj, 
				   names[d->origini][d->originj]);
	}
	sgfBoardText(node, i, j, names[d->origini][d->originj]);
	break;
      case WHITE:
	if (!names[d->origini][d->originj][0]) {
	  strcpy(names[d->origini][d->originj], next_white);
	  sgfPrevString(next_white);
	  sgfPrintSingleDragonInfo(node, d->origini, d->originj,
				   names[d->origini][d->originj]);
	}
	sgfBoardText(node, i, j, names[d->origini][d->originj]);
	break;

      case BLACK_BORDER: sgfTerritory(node, i, j, BLACK); break;
      case WHITE_BORDER: sgfTerritory(node, i, j, WHITE); break;
      case GRAY_BORDER:  ; break;
      case EMPTY:          break;
      default:
	fprintf(stderr, "%s: illegal color %i at %i %i\n",
		__FUNCTION__, d->color, i, j);
	exit(-1);
      }
    }
  }
}


/*
 * Prints the info of the worm at (i,j) with the name c.
 */

void
sgfPrintSingleWormInfo(SGFNode *pr,int i, int j, char *c)
{
  char text[128];

  gg_snprintf(text, 128, "%s %s size=%i effective size=%f", c,
	      worm[i][j].inessential?"inessential":"essential",
	      worm[i][j].size, worm[i][j].effective_size);
  sgfAddComment(pr, text);
}


static void
sgfShowWormInfo(SGFNode *node, int mode)
{
  struct worm_data *w=&worm[0][0];
  int i,j=0;
  char next_black[4], next_white[4];
  char names[MAX_BOARD][MAX_BOARD][4];

  /* count down from 'z' to minimise confusion. 
    Avoid % operator on -ve quantities */
  strncpy(next_white, "z", 4);
  strncpy(next_black, "A", 4); /* avoid 0, which means unsigned */
  for(i=0;i<board_size;i++)
    for(j=0;j<board_size;j++)
      names[i][j][0]='\0';

  if (mode<0)
    return;

  assert(node);
  node=sgfStartVariant(node);
  node=sgfAddComment(node,"Worm Info\n");
  for(i=0;i<board_size;i++) {
    for(j=0;j<board_size;j++) {
      w=&(worm[i][j]);
      switch(w->color) {
      case BLACK:
	if (!names[w->origini][w->originj][0]) {
	  strcpy(names[w->origini][w->originj],next_black);
	  sgfNextString(next_black);
	  sgfPrintSingleWormInfo(node, w->origini, w->originj, 
				 names[w->origini][w->originj]);
	}
	sgfBoardText(node, i, j, names[w->origini][w->originj]);
	break;

      case WHITE:
	if (!names[w->origini][w->originj][0]) {
	  strcpy(names[w->origini][w->originj],next_white);
	  sgfPrevString(next_white);
	  sgfPrintSingleWormInfo(node, w->origini, w->originj,
				 names[w->origini][w->originj]);
	}
              
	sgfBoardText(node, i, j, names[w->origini][w->originj]);
	break;

      case BLACK_BORDER: sgfTerritory(node,i,j,BLACK); break;
      case WHITE_BORDER: sgfTerritory(node,i,j,WHITE); break;
      case GRAY_BORDER:  ; break;
      case EMPTY:        break;
      default:
	fprintf(stderr, "%s: illegal color %i at %i %i\n",
		__FUNCTION__, w->color, i, j);
	exit(-1);
      }
    }
  }
}


/*
 * Shows the capturing move for all capturable worms.
 */

static void
sgfShowCapturingMoves(SGFNode *node)
{
  int *worms=xalloc(MAX_BOARD*MAX_BOARD*sizeof(int));
  int wp=0;
  int i,j,k;
  int wormname;

  for(i=0;i<board_size;i++) {
    for(j=0;j<board_size;j++) {
      if ((p[i][j]!=EMPTY) && (worm[i][j].liberties>1)) {
	wormname= (worm[i][j].origini<<8) + worm[i][j].originj;
	k=0;
	while((k<wp)&&(wormname!=worms[k])) k++;
	if (k==wp) {
	  worms[wp]=wormname;
	  wp++;
	  /*     gprintf("%s %m %d\n",__FUNCTION__,i,j,wormname);*/
	  sgfCaptureString(node, i, j);
	}
      }
    }
  }
}


/*
 * Shows how to capture the string at (i,j).
 */

void
sgfCaptureString(SGFNode *node, int i, int j)
{
  char move[3];
  int ki,kj;

  if (!attack(i, j, &ki, &kj)) 
    return;

  node=sgfStartVariant(node->child);
  sprintf(move, "%c%c", kj+'a', ki+'a');
  sgfAddProperty(node, p[i][j]==WHITE ? "B":"W", move);
  sgfMarkWorm(node, i, j, "MA");
  sgfAddComment(node, "Capturing move");
}


/*
 * Shows the defending move for all capturable worms.
 */

static void
sgfShowDefendingMoves(SGFNode *node)
{
  int *worms=xalloc(MAX_BOARD*MAX_BOARD*sizeof(int));
  int wp=0;
  int i,j,k;
  int wormname;

  for(i=0;i<board_size;i++) {
    for(j=0;j<board_size;j++) {
      if ((p[i][j]!=EMPTY) && (worm[i][j].liberties>1)) {
	wormname= (worm[i][j].origini<<8) + worm[i][j].originj;
	k=0;
	while ((k<wp) && (wormname!=worms[k]))
	  k++;
	if (k==wp) {
	  worms[wp]=wormname;
	  wp++;
	  /*     gprintf("%s %m %d\n",__FUNCTION__,i,j,wormname);*/
	  sgfDefendString(node, i, j);
	}
      }
    }
  }
}


/*
 * Shows how to capture the string at (i,j).
 */

void
sgfDefendString(SGFNode *node, int i, int j)
{
  char move[3];
  int ki,kj;

  if (!attack(i, j, &ki, &kj))
    return;
  if (!find_defense(i, j, &ki, &kj))
    return;

  node=sgfStartVariant(node->child);
  sprintf(move,"%c%c",kj+'a',ki+'a');
  sgfAddProperty(node, p[i][j]==WHITE ? "W":"B", move);
  sgfMarkWorm(node, i, j, "MA");
  sgfAddComment(node, "Defending move");
}


void
sgfHelp()
{
  fprintf(stderr, USAGE);
}

/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */

