/* -*-mode:c; c-style:k&r; c-basic-offset:4; -*- */


/* theme.c - gnect - a "Four In A Row" game for the GNOME.
 *
 * (c) 2000, 2001 Tim Musson <trmusson@ihug.co.nz>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */



/* Theme files look like this:

   # - Comments (#) and blank lines are optional.
   # - Items can appear in any order.
   # - The equals char can be space or tab padded.
   # - Lines can include leading/trailing white space.
   # - Item names are not case-sensitive (file names are).

   Title   = Marble Set 1
   Tileset = marbles1.png
   Player1 = Red
   Player2 = Blue

   # The items above are required.
   # The items below are optional.

   Background = whatever.jpg
   GridRGB = RGB:FF/FF/CC
   Tooltip = Any details you'd like to appear as a tooltip for this theme

*/



#include <sys/types.h>
#include <string.h>
#include <stdlib.h>
#include <dirent.h>
#include <gnome.h>

#include "main.h"
#include "gnect.h"
#include "file.h"



extern struct Gnect gnect;


static void   theme_add(THEMELIST_PTR thisTheme);
static THEMELIST_PTR theme_get_info(const gchar *pathnamedata, const gchar *filename);
static gchar *theme_file_read(const gchar *filename);
static gchar *theme_file_buffer_getline(const gchar *fbuffer, int *fpos);
static gchar *theme_string_bite(gchar *str, const gchar *biteChars);




void theme_init(void)
{
	/* Read all theme files into a theme info list, gnect.themes being the base */


	THEMELIST_PTR newTheme;
	DIR           *dir = NULL;
	struct dirent *e = NULL;
	gchar         *dname = NULL;
	gboolean      exists;
	gint          reading;


	gnect.themes = NULL;


	/* get the data directory's name */
	dname = filename_gnect_data(NULL, &exists);
	if (!exists) {
		g_free(dname);
		return;
	}


	/* read all .gnect files in install path and user's home dir */
	reading = 2;
	while(reading) {

		dir = opendir(dname);
		if (!dir) {
			if (reading == 2) {
				g_warning("gnect: theme_init: opendir failed (%s)\n", dname);
			}
			g_free(dname);
			return;
		}
		while ((e = readdir(dir)) != NULL) {
			gchar *fname = g_strdup(e->d_name);
			if (strstr(e->d_name, ".gnect")) {
				if ((newTheme = theme_get_info(dname, fname))) {
					theme_add(newTheme);
				}
			}
			g_free(fname);
		}
		closedir(dir);
		g_free(dname);
		reading--;
		if (reading) {
			dname = filename_expand_tilde(g_strdup(USER_THEME_DIR));
		}
	}
}



void theme_free(THEMELIST_PTR firstTheme)
{
	/* Free all theme info */


	THEMELIST_PTR thisTheme = firstTheme;
	THEMELIST_PTR nextTheme;


#ifdef GNECT_DEBUGGING
	if (gnect.debugging) {g_print("gnect: theme_free\n");}
#endif


	if (thisTheme) {

		while(thisTheme->prev) {
			thisTheme = thisTheme->prev;
		}

		while(thisTheme) {

			if (thisTheme->title) {
				g_free(thisTheme->title);
			}
			if (thisTheme->fnameTheme) {
				g_free(thisTheme->fnameTheme);
			}
			if (thisTheme->fnameTileset) {
				g_free(thisTheme->fnameTileset);
			}
			if (thisTheme->fnameBackground) {
				g_free(thisTheme->fnameBackground);
			}
			if (thisTheme->descrPlayer1) {
				g_free(thisTheme->descrPlayer1);
			}
			if (thisTheme->descrPlayer2) {
				g_free(thisTheme->descrPlayer2);
			}
			if (thisTheme->gridRGB) {
				g_free(thisTheme->gridRGB);
			}
			if (thisTheme->tooltip) {
				g_free(thisTheme->tooltip);
			}
			nextTheme = thisTheme->next;
			g_free(thisTheme);
			thisTheme = nextTheme;
		}
	}
}



static void theme_add(THEMELIST_PTR thisTheme)
{
	/* Insert theme in gnect.themes list alphabetically, according to title */


	if (gnect.themes == NULL) {
		gnect.themes = thisTheme;
	}
	else {

		gboolean endOfList = FALSE;
		gboolean addBefore = FALSE;


		while(gnect.themes->prev) {
			gnect.themes = gnect.themes->prev;
		}
		while(!endOfList) {
			if ((addBefore = strcasecmp(thisTheme->title, gnect.themes->title) <= 0)) {
				break;
			}
			if (!(endOfList = (gnect.themes->next == NULL))) {
				gnect.themes = gnect.themes->next;
			}
		}
		if (addBefore) {
			thisTheme->next = gnect.themes;
			thisTheme->prev = gnect.themes->prev;
			if (gnect.themes->prev) {
				gnect.themes->prev->next = thisTheme;
			}
			gnect.themes->prev = thisTheme;
		}
		else {
			gnect.themes->next = thisTheme;
			thisTheme->prev = gnect.themes;
		}
	}
}



THEMELIST_PTR theme_pointer_from_filename(const gchar *fnameTheme)
{
	/* Given the filename of a theme file, return a
	 * pointer to that theme's theme list info.
	 */


	THEMELIST_PTR thisTheme = gnect.themes;
	gboolean found = FALSE;


	if (thisTheme && fnameTheme) {

		while(thisTheme->prev) {
			thisTheme = thisTheme->prev;
		}
		while(thisTheme && !found) {
			found = (strcmp(fnameTheme, thisTheme->fnameTheme) == 0);
			if (!found) {
				thisTheme = thisTheme->next;
			}
		}
		if (!found) {
			thisTheme = NULL;
		}
	}

	return(thisTheme);
}



static THEMELIST_PTR theme_get_info(const gchar *pathnamedata, const gchar *filename)
{
	/* given a pathname and a filename, read the file, making a new
	 * theme info structure from it. Return NULL on failure, a pointer
	 * to the new theme info on success. Success includes verifying that
	 * the theme's tile set file actually exixts. Yes, it's a mess.
	 */


	THEMELIST_PTR newTheme;
	gchar         *fbuffer = NULL, *strLine = NULL, *strItem = NULL;
	gchar         *fname = NULL;
	gchar         *userDataDir = filename_expand_tilde(g_strdup(USER_THEME_DIR));
	gchar         *userPixmapDir = filename_expand_tilde(g_strdup(USER_PIXMAP_DIR));
	gint          fpos = 0, lineCount = 0;
	gboolean      done = FALSE, tilesetExists, backgroundExists;



	/* allocate a new theme info structure */
	if ( (newTheme = (THEMELIST_PTR)g_malloc(sizeof(struct ThemeList))) == NULL) {
		gnect_error("gnect: theme_get_info: g_malloc failed\n");
	}
	newTheme->title           = NULL;
	newTheme->fnameTheme      = NULL;
	newTheme->fnameTileset    = NULL;
	newTheme->fnameBackground = NULL;
	newTheme->descrPlayer1    = NULL;
	newTheme->descrPlayer2    = NULL;
	newTheme->gridRGB         = g_strdup(DEFAULT_GRID_COLOUR);
	newTheme->tooltip         = NULL;
	newTheme->isUserTheme     = (strcasecmp(pathnamedata, userDataDir) == 0);
	newTheme->prev            = NULL;
	newTheme->next            = NULL;



	/* get this theme's full path and filename */
	fname = g_strdup_printf("%s/%s", pathnamedata, filename);


	/* copy the file to fbuffer */
	if ( (fbuffer = theme_file_read(fname)) == NULL ) {
		g_warning("gnect: theme_get_info: file_read failed (%s)\n", fname);
		g_free(fname);
		g_free(userDataDir);
		g_free(userPixmapDir);
		g_free(newTheme);
		return(NULL);
	}


	/* parse (in a messy sorta way) the theme file stored in fbuffer */
	while(!done) {
		strLine = theme_file_buffer_getline(fbuffer, &fpos);
		lineCount++;
		done = (strLine == NULL);

		if (!done && strLine[0] != '\0') {

			/* grab keyword or full line (strItem), removing it from strLine */
			strItem = theme_string_bite(strLine, "=");

			/* strip any leading/trailing white space from strItem */
			strItem = g_strstrip(strItem);

			if (g_strcasecmp(strItem, STR_THEME_KEYWORD_TITLE)==0) {
				if (newTheme->title) {
					g_free(newTheme->title);
				}
				newTheme->title = g_strstrip(g_strdup(strLine));
			}
			else if (g_strcasecmp(strItem, STR_THEME_KEYWORD_TOOLTIP)==0) {
				if (newTheme->tooltip) {
					g_free(newTheme->tooltip);
				}
				newTheme->tooltip = g_strstrip(g_strdup(strLine));
			}
			else if (g_strcasecmp(strItem, STR_THEME_KEYWORD_PLAYER1)==0) {
				if (newTheme->descrPlayer1) {
					g_free(newTheme->descrPlayer1);
				}
				if (strlen(strLine) > MAX_LENGTH_PLAYER_DESCR) {
					/* make sure it's a sensible length */
					strLine[MAX_LENGTH_PLAYER_DESCR] = '\0';
				}
				newTheme->descrPlayer1 = g_strstrip(g_strdup(strLine));
			}
			else if (g_strcasecmp(strItem, STR_THEME_KEYWORD_PLAYER2)==0) {
				if (newTheme->descrPlayer2) {
					g_free(newTheme->descrPlayer2);
				}
				if (strlen(strLine) > MAX_LENGTH_PLAYER_DESCR) {
					/* make sure it's a sensible length */
					strLine[MAX_LENGTH_PLAYER_DESCR] = '\0';
				}
				newTheme->descrPlayer2 = g_strstrip(g_strdup(strLine));
			}
			else if (g_strcasecmp(strItem, STR_THEME_KEYWORD_TILE_SET)==0) {
				if (newTheme->fnameTileset) {
					g_free(newTheme->fnameTileset);
				}
				newTheme->fnameTileset = g_strstrip(g_strdup(strLine));
			}
			else if (g_strcasecmp(strItem, STR_THEME_KEYWORD_BACKGROUND)==0) {
				if (newTheme->fnameBackground) {
					g_free(newTheme->fnameBackground);
				}
				newTheme->fnameBackground = g_strstrip(g_strdup(strLine));
			}
			else if (g_strcasecmp(strItem, STR_THEME_KEYWORD_GRID_RGB)==0) {
				if (newTheme->gridRGB) {
					g_free(newTheme->gridRGB);
				}
				newTheme->gridRGB = g_strstrip(g_strdup(strLine));
			}
			else if (g_strcasecmp(strItem, STR_THEME_KEYWORD_NO_GRID)==0) {
				if (newTheme->gridRGB) {
					g_free(newTheme->gridRGB);
				}
				newTheme->gridRGB = NULL;
			}
			else if (strItem[0] == '#' || strItem[0] == '\0') {
				/* comment or blank line - do nothing */
			}
			else {
				g_warning("gnect: error in theme file (%s): line %d", filename, lineCount);
			}


			g_free(strItem);
			strItem = NULL;

		}
		g_free(strLine);
		strLine = NULL;

	}
	g_free(fbuffer);
	fbuffer = NULL;

	g_free(fname);


	/* make sure the specified tile set exists */
	if (newTheme->fnameTileset && newTheme->fnameTileset[0] != '\0') {
		if (newTheme->isUserTheme) {
			fname = g_strdup_printf("%s%s%s", userPixmapDir, PATH_SEP_STR, newTheme->fnameTileset);
			tilesetExists = g_file_test(fname, G_FILE_TEST_ISFILE);
			if (!tilesetExists) {
				/* not in user dir - try Gnect-installed pixmap dir */
				g_free(fname);
				fname = filename_gnect_pixmap(newTheme->fnameTileset, &tilesetExists);
			}
		}
		else {
			fname = filename_gnect_pixmap(newTheme->fnameTileset, &tilesetExists);
		}
		g_free(fname);
	}


	/* check that the specified background exists */
	if (tilesetExists) {
		if (newTheme->fnameBackground && newTheme->fnameBackground[0] != '\0') {
			if (newTheme->isUserTheme) {
				fname = g_strdup_printf("%s%s%s", userPixmapDir, PATH_SEP_STR, newTheme->fnameBackground);
				backgroundExists = g_file_test(fname, G_FILE_TEST_ISFILE);
				if (!backgroundExists) {
					/* not in user dir - try Gnect-installed pixmap dir */
					g_free(fname);
					fname = filename_gnect_pixmap(newTheme->fnameBackground, &backgroundExists);
				}
			}
			else {
				fname = filename_gnect_pixmap(newTheme->fnameBackground, &backgroundExists);
			}
			g_free(fname);
		}
		if (!backgroundExists) {
			/* complain, but continue anyway, if we can't get the background image */
			g_warning("gnect: theme_get_info: background image (%s) for theme (%s) does not exist\n",
					  newTheme->fnameBackground, filename);
			g_free(newTheme->fnameBackground);
			newTheme->fnameBackground = NULL;
		}
	}

	g_free(userDataDir);
	g_free(userPixmapDir);

	/* check that required items were found  */
	if (newTheme->title == NULL || newTheme->title[0] == '\0' ||
		newTheme->descrPlayer1 == NULL || newTheme->descrPlayer1[0] == '\0' ||
		newTheme->descrPlayer2 == NULL || newTheme->descrPlayer2[0] == '\0' ||
		newTheme->fnameTileset == NULL || newTheme->fnameTileset[0] == '\0' ||
		!tilesetExists ) {

		if (tilesetExists) {
			g_warning("gnect: theme_get_info: error in theme (%s) (required item missing?)\n", filename);
		}
		else {
			g_warning("gnect: theme_get_info: tile set specified in theme (%s) does not exist\n", filename);
		}

		theme_free(newTheme);

		return(NULL);
	}

	newTheme->fnameTheme = g_strdup(filename);

	return(newTheme);
}



static gchar *theme_string_bite(gchar *str, const gchar *biteChars)
{
	/* Find first occurance in str of any char in biteChars, then
	 * - return the string preceeding that char
	 * - cut str down to the string following that char
	 */


	gchar *biteStr = NULL;
	gint   strPos = 0, i;


	while(strPos < strlen(str)) {
		i = 0;
		while(i < strlen(biteChars)) {
			if (str[strPos] == biteChars[i]) {
				biteStr = g_strndup(str, strPos);
				strcpy(str, &str[strPos+1]);
				return(biteStr);
			}
			i++;
		}
		strPos++;
	}
	biteStr = g_strdup(str);

	return(biteStr);

}



static gchar *theme_file_read(const gchar *filename)
{
	/* Read a file and return its contents as a string */


	FILE     *fopen(), *fh = NULL;
	gchar    *fbuffer = NULL, *tempStr = NULL, ch;
	gint     fsize = 0, charCount = 0;
	gboolean done = FALSE;


	if ( (fh = fopen(filename, "r")) == NULL ) {
		return(NULL);
	}

	if ( (fbuffer = g_strnfill(THEME_FILEBUFFER_GROW_SIZE, '\0')) == NULL ) {
		fclose(fh);
		gnect_error("gnect: file_read: malloc failed\n");
	}


	while(!done) {

		ch = fgetc(fh);

		if (ch != '\0' && ch != -1){

			fbuffer[fsize] = ch;
			charCount++;
			fsize++;

			if (charCount == THEME_FILEBUFFER_GROW_SIZE){

				/* increase file buffer size */
				tempStr = g_strndup(fbuffer, fsize);
				tempStr[fsize] = '\0';
				g_free(fbuffer);

				if ( (fbuffer = g_strnfill(fsize + THEME_FILEBUFFER_GROW_SIZE + 1, '\0')) == NULL) {
					fclose(fh);
					g_free(tempStr);
					gnect_error("gnect: file_read: malloc failed\n");
				}

				g_snprintf(fbuffer, fsize + 1, "%s", tempStr);
				g_free(tempStr);
				charCount = 0;

			}
		}
		done = ch == -1;
	}

	fclose(fh);
	fbuffer[fsize] = '\0';

	return(fbuffer);
}



static gchar *theme_file_buffer_getline(const gchar *fbuffer, int *fpos)
{
	/* Return one line of a string (as returned by file_read),
	 * setting fpos to index the starting point of the next line.
	 */


	gchar *newStr = NULL;
	gint  len = 0;


	if (fbuffer == NULL || fbuffer[*fpos] == '\0') {
		return(NULL);
	}

	while(fbuffer[*fpos + len] != '\0' &&
		  fbuffer[*fpos + len] != '\n') {
		len++;
	}
	newStr = g_strndup(&fbuffer[*fpos], len);
	*fpos = *fpos + len + 1;

	return(newStr);
}
