/* -*-mode:c; c-style:k&r; c-basic-offset:4; -*- */


/* prefs.c - gnect - a "Four In A Row" game for the GNOME.
 *
 * (c) 2000, 2001 Tim Musson <trmusson@ihug.co.nz>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */



#include <gnome.h>
#include "main.h"
#include "gnect.h"
#include "dialog.h"
#include "file.h"



#define APPLY_RESETS_SCORES            1
#define APPLY_CHANGES_DESCRIPTIONS     2
#define APPLY_RESIZES_BOARD            4
#define APPLY_LOADS_IMAGES             8
#define APPLY_KILLS_GAME              16


extern struct Gnect gnect;
extern gchar msgNewGame[]; /* gui.c */


static GtkWidget *checkbuttonWipe;
static GtkWidget *entryKeyboardLeft;
static GtkWidget *entryKeyboardRight;
static GtkWidget *entryKeyboardDrop;
static GtkWidget *optionmenuTheme;
static GtkWidget *optionmenuTheme_menu;
static GtkWidget *entryDescrOne;
static GtkWidget *entryDescrTwo;

static struct Prefs prefsTemp;
static gint applyFlags;




void prefs_init(void)
{
	/* Read user config */


	gnect.prefs.player1       = gnome_config_get_int("/" PACKAGE "/Preferences/Player1="   DEFAULT_PREFS_PLAYER1);
	gnect.prefs.player2       = gnome_config_get_int("/" PACKAGE "/Preferences/Player2="   DEFAULT_PREFS_PLAYER2);
	gnect.prefs.startMode     = gnome_config_get_int("/" PACKAGE "/Preferences/StartMode=" DEFAULT_PREFS_STARTMODE);
	gnect.prefs.animDelay     = gnome_config_get_int("/" PACKAGE "/Preferences/AnimDelay=" DEFAULT_PREFS_ANIM_DELAY);
	gnect.prefs.keyLeft       = gnome_config_get_int("/" PACKAGE "/Preferences/KeyLeft="   DEFAULT_PREFS_KEY_LEFT);
	gnect.prefs.keyRight      = gnome_config_get_int("/" PACKAGE "/Preferences/KeyRight="  DEFAULT_PREFS_KEY_RIGHT);
	gnect.prefs.keyDrop       = gnome_config_get_int("/" PACKAGE "/Preferences/KeyDrop="   DEFAULT_PREFS_KEY_DROP);
	gnect.prefs.soundMode     = gnome_config_get_int("/" PACKAGE "/Preferences/SoundMode=" DEFAULT_PREFS_SOUNDMODE);
	gnect.prefs.doAnimate     = gnome_config_get_bool("/" PACKAGE "/Preferences/Animate="  DEFAULT_PREFS_DO_ANIMATE);
	gnect.prefs.doWipes       = gnome_config_get_bool("/" PACKAGE "/Preferences/Wipes="    DEFAULT_PREFS_DO_WIPES);
	gnect.prefs.doGrids       = gnome_config_get_bool("/" PACKAGE "/Preferences/Grids="    DEFAULT_PREFS_DO_GRIDS);
	gnect.prefs.fnameTheme    = gnome_config_get_string("/" PACKAGE "/Preferences/Theme="  DEFAULT_PREFS_THEME);
	gnect.prefs.descrPlayer1  = NULL;
	gnect.prefs.descrPlayer1  = NULL;
	gnect.prefs.toggleToolbar = gnome_config_get_bool("/" PACKAGE "/Preferences/Toolbar="  DEFAULT_PREFS_TOGGLE_TOOLBAR);
	gnect.prefs.toggleSound   = gnome_config_get_bool("/" PACKAGE "/Preferences/Sound="    DEFAULT_PREFS_TOGGLE_SOUND);
	gnect.prefs.theme         = NULL;
}



void prefs_free(void)
{
#ifdef GNECT_DEBUGGING
	if (gnect.debugging) {g_print("gnect: prefs_free\n");}
#endif

	if (gnect.prefs.fnameTheme) {
		g_free(gnect.prefs.fnameTheme);
	}
	if (gnect.prefs.descrPlayer1) {
		g_free(gnect.prefs.descrPlayer1);
	}
	if (gnect.prefs.descrPlayer2) {
		g_free(gnect.prefs.descrPlayer2);
	}
	if (prefsTemp.fnameTheme) {
		g_free(prefsTemp.fnameTheme);
	}
	if (prefsTemp.descrPlayer1) {
		g_free(prefsTemp.descrPlayer1);
	}
	if (prefsTemp.descrPlayer2) {
		g_free(prefsTemp.descrPlayer2);
	}
}



void prefs_check(void)
{
	/* Sanity check user config */


	gnect.prefs.doAnimate = gnect.prefs.doAnimate != 0;
	gnect.prefs.doWipes   = gnect.prefs.doWipes   != 0;
	gnect.prefs.doGrids   = gnect.prefs.doGrids   != 0;

	gnect.prefs.toggleToolbar = gnect.prefs.toggleToolbar != 0;
	gnect.prefs.toggleSound   = gnect.prefs.toggleSound   != 0;

	if (gnect.prefs.animDelay < 0) {
		gnect.prefs.animDelay = 0;
		gnect.prefs.doAnimate = FALSE;
	}
	if (gnect.prefs.startMode < 0 || gnect.prefs.startMode > 3) {
		gnect.prefs.startMode = atoi(DEFAULT_PREFS_STARTMODE);
	}
	if (gnect.prefs.player1 < 0 || gnect.prefs.player1 > 4) {
		gnect.prefs.player1 = atoi(DEFAULT_PREFS_PLAYER1);
	}
	if (gnect.prefs.player2 < 0 || gnect.prefs.player2 > 4) {
		gnect.prefs.player2 = atoi(DEFAULT_PREFS_PLAYER2);
	}
	if (gnect.prefs.soundMode < 0 || gnect.prefs.soundMode > 2) {
		gnect.prefs.soundMode = atoi(DEFAULT_PREFS_SOUNDMODE);
	}
}



static void prefs_save(void)
{
	/* Save current settings to user config file */


	gnome_config_set_int("/" PACKAGE "/Preferences/Player1",   gnect.prefs.player1);
	gnome_config_set_int("/" PACKAGE "/Preferences/Player2",   gnect.prefs.player2);
	gnome_config_set_int("/" PACKAGE "/Preferences/StartMode", gnect.prefs.startMode);
	gnome_config_set_int("/" PACKAGE "/Preferences/AnimDelay", gnect.prefs.animDelay);
	gnome_config_set_int("/" PACKAGE "/Preferences/KeyLeft",   gnect.prefs.keyLeft);
	gnome_config_set_int("/" PACKAGE "/Preferences/KeyRight",  gnect.prefs.keyRight);
	gnome_config_set_int("/" PACKAGE "/Preferences/KeyDrop",   gnect.prefs.keyDrop);
	gnome_config_set_int("/" PACKAGE "/Preferences/SoundMode", gnect.prefs.soundMode);
	gnome_config_set_bool("/" PACKAGE "/Preferences/Animate",  gnect.prefs.doAnimate);
	gnome_config_set_bool("/" PACKAGE "/Preferences/Wipes",    gnect.prefs.doWipes);
	gnome_config_set_bool("/" PACKAGE "/Preferences/Grids",    gnect.prefs.doGrids);
	gnome_config_set_string("/" PACKAGE "/Preferences/Theme",  gnect.prefs.fnameTheme);

	gnome_config_sync();
	gui_status_message(_(" Preferences saved"), GNECT_STATUS_MSG_FLASH);
}



static gboolean prefs_changed(void)
{
	/* Has user changed any settings? T/F */


	return((prefsTemp.player1   != gnect.prefs.player1) ||
		   (prefsTemp.player2   != gnect.prefs.player2) ||
		   (prefsTemp.startMode != gnect.prefs.startMode) ||
		   (prefsTemp.keyLeft   != gnect.prefs.keyLeft) ||
		   (prefsTemp.keyRight  != gnect.prefs.keyRight) ||
		   (prefsTemp.keyDrop   != gnect.prefs.keyDrop) ||
		   (prefsTemp.soundMode != gnect.prefs.soundMode) ||
		   (prefsTemp.doAnimate != gnect.prefs.doAnimate) ||
		   (prefsTemp.doWipes   != gnect.prefs.doWipes) ||
		   (strcmp(prefsTemp.fnameTheme, gnect.prefs.fnameTheme) != 0));
}



static void prefs_update_player_labels(void)
{
	/* Update player description labels in prefs dialogue */


	gtk_entry_set_text(GTK_ENTRY(entryDescrOne), prefsTemp.descrPlayer1);
	gtk_entry_set_text(GTK_ENTRY(entryDescrTwo), prefsTemp.descrPlayer2);
}



void prefs_set_player_descriptions(void)
{
	/* Get gnect.prefs.descrPlayer1 and 2 from theme pointed to by gnect.prefs.theme */


	if (gnect.prefs.descrPlayer1) {
		g_free(gnect.prefs.descrPlayer1);
	}
	if (gnect.prefs.descrPlayer2 != NULL) {
		g_free(gnect.prefs.descrPlayer2);
	}
	gnect.prefs.descrPlayer1 = g_strdup(gnect.prefs.theme->descrPlayer1);
	gnect.prefs.descrPlayer2 = g_strdup(gnect.prefs.theme->descrPlayer2);
}



static void cb_prefs_close(const GtkWidget *widget, const gpointer *data)
{
	/* Close the prefs dialogue */


	gtk_widget_destroy(gnect.gui.prefsDialog);
	gnect.gui.prefsDialog = 0;
}



static void cb_prefs_apply(const GtkWidget *widget, const gpointer *data)
{
	/* Apply and save prefs dialogue changes */


	THEMELIST_PTR theme = NULL;


	if ((gint)data != -1) {
		return;
	}


	if (!prefs_changed()) {
		applyFlags = 0;
		return;
	}


	if (prefsTemp.player1 != gnect.prefs.player1 ||
		prefsTemp.player2 != gnect.prefs.player2) {

		applyFlags |= APPLY_CHANGES_DESCRIPTIONS;
		applyFlags |= APPLY_RESETS_SCORES;
		applyFlags |= APPLY_KILLS_GAME;

	}


	/* apply changes to real prefs */

	gnect.prefs.player1   = prefsTemp.player1;
	gnect.prefs.player2   = prefsTemp.player2;
	gnect.prefs.startMode = prefsTemp.startMode;
	gnect.prefs.keyLeft   = prefsTemp.keyLeft;
	gnect.prefs.keyRight  = prefsTemp.keyRight;
	gnect.prefs.keyDrop   = prefsTemp.keyDrop;
	gnect.prefs.doAnimate = prefsTemp.doAnimate;
	gnect.prefs.doWipes   = prefsTemp.doWipes;
	gnect.prefs.soundMode = prefsTemp.soundMode;


	/* perform any special actions according to applyFlags */

	if (applyFlags & APPLY_KILLS_GAME) {
		gnect.over = TRUE;
	}
	if (applyFlags & APPLY_RESETS_SCORES) {
		gnect_reset_scores(GNECT_REFRESH_DISPLAY_TRUE);
	}
	if (applyFlags & APPLY_KILLS_GAME) {
		gnect_reset(GNECT_RESET_AND_WAIT);
		gui_status_message(msgNewGame, GNECT_STATUS_MSG_SET);
	}
	if (applyFlags & APPLY_LOADS_IMAGES) {
		theme = theme_pointer_from_filename(prefsTemp.fnameTheme);
		if (theme) { /* load new theme */
			if (!gfx_load_pixmaps(theme)) {
				gchar *errMsg = g_strdup_printf(_("Unable to load images (%s)"), theme->fnameTheme);
				gnome_app_warning(GNOME_APP(gnect.gui.app), errMsg);
				g_free(errMsg);
			}
			else {
				/* images okay, so allow this to become the new preference */
				if (gnect.prefs.fnameTheme) {
					g_free(gnect.prefs.fnameTheme);
				}
				gnect.prefs.fnameTheme = g_strdup(theme->fnameTheme);
				gnect.prefs.theme = theme;
				prefs_set_player_descriptions();
				if (gnect_n_players() != 1) {
					/* update descriptions in score dialogue unless it's just "Me" vs "You" */
					dialog_score_update();
				}
				if (gnect.over) {
					/* just in case, for example, the status bar is saying
					   "Blue wins!" but blue is now purple... */
					gui_status_message(msgNewGame, GNECT_STATUS_MSG_SET);
				}
			}
		}
	}

	applyFlags = 0;
	prefs_save();

}



static void cb_prefs_animate_select(const GtkWidget *widget, const gpointer *data)
{
	prefsTemp.doAnimate = GTK_TOGGLE_BUTTON(widget)->active;

	/* sensitize/desensitize animated wipes option accordingly */
	if (prefsTemp.doAnimate) {
		gtk_widget_set_sensitive(GTK_WIDGET(checkbuttonWipe), TRUE);
	}
	else {
		gtk_widget_set_sensitive(GTK_WIDGET(checkbuttonWipe), FALSE);
	}
	gnome_property_box_changed(GNOME_PROPERTY_BOX(gnect.gui.prefsDialog));
}



static void cb_prefs_wipe_select(const GtkWidget *widget, const gpointer *data)
{
	prefsTemp.doWipes = GTK_TOGGLE_BUTTON(widget)->active;
	gnome_property_box_changed(GNOME_PROPERTY_BOX(gnect.gui.prefsDialog));
}



static void cb_prefs_sound_select(const GtkWidget *widget, const gpointer *data)
{
	if ( (prefsTemp.soundMode = (gint)data) != gnect.prefs.soundMode ) {
		gnome_property_box_changed(GNOME_PROPERTY_BOX(gnect.gui.prefsDialog));
	}
}



static void cb_prefs_player1_select(const GtkWidget *widget, const gpointer *data)
{
	if ( (prefsTemp.player1 = (gint)data) != gnect.prefs.player1 ) {
		gnome_property_box_changed(GNOME_PROPERTY_BOX(gnect.gui.prefsDialog));
	}
}



static void cb_prefs_player2_select(const GtkWidget *widget, const gpointer *data)
{
	if ( (prefsTemp.player2 = (gint)data) != gnect.prefs.player2 ) {
		gnome_property_box_changed(GNOME_PROPERTY_BOX(gnect.gui.prefsDialog));
	}
}



static void cb_prefs_startmode_select(const GtkWidget *widget, const gpointer *data)
{
	if ( (prefsTemp.startMode = (gint)data) != gnect.prefs.startMode ) {
		gnome_property_box_changed(GNOME_PROPERTY_BOX(gnect.gui.prefsDialog));
	}
}



static void cb_prefs_theme_select(const GtkWidget *widget, const gpointer *data)
{
	THEMELIST_PTR thisTheme = (THEMELIST_PTR)data;


	/* get this theme's filename */
	if (prefsTemp.fnameTheme) {
		g_free(prefsTemp.fnameTheme);
	}
	prefsTemp.fnameTheme = g_strdup(thisTheme->fnameTheme);

	/* get this theme's player descriptions */
	if (prefsTemp.descrPlayer1) {
		g_free(prefsTemp.descrPlayer1);
	}
	if (prefsTemp.descrPlayer2) {
		g_free(prefsTemp.descrPlayer2);
	}
	prefsTemp.descrPlayer1 = g_strdup(thisTheme->descrPlayer1);
	prefsTemp.descrPlayer2 = g_strdup(thisTheme->descrPlayer2);

	prefs_update_player_labels();

	if (strcmp(prefsTemp.fnameTheme, gnect.prefs.fnameTheme) != 0) {
		applyFlags |= APPLY_LOADS_IMAGES;
		applyFlags |= APPLY_CHANGES_DESCRIPTIONS;
		gnome_property_box_changed(GNOME_PROPERTY_BOX(gnect.gui.prefsDialog));
	}
}



static void cb_prefs_key_select(const GtkWidget *widget, const GdkEventKey *data)
{
	gtk_entry_set_text(GTK_ENTRY(widget), gdk_keyval_name(data->keyval));

	if (widget == entryKeyboardLeft) {
		prefsTemp.keyLeft = data->keyval;
	}
	else if (widget == entryKeyboardRight) {
		prefsTemp.keyRight = data->keyval;
	}
	else {
		prefsTemp.keyDrop = data->keyval;
	}
	gnome_property_box_changed(GNOME_PROPERTY_BOX(gnect.gui.prefsDialog));
}



static void cb_prefs_help(const GtkWidget *widget, const gpointer *data)
{
	/* Open a browser and go to the prefs section of gnect's help page */


	gchar *helpfile;


	helpfile = gnome_help_file_find_file(PACKAGE, "index.html");

	if (helpfile) {

		gchar *helpurl;

		helpurl = g_strconcat("file:", helpfile, NULL);
		gnome_help_goto(NULL, helpurl);
		g_free(helpurl);
		g_free(helpfile);
	}
	else {
		g_warning("gnect: couldn't find help file\n");
	}
}



static void prefs_fill_optionmenu_theme(void)
{
	/* Add theme titles to theme selection menu */


	THEMELIST_PTR thisTheme = gnect.themes;
	gint itemno = 0;


	if (thisTheme == NULL) {
		return;
	}

	while(thisTheme->prev) {
		thisTheme = thisTheme->prev;
	}

	while(thisTheme) {

		GtkWidget *item;

		gchar *title = thisTheme->title;

		item = gtk_menu_item_new_with_label(title);
		if (thisTheme->tooltip) {
			gui_set_tooltip(GTK_WIDGET(item), thisTheme->tooltip);
		}
		gtk_widget_show(item);
		gtk_menu_append(GTK_MENU (optionmenuTheme_menu), item);
		gtk_signal_connect(GTK_OBJECT(item), "activate", GTK_SIGNAL_FUNC(cb_prefs_theme_select), thisTheme);

		/* select current theme */
		if (strcmp(gnect.prefs.fnameTheme, thisTheme->fnameTheme) == 0) {
			gtk_menu_set_active(GTK_MENU(optionmenuTheme_menu), itemno);
		}
		itemno++;

		thisTheme = thisTheme->next;
	}
}



void cb_prefs(const GtkWidget *widget, const gpointer *data)
{
	/* The preferences dialogue - originally made with Glade */


	GtkWidget *notebookPrefs;

	GtkWidget *vboxPlayers1;
	GtkWidget *hboxPlayers1;

	GtkWidget *framePlayerOne;
	GtkWidget *vboxPlayerOne;
	GSList    *vboxPlayerOne_group = NULL;
	GtkWidget *radiobuttonPlayer1_0;
	GtkWidget *radiobuttonPlayer1_1;
	GtkWidget *radiobuttonPlayer1_2;
	GtkWidget *radiobuttonPlayer1_3;
	GtkWidget *radiobuttonPlayer1_4;
	
	GtkWidget *framePlayerTwo;
	GtkWidget *vboxPlayerTwo;
	GSList    *vboxPlayerTwo_group = NULL;
	GtkWidget *radiobuttonPlayer2_0;
	GtkWidget *radiobuttonPlayer2_1;
	GtkWidget *radiobuttonPlayer2_2;
	GtkWidget *radiobuttonPlayer2_3;
	GtkWidget *radiobuttonPlayer2_4;

	GtkWidget *framePlayerStart;
	GtkWidget *hboxPlayerStart;
	GtkWidget *optionmenuPlayerStart;
	GtkWidget *optionmenuPlayerStart_menu;
	GtkWidget *labelPlayersTab;

	GtkWidget *glade_menuitem;

	GtkWidget *vboxGraphics;
	GtkWidget *frameTheme;
	GtkWidget *vboxTheme;
	GtkWidget *hboxTheme;
	GtkWidget *tableDescr;
	GtkWidget *labelDescrOne;
	GtkWidget *labelDescrTwo;

	GtkWidget *frameAnimation;
	GtkWidget *vboxAnimation;
	GtkWidget *labelAppearanceTab;
	GtkWidget *checkbuttonAnimate;

	GtkWidget *vboxKeyboard;
	GtkWidget *frameKeyboard;
	GtkWidget *hboxKeyboard;
	GtkWidget *tableKeyboard;
	GtkWidget *labelKeyboardDrop;
	GtkWidget *labelKeyboardRight;
	GtkWidget *labelKeyboardLeft;
	GtkWidget *frameSound;
	GtkWidget *hboxSound;
	GtkWidget *optionmenuSound;
	GtkWidget *optionmenuSound_menu;
	GtkWidget *labelKeyboardTab;
#ifdef ANNOYING_TOOLTIPS
	gchar msgPrefsWarning[] = N_("Changing this option will end a game in progress");
#endif


	/* don't bother if it's already open */
	if (gnect.gui.prefsDialog) {
		return;
	}


	applyFlags = 0;


	/* temporary copies of current settings, in case user cancels */
	prefsTemp.player1   = gnect.prefs.player1;
	prefsTemp.player2   = gnect.prefs.player2;
	prefsTemp.startMode = gnect.prefs.startMode;
	prefsTemp.keyLeft   = gnect.prefs.keyLeft;
	prefsTemp.keyRight  = gnect.prefs.keyRight;
	prefsTemp.keyDrop   = gnect.prefs.keyDrop;
	prefsTemp.doAnimate = gnect.prefs.doAnimate;
	prefsTemp.doWipes   = gnect.prefs.doWipes;
	prefsTemp.soundMode = gnect.prefs.soundMode;

	if (prefsTemp.fnameTheme) {
		g_free(prefsTemp.fnameTheme);
	}
	if (prefsTemp.descrPlayer1) {
		g_free(prefsTemp.descrPlayer1);
	}
	if (prefsTemp.descrPlayer2) {
		g_free(prefsTemp.descrPlayer2);
	}
	prefsTemp.fnameTheme   = g_strdup(gnect.prefs.fnameTheme);
	prefsTemp.descrPlayer1 = g_strdup(gnect.prefs.descrPlayer1);
	prefsTemp.descrPlayer2 = g_strdup(gnect.prefs.descrPlayer2);


	/* build the dialogue... */
	gnect.gui.prefsDialog = gnome_property_box_new();
	gtk_object_set_data(GTK_OBJECT(gnect.gui.prefsDialog), "prefsDialog", gnect.gui.prefsDialog);

	gtk_window_set_title(GTK_WINDOW(&GNOME_PROPERTY_BOX(gnect.gui.prefsDialog)->dialog.window), _("Gnect Preferences"));
	gtk_window_set_wmclass(GTK_WINDOW(&GNOME_PROPERTY_BOX(gnect.gui.prefsDialog)->dialog.window), PACKAGE, "preferences");

	notebookPrefs = GNOME_PROPERTY_BOX(gnect.gui.prefsDialog)->notebook;
	gtk_object_set_data(GTK_OBJECT(gnect.gui.prefsDialog), "notebookPrefs", notebookPrefs);
	gtk_widget_show(notebookPrefs);
	gtk_notebook_set_tab_hborder(GTK_NOTEBOOK(notebookPrefs), 10);
	gtk_notebook_set_tab_vborder(GTK_NOTEBOOK(notebookPrefs), 4);

	/* P L A Y E R S   T A B */
	vboxPlayers1 = gtk_vbox_new(FALSE, 0);
	gtk_widget_ref(vboxPlayers1);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "vboxPlayers1", vboxPlayers1, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(vboxPlayers1);
	gtk_container_add(GTK_CONTAINER(notebookPrefs), vboxPlayers1);

	/* p l a y e r   s e l e c t i o n */
	hboxPlayers1 = gtk_hbox_new(FALSE, 0);
	gtk_widget_ref(hboxPlayers1);
	gtk_object_set_data_full(GTK_OBJECT (gnect.gui.prefsDialog), "hboxPlayers1", hboxPlayers1, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(hboxPlayers1);
	gtk_box_pack_start(GTK_BOX(vboxPlayers1), hboxPlayers1, TRUE, TRUE, 0);

	/* player one */
	framePlayerOne = gtk_frame_new(_(" Player One "));
	gtk_widget_ref(framePlayerOne);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "framePlayerOne", framePlayerOne, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(framePlayerOne);
	gtk_box_pack_start(GTK_BOX(hboxPlayers1), framePlayerOne, TRUE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(framePlayerOne), 5);

	vboxPlayerOne = gtk_vbox_new(FALSE, 0);
	gtk_widget_ref(vboxPlayerOne);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "vboxPlayerOne", vboxPlayerOne, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (vboxPlayerOne);
	gtk_container_add(GTK_CONTAINER(framePlayerOne), vboxPlayerOne);
	gtk_container_set_border_width(GTK_CONTAINER(vboxPlayerOne), 5);

	radiobuttonPlayer1_0 = gtk_radio_button_new_with_label(vboxPlayerOne_group, _("Human"));
	vboxPlayerOne_group = gtk_radio_button_group(GTK_RADIO_BUTTON(radiobuttonPlayer1_0));
	gtk_widget_ref(radiobuttonPlayer1_0);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "radiobuttonPlayer1_0", radiobuttonPlayer1_0, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(radiobuttonPlayer1_0);
	gtk_box_pack_start(GTK_BOX(vboxPlayerOne), radiobuttonPlayer1_0, FALSE, FALSE, 0);
#ifdef ANNOYING_TOOLTIPS
	gui_set_tooltip(GTK_WIDGET(radiobuttonPlayer1_0), msgPrefsWarning);
#endif

	radiobuttonPlayer1_1 = gtk_radio_button_new_with_label(vboxPlayerOne_group, "Gnect");
	vboxPlayerOne_group = gtk_radio_button_group(GTK_RADIO_BUTTON(radiobuttonPlayer1_1));
	gtk_widget_ref(radiobuttonPlayer1_1);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "radiobuttonPlayer1_1", radiobuttonPlayer1_1, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(radiobuttonPlayer1_1);
	gtk_box_pack_start(GTK_BOX(vboxPlayerOne), radiobuttonPlayer1_1, FALSE, FALSE, 0);
#ifdef ANNOYING_TOOLTIPS
	gui_set_tooltip(GTK_WIDGET(radiobuttonPlayer1_1), msgPrefsWarning);
#endif

	radiobuttonPlayer1_2 = gtk_radio_button_new_with_label(vboxPlayerOne_group, _("Velena Engine: Weak"));
	vboxPlayerOne_group = gtk_radio_button_group(GTK_RADIO_BUTTON(radiobuttonPlayer1_2));
	gtk_widget_ref(radiobuttonPlayer1_2);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "radiobuttonPlayer1_2", radiobuttonPlayer1_2, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(radiobuttonPlayer1_2);
	gtk_box_pack_start(GTK_BOX(vboxPlayerOne), radiobuttonPlayer1_2, FALSE, FALSE, 0);
#ifdef ANNOYING_TOOLTIPS
	gui_set_tooltip(GTK_WIDGET(radiobuttonPlayer1_2), msgPrefsWarning);
#endif

	radiobuttonPlayer1_3 = gtk_radio_button_new_with_label(vboxPlayerOne_group, _("Velena Engine: Medium"));
	vboxPlayerOne_group = gtk_radio_button_group(GTK_RADIO_BUTTON(radiobuttonPlayer1_3));
	gtk_widget_ref(radiobuttonPlayer1_3);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "radiobuttonPlayer1_3", radiobuttonPlayer1_3, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(radiobuttonPlayer1_3);
	gtk_box_pack_start(GTK_BOX(vboxPlayerOne), radiobuttonPlayer1_3, FALSE, FALSE, 0);
#ifdef ANNOYING_TOOLTIPS
	gui_set_tooltip(GTK_WIDGET(radiobuttonPlayer1_3), msgPrefsWarning);
#endif

	radiobuttonPlayer1_4 = gtk_radio_button_new_with_label(vboxPlayerOne_group, _("Velena Engine: Strong"));
	vboxPlayerOne_group = gtk_radio_button_group(GTK_RADIO_BUTTON (radiobuttonPlayer1_4));
	gtk_widget_ref(radiobuttonPlayer1_4);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "radiobuttonPlayer1_4", radiobuttonPlayer1_4, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(radiobuttonPlayer1_4);
	gtk_box_pack_start(GTK_BOX(vboxPlayerOne), radiobuttonPlayer1_4, FALSE, FALSE, 0);
#ifdef ANNOYING_TOOLTIPS
	gui_set_tooltip(GTK_WIDGET(radiobuttonPlayer1_4), msgPrefsWarning);
#endif

	switch(gnect.prefs.player1) {
	case SELECT_PLAYER_HUMAN :
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radiobuttonPlayer1_0), TRUE);
		break;
	case SELECT_PLAYER_GNECT :
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radiobuttonPlayer1_1), TRUE);
		break;
	case SELECT_PLAYER_VELENA_WEAK :
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radiobuttonPlayer1_2), TRUE);
		break;
	case SELECT_PLAYER_VELENA_MEDIUM :
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radiobuttonPlayer1_3), TRUE);
		break;
	case SELECT_PLAYER_VELENA_STRONG :
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radiobuttonPlayer1_4), TRUE);
		break;
	}

	/* player two */
	framePlayerTwo = gtk_frame_new(_(" Player Two "));
	gtk_widget_ref(framePlayerTwo);
	gtk_object_set_data_full(GTK_OBJECT (gnect.gui.prefsDialog), "framePlayerTwo", framePlayerTwo, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(framePlayerTwo);
	gtk_box_pack_start(GTK_BOX(hboxPlayers1), framePlayerTwo, TRUE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(framePlayerTwo), 5);

	vboxPlayerTwo = gtk_vbox_new(FALSE, 0);
	gtk_widget_ref(vboxPlayerTwo);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "vboxPlayerTwo", vboxPlayerTwo, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(vboxPlayerTwo);
	gtk_container_add(GTK_CONTAINER(framePlayerTwo), vboxPlayerTwo);
	gtk_container_set_border_width(GTK_CONTAINER(vboxPlayerTwo), 5);

	radiobuttonPlayer2_0 = gtk_radio_button_new_with_label(vboxPlayerTwo_group, _("Human"));
	vboxPlayerTwo_group = gtk_radio_button_group(GTK_RADIO_BUTTON(radiobuttonPlayer2_0));
	gtk_widget_ref(radiobuttonPlayer2_0);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "radiobuttonPlayer2_0", radiobuttonPlayer2_0, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(radiobuttonPlayer2_0);
	gtk_box_pack_start(GTK_BOX(vboxPlayerTwo), radiobuttonPlayer2_0, FALSE, FALSE, 0);
#ifdef ANNOYING_TOOLTIPS
	gui_set_tooltip(GTK_WIDGET(radiobuttonPlayer2_0), msgPrefsWarning);
#endif

	radiobuttonPlayer2_1 = gtk_radio_button_new_with_label(vboxPlayerTwo_group, "Gnect");
	vboxPlayerTwo_group = gtk_radio_button_group(GTK_RADIO_BUTTON(radiobuttonPlayer2_1));
	gtk_widget_ref(radiobuttonPlayer2_1);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "radiobuttonPlayer2_1", radiobuttonPlayer2_1, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(radiobuttonPlayer2_1);
	gtk_box_pack_start(GTK_BOX(vboxPlayerTwo), radiobuttonPlayer2_1, FALSE, FALSE, 0);
#ifdef ANNOYING_TOOLTIPS
	gui_set_tooltip(GTK_WIDGET(radiobuttonPlayer2_1), msgPrefsWarning);
#endif

	radiobuttonPlayer2_2 = gtk_radio_button_new_with_label(vboxPlayerTwo_group, _("Velena Engine: Weak"));
	vboxPlayerTwo_group = gtk_radio_button_group(GTK_RADIO_BUTTON(radiobuttonPlayer2_2));
	gtk_widget_ref(radiobuttonPlayer2_2);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "radiobuttonPlayer2_2", radiobuttonPlayer2_2, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(radiobuttonPlayer2_2);
	gtk_box_pack_start(GTK_BOX(vboxPlayerTwo), radiobuttonPlayer2_2, FALSE, FALSE, 0);
#ifdef ANNOYING_TOOLTIPS
	gui_set_tooltip(GTK_WIDGET(radiobuttonPlayer2_2), msgPrefsWarning);
#endif

	radiobuttonPlayer2_3 = gtk_radio_button_new_with_label(vboxPlayerTwo_group, _("Velena Engine: Medium"));
	vboxPlayerTwo_group = gtk_radio_button_group(GTK_RADIO_BUTTON(radiobuttonPlayer2_3));
	gtk_widget_ref(radiobuttonPlayer2_3);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "radiobuttonPlayer2_3", radiobuttonPlayer2_3, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(radiobuttonPlayer2_3);
	gtk_box_pack_start(GTK_BOX(vboxPlayerTwo), radiobuttonPlayer2_3, FALSE, FALSE, 0);
#ifdef ANNOYING_TOOLTIPS
	gui_set_tooltip(GTK_WIDGET(radiobuttonPlayer2_3), msgPrefsWarning);
#endif

	radiobuttonPlayer2_4 = gtk_radio_button_new_with_label(vboxPlayerTwo_group, _("Velena Engine: Strong"));
	vboxPlayerTwo_group = gtk_radio_button_group(GTK_RADIO_BUTTON(radiobuttonPlayer2_4));
	gtk_widget_ref(radiobuttonPlayer2_4);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "radiobuttonPlayer2_4", radiobuttonPlayer2_4, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(radiobuttonPlayer2_4);
	gtk_box_pack_start(GTK_BOX(vboxPlayerTwo), radiobuttonPlayer2_4, FALSE, FALSE, 0);
#ifdef ANNOYING_TOOLTIPS
	gui_set_tooltip(GTK_WIDGET(radiobuttonPlayer2_4), msgPrefsWarning);
#endif

	switch(gnect.prefs.player2) {
	case SELECT_PLAYER_HUMAN :
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radiobuttonPlayer2_0), TRUE);
		break;
	case SELECT_PLAYER_GNECT :
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radiobuttonPlayer2_1), TRUE);
		break;
	case SELECT_PLAYER_VELENA_WEAK :
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radiobuttonPlayer2_2), TRUE);
		break;
	case SELECT_PLAYER_VELENA_MEDIUM :
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radiobuttonPlayer2_3), TRUE);
		break;
	case SELECT_PLAYER_VELENA_STRONG :
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radiobuttonPlayer2_4), TRUE);
		break;
	}

	/* f i r s t   m o v e */
	framePlayerStart = gtk_frame_new(_(" When starting a new game, who moves first? "));
	gtk_widget_ref(framePlayerStart);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "framePlayerStart", framePlayerStart, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(framePlayerStart);
	gtk_box_pack_start(GTK_BOX(vboxPlayers1), framePlayerStart, TRUE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(framePlayerStart), 5);

	hboxPlayerStart = gtk_hbox_new(FALSE, 0);
	gtk_widget_ref(hboxPlayerStart);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "hboxPlayerStart", hboxPlayerStart, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(hboxPlayerStart);
	gtk_container_add(GTK_CONTAINER(framePlayerStart), hboxPlayerStart);
	gtk_container_set_border_width(GTK_CONTAINER(hboxPlayerStart), 5);

	optionmenuPlayerStart = gtk_option_menu_new();
	gtk_widget_ref(optionmenuPlayerStart);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "optionmenuPlayerStart", optionmenuPlayerStart, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(optionmenuPlayerStart);
	gtk_box_pack_start(GTK_BOX(hboxPlayerStart), optionmenuPlayerStart, TRUE, FALSE, 0);
	optionmenuPlayerStart_menu = gtk_menu_new();
	glade_menuitem = gtk_menu_item_new_with_label(_("Player One "));
	gtk_widget_show(glade_menuitem);
	gtk_menu_append(GTK_MENU(optionmenuPlayerStart_menu), glade_menuitem);
	gtk_signal_connect(GTK_OBJECT(glade_menuitem), "activate", GTK_SIGNAL_FUNC(cb_prefs_startmode_select), (gpointer) SELECT_STARTMODE_PLAYER_ONE);
	if (gnect.prefs.startMode == SELECT_STARTMODE_PLAYER_ONE) {
		gtk_menu_set_active(GTK_MENU(optionmenuPlayerStart_menu), SELECT_STARTMODE_PLAYER_ONE);
	}
	glade_menuitem = gtk_menu_item_new_with_label(_("Player Two "));
	gtk_widget_show(glade_menuitem);
	gtk_menu_append(GTK_MENU(optionmenuPlayerStart_menu), glade_menuitem);
	gtk_signal_connect(GTK_OBJECT(glade_menuitem), "activate", GTK_SIGNAL_FUNC(cb_prefs_startmode_select), (gpointer) SELECT_STARTMODE_PLAYER_TWO);
	if (gnect.prefs.startMode == SELECT_STARTMODE_PLAYER_TWO) {
		gtk_menu_set_active(GTK_MENU(optionmenuPlayerStart_menu), SELECT_STARTMODE_PLAYER_TWO);
	}
	glade_menuitem = gtk_menu_item_new_with_label(_("Pick a player at random "));
	gtk_widget_show(glade_menuitem);
	gtk_menu_append(GTK_MENU(optionmenuPlayerStart_menu), glade_menuitem);
	gtk_signal_connect(GTK_OBJECT(glade_menuitem), "activate", GTK_SIGNAL_FUNC(cb_prefs_startmode_select), (gpointer) SELECT_STARTMODE_RANDOM);
	if (gnect.prefs.startMode == SELECT_STARTMODE_RANDOM) {
		gtk_menu_set_active(GTK_MENU(optionmenuPlayerStart_menu), SELECT_STARTMODE_RANDOM);
	}
	glade_menuitem = gtk_menu_item_new_with_label(_("Players take turns "));
	gtk_widget_show(glade_menuitem);
	gtk_menu_append(GTK_MENU (optionmenuPlayerStart_menu), glade_menuitem);
	gtk_signal_connect(GTK_OBJECT(glade_menuitem), "activate", GTK_SIGNAL_FUNC(cb_prefs_startmode_select), (gpointer) SELECT_STARTMODE_ALTERNATE);
	if (gnect.prefs.startMode == SELECT_STARTMODE_ALTERNATE) {
		gtk_menu_set_active(GTK_MENU(optionmenuPlayerStart_menu), SELECT_STARTMODE_ALTERNATE);
	}
	gtk_option_menu_set_menu(GTK_OPTION_MENU(optionmenuPlayerStart), optionmenuPlayerStart_menu);

	labelPlayersTab = gtk_label_new(_("Player Selection"));
	gtk_widget_ref(labelPlayersTab);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "labelPlayersTab", labelPlayersTab, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(labelPlayersTab);
	gtk_notebook_set_tab_label(GTK_NOTEBOOK(notebookPrefs), gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebookPrefs), 0), labelPlayersTab);

	/* A N I M A T I O N   T A B */
	vboxGraphics = gtk_vbox_new(FALSE, 0);
	gtk_widget_ref(vboxGraphics);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "vboxGraphics", vboxGraphics, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(vboxGraphics);
	gtk_container_add(GTK_CONTAINER(notebookPrefs), vboxGraphics);

	/* t h e m e */
	frameTheme = gtk_frame_new(_(" Theme "));
	gtk_widget_ref(frameTheme);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "frameTheme", frameTheme, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(frameTheme);
	gtk_box_pack_start(GTK_BOX(vboxGraphics), frameTheme, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(frameTheme), 5);

	vboxTheme = gtk_vbox_new(FALSE, 0);
	gtk_widget_ref(vboxTheme);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "vboxTheme", vboxTheme, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(vboxTheme);
	gtk_container_add(GTK_CONTAINER(frameTheme), vboxTheme);

	hboxTheme = gtk_hbox_new(FALSE, 0);
	gtk_widget_ref(hboxTheme);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "hboxTheme", hboxTheme, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(hboxTheme);
	gtk_box_pack_start(GTK_BOX(vboxTheme), hboxTheme, FALSE, TRUE, 0);

	optionmenuTheme = gtk_option_menu_new();
	gtk_widget_ref(optionmenuTheme);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "optionmenuTheme", optionmenuTheme, (GtkDestroyNotify) gtk_widget_unref);

	optionmenuTheme_menu = gtk_menu_new();
	gtk_widget_ref(optionmenuTheme_menu);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "optionmenuTheme_menu", optionmenuTheme_menu, (GtkDestroyNotify) gtk_widget_unref);
	prefs_fill_optionmenu_theme();
	gtk_widget_show(glade_menuitem);
	gtk_option_menu_set_menu(GTK_OPTION_MENU(optionmenuTheme), optionmenuTheme_menu);
	gtk_box_pack_start(GTK_BOX(hboxTheme), optionmenuTheme, FALSE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER (optionmenuTheme), 5);

	/* player descriptions */
	tableDescr = gtk_table_new(2, 2, FALSE);
	gtk_widget_ref(tableDescr);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "tableDescr", tableDescr, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(tableDescr);
	gtk_box_pack_start(GTK_BOX(vboxTheme), tableDescr, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(tableDescr), 5);
	gtk_table_set_col_spacings(GTK_TABLE(tableDescr), 5);

	labelDescrOne = gtk_label_new(_("Player One:"));
	gtk_widget_ref(labelDescrOne);
	gtk_object_set_data_full (GTK_OBJECT(gnect.gui.prefsDialog), "labelDescrOne", labelDescrOne, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(labelDescrOne);
	gtk_table_attach(GTK_TABLE(tableDescr), labelDescrOne, 0, 1, 0, 1, (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);

	labelDescrTwo = gtk_label_new(_("Player Two: "));
	gtk_widget_ref(labelDescrTwo);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "labelDescrTwo", labelDescrTwo, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(labelDescrTwo);
	gtk_table_attach(GTK_TABLE(tableDescr), labelDescrTwo, 0, 1, 1, 2, (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);

	entryDescrOne = gtk_entry_new_with_max_length(MAX_LENGTH_PLAYER_DESCR + 1);
	gtk_widget_ref(entryDescrOne);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "entryDescrOne", entryDescrOne, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(entryDescrOne);
	gtk_table_attach(GTK_TABLE(tableDescr), entryDescrOne, 1, 2, 0, 1, (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (0), 0, 0);
	gtk_entry_set_editable(GTK_ENTRY(entryDescrOne), FALSE);
	gtk_entry_set_text(GTK_ENTRY(entryDescrOne), gnect.prefs.descrPlayer1);
	gtk_widget_set_sensitive(GTK_WIDGET(entryDescrOne), FALSE);

	entryDescrTwo = gtk_entry_new_with_max_length(MAX_LENGTH_PLAYER_DESCR + 1);
	gtk_widget_ref(entryDescrTwo);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "entryDescrTwo", entryDescrTwo, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(entryDescrTwo);
	gtk_table_attach(GTK_TABLE(tableDescr), entryDescrTwo, 1, 2, 1, 2, (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (0), 0, 0);
	gtk_entry_set_editable(GTK_ENTRY (entryDescrTwo), FALSE);
	gtk_entry_set_text(GTK_ENTRY(entryDescrTwo), gnect.prefs.descrPlayer2);
	gtk_widget_set_sensitive(GTK_WIDGET(entryDescrTwo), FALSE);

	/* a n i m a t i o n */
	frameAnimation = gtk_frame_new(_(" Animation "));
	gtk_widget_ref(frameAnimation);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "frameAnimation", frameAnimation, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(frameAnimation);
	gtk_box_pack_start(GTK_BOX(vboxGraphics), frameAnimation, FALSE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(frameAnimation), 5);

	vboxAnimation = gtk_vbox_new(FALSE, 0);
	gtk_widget_ref(vboxAnimation);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "vboxAnimation", vboxAnimation, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(vboxAnimation);
	gtk_container_add(GTK_CONTAINER(frameAnimation), vboxAnimation);

	/* animate */
	checkbuttonAnimate = gtk_check_button_new_with_label(_("Enable animation"));
	gtk_widget_ref(checkbuttonAnimate);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "checkbuttonAnimate", checkbuttonAnimate, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(checkbuttonAnimate);
	gtk_box_pack_start(GTK_BOX(vboxAnimation), checkbuttonAnimate, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(checkbuttonAnimate), 5);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbuttonAnimate), gnect.prefs.doAnimate);

	/* wipe */
	checkbuttonWipe = gtk_check_button_new_with_label(_("Enable animated wipes"));
	gtk_widget_ref(checkbuttonWipe);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "checkbuttonWipe", checkbuttonWipe, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(checkbuttonWipe);
	gtk_box_pack_start(GTK_BOX(vboxAnimation), checkbuttonWipe, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(checkbuttonWipe), 5);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbuttonWipe), gnect.prefs.doWipes);

	gtk_widget_set_sensitive(GTK_WIDGET(checkbuttonWipe), gnect.prefs.doAnimate);

	labelAppearanceTab = gtk_label_new(_("Appearance"));
	gtk_widget_ref(labelAppearanceTab);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "labelAppearanceTab", labelAppearanceTab, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(labelAppearanceTab);
	gtk_notebook_set_tab_label(GTK_NOTEBOOK(notebookPrefs), gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebookPrefs), 1), labelAppearanceTab);

	/* M I S C E L L A N E O U S   T A B */
	vboxKeyboard = gtk_vbox_new(FALSE, 0);
	gtk_widget_ref(vboxKeyboard);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "vboxKeyboard", vboxKeyboard, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(vboxKeyboard);
	gtk_container_add(GTK_CONTAINER(notebookPrefs), vboxKeyboard);

	/* s o u n d */
	frameSound = gtk_frame_new(_(" Sound "));
	gtk_widget_ref(frameSound);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "frameSound", frameSound,(GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(frameSound);
	gtk_box_pack_start(GTK_BOX (vboxKeyboard), frameSound, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(frameSound), 5);

	hboxSound = gtk_hbox_new(FALSE, 0);
	gtk_widget_ref(hboxSound);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "hboxSound", hboxSound, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(hboxSound);
	gtk_container_add(GTK_CONTAINER(frameSound), hboxSound);

	optionmenuSound = gtk_option_menu_new();
	gtk_widget_ref(optionmenuSound);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "optionmenuSound", optionmenuSound, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(optionmenuSound);
	gtk_box_pack_start(GTK_BOX(hboxSound), optionmenuSound, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(optionmenuSound), 5);

	optionmenuSound_menu = gtk_menu_new();

	glade_menuitem = gtk_menu_item_new_with_label(_("No sound"));
	gtk_widget_show(glade_menuitem);
	gtk_menu_append(GTK_MENU(optionmenuSound_menu), glade_menuitem);
	gtk_signal_connect(GTK_OBJECT(glade_menuitem), "activate", GTK_SIGNAL_FUNC(cb_prefs_sound_select), (gpointer) SELECT_SOUNDMODE_NONE);
	if (gnect.prefs.soundMode == SELECT_SOUNDMODE_NONE) {
		gtk_menu_set_active(GTK_MENU(optionmenuSound_menu), SELECT_SOUNDMODE_NONE);
	}

	glade_menuitem = gtk_menu_item_new_with_label(_("Speaker beep"));
	gtk_widget_show(glade_menuitem);
	gtk_menu_append(GTK_MENU(optionmenuSound_menu), glade_menuitem);
	gtk_signal_connect(GTK_OBJECT(glade_menuitem), "activate", GTK_SIGNAL_FUNC(cb_prefs_sound_select), (gpointer) SELECT_SOUNDMODE_BEEP);
	if (gnect.prefs.soundMode == SELECT_SOUNDMODE_BEEP) {
		gtk_menu_set_active(GTK_MENU(optionmenuSound_menu), SELECT_SOUNDMODE_BEEP);
	}

	glade_menuitem = gtk_menu_item_new_with_label(_("GNOME sound events"));
	gtk_widget_show(glade_menuitem);
	gui_set_tooltip(GTK_WIDGET(glade_menuitem), _("You can configure these via the GNOME Control Center"));
	gtk_menu_append(GTK_MENU(optionmenuSound_menu), glade_menuitem);
	gtk_signal_connect(GTK_OBJECT(glade_menuitem), "activate", GTK_SIGNAL_FUNC(cb_prefs_sound_select), (gpointer) SELECT_SOUNDMODE_PLAY);
	if (gnect.prefs.soundMode == SELECT_SOUNDMODE_PLAY) {
		gtk_menu_set_active(GTK_MENU(optionmenuSound_menu), SELECT_SOUNDMODE_PLAY);
	}

	gtk_option_menu_set_menu(GTK_OPTION_MENU(optionmenuSound), optionmenuSound_menu);

	/* k e y b o a r d */
	frameKeyboard = gtk_frame_new(_(" Keyboard "));
	gtk_widget_ref(frameKeyboard);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "frameKeyboard", frameKeyboard, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(frameKeyboard);
	gtk_box_pack_start(GTK_BOX(vboxKeyboard), frameKeyboard, FALSE, TRUE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(frameKeyboard), 5);

	hboxKeyboard = gtk_hbox_new(FALSE, 0);
	gtk_widget_ref(hboxKeyboard);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "hboxKeyboard", hboxKeyboard, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(hboxKeyboard);
	gtk_container_add(GTK_CONTAINER(frameKeyboard), hboxKeyboard);
	gtk_container_set_border_width(GTK_CONTAINER(hboxKeyboard), 10);

	tableKeyboard = gtk_table_new(3, 2, FALSE);
	gtk_widget_ref(tableKeyboard);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "tableKeyboard", tableKeyboard, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(tableKeyboard);
	gtk_box_pack_start(GTK_BOX(hboxKeyboard), tableKeyboard, FALSE, FALSE, 0);
	gtk_table_set_row_spacings(GTK_TABLE (tableKeyboard), 10);
	gtk_table_set_col_spacings(GTK_TABLE (tableKeyboard), 5);

	/* key left */
	entryKeyboardLeft = gtk_entry_new();
	gtk_widget_ref(entryKeyboardLeft);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "entryKeyboardLeft", entryKeyboardLeft, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(entryKeyboardLeft);
	gtk_table_attach(GTK_TABLE(tableKeyboard), entryKeyboardLeft, 1, 2, 0, 1, (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (0), 0, 0);

	gtk_entry_set_editable(GTK_ENTRY(entryKeyboardLeft), FALSE);
	gtk_entry_set_text(GTK_ENTRY(entryKeyboardLeft), gdk_keyval_name(gnect.prefs.keyLeft));

	/* key right */
	entryKeyboardRight = gtk_entry_new();
	gtk_widget_ref(entryKeyboardRight);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "entryKeyboardRight", entryKeyboardRight, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(entryKeyboardRight);
	gtk_table_attach(GTK_TABLE(tableKeyboard), entryKeyboardRight, 1, 2, 1, 2, (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (0), 0, 0);

	gtk_entry_set_editable(GTK_ENTRY(entryKeyboardRight), FALSE);
	gtk_entry_set_text(GTK_ENTRY(entryKeyboardRight), gdk_keyval_name(gnect.prefs.keyRight));

	/* key drop */
	entryKeyboardDrop = gtk_entry_new();
	gtk_widget_ref(entryKeyboardDrop);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "entryKeyboardDrop", entryKeyboardDrop, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(entryKeyboardDrop);
	gtk_table_attach(GTK_TABLE(tableKeyboard), entryKeyboardDrop, 1, 2, 2, 3, (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (0), 0, 0);

	gtk_entry_set_editable(GTK_ENTRY(entryKeyboardDrop), FALSE);
	gtk_entry_set_text(GTK_ENTRY(entryKeyboardDrop), gdk_keyval_name(gnect.prefs.keyDrop));

	/* key labels */
	labelKeyboardDrop = gtk_label_new(_("Drop Counter "));
	gtk_widget_ref(labelKeyboardDrop);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "labelKeyboardDrop", labelKeyboardDrop, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(labelKeyboardDrop);
	gtk_table_attach(GTK_TABLE(tableKeyboard), labelKeyboardDrop, 0, 1, 2, 3, (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);

	labelKeyboardRight = gtk_label_new(_("Move Right "));
	gtk_widget_ref(labelKeyboardRight);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "labelKeyboardRight", labelKeyboardRight, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(labelKeyboardRight);
	gtk_table_attach(GTK_TABLE(tableKeyboard), labelKeyboardRight, 0, 1, 1, 2, (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);

	labelKeyboardLeft = gtk_label_new(_("Move Left "));
	gtk_widget_ref(labelKeyboardLeft);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "labelKeyboardLeft", labelKeyboardLeft, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(labelKeyboardLeft);
	gtk_table_attach(GTK_TABLE (tableKeyboard), labelKeyboardLeft, 0, 1, 0, 1, (GtkAttachOptions) (0), (GtkAttachOptions) (0), 0, 0);

	labelKeyboardTab = gtk_label_new(_("Miscellaneous"));
	gtk_widget_ref(labelKeyboardTab);
	gtk_object_set_data_full(GTK_OBJECT(gnect.gui.prefsDialog), "labelKeyboardTab", labelKeyboardTab, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show(labelKeyboardTab);
	gtk_notebook_set_tab_label(GTK_NOTEBOOK(notebookPrefs), gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebookPrefs), 2), labelKeyboardTab);

	/* C A L L B A C K S */
	gtk_signal_connect(GTK_OBJECT(gnect.gui.prefsDialog), "destroy", GTK_SIGNAL_FUNC(cb_prefs_close), NULL);
	gtk_signal_connect(GTK_OBJECT(gnect.gui.prefsDialog), "apply", GTK_SIGNAL_FUNC(cb_prefs_apply), NULL);
	gtk_signal_connect(GTK_OBJECT(gnect.gui.prefsDialog), "help", GTK_SIGNAL_FUNC(cb_prefs_help), NULL);

	gtk_signal_connect(GTK_OBJECT(radiobuttonPlayer1_0), "toggled", GTK_SIGNAL_FUNC(cb_prefs_player1_select), (gpointer) SELECT_PLAYER_HUMAN);
	gtk_signal_connect(GTK_OBJECT(radiobuttonPlayer1_1), "toggled", GTK_SIGNAL_FUNC(cb_prefs_player1_select), (gpointer) SELECT_PLAYER_GNECT);
	gtk_signal_connect(GTK_OBJECT(radiobuttonPlayer1_2), "toggled", GTK_SIGNAL_FUNC(cb_prefs_player1_select), (gpointer) SELECT_PLAYER_VELENA_WEAK);
	gtk_signal_connect(GTK_OBJECT(radiobuttonPlayer1_3), "toggled", GTK_SIGNAL_FUNC(cb_prefs_player1_select), (gpointer) SELECT_PLAYER_VELENA_MEDIUM);
	gtk_signal_connect(GTK_OBJECT(radiobuttonPlayer1_4), "toggled", GTK_SIGNAL_FUNC(cb_prefs_player1_select), (gpointer) SELECT_PLAYER_VELENA_STRONG);

	gtk_signal_connect(GTK_OBJECT(radiobuttonPlayer2_0), "toggled", GTK_SIGNAL_FUNC(cb_prefs_player2_select), (gpointer) SELECT_PLAYER_HUMAN);
	gtk_signal_connect(GTK_OBJECT(radiobuttonPlayer2_1), "toggled", GTK_SIGNAL_FUNC(cb_prefs_player2_select), (gpointer) SELECT_PLAYER_GNECT);
	gtk_signal_connect(GTK_OBJECT(radiobuttonPlayer2_2), "toggled", GTK_SIGNAL_FUNC(cb_prefs_player2_select), (gpointer) SELECT_PLAYER_VELENA_WEAK);
	gtk_signal_connect(GTK_OBJECT(radiobuttonPlayer2_3), "toggled", GTK_SIGNAL_FUNC(cb_prefs_player2_select), (gpointer) SELECT_PLAYER_VELENA_MEDIUM);
	gtk_signal_connect(GTK_OBJECT(radiobuttonPlayer2_4), "toggled", GTK_SIGNAL_FUNC(cb_prefs_player2_select), (gpointer) SELECT_PLAYER_VELENA_STRONG);

	gtk_signal_connect(GTK_OBJECT(checkbuttonAnimate), "toggled", GTK_SIGNAL_FUNC(cb_prefs_animate_select), NULL);
	gtk_signal_connect(GTK_OBJECT(checkbuttonWipe), "toggled", GTK_SIGNAL_FUNC(cb_prefs_wipe_select), NULL);

	gtk_signal_connect(GTK_OBJECT(entryKeyboardLeft), "key_press_event", GTK_SIGNAL_FUNC(cb_prefs_key_select), NULL);
	gtk_signal_connect(GTK_OBJECT(entryKeyboardRight), "key_press_event", GTK_SIGNAL_FUNC(cb_prefs_key_select), NULL);
	gtk_signal_connect(GTK_OBJECT(entryKeyboardDrop), "key_press_event", GTK_SIGNAL_FUNC(cb_prefs_key_select), NULL);


	/* open it */
	gtk_widget_show_all(gnect.gui.prefsDialog);

}
