/* -*-mode:c; c-style:k&r; c-basic-offset:4; -*- */


/* main.c - gnect - a "Four In A Row" game for the GNOME.
 *
 * (c) 2000, 2001 Tim Musson <trmusson@ihug.co.nz>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */



#include <gnome.h>
#include "main.h"
#include "gnect.h"



struct Gnect gnect;

GnomeClient *client         = NULL;
gint        doVelengLogfile = FALSE;
gint        sessionX        = -1;
gint        sessionY        = -1;
unsigned    randSeed        = -1;


static const struct   poptOption options[] = {
#ifdef GNECT_DEBUGGING
	{"debugging", '\0', POPT_ARG_NONE, &gnect.debugging, 0, N_("Debugging mode on"), NULL},
#endif
	{"seed",      '\0', POPT_ARG_INT,  &randSeed,        0, N_("Random number seed"), NULL},
	{NULL,        'x',  POPT_ARG_INT,  &sessionX,        0, N_("Window X position"), NULL},
	{NULL,        'y',  POPT_ARG_INT,  &sessionY,        0, N_("Window Y position"), NULL},
	{NULL,        '\0', 0, NULL, 0}
};




static void gnect_init(void)
{
#ifdef GNECT_DEBUGGING
	gnect.debugging   = 0;
#endif
	gnect.velengBoard = NULL;
	gnect.themes      = NULL;

	gnect.gui.app              = NULL;
	gnect.gui.toolbarGDI       = NULL;
	gnect.gui.contentbox       = NULL;
	gnect.gui.drawArea         = NULL;
	gnect.gui.aboutDialog      = NULL;
	gnect.gui.prefsDialog      = NULL;
	gnect.gui.scoreDialog      = NULL;

	gnect.gfx.bufferPixmap     = NULL;
	gnect.gfx.tilesPixmap      = NULL;
	gnect.gfx.tilesMask        = NULL;
	gnect.gfx.backgroundPixmap = NULL;
	gnect.gfx.playerPixmap1    = NULL;
	gnect.gfx.playerPixmap2    = NULL;

	gnect_reset_scores(GNECT_REFRESH_DISPLAY_FALSE);
	gnect_reset_board(GNECT_REFRESH_DISPLAY_FALSE);
}



static void gnect_setup_first_game(void)
{
	/* Load initial theme, etc */


	gnect.prefs.theme = theme_pointer_from_filename(gnect.prefs.fnameTheme);
	if (!gnect.prefs.theme) {
		/* if user-specified theme isn't available, pick one that is */
		g_warning("gnect: user-specified theme is not available (%s)", gnect.prefs.fnameTheme);
		gnect.prefs.theme = gnect.themes;
		g_free(gnect.prefs.fnameTheme);
		gnect.prefs.fnameTheme = g_strdup(gnect.themes->fnameTheme);
	}
	if (!gfx_load_pixmaps(gnect.prefs.theme)) {
		/* theme_init checked that the images existed, so this error is unlikely */
		gnect_error(_("Unable to load images!"));
	}

	/* grab player descriptions from gnect.prefs.theme->descrPlayer1 and 2 */
	prefs_set_player_descriptions();

	/* ready to start */
	gnect.whoStarts = gnect_random_number(2) - 1;
	gnect_reset_game(GNECT_RESET_FIRST_GAME);
}



static void gnect_die(GnomeClient *client, gpointer client_data)
{
	/* Session exit */

	gnect_cleanup();
	gtk_exit(0);
}



static gint gnect_session_save(GnomeClient *client, gint phase,
							   GnomeRestartStyle save_style, gint shutdown,
							   GnomeInteractStyle interact_style, gint fast,
							   gpointer client_data)
{
	/* Basic session management, just save window position */

	gchar *prefix = gnome_client_get_config_prefix(client);
	gchar *argv[] = { "rm", "-r", NULL };
	gint xpos, ypos;


	gnome_config_push_prefix(prefix);

	gdk_window_get_origin(gnect.gui.app->window, &xpos, &ypos);

	gnome_config_set_int("Game/WindowX", xpos);
	gnome_config_set_int("Game/WindowY", ypos);

	argv[2] = gnome_config_get_real_path(prefix);
	gnome_client_set_discard_command(client, 3, argv);

	gnome_config_pop_prefix();
	gnome_config_sync();

	return(TRUE);
}



static void gnect_session_restart(void)
{
	/* Grab session-managed window position */

	gchar    *prefix = gnome_client_get_config_prefix(client);


	gnome_config_push_prefix(prefix);
	
	if (sessionX == -1) {
		sessionX = gnome_config_get_int("Game/WindowX=-1");
	}
	if (sessionY == -1) {
		sessionY = gnome_config_get_int("Game/WindowY=-1");
	}

	gnome_config_pop_prefix();

}



static void gnect_session_init(char *argv[])
{
	client = gnome_master_client();
	gtk_object_ref(GTK_OBJECT(client));
	gtk_object_sink(GTK_OBJECT(client));
	gtk_signal_connect(GTK_OBJECT(client), "save_yourself", GTK_SIGNAL_FUNC(gnect_session_save), argv[0]);
	gtk_signal_connect(GTK_OBJECT(client), "die", GTK_SIGNAL_FUNC(gnect_die), argv[0]);
	if (GNOME_CLIENT_CONNECTED(client)) {
	    gnect_session_restart();
	}
}



gint main(gint argc, gchar *argv[])
{

#ifdef ENABLE_NLS
	bindtextdomain(PACKAGE, GNOMELOCALEDIR);
	textdomain(PACKAGE);
#endif

	gnect_init();

	if (gnome_init_with_popt_table(PACKAGE, VERSION, argc, argv, options, 0, NULL)) {
		gnect_error("gnome_init_with_popt_table() failed");
	}

	gnect_session_init(argv);
	gnect_randomize();
	gnect.velengBoard = veleng_init();
	prefs_init();
	theme_init();
	if (!gnect.themes) {
		gnect_error(_("No themes available!"));
	}
	gui_create_app_window();
	gnect_setup_first_game();
	gui_open();

	gtk_main();


#ifdef GNECT_DEBUGGING
	if (gnect.debugging) {g_print("gnect: exit\n");}
#endif
	exit(0);
}
