/* -*-mode:c; c-style:k&r; c-basic-offset:4; -*- */


/* gui.c - gnect - a "Four In A Row" game for the GNOME.
 *
 * (c) 2000, 2001 Tim Musson <trmusson@ihug.co.nz>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */



#include <gnome.h>
#include "main.h"
#include "gnect.h"
#include "dialog.h"
#include "sound.h"



extern struct Gnect gnect;

extern gint winningRow;
extern gint winningCol;
extern gint sessionX;
extern gint sessionY;


static gint      cb_gui_quit(const GtkWidget *widget, const GdkEvent *event, const gpointer data);
static void      cb_gui_quit_test(const GtkWidget *widget, const gpointer *data);
static void      cb_gui_quit_verify(const GtkWidget *widget, const gpointer *data);
static void      cb_gui_new_game_test(const GtkWidget *widget, const gpointer *data);
static void      cb_gui_new_game(const GtkWidget *widget, const gpointer *data);
static void      cb_gui_get_hint(const GtkWidget *widget, const gpointer *data);
static void      cb_gui_undo_move(const GtkWidget *widget, const gpointer *data);
static void      cb_gui_toggle_toolbar(const GtkWidget *widget, const gpointer *data);
static void      cb_gui_toggle_sound(const GtkWidget *widget, const gpointer *data);
static void      cb_gui_toggle_grid(const GtkWidget *widget, const gpointer *data);
static void      cb_gui_board_event(GtkWidget *widget, const GdkEvent *event);
static void      cb_gui_key_press(const GtkWidget* widget, const GdkEventKey* event, const gpointer data);


gchar msgNewGame[] = N_(" \"New game\" to begin...");



/* ========== menus ========== */
GnomeUIInfo gameMenu[] = {
	GNOMEUIINFO_MENU_NEW_GAME_ITEM(cb_gui_new_game, NULL),
	GNOMEUIINFO_MENU_HINT_ITEM(cb_gui_get_hint, NULL),
	GNOMEUIINFO_MENU_UNDO_MOVE_ITEM(cb_gui_undo_move, NULL),
	GNOMEUIINFO_MENU_SCORES_ITEM(cb_dialog_score, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_EXIT_ITEM(cb_gui_quit_verify, NULL),
	GNOMEUIINFO_END
};
GnomeUIInfo settingsMenu[] = {
	GNOMEUIINFO_MENU_PREFERENCES_ITEM(cb_prefs, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_TOGGLEITEM_DATA(N_("Show toolbar"), N_("Show or hide the toolbar"), cb_gui_toggle_toolbar, NULL, NULL),
	GNOMEUIINFO_TOGGLEITEM_DATA(N_("Enable sound"), N_("Enable or disable sound"), cb_gui_toggle_sound, NULL, NULL),
	GNOMEUIINFO_TOGGLEITEM_DATA(N_("Draw grid"), N_("Enable or disable grid"), cb_gui_toggle_grid, NULL, NULL),
	GNOMEUIINFO_END
};
GnomeUIInfo helpMenu[] = {
	GNOMEUIINFO_HELP(PACKAGE),
	GNOMEUIINFO_MENU_ABOUT_ITEM(cb_dialog_about, NULL),
	GNOMEUIINFO_END
};
GnomeUIInfo menubar[] = {
	GNOMEUIINFO_MENU_GAME_TREE(gameMenu),
	GNOMEUIINFO_SUBTREE("_Settings", settingsMenu),
	GNOMEUIINFO_SUBTREE("_Help", helpMenu),
	GNOMEUIINFO_END
};



/* ========== toolbar ========== */
GnomeUIInfo toolbar[] = {
	GNOMEUIINFO_ITEM_STOCK(N_("New"), N_("Start a new game"), cb_gui_new_game, GNOME_STOCK_PIXMAP_NEW),
	GNOMEUIINFO_ITEM_STOCK(N_("Hint"), N_("Get a hint for your next move"), cb_gui_get_hint, GNOME_STOCK_PIXMAP_HELP),
	GNOMEUIINFO_ITEM_STOCK(N_("Undo"), N_("Undo the last move"), cb_gui_undo_move, GNOME_STOCK_PIXMAP_UNDO),
	GNOMEUIINFO_ITEM_STOCK(N_("Scores"), N_("View the scores"), cb_dialog_score, GNOME_STOCK_PIXMAP_SCORES),
	/*	GNOMEUIINFO_ITEM_STOCK(N_("Prefs"), N_("Configure the application"), cb_prefs, GNOME_STOCK_PIXMAP_PREFERENCES), */
	GNOMEUIINFO_ITEM_STOCK(N_("Exit"), N_("Exit the program"), cb_gui_quit_verify, GNOME_STOCK_PIXMAP_EXIT),
	GNOMEUIINFO_END
};




void gui_set_tooltip(GtkWidget *widget, const gchar *tipStr)
{
	/* Add tooltip tipStr to widget */


	GtkTooltips *t = gtk_tooltips_new();


	gtk_tooltips_set_tip(t, widget, tipStr, NULL);
}



void gui_status_message(const gchar *message, gint mode)
{
	/* Write message on status bar according to mode */


	if (mode == GNECT_STATUS_MSG_SET) {
		gnome_appbar_pop(GNOME_APPBAR(gnect.gui.appbar));
		gnome_appbar_push(GNOME_APPBAR(gnect.gui.appbar), message);
	}
	else if (mode == GNECT_STATUS_MSG_FLASH) {
		gnome_app_flash(GNOME_APP(gnect.gui.app), message);
	}
	else if (mode == GNECT_STATUS_MSG_CLEAR) {
		gnome_appbar_clear_stack(GNOME_APPBAR(gnect.gui.appbar));
		gnome_appbar_refresh(GNOME_APPBAR(gnect.gui.appbar));
	}
}



static void cb_gui_new_game_test(const GtkWidget *widget, const gpointer *data)
{
	/* Called by cb_gui_new_game */


	if ((gint *)data == 0) {
		/* gfx_wipe_board(); */
		/* a wipe here causes the verify box to stick around till the wipe is done */
		gnect_reset(GNECT_RESET_AND_START);
	}
}



static void cb_gui_new_game(const GtkWidget *widget, const gpointer *data)
{
	/* Reset and start a new game */


	GtkWidget *newverify = NULL;


	/* only verify if there's a game in progress */

	if (!gnect.over) {
		newverify = gnome_message_box_new(_("End the current game?"), GNOME_MESSAGE_BOX_QUESTION, GNOME_STOCK_BUTTON_YES, GNOME_STOCK_BUTTON_NO, NULL);
		gnome_dialog_set_parent(GNOME_DIALOG(newverify), GTK_WINDOW(gnect.gui.app));
		gnome_dialog_set_default(GNOME_DIALOG(newverify), 0);
		gtk_window_set_modal(GTK_WINDOW(newverify), TRUE);
		gtk_signal_connect(GTK_OBJECT(newverify), "clicked", cb_gui_new_game_test, NULL);
		gtk_widget_show(newverify);
	}
	else {
		gfx_wipe_board();
		gnect_reset(GNECT_RESET_AND_START);
	}
}



static void cb_gui_get_hint(const GtkWidget *widget, const gpointer *data)
{
	/* Flash a hint on the status bar */


	gchar ch = gnect.velengStr[0];
	gchar *hintstr = NULL;


	if (!gnect.over && !gnect_is_computers_move()) {
		/* get Velena Engine's strongest move */
		gnect.velengStr[0] = gnect_get_veleng_level(SELECT_PLAYER_VELENA_STRONG);
		hintstr = g_strdup_printf(_(" Hint: Column %d"), playgame(gnect.velengStr, gnect.velengBoard));
		gnect.velengStr[0] = ch;
	}
	else if (!gnect.over) {
		hintstr = g_strdup_printf(_(" Hey! Wait for your turn!"));
	}
	else {
		/* no game in progress */
		switch(gnect_random_number(16)) {

		case 1 :
			hintstr = g_strdup_printf(_(" Time for a coffee break."));
			break;

		case 2 :	
			hintstr = g_strdup_printf(_(" Beware of low flying butterflies."));
			break;

		case 3 :
			hintstr = g_strdup_printf(_(" Strike while the iron is hot."));
			break;

		case 4 :
			hintstr = g_strdup_printf(_(" There is a screw loose somewhere."));
			break;

		case 5 :
			hintstr = g_strdup_printf(_(" Reserve the master blow."));
			break;

		case 6 :
			hintstr = g_strdup_printf(_(" Don't order the sausage."));
			break;

		case 7 :
			hintstr = g_strdup_printf(_(" The world is unstable."));
			break;

		case 8 :
			hintstr = g_strdup_printf(_(" Reserve the master blow."));
			break;

		case 9 :
			hintstr = g_strdup_printf(_(" Shake container well prior to use."));
			break;

		case 10 :
			hintstr = g_strdup_printf(_(" Abstain from beans."));
			break;

		case 11 :
			hintstr = g_strdup_printf(_(" Dave, my mind is going. I can feel it."));
			break;

		case 12 :
			hintstr = g_strdup_printf(_(" The Astrodome. Good hot dogs."));
			break;

		case 13 :
			hintstr = g_strdup_printf(_(" Funny noises are not funny."));
			break;

		case 14 :
			hintstr = g_strdup_printf(_(" Goldfish don't bounce."));
			break;

		case 15 :
			hintstr = g_strdup_printf(_(" Harris's Lament: All the good ones are taken."));
			break;

		default:
			hintstr = g_strdup_printf(msgNewGame);
			break;

		}
	}

	if (!gnect.over) {
		gui_status_message(hintstr, GNECT_STATUS_MSG_SET);
	}
	else {
		gui_status_message(hintstr, GNECT_STATUS_MSG_FLASH);
	}
	g_free(hintstr);

}



static void cb_gui_undo_move(const GtkWidget *widget, const gpointer *data)
{
	/* Undo the last move */


	if (gnect_n_players()) {
		/* if there's at least one human playing... */
		if (!gnect_is_computers_move() || gnect.over) {
			/* and either it's a human's turn to move or the game is over... */
			gnect_undo_move(UNDO_IS_NORMAL);
		}
		else {
			gui_status_message(_(" Hey! It's not your turn!"), GNECT_STATUS_MSG_FLASH);
		}
	}
	else {
		gui_status_message(_(" No human moves to undo"), GNECT_STATUS_MSG_FLASH);
	}
}



static void cb_gui_toggle_toolbar(const GtkWidget *widget, const gpointer *data)
{
	/* Show or hide the toolbar - save state to prefs */


	gnect.prefs.toggleToolbar = GTK_CHECK_MENU_ITEM(widget)->active;

	if (gnect.prefs.toggleToolbar) {
		gtk_widget_show(GTK_WIDGET(gnect.gui.toolbarGDI));
	}
	else {
		gtk_widget_hide(GTK_WIDGET(gnect.gui.toolbarGDI));
		/* gtk_widget_queue_resize(gnect.gui.app); */
	}
	gnome_config_set_bool("/" PACKAGE "/Preferences/Toolbar", gnect.prefs.toggleToolbar);
	gnome_config_sync();
}



static void cb_gui_toggle_sound(const GtkWidget *widget, const gpointer *data)
{
	/* Enable or disable sound - save to prefs */


	gnect.prefs.toggleSound = GTK_CHECK_MENU_ITEM(widget)->active;

	gnome_config_set_bool("/" PACKAGE "/Preferences/Sound", gnect.prefs.toggleSound);
	gnome_config_sync();
}



static void cb_gui_toggle_grid(const GtkWidget *widget, const gpointer *data)
{
	/* Enable or disable grid */


	gnect.prefs.doGrids = GTK_CHECK_MENU_ITEM(widget)->active;

	gfx_load_pixmaps(gnect.prefs.theme);

	gnome_config_set_bool("/" PACKAGE "/Preferences/Grids", gnect.prefs.doGrids);
	gnome_config_sync();
}



static gint cb_gui_quit(const GtkWidget *widget, const GdkEvent *event, const gpointer data)
{
	/* Called if window is closed by window manager */


	gnect_cleanup();
	gtk_main_quit();
	return(FALSE);
}



static void cb_gui_quit_test(const GtkWidget *widget, const gpointer *data)
{
	/* Called by cb_gui_quit_verify */


	if ((gint *)data == 0) {
		gnect_cleanup();
		gtk_main_quit();
	}
}



static void cb_gui_quit_verify(const GtkWidget *widget, const gpointer *data)
{
	/* Called if user hits exit on toolbar or in menu */


	GtkWidget *quitverify = NULL;
	

	/* only verify if there's a game in progress */

	if (!gnect.over) {
		quitverify = gnome_message_box_new(_("End the current game?"), GNOME_MESSAGE_BOX_QUESTION, GNOME_STOCK_BUTTON_YES, GNOME_STOCK_BUTTON_NO, NULL);
		gnome_dialog_set_parent(GNOME_DIALOG(quitverify), GTK_WINDOW(gnect.gui.app));
		gnome_dialog_set_default(GNOME_DIALOG(quitverify), 0);
		gtk_window_set_modal(GTK_WINDOW(quitverify), TRUE);
		gtk_signal_connect(GTK_OBJECT(quitverify), "clicked", cb_gui_quit_test, NULL);
		gtk_widget_show(quitverify);
	}
	else {
		gnect_cleanup();
		gtk_main_quit();
	}
}



static void cb_gui_board_event(GtkWidget *widget, const GdkEvent *event)
{
	/* Handle expose and mouse events */


	GdkEventExpose *expose;
	gint x, y;


	switch (event->type) {

	case GDK_EXPOSE :
		expose = (GdkEventExpose *) event;
		gfx_redraw_area(&expose->area);
		break;
	
	case GDK_BUTTON_PRESS :
		if (!gnect.over) {
			gtk_widget_get_pointer(widget, &x, &y);
			gnect_process_move(gnect_mouse_x_to_col(x));
		}
		else {
			sound_event(SOUND_ID_CANT_MOVE);
			if (gnect.winner != -1 && gnect.winner != DRAWN_GAME) {
				gfx_flash_winning_lines(winningRow, winningCol, 1);
			}
			gui_status_message(msgNewGame, GNECT_STATUS_MSG_FLASH);
		}

	default:
		break;

	}

}



static void cb_gui_key_press(const GtkWidget* widget, const GdkEventKey* event, const gpointer data)
{
	/* Handle keyboard events */


	/* ignore if computer's playing */
	if (gnect_is_computers_move()) {
		return;
	}

	/* ignore if not an assigned key */
	if (event->keyval != gnect.prefs.keyLeft &&
		event->keyval != gnect.prefs.keyRight &&
		event->keyval != gnect.prefs.keyDrop) {
		return;
	}

	/* complain if no game in progress */
	if (gnect.over) {
		sound_event(SOUND_ID_CANT_MOVE);
		if (gnect.winner != -1 && gnect.winner != DRAWN_GAME) {
			gfx_flash_winning_lines(winningRow, winningCol, 1);
		}
		gui_status_message(msgNewGame, GNECT_STATUS_MSG_FLASH);
		return;
	}

	/* otherwise, try to act on the key-press */
	if (event->keyval == gnect.prefs.keyLeft && gnect.cursorCol > 0) {
		gfx_move_cursor(gnect.cursorCol - 1);
	}
	else if (event->keyval == gnect.prefs.keyRight && gnect.cursorCol < GAME_N_COLS - 1) {
		gfx_move_cursor(gnect.cursorCol + 1);
	}
	else if (event->keyval == gnect.prefs.keyDrop) {
		gnect_process_move(gnect.cursorCol);
	}
}



void gui_create_app_window(void)
{
	/* Build the main window */


	GtkWidget *hbox = NULL;


	gnect.gui.app = gnome_app_new(PACKAGE, "Gnect");
	gtk_widget_realize(gnect.gui.app);
	gtk_window_set_policy(GTK_WINDOW(gnect.gui.app), FALSE, FALSE, TRUE);
	gtk_signal_connect(GTK_OBJECT(gnect.gui.app), "delete_event", GTK_SIGNAL_FUNC(cb_gui_quit), NULL);

	gnect.gui.appbar = gnome_appbar_new(FALSE, TRUE, GNOME_PREFERENCES_USER);
	gnome_app_set_statusbar(GNOME_APP(gnect.gui.app), GTK_WIDGET(gnect.gui.appbar));

	gnome_app_create_menus(GNOME_APP(gnect.gui.app), menubar);
	gnome_app_install_menu_hints(GNOME_APP(gnect.gui.app), menubar);

	gnome_app_create_toolbar(GNOME_APP(gnect.gui.app), toolbar);
	gnect.gui.toolbarGDI = gnome_app_get_dock_item_by_name(GNOME_APP(gnect.gui.app), GNOME_APP_TOOLBAR_NAME);

	GTK_CHECK_MENU_ITEM(settingsMenu[GNECT_ID_MENU_ITEM_TOGGLE_TOOLBAR].widget)->active = gnect.prefs.toggleToolbar;
	GTK_CHECK_MENU_ITEM(settingsMenu[GNECT_ID_MENU_ITEM_TOGGLE_SOUND].widget)->active = gnect.prefs.toggleSound;
	GTK_CHECK_MENU_ITEM(settingsMenu[GNECT_ID_MENU_ITEM_TOGGLE_GRID].widget)->active = gnect.prefs.doGrids;

	gnect.gui.contentbox = gtk_vbox_new(FALSE,5);
	gnome_app_set_contents(GNOME_APP(gnect.gui.app), gnect.gui.contentbox);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_widget_show(hbox);
	gtk_box_pack_start(GTK_BOX(gnect.gui.contentbox), hbox, TRUE, FALSE, 0);

	gtk_widget_push_visual(gdk_imlib_get_visual());
	gtk_widget_push_colormap(gdk_imlib_get_colormap());

	gnect.gui.drawArea = gtk_drawing_area_new();

	gtk_widget_pop_colormap();
	gtk_widget_pop_visual();

	gtk_box_pack_start(GTK_BOX(hbox), gnect.gui.drawArea, TRUE, FALSE, 0);

	gtk_widget_set_events(gnect.gui.drawArea, GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK);

	gtk_widget_realize(gnect.gui.drawArea);

}



void gui_open(void)
{
	/* Open main window */


	gtk_signal_connect(GTK_OBJECT(gnect.gui.drawArea), "event", GTK_SIGNAL_FUNC(cb_gui_board_event), NULL);
	gtk_signal_connect(GTK_OBJECT(gnect.gui.app), "key_press_event", GTK_SIGNAL_FUNC(cb_gui_key_press), NULL);


	/* set window position */
    if (sessionX >= 0 && sessionY >= 0) {
		gtk_widget_set_uposition(gnect.gui.app, sessionX, sessionY);
    }

	gtk_widget_show_all(gnect.gui.app);

	/* hide toolbar if it's not wanted...
	   is possible to do this _before_ showing the window? */
	if (!gnect.prefs.toggleToolbar) {
		gtk_widget_hide(GTK_WIDGET(gnect.gui.toolbarGDI));
	}

	gfx_draw_all_tiles(GNECT_REFRESH_DISPLAY_FALSE);

	gui_status_message(_(" Welcome to Gnect!"), GNECT_STATUS_MSG_FLASH);
}
