/* -*-mode:c; c-style:k&r; c-basic-offset:4; -*- */


/* gnect.c - gnect - a "Four In A Row" game for the GNOME.
 *
 * (c) 2000, 2001 Tim Musson <trmusson@ihug.co.nz>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */



#include <gnome.h>
#include "main.h"
#include "gnect.h"
#include "sound.h"
#include "dialog.h"
#include "brain.h"



extern struct Gnect gnect;
extern unsigned randSeed;


gint lineX1;
gint lineY1;
gint lineX2;
gint lineY2;
gint winningRow = 0;
gint winningCol = 0;
static gint sourceid = 0;



static void      gnect_reset_velengstr(gint level);
static gint      gnect_computer_move(void);
static void      gnect_announce_winner(const gint row, const gint col);




gint gnect_check_computer_move(gpointer data)
{
	/* Function to hook computer move calculations into the idle loop (DN, 2001.01.18) */

	gnect_process_move(gnect_computer_move());
	return(FALSE);
}



void gnect_randomize(void)
{
	/* Seed random number generator */


	if (randSeed == -1) {
		randSeed = (unsigned)time(NULL);
		srand(randSeed);
	}
	else {
		srand((unsigned)randSeed);
	}
#ifdef GNECT_DEBUGGING
	if (gnect.debugging) {g_print("gnect: gnect_randomize: randSeed = %d\n", randSeed);}
#endif
}



gint gnect_random_number(gint top)
{
	/* Return a random integer in the range 1..top */


	return( rand()%top + 1 );
}



void gnect_cleanup(void)
{
	/* Free and close everything, ready to exit */


#ifdef GNECT_DEBUGGING
	if (gnect.debugging) {g_print("gnect: gnect_cleanup\n");}
#endif

	gfx_free_pixmaps();
	theme_free(gnect.themes);
	gnect.prefs.theme = NULL;
	prefs_free();
	veleng_free(gnect.velengBoard);
	if (gnect.gui.app) {
		gtk_widget_destroy(gnect.gui.app);
	}

#ifdef GNECT_DEBUGGING
	if (gnect.debugging) {g_print("gnect: gnect_cleanup: done\n");}
#endif
}



void gnect_error(const gchar *message)
{
	/* Clean up, write a message, abort the program */


	gnect_cleanup();

	if (message == NULL) {
		g_warning("gnect: Can't continue");
	}
	else {
		g_warning("gnect: %s", message);
	}
	gtk_exit(0);
}



gint gnect_mouse_x_to_col(gint x)
{
	/* Convert mouse x position to game board column */


	gint col;


	col = x / gnect.gfx.tileWidth;

	if (col > GAME_N_COLS - 1) {
		col--;
	}

	return(col);
}



gboolean gnect_is_full_column(gint col)
{
	/* Return non-zero if col is a full column */


	return(gnect.board[1][col] != TILE_CLEAR);
}



gboolean gnect_is_full_board(void)
{
	/* Return TRUE if all columns are full */


	gint col;


	for (col = 0; col < GAME_N_COLS; col++) {
		if (!gnect_is_full_column(col)) {
			return(FALSE);
		}
	}

	return(TRUE);
}



gint gnect_top_used_row(gint col)
{
	/* Find the top-most non-empty row in column col */


	gint row = 1;


	while(row < GAME_N_ROWS && gnect.board[row][col] == TILE_CLEAR) {
		row++;
	}

	return(row);
}



gboolean gnect_player_is_human(gint player)
{
	/* Return non-zero if this player is human */


	return(player == SELECT_PLAYER_HUMAN);
}



gboolean gnect_player_is_gnect(gint player)
{
	/* Return non-zero if this player is gnect's dummy brain */


	return(player == SELECT_PLAYER_GNECT);
}



gboolean gnect_player_is_veleng(gint player)
{
	/* Return non-zero if this player is the Velena Engine */


	return(player != SELECT_PLAYER_HUMAN &&
		   player != SELECT_PLAYER_GNECT);
}



gboolean gnect_is_computers_move(void)
{
	/* return non-zero if currentPlayer is non-human */


	return((gnect.currentPlayer == PLAYER_ONE && !gnect_player_is_human(gnect.prefs.player1)) ||
		   (gnect.currentPlayer == PLAYER_TWO && !gnect_player_is_human(gnect.prefs.player2)));
}



gint gnect_n_players(void)
{
	/* How many humans are playing? (0, 1 or 2) */


	if (gnect.prefs.player1 == SELECT_PLAYER_HUMAN &&
		gnect.prefs.player2 == SELECT_PLAYER_HUMAN) {
		return(2);
	}
	if (gnect.prefs.player1 != SELECT_PLAYER_HUMAN &&
		gnect.prefs.player2 != SELECT_PLAYER_HUMAN) {
		return(0);
	}
	return(1);
}



gchar gnect_get_veleng_level(gint player)
{
	/* Return a Velena Engine level specifier ('a', 'b' or 'c'), depending on player */


	switch(player) {
	case SELECT_PLAYER_VELENA_WEAK :
		return('a');
		break;
	case SELECT_PLAYER_VELENA_MEDIUM :
		return('b');
		break;
	default:
		return('c');
		break;
	}
}



void gnect_switch_players(void)
{
	/* If currentPlayer is PLAYER_ONE, switch to PLAYER_TWO (and vice versa) */


	if (gnect.currentPlayer == PLAYER_ONE) {
		gnect.currentPlayer = PLAYER_TWO;
	}
	else {
		gnect.currentPlayer = PLAYER_ONE;
	}

	/* redraw cursor */
	gfx_move_cursor(gnect.cursorCol);
}



gboolean gnect_undo_move(gboolean isWipe)
{
	/* Undo the last move or two (depending on whether it's a human vs computer
	 * game or a human vs human game). Return FALSE if there's nothing to undo.
	 * Extra messy because it's also used by the gfx_wipe_board function.
	 */


	gint col;
	gint undo = strlen(gnect.velengStr) - 2;


	/* if there's something on the board, other than a first move by a computer player... */
	if (undo && !(gnect_n_players() == 1 && undo == 1)) {

		/* if we're not part of a fancy board wipe... */
		if (!isWipe) {

			gui_status_message(NULL, GNECT_STATUS_MSG_CLEAR);

			/* if game in progress */
			if (!gnect.over) {
				gnect_switch_players();
			}
			/* else if the game was over */
			else {
				if (gnect.score[gnect.winner]) {
					gnect.score[gnect.winner]--;
				}
				gnect.over = FALSE; /* it's no longer over */
				gnect.winner = -1;  /* nobody has won */
				dialog_score_update();
			}

		}
		/* else we're just a part of a board wipe... */
		else {
			gnect_switch_players();
		}

		/* undo the last move, whoever made it... */

		col = gnect.velengStr[undo] - '1'; /* translate char '1' to int 0, '7' to 6, etc. */
		gnect.velengStr[undo + 1] = '0';
		gnect.velengStr[undo + 2] = '\0';
		undo--;
		gfx_move_cursor(col);
		gfx_suck_counter(col);


		/* if it's a human vs computer game and currentPlayer is
		   the computer, we still need to undo the human's move... */

		if (gnect_n_players() == 1 && gnect_is_computers_move()) {
			gnect_switch_players();
			col = gnect.velengStr[undo] - '1';
			gnect.velengStr[undo + 1] = '0';
			gnect.velengStr[undo + 2] = '\0';
			gfx_move_cursor(col);
			gfx_suck_counter(col);
		}
	}
	else {
		if (!isWipe) {
			if (undo) {
				/* there's a counter left, but it's not the user's */
				gui_status_message(_(" You have no moves to undo"), GNECT_STATUS_MSG_FLASH);
			}
			else {
				/* no counters on board */
				gui_status_message(_(" Nothing to undo"), GNECT_STATUS_MSG_FLASH);
			}
		}
		return(FALSE); /* nothing more can be undone */
	}

	return(TRUE); /* more moves can be undone */
}



void gnect_reset(gint resetMode)
{
	/* Reset game state and display for a new game */


#ifdef GNECT_DEBUGGING
	if (gnect.debugging) {g_print("gnect: gnect_reset\n");};
#endif

	/* No harm in removing the source, even if it no longer exists */
	if (sourceid != 0) {
	    g_source_remove(sourceid);
	}

	gnect_reset_game(resetMode);
	gnect_reset_display();

	if (resetMode == GNECT_RESET_AND_START) {
		gnect.cursorCol = CURSOR_START_COL;
		gfx_move_cursor(gnect.cursorCol);
		if (gnect_is_computers_move()) {
			gnect_process_move(gnect_computer_move());
		}
	}
}



void gnect_reset_game(gint resetMode)
{
	/* Reset game variables and select who starts */


#ifdef GNECT_DEBUGGING
	if (gnect.debugging) {g_print("gnect: gnect_reset_game\n");};
#endif

	gnect.winner = -1;
	gnect.over = resetMode == GNECT_RESET_AND_WAIT || resetMode == GNECT_RESET_FIRST_GAME;


	gnect_reset_velengstr(SELECT_PLAYER_VELENA_STRONG);
	gnect_reset_board(GNECT_REFRESH_DISPLAY_FALSE);

	switch(gnect.prefs.startMode) {

	case SELECT_STARTMODE_PLAYER_ONE :
		gnect.currentPlayer = PLAYER_ONE;
		break;

	case SELECT_STARTMODE_PLAYER_TWO :
		gnect.currentPlayer = PLAYER_TWO;
		break;

	case SELECT_STARTMODE_RANDOM :
		if (gnect_random_number(2) == 1) {
			gnect.currentPlayer = PLAYER_ONE;
		}
		else {
			gnect.currentPlayer = PLAYER_TWO;
		}
		break;

	case SELECT_STARTMODE_ALTERNATE :
		if (gnect.whoStarts == PLAYER_ONE) {
			gnect.currentPlayer = PLAYER_TWO;
		}
		else {
			gnect.currentPlayer = PLAYER_ONE;
		}
		break;

	default:
		/* impossible to get here, actually (see prefs_check) */
		gnect.currentPlayer = PLAYER_ONE;
		gnect.prefs.startMode = SELECT_STARTMODE_ALTERNATE;
		break;

	}
	gnect.whoStarts = gnect.currentPlayer;

}



void gnect_reset_display(void)
{
	/* Draw current board state and clear status bar */


#ifdef GNECT_DEBUGGING
	if (gnect.debugging) {g_print("gnect: gnect_reset_display\n");};
#endif

	gui_status_message(NULL, GNECT_STATUS_MSG_CLEAR);
	gfx_draw_all_tiles(GNECT_REFRESH_DISPLAY_FALSE);
}



static void gnect_reset_velengstr(gint level)
{
	/* Clear the string sent to the Velena Engine */


	gint i;


	for (i = 0; i < MAX_LEN_VELENGSTR; i++) {
		gnect.velengStr[i] = '\0';
	}

	/* set level specifier according to requested level */
	gnect.velengStr[0] = gnect_get_veleng_level(level);
	gnect.velengStr[1] = '0';
}



void gnect_reset_board(gboolean refreshDisplay)
{
	/* Clear board state and display - optional gdk_flush */


	gint row, col;


#ifdef GNECT_DEBUGGING
	if (gnect.debugging) {g_print("gnect: gnect_reset_board\n");};
#endif

	for (row = 0; row < GAME_N_ROWS; row++) {
		for (col = 0; col < GAME_N_COLS; col++) {
			gnect.board[row][col] = TILE_CLEAR;
			if (refreshDisplay) {
				gfx_draw_tile(row, col, TILE_CLEAR, GNECT_REFRESH_DISPLAY_FALSE);
			}
		}
	}
	if (refreshDisplay) {
		gdk_flush();
	}
}



void gnect_reset_scores(gboolean refreshDisplay)
{
	/* Set scores to zero - optionally update score dialogue */


	gnect.score[PLAYER_ONE] = 0;
	gnect.score[PLAYER_TWO] = 0;
	gnect.score[DRAWN_GAME] = 0;

	if (refreshDisplay) {
		dialog_score_update();
	}
}



void gnect_process_move(gint col)
{
	/* currentPlayer drops a counter in column col - deal with it */


	gint row;
	gint lenVelengStr;


	gui_status_message(NULL, GNECT_STATUS_MSG_CLEAR);

	gfx_move_cursor(col);

	/* if the column's not full... */
	if (!gnect_is_full_column(col)) {

		/* add this move to the Velena Engine string */
		lenVelengStr = strlen(gnect.velengStr);
		gnect.velengStr[lenVelengStr - 1] = '1' + col;
		gnect.velengStr[lenVelengStr] = '0';
#ifdef GNECT_DEBUGGING
		if (gnect.debugging) {printf("gnect: gnect.velengStr: %s\n", gnect.velengStr);}
#endif

		/* drop counter */
		row = gfx_drop_counter(col);

		/* check for a win */
		if (gnect_makes_line(gnect.currentPlayer, row, col, GAME_LINE_LENGTH)) {
			gnect.over = TRUE;
			gnect.score[gnect.currentPlayer]++;
			dialog_score_update();
			gnect_announce_winner(row, col);
		}
		else {

			/* if not a win, check for a draw */
			if (gnect_is_full_board()) {
				gnect.over = TRUE;
				gnect.score[DRAWN_GAME]++;
				gui_status_message(_(" It's a draw!"), GNECT_STATUS_MSG_SET);
				gnect.winner = DRAWN_GAME;
				dialog_score_update();
				sound_event(SOUND_ID_DRAWN_GAME);
			}
			else {

				/* if nothing interesting happened, it's the next player's turn */
				gnect_switch_players();

				/* Add computer move to idle loop. DN*/
				if (!gnect.over && gnect_is_computers_move()) {
				    if (sourceid !=0) {
						g_source_remove(sourceid);
				    }
				    sourceid = g_idle_add(gnect_check_computer_move, NULL);
				}
			}
		}
	}
	else {

		/* full column, complain */
		sound_event(SOUND_ID_CANT_MOVE);
		gui_status_message(_(" Sorry, full column"), GNECT_STATUS_MSG_FLASH);

	}

}



gboolean gnect_makes_line(gint counter, gint row, gint col, gint len)
{
	/* Return non-zero if there's a line of counter based on row, col */


	return(gnect_makes_line_horizontal(counter, row, len) ||
		   gnect_makes_line_vertical(counter, col, len) ||
		   gnect_makes_line_diagonal1(counter, row, col, len) ||
		   gnect_makes_line_diagonal2(counter, row, col, len) );

}



gboolean gnect_makes_line_horizontal(gint counter, gint row, gint len)
{
	/* Return TRUE if there's a horizontal line of counter in row */


	gint     col = 0;
	gint     lineLen = 0;
	gboolean marked = FALSE;
	gboolean gotLine = FALSE;


	lineY1 = row;
	lineY2 = row;

	while(col < GAME_N_COLS) {

		if (gnect.board[row][col] == counter) {
			if (!marked){
				marked = TRUE;
				lineX1 = col;
			}
			lineLen++;
			if (lineLen == len /* GAME_LINE_LENGTH */){
				gotLine = TRUE;
				lineX2 = col;
				/* return(TRUE); */
			}
		}
		else {
			if (gotLine) {
				lineX2 = col-1;
				return(gotLine);
			}
			marked = FALSE;
			lineLen = 0;
		}
		col++;
	}

	lineX2 = col - 1;

	return(gotLine);
}



gboolean gnect_makes_line_vertical(gint counter, gint col, gint len)
{
	/* Return TRUE if there's a vertical line of counter in column col */


	gint row = 1;
	gint lineLen = 0;
	gboolean marked = FALSE;


	lineX1 = col;
	lineX2 = col;

	while(row < GAME_N_ROWS) {

		if (gnect.board[row][col] == counter) {
			if (!marked) {
				marked = TRUE;
				lineY1 = row;
			}
			lineLen++;
			if (lineLen == len /* GAME_LINE_LENGTH */) {
				lineY2 = row;
				return(TRUE);
			}
		}
		else {
			marked = FALSE;
			lineLen = 0;
		}
		row++;
	}

	return(FALSE);
}



gboolean gnect_makes_line_diagonal1(gint counter, gint row, gint col, gint len)
{
	/* Return TRUE if there's a diagonal line of counter based on row, col */


	gint r = row;
	gint c = col;
	gint lineLen = 0;
	gboolean marked = FALSE;
	gboolean gotLine = FALSE;


	/* check for lines diagonal (\) */

	while(r > 1 && c){
		r--;
		c--;
	}

	while(c < GAME_N_COLS && r < GAME_N_ROWS) {

		if (gnect.board[r][c] == counter) {
			if (!marked) {
				marked = TRUE;
				lineX1 = c;
				lineY1 = r;
			}
			lineLen++;
			if (lineLen == len /* GAME_LINE_LENGTH */) {
				gotLine = TRUE;
				lineX2 = c;
				lineY2 = r;

				/* return(TRUE); */
			}
		}
		else {
			if (gotLine) {
				lineX2 = c - 1;
				lineY2 = r - 1;
				return(gotLine);
			}
			marked = FALSE;
			lineLen = 0;
		}
		r++;
		c++;
	}

	if (gotLine) {
		lineX2 = c - 1;
		lineY2 = r - 1;
		return(gotLine);
	}
  
	return(gotLine);
}



gboolean gnect_makes_line_diagonal2(gint counter, gint row, gint col, gint len)
{
	/* Return TRUE if there's a diagonal line of counter based on row, col */


	gint r = row;
	gint c = col;
	gint lineLen = 0;
	gboolean marked = FALSE;
	gboolean gotLine = FALSE;


	/* check for lines diagonal (/) */

	while(r > 1 && c < GAME_N_COLS - 1) {
		r--;
		c++;
	}

	while(r < GAME_N_ROWS && c > -1) {

		if (gnect.board[r][c] == counter) {
			if (!marked) {
				marked = TRUE;
				lineX1 = c;
				lineY1 = r;
			}
			lineLen++;
			if (lineLen == len /* GAME_LINE_LENGTH */) {
				gotLine = TRUE;
				lineX2 = c;
				lineY2 = r;

				/* return(TRUE); */
			}
		}
		else {
			if (gotLine) {
				lineX2 = c + 1;
				lineY2 = r - 1;
				return(gotLine);
			}
			marked = FALSE;
			lineLen = 0;
		}
		r++;
		c--;
	}

	if (gotLine) {
		lineX2 = c + 1;
		lineY2 = r - 1;
		return(gotLine);
	}
  
	return(gotLine);
}



void gnect_tantrum(void)
{
	/* Can be called by the computer player when it gets frustrated
	 * - flashes a message on the status bar and makes a noise.
	 * Velena Engine does not use this - Gnect's dummy brain does.
	 */


	switch(gnect_random_number(5)) {

	case 1 :
		gui_status_message(_(" Hey!"), GNECT_STATUS_MSG_FLASH);
		break;
	case 2 :
		gui_status_message(_(" Ouch!"), GNECT_STATUS_MSG_FLASH);
		break;
	case 3 :
		gui_status_message(_(" *grumble*"), GNECT_STATUS_MSG_FLASH);
		break;
	case 4 :
		gui_status_message(_(" Now what can I do...?"), GNECT_STATUS_MSG_FLASH);
		break;
	default :
		gui_status_message(_(" Hmmm..."), GNECT_STATUS_MSG_FLASH);
		break;

	}

	sound_event(SOUND_ID_TANTRUM);

}



static gint gnect_computer_move(void)
{
	/* Assuming is_computers_move(), return a move from one of the computer players */


	gint player, result;


	if (gnect.currentPlayer == PLAYER_ONE) {
		player = gnect.prefs.player1;
	}
	else {
		player = gnect.prefs.player2;
	}


	if (gnect_player_is_gnect(player)) {

		/* Gnect */
		result = brain_get_computer_move();

	}
	else {

		/* Velena Engine */
		gnect.velengStr[0] = gnect_get_veleng_level(player);
		result = playgame(gnect.velengStr, gnect.velengBoard) - 1;

	}

	return(result);
}



static void gnect_announce_winner(const gint row, const gint col)
{
	/* currentPlayer has won - sing and dance */


	gchar *strOne = NULL;
	gchar *strTwo = NULL;


	if (gnect_n_players() == 1) {

		if (gnect_is_computers_move()) {
			gui_status_message(_(" I win!"), GNECT_STATUS_MSG_SET);
			sound_event(SOUND_ID_I_WIN);
		}
		else {
			gui_status_message(_(" You win!"), GNECT_STATUS_MSG_SET);
			sound_event(SOUND_ID_YOU_WIN);
		}

	}
	else {

		strOne = g_strdup_printf(_(" %s wins!"), gnect.prefs.descrPlayer1);
		strTwo = g_strdup_printf(_(" %s wins!"), gnect.prefs.descrPlayer2);

		if (gnect.currentPlayer == PLAYER_ONE) {
			gui_status_message(strOne, GNECT_STATUS_MSG_SET);
		}
		else {
			gui_status_message(strTwo, GNECT_STATUS_MSG_SET);
		}

		g_free(strTwo);
		g_free(strOne);

		sound_event(SOUND_ID_WIN);

	}

	gnect.winner = gnect.currentPlayer;

	winningRow = row;
	winningCol = col;

	gfx_flash_winning_lines(row, col, N_WINNING_LINE_FLASHES);
}
