/* -*-mode:c; c-style:k&r; c-basic-offset:4; -*- */


/* gfx.c - gnect - a "Four In A Row" game for the GNOME.
 *
 * (c) 2000, 2001 Tim Musson <trmusson@ihug.co.nz>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */



#include <gnome.h>
#include "main.h"
#include "gnect.h"
#include "file.h"



extern struct Gnect gnect;

extern gint lineX1;
extern gint lineY1;
extern gint lineX2;
extern gint lineY2;


static void       gfx_draw_grid(const gchar *gridRGB, GdkPixmap *pixmap);
static GdkPixmap *gfx_make_player_pixmap(GdkImlibImage *tilesetImage,
										 GdkImlibImage *backgroundImage,
										 gint player, const gchar *gridRGB);



void gfx_redraw_area(const GdkRectangle *area)
{
	/* Redraw all tiles touched by 'area' - eg., called after an expose event */


	gint x1, y1, x2, y2;
	gint row, col;


	/* convert pixel coords to row, col cells */
	x1 = area->x / gnect.gfx.tileWidth;
	y1 = area->y / gnect.gfx.tileHeight;
	x2 = (area->x + area->width) / gnect.gfx.tileWidth;
	y2 = (area->y + area->height) / gnect.gfx.tileHeight;

	if (x2 >= GAME_N_COLS) {
		x2 = GAME_N_COLS - 1;
	}
	if (y2 >= GAME_N_ROWS) {
		y2 = GAME_N_ROWS - 1;
	}

	/* draw affected tiles */
	for (row = y1; row <= y2; row++) {
		for (col = x1; col <= x2; col++) {
			gfx_draw_tile(row, col, gnect.board[row][col], GNECT_REFRESH_DISPLAY_FALSE);
		}
	}
	gdk_flush();

}



void gfx_draw_all_tiles(gboolean refreshDisplay)
{
	/* Draw every tile on the board - optional gdk_flush */


	gint row, col;


	for (row = 0; row < GAME_N_ROWS; row++) {
		for (col = 0; col < GAME_N_COLS; col++) {
			gfx_draw_tile(row, col, gnect.board[row][col], GNECT_REFRESH_DISPLAY_FALSE);
		}
	}
	if (refreshDisplay) {
		gdk_flush();
	}
}



void gfx_move_cursor(gint col)
{
	/* Move currentPlayer's cursor to col */


	if (!gnect.prefs.doAnimate) {

		/* non-animated */

		gnect.board[0][gnect.cursorCol] = TILE_CLEAR;
		gfx_draw_tile(0, gnect.cursorCol, TILE_CLEAR, GNECT_REFRESH_DISPLAY_FALSE);
		gnect.cursorCol = col;
		gnect.board[0][gnect.cursorCol] = gnect.currentPlayer;
		gfx_draw_tile(0, gnect.cursorCol, gnect.currentPlayer, GNECT_REFRESH_DISPLAY_TRUE);
		return;
	}

	/* animated */

	gfx_draw_tile(0, gnect.cursorCol, TILE_CLEAR, GNECT_REFRESH_DISPLAY_FALSE);
	gnect.board[0][gnect.cursorCol] = TILE_CLEAR;

	while(gnect.cursorCol < col) {
		gfx_draw_tile(0, gnect.cursorCol, TILE_CLEAR, GNECT_REFRESH_DISPLAY_FALSE);
		gnect.cursorCol++;
		gfx_draw_tile(0, gnect.cursorCol, gnect.currentPlayer, GNECT_REFRESH_DISPLAY_TRUE);
		usleep(gnect.prefs.animDelay * GNECT_FRAMERATE);
	}
	while(gnect.cursorCol > col) {
		gfx_draw_tile(0, gnect.cursorCol, TILE_CLEAR, GNECT_REFRESH_DISPLAY_FALSE);
		gnect.cursorCol--;
		gfx_draw_tile(0, gnect.cursorCol, gnect.currentPlayer, GNECT_REFRESH_DISPLAY_TRUE);
		usleep(gnect.prefs.animDelay * GNECT_FRAMERATE);
	}
	gfx_draw_tile(0, gnect.cursorCol, gnect.currentPlayer, GNECT_REFRESH_DISPLAY_TRUE);
	gnect.board[0][gnect.cursorCol] = gnect.currentPlayer;
}



gint gfx_drop_counter(gint col)
{
	/* Drop currentPlayer's counter into column col, returning the row it landed in */


	gint row = 1;


	if (!gnect.prefs.doAnimate) {

		/* non-animated */

		while(row < GAME_N_ROWS-1 && gnect.board[row + 1][col] == TILE_CLEAR) {
			row++;
		}
		gfx_draw_tile(row, col, gnect.currentPlayer, GNECT_REFRESH_DISPLAY_TRUE);
		gnect.board[row][col] = gnect.currentPlayer;
		return(row);
	}

	/* animated */

	gfx_draw_tile(row, col, gnect.currentPlayer, GNECT_REFRESH_DISPLAY_TRUE);
	while(row < GAME_N_ROWS - 1 && gnect.board[row + 1][col] == TILE_CLEAR) {
		usleep(gnect.prefs.animDelay * GNECT_FRAMERATE);
		gfx_draw_tile(row, col, TILE_CLEAR, GNECT_REFRESH_DISPLAY_FALSE);
		row++;
		gfx_draw_tile(row, col, gnect.currentPlayer, GNECT_REFRESH_DISPLAY_TRUE);
	}
	gnect.board[row][col] = gnect.currentPlayer;

	return(row);
}



void gfx_suck_counter(gint col)
{
	/* Remove topmost counter from column */


	gint row;
	gint counter;
	gint lenVelengStr = strlen(gnect.velengStr);


	/* keep the Velena Engine in sync */
	gnect.velengStr[lenVelengStr - 2] = '0';
	gnect.velengStr[lenVelengStr - 1] = '\0';
#ifdef GNECT_DEBUGGING
	if (gnect.debugging) {g_print("gnect: gnect.velengStr: %s\n", gnect.velengStr);}
#endif

	if (!gnect.prefs.doAnimate) {

		/* non-animated */

		row = gnect_top_used_row(col);
		gfx_draw_tile(row, col, TILE_CLEAR, GNECT_REFRESH_DISPLAY_TRUE);
		gnect.board[row][col] = TILE_CLEAR;
		return;
	}

	/* animated */

	row = gnect_top_used_row(col);
	counter = gnect.board[row][col];
	gnect.board[row][col] = TILE_CLEAR;

	while(row > 1) {
		gfx_draw_tile(row, col, TILE_CLEAR, GNECT_REFRESH_DISPLAY_FALSE);
		row--;
		gfx_draw_tile(row, col, counter, GNECT_REFRESH_DISPLAY_TRUE);
		usleep(gnect.prefs.animDelay * GNECT_FRAMERATE);
	}
	gfx_draw_tile(row, col, TILE_CLEAR, GNECT_REFRESH_DISPLAY_TRUE);
}



gboolean gfx_load_pixmaps(THEMELIST_PTR theme)
{
	/* Load theme's tile set. If specified, also load background image and draw gird */

	gchar         *filename = NULL, *userPixmapDir = NULL;
	GdkImlibImage *imageTileset = NULL;
	GdkImlibImage *imageBackground = NULL;
	GdkVisual     *visual = NULL;


	/* load tile set */
	if (theme->isUserTheme) {
		userPixmapDir = filename_expand_tilde(g_strdup(USER_PIXMAP_DIR));
		filename = g_strdup_printf("%s%s%s", userPixmapDir, PATH_SEP_STR, theme->fnameTileset);
		if (!g_file_test(filename, G_FILE_TEST_ISFILE)) {
			/* not in user dir - try Gnect-installed pixmap dir */
			g_free(filename);
			filename = filename_gnect_pixmap(theme->fnameTileset, NULL);
		}
	}
	else {
		filename = filename_gnect_pixmap(theme->fnameTileset, NULL);
	}
	if (!(imageTileset = gdk_imlib_load_image(filename))) {
		g_warning("gnect: gfx_load_pixmaps: gdk_imlib_load_image failed (%s)", filename);
		g_free(filename);
		g_free(userPixmapDir);
		return(FALSE);
	}

	visual = gdk_imlib_get_visual();
	if (!visual) {
		g_warning("gnect: gfx_load_pixmaps: gdk_imlib_get_visual failed");
		gdk_imlib_destroy_image(imageTileset);
		g_free(filename);
		g_free(userPixmapDir);
		return(FALSE);
	}
	if(visual->type != GDK_VISUAL_TRUE_COLOR) {
		gdk_imlib_set_render_type(RT_PLAIN_PALETTE);
	}


	/* new tile set seems to be fine, so ditch any previously loaded graphics */
	gfx_free_pixmaps();

	g_free(filename);


	gdk_imlib_render(imageTileset, imageTileset->rgb_width, imageTileset->rgb_height);

	gnect.gfx.tilesPixmap = gdk_imlib_move_image(imageTileset);
	gnect.gfx.tilesMask   = gdk_imlib_move_mask(imageTileset);
	gnect.gfx.tileWidth   = imageTileset->rgb_width / N_TILES_IN_SET;
	gnect.gfx.tileHeight  = imageTileset->rgb_height;

	/* get a pixel offset for each tile in the set */
	gnect.gfx.tileOffset[TILE_PLAYER_ONE]        = 0;
	gnect.gfx.tileOffset[TILE_PLAYER_TWO]        = gnect.gfx.tileWidth;
	gnect.gfx.tileOffset[TILE_CLEAR]             = gnect.gfx.tileWidth * 2;
	gnect.gfx.tileOffset[TILE_CLEAR_CURSOR]      = gnect.gfx.tileWidth * 3;
	gnect.gfx.tileOffset[TILE_PLAYER_ONE_CURSOR] = gnect.gfx.tileWidth * 4;
	gnect.gfx.tileOffset[TILE_PLAYER_TWO_CURSOR] = gnect.gfx.tileWidth * 5;


	/* load background if specified */
	if (theme->fnameBackground) {

		if (theme->isUserTheme) {
			filename = g_strdup_printf("%s%s%s", userPixmapDir, PATH_SEP_STR, theme->fnameBackground);
			if (!g_file_test(filename, G_FILE_TEST_ISFILE)) {
				/* not in user dir - try Gnect-installed pixmap dir */
				g_free(filename);
				filename = filename_gnect_pixmap(theme->fnameBackground, NULL);
			}
		}
		else {
			filename = filename_gnect_pixmap(theme->fnameBackground, NULL);
		}
		if (!(imageBackground = gdk_imlib_load_image(filename))) {
			g_warning("gnect: gfx_load_pixmaps: gdk_imlib_load_image failed (%s)", filename);
		}
		else {

			gdk_imlib_render(imageBackground, imageBackground->rgb_width, imageBackground->rgb_height);
			gnect.gfx.backgroundPixmap = gdk_imlib_copy_image(imageBackground);
			if (!gnect.gfx.backgroundPixmap) {
				g_warning("gnect: gfx_load_pixmaps: gdk_imlib_copy_image failed\n");
			}
			else {

				/* if theme specifies a grid colour, draw a grid */
				if (gnect.prefs.doGrids && theme->gridRGB) {
					gfx_draw_grid(theme->gridRGB, gnect.gfx.backgroundPixmap);
				}

				/* create two new pixmaps:
				 *  > a copy of the background image filled with player1 tiles
				 *  > a copy of the background image filled with player2 tiles
				 */
				gnect.gfx.playerPixmap1 = gfx_make_player_pixmap(imageTileset, imageBackground, TILE_PLAYER_ONE, theme->gridRGB);
				gnect.gfx.playerPixmap2 = gfx_make_player_pixmap(imageTileset, imageBackground, TILE_PLAYER_TWO, theme->gridRGB);

				/* check that it worked */
				if (!gnect.gfx.playerPixmap1 || !gnect.gfx.playerPixmap2) {
					g_warning("gnect: gfx_load_pixmaps: gfx_make_player_pixmap failed\n");
					if (gnect.gfx.playerPixmap1) {
						gdk_pixmap_unref(gnect.gfx.playerPixmap1);
						gnect.gfx.playerPixmap1 = NULL;
					}
					if (gnect.gfx.playerPixmap2) {
						gdk_pixmap_unref(gnect.gfx.playerPixmap2);
						gnect.gfx.playerPixmap2 = NULL;
					}
					if (gnect.gfx.backgroundPixmap) {
						gdk_pixmap_unref(gnect.gfx.backgroundPixmap);
						gnect.gfx.backgroundPixmap = NULL;
					}
					/* At this point we have a tile set but no background. Something
					 * weird happened. But a background isn't required, so keep going.
					 */
				}
			}

			gdk_imlib_destroy_image(imageBackground);

		}

		g_free(filename);

	}

	gdk_imlib_destroy_image(imageTileset);


	/* update display */
	gtk_widget_set_usize(GTK_WIDGET(gnect.gui.drawArea),
						 gnect.gfx.tileWidth * GAME_N_COLS,
						 gnect.gfx.tileHeight * GAME_N_ROWS);
	gtk_widget_draw(gnect.gui.drawArea, NULL);


	if (userPixmapDir) {
		g_free(userPixmapDir);
	}


	return(TRUE);
}



void gfx_free_pixmaps(void)
{
#ifdef GNECT_DEBUGGING
	if (gnect.debugging) {g_print("gnect: gfx_free_pixmaps\n");}
#endif

	if (gnect.gfx.bufferPixmap) {
		gdk_pixmap_unref(gnect.gfx.bufferPixmap);
		gnect.gfx.bufferPixmap = NULL;
	}
	if (gnect.gfx.tilesPixmap) {
		gdk_pixmap_unref(gnect.gfx.tilesPixmap);
		gnect.gfx.tilesPixmap = NULL;
	}
	if (gnect.gfx.tilesMask) {
		gdk_pixmap_unref(gnect.gfx.tilesMask);
		gnect.gfx.tilesMask = NULL;
	}

	if (gnect.gfx.playerPixmap1) {
		gdk_pixmap_unref(gnect.gfx.playerPixmap1);
		gnect.gfx.playerPixmap1 = NULL;
	}
	if (gnect.gfx.playerPixmap2) {
		gdk_pixmap_unref(gnect.gfx.playerPixmap2);
		gnect.gfx.playerPixmap2 = NULL;
	}
	if (gnect.gfx.backgroundPixmap) {
		gdk_pixmap_unref(gnect.gfx.backgroundPixmap);
		gnect.gfx.backgroundPixmap = NULL;
	}
}



static GdkPixmap *gfx_make_player_pixmap(GdkImlibImage *tilesetImage,
										 GdkImlibImage *backgroundImage,
										 gint player, const gchar *gridRGB)
{
	/* Return a pixmap containing the background image filled with player's counters */


	GdkImlibImage *playerImage = NULL;
	GdkPixmap     *playerPixmap = NULL;
	gint          x, y;


	/* copy background image for this player */
	playerPixmap = gdk_imlib_copy_image(backgroundImage);
	if (!playerPixmap) {
		return(NULL);
	}

	/* draw grid on background */
	if (gnect.prefs.doGrids && gridRGB) {
		gfx_draw_grid(gridRGB, playerPixmap);
	}

	/* What follows is slow. Too slow to be done on the fly, which
	 * is why we're making these extra player pixmaps in the first place.
	 * Other GNOME games I've checked do the same thing, so I guess this
	 * is just a case of "shit happens".
	 */

	/* draw player's tiles on background */

	for (x = 0; x < GAME_N_COLS; x++) {

		for (y = 0; y < GAME_N_ROWS; y++) {


			if (y == 0) { /* top row */
				playerImage = gdk_imlib_crop_and_clone_image(tilesetImage,
															 gnect.gfx.tileOffset[player + 4], 0,
															 gnect.gfx.tileWidth, gnect.gfx.tileHeight);
			}
			else { /* main board */
				playerImage = gdk_imlib_crop_and_clone_image(tilesetImage,
															 gnect.gfx.tileOffset[player], 0,
															 gnect.gfx.tileWidth, gnect.gfx.tileHeight);
			}
			if (!playerImage) {
				gdk_pixmap_unref(playerPixmap);
				return(NULL);
			}

			gdk_imlib_paste_image(playerImage, playerPixmap,
								  x * gnect.gfx.tileWidth, y * gnect.gfx.tileHeight,
								  gnect.gfx.tileWidth, gnect.gfx.tileHeight);

			gdk_imlib_destroy_image(playerImage);
			playerImage = NULL;

		}

	}

	return(playerPixmap);
}



static void gfx_draw_grid(const gchar *gridRGB, GdkPixmap *pixmap)
{
	/* Draw a grid on the pixmap, colour specified by gridRGB */


	GdkColor    colour;
	GdkGCValues gcValues;
	GdkGC       *gc;
	gint        x, y;


	gc = gnect.gui.drawArea->style->black_gc;
	gdk_gc_get_values(gc, &gcValues);

	if (gdk_color_parse(gridRGB, &colour)) {

		gdk_imlib_best_color_get(&colour);
		gdk_gc_set_foreground(gc, &colour);

#ifdef GRID_FULL_HEIGHT /* grid covers all 7 rows */
		for (x = gnect.gfx.tileWidth; x < gnect.gfx.tileWidth * GAME_N_COLS; x = x + gnect.gfx.tileWidth) {
			gdk_draw_line(pixmap, gc, x, 0, x, gnect.gfx.tileHeight * GAME_N_COLS);
		}
#else /* grid covers bottom 6 rows only */
		for (x = gnect.gfx.tileWidth; x < gnect.gfx.tileWidth * GAME_N_COLS; x = x + gnect.gfx.tileWidth) {
			gdk_draw_line(pixmap, gc, x, gnect.gfx.tileHeight, x, gnect.gfx.tileHeight * GAME_N_COLS);
		}
#endif
		for (y = gnect.gfx.tileHeight; y < gnect.gfx.tileHeight * GAME_N_ROWS; y = y + gnect.gfx.tileHeight) {
			gdk_draw_line(pixmap, gc, 0, y, gnect.gfx.tileWidth * GAME_N_ROWS, y);
		}

		/* restore original foreground colour */
		gdk_gc_set_foreground(gc, &gcValues.foreground);
	}
	else {
		g_warning("gnect: bad gridRGB value (%s)\n", gridRGB);
	}
}



void gfx_draw_tile(gint row, gint col, gint tileSelector, gboolean refreshDisplay)
{
	/* Draw tile on board at row/col - optional gdk_flush */


	gint x = col * gnect.gfx.tileWidth;
	gint y = row * gnect.gfx.tileHeight;


	/* so what sort of theme is this? */

	if (gnect.gfx.backgroundPixmap) {

		/* theme uses full-window background image */

		if (tileSelector == TILE_CLEAR) {

			gdk_draw_pixmap(gnect.gui.drawArea->window,
							gnect.gui.drawArea->style->black_gc,
							gnect.gfx.backgroundPixmap,
							x, y,
							x, y,
							gnect.gfx.tileWidth, gnect.gfx.tileHeight);

		}
		else {

			if (tileSelector == TILE_PLAYER_ONE) {
				gdk_draw_pixmap(gnect.gui.drawArea->window,
								gnect.gui.drawArea->style->black_gc,
								gnect.gfx.playerPixmap1,
								x, y,
								x, y,
								gnect.gfx.tileWidth, gnect.gfx.tileHeight);
			}
			else { /* TILE_PLAYER_TWO */
				gdk_draw_pixmap(gnect.gui.drawArea->window,
								gnect.gui.drawArea->style->black_gc,
								gnect.gfx.playerPixmap2,
								x, y,
								x, y,
								gnect.gfx.tileWidth, gnect.gfx.tileHeight);
			}
		}

	}
	else {

		/* theme uses a tile set with no full-window background image */

		gint offset = tileSelector;


		/* if row is the top row, first convert the
		   tile selector to a cursor row tile */

		if (row == 0) {
			if (tileSelector == TILE_PLAYER_ONE) {
				offset = TILE_PLAYER_ONE_CURSOR;
			}
			else if (tileSelector == TILE_PLAYER_TWO) {
				offset = TILE_PLAYER_TWO_CURSOR;
			}
			else {
				offset = TILE_CLEAR_CURSOR;
			}
		}
		gdk_draw_pixmap(gnect.gui.drawArea->window,
						gnect.gui.drawArea->style->black_gc,
						gnect.gfx.tilesPixmap,
						gnect.gfx.tileOffset[offset], 0,
						x, y,
						gnect.gfx.tileWidth, gnect.gfx.tileHeight);

	}


	if (refreshDisplay) {
		gdk_flush();
	}

}



static void gfx_flash_one_winning_line(gint nFlashes)
{
	/* Blink a line nFlashes times. The line is specified
	   by lineX1, lineY1, lineX2, lineY2 */


	gint row, col, i;


	if (lineX1 == lineX2) {

		/* vertical */

		for (i = 0; i < nFlashes; i++){
			for (row = lineY1; row < lineY2 + 1; row++) {
				gfx_draw_tile(row, lineX1, TILE_CLEAR, GNECT_REFRESH_DISPLAY_FALSE);
			}
			gdk_flush();
			usleep(5 * GNECT_FRAMERATE);
			for (row = lineY1; row < lineY2 + 1; row++) {
				gfx_draw_tile(row, lineX1, gnect.winner, GNECT_REFRESH_DISPLAY_FALSE);
			}
			gdk_flush();
			usleep(5 * GNECT_FRAMERATE);
		}

	}
	else if (lineY1 == lineY2) {

		/* horizontal */

		for (i = 0; i < nFlashes; i++){
			for (col = lineX1; col < lineX2 + 1; col++) {
				gfx_draw_tile(lineY1, col, TILE_CLEAR, GNECT_REFRESH_DISPLAY_FALSE);
			}
			gdk_flush();
			usleep(5 * GNECT_FRAMERATE);
			for (col = lineX1; col < lineX2 + 1; col++) {
				gfx_draw_tile(lineY1, col, gnect.winner, GNECT_REFRESH_DISPLAY_FALSE);
			}
			gdk_flush();
			usleep(5 * GNECT_FRAMERATE);
		}

	}
	else if (lineX1 < lineX2 && lineY1 < lineY2) {

		/* diagonal (\) top down */

		for (i = 0; i < nFlashes; i++) {
			row = lineY1;
			col = lineX1;
			while(row < lineY2 + 1) {
				gfx_draw_tile(row, col, TILE_CLEAR, GNECT_REFRESH_DISPLAY_FALSE);
				row++;
				col++;
			}
			gdk_flush();
			usleep(5 * GNECT_FRAMERATE);
			row = lineY1;
			col = lineX1;
			while(row < lineY2 + 1) {
				gfx_draw_tile(row, col, gnect.winner, GNECT_REFRESH_DISPLAY_FALSE);
				row++;
				col++;
			}
			gdk_flush();
			usleep(5 * GNECT_FRAMERATE);
		}

	}
	else {

		/* diagonal (/) top down */

		for (i = 0; i < nFlashes; i++) {
			row = lineY1;
			col = lineX1;
			while(row < lineY2 + 1) {
				gfx_draw_tile(row, col, TILE_CLEAR, GNECT_REFRESH_DISPLAY_FALSE);
				row++;
				col--;
			}
			gdk_flush();
			usleep(5 * GNECT_FRAMERATE);
			row = lineY1;
			col = lineX1;
			while(row < lineY2 + 1) {
				gfx_draw_tile(row, col, gnect.winner, GNECT_REFRESH_DISPLAY_FALSE);
				row++;
				col--;
			}
			gdk_flush();
			usleep(5 * GNECT_FRAMERATE);
		}

	}
}



void gfx_flash_winning_lines(gint row, gint col, gint nFlashes)
{
	/* Indicate all winning lines by blinking them on and off */


	if (!gnect.prefs.doAnimate || gnect.winner == -1) {
		return;
	}

	/* The line detection functions set lineX1, lineY1, lineX2 and lineY2.
	   These co-ords are used by flash_one_winning_line(), above */

	if (gnect_makes_line_horizontal(gnect.winner, row, GAME_LINE_LENGTH)) {
		gfx_flash_one_winning_line(nFlashes);
	}
	if (gnect_makes_line_diagonal1(gnect.winner, row, col, GAME_LINE_LENGTH)) {
		gfx_flash_one_winning_line(nFlashes);
	}
	if (gnect_makes_line_vertical(gnect.winner, col, GAME_LINE_LENGTH)) {
		gfx_flash_one_winning_line(nFlashes);
	}
	if (gnect_makes_line_diagonal2(gnect.winner, row, col, GAME_LINE_LENGTH)) {
		gfx_flash_one_winning_line(nFlashes);
	}

}



void gfx_wipe_board(void)
{
	/* Some pretty (er, needless) wipes to clear the board */


	gint wipeID;
	gint row;
	gint col;
	gint i;


	/* don't bother unless user wants to */
	if (!gnect.prefs.doAnimate || !gnect.prefs.doWipes) {
		return;
	}

	/* pick a wipe style at random, but don't use the "undo"
	 * wipe if the board's large or if it's a computer vs computer game
	 */
	if (gnect_n_players() && GAME_N_COLS <= 7 && GAME_N_ROWS <= 7) {
		wipeID = gnect_random_number(5);
	}
	else { /* the first 4 wipes are quick */
		wipeID = gnect_random_number(4);
	}

	switch(wipeID) {

	case 1 :

		/* M O V E   R O W S   D O W N */
		for (i = 0; i < GAME_N_ROWS; i++) {
			for (row = GAME_N_ROWS - 1; row > 0; row--) {
				for (col = 0; col < GAME_N_COLS; col++) {
					if (row > 1) {
						gnect.board[row][col] = gnect.board[row - 1][col];
					}
					else {
						gnect.board[row][col] = TILE_CLEAR;
					}
					gfx_draw_tile(row, col, gnect.board[row][col], GNECT_REFRESH_DISPLAY_FALSE);
				}
			}
			gdk_flush();
			usleep(gnect.prefs.animDelay * GNECT_FRAMERATE);
		}
		break;

	case 2 :

		/* M O V E   R O W S   U P */
		for (i = 0; i < GAME_N_ROWS; i++) {
			for (col = 0; col < GAME_N_COLS; col++) {
				for (row = 0; row < GAME_N_ROWS; row++) {
					if (row < GAME_N_ROWS - 1) {
						gnect.board[row][col] = gnect.board[row + 1][col];
					}
					else {
						gnect.board[row][col] = TILE_CLEAR;
					}
					gfx_draw_tile(row, col, gnect.board[row][col], GNECT_REFRESH_DISPLAY_FALSE);
				}
			}
			gdk_flush();
			usleep(gnect.prefs.animDelay * GNECT_FRAMERATE);
		}
		break;

	case 3 :

		/* M O V E   C O L U M N S   L E F T */
		for (i = 0; i < GAME_N_COLS; i++) {
			for (row = 0; row < GAME_N_ROWS; row++) {
				for (col = 0; col < GAME_N_COLS; col++) {
					if (col < GAME_N_COLS - 1) {
						gnect.board[row][col] = gnect.board[row][col + 1];
					}
					else {
						gnect.board[row][col] = TILE_CLEAR;
					}
					gfx_draw_tile(row, col, gnect.board[row][col], GNECT_REFRESH_DISPLAY_FALSE);
				}
			}
			gdk_flush();
			usleep(gnect.prefs.animDelay * GNECT_FRAMERATE);
		}
		break;

	case 4 :

		/* M O V E   C O L U M N S   R I G H T */
		for (i = 0; i < GAME_N_COLS; i++) {
			for (row = 0; row < GAME_N_ROWS; row++) {
				for (col = GAME_N_COLS - 1; col > -1; col--) {
					if (col > 0) {
						gnect.board[row][col] = gnect.board[row][col - 1];
					}
					else {
						gnect.board[row][col] = TILE_CLEAR;
					}
					gfx_draw_tile(row, col, gnect.board[row][col], GNECT_REFRESH_DISPLAY_FALSE);
				}
			}
			gdk_flush();
			usleep(gnect.prefs.animDelay * GNECT_FRAMERATE);
		}
		break;

	case 5 :

		/* U N P L A Y   G A M E */
		while(gnect_undo_move(UNDO_IS_WIPE));
		break;

	default :
		break;

	}

}
