/* -*-mode:c; c-style:k&r; c-basic-offset:4; -*- */


/* file.c - gnect - a "Four In A Row" game for the GNOME.
 *
 * (c) 2000, 2001 Tim Musson <trmusson@ihug.co.nz>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */



#include <gnome.h>
#include "main.h"
#include "file.h"



/* These functions are used instead of gnome_datadir_file, etc., so that Gnect's
 * install location can be a bit more flexible. Still, GNOME apps really prefer
 * a location that reflects the main GNOME installation location on your system.
 * Eg., if GNOME lives in /usr, Gnect will also be happiest in /usr ...but if
 * GNOME lives in /usr and Gnect lives in /usr/local, Gnect will work, but won't
 * have access to its help files or be added to the GNOME Games menu.
 */


gchar *filename_gnect_data(const gchar *fname, gboolean *verify)
{
	/* - Return full path + filename of fname
	 *   (eg. /usr/share/gnect/whatever.file)
	 * - If fname is NULL, just return the program's data
	 *   installation path (eg. /usr/share/gnect)
	 * - If verify, test that the path or file exists,
	 *   setting verify to TRUE or FALSE.
	 */

	gchar *result = NULL;


	if (fname) {
		result = g_strdup_printf("%s%sshare%s%s%s%s",
								 PACKAGE_INSTALL_PREFIX,
								 PATH_SEP_STR, PATH_SEP_STR,
								 PACKAGE,
								 PATH_SEP_STR,
								 fname);
		if (verify) {
			if (g_file_test(result, G_FILE_TEST_ISFILE)) {
				*verify = TRUE;
			}
			else {
				*verify = FALSE;
			}
		}
	}
	else {
		result = g_strdup_printf("%s%sshare%s%s",
								 PACKAGE_INSTALL_PREFIX,
								 PATH_SEP_STR, PATH_SEP_STR,
								 PACKAGE);
		if (verify) {
			if (g_file_test(result, G_FILE_TEST_ISDIR)) {
				*verify = TRUE;
			}
			else {
				*verify = FALSE;
			}
		}
	}
	return(result);
}



gchar *filename_gnect_pixmap(const gchar *fname, gboolean *verify)
{
	/* - Return full path + filename of fname
	 *   (eg. /usr/share/pixmaps/gnect/test.png)
	 * - If fname is NULL, just return the program's
	 *   pixmap installation path (eg. /usr/local/share/pixmaps/gnect)
	 * - If verify, test that the path or file exists,
	 *   setting verify to TRUE or FALSE.
	 */

	gchar *result = NULL;


	if (fname) {
		result = g_strdup_printf("%s%sshare%spixmaps%s%s%s%s",
								 PACKAGE_INSTALL_PREFIX,
								 PATH_SEP_STR, PATH_SEP_STR, PATH_SEP_STR,
								 PACKAGE,
								 PATH_SEP_STR,
								 fname);
		if (verify) {
			if (g_file_test(result, G_FILE_TEST_ISFILE)) {
				*verify = TRUE;
			}
			else {
				*verify = FALSE;
			}
		}
	}
	else {
		result = g_strdup_printf("%s%sshare%spixmaps%s%s",
								 PACKAGE_INSTALL_PREFIX,
								 PATH_SEP_STR, PATH_SEP_STR, PATH_SEP_STR,
								 PACKAGE);
		if (verify) {
			if (g_file_test(result, G_FILE_TEST_ISDIR)) {
				*verify = TRUE;
			}
			else {
				*verify = FALSE;
			}
		}
	}
	return(result);
}



gchar *filename_gnect_base_pixmap(const gchar *fname, gboolean *verify)
{
	/* - Return full path + filename of fname
	 *   (eg. /usr/share/pixmaps/gnect-icon.png)
	 * - If fname is NULL, just return the base pixmap path
	 *   (eg. /usr/share/pixmaps)
	 * - If verify, test that the path or file exists,
	 *   setting verify to TRUE or FALSE.
	 */

	gchar *result = NULL;


	if (fname) {
		result = g_strdup_printf("%s%sshare%spixmaps%s%s",
								 PACKAGE_INSTALL_PREFIX,
								 PATH_SEP_STR, PATH_SEP_STR, PATH_SEP_STR,
								 fname);
		if (verify) {
			if (g_file_test(result, G_FILE_TEST_ISFILE)) {
				*verify = TRUE;
			}
			else {
				*verify = FALSE;
			}
		}
	}
	else {
		result = g_strdup_printf("%s%sshare%spixmaps",
								 PACKAGE_INSTALL_PREFIX,
								 PATH_SEP_STR, PATH_SEP_STR);
		if (verify) {
			if (g_file_test(result, G_FILE_TEST_ISDIR)) {
				*verify = TRUE;
			}
			else {
				*verify = FALSE;
			}
		}
	}
	return(result);
}



gchar *filename_expand_tilde(gchar *dname)
{
	/* Expand tilde (~) in dname to user home name */


	gchar *result = NULL;


	if (dname && dname[0] == '~') {
		result = g_strdup_printf("%s%s", getenv("HOME"), &dname[1]);
		g_free(dname);
		return(result);
	}
	return(dname);
}
