/* -*-mode:c; c-style:k&r; c-basic-offset:4; -*- */


/* brain.c - gnect - a "Four In A Row" game for the GNOME.
 *
 * (c) 2000, 2001 Tim Musson <trmusson@ihug.co.nz>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */




/* This is Gnect's original brain. It served as a place holder
 * while I built the GUI - but I've kept it on as a very easy level.
 * It has four stages:
 *
 * 1) Look for the immediate consequence of dropping a counter in each column.
 * 2) Avoid easy horizontal traps.
 * 3) If there's nothing else to do, play connect three (as close as
 *    the dummy gets to trying to build its own lines and get in the
 *    way of its opponent).
 * 4) Avoid a few more easily detected traps.
 *
 * So, there's no planning going on - no looking ahead more than two moves
 * (more like one and a half ;) - and no serious strategy.
 *
 */


#include <gnome.h>
#include "main.h"
#include "gnect.h"
#include "brain.h"



#define BRAIN_CHOICE_WIN           12 /* greatest incentive to drop counter */
#define BRAIN_CHOICE_THWART_1      11
#define BRAIN_CHOICE_THWART_2      10
#define BRAIN_CHOICE_GOOD_1         9
#define BRAIN_CHOICE_GOOD_2         8
#define BRAIN_CHOICE_GOOD_3         7
#define BRAIN_CHOICE_GOOD_4         6
#define BRAIN_CHOICE_NO_VALUE       5
#define BRAIN_CHOICE_BAD_2          4
#define BRAIN_CHOICE_BAD_1          3
#define BRAIN_CHOICE_MY_TRAP        2
#define BRAIN_CHOICE_OPPONENTS_TRAP 1
#define BRAIN_CHOICE_FULL_COLUMN    0 /* least incentive to drop counter    */


extern struct Gnect gnect;


static void     brain_pass_1(void);
static void     brain_pass_2(void);
static void     brain_pass_3(void);
static void     brain_pass_4(void);
static gboolean brain_row_compare(gint row, gint col, const gchar *rowStr, const gchar *belowStr);
static gboolean brain_block_compare(gint row, gint col, const gchar *str1, const gchar *str2, const gchar *str3, const gchar *str4);
static gint     brain_test_drop(gint counter, gint col, gint len);
static gint     brain_has_multi_choice(gint val);
static gint     brain_col_is_trap(gint counter, gint col, gint len);
static gint     brain_best_choice(void);


static gint me, opponent, choice[GAME_N_COLS + 1];




gint brain_get_computer_move(void)
{
	/* Return the best move (column) Gnect can come up with */


	me = gnect.currentPlayer;
	if (me == PLAYER_ONE) {
		opponent = PLAYER_TWO;
	}
	else {
		opponent = PLAYER_ONE;
	}

	brain_pass_1();
	brain_pass_2();
	brain_pass_3();
	brain_pass_4();

	return(brain_best_choice());
}



static void brain_pass_1(void)
{
	/* Work out the immediate value of dropping the computer's
	   counter into each column. Store in the 'choice' array */


	gint col = 0;


	while(col < GAME_N_COLS) {

		if (gnect.prefs.doAnimate) {
			gfx_move_cursor(col);
		}

		if (gnect_is_full_column(col)) {
			/* Column full */
			choice[col] = BRAIN_CHOICE_FULL_COLUMN;
		}
		else if (brain_test_drop(me, col, GAME_LINE_LENGTH)) {
			/* I get an immediate win with this column */
			choice[col] = BRAIN_CHOICE_WIN;
		}
		else if (brain_test_drop(opponent, col, GAME_LINE_LENGTH)) {
			/* opponent wins with this column */
			choice[col] = BRAIN_CHOICE_THWART_1;
		}
		else if (brain_col_is_trap(me, col, GAME_LINE_LENGTH)) {
			/* if I drop counter here, opponent's next move could win */
			choice[col] = BRAIN_CHOICE_OPPONENTS_TRAP;
		}
		else if (brain_col_is_trap(opponent, col, GAME_LINE_LENGTH)) {
			/* if opponent drops counter here, I could win */
			choice[col] = BRAIN_CHOICE_MY_TRAP;
		}
		else {
			/* No immediate reason to drop a counter here */
			choice[col] = BRAIN_CHOICE_NO_VALUE;
		}
		col++;
	}
}



static void brain_pass_2(void)
{
	gint row, col;


	/* yes, well, I'm just a dummy too... */

	for (col = 0; col < GAME_N_COLS; col++) {
		if (choice[col] != BRAIN_CHOICE_FULL_COLUMN) {

			row = gnect_top_used_row(col) - 1;

			if (brain_row_compare(row, col, "*.11.", "xxxx.") ||
				brain_row_compare(row, col, "*.22.", "xxxx.")) {
				if (choice[col + 1] == BRAIN_CHOICE_NO_VALUE) {choice[col + 1] = BRAIN_CHOICE_GOOD_1;}
				if (choice[col + 4] == BRAIN_CHOICE_NO_VALUE) {choice[col + 4] = BRAIN_CHOICE_BAD_1;}
			}
			if (brain_row_compare(row, col, ".11.*", ".xxxx") ||
				brain_row_compare(row, col, ".22.*", ".xxxx")) {
				if (choice[col + 3] == BRAIN_CHOICE_NO_VALUE) {choice[col + 3] = BRAIN_CHOICE_GOOD_1;}
				if (choice[col] == BRAIN_CHOICE_NO_VALUE) {choice[col] = BRAIN_CHOICE_BAD_1;}
			}
			if (brain_row_compare(row, col, ".11..", "xxx.x") ||
				brain_row_compare(row, col, ".22..", "xxx.x")) {
				if (choice[col] == BRAIN_CHOICE_NO_VALUE) {choice[col] = BRAIN_CHOICE_GOOD_1;}
			}
			if (brain_row_compare(row, col, "..11.", "x.xxx") ||
				brain_row_compare(row, col, "..22.", "x.xxx")) {
				if (choice[col + 4] == BRAIN_CHOICE_NO_VALUE) {choice[col + 4] = BRAIN_CHOICE_GOOD_1;}
			}
			if (brain_row_compare(row, col, ".11..", "xxxxx") ||
				brain_row_compare(row, col, ".22..", "xxxxx")) {
				if (choice[col + 3] == BRAIN_CHOICE_NO_VALUE) {choice[col + 3] = BRAIN_CHOICE_GOOD_1;}
			}
			if (brain_row_compare(row, col, "..11.", "xxxxx") ||
				brain_row_compare(row, col, "..22.", "xxxxx")) {
				if (choice[col + 1] == BRAIN_CHOICE_NO_VALUE) {choice[col + 1] = BRAIN_CHOICE_GOOD_1;}
			}
			if (brain_row_compare(row, col, ".1.1.", ".xxxx") ||
				brain_row_compare(row, col, ".2.2.", ".xxxx")) {
				if (choice[col + 2] == BRAIN_CHOICE_NO_VALUE) {choice[col + 2] = BRAIN_CHOICE_GOOD_1;}
			}
			if (brain_row_compare(row, col, ".1.1.", "xxxx.") ||
				brain_row_compare(row, col, ".2.2.", "xxxx.")) {
				if (choice[col + 2] == BRAIN_CHOICE_NO_VALUE) {choice[col + 2] = BRAIN_CHOICE_GOOD_1;}
			}
			if (brain_row_compare(row, col, ".1.1.", "xxxxx") ||
				brain_row_compare(row, col, ".2.2.", "xxxxx")) {
				if (choice[col + 2] == BRAIN_CHOICE_NO_VALUE) {choice[col + 2] = BRAIN_CHOICE_GOOD_1;}
			}
			if (brain_row_compare(row, col, ".1.1.", "xx.xx") ||
				brain_row_compare(row, col, ".2.2.", "xx.xx")) {
				if (choice[col] == BRAIN_CHOICE_NO_VALUE) {choice[col] = BRAIN_CHOICE_GOOD_1;}
				if (choice[col + 4] == BRAIN_CHOICE_NO_VALUE) {choice[col + 4] = BRAIN_CHOICE_GOOD_1;}
			}
		}
	}
}



static void brain_pass_3(void)
{
	/* Where there's nothing worth doing, pretend it's Connect Three - not great, but hey. */


	gint col = 0;


	while(col < GAME_N_COLS) {

		if (choice[col] == BRAIN_CHOICE_NO_VALUE) {

			if (brain_test_drop(me, col, GAME_LINE_LENGTH - 1)) {
				choice[col] = BRAIN_CHOICE_GOOD_2;
			}
			if (brain_test_drop(opponent, col, GAME_LINE_LENGTH - 1)) {
				choice[col] = BRAIN_CHOICE_GOOD_2;
			}

			if (brain_col_is_trap(me, col, GAME_LINE_LENGTH - 1)) {
				choice[col] = BRAIN_CHOICE_BAD_2;
			}
			else if (brain_col_is_trap(opponent, col, GAME_LINE_LENGTH - 1)) {
				choice[col] = BRAIN_CHOICE_BAD_1;
			}

		}
		col++;

	}

}



static void brain_pass_4(void)
{

	gboolean test;
	gint col, row;


	for (row = 1; row < GAME_N_ROWS - 1; row++) {
		for (col = 0; col < GAME_N_COLS; col++) {

			if (brain_block_compare(row, col, "11..", "x11.", "1xxx", "xxxx") ||
				brain_block_compare(row, col, "22..", "x22.", "2xxx", "xxxx")) {
				if (choice[col+2] && choice[col+2] < BRAIN_CHOICE_GOOD_1) {choice[col+2] = BRAIN_CHOICE_GOOD_1;}
			}
			if (brain_block_compare(row, col, "..11", ".11x", "xxx1", "xxxx") ||
				brain_block_compare(row, col, "..22", ".22x", "xxx2", "xxxx")) {
				if (choice[col+1] && choice[col+1] < BRAIN_CHOICE_GOOD_1) {choice[col+1] = BRAIN_CHOICE_GOOD_1;}
			}
			if (brain_block_compare(row, col, ".11", "1xx", "1xx", "xxx") ||
				brain_block_compare(row, col, ".22", "2xx", "2xx", "xxx")) {
				if (choice[col] && choice[col] < BRAIN_CHOICE_GOOD_1) {choice[col] = BRAIN_CHOICE_GOOD_1;}
			}
			if (brain_block_compare(row, col, "11.", "xx1", "xx1", "xxx") ||
				brain_block_compare(row, col, "22.", "xx2", "xx2", "xxx")) {
				if (choice[col+2] && choice[col+2] < BRAIN_CHOICE_GOOD_1) {choice[col+2] = BRAIN_CHOICE_GOOD_1;}
			}
			if (brain_block_compare(row, col, ".1.", "1xx", "1xx", "xxx") ||
				brain_block_compare(row, col, ".2.", "2xx", "2xx", "xxx")) {
				if (choice[col+2] && choice[col+2] < BRAIN_CHOICE_GOOD_2) {choice[col+2] = BRAIN_CHOICE_GOOD_2;}
			}
			if (brain_block_compare(row, col, ".1.", "xx1", "xx1", "xxx") ||
				brain_block_compare(row, col, ".2.", "xx2", "xx2", "xxx")) {
				if (choice[col] && choice[col] < BRAIN_CHOICE_GOOD_2) {choice[col] = BRAIN_CHOICE_GOOD_2;}
			}
			if (brain_block_compare(row, col, "1.1", "x1x", "xxx", "xxx") ||
				brain_block_compare(row, col, "2.2", "x2x", "xxx", "xxx")) {
				if (choice[col+1] && choice[col+1] < BRAIN_CHOICE_THWART_2) {choice[col+1] = BRAIN_CHOICE_THWART_2;}
			}
			if (brain_block_compare(row, col, "1.1", "x1x", "xxx", "xxx") ||
				brain_block_compare(row, col, "2.2", "x2x", "xxx", "xxx")) {
				if (choice[col+1] && choice[col+1] < BRAIN_CHOICE_THWART_2) {choice[col+1] = BRAIN_CHOICE_THWART_2;}
			}
			if (brain_block_compare(row, col, ".*1", ".11", "xx1", "xxx") ||
				brain_block_compare(row, col, ".*2", ".22", "xx2", "xxx")) {
				if (choice[col] && choice[col] < BRAIN_CHOICE_GOOD_1) {choice[col] = BRAIN_CHOICE_GOOD_1;}
			}
			if (brain_block_compare(row, col, "1*.", "11.", "1xx", "xxx") ||
				brain_block_compare(row, col, "2*.", "22.", "2xx", "xxx")) {
				if (choice[col+2] && choice[col+2] < BRAIN_CHOICE_GOOD_1) {choice[col+2] = BRAIN_CHOICE_GOOD_1;}
			}
			if (brain_block_compare(row, col, ".1*", "111", "xxx", "xxx") ||
				brain_block_compare(row, col, ".2*", "222", "xxx", "xxx")) {
				if (choice[col] && choice[col] < BRAIN_CHOICE_GOOD_1) {choice[col] = BRAIN_CHOICE_GOOD_1;}
			}
			if (brain_block_compare(row, col, "*1.", "111", "xxx", "xxx") ||
				brain_block_compare(row, col, "*2.", "222", "xxx", "xxx")) {
				if (choice[col+2] && choice[col+2] < BRAIN_CHOICE_GOOD_1) {choice[col+2] = BRAIN_CHOICE_GOOD_1;}
			}
		}
	}



	/* try filling the middle columns */
	test = TRUE;
	for (col = 0; col < GAME_N_ROWS; col++) {
		if (choice[col] != BRAIN_CHOICE_NO_VALUE && choice[col] != BRAIN_CHOICE_FULL_COLUMN) {
			test = FALSE;
		}
	}
	if (test) {
		gint middle = GAME_N_COLS / 2;
		if (gnect.board[GAME_N_ROWS - 1][middle] == TILE_CLEAR) {
			if (choice[middle] != BRAIN_CHOICE_FULL_COLUMN) {choice[middle] = BRAIN_CHOICE_GOOD_1;}
		}
		else {
			if (choice[middle] != BRAIN_CHOICE_FULL_COLUMN) {choice[middle] = BRAIN_CHOICE_GOOD_2;}
		}
		if (choice[middle-2] != BRAIN_CHOICE_FULL_COLUMN) {choice[middle-2] = BRAIN_CHOICE_GOOD_4;}
		if (choice[middle-1] != BRAIN_CHOICE_FULL_COLUMN) {choice[middle-1] = BRAIN_CHOICE_GOOD_3;}
		if (choice[middle+1] != BRAIN_CHOICE_FULL_COLUMN) {choice[middle+1] = BRAIN_CHOICE_GOOD_3;}
		if (choice[middle+2] != BRAIN_CHOICE_FULL_COLUMN) {choice[middle+2] = BRAIN_CHOICE_GOOD_4;}
	}

}



static gboolean brain_block_compare(gint row, gint col,	const gchar *str1, const gchar *str2, const gchar *str3, const gchar *str4)
{
	if (brain_row_compare(row, col, str1, str2) &&
		brain_row_compare(row + 2, col, str3, str4)) {
		return(TRUE);
	}
	return(FALSE);
}



static gboolean brain_row_compare(gint row, gint col,
								  const gchar *rowStr,
								  const gchar *belowStr)
{
	/* Return TRUE if board[row][col]..board[row][col + strlen rowStr] matches rowStr
	 * and board[row + 1][col]..board[row + 1][col + strlen rowStr] matches belowStr
	 */

	gboolean result = TRUE;
	gboolean ok;
	gint l, i;



	l = strlen(rowStr);

	if (col + l > GAME_N_COLS - 1) {
		return(FALSE);
	}


	/*
	 *    . = empty
	 *    1 = player 1
	 *    2 = player 2
	 *    x = 1 or 2
	 *    * = 1, 2 or .
	 */


	for (i = 0; i < l; i++) {
		ok = FALSE;
		switch(rowStr[i]) {
		case '.' :
			ok = (gnect.board[row][col + i] == TILE_CLEAR);
			break;
		case '1' :
			ok = (gnect.board[row][col + i] == TILE_PLAYER_ONE);
			break;
		case '2' :
			ok = (gnect.board[row][col + i] == TILE_PLAYER_TWO);
			break;
		case 'x' :
			ok = (gnect.board[row][col + i] != TILE_CLEAR);
			break;
		default :
			ok = TRUE;
			break;
		}

		if (ok) {

			if (row != GAME_N_ROWS - 1) {

				switch(belowStr[i]) {
				case '.' :
					if (gnect.board[row + 1][col + i] != TILE_CLEAR) {
						result = FALSE;
					}
					break;
				case '1' :
					if (gnect.board[row + 1][col + i] != TILE_PLAYER_ONE) {
						result = FALSE;
					}
					break;
				case '2' :
					if (gnect.board[row + 1][col + i] != TILE_PLAYER_TWO) {
						result = FALSE;
					}
					break;
				case 'x' :
					if (gnect.board[row + 1][col + i] == TILE_CLEAR) {
						result = FALSE;
					}
					break;
				default :
					break;
				}
			}
		}
		else {
			result = FALSE;
		}
	}


	return(result);
}



static gint brain_test_drop(gint counter, gint col, gint len)
{
	/* Simulate dropping counter into col to see if it gives
	   counter a winning line. Return TRUE if it does */


	gint result = FALSE;
	gint row = 1;


	if (gnect.board[1][col] == TILE_CLEAR) {

		while(row < GAME_N_ROWS - 1 && gnect.board[row + 1][col] == TILE_CLEAR) {
			row++;
		}
		gnect.board[row][col] = counter;
		result = gnect_makes_line(counter, row, col, len);
		gnect.board[row][col] = TILE_CLEAR;

	}

	return(result);
}



static gint brain_has_multi_choice(gint val)
{
	/* Return TRUE if more than one column has a choice with
	 * weight 'val'.
	 */


	gint col = 0;
	gint result = 0;


	while(col < GAME_N_COLS) {
		if (choice[col] == val) {
			result++;
		}
		col++;
	}
	return(result > 1);
}



static gint brain_col_is_trap(gint counter, gint col, gint len)
{
	/* Return TRUE if dropping counter here lets the anti-counter's
	   next move win */


	gint isTrap = FALSE;
	gint row = 0;
	gint antiCounter = me;


	if (gnect.board[2][col] == TILE_CLEAR) {

		if (counter == me) {
			antiCounter = opponent;
		}

		while (row < GAME_N_ROWS && gnect.board[row + 1][col] == TILE_CLEAR) {
			row++;
		}

		gnect.board[row][col] = counter;
		gnect.board[row - 1][col] = antiCounter;

		isTrap = gnect_makes_line(antiCounter, row - 1, col, len);

		gnect.board[row - 1][col] = TILE_CLEAR;
		gnect.board[row][col] = TILE_CLEAR;
	}

	return(isTrap);
}



static gint brain_best_choice(void)
{
	/* Based on the choice value of each column, return
	   the bestCol in which to drop my counter */


	gint bestCol = 0;
	gint bestVal = 0;
	gint col;
	gint done = FALSE;


	for (col = 0; col < GAME_N_COLS; col++) {

		if (bestVal < choice[col]) {
			bestCol = col;
			bestVal = choice[bestCol];
		}
	}

	/* Pick between columns of highest value at random.
	   (this method wastes time, but computers are speedy...) */
	if (brain_has_multi_choice(bestVal)) {

		while(!done){
			bestCol = gnect_random_number(GAME_N_COLS) - 1;
			if (choice[bestCol] == bestVal) {
				done = TRUE;
			}
		}
	}

	if (choice[bestCol] == BRAIN_CHOICE_FULL_COLUMN) {

		/* best choice is a full column - shouldn't be here */

		gnect_error("gnect: dummy brain tried a full column\n");
	}

	if (choice[bestCol] == BRAIN_CHOICE_OPPONENTS_TRAP) {
		gnect_tantrum();
	}

	return(bestCol);
}
