/* GnomerMind - A MasterMind(R)-based game for Gnome
 * (C) 2001 Germano Rizzo
 *
 * prefs.c - preferences selection window implementation
 * Author: Germano Rizzo
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"
#include "gnomermind.h"
#include "face.h"
#include "keyboard.h"
#include "themes.h"
#include "sounds.h"
#include "save.h"

#define SAVE_PREFS 1

static const gchar *msg[] = {
  N_("You'll end the current game!"),
  N_("Are you sure?"),
  N_("Make default")
};

gint h, c, r;
gboolean tempinf, temprep, tempbla;

static gboolean exitable, repstore;

static GtkWidget *pref, *diff1, *diff2, *kd, *rlab;
static GtkWidget *ibut, *rbut, *bbut;
static GtkWidget *abut, *dbut;
#ifdef HAVE_ESD
static GtkWidget *sbut;
#endif

/**
 * rebuild
 * @reply: the pressed button
 * @data: ignored, only for callback
 *
 * Rebuilds the field with new parameters
 **/
static void
rebuild (gint reply, gpointer data)
{
  if (reply == GNOME_YES)
    {
      inf = tempinf;
      rep = temprep;
      bla = tempbla;
      inflate_board ();
      newall (GNOME_YES, NULL);
    }
}

/**
 * close_call
 *
 * Decides whether to close the dialog or not
 **/
static gboolean
close_call ()
{
  gboolean e;

  e = !exitable;
  exitable = TRUE;

  return e;
}

/**
 * button_pressed
 * @widget: ignored, only for callback
 * @button: the pressed button
 * @data:  ignored, only for callback
 *
 * Manages the pression of a button
 **/
static inline void
button_pressed (GtkWidget * widget, gint button, gpointer data)
{
  if (button == GNOME_OK)
    {
      if (!theme_compare ())
	if (theme_set ())
	  retheme_board ();

      if ((h != holez) || (c != colorz) || (r != roundz) || (tempinf != inf)
	  || (temprep != rep) || (tempbla != bla))
	{
	  GtkWidget *dialog;

	  if (ask)
	    dialog =
	      gnome_question_dialog_parented (g_strconcat
					      (_(msg[0]), "\n",
					       _(msg[1]), NULL),
					      rebuild, NULL,
					      GTK_WINDOW (gwin));
	  else
	    rebuild (GNOME_YES, NULL);
	}
    }

  if (button == SAVE_PREFS)
    {
      save_defaults ();
      exitable = FALSE;
    }
}

/**
 * rlab_upd
 *
 * Updates the content of the rounds spin box label
 **/
static inline void
rlab_upd ()
{
  if (tempinf)
    gtk_label_set_text (GTK_LABEL (rlab), _("Visible rounds"));
  else
    gtk_label_set_text (GTK_LABEL (rlab), _("Available rounds"));
}

/**
 * l2m
 * @list: the GList
 * @cnt: its members number
 *
 * Generates a GtkMenu out of the GList given
 **/
static GtkWidget *
l2m (GList * list, guint cnt, gint type)
{
  GtkWidget *menu, *mi[cnt];
  GList *tmp;
  gint i;

  tmp = list;

  menu = gtk_menu_new ();

  for (i = 0; i < cnt; i++)
    {
      mi[i] = gtk_menu_item_new_with_label (tmp->data);

      switch (type)
	{
	case 0:
	  gtk_signal_connect (GTK_OBJECT (mi[i]), "activate",
			      GTK_SIGNAL_FUNC (key_sel),
			      (gpointer) tmp->data);
	  break;
	case 1:
	  gtk_signal_connect (GTK_OBJECT (mi[i]), "activate",
			      GTK_SIGNAL_FUNC (theme_sel),
			      (gpointer) tmp->data);
	  break;
	default:
	  g_assert_not_reached ();
	}

      if ((type != 1) || (strcmp (theme, tmp->data)))
	gtk_menu_append (GTK_MENU (menu), mi[i]);
      else
	gtk_menu_prepend (GTK_MENU (menu), mi[i]);

      tmp = g_list_next (tmp);
    }

  g_assert (menu);

  return menu;
}

/**
 * list2menu
 * @list: the GList
 * 
 * Generates a GtkMenu out of the GList given
 **/
static GtkWidget *
list2menu (GList * list, gint type)
{
  return l2m (list, g_list_length (list), type);
}

/**
 * diff_calc
 *
 * Estimates the value of difficulty
 **/
static inline void
diff_calc ()
{
  gdouble d;
  gchar *label;

  /*
   * original method, less realistic, IMHO
   * d = (150 * h * c - 200 * r) / (13 * r);
   */

  d = (150 * h + 150 * c - 100 * r) / (3 * r);

  /*
   * difficulty -40% if infinite rounds
   */

  if (tempinf)
    d = d * 3 / 5;

  /*
   * difficulty -10% if repetitions aren't allowed
   */

  if (!temprep)
    d = d * 9 / 10;

  /*
   * difficulty -20% if blank guessings are allowed
   */

  if (tempbla)
    d = d * 4 / 5;

  g_assert (d >= 0 && d <= 100);

  label = g_strdup_printf (_("Estimated difficulty level: %d%%"), (gint) d);
  gtk_label_set_text (GTK_LABEL (diff1), label);
  gtk_label_set_text (GTK_LABEL (diff2), label);
}

/**
 * itog
 *
 * Manages the check button for the `inf' parameter
 **/
static void
itog ()
{
  tempinf = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ibut));
  rlab_upd ();
  diff_calc ();
}

/**
 * rtog
 *
 * Manages the check button for the `rep' parameter
 **/
static void
rtog ()
{
  temprep = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (rbut));
  diff_calc ();
}

/**
 * rep_check
 *
 * Validates the repetitions check button value.
 * If the number of different tokens is less than
 * the one of the places to guess, it turns
 * the widget off.
 **/
static void
rep_check ()
{
  if (c < h)
    {
      if (!temprep)
	{
	  repstore = FALSE;
	  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rbut), TRUE);
	}
      gtk_widget_set_sensitive (rbut, FALSE);
    }
  else
    {
      if (!repstore)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rbut), FALSE);
      gtk_widget_set_sensitive (rbut, TRUE);
      repstore = TRUE;
    }
}

/**
 * btog
 *
 * Manages the check button for the `bla' parameter
 **/
static void
btog ()
{
  tempbla = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (bbut));
  diff_calc ();
}

/**
 * dtog
 *
 * Manages the check button for the `drop' parameter
 **/
static void
dtog ()
{
  drop = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dbut));
}

/**
 * atog
 *
 * Manages the check button for the `ask' parameter
 **/
static void
atog ()
{
  ask = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (abut));
}

#ifdef HAVE_ESD

/**
 * stog
 *
 * Manages the check button for the `sound_enable' parameter
 **/
static void
stog ()
{
  sound_enable = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (sbut));
}

#endif

/**
 * h_get
 * @adj: the changed Adjustment
 * @spin: the Spin Button related to it
 *
 * Manages a change in the value of the `holes' spin button
 **/
static void
h_get (GtkAdjustment * adj, GtkSpinButton * spin)
{
  h = gtk_spin_button_get_value_as_int (spin);
  diff_calc ();
}

/**
 * c_get
 * @adj: the changed Adjustment
 * @spin: the Spin Button related to it
 *
 * Manages a change in the value of the `colors' spin button
 **/
static void
c_get (GtkAdjustment * adj, GtkSpinButton * spin)
{
  c = gtk_spin_button_get_value_as_int (spin);
  diff_calc ();
}

/**
 * r_get
 * @adj: the changed Adjustment
 * @spin: the Spin Button related to it
 *
 * Manages a change in the value of the `rounds' spin button
 **/
static void
r_get (GtkAdjustment * adj, GtkSpinButton * spin)
{
  r = gtk_spin_button_get_value_as_int (spin);
  diff_calc ();
}

/**
 * kd_upd
 * @lbl: the label to show
 *
 * Updates the key display (kd) widget
 **/
inline void
kd_upd (gchar * lbl)
{
  gtk_label_set_text (GTK_LABEL (kd), lbl);
}

/**
 * prefs_show
 *
 * Builds and shows the preferences window
 **/
void
prefs_show ()
{
  GtkWidget *hspin, *cspin, *rspin;
  GtkWidget *hlab, *clab;
  GtkWidget *hbox, *cbox, *rbox;
  GtkWidget *ktab, *kom, *kl1, *kl2, *kb;
  GtkWidget *tom, *tl;
  GtkWidget *preftab, *vbox1, *lab1, *vbox2, *lab2, *vbox3, *lab3, *vbox4,
    *lab4;
  GtkAdjustment *hadj, *cadj, *radj;

  if (GTK_IS_WIDGET (pref))
    /*  if(GTK_WIDGET_VISIBLE(pref))  */
    return;

  g_assert (validate (theme));

  exitable = TRUE;

  h = holez;
  c = colorz;
  r = roundz;
  tempinf = inf;
  temprep = rep;
  tempbla = bla;
  repstore = TRUE;

  /*
   * first page
   */

  hadj = (GtkAdjustment *) gtk_adjustment_new (h, MINH, MAXH, 1, 2, 2);
  cadj = (GtkAdjustment *) gtk_adjustment_new (c, MINC, MAXC, 1, 2, 2);
  radj = (GtkAdjustment *) gtk_adjustment_new (r, MINR, MAXR, 1, 2, 2);

  hspin = gtk_spin_button_new (hadj, 0, 0);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (hspin),
				     GTK_UPDATE_IF_VALID);
  gtk_signal_connect (GTK_OBJECT (hadj), "value_changed",
		      GTK_SIGNAL_FUNC (h_get), (gpointer) hspin);
  gtk_signal_connect (GTK_OBJECT (hadj), "value_changed",
		      GTK_SIGNAL_FUNC (rep_check), (gpointer) hspin);

  cspin = gtk_spin_button_new (cadj, 0, 0);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (cspin),
				     GTK_UPDATE_IF_VALID);
  gtk_signal_connect (GTK_OBJECT (cadj), "value_changed",
		      GTK_SIGNAL_FUNC (c_get), (gpointer) cspin);
  gtk_signal_connect (GTK_OBJECT (cadj), "value_changed",
		      GTK_SIGNAL_FUNC (rep_check), (gpointer) cspin);

  rspin = gtk_spin_button_new (radj, 0, 0);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (rspin),
				     GTK_UPDATE_IF_VALID);
  gtk_signal_connect (GTK_OBJECT (radj), "value_changed",
		      GTK_SIGNAL_FUNC (r_get), (gpointer) rspin);
  gtk_signal_connect (GTK_OBJECT (radj), "value_changed",
		      GTK_SIGNAL_FUNC (rep_check), (gpointer) rspin);

  hlab = gtk_label_new (_("Tokens to guess"));
  clab = gtk_label_new (_("Token range extension"));
  rlab = gtk_label_new ("");
  rlab_upd ();

  hbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox), hspin, FALSE, FALSE, GNOME_PAD);
  gtk_box_pack_start (GTK_BOX (hbox), hlab, FALSE, FALSE, GNOME_PAD);

  cbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (cbox), cspin, FALSE, FALSE, GNOME_PAD);
  gtk_box_pack_start (GTK_BOX (cbox), clab, FALSE, FALSE, GNOME_PAD);

  rbox = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (rbox), rspin, FALSE, FALSE, GNOME_PAD);
  gtk_box_pack_start (GTK_BOX (rbox), rlab, FALSE, FALSE, GNOME_PAD);

  ibut = gtk_check_button_new_with_label (_("Play on infinite rounds"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ibut), tempinf);
  gtk_signal_connect (GTK_OBJECT (ibut), "toggled", GTK_SIGNAL_FUNC (itog),
		      NULL);

  diff1 = gtk_label_new ("");

  vbox1 = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox1), hbox, FALSE, FALSE, GNOME_PAD);
  gtk_box_pack_start (GTK_BOX (vbox1), cbox, FALSE, FALSE, GNOME_PAD);
  gtk_box_pack_start (GTK_BOX (vbox1), rbox, FALSE, FALSE, GNOME_PAD);
  gtk_box_pack_start (GTK_BOX (vbox1), ibut, FALSE, FALSE, GNOME_PAD);
  gtk_box_pack_end (GTK_BOX (vbox1), diff1, FALSE, FALSE, GNOME_PAD);

  /*
   * second page
   */

  rbut =
    gtk_check_button_new_with_label (_
				     ("Allow repetitions in the row to guess"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (rbut), temprep);
  gtk_signal_connect (GTK_OBJECT (rbut), "toggled", GTK_SIGNAL_FUNC (rtog),
		      NULL);

  bbut =
    gtk_check_button_new_with_label (_
				     ("Allow to leave blank places in guessings"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (bbut), tempbla);
  gtk_signal_connect (GTK_OBJECT (bbut), "toggled", GTK_SIGNAL_FUNC (btog),
		      NULL);

  diff2 = gtk_label_new ("");

  vbox2 = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox2), rbut, FALSE, FALSE, GNOME_PAD);
  gtk_box_pack_start (GTK_BOX (vbox2), bbut, FALSE, FALSE, GNOME_PAD);
  gtk_box_pack_end (GTK_BOX (vbox2), diff2, FALSE, FALSE, GNOME_PAD);

  /*
   * third page
   */

  dbut =
    gtk_check_button_new_with_label (_("Drop the token after each move"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dbut), drop);
  gtk_signal_connect (GTK_OBJECT (dbut), "toggled", GTK_SIGNAL_FUNC (dtog),
		      NULL);
  gtk_signal_connect (GTK_OBJECT (dbut), "toggled",
		      GTK_SIGNAL_FUNC (key_grab_release), NULL);

  abut =
    gtk_check_button_new_with_label (_("Activate the confirmation dialogs"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (abut), ask);
  gtk_signal_connect (GTK_OBJECT (abut), "toggled", GTK_SIGNAL_FUNC (atog),
		      NULL);
  gtk_signal_connect (GTK_OBJECT (abut), "toggled",
		      GTK_SIGNAL_FUNC (key_grab_release), NULL);

#ifdef HAVE_ESD

  sbut = gtk_check_button_new_with_label (_("Sounds active"));
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (sbut), sound_enable);
  gtk_signal_connect (GTK_OBJECT (sbut), "toggled", GTK_SIGNAL_FUNC (stog),
		      NULL);
  gtk_signal_connect (GTK_OBJECT (sbut), "toggled",
		      GTK_SIGNAL_FUNC (key_grab_release), NULL);

#endif

  ktab = gtk_table_new (0, 0, FALSE);

  kl1 = gtk_label_new (_("Command:"));
  gtk_table_attach_defaults (GTK_TABLE (ktab), kl1, 0, 1, 0, 1);

  kl2 = gtk_label_new (_("Key:"));
  gtk_table_attach_defaults (GTK_TABLE (ktab), kl2, 2, 3, 0, 1);

  kb = gtk_button_new_with_label (_("Set"));
  gtk_signal_connect (GTK_OBJECT (kb), "clicked", GTK_SIGNAL_FUNC (key_new),
		      NULL);
  gtk_widget_add_events (kb, GDK_KEY_PRESS_MASK);
  gtk_signal_connect (GTK_OBJECT (kb), "key_press_event",
		      GTK_SIGNAL_FUNC (key_new_press), NULL);

  gtk_table_attach_defaults (GTK_TABLE (ktab), kb, 1, 2, 1, 2);

  kd = gtk_label_new (gdk_keyval_name (key_get ()));
  gtk_table_attach_defaults (GTK_TABLE (ktab), kd, 2, 3, 1, 2);

  kom = gtk_option_menu_new ();
  gtk_option_menu_set_menu (GTK_OPTION_MENU (kom),
			    list2menu (gen_key_list (), 0));
  gtk_signal_connect (GTK_OBJECT (kom), "clicked",
		      GTK_SIGNAL_FUNC (key_grab_release), NULL);

  gtk_table_attach_defaults (GTK_TABLE (ktab), kom, 0, 1, 1, 2);

  gtk_table_set_row_spacings (GTK_TABLE (ktab), 3);
  gtk_table_set_col_spacings (GTK_TABLE (ktab), 3);

  vbox3 = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox3), dbut, FALSE, FALSE, GNOME_PAD);
  gtk_box_pack_start (GTK_BOX (vbox3), abut, FALSE, FALSE, GNOME_PAD);
#ifdef HAVE_ESD
  gtk_box_pack_start (GTK_BOX (vbox3), sbut, FALSE, FALSE, GNOME_PAD);
#endif
  gtk_box_pack_start (GTK_BOX (vbox3), ktab, FALSE, FALSE, GNOME_PAD);

  /*
   * fourth page
   */

  tom = gtk_option_menu_new ();
  gtk_option_menu_set_menu (GTK_OPTION_MENU (tom), list2menu (theme_list, 1));

  tl = gtk_label_new (_("Switch to theme:"));

  vbox4 = gtk_vbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (vbox4), tl, FALSE, FALSE, GNOME_PAD);
  gtk_box_pack_start (GTK_BOX (vbox4), tom, FALSE, FALSE, GNOME_PAD);

  /*
   * bind them all
   */

  diff_calc ();

  pref =
    gnome_dialog_new (_("Preferences"), GNOME_STOCK_BUTTON_OK,
		      _(msg[2]), GNOME_STOCK_BUTTON_CANCEL, NULL);
  gtk_window_set_wmclass (GTK_WINDOW (pref), "GnomerMind", "preferences");
  gnome_dialog_set_close (GNOME_DIALOG (pref), TRUE);
  gnome_dialog_set_parent (GNOME_DIALOG (pref), GTK_WINDOW (gwin));
  gtk_signal_connect (GTK_OBJECT (pref), "clicked",
		      GTK_SIGNAL_FUNC (button_pressed), NULL);
  gtk_signal_connect (GTK_OBJECT (pref), "close",
		      GTK_SIGNAL_FUNC (close_call), NULL);
  gtk_signal_connect (GTK_OBJECT (pref), "focus-out-event",
		      GTK_SIGNAL_FUNC (key_grab_release), NULL);

  preftab = gtk_notebook_new ();
  gtk_signal_connect (GTK_OBJECT (preftab), "switch-page",
		      GTK_SIGNAL_FUNC (key_grab_release), NULL);

  lab1 = gtk_label_new (_("Game settings"));
  lab2 = gtk_label_new (_("Advanced"));
  lab3 = gtk_label_new (_("Controls"));
  lab4 = gtk_label_new (_("Themes"));

  gtk_notebook_append_page (GTK_NOTEBOOK (preftab), vbox1, lab1);
  gtk_notebook_append_page (GTK_NOTEBOOK (preftab), vbox2, lab2);
  gtk_notebook_append_page (GTK_NOTEBOOK (preftab), vbox3, lab3);
  gtk_notebook_append_page (GTK_NOTEBOOK (preftab), vbox4, lab4);
  gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (pref)->vbox), preftab, FALSE,
		      FALSE, GNOME_PAD);
/*
  pref=gnome_property_box_new();

  gnome_property_box_append_page(GNOME_PROPERTY_BOX(pref), vbox1, lab1);
  gnome_property_box_append_page(GNOME_PROPERTY_BOX(pref), vbox2, lab2);
  gnome_property_box_append_page(GNOME_PROPERTY_BOX(pref), vbox3, lab3);
  gnome_property_box_append_page(GNOME_PROPERTY_BOX(pref), vbox4, lab4);

  gtk_signal_connect (GTK_OBJECT (pref), "close",
		      GTK_SIGNAL_FUNC (close_call), NULL);
  gtk_signal_connect (GTK_OBJECT (pref), "focus-out-event",
		      GTK_SIGNAL_FUNC (key_grab_release), NULL);
*/
  rep_check ();

  gtk_widget_show_all (pref);
}
