# GNU Solfege - eartraining for GNOME
# Copyright (C) 2000-2001  Tom Cato Amundsen
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import gtk, GDK, gnome
import mpd, gu, cfg
import types
from i18n import _

class NotenameSpinButton(gtk.GtkHBox):
    def __init__(self, value="c", lower=mpd.LOWEST_NOTENAME, 
                 upper=mpd.HIGHEST_NOTENAME, callback=None):
        assert type(value) == types.StringType
        assert type(lower) == type(upper) == types.StringType
        gtk.GtkHBox.__init__(self)
        self.m_lower = mpd.notename_to_int(lower)
        self.m_upper = mpd.notename_to_int(upper)
        assert self.m_lower <= mpd.notename_to_int(value) <= self.m_upper
        self.g_entry = gtk.GtkEntry()
        self.g_entry.set_text(value)
        self.pack_start(self.g_entry, gtk.FALSE, gtk.FALSE)
        self.g_entry.set_usize(40, -1)
        self.g_entry.set_editable(gtk.FALSE)

        box = gtk.GtkVBox()
        self.pack_start(box, gtk.FALSE, gtk.FALSE)
        eb = gtk.GtkEventBox()
        eb.connect('button_press_event', self.on_arrowclick, 1)
        self.g_up = gtk.GtkArrow(at=gtk.ARROW_UP)
        eb.add(self.g_up)
        box.pack_start(eb)
        eb = gtk.GtkEventBox()
        eb.connect('button_press_event', self.on_arrowclick, -1)
        self.g_down = gtk.GtkArrow(at=gtk.ARROW_DOWN)
        eb.add(self.g_down)
        box.pack_start(eb)
        # just to get value checked
        self.change_value(0)
        self.g_entry.connect('event', self.on_entry_event)
        if callback:
            self.g_entry.connect('changed', callback)
    def set_range(self, lower, upper):
        self.m_lower = mpd.notename_to_int(lower)
        self.m_upper = mpd.notename_to_int(upper)
        # just to get value checked
        self.change_value(0)
    def on_entry_event(self, entry, event):
        if event.type == GDK.KEY_PRESS:
            if event.keyval == GDK.Page_Up:
                self.change_value(12)
                return 1
            elif event.keyval == GDK.Page_Down:
                self.change_value(-12)
                return 1
            elif event.keyval == GDK.Up:
                self.change_value(1)
                return 1
            elif event.keyval == GDK.Down:
                self.change_value(-1)
                return 1
    def on_arrowclick(self, _o, event, v):
        if event.type == GDK.BUTTON_PRESS:
            assert v in (-1, 1)
            if event.button == 1:
                self.change_value(v)
            elif event.button == 2: # middle button
                self.change_value(v*12)
            elif event.button == 3: # right button
                if v == 1:
                    self.set_value(self.m_upper)
                else:
                    self.set_value(self.m_lower)
    def get_value(self):
        return self.g_entry.get_text()
    def change_value(self, v):
        self.set_value(mpd.notename_to_int(self.get_value()) + v)
    def set_value(self, n):
        if type(n) == types.StringType: 
            n = mpd.notename_to_int(n)
        assert type(n) == types.IntType
        assert self.m_lower <= self.m_upper
        if n < self.m_lower:
            n = self.m_lower
        elif n > self.m_upper:
            n = self.m_upper
        # only change the entry if the value has actually changed, to
        # avoid unnecessary 'changed' signals emitted.
        if self.g_entry.get_text() != mpd.int_to_notename(n):
            self.g_entry.set_text(mpd.int_to_notename(n))

 
class nNotenameSpinButton(NotenameSpinButton, cfg.ConfigUtils):
    def __init__(self, exname, varname, default_value="c",
                 lower=mpd.LOWEST_NOTENAME, upper=mpd.HIGHEST_NOTENAME,
                 callback=None):
        NotenameSpinButton.__init__(self, default_value, lower, upper,
                                    callback=self.on_changed)
        cfg.ConfigUtils.__init__(self, exname)
        self.m_name = varname
        self.m_callback = None
        p = self.get_string(self.m_name)
        if p:
            self.g_entry.set_text(p)
        self.m_callback = callback
        self.add_watch(self.m_name, self._watch_cb)
    def _watch_cb(self, name):
        self.g_entry.set_text(self.get_string(name))
    def on_changed(self, _o=None):
        self.set_string(self.m_name, self.get_value())
        if self.m_callback is not None:
            self.m_callback()

class nNotenameRange(gtk.GtkHBox, cfg.ConfigUtils):
    def __init__(self, exname, lowname, highname, 
                 lowest_value=mpd.LOWEST_NOTENAME,
                 highest_value=mpd.HIGHEST_NOTENAME):
        """
        lowest_value and higest_value are notename that tell the tones the
        range has to be within.
        """
        gtk.GtkHBox.__init__(self)
        cfg.ConfigUtils.__init__(self, exname)
        assert mpd.compare_notenames(lowest_value, highest_value) <= 0
        self.m_lowest_value = lowest_value
        self.m_highest_value = highest_value
        self.g_low = nNotenameSpinButton(exname, lowname, lowest_value,
                                         callback=self.low_cb)
        self.pack_start(self.g_low, gtk.FALSE)
        self.g_low.show()
        self.pack_start(gtk.GtkLabel(" %s " %_("to")), gtk.FALSE)
        self.g_high = nNotenameSpinButton(exname, highname, highest_value,
                                          callback=self.high_cb)
        self.set_range(lowest_value, highest_value)
        self.pack_start(self.g_high, gtk.FALSE)
        self.g_high.show()
    def set_range(self, lowest, highest):
        self.m_lowest_value = lowest
        self.m_highest_value = highest
        self.g_low.set_range(lowest, self.g_high.get_value())
        self.g_high.set_range(self.g_low.get_value(), highest)
    def low_cb(self):
        self.g_high.set_range(self.g_low.get_value(), self.m_highest_value)
    def high_cb(self):
        self.g_low.set_range(self.m_lowest_value, self.g_high.get_value())
