# GNU Solfege - eartraining for GNOME
# Copyright (C) 2000-2001  Tom Cato Amundsen
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import widgets, inputwidgets, abstract, const, random, soundcard, gu, mpd, gnome
import configureoutput
from i18n import _
import gtk
import gu
import statistics, statisticsviewer

class Teacher(abstract.Teacher):
    OCTAVES = [-2, -1, 0, 1, 2, 3]
    def __init__(self, exname, app):
        abstract.Teacher.__init__(self, exname, app)
        self.m_statistics = statistics.Statistics(self)
        self.m_ask_tones =   {}
        self.m_question = None
    def new_question(self):
        """
        return 1 if it is properly configured and has a new question, otherwise
        return None
        """
        v = []
        for n in ['c', 'cis', 'd', 'dis', 'e', 'f', 'fis', 
                  'g', 'gis', 'a', 'ais', 'b']:
            v.extend([n] * self.get_int(n+"_weight"))
        if not v:
            return
        self.m_question = random.choice(v)
        v = []
        for n in self.OCTAVES:
            if self.get_bool("octave"+str(n)):
                v.append(n)
        if not v:
            return
        self.m_octave = random.choice(v)
        self.q_status = const.NEW_QUESTION
        return 1
    def guess_answer(self, notename):
        if notename == self.m_question:
            if self.q_status == const.NEW_QUESTION:
                self.m_statistics.add_correct(notename)
            self.q_status = const.SOLVED
            return 1
        else:
            if self.q_status == const.NEW_QUESTION:
                self.m_statistics.add_wrong(self.m_question, notename)
            self.q_status = const.WRONG_GUESS
    def play_question(self):
        soundcard.play_note(self.get_int('config/preferred_instrument'), 4, 0, 
             mpd.notename_to_int(self.m_question)+self.m_octave*12,
             self.get_int('config/preferred_instrument_velocity'))
    def spank_me_play_question(self):
        track = soundcard.Track()
        track.note(8, 9, 71, mpd.const.MF)
        track.note(4, 0, mpd.notename_to_int(self.m_question)+self.m_octave*12, 90)
        soundcard.synth.play_track(track)
    def spank_me(self):
        soundcard.play_note(0, 4, 9, 71, 100)
    def end_practise(self):
        self.q_status = const.NO_QUESTION

class Gui(abstract.Gui):
    def __init__(self, teacher, window):
        abstract.Gui.__init__(self, teacher, window)
        if not configureoutput.WITH_GNOME:
            self.practise_box.add(gtk.GtkLabel(_(
"""This exercise don't work without GNOME.
It can pretty easy be converted to use GtkPixmap instead
of GnomeCanvas, but I (tca) don't have time to do this
right now. But I can answer questions if needed.""")))
            self.practise_box.show_all()
            return
        self.m_key_bindings = {'start_ak': self.start_practise,
                               'repeat_ak': self.m_t.play_question}
        v = []
        for k in ('c', 'cis', 'd', 'dis', 'e', 'f', 'fis',
                  'g', 'gis', 'a', 'ais', 'b'):
            self.m_key_bindings['tone_%s_ak' % k] \
                = lambda self=self, k=k: self.on_answer_from_user(k)
            v.append(self.get_string('tone_%s_ak' % k))

        self.g_piano = inputwidgets.PianoOctaveWithAccelName(
                       self.on_answer_from_user, v)
        self.practise_box.pack_start(self.g_piano)

        self.g_flashbar = gu.FlashBar()
        self.g_flashbar.show()
        self.practise_box.pack_start(self.g_flashbar, gtk.FALSE)
        self.practise_box.set_spacing(gnome.uiconsts.PAD)

        gu.bButton(self.action_area, _("Start"), self.start_practise)
        self.g_repeat = gu.bButton(self.action_area, _("Repeat"),
                  lambda _o, self=self: self.m_t.play_question())
        self.g_repeat.set_sensitive(gtk.FALSE)
        self.practise_box.show_all()
        ##############
        # config_box #
        ##############
        table = gtk.GtkTable()
        self.config_box.pack_start(table, gtk.FALSE)
        for x, n in [(1, 'cis'), (3, 'dis'), (7, 'fis'),
                     (9, 'gis'), (11, 'ais')]:
             label = gtk.GtkLabel(n)
             label.set_name("Heading2")
             label.set_alignment(0.2, 1.0)
             table.attach(label, x, x+2, 0, 1, xoptions=gtk.FILL)
             b = gu.nSpinButton(self.m_exname, n+"_weight",
                      gtk.GtkAdjustment(1, 0, 1000, 1, 10), digits=0)
             table.attach(b, x, x+2, 1, 2, xoptions=gtk.FILL)
        for x, n in [(0, 'c'), (2, 'd'), (4, 'e'), (6, 'f'),
                      (8, 'g'), (10, 'a'), (12, 'b')]:
             label = gtk.GtkLabel(n)
             label.set_name("Heading2")
             label.set_alignment(0.35, 1.0)
             table.attach(label, x, x+2, 2, 3, xoptions=gtk.FILL)
             b = gu.nSpinButton(self.m_exname, n+"_weight",
                   gtk.GtkAdjustment(1, 0, 1000, 1, 10), digits=0)
             table.attach(b, x, x+2, 3, 4, xoptions=gtk.FILL)
             
        hbox = gu.bHBox(self.config_box, gtk.FALSE)
        hbox.pack_start(gtk.GtkLabel(_("Octave:")), gtk.FALSE, padding=4)
        for oct in self.m_t.OCTAVES:
            b = gu.nCheckButton(self.m_exname, "octave"+str(oct), str(oct),
                                default_value=1)
            hbox.pack_start(b, gtk.FALSE)
        frame = gtk.GtkFrame(_("When you guess wrong"))
        vbox = gtk.GtkVBox()
        vbox.set_border_width(gnome.uiconsts.PAD_SMALL)
        frame.add(vbox)
        vbox.pack_start(gu.nCheckButton(self.m_exname, 
                    "warning_sound", _("Play warning sound")))
        vbox.pack_start(gu.nCheckButton(self.m_exname, "one_chance_only",
                  _("Only one chance to get the answer right")))
        vbox.pack_start(gu.nCheckButton(self.m_exname, "repeat_question",
                  _("Repeat question")))
        self.config_box.pack_start(frame, gtk.FALSE)
        self.config_box.set_spacing(gnome.uiconsts.PAD)
        self.config_box.show_all()
        ##############
        # statistics #
        ##############
        self.setup_statisticsviewer(statisticsviewer.StatisticsViewer,
                                   _("idtone"))

    def start_practise(self, widget=None):
        if not self.m_t.new_question():
            gnome.ui.GnomeErrorDialog("You have to select the tones to use")
            return
        self.g_repeat.set_sensitive(gtk.TRUE)
        self.g_flashbar.flash(_("First tone is %s") % self.m_t.m_question)
        self.m_t.play_question()
    def on_answer_from_user(self, notename):
        if self.m_t.q_status == const.NO_QUESTION:
            self.g_flashbar.flash(_("Click on 'Start' to begin."))
            return
        elif self.m_t.q_status == const.SOLVED:
            if self.m_t.guess_answer(notename):
                self.g_flashbar.flash(_("Correct, but you have already solved this question"))
            else:
                self.g_flashbar.flash(_("Wrong, but you have already solved this question"))
        else:
            if self.m_t.guess_answer(notename):
                self.g_flashbar.flash(_("Correct"))
                self.m_t.new_question()
                self.m_t.play_question()
            else:
                self.g_flashbar.flash(_("Wrong"))
                if self.get_bool('one_chance_only'):
                    self.m_t.new_question()
                    if self.get_bool("warning_sound"):
                        self.m_t.spank_me_play_question()
                    else:
                        self.m_t.play_question()
                else:
                    if self.get_bool("warning_sound"):
                        if self.get_bool("repeat_question"):
                            self.m_t.spank_me_play_question()
                        else:
                            self.m_t.spank_me()
                    else:
                        if self.get_bool("repeat_question"):
                            self.m_t.play_question()
    def on_start_practise(self):
        if configureoutput.WITH_GNOME:
            gtk.timeout_add(const.SHORT_WAIT, lambda self=self:
                self.g_flashbar.flash(_("Click on 'Start' to begin.")))
    def on_end_practise(self):
        import configureoutput
        if configureoutput.WITH_GNOME:
            self.g_repeat.set_sensitive(gtk.FALSE)
            self.m_t.end_practise()
