# GNU Solfege - eartraining for GNOME
# Copyright (C) 2000-2001  Tom Cato Amundsen
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import random
import gtk, gnome
import gu, widgets, inputwidgets
import soundcard, mpd
import abstract, const, utils
from i18n import _
import statistics, statisticsviewer

class Teacher(abstract.Teacher):
    OK = 0
    ERR_PICKY = 1
    ERR_NO_INTERVALLS = 2
    ERR_NOTERANGE = 3
    def __init__(self, exname, app):
        abstract.Teacher.__init__(self, exname, app)
        self.m_statistics = statistics.IntervallStatistics(self)
        self.m_tonika = None
        self.m_intervall = None
    def new_question(self):
        """
        Return values:
        OK: new question created, all ok
        ERR_NO_INTERVALLS: no new question because no intervalls are selected
        ERR_PICKY: you have to solve this question before you are allowed to create new
        ERR_RANGE: to small range of tones for the selected intervalls
        """
        if self.m_timeout_handle:
            gtk.timeout_remove(self.m_timeout_handle)
            self.m_timeout_handle = None
            
        if self.get_bool('config/picky_on_new_question') \
                 and self.q_status in [const.NEW_QUESTION, const.WRONG_GUESS]:
             return Teacher.ERR_PICKY

        self.q_status = const.NO_QUESTION

        if self.get_string('intervalls') == []:
            return Teacher.ERR_NO_INTERVALLS
        last_question = self.m_intervall
        last_tonika = self.m_tonika
        while 1:
            self.m_tonika, self.m_intervall = \
                  utils.random_tonika_and_intervall(
                    self.get_string('lowest_tone'),
                    self.get_string('highest_tone'),
                    self.get_list('intervalls'))
            if self.m_tonika is None:
                return Teacher.ERR_NOTERANGE
            if last_question is None:
                break
            v = utils.filter_intervalls_within_range(
                                 self.get_string('lowest_tone'),
                                 self.get_string('highest_tone'),
                                 self.get_list('intervalls'))
            d =  mpd.notename_to_int(self.get_string('highest_tone'))\
                  - mpd.notename_to_int(self.get_string('lowest_tone'))
            if (self.m_intervall == last_question
                and self.m_tonika == last_tonika) \
                and (len(v) > 1 or v[0] < d):
                continue
            break
        assert self.m_tonika
        self.q_status = const.NEW_QUESTION
        return Teacher.OK
    def guess_answer(self, answer):
        """
        Return: 1 if correct, None if wrong
        """
        assert self.q_status != const.NO_QUESTION
        if self.m_intervall == answer:
            if self.q_status == const.NEW_QUESTION:
                self.m_statistics.add_correct(self.m_intervall)
            self.maybe_auto_new_question()
            self.q_status = const.SOLVED
            return 1
        else:
            if self.q_status == const.NEW_QUESTION:
                self.m_statistics.add_wrong(self.m_intervall, answer)
                self.q_status = const.WRONG_GUESS
    def play_question(self):
        if self.q_status == const.NO_QUESTION:
            return
        instr_low, instr_low_vel, instr_high, instr_high_vel = self.get_instrument_config(2)
        low_tone = self.m_tonika.semitone_pitch()
        high_tone = (self.m_tonika+self.m_intervall).semitone_pitch()

        m = soundcard.Track()
        m.set_bpm(self.get_int('config/default_bpm'))
        m.set_patch(0, instr_low)
        m.set_patch(1, instr_high)
        m.start_note(0, low_tone, instr_low_vel)
        m.start_note(1, high_tone, instr_high_vel)
        m.notelen_time(4)
        m.stop_note(0, low_tone, instr_low_vel)
        m.stop_note(1, high_tone, instr_high_vel)
        soundcard.synth.play_track(m)
        return 1
    def play_melodic(self):
        if self.q_status == const.NO_QUESTION:
            return
        if self.get_bool('override_default_instrument'):
            instr_low = self.get_int('lowest_instrument')
            instr_low_vel = self.get_int('lowest_instrument_velocity')
            instr_high = self.get_int('highest_instrument')
            instr_high_vel = self.get_int('highest_instrument_velocity')
        else:
            instr_low = instr_high = self.get_int('config/preferred_instrument')
            instr_low_vel = instr_high_vel \
                    = self.get_int('config/preferred_instrument_velocity')
        low_tone = self.m_tonika.semitone_pitch()
        high_tone = (self.m_tonika+self.m_intervall).semitone_pitch()
        m = soundcard.Track()
        m.set_bpm(self.get_int('config/default_bpm'))
        m.set_patch(0, instr_low)
        m.set_patch(1, instr_high)
        m.note(4, 0, low_tone, instr_low_vel)
        m.note(4, 1, high_tone, instr_high_vel) 
        soundcard.synth.play_track(m)


class Gui(abstract.IntervallGui):
    def __init__(self, teacher, window):
        abstract.IntervallGui.__init__(self, teacher, window)
        ################
        # practice_box #
        ################
        self.g_repeat_melodic = gu.bButton(self.action_area, 
            _("Repeat melodic"), lambda _o, self=self: self.m_t.play_melodic())

        self.g_repeat_melodic.set_sensitive(gtk.FALSE)
        ##############
        # config_box #        
        ##############
        frame = gtk.GtkFrame(_("Ask for these intervalls"))
        self.config_box.pack_start(frame, gtk.FALSE)
        
        self.g_intervall_selector = widgets.IntervallCheckBox(
                                          self.m_exname, 'intervalls')

        self.g_intervall_selector.set_border_width(gnome.uiconsts.PAD)
        frame.add(self.g_intervall_selector)
        #------we need some space
        self.config_box.pack_start(gtk.GtkHBox(), gtk.FALSE,
                                   padding=gnome.uiconsts.PAD_SMALL)
        # -------------------------------------------------
        hbox = gu.bHBox(self.config_box, gtk.FALSE)
        hbox.set_spacing(gnome.uiconsts.PAD)
        hbox.pack_start(gtk.GtkLabel(_("Try to keep question with range from")),
                        gtk.FALSE)
        self.g_notenamerange = widgets.nNotenameRange(self.m_exname,
                         "lowest_tone", "highest_tone")
        hbox.pack_start(self.g_notenamerange)
        
        # ------------------------------------------
        self._add_auto_new_question_gui(self.config_box)
        # ----------------------------------------------
        hbox = gu.bHBox(self.config_box, gtk.FALSE)
        hbox.set_spacing(gnome.uiconsts.PAD_SMALL)
        gu.bLabel(hbox, _("Input interface:"), gtk.FALSE)
        combo = gu.nCombo(self.m_exname, 'inputwidget', _("Buttons"),
                   inputwidgets.inputwidget_names)
        hbox.pack_start(combo, gtk.FALSE)
        combo.entry.connect('changed', 
               lambda entry, self=self, inputwidgets=inputwidgets:
               self.use_inputwidget(inputwidgets.name_to_inputwidget(
                           entry.get_text(), self.click_on_intervall)))
        # ------------ frame -------------------
        self.g_instrument_configurator = widgets.InstrumentConfigurator(
                                                             self.m_exname, 2)
        self.config_box.pack_start(self.g_instrument_configurator, gtk.FALSE)
        self.config_box.set_spacing(0)
        self.config_box.show_all()
        ##############
        # statistics #
        ##############
        self.setup_statisticsviewer(statisticsviewer.StatisticsViewer,
                                   _("Harmonic intervall"))
        self.m_key_bindings['repeat_melodic_ak'] = self.m_t.play_melodic
        self.select_inputwidget()
    def click_on_intervall(self, mouse_button, intervall, midi_int):
        if mouse_button == 1:
            if self.m_t.q_status == const.NO_QUESTION:
                self.g_flashbar.flash(_("Click 'New intervall' to begin."))
                return
            if not -17 < intervall < 17:
                self.g_flashbar.flash(_("Ignoring intervalls greater than decim."))
                return
            if self.m_t.q_status == const.SOLVED:
                if self.m_t.guess_answer(intervall):
                    self.g_flashbar.flash(_("Correct, but you have already solved this question"))
                else:
                    self.g_flashbar.flash(_("Wrong, but you have already solved this question"))
            else:
                if self.m_t.guess_answer(intervall):
                    self.g_flashbar.flash(_("Correct"))
                    self.g_new_intervall.set_sensitive(gtk.TRUE)
                else:
                    self.g_flashbar.flash(_("Wrong"))
                    if self.get_bool("config/auto_repeat_question_if_wrong_answer"):
                        self.m_t.play_question()
            self.g_input.set_first_note(self.m_t.m_tonika)
        elif mouse_button == 2:
            soundcard.play_note(self.get_int('config/preferred_instrument'),
                          4, 0, midi_int,
                          self.get_int('config/preferred_instrument_velocity'))
        elif mouse_button == 3 and self.m_t.q_status != const.NO_QUESTION:
            instr_low, instr_low_vel, instr_high, instr_high_vel = self.m_t.get_instrument_config(2)
            n2 = self.m_t.m_tonika + intervall
            m = soundcard.Track()
            m.set_patch(0, instr_low)
            m.set_patch(1, instr_high)
            m.start_note(0, self.m_t.m_tonika.semitone_pitch(), instr_low_vel)
            m.start_note(1, n2.semitone_pitch(), instr_low_vel)
            m.notelen_time(4)
            m.stop_note(0, self.m_t.m_tonika.semitone_pitch(), instr_high_vel)
            m.stop_note(1, n2.semitone_pitch(), instr_high_vel)
            soundcard.synth.play_track(m)
    def new_question(self, _o=None):
        g = self.m_t.new_question()
        if g == Teacher.OK: # new question, everything is OK
            self.g_repeat.set_sensitive(gtk.TRUE)
            self.g_repeat_melodic.set_sensitive(gtk.TRUE)
            self.g_input.set_first_note(self.m_t.m_tonika)
            qstr = self.m_t.play_question()
            self.g_new_intervall.set_sensitive(
              not self.get_bool('config/picky_on_new_question'))
        elif g == Teacher.ERR_PICKY:
                self.g_flashbar.flash(_("You have to solve this question first."))
        else:
            self.g_repeat.set_sensitive(gtk.FALSE)
            self.g_repeat_melodic.set_sensitive(gtk.FALSE)

            if g == Teacher.ERR_NO_INTERVALLS:
                self.g_flashbar.flash(
                      _("You have to select some intervalls to practise."))
            elif g == Teacher.ERR_NOTERANGE:
                self.g_flashbar.flash(_("Please configure with a wider range of notes"))
            else:
                raise "NEVER HERE"
    def on_end_practise(self):
        self.g_new_intervall.set_sensitive(gtk.TRUE)
        self.g_repeat.set_sensitive(gtk.FALSE)
        self.g_repeat_melodic.set_sensitive(gtk.FALSE)
        self.g_input.clear()
        self.m_t.end_practise()
