# GNU Solfege - eartraining for GNOME
# Copyright (C) 2001  Tom Cato Amundsen
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import re, string, operator, random, sys, os

from i18n import _
import gettext

blockstart_re = re.compile("\s*(\w+)\s*{")
assignment_re = re.compile("\s*(([a-zA-Z][\w_\-]*)(\([a-z]+\))*)\s*=\s*")
blockend_re = re.compile("\s*}")
whitespace_re = re.compile("\s+")
comment_re = re.compile("\s*#.*")
include_re = re.compile("\s*include\((.*?)\)")
tot_re = re.compile(r"""
            (\s*(?P<keyword>[a-zA-Z]+[\w_-]*(\(.*?\))?))|
        ###### xstring
          (\s*(
           \"\"\"(?P<xstring>(.*?))\"\"\"
           ))|
        ###### string
            (\s*((?P<i18npre>_\()?\"
            (?P<string>[^\"]*)
            \"(?P<i18npost>\))?))|
        ###### rational
            (\s*(?P<rat_num>\d+)/(?P<rat_den>\d+))|
        ###### integer
            (\s*(?P<integer>-?\d+))|
        ###### comment
            (\s*(?P<comment>\#.*?$))
            """, re.VERBOSE|re.MULTILINE|re.DOTALL)


class i18nString:
    def __init__(self, s):
        self.m_string = s

class DataparserException:
    def __init__(self, filename, lineno, badcode):
        self.filename = filename
        self.lineno = lineno
        self.badcode = badcode
    def __str__(self):
        return "dataparser-error, lineno: %i\nThis code is invalid: %s: %s" % (self.filename, self.lineno, self.badcode)

class UnknownVariableException(DataparserException):
    def __init__(self, filename, lineno, badcode):
        DataparserException.__init__(self, filename, lineno, badcode)
    def __str__(self):
        return "unknown variable in file %s, lineno: %i\n%s" % (self.filename, self.lineno, self.badcode)


class Dataparser:
    def __init__(self, globals={}, gd=[]):
        self.m_globals = globals.copy()
        # list with names of the variables that will be set from in every
        # block from the global namespace. #ugh explain better
        self.m_gd = gd
        self.m_blocks = []
        self.m_context = self.m_globals
    def start_block(self, blockname):
        if self.m_context != self.m_globals:
            self.raise_error_exception()
        self.m_blocks.append((blockname, {}))
        for v in self.m_gd:
            self.m_blocks[-1][1][v] = self.m_globals[v]
        self.m_context = self.m_blocks[-1][1]
    def end_block(self):
        if self.m_context == self.m_globals:
            self.raise_error_exception()
        self.m_context = self.m_globals
    def advance_counters(self, s):
        self._idx = self._idx + len(s)
        self._lineno = self._lineno + string.count(s, "\n")
    def get_assignment_rh(self):
        # ugh: it works, but it can be done better
        v = []
        while 1:
            m = tot_re.match(self._data[self._idx:])
            if m.group('integer'):
                self.advance_counters(m.group())
                v.append(('val', int(m.group('integer'))))
            elif m.group('xstring'):
                self.advance_counters(m.group())
                v.append(('val', m.group('xstring')))
            elif m.group('string'):
                self.advance_counters(m.group())
                if m.group('i18npre') and m.group('i18npost'):
                    v.append(('val', i18nString(m.group('string'))))
                else:
                    v.append(('val', m.group('string')))
            elif m.group('keyword'):
                self.advance_counters(m.group())
                if self.m_context.has_key(m.group('keyword')):
                    tmp = self.m_context[m.group('keyword')]
                elif self.m_context != self.m_globals and \
                        self.m_globals.has_key(m.group('keyword')):
                    tmp = self.m_globals[m.group('keyword')]
                else:
                    self.raise_error_exception(UnknownVariableException)
                v.append(('val', tmp))
            elif m.group('rat_num'):
                self.advance_counters(m.group())
                v.append(('val', (int(m.group('rat_num')),
                                  int(m.group('rat_den')))))
            elif m.group('comment'):
                # denne fanger kommentarer etter operator og fr frste val
                self.advance_counters(m.group())
                continue
            # denne fanger opp kommentar fr operator
            while 1:
                m = comment_re.match(self._data[self._idx:])
                if m:
                    self.advance_counters(m.group())
                else:
                    break
            
            self.eat_whitespace()
            operator_re = re.compile("\s*(,|%|\+)")
            m = operator_re.match(self._data[self._idx:])
            if m:
                self.advance_counters(m.group())
                c = m.group(1)
                if c == "%" or c == '+':
                    v.append((c,))
            else:
                break
        retval = self.evaluate(v)
        if len(retval) == 1:
            return retval[0][1]
        v = []
        for x in retval:
            v.append(x[1])
        return v
    def raise_error_exception(self, e=DataparserException):
        x = self._idx - 1
        y = self._idx
        for c in range(2):
            while x > 0 and self._data[x] != "\n":
                x = x - 1
            x = x - 1
        while y < len(self._data) and self._data[y] != "\n":
            y = y + 1
        raise e(self.m_filename, self._lineno, self._data[x+1:y])
    def evaluate(self, v):
        if len(v) < 3:
            return v
        for op, f in (("%", operator.__mod__),
                      ("+", operator.__add__)):
            x = 0
            while x + 2 < len(v) > 2:
                if (v[x][0], v[x+1][0], v[x+2][0]) == ('val', op, 'val'):
                    v = v[:x] + [('val', f(v[x][1], v[x+2][1]))] + v[x+3:]
                else:
                    x = x + 1
        return v
    def eat_whitespace(self):
        m = whitespace_re.match(self._data[self._idx:])
        if m:
            self.advance_counters(m.group())
    def parse_file(self, filename):
        self.m_filename = filename
        f = open(filename, "r")
        s = f.read()
        f.close()
        return self._parse(s)
    def _parse(self, s):
        self._lineno = 1
        self._idx = 0
        self._data = s
        _stack = []
        while 1:
            if self._idx >= len(self._data):
                if not _stack:
                    break
                self._idx, self._lineno, self._data, self.m_filename = _stack.pop()
            m = blockstart_re.match(self._data[self._idx:])
            if m:
                self.advance_counters(m.group())
                self.start_block(m.group(1))
                continue
            m = include_re.match(self._data[self._idx:])
            if m:
                self.advance_counters(m.group())
                _stack.append(self._idx, self._lineno, self._data, self.m_filename)
                self.m_filename = os.path.join(os.path.dirname(self.m_filename), m.group(1))
                f = open(self.m_filename, "r")
                self._data = f.read()
                f.close()
                self._idx = 0
                self._lineno = 1
                continue
            m = assignment_re.match(self._data[self._idx:])
            if m:
                self.advance_counters(m.group())
                d = self.get_assignment_rh()
                if isinstance(d, i18nString):
                    self.m_context[m.group(1)] = d.m_string
                    self.m_context["%s(%s)" % (m.group(1), gettext.lang[0])] = _(d.m_string)
                else:
                    self.m_context[m.group(1)] = d
                continue
            m = blockend_re.match(self._data[self._idx:])
            if m:
                self.advance_counters(m.group())
                self.end_block()
                continue
            m = comment_re.match(self._data[self._idx:])
            if m:
                self.advance_counters(m.group())
                continue
            m = tot_re.match(self._data[self._idx:])
            if m:
                s = self.get_assignment_rh()
                if type(s) != type(""):
                    self.raise_error_exception()
                self.m_context['music'] = s
                continue
            m = whitespace_re.match(self._data[self._idx:])
            if m:
                self.advance_counters(m.group())
                continue
            if len(self._data) != self._idx:
                self.raise_error_exception()


def get_translated_string(dict, name):
    for n in gettext.lang:
        s = string.lower(n)
        if dict.has_key("%s(%s)" % (name, s)):
            return dict["%s(%s)" % (name, s)]
    return dict[name]


def __f():
    if len(sys.argv) == 1:
        print "Give the file to parse as command line argument."
        sys.exit(-1)
    p = Dataparser({'dictation': 'dictation',
                      'progression': 'progression',
                      'harmony': 'harmony',
                      'sing-chord': 'sing-chord',
                      'chord': 'chord',
                      'id-by-name': 'id-by-name',
                      'satb': 'satb',
                      'horiz': 'horiz',
                      'vertic': 'vertic',
                      'yes': 1,
                      'no': 0,
                      'tempo': (160, 4)})
    import time
    t1 = time.clock()
    for x in range(1):
       p.parse_file(sys.argv[1])
    print time.clock() - t1
    #print p.m_blocks


if __name__ == "__main__":
    import i18n
    i18n.initialise("share/locale")
    from i18n import _

    __f()
    sys.exit()
    import profile, pstats
    profile.run("__f()", "profile.txt")
    s = pstats.Stats("profile.txt")
    s.strip_dirs().sort_stats('cumulative').print_stats()
