# GNU Solfege - eartraining for GNOME
# Copyright (C) 2000-2001  Tom Cato Amundsen
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""
REMEMBER: down is positive, up is negative.

All voices begin at the beginning of the staff. It is
not possible to split a voice in two like in Lilypond.

The parser will not handle fis and f in the same octave on one stem.

Rules:
 * The clef has to be set in the beginning of the staff, and it cannot
   be changed.
 * The clef has to be set in \staff, not in \addvoice
 * It can be different timesignatures in different staffs.
 * \key has to come before \time

The parser does not care if you have correct number of notes in a bar.
To get bar lines you have to insert a '|'
"""

import string, re
from requests import *
from duration import Duration
from musicalpitch import MusicalPitch
import engravers
from engravers import *
from soundcard.rat import Rat
import const
import soundcard
import musicdisplayer
import types
import utils

def musicalpitch_relative(first, second):
    """
    think:  \relative c'{ first second }
    Tritonus handling should be the same as GNU Lilypond
    0 up #  # dn 0
    b up 0  0 dn b
    b up #  # dn b
    F up B  B dn F

    I placed here instead of in MusicalPitch since it is only used
    once in parse_to_score_object and I don't think anyone need this
    in MusicalPitch.
    """
    assert isinstance(first, MusicalPitch)
    assert isinstance(second, MusicalPitch)
    n1 = MusicalPitch(second)
    n1.m_octave_i = first.m_octave_i
    n2 = MusicalPitch(n1)
    if n1 < first:
        n2.m_octave_i = n2.m_octave_i + 1
    else:
        n2.m_octave_i = n2.m_octave_i - 1
    # just make sure n1 < first < n2
    if n2 < n1:
       n1, n2 = n2, n1
    C = cmp(int(first) - int(n1), int(n2) - int(first))
    if C == 0:
        if first.steps() - n1.steps() > n2.steps() - first.steps():
            n2.m_octave_i = n2.m_octave_i + second.m_octave_i
            return n2
        else:
            n1.m_octave_i = n1.m_octave_i + second.m_octave_i
            return n1
    elif C < 0:
        n1.m_octave_i = n1.m_octave_i + second.m_octave_i
        return n1
    else:
        n2.m_octave_i = n2.m_octave_i + second.m_octave_i
        return n2



class TimeSignature:
    def __init__(self, num, den):
        self.m_num = num
        self.m_den = den


class ScoreColumn:
    """
    The class is used to remember how wide the different elements of
    a score column is.
    """
    def __init__(self):
        self.m_clef = 0
        self.m_keysignature = 0
        self.m_barline = 0
        self.m_timesignature = 0
        self.m_accidentals = 0
        self.m_music = 0
        self.m_leftshift = 0
        self.m_rightshift = 0
        
class Score:
    def help__del__(self):
        """
        This should be called when you are finished with the score object,
        to avoid memory leaks caused by circular reference.
        """
        for staff in self.m_staffs:
           for v in staff.m_voice_list:
               v.m_parent_staff = None
               del v
           del staff
        del self.m_staffs
    def __init__(self):
        self.m_timeposdict = {}
        self.m_staffs = []
    def announce_timepos(self, pos):
        if not self.m_timeposdict.has_key(pos):
            self.m_timeposdict[pos] = ScoreColumn()
    def get_engravers(self, fontsize):
        tv = self.m_timeposdict.keys()
        tv.sort()
        return self._generate_engravers(tv, fontsize)
    def get_first_engravers(self, fontsize):
        tv = self.m_timeposdict.keys()
        tv.sort()
        return self._generate_engravers([tv[0]], fontsize)
    def _generate_engravers(self, tv, fontsize):
        V = []
        xpos = 30
        max_xpos = xpos
        clef = None
        self.m_beam_list = []
        self.m_stem_list = []
        for staff in self.m_staffs:
            staff.accbarline(("c", "major"))
            key = ('c', 'major')
            se = []
            V.append(se)
            ###################
            beam = None
            for voice in staff.m_voice_list:
                pv = tv
                pv.sort()
                for pos in pv:
                    if not staff.m_coldict.has_key(pos):
                        continue
                    if not voice.m_coldict.has_key(pos):
                        continue
                    if staff.m_coldict[pos].m_clef:
                        clef = staff.m_coldict[pos].m_clef
                    ########
                    # stems
                    v = []
                    for n in voice.m_coldict[pos].m_music.values():
                        v.append(steps_to_ylinepos(n.m_pitch.steps(), clef))
                    if v and (voice.m_coldict[pos].m_music.values()[0].m_duration.m_nh > 1):
                        v.sort()
                        if voice.m_coldict[pos].m_beaminfo == 'start':
                            beam = BeamEngraver(fontsize)
                            self.m_beam_list.append(beam)
                            se.append(beam)
                        if beam and not voice.m_coldict[pos].m_beaminfo:
                            beam = None
                        se.append(StemEngraver(pos, fontsize, v, 
                                  voice.m_coldict[pos],
                                  beam != None))
                        self.m_stem_list.append(se[-1])
                        if beam:
                            beam.add_stem(se[-1])
            ###################
            pv = tv
            pv.sort()
            # this loop takes care of stuff that is decided on a per-staff-basis
            for pos in pv:
                if not staff.m_coldict.has_key(pos):
                    continue
                # clef
                if staff.m_coldict[pos].m_clef:
                    clef = staff.m_coldict[pos].m_clef
                    se.append(ClefEngraver(pos, fontsize,
                                           staff.m_coldict[pos].m_clef))
                    self.m_timeposdict[pos].m_clef \
                       = max(self.m_timeposdict[pos].m_clef, se[-1].get_width())
                #key signature
                if staff.m_coldict[pos].m_keysignature:
                    se.append(KeySignatureEngraver(pos, fontsize,
                                     staff.m_coldict[pos].m_keysignature, clef))
                    self.m_timeposdict[pos].m_keysignature \
                       = max(self.m_timeposdict[pos].m_keysignature,
                             se[-1].get_width())
                    key = staff.m_coldict[pos].m_keysignature
                    staff.accbarline(key)
                # barline
                if staff.m_coldict[pos].m_barline:
                    se.append(BarlineEngraver(pos, fontsize, "|"))
                    self.m_timeposdict[pos].m_barline \
                       = max(self.m_timeposdict[pos].m_barline, 
                             se[-1].get_width())
                    staff.accbarline(key)
                # time signature
                if staff.m_coldict[pos].m_timesignature:
                    se.append(TimeSignatureEngraver(pos, fontsize,
                       staff.m_coldict[pos].m_timesignature))
                    self.m_timeposdict[pos].m_timesignature \
                       = max(self.m_timeposdict[pos].m_timesignature,
                             se[-1].get_width())
                ##############
                # accidentals
                v = {}
                for voice in staff.m_voice_list:
                    if not voice.m_coldict.has_key(pos):
                        continue
                    for musicalpitch in voice.m_coldict[pos].m_music.keys():
                        m = voice.m_coldict[pos].m_music[musicalpitch].m_pitch
                        e = staff.needed_accidental(m)
                        if e is not None:
                            v[steps_to_ylinepos(m.steps(), clef)] = e
                se.append(AccidentalsEngraver(pos, fontsize, v))
                self.m_timeposdict[pos].m_accidentals \
                   = max(self.m_timeposdict[pos].m_accidentals,
                        se[-1].get_width())
                ################################
                # xshift noteheads that need it
                for voice in staff.m_voice_list:
                    if not voice.m_coldict.has_key(pos):
                        continue
                    nd = {}
                    for musicalpitch in voice.m_coldict[pos].m_music.keys():
                        nd[steps_to_ylinepos(voice.m_coldict[pos].m_music[musicalpitch].m_pitch.steps(), clef)] = voice.m_coldict[pos].m_music[musicalpitch]
                    v = nd.keys()
                    v.sort()
                    for n in range(1, len(v)):
                        if nd[v[n]].m_pitch.steps()+1 == nd[v[n-1]].m_pitch.steps() and (not nd[v[n-1]].m_shift):
                            if voice.m_coldict[pos].m_stemdir == const.UP:
                                nd[v[n]].m_shift = 1
                                self.m_timeposdict[pos].m_rightshift = 1
                            else:
                                nd[v[n]].m_shift = -1
                                self.m_timeposdict[pos].m_leftshift = 1
                ####################
                # notehead and rest
                for voice in staff.m_voice_list:
                    if not voice.m_coldict.has_key(pos):
                        continue
                    for k in voice.m_coldict[pos].m_music.keys():
                        m = voice.m_coldict[pos].m_music[k]
                        if m.m_duration.m_nh < 2:
                            head = const.NOTEHEAD_0
                        elif m.m_duration.m_nh > 2:
                            head = const.NOTEHEAD_2
                        else:
                            head = const.NOTEHEAD_1
                        ylinepos = steps_to_ylinepos(m.m_pitch.steps(), clef)
                        se.append(NoteheadEngraver(pos, fontsize, m.m_shift,
                             ylinepos, head, m.m_duration.m_dots, 
                             m.m_pitch.semitone_pitch(),
                             voice.m_coldict[pos]))
                        self.m_timeposdict[pos].m_music \
                           = max(self.m_timeposdict[pos].m_music, se[-1].get_width())
                    l = self.m_timeposdict[pos].m_leftshift
                    r = self.m_timeposdict[pos].m_rightshift
                    self.m_timeposdict[pos].m_music \
                       = max(self.m_timeposdict[pos].m_music, (1+l+r)*dimentions[fontsize].xshift)
                    #
                    if voice.m_coldict[pos].m_rest:
                        se.append(RestEngraver(pos, fontsize, 0,
                              voice.m_coldict[pos].m_rest.m_duration))
                        self.m_timeposdict[pos].m_music \
                           = max(self.m_timeposdict[pos].m_music, se[-1].get_width())
                ####################################
                # Find out if wee need ledger lines
                up = 0
                down = 0
                for voice in staff.m_voice_list:
                    if not voice.m_coldict.has_key(pos):
                        continue
                    for k in voice.m_coldict[pos].m_music.keys():
                        ypos = voice.m_coldict[pos].m_music[k].m_pitch.steps()
                        ypos = steps_to_ylinepos(ypos, clef)
                        if up > ypos < -5:
                            up = ypos
                        if down < ypos > 5:
                            down = ypos
                ###############################
                # Create ledger line engravers
                if staff.m_coldict.has_key(pos):
                    if up:
                        up = - up / 2 - 2
                    else:
                        up = 0
                    if down:
                        down = down / 2 - 2
                    else:
                        down = 0
                    e = LedgerLineEngraver(pos, fontsize, up, down)
                    se.append(e)
            for voice in staff.m_voice_list:
                for tie in voice.m_ties:
                    sh1 = tie[3].m_shift
                    sh2 = tie[4].m_shift
                    se.append(TieEngraver(fontsize, tie[0], tie[1],
                                          sh1, sh2,
                                          steps_to_ylinepos(tie[2][0], clef)))
        #########################################
        for e in self.m_beam_list:
            e.do_layout()
        xv = {}
        p = 0
        pv = self.m_timeposdict.keys()
        pv.sort()
        for pos in pv:
            xv[pos] = ScoreColumn()
            xv[pos].m_clef = p
            p = p + self.m_timeposdict[pos].m_clef
            xv[pos].m_keysignature = p
            p = p + self.m_timeposdict[pos].m_keysignature
            xv[pos].m_barline = p
            p = p + self.m_timeposdict[pos].m_barline
            xv[pos].m_timesignature = p
            p = p + self.m_timeposdict[pos].m_timesignature
            xv[pos].m_accidentals = p
            p = p + self.m_timeposdict[pos].m_accidentals
            if self.m_timeposdict[pos].m_leftshift:
                p = p + 10
            xv[pos].m_music = p
            p = p + self.m_timeposdict[pos].m_music
        for ev in V:
            for e in ev:
                e.set_xpos(xv)
        # we delete it because it is not used any more, and to help avoid
        # circular references
        del self.m_beam_list
        for e in self.m_stem_list:
            e.xshift_stem()
        del self.m_stem_list
        return V
    def add_staff(self):
        staff = Staff(self)
        self.m_staffs.append(staff)
        return staff
    def display(self):
        print self.m_timeposdict
    def get_midi_events(self, start=None, end=None):
        kv = self.m_timeposdict.keys()
        kv.sort()
        if start is None and end is not None:
            return self._generate_midi_events(
                     filter(lambda i, end=end: i < end, kv))
        elif start is not None and end is None:
            return self._generate_midi_events(
                     filter(lambda i, start=start: i >= start, kv))
        elif start is None and end is None:
            return self._generate_midi_events(kv)
        else:
            assert start is not None and end is not None
            return self._generate_midi_events(
               filter(lambda i, start=start, end=end: start <= i < end, kv))
    def get_first_beat_midi_events(self):
        kv = self.m_timeposdict.keys()
        kv.sort()
        return self._generate_midi_events([kv[0]])
    def get_last_beat_midi_events(self):
        kv = self.m_timeposdict.keys()
        kv.sort()
        return self._generate_midi_events([kv[-1]])
    def get_midi_events_with_channel(self, chan):
        """Same as get_midi_events, but all voices will use channel chan.
        This is used by the rhythm exercise.
        """
        kv = self.m_timeposdict.keys()
        kv.sort()
        track_list = []
        for staff in self.m_staffs:
            for voice in staff.m_voice_list:
               track_list.append(voice.generate_track_for_voice(staff, kv, chan))
        return track_list

    def _generate_midi_events(self, kv):
        """
        kv is a list of rat.Rat that represent the timepos of columns
        
        Return a list of tracks, one track for each voice.
        """
        track_list = []
        chan = 0
        for staff in self.m_staffs:
            for voice in staff.m_voice_list:
               track_list.append(voice.generate_track_for_voice(staff, kv, chan))
               #chan = chan + 1
        return track_list

def semitonepitch_to_ylinepos(i, clef):
    n = MusicalPitch(i)
    return steps_to_ylinepos(n.steps(), clef)

def steps_to_ylinepos(steps, clef):
    return {'treble': 13, #treble and violin are two names for the same clef
            'violin': 13,
            'subbass': -1,
            'bass': 1,
            'baritone': 3,
            'varbaritone': 3, 
            'tenor': 5,
            'alto': 7,
            'mezzosoprano': 9,
            'soprano': 11,
            'french': 15}[clef] - steps
    raise "unknown clef"


class ColObj:
    def __init__(self):
        self.m_timesignature = None
        self.m_keysignature = None
        self.m_clef = None
        self.m_barline = None
        self.m_ledger_up = 0
        self.m_ledger_down = 0


#count_staff = 0
class Staff:
#    def __del__(self):
#        global count_staff
#        count_staff = count_staff - 1
#        print "del count_staff", count_staff
    def __init__(self, score):
#        global count_staff
#        count_staff = count_staff + 1
#        print "init count_staff", count_staff
        self.m_score = score
        self.m_coldict = {}
        self.m_voice_list = []
    def add_voice(self):
        voice = Voice(self)
        self.m_voice_list.append(voice)
        return voice
    def accbarline(self, key):
        """Do accidental stuff on barline.
        """
        self.m_i = {}
        for step in range(MusicalPitch.LOWEST_STEPS, MusicalPitch.HIGHEST_STEPS+1):
            self.m_i[step] = 0
        for a in utils.key_to_accidentals(key):
            n = MusicalPitch(a)
            for oct in range(-4, 7):
                n.m_octave_i = oct
                if n.semitone_pitch() < 128:
                    if a[-2:] == 'es':
                        self.m_i[n.steps()] = self.m_i[n.steps()] - 1
                    else:
                        self.m_i[n.steps()] = self.m_i[n.steps()] + 1
    def needed_accidental(self, m):
        if m.m_accidental_i != self.m_i[m.steps()]:
            self.m_i[m.steps()] = m.m_accidental_i
            return m.m_accidental_i
    def barline(self, pos):
        #assert not self.m_coldict.has_key(pos)
        self.announce_timepos(pos)
        self.m_coldict[pos] = ColObj()
        self.m_coldict[pos].m_barline = 1
    def add_timesignature(self, t, pos):
        self.announce_timepos(pos)
        self.m_coldict[pos].m_timesignature = t
    def add_keysignature(self, pos, key):
        self.announce_timepos(pos)
        self.m_coldict[pos].m_keysignature = key
    def add_clef(self, pos, clef):
        self.announce_timepos(pos)
        self.m_coldict[pos].m_clef = clef
    def announce_timepos(self, pos):
        if not self.m_coldict.has_key(pos):
            self.m_coldict[pos] = ColObj()
        self.m_score.announce_timepos(pos)
    def maybe_ledger_lines(self, pos, ypos):
        if not self.m_coldict.has_key(pos):
            self.m_coldict[pos] = ColObj()
        if self.m_coldict[pos].m_ledger_up > ypos < -5:
            self.m_coldict[pos].m_ledger_up = ypos
        if self.m_coldict[pos].m_ledger_down < ypos > 5:
            self.m_coldict[pos].m_ledger_down = ypos


class VoiceColObj:
    def __init__(self):
        self.m_rest = None
        # the key for this dictionary is semitonepitch for the notehead
        self.m_music = {}
        self.m_beaminfo = None
        self.m_duration = None

class Voice:
    def __init__(self, parent_staff):
        self.m_parent_staff = parent_staff
        self.m_coldict = {}
        self._tmp_tie = {}
        self.m_ties = []
        self.m_beams = []
        self.m_is_beaming = None
    def add_notehead(self, pos, music, stemdir):
        assert self.m_coldict.has_key(pos)
        key = (music.m_pitch.steps(), music.m_pitch.m_accidental_i)
        if self.m_coldict[pos].m_music.has_key(key):
            print "warning, adding the same notehead twice", key
        if self.m_coldict[pos].m_duration == None:
            self.m_coldict[pos].m_duration = music.m_duration
        elif self.m_coldict[pos].m_duration != music.m_duration:
            print "mpd: warning: All noteheads on the same stem has to have the same length"
        if self.m_is_beaming and self.m_coldict[pos].m_duration.get_rat_value() >= Rat(1, 4):
            print "mpd: warning: beamed stems has to be 1/8-note or shorter. Ignoring invalid beam request."
            self.m_is_beaming = None
        music.m_shift = 0
        self.m_coldict[pos].m_music[key] = music
        self.m_coldict[pos].m_stemdir = stemdir
    def add_rest(self, rest, pos):
        assert self.m_coldict.has_key(pos)
        self.m_coldict[pos].m_rest = rest
    def announce_timepos(self, pos):
        if not self.m_coldict.has_key(pos):
            self.m_coldict[pos] = VoiceColObj()
            if self.m_is_beaming:
                self.m_coldict[pos].m_beaminfo = 'continue'
        self.m_parent_staff.announce_timepos(pos)
    def start_beam(self, pos):
        if self.m_is_beaming:
            print "mpd-warning: we are already beaming, ignoring start_beam request"
            return
        self.m_is_beaming = 1
        self.m_coldict[pos].m_beaminfo = 'start'
    def end_beam(self):
        if not self.m_is_beaming:
            print "mpd-warning: we are not beaming, ignoring stop_beam request"
        self.m_is_beaming = 0
    def do_tie_end(self, pos1, pos2):
        for p in self.m_coldict[pos1].m_music.keys():
            if self.m_coldict[pos2].m_music.has_key(p):
                self.m_ties.append((pos1, pos2, p,
                       # need these to generate midi track
                       self.m_coldict[pos1].m_music[p],
                       self.m_coldict[pos2].m_music[p]))
    def generate_track_for_voice(self, staff, kv, chan):
        # first we find the id()'s of music to tie
        tie_from_v = []
        tie_to_v = []
        for t in self.m_ties:
           tie_from_v.append(id(t[3]))
           tie_to_v.append(id(t[4]))
        ################
        D = {}
        i = 2
        musictimepos = Rat(0, 1)
        last_timepos = kv[0]
        id_D = {}
        for idx in range(len(kv)):
            coltimepos = kv[idx]
            musictimepos = musictimepos + (coltimepos - last_timepos)
            if self.m_coldict.has_key(coltimepos):
                for n in self.m_coldict[coltimepos].m_music.values():
                    #ugh this is ugly
                    if id(n) in tie_from_v and (id(n) not in tie_to_v):
                        id_D[n.m_pitch.semitone_pitch()] = i
                        if not D.has_key(musictimepos):
                            D[musictimepos] = []
                        D[musictimepos].append((i, const.START_NOTE, n.m_pitch.semitone_pitch()))
                    elif id(n) in tie_to_v:
                        stop_pos = musictimepos + n.m_duration.get_rat_value()
                        if not D.has_key(stop_pos):
                            D[stop_pos] = []
                        D[stop_pos].append((id_D[n.m_pitch.semitone_pitch()], 
                               const.STOP_NOTE, n.m_pitch.semitone_pitch()))
                    else:
                        if not D.has_key(musictimepos):
                            D[musictimepos] = []
                        D[musictimepos].append((i, const.START_NOTE, n.m_pitch.semitone_pitch()))
                        
                        stop_pos = musictimepos + n.m_duration.get_rat_value()
                        if not D.has_key(stop_pos):
                            D[stop_pos] = []
                        D[stop_pos].append((i, const.STOP_NOTE, n.m_pitch.semitone_pitch()))
                        i = i + 1
            last_timepos = coltimepos
        return self.__gen_midi_last_step(D, chan)
    def __gen_midi_last_step(self, D, chan):
        keys = D.keys()
        keys.sort()
        prev_time = Rat(0)
        ms = soundcard.Track()
        for k in keys:
            delta = None
            if k != Rat(0, 1):
                delta = 1/(k-prev_time)
            prev_time = k
            for e in D[k]:
                if e[1] == const.START_NOTE:
                    if delta:
                        ms.notelen_time(delta)
                    ms.start_note(chan, e[2], const.MF)
                elif e[1] == const.STOP_NOTE:
                    if delta:
                        ms.notelen_time(delta)
                    ms.stop_note(chan, e[2], const.MF)
                delta = None
        return ms
    
def parse_to_score_object(music):
        lexer = Lexer(music)
        relative_mode = None
        relto=None
        transpose_pitch = None
        TOPLEVEL = 1#'toplevel'
        NOTES = 2#'notes'
        START_OF_CHORD = 3#'start-of-chord'
        CHORD = 4#'chord'
        context = TOPLEVEL
        score = Score()
        chord_duration = None
        tie_is_in_the_air = 0
        x = len(lexer.m_string)
        while lexer.m_idx < x:
            toc, toc_data = lexer.get()
            if toc == Lexer.STAFF:
                assert context == TOPLEVEL
                cur_staff = score.add_staff()
                cur_voice = cur_staff.add_voice()
                stem_dir = const.BOTH
                relative_mode = None
                pos = Rat(0)
                last_pos = pos
                cur_staff.announce_timepos(pos)
                cur_staff.add_clef(pos, "violin")
            elif toc == Lexer.VOICE:
                stemdir = const.BOTH
                relative_mode = None
                pos = Rat(0)
                cur_voice = cur_staff.add_voice()
            elif toc == Lexer.RELATIVE:
                assert not relative_mode
                relative_mode = 1
                relto = toc_data
            elif toc == Lexer.TRANSPOSE:
                transpose_pitch = toc_data
            elif toc == Lexer.TIME:
                cur_staff.add_timesignature(toc_data, pos)
            elif toc == Lexer.KEY:
                p = MusicalPitch(toc_data[0])
                if transpose_pitch:
                    p.transpose_by_musicalpitch(transpose_pitch)
                k = (p.notename(), toc_data[1])
                cur_staff.announce_timepos(pos)
                cur_staff.add_keysignature(pos, k)
            elif toc == Lexer.CLEF:
                cur_staff.announce_timepos(pos)
                cur_staff.add_clef(pos, toc_data)
            elif toc == '|':
                cur_staff.barline(pos)
            elif toc == '{':
                if (context == TOPLEVEL):
                    context = NOTES
                    if not cur_staff.m_coldict[Rat(0, 1)].m_keysignature:
                        if transpose_pitch:
                            k = (transpose_pitch.notename(), 'major')
                        else:
                            k = ('c', 'major')
                        cur_staff.add_keysignature(Rat(0, 1), k)
                else:
                    raise "parse error {"
            elif toc == '<':
                if context == NOTES:
                    context = START_OF_CHORD
                else:
                    raise "'<' not allowed here"
            elif toc == '>':
                if context == CHORD:
                    if tie_is_in_the_air:
                        tie_is_in_the_air = 0
                        cur_voice.do_tie_end(last_pos, pos)
                    last_pos = pos
                    pos = pos + chord_duration.get_rat_value()
                    chord_duration = None
                    relto = relto_backup; relto_backup = None
                    context = NOTES
                else:
                    raise "error parsing '>'"
            elif toc == '}':
                if context == NOTES:
                    context = TOPLEVEL
                else:
                    raise "error parsing '}'"
            elif toc == '[':
                cur_voice.announce_timepos(pos)
                cur_voice.start_beam(pos)
            elif toc == ']':
                cur_voice.end_beam()
                cur_voice.announce_timepos(pos)
            elif toc == '~':
                tie_is_in_the_air = 1
            elif toc == Lexer.NOTE and (context in [NOTES, CHORD, START_OF_CHORD]):
                if context in [NOTES, START_OF_CHORD]:
                    cur_voice.announce_timepos(pos)
                if relative_mode:
                    toc_data.m_pitch = musicalpitch_relative(
                                              relto, toc_data.m_pitch)
                    relto = MusicalPitch(toc_data.m_pitch)
                if transpose_pitch:
                    toc_data.transpose(transpose_pitch)
                cur_voice.add_notehead(pos, toc_data, stem_dir)
                if context == NOTES:
                    if tie_is_in_the_air:
                        cur_voice.do_tie_end(last_pos, pos)
                        tie_is_in_the_air = 0
                    last_pos = pos
                    pos = pos + toc_data.m_duration.get_rat_value()
                if context == START_OF_CHORD:
                    relto_backup = relto
                    chord_duration = toc_data.m_duration
                    context = CHORD
            elif toc == Lexer.REST and context == NOTES:
                cur_voice.announce_timepos(pos)
                cur_voice.add_rest(toc_data, pos)
                last_pos = pos
                pos = pos + toc_data.m_duration.get_rat_value()
            elif toc == Lexer.STEMDIR:
                stem_dir = toc_data
            else:
                print "*%s*" % toc, type(toc), lexer.m_idx
                print "music:", music[lexer.m_idx-4:]
                raise "Parse error", toc
        return score


class Lexer:
    STAFF = 1
    VOICE = 2
    CLEF = 3
    STEMDIR = 4
    TRANSPOSE = 5
    TIME = 6
    KEY = 7
    NOTE = 8
    REST = 9
    RELATIVE = 10
    def __init__(self, s):
        self.m_string = string.strip(s)
        self.m_notelen = Duration(4, 0)
        self.re_staff = re.compile(r"\s*\\staff")
        self.re_voice = re.compile(r"\s*\\addvoice")
        self.re_clef = re.compile(r"\s*\\clef\s+(.*?);")
        self.re_stem_updown = re.compile(r"\s*(\\stem)(Up|Down|Both)\s+")
        self.re_relative = re.compile(r"\s*\\relative\s+(([a-zA-Z]+)([',]*))")
        self.re_transpose = re.compile(r"\s*\\transpose\s+(([a-zA-Z]+)([',]*))")
        self.re_rest = re.compile(r"\s*(r)([\d]*)(\.*)")
        #ugh we are a little more strict than Lilypond, since ~ has to
        # be before ]
        #don't use named rexec if we don't need it.
        self.re_melodic = re.compile(r"""(?x)
                            \s*
                             ((?P<notename>[a-zA-Z]+)
                             (?P<octave>[',]*))
                             (?P<len>[\d]*)
                             (?P<dots>\.*)""")
        #self.re_melodic = re.compile(r"""\s*(([a-zA-Z]+)([',]*))([\d]*)(\.*)""")
        self.re_tempo = re.compile(r"\s*\\tempo\s+(\d+)\s*=+s*(\d+);")
        self.re_time = re.compile(r"\s*\\time\s+(\d+)\s*/\s*(\d+);")
        self.re_key = re.compile(r"\s*\\key\s+([a-z]+)\s*\\(major|minor);")
        self.m_idx = 0
    def get(self):
        m = self.re_rest.match(self.m_string, self.m_idx)
        if m:
            self.m_idx = self.m_idx + len(m.group())
            notelen = m.group(2)
            if notelen:
                notelen = int(notelen)
            else:
                notelen = 0
            numdots = len(m.group(3))
            r = RestRequest(notelen, numdots)
            if not r.m_duration:
                r.m_duration = self.m_notelen
                if numdots:
                    self.m_notelen.m_dots = numdots
            else:
                self.m_notelen = r.m_duration
            return self.REST, r

        m = self.re_melodic.match(self.m_string, self.m_idx)
        if m:
            self.m_idx = self.m_idx + len(m.group())
            notename, IGN1, IGN2, notelen, dots = m.groups()
            if notelen:
                notelen = int(notelen)
            else:
                notelen = 0
            numdots = len(dots)
            n = Music(notename, notelen, numdots)
            if not n.m_duration:
                n.m_duration = self.m_notelen
                if numdots:
                    self.m_notelen.m_dots = numdots
            else:
                self.m_notelen = n.m_duration
            return self.NOTE, n
        m = self.re_staff.match(self.m_string, self.m_idx)
        if m:
            self.m_idx = self.m_idx + len(m.group())
            return self.STAFF, None
        m = self.re_voice.match(self.m_string, self.m_idx)
        if m:
            self.m_idx = self.m_idx + len(m.group())
            return self.VOICE, None
        m = self.re_relative.match(self.m_string, self.m_idx)
        if m:
            self.m_idx = self.m_idx + len(m.group())
            return self.RELATIVE, MusicalPitch(m.group(1))
        m = self.re_clef.match(self.m_string, self.m_idx)
        if m:
            self.m_idx = self.m_idx + len(m.group())
            return self.CLEF, m.group(1)
        m = self.re_stem_updown.match(self.m_string, self.m_idx)
        if m:
            self.m_idx = self.m_idx + len(m.group())
            d = [const.UP, const.DOWN, const.BOTH][['Up', 'Down', 'Both'].index(m.group(2))]
            return self.STEMDIR, d
        m = self.re_transpose.match(self.m_string, self.m_idx)
        if m:
            self.m_idx = self.m_idx + len(m.group())
            return self.TRANSPOSE, MusicalPitch(m.group(1))
        m = self.re_time.match(self.m_string, self.m_idx)
        if m:
            self.m_idx = self.m_idx + len(m.group())
            return self.TIME, TimeSignature(int(m.group(1)), int(m.group(2)))
        m = self.re_key.match(self.m_string, self.m_idx)
        if m:
            self.m_idx = self.m_idx + len(m.group())
            return self.KEY, (m.group(1), m.group(2))
        while self.m_string[self.m_idx] in [' ', '\n', '\t']:
            self.m_idx = self.m_idx + 1
        self.m_idx = self.m_idx + 1
        return self.m_string[self.m_idx-1], None
