/** Laser 0.9 Unix source code *******************************/
/**                                                         **/
/**                        Laser 0.9 (Unix)                 **/
/**                                                         **/
/** Copyright (C) 2000 Domenico Dato (dato@di.unipi.it)     **/
/** Ported from original code by Mickey McMurray (Lopantu)  **/
/**                                                         **/
/**     You are not allowed to distribute this software     **/
/**     commercially.                                       **/
/*************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include "LibUnix.h"

/** Standard Unix/X #includes ********************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>


#include "z80.h"

static byte *XBuf, XPal[4], XPal0;
static Image Img;

/** Various X-related variables ******************************/
static Display *Dsp;
static Window Wnd;
static Colormap DefaultCMap;
static GC DefaultGC;
static unsigned long White, Black;
char *Title = "Laser 0.9 (Unix version)";	/* Window/command line title */
int UseSHM = 1;			/* 1 = use MITSHM            */
int Verbose = 1;		/* Debug ON or OFF           */
byte ExitNow = 0;		/* 1: Exit the emulator      */



byte IType;
int WIDTH;
int HEIGHT;
int refresh = 0;
byte bValue = 0x81;
byte bShiftValue = 0;
byte bBitmap = 0;
byte bBitmapPrevious = 0;
int horizontaltester = 0;

typedef struct
  {
    char *romName;
    int address;
    int romSize;
    byte romCount;
  }
tagRomList;

tagRomList lrescueList[] =
{
  {"roms/lrescue/lrescue.1", 0x0000, 0x0800, 6},
  {"roms/lrescue/lrescue.2", 0x0800, 0x0800, 0},
  {"roms/lrescue/lrescue.3", 0x1000, 0x0800, 0},
  {"roms/lrescue/lrescue.4", 0x1800, 0x0800, 0},
  {"roms/lrescue/lrescue.5", 0x4000, 0x0800, 0},
  {"roms/lrescue/lrescue.6", 0x4800, 0x0800, 0}
};

tagRomList invadersList[] =
{
  {"roms/invaders/invaders.h", 0x0000, 0x0800, 4},
  {"roms/invaders/invaders.g", 0x0800, 0x0800, 0},
  {"roms/invaders/invaders.f", 0x1000, 0x0800, 0},
  {"roms/invaders/invaders.e", 0x1800, 0x0800, 0}
};

tagRomList invadersdxList[] =
{
  {"roms/invdelux/invdelux.h", 0x0000, 0x0800, 5},
  {"roms/invdelux/invdelux.g", 0x0800, 0x0800, 0},
  {"roms/invdelux/invdelux.f", 0x1000, 0x0800, 0},
  {"roms/invdelux/invdelux.e", 0x1800, 0x0800, 0},
  {"roms/invdelux/invdelux.d", 0x4000, 0x0800, 0}
};

tagRomList galaxywarsList[] =
{
  {"roms/galxwars/galxwars.0", 0x0000, 0x0400, 6},
  {"roms/galxwars/galxwars.1", 0x0400, 0x0400, 0},
  {"roms/galxwars/galxwars.2", 0x0800, 0x0400, 0},
  {"roms/galxwars/galxwars.3", 0x0C00, 0x0400, 0},
  {"roms/galxwars/galxwars.4", 0x4000, 0x0400, 0},
  {"roms/galxwars/galxwars.5", 0x4400, 0x0400, 0}
};

tagRomList spaceattack2List[] =
{
  {"roms/spaceatt/spaceatt.h", 0x0000, 0x0800, 4},
  {"roms/spaceatt/spaceatt.g", 0x0800, 0x0800, 0},
  {"roms/spaceatt/spaceatt.f", 0x1000, 0x0800, 0},
  {"roms/spaceatt/spaceatt.e", 0x1800, 0x0800, 0},
};

tagRomList invrevengeList[] =
{
  {"roms/invrvnge/invrvnge.h", 0x0000, 0x0800, 4},
  {"roms/invrvnge/invrvnge.g", 0x0800, 0x0800, 0},
  {"roms/invrvnge/invrvnge.f", 0x1000, 0x0800, 0},
  {"roms/invrvnge/invrvnge.e", 0x1800, 0x0800, 0},
};

tagRomList destearthList[] =
{
  {"roms/destinat/36_h.bin", 0x0000, 0x0800, 7},
  {"roms/destinat/35_g.bin", 0x0800, 0x0800, 0},
  {"roms/destinat/34_f.bin", 0x1000, 0x0800, 0},
  {"roms/destinat/33_e.bin", 0x1800, 0x0800, 0},
  {"roms/destinat/32_d.bin", 0x4000, 0x0800, 0},
  {"roms/destinat/31_c.bin", 0x4800, 0x0800, 0},
  {"roms/destinat/42_b.bin", 0x5000, 0x0800, 0},
};

tagRomList earthinvList[] =
{
  {"roms/earthinv/earthinv.h", 0x0000, 0x0800, 4},
  {"roms/earthinv/earthinv.g", 0x0800, 0x0800, 0},
  {"roms/earthinv/earthinv.f", 0x1000, 0x0800, 0},
  {"roms/earthinv/earthinv.e", 0x1800, 0x0800, 0},
};

tagRomList boothillList[] =
{
  {"roms/boothill/Romh.cpu", 0x0000, 0x0800, 4},
  {"roms/boothill/Romg.cpu", 0x0800, 0x0800, 0},
  {"roms/boothill/Romf.cpu", 0x1000, 0x0800, 0},
  {"roms/boothill/Rome.cpu", 0x1800, 0x0800, 0},
};

tagRomList sicvList[] =
{
  {"roms/sicv/Cv17.bin", 0x0000, 0x0800, 4},
  {"roms/sicv/Cv18.bin", 0x0800, 0x0800, 0},
  {"roms/sicv/Cv19.bin", 0x1000, 0x0800, 0},
  {"roms/sicv/Cv20.bin", 0x1800, 0x0800, 0},
};

tagRomList cosmicmoList[] =
{
  {"roms/cosmicmo/cosmicmo.1", 0x0000, 0x0400, 7},
  {"roms/cosmicmo/cosmicmo.2", 0x0400, 0x0400, 0},
  {"roms/cosmicmo/cosmicmo.3", 0x0800, 0x0400, 0},
  {"roms/cosmicmo/cosmicmo.4", 0x0c00, 0x0400, 0},
  {"roms/cosmicmo/cosmicmo.5", 0x4000, 0x0400, 0},
  {"roms/cosmicmo/cosmicmo.6", 0x4400, 0x0400, 0},
  {"roms/cosmicmo/cosmicmo.7", 0x4800, 0x0400, 0},
};

tagRomList laserList[] =
{
  {"roms/laser/1.u36", 0x0000, 0x0800, 4},
  {"roms/laser/2.u35", 0x0800, 0x0800, 0},
  {"roms/laser/3.u34", 0x1000, 0x0800, 0},
  {"roms/laser/4.u33", 0x1800, 0x0800, 0},
};

tagRomList spcekingList[] =
{
  {"roms/spceking/invaders.h", 0x0000, 0x0800, 4},
  {"roms/spceking/spcekng2", 0x0800, 0x0800, 0},
  {"roms/spceking/spcekng3", 0x1000, 0x0800, 0},
  {"roms/spceking/spcekng4", 0x1800, 0x0800, 0},
};

tagRomList checkmateList[] =
{
  {"roms/checkmate/checkmat.h", 0x0000, 0x0400, 4},
  {"roms/checkmate/checkmat.g", 0x0400, 0x0400, 0},
  {"roms/checkmate/checkmat.f", 0x0800, 0x0400, 0},
  {"roms/checkmate/checkmat.e", 0x0c00, 0x0400, 0},
};

tagRomList spcewr3List[] =
{
  {"roms/spcewr3/ic36.bin", 0x0000, 0x0800, 5},
  {"roms/spcewr3/ic35.bin", 0x0800, 0x0800, 0},
  {"roms/spcewr3/ic34.bin", 0x1000, 0x0800, 0},
  {"roms/spcewr3/ic33.bin", 0x1800, 0x0800, 0},
  {"roms/spcewr3/ic32.bin", 0x4000, 0x0800, 0},

};

struct
  {
    char *gameName;
    tagRomList *romList;
  }

gameList[] =
{
  {
    "Space Invaders", invadersList
  }
  ,
  {
    "Space Invaders Part II", invadersdxList
  }
  ,
  {
    "Lunar Rescue", lrescueList
  }
  ,
  {
    "Galaxy Wars", galaxywarsList
  }
  ,
  {
    "Space Attack II", spaceattack2List
  }
  ,
  {
    "Destination Earth", destearthList
  }
  ,
  {
    "Invaders Revenge", invrevengeList
  }
  ,
  {
    "Super Earth Invasion", earthinvList
  }
  ,
  {
    "Boot Hill", boothillList
  }
  ,
  {
    "Space Invaders (Taito)", sicvList
  }
  ,
  {
    "Cosmic Monsters", cosmicmoList
  }
  ,
  {
    "Space Laser", laserList
  }
  ,
  {
    "Space King", spcekingList
  }
  ,
  {
    "Check Mate", checkmateList
  }
  ,
  {
    "Space War 3", spcewr3List
  }
  ,
};

byte gameCount = sizeof (gameList) / sizeof (gameList[0]);

/** This function is called on signals ***********************/
static void 
OnBreak (int Arg)
{
  ExitNow = 1;
  signal (Arg, OnBreak);
}

/** InitMachine() ********************************************/
/** Allocate resources needed by Unix/X-dependent code.     **/
/*************************************************************/
int 
InitMachine (void)
{
  Screen *Scr;
  int J, I;

  /* Open display */
  if (Verbose)
    printf ("Initializing Unix/X drivers:\n  Opening display...");
  if (!(Dsp = XOpenDisplay (NULL)))
    {
      if (Verbose)
	puts ("FAILED");
      return (0);
    }

  /* Set internal variables */
  Scr = DefaultScreenOfDisplay (Dsp);
  White = WhitePixelOfScreen (Scr);
  Black = BlackPixelOfScreen (Scr);
  DefaultGC = DefaultGCOfScreen (Scr);
  DefaultCMap = DefaultColormapOfScreen (Scr);

  /* Reset the palette */
  for (XPal0 = Black, J = 0; J < 16; J++)
    XPal[J] = Black;

  /* Keys do not autorepeat */
  XAutoRepeatOff (Dsp);

  /* Initialize LibUnix toolkit */
  if (Verbose)
    printf ("OK\n  Initializing LibUnix...");
  if (!InitLibUnix (Dsp))
    {
      if (Verbose)
	puts ("FAILED");
      return (0);
    }

  /* Create a window */
  if (Verbose)
    printf ("OK\n  Opening window...");
  if (!(Wnd = X11Window (Title, WIDTH, HEIGHT)))
    {
      if (Verbose)
	puts ("FAILED");
      return (0);
    }

  /* Create an image */
  if (Verbose)
    printf ("OK\n  Allocating screen buffer...");
  J = X11NewImage (&Img, WIDTH, HEIGHT, UseSHM ? USE_SHM : 0);
  if (Verbose)
    puts (J ? "OK" : "FAILED");
  if (!J)
    return (0);
  XBuf = Img.Data;

  /* Catch all signals */
  //signal(SIGHUP,OnBreak);signal(SIGINT,OnBreak);
  //signal(SIGQUIT,OnBreak);signal(SIGTERM,OnBreak);

  return (1);
}

/** TrashMachine() *******************************************/
/** Deallocate all resources taken by InitMachine().        **/
/*************************************************************/
void 
TrashMachine (void)
{
  unsigned long L;
  int J;

  if (Verbose)
    printf ("Shutting down...\n");

  if (Dsp)
    {
      for (XPal[0] = XPal0, J = 0; J < 4; J++)
	if (XPal[J] != Black)
	  {
	    L = XPal[J];
	    XFreeColors (Dsp, DefaultCMap, &L, 1, 0);
	  }

      X11FreeImage (&Img);
      XAutoRepeatOn (Dsp);
      XCloseDisplay (Dsp);
    }
}


/** SetColor() ***********************************************/
/** Allocate a given color.                                 **/
/*************************************************************/
void 
SetColor (byte N, byte R, byte G, byte B)
{
  XColor Color;
  unsigned long L;

  if (XPal[N] != Black)
    {
      L = XPal[N];
      XFreeColors (Dsp, DefaultCMap, &L, 1, 0);
    }

  if (!R && !G && !B)
    XPal[N] = Black;
  else
    {
      Color.flags = DoRed | DoGreen | DoBlue;
      Color.red = (int) R << 8;
      Color.green = (int) G << 8;
      Color.blue = (int) B << 8;
      XPal[N] = XAllocColor (Dsp, DefaultCMap, &Color) ? Color.pixel : Black;
    }

  XPal0 = XPal[0];
}

/** PutImage() ***********************************************/
/** Put an image on the screen.                             **/
/*************************************************************/
void 
PutImage (void)
{
  X11PutImage (Wnd, &Img, 0, 0, 0, 0, WIDTH, HEIGHT);
}


short 
getGameNum (char *parm)
{
  word temp;
  if (parm[0] < '0' || parm[0] > '9')
    return -1;
  temp = atoi (parm);
  if (temp < 1 || temp > gameCount)
    return -1;
  return temp;
}

void 
gameNmi ()
{
  IFlag = 1;
  IType = (IType == 1 ? 2 : 1);
  refresh++;
  if (refresh == 10)
    {
      refresh = 0;
      PutImage ();
    }
  if (ExitNow)
    {
      TrashMachine ();
      exit (1);
    }
}

void 
ShowHelp ()
{
  printf ("Laser 0.9.  Based on SinVader by Mike Green, and Spaders by Tony Toole\n");
  printf ("            DOS version by Lopantu\n");
  printf ("            Ported to Unix by Domenico Dato (dato@di.unipi.it)\n");
  printf ("Usage:     laser gamename [options]\n");
  printf ("Options:   -?,-h        Show this help text\n");
  printf ("\n");
  printf ("Games supported: \n");
  printf ("\n");
  printf ("Game                   | Gamename\n");
  printf ("-----------------------+----------\n");
  printf ("Space Invaders         | invaders\n");
  printf ("Space Invaders Part II | invdelux\n");
  printf ("Lunar Rescue           | lrescue\n");
  printf ("Galaxy Wars            | galxwars\n");
  printf ("Space Attack II        | spaceatt\n");
  printf ("Destination Earth      | destinat\n");
  printf ("Invaders Revenge       | invrvnge\n");
  printf ("Super Earth Invasion   | earthinv\n");
  printf ("Boot Hill (not working)| boothill\n");
  printf ("Space Invaders (Taito) | sicv\n");
  printf ("Cosmic Monsters        | cosmicmo\n");
  printf ("Space Laser            | laser\n");
  printf ("Space King             | spceking\n");
  printf ("CheckMate (not working)| checkmate\n");
  printf ("Space War 3            | spcewr3\n");
  exit (0);
}

int 
FindOption (char *option, int p, int h, int u, int argc, char **argv)
{
  static int t[] =
  {0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0};
  int i;
  char *c;

  if (argc > 128)
    argc = 128;			/* maximum this function can handle is 128 */

  /*
   * if p = 1 and h = 0 will find option and return decimal value of
   * argv[i+1], if h = 1 will read it as hex.
   * if p = 0 then it will return index of option in argv[], 0 not found
   * if u = 1 will return index of first occurance of untouched option
   */

  if (u)			/* find first untouched element */
    {
      for (i = 1; i < argc; i++)
	{
	  if (!t[i])		/* 0 indicates untouched */
	    return i;
	}
      return 0;
    }

  if (p)			/* find option and return integer value following it */
    {
      for (i = 1; i < argc; i++)
	{
	  if (strcmp (argv[i], option) == 0)	/* found */
	    {
	      if (i >= argc)	/* bounds! */
		return 0;
	      t[i + 1] = t[i] = 1;	/* touched */
	      if (!h)
		return atoi (argv[i + 1]);
	      else
		return strtoul (argv[i + 1], &c, 16);
	    }
	}
      return 0;			/* no match */
    }
  else
    /* find option and return position */
    {
      for (i = 1; i < argc; i++)
	{
	  if (strcmp (argv[i], option) == 0)
	    {
	      t[i] = 1;
	      return i;		/* found! return position */
	    }
	}
      return 0;
    }
  return 0;
}


void 
putpixel (byte * bits, int x, int y, int color)
{
  bits += 10 + x + (248 * y);
  *bits = XPal[color];
}


void setMode (byte theMode);
void pal (int index, char r, char g, char b);
byte *RAM;			// 64K RAM for the game.

reg R;
byte port2 = 0, port4 = 0, oldPort4 = 0;

int 
main (int argc, char *argv[])
{
  FILE *ap;
  int i, J;
  int gameNum = 0;
  struct
    {
      byte R, G, B;
    }
  Palette[4] =
  {
    {
      0x00, 0x00, 0x00
    }
    ,
    {
      0xff, 0x00, 0x00
    }
    ,
    {
      0x00, 0xff, 0x00
    }
    ,
    {
      0xff, 0xff, 0xff
    }
  };

  /* check command line for no options, process help */
  if (argc == 1)		/* show help */
    ShowHelp ();
  if (FindOption ("-?", 0, 0, 0, argc, argv) ||
      FindOption ("-h", 0, 0, 0, argc, argv))
    ShowHelp ();


  if (argc > 1)
    {
      gameNum = getGameNum (argv[1]);
      gameNum--;
    }

  if (FindOption ("invaders", 0, 0, 0, argc, argv))
    {
      gameNum = 0;
    }

  else if (FindOption ("invdelux", 0, 0, 0, argc, argv))
    {
      gameNum = 1;
    }

  else if (FindOption ("lrescue", 0, 0, 0, argc, argv))
    {
      gameNum = 2;
    }

  else if (FindOption ("galxwars", 0, 0, 0, argc, argv))
    {
      gameNum = 3;
    }

  else if (FindOption ("spaceatt", 0, 0, 0, argc, argv))
    {
      gameNum = 4;
    }

  else if (FindOption ("destinat", 0, 0, 0, argc, argv))
    {
      gameNum = 5;
    }

  else if (FindOption ("invrvnge", 0, 0, 0, argc, argv))
    {
      gameNum = 6;
    }

  else if (FindOption ("earthinv", 0, 0, 0, argc, argv))
    {
      gameNum = 7;
    }

  else if (FindOption ("boothill", 0, 0, 0, argc, argv))
    {
      horizontaltester = 1;
      gameNum = 8;
    }

  else if (FindOption ("sicv", 0, 0, 0, argc, argv))
    {
      gameNum = 9;
    }

  else if (FindOption ("cosmicmo", 0, 0, 0, argc, argv))
    {
      gameNum = 10;
    }

  else if (FindOption ("laser", 0, 0, 0, argc, argv))
    {
      gameNum = 11;
    }

  else if (FindOption ("spceking", 0, 0, 0, argc, argv))
    {
      gameNum = 12;
    }

  else if (FindOption ("checkmate", 0, 0, 0, argc, argv))
    {
      horizontaltester = 1;
      gameNum = 13;
    }

  else if (FindOption ("spcewr3", 0, 0, 0, argc, argv))
    {
      gameNum = 14;
    }

  else
    {
      gameNum = 0;
    }

  if (!(RAM = malloc (sizeof (char) * 0x10000)))
    {
      printf ("Too low on memory to continue.\n");
      exit (1);
    }

  // Load the ROM data into the z80 memory location.
  for (i = 0; i < gameList[gameNum].romList[0].romCount; i++)
    {
      if (!(ap = fopen (gameList[gameNum].romList[i].romName, "rb")))
	{
	  printf ("Unable to open '%s'.\n", gameList[gameNum].romList[i].romName);
	  exit (1);
	}
      if (fread (RAM + gameList[gameNum].romList[i].address, 1, gameList[gameNum].romList[i].romSize, ap) != gameList[gameNum].romList[i].romSize)
	{
	  printf ("Error reading from file '%s'.\n", gameList[gameNum].romList[i].romName);
	  exit (1);
	}
      fclose (ap);
    }

  WIDTH = 248;
  HEIGHT = 256;
  if (!InitMachine ())
    return (1);

  /* Set up the palette */
  for (J = 0; J < 4; J++)
    SetColor (J, Palette[J].R, Palette[J].G, Palette[J].B);

  ResetZ80 (&R);		// Reset the CPU to its proper values

  Z80 (R);			// Actually starts the emulation

}


// Filler functions for the Z80 library.
// byte and word are defined in z80.h

byte 
M_RDMEM (word addr)
{

  return RAM[addr];
}

void 
M_WRMEM (word A, byte V)
{
  if (A < 0x2000)
    return;			// Trying to write to the game ROM...

  RAM[A] = V;			// Writting to system memory...

  if (A >= 0x2400 && A < 0x4400)	// Video RAM...

    {
      // Graphics consist of 3 color bands, White, Red and Green..
      int y, x, i, color = 3;
      A -= 0x2400;
      x = A / 32;
      y = A - (x * 32);

      if (horizontaltester)
	{
	  if (y < 8)
	    color = 3;
	  if (y > 23 && y < 28)
	    color = 3;
	  if (x < 16 && y < 2)
	    color = 3;		// White 'ships left' number

	  if (x > 135 && y < 2)
	    color = 3;		// White 'Credits' word.

	}
      else
	{
	  if (y < 8)
	    color = 2;
	  if (y > 23 && y < 28)
	    color = 1;
	  if (x < 16 && y < 2)
	    color = 3;		// White 'ships left' number

	  if (x > 135 && y < 2)
	    color = 3;		// White 'Credits' word.

	}

      for (i = 0; i < 8; i++)
	if (horizontaltester)
	  putpixel (XBuf, (y * 8 + i), x, (V & (1 << i)) ? color : 0);
	else
	  putpixel (XBuf, x, (256 - (y * 8 + i)), (V & (1 << i)) ? color : 0);
    }
}

word 
Interrupt ()
{
  return (IType == 1 ? 0x0008 : 0x0010);
}

byte 
DoIn (byte Port)
{
  XEvent E;
  word J;
  word wValue;
  byte lbValue;

  switch (Port)
    {
    case 1:
      if (XCheckWindowEvent (Dsp, Wnd, KeyPressMask | KeyReleaseMask, &E))
	{
	  J = XLookupKeysym ((XKeyEvent *) & E, 0);
	  if (E.type == KeyPress)
	    {
	      switch (J)
		{
		case XK_Escape:
		  ExitNow = 1;
		  break;
		case XK_1:
		  bValue |= 0x04;
		  break;	// 1 player

		case XK_2:
		  bValue |= 0x02;
		  break;	// 2 players

		case XK_3:
		  bValue &= ~0x01;
		  break;	// give us a credit;

		case XK_Left:
		  bValue |= 0x20;
		  break;
		case XK_Right:
		  bValue |= 0x40;
		  break;
		case XK_space:
		  bValue |= 0x10;
		  break;
		}
	      return bValue;
	    }

	  if (E.type == KeyRelease)
	    {
	      bValue = 0x81;
	      return bValue;
	    }
	}
      return bValue;

    case 3:
      wValue = (bBitmap << 8) | bBitmapPrevious;
      wValue = wValue << bShiftValue;
      lbValue = (byte) (wValue >> 8);
      return (lbValue);

    default:
      return 0;
    }
}

void 
DoOut (byte Port, byte Value)
{
  switch (Port)
    {
    case 2:
      bShiftValue = Value;
      break;
    case 4:
      bBitmapPrevious = bBitmap;
      bBitmap = Value;
      break;
    }
}

void 
Patch (reg * R)
{
}
