/*       chat.c        */

/* Peer-to-Peer Chat Demo Program v1.01  */
/*  (c) 1992 Racal-Datacom, Inc.         */
/*  Frank Consiglio                      */

/*  JED     Aug-26-92     Beta release.  Add handle to ResetInterLanAT. */
/*  JED     Jan-11-93     Change rev. to v1.01 due to new driver v3.01. */

/* This program links to the services of an embedded ILAN_AT/XT driver. */

/* The basic services provided are as follows:						 */
/*																					 */
/* 	RegisterEthertype                                         */
/* 	SendPacket                                                */
/* 	GetEthernetAddress                                        */
/* 	InitInterLanAT                                            */
/* 	ResetInterLanAT                                           */
/* 	ReleaseInterLanAT                                         */
/*																					 */

/* This program demonstrates the use of the basic services in a simple 		*/
/* communication program. This is run on two or more wrkstatns after the 	*/
/* INTERLAN_AT board is installed in the system.                       		*/

/** Standard Microsoft include files *******************************/

#define     LINT_ARGS
#include    <conio.h>
#include    <stdio.h>
#include    <dos.h>
#include    <stdlib.h>
#include    <bios.h>
#include    <memory.h>
#include    <string.h>
#include    <ctype.h>


/** Racal_Datacom Toolkit include file *****************************/

#include    <at_ddk.h>


/** Constants ******************************************************/

#define MAC_HEADER          14
#define BUFSIZE             64     /* small buffer for this example */
#define ETHERTYPE            2
#define VALID_DATA           1
#define SUCCESS              0     /* Return zero means "OK" */

#define INSPECT              0
#define COMPLETE             1

#define BEL                  7
#define ESC                 27
#define BS                   8
#define CR                  13
#define LF                  10

#define ILAN_DIAG_TYPE  0x7030     /* Racal Diag PacketType (byte-swapped) */


/** Typedefs *******************************************************/

typedef	unsigned int  UINT;


/** Macros *********************************************************/

#define hexval(cp)	((*cp <= '9') ? (*cp++ - '0') : (*cp++ - 'A' + 10))
	/* side effect: *cp is incremented */


/** Prototypes *****************************************************/

extern int InitInterLanAT(void);
extern int RegisterEthertype(unsigned int etype, unsigned int far *handleptr);
extern int SendPacket(byte far * packet, unsigned int length);
extern void GetEthernetAddress(byte far * addr_buf);
extern void ResetInterLanAT(unsigned int handle);
extern void ReleaseInterLanAT(void);

int ReceiveProcedure(RcvBlock far *Rcv);




void box(int,int,int,int);
void clr_scrn(void);
void clr_wndo(int,int,int);

int  edt_line(int,int,char *);

void locate(int,int,int,char *);
void putfstr(char *);
void rst_cursor(void);
void sav_cursor(void);
void scroll_up(int,int,int,int);
void scroll_wndo(int,int,int,int);
void set_cursor(int,int);


void bld_wndo(void);           /* builds dialogue boxes */
void box(int,int,int,int);     /* creates a box on the stdio */
void clr_scrn(void);           /* clears stdout */
void clr_wndo(int,int,int);	 /* clears given window */
void receiver(RcvBlock far *); /* background processing for driver */
void get_peer(byte *);         /* requests user for the peer node */
void driver_fail(void);        /* terminates program on call to driver failure */

void print_error(RETURN_CODE);


/** Structures *****************************************************/

typedef struct pbuf
 {                                     /* actual packet buffer */
  byte  dstadd[6];                     /* destination address */
  byte  srcadd[6];                     /* source address */
  UINT  typfld;                        /* type field */
  char  data[BUFSIZE];                 /* an array of bytes */
 } PacketBuf;


/** Global variables ************************************************/

PacketBuf  * RxBuf;                   /* Packet buffer for receiving */
PacketBuf  * TxBuf;                   /* Packet buffer for transmitting */

RcvBlock   * RxBlock;                 /* structure filled by driver */

int  flag = NULL;                     /* data-is-present/valid flag */



/**************************************************************************
*                                                                         *
*                                main                                     *
*                                                                         *
**************************************************************************/

void main(void)
{
   RETURN_CODE r_code;         /* return code for every Packet Driver call */
   UINT handle = -1;              
   UINT i = 0;
   int ch;
   int data_length = BUFSIZE;
   int send_length;


   /* Allocate the receive block structure, set its buffer length field. */

   RxBlock = (RcvBlock *) malloc(sizeof(RcvBlock));
   RxBlock->buffer_len = sizeof(PacketBuf);

   /* Allocate a packet buffer for frame reception. */
   /* Allocate a packet buffer for frame transmission. */

   RxBuf = (PacketBuf *) malloc(sizeof(PacketBuf));
   TxBuf = (PacketBuf *) malloc(sizeof(PacketBuf));

   TxBuf->typfld = ILAN_DIAG_TYPE;     /* Set transmit ethertype */


   /* Try and initialize the InterLan AT ethernet adapter and driver. */

   if ((r_code = InitInterLanAT()) != SUCCESS)
   {
       print_error(r_code);
       driver_fail();
   }

   /* Initialize the screen, formatting it for the chat program. */

   clr_scrn();
   printf(" Peer-to-Peer Chat Demo Program v1.01  (c) 1992 Racal-Datacom, Inc.");
   bld_wndo();


   /* Request that packets with our ethertype be directed by the driver */
   /*  to the C routine "ReceiveProcedure". */

   r_code = RegisterEthertype(ILAN_DIAG_TYPE, (UINT far *) &handle);

   if (r_code != SUCCESS)
   {
      print_error(r_code);
      driver_fail();
   }

   /* Have the driver copy our card's ethernet address into our Tx buffer. */

   GetEthernetAddress((byte far *) TxBuf->srcadd);


   /* Print the ethernet address to the screen for the user's benefit. */
   set_cursor(4, 2);
   printf("This node's address: %02X%02X%02X%02X%02X%02X\n",
        (unsigned int) *(TxBuf->srcadd+0), (unsigned int) *(TxBuf->srcadd+1),
        (unsigned int) *(TxBuf->srcadd+2), (unsigned int) *(TxBuf->srcadd+3),
        (unsigned int) *(TxBuf->srcadd+4), (unsigned int) *(TxBuf->srcadd+5)
         );


   /* Ask the user for our peer node's ethernet address. */

   get_peer(TxBuf->dstadd);


   /* Prompt the user for message. */

   set_cursor(7, 2);
   printf("Enter message to send to peer node:");
   set_cursor(8, 2);


   /* Loop getting messages to send, sending them, and printing */
   /*  messages received.  The [Esc] key press breaks the loop. */

   for (i = 0; ; )
   {
      if (flag == VALID_DATA)
      {
         /* Display new message from the peer. */

		   clr_wndo(22, 2, 65);
         printf("%s", RxBuf->data);
         flag = NULL;
         clr_wndo(8, 2, 64);
         i = 0;
      }
      else
      {
         if (kbhit())            /* If the user pressed a key... */
         {
            /* Read the keystroke. */
            ch = getch();        

            switch(ch)
            {
               case ESC:
                  /* Leave this application. */
                  break;

               case 0x00:                 
               case 0x0E:
                  /* 00h or 0Eh indicate that the next character */
                  /*  is an extended scan code. Read it. */
                  ch = getch();
                  break;

               case BS:                   
                  if (i != 0)
                  {
                     /* Erase previous character */
                     printf("%c %c", BS, BS);
                     i -= 1;
                     break;
                  }
                  else
                     break;

               case CR:
               case LF:
                  /* [Enter]: transmit what the user typed. */
                  if (i != 0)
                  {
                     TxBuf->data[i] = NULL;
                     send_length = i + MAC_HEADER;
                     r_code = SendPacket((byte far *) TxBuf,
                                  (send_length < 60) ? (60) : (send_length));
                     clr_wndo(8, 2, i);
                     i = 0;
                     break;
                  }
                  else
                     break;

               default:
                  if (i == data_length)
                  {
                     putch(ch);
                     TxBuf->data[i] = NULL;
                     r_code = SendPacket((byte far *) TxBuf, i);
                     i = 0;
                  }
                  else
                  {
                     putch(ch);
                     TxBuf->data[i] = (byte) ch;
                     i++;
                  }

            }   /* end of switch */

            if (ch == ESC)
               break;

         }   /* end of kbhit */

      }

   }   /* end for */

   if (r_code != SUCCESS)
   {
      print_error(r_code);
      driver_fail();
   }

   /* Clean up and terminate */

   ReleaseInterLanAT();                /* Release interrupt, etc. */

   clr_scrn();
   printf("Peer-to-Peer Chat Demo Program v1.01  (c) 1993 Racal-Datacom, Inc.\n");
   printf("Goodbye\n");
   exit(0);
}


/************************************************************************
*
* name     -  ReceiveProcedure
*                                                                       
* synopsis -  int ReceiveProcedure(RcvBlock far * Rcv);
*                                                                       
* function -  Every time the driver receives a packet with our registered
*             ethertype, this routine is called.  If we want the packet,
*             we write the size and address of the contiguous buffer into
*             which we want the packet placed, into the RcvBlock passed to
*             us on the stack by the driver.  If we accepted the packet
*             on the first call, this routine will be called a second
*             time once the packet has been completely copied into the 
*             buffer.
*                                                                       
*************************************************************************/

#pragma check_stack(off)      /* This is NOT optional! */

int ReceiveProcedure(RcvBlock far * Rcv)
{
   if (Rcv->function_flag == INSPECT)
   {			 
      /* We are being called by the driver to ask if we want the
       * current frame.  The frame contains our ethertype.  In this
       * example program, we accept ANY frame with our ethertype.
       */

      /* Set the buffer length field. */
      Rcv->buffer_len = (UINT) sizeof(struct pbuf);

      /* Give the driver a buffer into which it can place the packet. */
      Rcv->buffer = (char far *) RxBuf;
   }
   else if (Rcv->function_flag == COMPLETE)
   {
	   /* The driver has finished copying the packet into buffer. */

      /* Our foreground program watches this 'flag' variable and
       * displays data when the VALID_DATA bit(s) are set.  It then
       * clears the VALID_DATA bit(s).
       */
      flag |= VALID_DATA;
   }

   return SUCCESS;
}

/**************************************************************************
*                                                                         *
*  NOTE:                                                                  *
*                                                                         *
*  Everything below this header is irrelevent to a network application.   *
*                                                                         *
*                                                                         *
**************************************************************************/


   /* Video_io definitions	*/

#define  VIDEO_IO    0x10  /* video BIOS interrupt number */

   /* Video_io commands	*/

#define SET_MODE     0     /* set video mode	*/
#define CURSOR_TYPE  1     /* set cursor type	*/
#define SET_CURSOR   2     /* set cursor coordinates	*/
#define GET_COORD    3     /* get cursor coordinates	*/
#define SCROLL       7     /* scroll screen	*/
#define READ_AC      8     /* read attribute/character	*/
#define WRITE_AC     9     /* write attribute/character	*/
#define WRITE_TTY   14     /* write character	*/

	/*	Character attributes	*/

#define UNDERLINE 0x01     /* normal character underlined	*/
#define NORMAL    0x07     /* white character / black background	*/
#define INTENSITY 0x0F     /* high intensity	*/
#define REVERSE   0x70     /* reverse video	*/

   /* Graphics Character Definitions */

#define UL_CORNER 0xC9
#define UR_CORNER 0xBB
#define LL_CORNER 0xC8
#define LR_CORNER 0xBC
#define V_BAR     0xBA
#define H_BAR     0xCD
#define T_RIGHT   0xCC
#define T_LEFT    0xB9
#define T_DOWN    0xCB
#define T_UP      0xCA

#define DEC       0
#define HEX       1
#define STR       2

#define FOREVER  for(;;)

/************************************************************************
*                                                                       *
* name     -  build_window                                              *
*                                                                       *
* synopsis -  void bld_wndo(void);					                         *
*                                                                       *
* function -  Clears screen window starting at coordinates (row,column) *
*             and continuing for (length) characters.  The cursor is    *
*             left positioned at the beginning of the window. Writes    *
*				  In the box titles and exits										 *
*				  Calls: box																				 *	
* returns -  nothing                                                    *
*                                                                       *
************************************************************************/

void bld_wndo(void)
{
	box(2, 1, 10, 78);
	set_cursor(3, 2);
	printf("DIALOG BOX");
	box(18, 1, 24, 78);
	set_cursor(19, 2);
	printf("IN BOX");
}

/*****************************************************************************
* get_peer()
* Promts user to type in the peer node address
* Calls:  MACRO hexval
* Fills dst_addr with the 12 nibbles of address 
*
***************************************************************************/
void get_peer(byte *addrptr)

{
	char	ad_buf[12], *cptr;
	int 	ch, j, byte_count;
	int  ad_length;
	byte	*bptr;
   int i = 0;

	set_cursor(5, 2);
	printf("Enter the 12 hex digits for peer address: ");
	set_cursor(6, 2);
	ad_length=12;

	FOREVER                    /* Fills buf with hex chars from stdio. */
	{
    while ( i < ad_length)
      {
      ch = getch();
      if (isxdigit(ch))
        {
        ad_buf[i]= ch;
        i ++;
        putch(ch);
        }
       else
        {
        switch(ch)
         {
         case ESC:
          continue;
               
         case 0x00:          /* special key hit */
         case 0x0E:
           {
           ch=getch();      /* throw away next key */
           continue;
           }
         case BS:
          if (i != 0)
           {
           printf("%c %c",BS,BS);
           i -= 1;
           continue;
           }
          else
           continue;
         }

       }  
     	}      /* end of while */
		/* fill the dstaddr array  with 12 nibbles */

	 for (j=0, cptr = ad_buf; j<6; j++)
      {
     	 *addrptr = hexval(cptr) << 4;			
     	 *addrptr++ |= (hexval(cptr) & 0x0F);
      }
    break;
   }
}
/*****************************************************************************
* driver_fail()
* Terminates program upon receiving a failure code from driver
*
* Calls: clr_wndo
* 
*
***************************************************************************/
void  driver_fail(void)
{
   clr_wndo(9,2,64);
   printf("Call to Packet Driver service failed.  Exiting program.");
   set_cursor(15,1);
   exit(0);
}

/************************************************************************
*                                                                       *
* NAME     -  clear screen                                              *
*                                                                       *
* SYNOPSIS -  void clr_scrn()                                           *
*                                                                       *
* FUNCTION -  Clears the display screen                                 *
*                                                                       *
* RETURNS  -  Nothing                                                   *
*                                                                       *
************************************************************************/

void clr_scrn()
{
   clr_wndo(0,0,2000);
}

/************************************************************************
*                                                                       *
* name     -  box                                                       *
*                                                                       *
* synopsis -  void box(row1,col1,row2,col2);                            *
*               int  row1, col1;  co-ordinates of upper-left corner     *
*               int  row2, col2;  co-ordinates of lower-right corner    *
*                                                                       *
* function -  Displays a box defined by upper-left and lower-right      *
*             corners on screen using graphics characters.              *
*                                                                       *
* returns  -  nothing                                                   *
*                                                                       *
************************************************************************/

void box(row1, col1, row2, col2)

   int     row1, col1, row2, col2;

{
   register int row, col;

   set_cursor(row1, col1);
   putchar(UL_CORNER);
   for (col = col1 + 1;  col < col2;  col++)
      putchar(H_BAR);
   putchar(UR_CORNER);
   for (row = row1 + 1;  row < row2;  row++)
   {
      set_cursor(row, col1);
      putchar(V_BAR);
      set_cursor(row, col2);
      putchar(V_BAR);
   }
   set_cursor(row2,col1);
   putchar(LL_CORNER);
   for (col = col1 + 1;  col < col2;  col++)
      putchar(H_BAR);
   putchar(LR_CORNER);
}

/************************************************************************
*                                                                       *
* name     -  clear window                                              *
*                                                                       *
* synopsis -  void clr_wndo(row, col, length);                          *
*               int   row, col      start row and column                *
*               int   length        number of characters in the window  *
*                                                                       *
* function -  Clears screen window starting at coordinates (row,column) *
*             and continuing for (length) characters.  The cursor is    *
*             left positioned at the beginning of the window.           *
*                                                                       *
* returns -  nothing                                                    *
*                                                                       *
************************************************************************/

static union REGS rv = {0};            /* register structure for int86() */

void clr_wndo(row, col, length)

    int         row, col, length;

{
   set_cursor(row, col);                /*  position cursor  */

   rv.h.ah = WRITE_AC;                 /*  function    */
   rv.h.bh = 0;                        /*  page number */
   rv.x.cx = length;
   rv.h.al = ' ';                      /*  space is clear character */
   rv.h.bl = NORMAL;                   /*  attribute */
   int86(VIDEO_IO, &rv, &rv);

   set_cursor(row, col);
}

/************************************************************************
*                                                                       *
* NAME     -  set cursor                                                *
*                                                                       *
* SYNOPSIS -  void set_cursor(row,col)                                  *
*               int     row,col ;       screen coordinates              *
*                                                                       *
* FUNCTION -  Places cursor at the coordinates specified.               *
*                                                                       *
* RETURNS  -  Nothing                                                   *
*                                                                       *
************************************************************************/

void set_cursor(row, col)

    int         row, col;

{
   rv.h.ah = SET_CURSOR;               /* function    */
   rv.h.bh = 0;                        /* page number */
   rv.h.dh = row;                      /* row  */
   rv.h.dl = col;                      /* column  */
   int86(VIDEO_IO, &rv, &rv);
}

/************************************************************************
*                                                                       *
* NAME     -  print_error                                               *
*                                                                       *
* SYNOPSIS -  void print_error(r_code)                                  *
*                                                                       *
* FUNCTION -  Prints error code returned from PKT DRVR call.            *
*                                                                       *
* RETURNS  -  Nothing                                                   *
*                                                                       *
************************************************************************/


void print_error(RETURN_CODE error)
{
   clr_wndo(8, 2, 64);
   printf("Error code returned =  %02X", error);
}

