#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <ctype.h>

#include "pktdate.lng"

#if defined(__MSDOS__) || defined(MSDOS)
#if defined(__TURBOC__)
#ifdef __HUGE__
#define HUGEPTR huge
#else
#error "You must use the HUGE memory model (use -mh on command line)!"
#endif
#elif defined(__DJGPP__)
/* DJGPP is a flat 32 bit compiler, no problems here */
#define HUGEPTR
#else
/* The DOS verions must be able to work with malloc'ed blocks larger than
   64KBytes. This is assured when using huge pointers - those are pointers that
   are always normalized to an offset < 16. Please give your compiler the
   appropriate command line parameters to use such a memory model. Then
   you may remove the following error statement.
*/
#error "You need a HUGE memory model on DOS - don't know how to do this with this compiler."
#endif
#else
#define HUGEPTR
#endif

#define ID "Pktdate Rev. 1.5"

/* ============================== Prototypes ============================ */

static int init(int argc, char **argv, int require_arguments);
void log(int loglevel, const char *fmt, ...);
static int process_pkt(const char *filename);

#define LOG_ERROR   1
#define LOG_WARNING 2
#define LOG_NORMAL  3
#define LOG_DEBUG   4

const char *log_chars = " ?! d";

typedef unsigned long flag_t;  /* for at least 32 bit flags */

                                /* constants that describe errors    */
#define FTSC_FLAWY  1           /* FTSC field has correctable errors */
#define FTSC_BROKEN 2           /* FTSC field can't even be parsed   */
#define FTSC_SPILLED_ZERO 4     /* FTSC field too long -> msg looks  */
                                /* like it has a zero length subject */
#define UNPLAUSIBLE_DATE 8      /* date not within defined tolerance */
                                /* boundaries                        */
#define FTSC_SEADOG 16          /* Seadog style string in the FTSC   */
                                /* date field                        */
#define UD_SQUISH_BUG 32        /* Unplausible date: 7 bit overflow  */
#define UD_YEAR_DISP  64        /* Only year is wrong                */
#define FTSC_TS_BROKEN 128      /* Only timestamp broken, date is OK */
#define UD_FIELD_UNINITIALISED 256 /* date field uninitialised       */
#define FTSC_NO_ZERO 512        /* no \0 in date field               */
#define FTSC_TOO_SHORT 1024     /* FTSC Date field was too short     */
#define PKT_HEADER_DATE 2048    /* PKT header date is unplausible    */

#define FATAL_ERROR ((unsigned long)-1L)

/* =========================== Global variables ========================= */

struct _config
{
    int fix : 1;               /* -c : try to correct errors       */
    int fixseadog : 1;         /* -S : fix Seadog style date field */
    int qqq : 1;               /* -q : accept Fastecho QQQ files   */
    int keep: 1;               /* -k : keep copies of broken files */
    const char *logfilename;   /* -L : log file name               */
    int   maxlog;              /* -l : maximum log level           */
    unsigned long futureTol;   /* -f : Tolerance for date in days  */
                               /*      in future direction         */
    unsigned long pastTol;     /* -p : Tolerance into the past     */
    unsigned long sfutureTol;  /* -F : for speculative error corr. */
    unsigned long spastTol;    /* -P : for speculative error corr. */

    FILE *logfile;
} config;


/* ============================= Main Program =========================== */

int main(int argc, char **argv)
{
    int i;
    int rv=0, rvt;

    for (i = init (argc, argv, 1); i < argc; i++)
    {
        rvt = process_pkt(argv[i]);
        if (rvt > rv)
            rv = rvt;
    }

    return rv;
}


/* ======================== Process a PKT file ========================== */

struct msginpkt
{
    long absoffset;      /* ftell() result at message start in pkt file */
    long reloffsets[4];  /* From, To, Subject, Body */
    long lengths[4];     /* byte lengths including terminating zeros */

    char date_string[22];
    char new_date_string[22];
    flag_t needs_fix;
};

struct pktfile
{
    FILE *f; const char *filename;
    int day, month, year;
    int nday, nmonth, nyear;
    int node, net, zone, point;
    unsigned int capability_word;
    size_t nmsgs, nmaxmsgs;
    struct msginpkt HUGEPTR *msg;
    int is_qqq;  /* if 1: do not write trailing 00 00 when fixing up! */
    int keep_me; /* if 1: keep a copy of the original corrupted file */
    unsigned int baudrate;

    flag_t needs_fix;
};

static int  read_pkt  (struct pktfile *ppkt, FILE *f, const char *filename);
static void fix_pkt   (struct pktfile *ppkt);
static void show_pkt  (struct pktfile *ppkt);
static int  write_pkt (struct pktfile *ppkt);
int movefile (const char *old_name, const char *new_name);
int fexist (const char *filename);
long daydiff(struct tm*, struct tm*);

static int process_pkt (const char *filename)
{
    FILE *f;
    struct pktfile pkt;

    if ((f = fopen(filename, "rb")) == NULL)
    {
        log (LOG_ERROR, LNG(3), filename);
        return FATAL_ERROR;
    }

    if (read_pkt(&pkt, f, filename) == -1)
    {
        if (pkt.f   != NULL) fclose(f);
        if (pkt.msg != NULL) free(pkt.msg);
        return FATAL_ERROR;
    }

    fix_pkt(&pkt);
    show_pkt(&pkt);

    if (pkt.needs_fix && config.fix)
    {
        if (config.keep)
            pkt.keep_me = 1;
        if (!write_pkt(&pkt))
        {
            log (LOG_NORMAL, LNG(4), pkt.filename);
        }
    }


    if (pkt.f != NULL) fclose(f);
    free(pkt.msg);
    return (int)pkt.needs_fix;
}

static int read_pkt(struct pktfile *ppkt, FILE *f, const char *filename)
{
    unsigned char buffer[0x3A];
    long ofs, safe_ofs;
    int i, j, spz;

    ppkt->f = f;
    ppkt->filename = filename;
    ppkt->nmaxmsgs = 0; ppkt->nmsgs = 0; ppkt->msg = NULL;
    ppkt->needs_fix = 0;
    ppkt->is_qqq = 0;
    ppkt->keep_me = 0;
    ppkt->node = ppkt->net = ppkt->zone = ppkt->point = 0;

    i = fread(buffer, 1, ofs = 0x3A, f);

    if (i != 0x3A)
    {
        if (i == 0 || feof(f))
        {
           log (LOG_WARNING, LNG(5), ppkt->filename);
           return -1;
        }

        log (LOG_ERROR, LNG(6), ppkt->filename);
        return -1;
    }


    if (ppkt->baudrate != 2)
    {
        ppkt->nday   = ppkt->day   = (((unsigned)buffer[9])<<8) + buffer[8];
        ppkt->nmonth = ppkt->month = (((unsigned)buffer[7])<<8) + buffer[6] + 1;
        ppkt->nyear  = ppkt->year  = (((unsigned)buffer[5])<<8) + buffer[4];
    }

    ppkt->node = (((unsigned)buffer[1]) << 8) + buffer[0];
    ppkt->net  = (((unsigned)buffer[21]) << 8) + buffer[20];
    ppkt->zone = (((unsigned)buffer[35]) << 8) + buffer[34];

    ppkt->baudrate = buffer[0x10];
    ppkt->capability_word = buffer[0x2C] + (buffer[0x2D] << 8);
    if (ppkt->capability_word != (buffer[0x28] << 8) + buffer[0x29])
    {
        ppkt->capability_word = 0;  /* stone age! */
        ppkt->point = 0;
    }
    else
    {
        if (buffer[46] != 0 || buffer[47] != 0)
            ppkt->zone = (((unsigned)buffer[47]) << 8) + buffer[46];
        ppkt->point = (((unsigned)buffer[51]) << 8) + buffer[50];
    }


    if (ppkt->capability_word != 0 && ppkt->capability_word != 1)
    {
        log (LOG_ERROR,  LNG(7), ppkt->filename);
        return -1;
    }

    while (1)
    {
        if (fread(buffer,1,2,f) != 2)
        {
            if(config.qqq)
            {
                ppkt->is_qqq = 1;
                break;  /* end of FE QQQ file */
            }
            else
            {
                log (LOG_ERROR, LNG(8), ppkt->filename);
                return -1;
            }
        }
        ofs += 2;

        if ((buffer[0] == buffer[1]) && (buffer[1] == 0))  /* end of pkt */
        {
            break;
        }
        if ((buffer[0] != 2) || (buffer[1] != 0))          /* bad pkt */
        {
            safe_ofs = ofs - 2;

            if (ppkt->nmsgs > 0 &&
                (ppkt->msg[ppkt->nmsgs-1].needs_fix & FTSC_NO_ZERO))
            {
                /* Speculative read ahead; maybe an overflowing date field
                   has misled us to assume an empty "to" field, so we seek
                   to the next null and see if another message starts after
                   that one. This is the most complicated code of the whole
                   program ...*/

                while (buffer[0] != 0 && (!feof(f)))
                {
                    buffer[0] = buffer[1];
                    buffer[1] = fgetc(f);
                    ofs++;
                }

                spz = -1;     /* no sucess by default */

                if (feof(f) && config.qqq && buffer[0] == 0)
                {
                    spz = 1;  /* reached end of a QQQ file */
                    ppkt->is_qqq = 1;
                }
                else if (buffer[0] == 0)
                {             /* reached end of this message */
                    buffer[0] = buffer[1]; buffer[1] = fgetc(f); ofs++;
                    if ((buffer[0] == 2 || buffer[0] == 0) && buffer[1] == 0)
                    {
                        spz = 2;
                    }
                }

                if (spz != -1)
                {
                    /* bingo! we found the end of the last message */

                    ppkt->msg[ppkt->nmsgs - 1].needs_fix |=
                        FTSC_SPILLED_ZERO;
                    ppkt->needs_fix |= FTSC_SPILLED_ZERO;

                    for (i = 0; i < 3; i++)
                    {
                        ppkt->msg[ppkt->nmsgs - 1].reloffsets[i] =
                            ppkt->msg[ppkt->nmsgs - 1].reloffsets[i + 1];
                        ppkt->msg[ppkt->nmsgs - 1].lengths[i] =
                            ppkt->msg[ppkt->nmsgs - 1].lengths[i + 1];
                    }

                    ppkt->msg[ppkt->nmsgs - 1].reloffsets[3] =
                        safe_ofs - ppkt->msg[ppkt->nmsgs - 1].absoffset;
                    ppkt->msg[ppkt->nmsgs - 1].lengths[3] =
                        ofs - spz - safe_ofs;

                    if (buffer[0] == 0 || (feof(f) && config.qqq))
                    {
                        break;
                    }
                }
            }
            if ( ppkt->nmsgs < 1 ||
                 (!(ppkt->msg[ppkt->nmsgs - 1].needs_fix & FTSC_SPILLED_ZERO))
               )
            {
                log (LOG_ERROR, LNG(9), ppkt->filename);
                return -1;
            }
        }

        if (fread(buffer+2,1,0x20,f) != 0x20)
        {
            log (LOG_ERROR, LNG(8), ppkt->filename);
            return -1;
        }
        ofs += 0x20;

        if (ppkt->nmaxmsgs <= ppkt->nmsgs)
        {
            if (ppkt->msg == NULL)
            {
                if ((ppkt->msg = malloc(sizeof(struct msginpkt) *
                                        (ppkt->nmaxmsgs = 10))) == NULL)
                {
                    log (LOG_ERROR, LNG(10));
                    return -1;
                }
            }
            else
            {
                if ((ppkt->msg = realloc(ppkt->msg,
                                         sizeof(struct msginpkt) *
                                         (ppkt->nmaxmsgs *= 2))) == NULL)
                {
                    log (LOG_ERROR, LNG(10));
                    return -1;
                }
            }
        }

        ppkt->msg[ppkt->nmsgs].absoffset = ofs - 0x22;
        ppkt->msg[ppkt->nmsgs].needs_fix = 0;

        memcpy(ppkt->msg[ppkt->nmsgs].date_string, buffer+0xE, 20);
        ppkt->msg[ppkt->nmsgs].date_string[20] = '\0';
        if ((j = strlen(ppkt->msg[ppkt->nmsgs].date_string)) == 20)
            ppkt->msg[ppkt->nmsgs].needs_fix |= FTSC_NO_ZERO;

        /* Fix for problem reported by Stas: Date field too short (!),
           and Recipient name begins right after the zero, instead of
           at the expected position */

        for (j++, i=0; j < 20; i++, j++)
           if (!((ppkt->msg[ppkt->nmsgs].date_string[j] >= 'A' &&
                  ppkt->msg[ppkt->nmsgs].date_string[j] <= 'Z') ||
                 (ppkt->msg[ppkt->nmsgs].date_string[j] >= 'a' &&
                  ppkt->msg[ppkt->nmsgs].date_string[j] <= 'z')))
           {
               i = 0; break;
               /* This is probably not the recipient's name, but garbage */
           }
        if (i)
        {
            ofs -= i;
            if (fseek(ppkt->f, -i, SEEK_CUR) != 0)
            {
                log (LOG_ERROR, LNG(8), ppkt->filename);
                return -1;
            }
            ppkt->msg[ppkt->nmsgs].needs_fix |= FTSC_TOO_SHORT;
        }

        for (i=0;i<4;i++)
        {
            int c;
            ppkt->msg[ppkt->nmsgs].reloffsets[i] = ofs -
                ppkt->msg[ppkt->nmsgs].absoffset;

            while ((c=fgetc(f)) != 0)
            {
                ofs++;
                if (c==EOF)
                {
                    log (LOG_ERROR, LNG(8), ppkt->filename);
                    return -1;
                }
            }
            ofs++;

            ppkt->msg[ppkt->nmsgs].lengths[i] =
                ofs - (ppkt->msg[ppkt->nmsgs].reloffsets[i] +
                       ppkt->msg[ppkt->nmsgs].absoffset);
        }
        ppkt->nmsgs ++;

    }

    return 0;
}

#define BUFFER_SIZE 32768       /* Any system should be able to malloc 32K, and
                                   it gives tolerable good performance */

static int  write_pkt (struct pktfile *ppkt)
{
    char *tmp_file_name;
    char *y2k_file_name;
    FILE *tmp_file;
    unsigned char *buffer;
    size_t i,k;
    struct msginpkt HUGEPTR *pmsg;
    int j;
    long n;
    size_t toread;

    if ((buffer = malloc(BUFFER_SIZE)) == NULL)
    {
        log (LOG_ERROR, LNG(10));
        return -1;
    }

    if (((tmp_file_name = tmpnam(NULL)) == NULL) ||
        ((tmp_file = fopen(tmp_file_name, "wb")) == NULL))
    {
        log (LOG_ERROR, LNG(11));
        free(buffer);
        return -1;
    }

    /* transfer the pkt header */
    if (fseek(ppkt->f, 0, SEEK_SET) != 0) goto read_error;
    if (fread(buffer, 0x3A, 1, ppkt->f) != 1) goto read_error;
    if (ppkt->baudrate != 2)
    {
        if (config.fix && (ppkt->needs_fix & PKT_HEADER_DATE))
        {
            buffer[4] = (unsigned char)(ppkt->nyear & 0xFF);
            buffer[5] = (unsigned char)((ppkt->nyear >> 8) & 0xFF);
            buffer[6] = (unsigned char)((ppkt->nmonth - 1) & 0xFF);
            buffer[7] = (unsigned char)(((ppkt->nmonth - 1) >> 8) & 0xFF);
            buffer[8] = (unsigned char)(ppkt->nday & 0xFF);
            buffer[9] = (unsigned char)((ppkt->nday >> 8) & 0xFF);
        }
    }
    if (fwrite(buffer, 0x3A, 1, tmp_file) != 1) goto write_error;

    for (i=0; i < ppkt->nmsgs; i++)
    {
        pmsg = ppkt->msg + i;

        /* read msg header, fix if necessary, and save it again */
        if (fseek(ppkt->f, pmsg->absoffset, SEEK_SET) != 0) goto read_error;
        if (fread(buffer, 0x22, 1, ppkt->f) != 1) goto read_error;
        if (buffer[0] != 2 || buffer[1] != 0) goto integrity_error;
        if (pmsg->needs_fix)
        {
            memcpy(buffer + 0xe, pmsg->new_date_string, 20);
        }
        if (fwrite(buffer, 0x22, 1, tmp_file) != 1) goto write_error;

        for (j = 0; j < 4; j++)
        {
            if (fseek(ppkt->f, pmsg->absoffset + pmsg->reloffsets[j], SEEK_SET)
                != 0) goto read_error;

            for (n = 0; n < pmsg->lengths[j]; n += (long) toread)
            {
                toread = (n + BUFFER_SIZE < pmsg->lengths[j])
                    ? BUFFER_SIZE  : (size_t)(pmsg->lengths[j] - n);

                if (fread(buffer, toread, 1, ppkt->f) != 1) goto read_error;
                if (fwrite(buffer, toread, 1, tmp_file) != 1) goto write_error;
            }
        }
    }

    /* write 00 00, unless it is a Fastecho .QQQ file */
    if (!ppkt->is_qqq)
    {
        buffer[0] = 0; buffer[1] = 0;
        if (fwrite(buffer, 2, 1, tmp_file) != 1) goto write_error;
    }

    fclose(tmp_file);
    free(buffer);
    fclose(ppkt->f); ppkt->f = NULL;

    if (ppkt->keep_me)
    {
        /* we want to keep a copy of the file renamed to .y2k */

        i = strlen(ppkt->filename);
        if (!i || (y2k_file_name = malloc(i + 5)) == NULL)
        {
            log (LOG_ERROR, LNG(12));
            remove(ppkt->filename);
        }
        else
        {
            /* construct the new file name */

            memmove(y2k_file_name, ppkt->filename, i + 1);
            for (k = i - 1; k > 0 && y2k_file_name[k] != '.'
                 && y2k_file_name[k] != '/' && y2k_file_name[k] != '\\'
                 && y2k_file_name[k] != ':'; k--);
            if (y2k_file_name[k] != '.')
            {
                k = i;
            }
            /* now k is the position where we can put the new extension */

            /* try the default extension first */
            strcpy(y2k_file_name + k,  ".y2k");

            /* but if it exists, try, .Y## extensions */
            for (i = 1; fexist(y2k_file_name) && i < 100; i++)
            {
                sprintf (y2k_file_name + k, ".y%02d", (int)i);
            }

            if (i == 100) /* we can't fine any free name - TROUBLE! */
            {
                log (LOG_ERROR, LNG(13));
                remove(ppkt->filename);
            }
            else
            {
                log (LOG_WARNING, LNG(14), y2k_file_name);
                if (movefile(ppkt->filename, y2k_file_name) == -1)
                {
                    log(LOG_ERROR, LNG(15),
                        ppkt->filename, y2k_file_name);
                    remove(ppkt->filename);
                }
            }
        }
    }
    else
    {
        remove(ppkt->filename);
    }

    /* move the new file in place */;
    movefile(tmp_file_name, ppkt->filename);
    return 0;

 write_error:
    log (LOG_ERROR, LNG(16), tmpnam);
    fclose(tmp_file);
    remove(tmp_file_name);
    free(buffer);
    return -1;

 integrity_error:
    log (LOG_ERROR, LNG(17), tmpnam);
    fclose(tmp_file);
    remove(tmp_file_name);
    free(buffer);
    return -1;

 read_error:
    log (LOG_ERROR, LNG(6), ppkt->filename);
    fclose(tmp_file);
    remove(tmp_file_name);
    free(buffer);
    return -1;
}

static flag_t parse_ftsc_date(struct tm *, struct tm *, char *);
static void   make_ftsc_date (char *, const struct tm *);

static void fix_pkt  (struct pktfile *ppkt)
{
    struct tm tm, tms, tmpkt;
    struct tm *pnow;
    time_t t_now;
    struct msginpkt HUGEPTR *pmsg;
    size_t i;
    int s;
    int currentreason = 0;
    int parseresult;

    time(&t_now);
    pnow = localtime(&t_now);   /* get the current time */

    /* First, fix the header. This is not as sophisticated as the message
       fixing functions, but the header is also not that important.  */

    if (ppkt->baudrate != 2) /* == 2 means type 2.2, i.e. no pkt header date */
    {
        tmpkt.tm_year = ppkt->year - 1900;
        tmpkt.tm_mon  = ppkt->month - 1;
        tmpkt.tm_mday = ppkt->day;
        tmpkt.tm_sec = 0;
        tmpkt.tm_min = 0;
        tmpkt.tm_hour = 0;

        if (ppkt->year < 1900 ||
            ppkt->month < 1 || ppkt->month > 12 ||
            ppkt->day < 1 || ppkt->day > 31 ||
            daydiff(&tmpkt, pnow) > (long) config.futureTol ||
            daydiff(pnow, &tmpkt) > (long) config.pastTol)
        {
            ppkt->needs_fix |= PKT_HEADER_DATE;
            ppkt->nyear  = pnow->tm_year + 1900;
            ppkt->nmonth = pnow->tm_mon + 1;
            ppkt->nday   = pnow->tm_mday;
        }
    }

    for (i = 1; i <= ppkt->nmsgs; i++)
    {
        pmsg = ppkt->msg + i - 1;

                                /* parse the FTSC date field */
        memcpy(pmsg->new_date_string, pmsg->date_string,
               sizeof(pmsg->new_date_string));


        parseresult = parse_ftsc_date(&tm, pnow, pmsg->new_date_string);

        pmsg->needs_fix |= parseresult;

        if (parseresult & FTSC_BROKEN) /* parsing failed fatally */
        {
            memcpy(&tm, pnow, sizeof(struct tm));
        } else
        {
            /* date plausibility checks  */
            if (daydiff(&tm, pnow) > (long)config.futureTol ||
                daydiff(pnow, &tm) > (long)config.pastTol)
            {
                pmsg->needs_fix |= UNPLAUSIBLE_DATE;

                /* Speculative error correction */

                for (s = 1; s <= 6; s++)
                {
                    memcpy(&tms, &tm, sizeof(struct tm));

                    switch (s)
                    {
                    case 1:
                        if (tm.tm_year == 70 && tm.tm_mon == 0 &&
                            tm.tm_mday == 1 && tm.tm_hour == 0 &&
                            tm.tm_min == 0 && tm.tm_sec == 0)
                        {
                            /* uninitialised date field */
                            memcpy(&tms, pnow, sizeof(struct tm));
                            currentreason = UD_FIELD_UNINITIALISED;
                        }
                        break;

                    case 2:

                        /* #1: Try to undo the 7 bit overflow bug (happens
                           when 00 is interpreted as "1900" and then a
                           timestamp structure with 7 bit year offset
                           (rel. 1980) is filled in. Observed in Squish. */

                        tms.tm_year = tm.tm_year - 28;
                        currentreason = UD_SQUISH_BUG;
                        break;

                    case 3:

                        /* #2..#4: Assume the sysop has just put his clock back
                           by some or several years */

                        tms.tm_year = pnow->tm_year;
                        currentreason = UD_YEAR_DISP;
                        break;

                    case 4:
                        tms.tm_year = pnow->tm_year - 1;
                        currentreason = UD_YEAR_DISP;
                        break;

                    case 5:
                        tms.tm_year = pnow->tm_year + 1;
                        currentreason = UD_YEAR_DISP;
                        break;

                    case 6:
                        /* no plan - set it to current time */
                        memcpy(&tms, pnow, sizeof(struct tm));
                        currentreason = 0;
                        break;
                    }

                    if (daydiff(&tms, pnow) < (long)config.sfutureTol &&
                        daydiff(pnow, &tms) < (long)config.spastTol)
                    {
                        memcpy(&tm, &tms, sizeof(struct tm));
                        pmsg->needs_fix |= currentreason;
                        break; /* leave the loop */
                    }
                }
            }
        }

        if (pmsg->needs_fix)
        {
            make_ftsc_date(pmsg->new_date_string, &tm);
            ppkt->needs_fix |= pmsg->needs_fix;
        }
    }
}

static void show_pkt(struct pktfile *ppkt)
{
    int i, j;
    struct msginpkt HUGEPTR *pmsg;
    int len;
    char *buf, *pbuf1, *pbuf2;
    char fromaddr[80];
    char fromname[50];

    log ((ppkt->needs_fix) ? LOG_WARNING : LOG_NORMAL, LNG(18),
         ID,
         ppkt->filename,
         ppkt->baudrate == 2 ? "2.2":
         (ppkt->capability_word == 1 ? "2+" : "stone-aged"));

    if (ppkt->baudrate != 2)
    {
        if ((ppkt->needs_fix & PKT_HEADER_DATE) && (config.fix))
        {
            log (LOG_WARNING, LNG(19),
                 ppkt->day, ppkt->month, ppkt->year,
                 ppkt->nday, ppkt->nmonth, ppkt->nyear);
        }
        else if (ppkt->needs_fix & PKT_HEADER_DATE)
        {
            log (LOG_WARNING, LNG(20),
                 ppkt->day, ppkt->month, ppkt->year,
                 ppkt->nday, ppkt->nmonth, ppkt->nyear);
        }
        else
        {
            log ((ppkt->needs_fix) ? LOG_WARNING : LOG_NORMAL, LNG(21),
                 ppkt->day, ppkt->month, ppkt->year);
        }
    }


    log ((ppkt->needs_fix) ? LOG_WARNING : LOG_NORMAL, LNG(22),
         ppkt->zone, ppkt->net, ppkt->node, ppkt->point);

    for (i = 1; i <= ppkt->nmsgs; i++)
    {
        pmsg = ppkt->msg + i - 1;

        fromaddr[0] = '\0';
        fromname[0] = '\0';

        if (pmsg->needs_fix)
        {

            /* find out the name of the sender */

            len = pmsg->lengths[1] < sizeof(fromname) ?
                  pmsg->lengths[1] : sizeof(fromname);

            if (fseek(ppkt->f, pmsg->absoffset + pmsg->reloffsets[1],
                      SEEK_SET) == 0 &&
                fread(fromname, len, 1, ppkt->f) == 1)
            {
                fromname[sizeof(fromname) - 1] = 0;
            }
            else
            {
                fromname[0] = '\0';
            }

            /* Find out the AKA of the sender of this message.
               This code takes it from the MSGID kludge.
               This is not the best way, but it works. */

            len = pmsg->lengths[3] > 32768 ? 32768 : pmsg->lengths[3];
            /* if the MSGID is not in the first 32K, it is nowhere */
            buf = malloc(len);

            if (buf != NULL)
            {
                if (fseek(ppkt->f, pmsg->absoffset + pmsg->reloffsets[3],
                          SEEK_SET) == 0 &&
                    fread(buf, len, 1, ppkt->f) == 1)
                {
                    pbuf1 = buf;
                    while ((pbuf2 = memchr(pbuf1, 0x01, len)) != NULL)
                    {
                        pbuf2++;
                        len = len - (pbuf2 - pbuf1);
                        pbuf1 = pbuf2;
                        if (!strncmp(pbuf1, "MSGID: ", 7))
                        {
                            pbuf1 += 7;
                            for (j = 0; j + 1< sizeof(fromaddr) &&
                                 pbuf1[j] != ' ' && pbuf1[j] && j < len;
                                 j++)
                                fromaddr[j] = pbuf1[j];
                            fromaddr[j] = '\0';
                        }
                    }
                }
                free(buf);
            }
        }

        if (pmsg->needs_fix && config.fix)
        {
            log (LOG_WARNING, LNG(23),
                 i, pmsg->date_string,
                 pmsg->new_date_string, (unsigned long) pmsg->needs_fix);
        }
        else if (pmsg->needs_fix)
        {
            log (LOG_WARNING, LNG(24),
                 i, pmsg->date_string,
                 pmsg->new_date_string,  (unsigned long) pmsg->needs_fix);

        }
        else
        {
            log (LOG_NORMAL, LNG(25), i, pmsg->date_string);
        }

        if (pmsg->needs_fix && fromaddr[0] && fromname[0])
        {
            log (LOG_NORMAL,  LNG(26), i, fromname, fromaddr);
        }
    }
}


/* ======================== general utility routines ==================== */


/* Some toupper routines crash when they get invalid input. As this program
   is intended to be portable and deal with any sort of malformed input,
   we have to provide our own toupper routine. */
char safe_toupper(char c)
{
    const char *from_table = "abcdefghijklmnopqrstuvwxyz";
    const char *to_table   = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    const char *p;

    if ((p = strchr(from_table, c)) != NULL)
    {
        return (to_table[p - from_table]);
    }

    return c;
}


int fexist(const char *filename)
{
    FILE *f = fopen(filename, "rb");

    if (f != NULL)
    {
        fclose(f);
        return 1;
    }
    else
    {
        /* at least we don't have write access to the file - this is good */
        /* enough for our purposes: we cannot destroy anything at least.  */
        return 0;
    }
}

#define MOVE_FILE_BUFFER_SIZE 16384

int movefile(const char *from, const char *to)
{
    int rc;
    char *buffer;
    size_t read;
    FILE *fin, *fout;


    rc = rename(from, to);
    if (!rc) return 0;        /* rename succeeded. fine! */

    /* Rename did not succeed, probably because the move is accross
       file system boundaries. We have to copy the file. */

    buffer = malloc(MOVE_FILE_BUFFER_SIZE);
    if (buffer == NULL) return -1;

    fin = fopen(from, "rb");
    if (fin == NULL) { free(buffer); return -1; }

    fout = fopen(to, "wb");
    if (fout == NULL) { free(buffer); return -1; }

    while ((read = fread(buffer, 1, MOVE_FILE_BUFFER_SIZE, fin)) > 0)
    {
    if (fwrite(buffer, 1, read, fout) != read)
    {
        fclose(fout); fclose(fin); remove(to); free(buffer);
        return -1;
    }
    }

    if (ferror(fout) || ferror(fin))
    {
    fclose(fout);
    fclose(fin);
    free(buffer);
    remove(to);
    return -1;
    }

    fclose(fout);
    fclose(fin);
    free(buffer);
    remove(from);
    return 0;
}


/* ======================== date utility routines ======================= */


/* The Julian Date is a unique day number. It can be used to calculate the */
/* difference in days between dates, or to calculate the day of week.      */
/* Works from May 1600 onwards. */

long get_julian_date(int day, int month, int year)
{
    if (month > 2)
    {
        month -= 3;
    }
    else
    {
        month += 9;
        year--;
    }
    year -= 1600;
    return (((((long)year / 100)  *146097L) / 4) +
            ((((long)year % 100)  *1461) / 4) +
            (((153 * (long)month) + 2) / 5) + day + 59);
}

long daydiff(struct tm* first, struct tm* second)
{
    long diff =
           - get_julian_date(second->tm_mday,
                             second->tm_mon + 1,
                             second->tm_year + 1900)
           + get_julian_date(first->tm_mday,
                             first->tm_mon + 1,
                             first->tm_year + 1900);
    return diff;
}



static const char *months[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun",
                               "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};

int get_month(const char *pmon, flag_t *flag)
{
    int i;

    if (strlen(pmon) != 3 && flag != NULL)
    {
        (*flag) |= FTSC_FLAWY;
    }

    for (i=0; i < 12; i++)
    {
        if (pmon[0] == months[i][0] &&
            pmon[1] == months[i][1] &&
            pmon[2] == months[i][2])
        {
            return i;
        }
    }

    for (i=0; i < 12; i++)
    {
        if (safe_toupper(pmon[0]) == safe_toupper(months[i][0]) &&
            safe_toupper(pmon[1]) == safe_toupper(months[i][1]) &&
            safe_toupper(pmon[2]) == safe_toupper(months[i][2]))
        {
            (*flag) |= FTSC_FLAWY;
            return i;
        }
    }

    (*flag) |= FTSC_BROKEN;
    return 0;
}


static flag_t parse_ftsc_date(struct tm * ptm, struct tm * pnow,
                              char *pdatestr)
{
    const char *pday, *pmon, *pyear, *phour, *pminute, *psecond;
    flag_t rval;
    char buf[22];

    pday = pmon = pyear = phour = pminute = psecond = NULL;

    rval = FTSC_BROKEN;

    memcpy(buf, pdatestr, 21); buf[21] = 0;

    if ((pday = strtok(pdatestr, " ")) != NULL)
        if ((pmon = strtok(NULL, " ")) != NULL)
            if ((pyear = strtok(NULL, " ")) != NULL)
                if ((phour = strtok(NULL, ":")) != NULL)
                    if ((pminute = strtok(NULL, ":")) != NULL)
                        if ((psecond = strtok(NULL, " ")) != NULL)
                            rval = 0;

    if (rval == FTSC_BROKEN)
    {
                 /* let's try and see if it might be the old SeaDog format */

        rval = FTSC_BROKEN;

        if ((strtok(buf, " ")) != NULL)
            if ((pday = strtok(NULL, " ")) != NULL)
                if ((pmon = strtok(NULL, " ")) != NULL)
                    if ((pyear = strtok(NULL, " ")) != NULL)
                        if ((phour = strtok(NULL, ": ")) != NULL)
                            if ((pminute = strtok(NULL, ": ")) != NULL)
                            {
                                psecond = NULL;
                                if (config.fixseadog)
                                    rval = FTSC_SEADOG;
                                else
                                    rval = 0;
                            }
    }


    ptm->tm_sec = ptm->tm_min = ptm->tm_hour = ptm->tm_mday = ptm->tm_mon =
        ptm->tm_year = 0;

    while (rval != FTSC_BROKEN)    /* at least we could tokenize it! */
    {
        if (psecond != NULL)
        {
            ptm->tm_sec = atoi(psecond);   /* Is the number of seconds valid? */
            if (strlen(psecond) == 1)
            {
                rval |= FTSC_FLAWY;
                if (ptm->tm_sec < 6) (ptm->tm_sec *= 10);
            }
            if (ptm->tm_sec < 0 || ptm->tm_sec > 59)
            {
                rval |= FTSC_TS_BROKEN; ptm->tm_sec = 0;
            }
        }
        else
        {
            ptm->tm_sec = 0;
        }

        ptm->tm_min = atoi(pminute);   /* Is the number of minutes valid? */
        if (ptm->tm_min < 0 || ptm->tm_min > 59)
        {
            rval |= FTSC_TS_BROKEN; ptm->tm_min = 0;
        }

        ptm->tm_hour = atoi(phour);    /* Is the number of hours valid? */
        if (ptm->tm_hour < 0 || ptm->tm_hour>23)
        {
            rval |= FTSC_TS_BROKEN; ptm->tm_hour = 0;
        }

        ptm->tm_mday = atoi(pday);     /* Is the day in the month valid? */
        if (ptm->tm_mday < 1 || ptm->tm_mday>31) { rval |= FTSC_BROKEN; break; }

        ptm->tm_mon = get_month(pmon, &rval); /* Is the month valid? */

        if (strlen(pyear) != 2)        /* year field format check */
        {
            rval |= FTSC_FLAWY;
        }

        if (*pyear)
        {
            ptm->tm_year = (*pyear - '0'); /* allows for the ":0" bug */
            for (pyear++; isdigit((int)(*pyear)); pyear++)
            {
                ptm->tm_year *= 10;
                ptm->tm_year += (*pyear - '0');
            }
            if (*pyear)
            {
                rval |= FTSC_BROKEN;
                break;
            }
        }
        else
        {
            rval |= FTSC_BROKEN;
            break;
        }

        if (ptm->tm_year < 100)  /* correct date field! */
        {
            while (pnow->tm_year - ptm->tm_year > 50)
            {                         /* sliding window adaption */
                ptm->tm_year += 100;
            }
        }
        else if (ptm->tm_year < 1900) /* probably the field directly */
                                      /* contains tm_year, like produced */
                                      /* by the Timed/Netmgr bug and others */
        {
            rval |= FTSC_FLAWY;
        }
        else                          /* 4 digit year field, not correct! */
        {
            ptm->tm_year -= 1900;
            rval |= FTSC_FLAWY;
        }
        break;
    }


    return rval;

}

static void make_ftsc_date(char *pdate, const struct tm *ptm)
{
    sprintf(pdate, "%02d %-3.3s %02d  %02d:%02d:%02d",
            ptm->tm_mday % 100, months[ptm->tm_mon], ptm->tm_year % 100,
            ptm->tm_hour % 100, ptm->tm_min % 100, ptm->tm_sec % 100);
}


/* ========================== Routine for logging ======================= */
void log(int loglevel, const char *fmt, ...)
{
    va_list argptr;
    int neednl = fmt[strlen(fmt) - 1] != '\n';

    time_t t;
    struct tm *tm;
    char cpTime[25];
    static char *cpMonths[]={"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul",
                             "Aug", "Sep", "Oct", "Nov", "Dec", "???"};

    va_start(argptr, fmt);
    if (config.maxlog >= loglevel)
    {
        time(&t);
        tm=localtime(&t);
        sprintf(cpTime,"%c %02d %-3.3s %02d %02d:%02d:%02d PKTD ",
                log_chars[loglevel],
                tm->tm_mday, cpMonths[tm->tm_mon], tm->tm_year % 100,
                tm->tm_hour, tm->tm_min, tm->tm_sec);

        if (config.logfile != NULL)
        {
            fprintf(config.logfile, "%s", cpTime);
            vfprintf(config.logfile, fmt, argptr);
            if (neednl) fprintf(config.logfile, "\n");
        }
        vfprintf(stdout, fmt, argptr);
        if (neednl) fprintf(stdout, "\n");
    }
}



/* ===================== Command line analysator ======================== */
/* Parses a command line according to the POSIX standards.                */
/* Option clustering (-abc instead of -a -b -c) is not supported.         */

static const char *valid_options="?qcSkl:m:L:f:p:F:P:";

static void initoptions (void)
{
    config.fix = 0;
    config.logfilename = NULL;    /* "pktdate.log"; */
    config.maxlog = LOG_NORMAL;
    config.futureTol  = 31L;      /* One month into the future. */
    config.pastTol    = 364L;     /* Almost one year into the past. */
    config.sfutureTol = 1L;       /* Acceptance window for error correction */
    config.spastTol   = 31L;      /* Acceptance window for error correction */

}

static void finalizeconfig (void)
{
    if (config.logfilename != NULL)
    {
        config.logfile = fopen (config.logfilename, "a");
    }
    else
    {
        config.logfile = NULL;
    }
}

static void processoption (char option, char *argument)
{
    int flag;

    switch (option)
    {
    case 'c':
        config.fix = 1;
        break;

    case 'q':
        config.qqq = 1;
        break;

    case 'k':
        config.keep = 1;
        break;

    case 'S':
        config.fixseadog = 1;
        break;

    case 'L':
        config.logfilename = argument;
        break;

    case 'l':
        if (atoi(argument) != 0)
        {
            config.maxlog = atoi(argument);
        }
        break;
    case 'm':
        if  (argument[0] == 'g' || argument[0] == 'G')
            lng = lng_german;
        else if (argument[0] == 'e' || argument[0] == 'E')
            lng = lng_english;
        else if (argument[0] == 'r' || argument[0] == 'R')
        {
            if (argument[1] == '8')
                lng = lng_rus866;
            else
                lng = lng_ruskoi;
        }
        break;


    case 'f':
    case 'p':
    case 'F':
    case 'P':
        if (*argument && safe_toupper(argument[strlen(argument) - 1]) == 'Y')
        {
            argument[strlen(argument) - 1] = '\0';
            flag = 365;
        }
        else
        if (*argument && safe_toupper(argument[strlen(argument) - 1]) == 'M')
        {
            flag = 31;
        }
        else
        {
            flag = 1;
        }
        if (atol(argument) != 0L)
        {
            if (option == 'f')
            {
                config.futureTol = atol(argument) * ((unsigned long)flag);
            }
            else if (option == 'p')
            {
                config.pastTol = atol(argument) * ((unsigned long) flag);
            }
            else if (option == 'F')
            {
                config.futureTol = atol(argument) * ((unsigned long) flag);
            }
            else if (option == 'P')
            {
                config.spastTol = atol(argument) * ((unsigned long) flag);
            }
        }
        break;

    default:
        if (option != '?')
        {
            fprintf (stderr, LNG(27), option);
        }
        else
        {
            printf (LNG(29), ID);
            if (argument != NULL)
            {
                fprintf (stderr, LNG(30), argument);
            }
            printf ("%s\n", LNG(28));
        }
        exit ((argument == NULL) ? 0 : 8);

    }
}

static void set_default_language(void)
{
    char *envlng = getenv("LANG");

    lng = lng_english; /* this is the default */

    if (envlng != NULL)
    {
        if ((envlng[0]=='r' || envlng[0]=='R') &&
            (envlng[1]=='u' || envlng[1]=='U'))
        {
#if defined(unix) || defined(__unix__) || defined(linux) || defined(__linux__) \
 || defined(UNIX) || defined(BSD)
             lng = lng_ruskoi;
#else
             lng = lng_rus866;
#endif
        }
        else if ((envlng[0]=='d' || envlng[0]=='D') &&
                 (envlng[1]=='e' || envlng[1]=='E'))
        {
            lng = lng_german;
        }
    }
}


static int init(int argc, char **argv, int require_arguments)
{
    int i, firstargument = 0;
    const char *cp;
    char errmsg[160];

    /* this sets all configuration parameters to their default values */
    initoptions();

    /* sets the message language */
    set_default_language();

    /* process all options. stop as soon as we find an argument.
       (options like -l1 MODIFY the behaviour of the program, arguments
       like filenames tell it what to do at all. Optoins come first, then come
       the filenames, optionally separated by a double dash */
    for (i = 1; i < argc && firstargument == 0; i++)
    {
        switch (argv[i][0])
        {
        case '-':
            switch (argv[i][1])
            {
            case '-':    /* -- indicates end of options -- */
                firstargument = i + 1;
                break;

            case '\0':   /* single dash is probably a file name argument */
                firstargument = i;
                break;

            default:                              /* some sort of option */
                cp = strchr(valid_options, argv[i][1]);
                if (cp == NULL)                   /* invalid option */
                {
                    sprintf (errmsg, LNG(31), argv[i][1]);
                    processoption ('?', errmsg); /* error */
                }
                else if (cp[1] == ':')            /* option with argument */
                {
                    if (argv[i][2] == '\0' && i + 1 < argc)
                    {
                        processoption (*cp, argv[i + 1]);
                        i++;
                    }
                    else if (argv[i][2] != '\0')
                    {
                        processoption (*cp, argv[i]+2);
                    }
                    else
                    {
                        sprintf (errmsg, LNG(32), *cp);
                        processoption ('?', errmsg);
                    }
                }
                else                              /* option w/o argument */
                {
                    if (argv[i][2] != '\0')       /* error - unexp. opt arg */
                    {
                        sprintf (errmsg, LNG(33), i);
                        processoption ('?', errmsg);
                    }
                    else
                    {
                        processoption (*cp, NULL);
                    }
                }
            }
            break;
        default:
            firstargument = i;
        }
    }

    if (firstargument == 0 && require_arguments)
    {
        sprintf (errmsg, LNG(34));
        processoption ('?', errmsg);
    }

    finalizeconfig();

    return firstargument;
}
