/*
 * MultiMail offline mail reader
 * message display and editing

 Copyright (c) 1996 Kolossvary Tamas <thomas@tvnet.hu>
 Copyright (c) 1997 John Zero <john@graphisoft.hu>
 Copyright (c) 1999 William McBrine <wmcbrine@clark.net>

 Distributed under the GNU General Public License.
 For details, see the file COPYING in the parent directory. */

#include "interfac.h"

Line::Line(int size)
{
	next = 0;
	text = (size ? new char[size] : 0);
	quoted = hidden = tearline = tagline = origin = false;
}

Line::~Line()
{
	delete[] text;
}

LetterWindow::LetterWindow()
{
	linelist = 0;
	NM.isSet = hidden = rot13 = false;
	NumOfLines = 0;
	To[0] = tagline1[0] = '\0';
	stripCR = mm.resourceObject->getInt(StripSoftCR);
	beepPers = mm.resourceObject->getInt(BeepOnPers);
	lynxNav = mm.resourceObject->getInt(UseLynxNav);
}

net_address &LetterWindow::PickNetAddr()
{
	Line *line;
	static net_address result;
	int i;

	for (i = 0; i < NumOfLines; i++)
		if (!strncmp(" * Origin:", linelist[i]->text, 10))
			break;

	if (i != NumOfLines) {
		line = linelist[i];
		i = line->length;
		while (((line->text[i - 1]) != '(') && i > 0)
			i--;
		//we have the opening bracket

		while ((i < (int) line->length) &&
			((line->text[i] < '0') || (line->text[i] > '9')))
				i++;
		//we have the begining of the address

		result = &line->text[i];
	} else
		result = mm.letterList->getNetAddr();

	return result;
}

void LetterWindow::ReDraw()
{
	headbar->wtouch();
	header->wtouch();
	text->wtouch();
	statbar->cursor_on();	// to get it out of the way
	statbar->wtouch();
	statbar->cursor_off();
}

void LetterWindow::DestroyChain()
{
	if (linelist) {
		while (NumOfLines)
			delete linelist[--NumOfLines];
		delete[] linelist;
		linelist = 0;
	}
	letter_in_chain = -1;
}

/* Take a message as returned by getBody() -- the whole thing as one C
   string, with paragraphs separated by '\n' -- and turn it into a linked
   list with an array index, wrapping long paragraphs if needed. Also
   flags hidden and quoted lines.
*/
void LetterWindow::MakeChain(int columns, bool rejoin)
{
	static const char *seenby = "SEEN-BY:";
	Line head(0), *curr;
	unsigned char each, compare = isoConsole ? 0xEC : 0x8D;
	char *j, *k, *begin;
	int c;
	bool end = false, wrapped = false;
	bool strip = mm.letterList->isLatin() ? false : stripCR;
	bool skipSeenBy = false;

	DestroyChain();
	j = (char *) mm.letterList->getBody();

	letterconv_in(j);

	letter_in_chain = mm.letterList->getCurrent();
	curr = &head;

	while (!end) {
	    if (!hidden)	// skip ^A lines
		while ((*j == 1) || (skipSeenBy && !strncmp(j, seenby, 8))) {
			do
				j++;
			while (*j && (*j != '\n'));
			while (*j == '\n')
				j++;
		}

	    if (*j) {
		curr->next = new Line(((columns > 80) || !rejoin) ?
			columns : 81);
		curr = curr->next;

		NumOfLines++;

		begin = 0;
		k = curr->text;
		c = 0;

		if (*j == 1) {		// ^A is hidden line marker
			j++;
			curr->hidden = true;
		} else
			if (skipSeenBy && !strncmp(j, seenby, 8))
				curr->hidden = true;

		while (!end && ((c < columns) || (rejoin &&
		    curr->quoted && (c < 80)))) {

			each = *j;

			// Strip soft CR's:

			if (strip && (each == compare))
				each = *++j;

			if (each == '\n') {
			    if (wrapped) {
				wrapped = false;

				// Strip any trailing spaces (1):

				while (c && (k[-1] == ' ')) {
					c--;
					k--;
				}

				// If the next character indicates a new
				// paragraph, quote or hidden line, this
				// is the end of the line; otherwise make
				// it a space:

				each = j[1];
				if (!each || (each == '\n') || (each == ' ')
				  || (each == '\t') || (each == '>') ||
				     (each == 1) || skipSeenBy) {
					j++;
					break;
				} else
					if (!c)
						j++;
					else
						each = ' ';
			    } else {		// if wrapped is false, EOL
				j++;
				break;
			    }
			}

			switch (each) {
			case ' ':
				// begin == start of last word (for wrapping):

				begin = 0;
				if (rejoin && !curr->quoted)
					if (c == (columns - 1))
						wrapped = true;
				break;
			case '>':		// Quoted line
				if (c < 5)
					curr->quoted = true;
			default:
				if (!begin)
					begin = j;

				if (rot13) {
				    if (each >= 'A' && each <= 'Z')
					each = (each - 'A' + 13) % 26 + 'A';
				    else if (each >= 'a' && each <= 'z')
					each = (each - 'a' + 13) % 26 + 'a';
				}
			}

			if (each) {
				*k++ = each;
				j++;
				c++;
			}

			end = !each;	// a 0-terminated string
		}

		k = curr->text;

		// Start a new line on a word boundary (if needed):

		if ((c == columns) && begin && ((j - begin) < columns) &&
		   !(rejoin && curr->quoted)) {
			c -= j - begin;
			j = begin;
			wrapped = rejoin;
		}

		// Strip any trailing spaces (2):

		while (c && (k[c - 1] == ' '))
			c--;

		k[c] = '\0';
		curr->length = c;

		// Check for taglines, tearlines, and origin lines:

                each = *k;
		if ((k[1] == each) && (k[2] == each) &&
		   ((k[3] == ' ') || (c == 3)))
			switch (each) {
			case '.':
				curr->tagline = true;
				break;
			case '-':
			case '~':
				curr->tearline = true;
			}
		else
			if (!strncmp(k, " * Origin:", 10)) {
				curr->origin = true;

				// SEEN-BY: should only appear after Origin:
				skipSeenBy = true;
			}
	    } else
		end = true;
	}

	linelist = new Line *[NumOfLines];
	curr = head.next;
	c = 0;
	while (curr) {
		linelist[c++] = curr;
		curr = curr->next;
	}
}

void LetterWindow::StatToggle(int mask)
{
	int stat = mm.letterList->getStatus();
	stat ^= mask;
	mm.letterList->setStatus(stat);
	interface->setAnyRead();
	DrawHeader();
}

void LetterWindow::Draw(bool redo)
{
	if (redo) {
		rot13 = false;
		position = 0;
		tagline1[0] = '\0';
		if (!interface->dontRead()) {
			if (!mm.letterList->getRead()) {
				mm.letterList->setRead(); // nem ide kene? de.
				interface->setAnyRead();
			}
			if (beepPers && mm.letterList->isPersonal())
				beep();
		}
	}

	if (letter_in_chain != mm.letterList->getCurrent())
		MakeChain(COLS);

	DrawHeader();
	DrawBody();
	DrawStat();
}

char *LetterWindow::netAdd(char *tmp)
{
	net_address &na = mm.letterList->getNetAddr();
	if (na.isSet) {
		const char *p = na;
		if (*p)
			tmp += sprintf(tmp, (na.isInternet ?
				" <%s>" : " @ %s"), p);
	}
	return tmp;
}

#define flagattr(x) header->attrib((x) ? C_LHFLAGSHI : C_LHFLAGS)

void LetterWindow::DrawHeader()
{
	char tmp[256], *p;
	int stat = mm.letterList->getStatus();
	bool hasTo = mm.areaList->hasTo();

	header->Clear(C_LHEADTEXT);

	header->put(0, 2, "Msg#:");
	header->put(1, 2, "From:");
	if (hasTo)
		header->put(2, 2, "  To:");
	else
		header->put(2, 2, "Addr:");
	header->put(3, 2, "Subj:");

	header->attrib(C_LHMSGNUM);
	sprintf(tmp, "%d (%d of %d)   ", mm.letterList->getMsgNum(),
		mm.letterList->getCurrent() + 1, mm.areaList->getNoOfLetters());
	header->put(0, 8, tmp);

	header->attrib(C_LHFROM);
	p = tmp + sprintf(tmp, "%.255s", mm.letterList->getFrom());
	if (mm.areaList->isEmail())
		netAdd(p);
	letterconv_in(tmp);
	header->put(1, 8, tmp);

	header->attrib(C_LHTO);
	if (hasTo) {
		p = (char *) mm.letterList->getTo();
		if (*p) {
			p = tmp + sprintf(tmp, "%.255s", p);
			if (mm.areaList->isReplyArea())
				if (p != tmp)
					netAdd(p);
		}
	} else
		sprintf(tmp, "%.255s",
			(const char *) mm.letterList->getNetAddr());
		
	letterconv_in(tmp);
	header->put(2, 8, tmp);

	header->attrib(C_LHSUBJ);
	int i = sprintf(tmp, "%.71s", mm.letterList->getSubject());
	letterconv_in(tmp);
	header->put(3, 8, tmp);

	for (i += 8; i < COLS; i++)
		header->put(3, i, ' ');

	header->attrib(C_LHEADTEXT);
	header->put(0, COLS - 33, "Date:");
	header->put(1, COLS - 33, "Line:");
	header->put(2, COLS - 33, "Stat: ");

	//header->put(4, 0, ACS_LLCORNER);
	//header->horizline(4, COLS-2);
	//header->put(4, COLS-1, ACS_LRCORNER);

	flagattr(mm.letterList->getPrivate());
	header->put(2, COLS - 27, "Pvt ");
	flagattr(stat & MS_READ);
	header->put(2, COLS - 23, "Read ");
	flagattr(stat & MS_REPLIED);
	header->put(2, COLS - 18, "Replied ");
	flagattr(stat & MS_MARKED);
	header->put(2, COLS - 10, "Marked  ");

	header->attrib(C_LHDATE);
	strcpy(tmp, mm.letterList->getDate());
	letterconv_in(tmp);
	header->put(0, COLS - 27, tmp);

	lineCount();
}

void LetterWindow::lineCount()
{
	char tmp[80];

	header->attrib(C_LHMSGNUM);
	sprintf(tmp, "%6d/%-21d", position + 1, NumOfLines);
	header->put(1, COLS-28, tmp);
	header->delay_update();
}

void LetterWindow::oneLine(int i)
{
	int j, length, z = position + i;

	if (z < NumOfLines) {
		Line *curr = linelist[z];

		text->attrib(curr->hidden ? C_LHIDDEN :
			(curr->origin ? C_LORIGIN :
			(curr->tearline ? C_LTEAR :
			(curr->tagline ? C_LTAGLINE :
			(curr->quoted ? C_LQTEXT : C_LTEXT)))));

		if (curr->tagline && (curr->length > 3)) {
			strncpy(tagline1, &curr->text[4], TAGLINE_LENGTH);
			tagline1[TAGLINE_LENGTH] = '\0';
		}

		length = text->put(i, 0, curr->text);
	} else
		length = 0;

	for (j = length; j < COLS; j++)
		text->put(i, j, ' ');
}

void LetterWindow::DrawBody()
{
	lineCount();

	for (int i = 0; i < y; i++)
		oneLine(i);
	text->delay_update();
}

void LetterWindow::DrawStat()
{
	static const char *helpmsg = "F1 or ? - Help ";
	char format[40], *tmp = new char[COLS + 1];

	int maxw = COLS - 16;

	bool collflag = false;
	if (mm.areaList->isCollection()) {
		if (mm.areaList->isReplyArea()) {
			maxw -= 10;
			sprintf(format, " REPLY in: %%-%d.%ds%%s",
				maxw, maxw);
		} else {
			maxw -= 13;
			sprintf(format, " PERSONAL in: %%-%d.%ds%%s",
				maxw, maxw);
		}
		mm.areaList->gotoArea(mm.letterList->getAreaID());
		collflag = true;
	} else
		sprintf(format, " %%-%d.%ds%%s", maxw, maxw);

	sprintf(tmp, format, mm.areaList->getDescription(), helpmsg);
	areaconv_in(tmp);

	if (collflag)
		interface->areas.Select();

	statbar->cursor_on();
	statbar->put(0, 0, tmp);
	statbar->delay_update();
	statbar->cursor_off();

	delete[] tmp;
}

void LetterWindow::MakeActive(bool redo)
{
	DestroyChain();

	y = LINES - 7;

	headbar = new Win(1, COLS, 0, C_LBOTTSTAT);
	header = new Win(5, COLS, 1, C_LHEADTEXT);
	text = new Win(y, COLS, 6, C_LTEXT);
	statbar = new Win(1, COLS, LINES - 1, C_LBOTTSTAT);

	char *tmp = new char[COLS + 1];
	int i = sprintf(tmp, " " MM_TOPHEADER, MM_NAME, MM_MAJOR,
		MM_MINOR);
	for (; i < COLS; i++)
		tmp[i] = ' ';
	tmp[i] = '\0';

	headbar->put(0, 0, tmp);
	headbar->delay_update();

	delete[] tmp;

 	Draw(redo);
}

bool LetterWindow::Next()
{
	if (mm.letterList->getActive() < (mm.letterList->noOfActive() - 1)) {
		interface->letters.Move(DOWN);
		mm.letterList->gotoActive(mm.letterList->getActive() + 1);
		Draw(true);
		return true;
	} else {
		interface->back();
		doupdate();
		interface->back();
		doupdate();
		interface->areas.KeyHandle('+');
	}
	return false;
}

bool LetterWindow::Previous()
{
	if (mm.letterList->getActive() > 0) {
		interface->letters.Move(UP);
		mm.letterList->gotoActive(mm.letterList->getActive() - 1);
		Draw(true);
		return true;
	} else {
		interface->back();
		doupdate();
		interface->back();
		doupdate();
		interface->areas.Move(UP);
	}
	return false;
}

void LetterWindow::Move(int dir)
{
	switch (dir) {
	case KEY_UP:
		if (position > 0) {
			position--;
			text->wscroll(-1);
			oneLine(0);
			text->delay_update();
			lineCount();
		}
		break;
	case KEY_DOWN:
		if (position < NumOfLines - y) {
			position++;
			lineCount();
			text->wscroll(1);
			oneLine(y - 1);
			text->delay_update();
		}
		break;
	case KEY_HOME:
		position = 0;
		DrawBody();
		break;
	case KEY_END:
		if (NumOfLines > y) {
			position = NumOfLines - y;
			DrawBody();
		}
		break;
	case 'B':
	case KEY_PPAGE:
		position -= ((y < position) ? y : position);
		DrawBody();
		break;
	case 'F':
	case KEY_NPAGE:
		if (position < NumOfLines - y) {
			position += y;
			if (position > NumOfLines - y)
				position = NumOfLines - y;
			DrawBody();
		}
		break;
	}
}

void LetterWindow::NextDown()
{
	position += y;
	if (position < NumOfLines)
		DrawBody();
	else
		Next();
}

void LetterWindow::Delete()
{
	DestroyChain();
	delete headbar;
	delete header;
	delete text;
	delete statbar;
}

bool LetterWindow::Save(int stype)
{
	FILE *fd;

	static const char *ntemplate[] = {
		"%.8s.%.03d", "%.8s.all", "%.8s.mkd"
	};

	static char keepname[3][128];
	char filename[128], oldfname[128];

	stype--;

	if (keepname[stype][0])
		strcpy(filename, keepname[stype]);
	else {
		switch (stype) {
		case 2:					// Marked
		case 1:					// All
			sprintf(filename, ntemplate[stype],
				mm.areaList->getName());
			break;
		case 0:					// This one
			sprintf(filename, ntemplate[0],
				mm.areaList->getName(),
					mm.letterList->getCurrent());
		}

		unspace(filename);
	}

	strcpy(oldfname, filename);

	if (interface->savePrompt("Save to file:", filename)) {
		mychdir(mm.resourceObject->get(SaveDir));
		if ((fd = fopen(filename, "at"))) {
			int num = mm.letterList->noOfActive();

			switch (stype) {
			case 2:
			case 1:
				for (int i = 0; i < num; i++) {
					mm.letterList->gotoActive(i);
					if ((stype == 1) ||
					  (mm.letterList->getStatus() &
					    MS_MARKED))
						write_to_file(fd);
				}
				break;
			case 0:
				write_to_file(fd);
			}
			fclose(fd);
			if (!stype)
				MakeChain(COLS);

			interface->setAnyRead();
		}
		if (strcmp(filename, oldfname))
			strcpy(keepname[stype], filename);

		return true;
	} else
		return false;

}

void LetterWindow::set_Letter_Params(net_address &nm, const char *to)
{
	NM = nm;
	strncpy(To, to ? to : "", 29);
}

void LetterWindow::set_Letter_Params(int area, char param_key)
{
	key = param_key;
	replyto_area = area;
}

void LetterWindow::QuoteText(FILE *reply)
{
	char TMP[81];
	int i;
	bool inet = (mm.areaList->isInternet() || mm.areaList->isUsenet());
	const char *TMP2 = mm.letterList->getFrom();

	sprintf(TMP, "-=> %.30s wrote to %.30s <=-\n\n",
		TMP2, mm.letterList->getTo());
	letterconv_in(TMP);
	fputs(TMP, reply);

	if (!inet) {
		char mg[4];

		strncpy(mg, TMP2, 2);
		mg[2] = '\0';
		mg[3] = '\0';

		i = 1;
		for (int j = 1; j < 3; j++) {
			bool end = false;
			while (TMP2[i] && !end) {
				if ((TMP2[i - 1] == ' ') && (TMP2[i] != ' ')) {
					mg[j] = TMP2[i];
					if (j == 1)
						end = true;
				}
				i++;
			}
		}
		letterconv_in(mg);

		sprintf(TMP, " %s> %%s\n", mg);
	} else
		strcpy(TMP, "> %s\n");

	MakeChain(inet ? 77 : 72, true);

	for (i = 0; i < NumOfLines; i++) {
		Line *curr = linelist[i];
		fprintf(reply, curr->length ? (curr->quoted ? (inet ?
			">%s\n" : ((curr->text[0] == ' ') ? "%s\n" :
			" %s\n")) : TMP) : "%s\n", curr->text);
	}

	MakeChain(COLS);
}

int LetterWindow::HeaderLine(ShadowedWin &win, char *buf, int limit,
				int pos, int color)
{
	areaconv_in(buf);
	int getit = win.getstring(pos, 8, buf, limit, color, color);
	areaconv_out(buf);
	return getit;
}

int LetterWindow::EnterHeader(char *FROM, char *TO, char *SUBJ, bool &privat)
{
	static const char *noyes[] = { "No", "Yes" };

	char NETADD[72];
	int result, current, maxitems = 2;
	bool end = false;
	bool hasNet = mm.areaList->isEmail();
	bool hasTo = mm.areaList->hasTo();

	if (hasNet) {
		strcpy(NETADD, NM);
		maxitems++;
	} else
		NM.isSet = false;

	if (hasTo)
		maxitems++;

	ShadowedWin rep_header(maxitems + 2, COLS - 2, (LINES / 2) - 3,
		C_LETEXT);

	rep_header.put(1, 2, "From:");
	if (hasTo)
		rep_header.put(2, 2, "  To:");
	if (hasNet)
		rep_header.put(3, 2, "Addr:");
	rep_header.put(maxitems, 2, "Subj:");

	rep_header.attrib(C_LEGET2);
	rep_header.put(1, 8, FROM);

	if (hasNet && !NM.isSet) {
		//NM.isInternet = mm.areaList->isInternet();
		TO[0] = '\0';
		current = 1;
	} else {
		if (hasTo) {
			rep_header.attrib(C_LEGET1);
			rep_header.put(2, 8, TO);
		}
		if (hasNet) {
			rep_header.attrib(C_LEGET2);
			rep_header.put(3, 8, NETADD);
		}
		current = maxitems - 1;
	}

	rep_header.update();

	do {
		result = HeaderLine(rep_header, (current == (maxitems - 1)) ?
			SUBJ : (((current == 2) && hasNet) ?
			NETADD : ((current && hasTo) ? TO : FROM)),
			((current == (maxitems - 1)) ||
			((current == 2) && NM.isInternet)) ? 69 :
			mm.areaList->maxToLen(), current + 1,
			(current & 1) ? C_LEGET1 : C_LEGET2);

		switch (result) {
		case 0:
			end = true;
			break;
		case 1:
			current++;
			if (current == maxitems)
				end = true;
			break;
		case 2:
			if (current > 0)
				current--;
			break;
		case 3:
			if (current < maxitems - 1)
				current++;
		}
	} while (!end);

	if (result) {
		int pmode = (!hasNet ? mm.areaList->hasPublic() : 0) +
			((mm.areaList->hasPrivate() &&
			!mm.areaList->isUsenet()) ? 2 : 0);

		if (hasNet)
			NM = NETADD;

		switch (pmode) {
		case 1:
			privat = false;
			break;
		case 2:
			privat = true;
			break;
		case 3:
			if (!interface->WarningWindow(
				"Make letter private?", privat ? 0 : noyes))
						privat = !privat;
		}
	}
	return result;
}

void LetterWindow::editletter(const char *reply_filename)
{
	char command[256];

	sprintf(command, "%s %s", mm.resourceObject->get(editor),
		canonize(reply_filename));
	mysystem(command);
}

long LetterWindow::reconvert(const char *reply_filename)
{
	FILE *reply;

	reply = fopen(reply_filename, "rt");
	fseek(reply, 0, SEEK_END);
	long replen = ftell(reply);
	rewind(reply);

	char *body = new char[replen + 1];
	replen = (long) fread(body, 1, replen, reply);
	fclose(reply);

	body[replen] = '\0';
	areaconv_out(body);
	while (body[replen - 1] == '\n')
		replen--;

	reply = fopen(reply_filename, "wt");
	fwrite(body, 1, replen, reply);
	fclose(reply);

	delete[] body;
	return replen;
}

void LetterWindow::setToFrom(char *TO, char *FROM)
{
	char format[7];
	sprintf(format, "%%.%ds", mm.areaList->maxToLen());

	bool usealias = mm.areaList->getUseAlias();
	if (usealias) {
		sprintf(FROM, format, mm.resourceObject->get(AliasName));
		if (!FROM[0])
			usealias = false;
	}
	if (!usealias)
		sprintf(FROM, format, mm.resourceObject->get(LoginName));

	if (key == 'E')
		strcpy(TO, (To[0] ? To : "All"));
	else
		sprintf(TO, format, (key == 'O') ? mm.letterList->getTo() :
			mm.letterList->getFrom());
}

void LetterWindow::EnterLetter()
{
	FILE *reply;
	char reply_filename[256], FROM[36], TO[36], SUBJ[78];
	const char *orig_id;

	mystat fileStat;
	time_t oldtime;

	long replen;
	int replyto_num;
	bool privat;

	mm.areaList->gotoArea(replyto_area);

	// HEADER

	setToFrom(TO, FROM);

	if (key == 'E')
		SUBJ[0] = '\0';	//we don't have subject yet
	else
		sprintf(SUBJ, "Re: %.65s",
			stripre(mm.letterList->getSubject()));

	privat = (key == 'E') ? false : mm.letterList->getPrivate();

	if (!EnterHeader(FROM, TO, SUBJ, privat)) {
		NM.isSet = false;
		interface->areas.Select();
		interface->redraw();
		return;
	}

	// Don't use refnum if posting in different area:

	replyto_num = ((key == 'E') || (key == 'N') || (replyto_area !=
		mm.letterList->getAreaID())) ? 0 : mm.letterList->getMsgNum();

	orig_id = replyto_num ? mm.letterList->getMsgID() : 0;

	// BODY

	mytmpnam(reply_filename);

	// Quote the old text 

	if (key != 'E') {
		reply = fopen(reply_filename, "wt");
		QuoteText(reply);
		fclose(reply);
	}
	fileStat.init(reply_filename);
	oldtime = fileStat.date;

	// Edit the reply

	editletter(reply_filename);
	interface->areas.Select();
	interface->redraw();

	// Check if modified

	fileStat.init(reply_filename);
	if (fileStat.date == oldtime)
		if (interface->WarningWindow("Cancel this letter?")) {
			remove(reply_filename);
			interface->redraw();
			return;
		}

	// Mark original as replied

	if (key != 'E') {
		int origatt = mm.letterList->getStatus();
		mm.letterList->setStatus(origatt | MS_REPLIED);
		if (!(origatt & MS_REPLIED))
			interface->setAnyRead();
	}

	reply = fopen(reply_filename, "at");

	// Signature

	const char *sg = mm.resourceObject->get(sigFile);
	if (sg && *sg) {
		FILE *s = fopen(sg, "rt");
		if (s) {
			char bufx[81];

			fputc('\n', reply);
			while (fgets(bufx, sizeof bufx, s))
				fputs(bufx, reply);
			fclose(s);
		}
	}

	// Tagline

	bool useTag = mm.resourceObject->get(UseTaglines) &&
		interface->Tagwin();
	if (useTag)
		fprintf(reply, "\n... %s\n",
			interface->taglines.getCurrent());
	else
		fprintf(reply, " \n");

	// Tearline (not for Blue Wave -- it does its own)

	if (mm.driverList->useTearline())
		fprintf(reply, "--- %s/%s v%d.%d\n", MM_NAME, sysname(),
			MM_MAJOR, MM_MINOR);

	fclose(reply);

	// Reconvert the text

	mm.areaList->gotoArea(replyto_area);
	replen = reconvert(reply_filename);
	mm.areaList->enterLetter(replyto_area, FROM, TO, SUBJ, orig_id,
		replyto_num, privat, NM, reply_filename, (int) replen);

	NM.isSet = false;
	To[0] = '\0';

	interface->areas.Select();
	interface->setUnsaved();
}

void LetterWindow::EditLetter(bool forwarding)
{
	FILE *reply;
	char reply_filename[256], FROM[36], TO[36], SUBJ[78], *body;
	const char *msgid;
	long siz;
	int replyto_num, replyto_area;
	bool privat;

	bool is_reply_area = (mm.areaList->getAreaNo() == REPLY_AREA);

	NM = mm.letterList->getNetAddr();

	replyto_area = interface->areaMenu();
	if (replyto_area == -1)
		return;

	// The refnum is only good for the original area:

	replyto_num = (replyto_area != mm.letterList->getAreaID()) ?
		0 : mm.letterList->getReplyTo();

	msgid = replyto_num ? mm.letterList->getMsgID() : 0;

	mm.areaList->gotoArea(replyto_area);

	if (forwarding) {
		if (mm.areaList->isEmail()) {
			interface->areas.Select();
			interface->addressbook();
			mm.areaList->gotoArea(replyto_area);
		} else
			NM.isSet = false;
		key = 'E';
		setToFrom(TO, FROM);
		sprintf(SUBJ, "%.63s (fwd)", mm.letterList->getSubject());
		privat = false;
	} else {
		strcpy(FROM, mm.letterList->getFrom());
		strcpy(TO, mm.letterList->getTo());
		sprintf(SUBJ, "%.69s", mm.letterList->getSubject());
		privat = mm.letterList->getPrivate();
	}

	if (!EnterHeader(FROM, TO, SUBJ, privat)) {
		NM.isSet = false;
		interface->areas.Select();
		interface->redraw();
		return;
	}

	DestroyChain();		// current letter's chain reset

	mytmpnam(reply_filename);

	body = (char *) mm.letterList->getBody();
	letterconv_in(body);
	reply = fopen(reply_filename, "wt");
	if (forwarding)
		write_header_to_file(reply);
	fwrite(body, strlen(body), 1, reply);
	fclose(reply);
	body = 0;		// it will be auto-dealloc'd by next getBody

	editletter(reply_filename);
	siz = reconvert(reply_filename);

	if (!forwarding)
		mm.areaList->killLetter(mm.letterList->getMsgNum());

	mm.areaList->enterLetter(replyto_area, FROM, TO, SUBJ, msgid,
		replyto_num, privat, NM, reply_filename, (int) siz);

	if (is_reply_area) {
		mm.letterList->rrefresh();
		interface->letters.ResetActive();
	}
	interface->areas.Select();

	NM.isSet = false;

	interface->setUnsaved();
}

bool LetterWindow::SplitLetter(int lines)
{
	static int eachmax = mm.resourceObject->getInt(MaxLines);

	if (!lines) {
		char maxlinesA[55];

		sprintf(maxlinesA, "%d", eachmax);
		if (!interface->savePrompt(
	    	"Max lines per part? (WARNING: Split is not reversible!)",
			maxlinesA) || !sscanf(maxlinesA, "%d", &eachmax))
				return false;
	}
	int maxlines = lines ? lines : eachmax;

	if (maxlines < 20)
		return false;

	MakeChain(80);
	int orglines = NumOfLines;
	int parts = (orglines - 1) / maxlines + 1;

	if (parts == 1)
		return false;

	NM = mm.letterList->getNetAddr();

	int replyto_area = mm.letterList->getAreaID();
	int replyto_num = mm.letterList->getReplyTo();

	const char *msgid = replyto_num ? mm.letterList->getMsgID() : 0;

	char FROM[36], TO[36], ORGSUBJ[69];

	strcpy(FROM, mm.letterList->getFrom());
	strcpy(TO, mm.letterList->getTo());
	sprintf(ORGSUBJ, "%.68s", mm.letterList->getSubject());

	bool privat = mm.letterList->getPrivate();

	int clines = 0;

	mm.areaList->gotoArea(replyto_area);
	mm.areaList->killLetter(mm.letterList->getMsgNum());

	for (int partno = 1; partno <= parts; partno++) {
		FILE *reply;
		char reply_filename[256], SUBJ[80];

		mytmpnam(reply_filename);

		int endline = (orglines > maxlines) ? maxlines: orglines;

		reply = fopen(reply_filename, "wt");
		for (int i = clines; i < (clines + endline); i++)
			fprintf(reply, "%s\n", linelist[i]->text);
		fclose(reply);

		mystat fileStat;
		fileStat.init(reply_filename);

		sprintf(SUBJ, "%s [%d/%d]", ORGSUBJ, partno, parts);

		mm.areaList->enterLetter(replyto_area, FROM, TO, SUBJ, msgid,
			replyto_num, privat, NM, reply_filename,
			(int) fileStat.size);

		clines += endline;
		orglines -= endline;
	}
	NM.isSet = false;

	mm.letterList->rrefresh();

	if (!lines) {
		interface->letters.ResetActive();
		interface->areas.Select();
		interface->setUnsaved();
	}
	return true;
}

void LetterWindow::GetTagline()
{
	interface->taglines.EnterTagline(tagline1);
	ReDraw();
}

void LetterWindow::write_header_to_file(FILE *fd)
{
	char Header[400], *p;
	int j;

	for (j = 0; j < 72; j++)
		fputc('=', fd);

	mm.areaList->gotoArea(mm.letterList->getAreaID());

	p = Header + sprintf(Header,
		"\n System: %.71s\n   Area: %.71s\n   Date: %.25s\n"
		"   From: %.45s", mm.resourceObject->get(BBSName),
		mm.areaList->getDescription(), mm.letterList->getDate(),
		mm.letterList->getFrom());

	interface->areas.Select();

	if (mm.areaList->isEmail())
		p = netAdd(p);

	p += sprintf(p, "\n     To: %.45s", mm.letterList->getTo());

	if (mm.areaList->isReplyArea())
		p = netAdd(p);

	sprintf(p, "\n   Subj: %.71s\n", mm.letterList->getSubject());

	letterconv_in(Header);
	fputs(Header, fd);

	for (j = 0; j < 72; j++)
		fputc('-', fd);
	fputc('\n', fd);
}

void LetterWindow::write_to_file(FILE *fd)
{
	write_header_to_file(fd);

	// write chain to file

	MakeChain(80);
	for (int j = 0; j < NumOfLines; j++)
		fprintf(fd, "%s\n", linelist[j]->text);
	fputc('\n', fd);

	// set read, unmarked -- not part of writing to a file, but anyway

	int stat = mm.letterList->getStatus();
	int oldstat = stat;
	stat |= MS_READ;
	stat &= ~MS_MARKED;
	mm.letterList->setStatus(stat);
	if (stat != oldstat)
		interface->setAnyRead();
}

// For searches (may want to start at position == -1):
void LetterWindow::setPos(int x)
{
	position = x;
}

int LetterWindow::getPos()
{
	return position;
}

searchret LetterWindow::search(const char *item)
{
	searchret found = False;

	for (int x = position + 1; (x < NumOfLines) && (found == False);
	    x++) {

		if (text->keypressed() == 27) {
			found = Abort;
			break;
		}

		found = searchstr(linelist[x]->text, item) ? True : False;

		if (found == True) {
			position = x;
			if (interface->active() == letter)
				DrawBody();
		}
	}

	return found;
}

bool LetterWindow::EditOriginal()
{
	int old_area = mm.letterList->getAreaID();
	int old_mnum = mm.letterList->getMsgNum();
	int orig_area = mm.areaList->getAreaNo();
	int orig_mnum = mm.letterList->getCurrent();

	letter_list *old_list = mm.letterList;
	mm.areaList->gotoArea(REPLY_AREA);
	mm.areaList->getLetterList();
	interface->areas.ResetActive();

	bool found = mm.letterList->findReply(old_area, old_mnum);

	if (found && interface->WarningWindow("A reply exists. Re-edit it?"))
		EditLetter(false);
	else
		found = false;

	delete mm.letterList;
	mm.letterList = old_list;
	mm.areaList->gotoArea(orig_area);
	interface->areas.ResetActive();
	mm.letterList->gotoLetter(orig_mnum);
	interface->letters.ResetActive();

	return found;
}

void LetterWindow::SplitAll(int lines)
{
	letter_list *old_list = 0;
	statetype state = interface->active();
	bool list_is_active = (state == letter) || (state == letterlist);

	interface->areas.Select();
	int orig_area = mm.areaList->getAreaNo();

	bool is_reply_area = (orig_area == REPLY_AREA) && list_is_active;

	if (!is_reply_area) {
		if (list_is_active)
			old_list = mm.letterList;

		mm.areaList->gotoArea(REPLY_AREA);
		mm.areaList->getLetterList();
		interface->areas.ResetActive();
	}

	bool anysplit = false;
	int last = mm.letterList->noOfActive();
	for (int i = 0; i < last; i++) {
		mm.letterList->gotoActive(i);
		if (SplitLetter(lines)) {
			i--;
			last--;
			anysplit = true;
		}
	}

	if (is_reply_area)
		interface->letters.ResetActive();
	else {
		delete mm.letterList;
		if (list_is_active)
			mm.letterList = old_list;
	}
	mm.areaList->gotoArea(orig_area);
	interface->areas.ResetActive();

	if ((state == letter) && !is_reply_area)
		MakeChain(COLS);

	if (anysplit)
		interface->nonFatalError("Some replies were split");
}

void LetterWindow::KeyHandle(int key)
{
	int t_area;

	switch (key) {
	case 'D':
		rot13 = !rot13;
		interface->redraw();
		break;
	case 'X':
		hidden = !hidden;
		interface->redraw();
		break;
	case 'I':
		stripCR = !stripCR;
		interface->redraw();
		break;
	case 'S':
		Save(1);
		DrawHeader();
		ReDraw();
		break;
	case '?':
	case KEY_F(1):
		interface->changestate(letter_help);
		break;
	case 'V':
	case 1: // CTRL-A
	case 11: // CTRL-V
		{
			int nextAns;
			bool cont = false;
			do {
			    nextAns = interface->ansiLoop(mm.letterList->
				getBody(), mm.letterList->getSubject());
			    if (nextAns == 1)
				cont = Next();
			    else if (nextAns == -1)
				cont = Previous();
			} while (nextAns && cont);
		}
		break;
	case KEY_RIGHT:
		if (lynxNav)
			break;
	case MM_PLUS:
		Next();
		break;
	case KEY_LEFT:
		if (lynxNav) {
			interface->back();
			break;
		}
	case MM_MINUS:
		Previous();
		break;
	case ' ':
		NextDown();
		break;
	case 6:				// Ctrl-F
		EditLetter(true);
		interface->redraw();
		break;
	default:
	    if (mm.areaList->isReplyArea()) {
		switch(key) {
		case 'R':
		case 'E':
			EditLetter(false);
			interface->redraw();
			break;
		case KEY_DC:
		case 'K':
			interface->kill_letter();
			break;
		case 2:				// Ctrl-B
			SplitLetter();
			interface->redraw();
			break;
		default:
			Move(key);
		}
	    } else {
		switch (key) {
		case '\t':
			interface->letters.Next();
			Draw(true);
			break;
		case 'M':
		case 'U':
			StatToggle((key == 'M') ? MS_MARKED : MS_READ);
			break;
		case 'R':	// Allow re-editing from here:
		case 'O':
			if (mm.letterList->getStatus() & MS_REPLIED)
				if (EditOriginal()) {
					interface->redraw();
					break;
				}
		case 'E':
			t_area = interface->areaMenu();
			if (t_area != -1) {
				mm.areaList->gotoArea(t_area);
				set_Letter_Params(t_area, key);

				if (mm.areaList->isEmail())
					if (key == 'E')
						interface->addressbook();
					else {
						net_address nm = PickNetAddr();
						set_Letter_Params(nm, 0);
					}
				EnterLetter();
			}
			break;
		case 'N':
			t_area = (mm.areaList->isUsenet() ||
				mm.areaList->isInternet()) ?
				mm.areaList->findInternet() :
				mm.areaList->findNetmail();
			if (t_area != -1) {
			    net_address nm = PickNetAddr();
			    if (nm.isSet) {
				set_Letter_Params(t_area, 'N');
				set_Letter_Params(nm, 0);
				EnterLetter();
			    } else
				interface->nonFatalError(
					"No reply address");
			} else
				interface->nonFatalError(
					"Netmail is not available");
			break;
		case 'T':
			GetTagline();
			break;
		default:
			Move(key);
		}
	    }
	}
}
